# ui-activesourcemgr.tcl --
#
#       ActiveSourceManager is used to dispatch and manage ActiveSource
#       objects.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/vic/ui-activesourcemgr.tcl,v 1.17 2002/02/23 03:30:03 chema Exp $


import Observer ActiveSource UserWindow Configuration

#
# ActiveSourceManager is used to dispatch and manage ActiveSource
# objects.  It is an Observer of VideoAgent.
#
Class ActiveSourceManager -superclass {Observer} -configuration {
	tile 1
}

#
# Tile out in a grid the video of the ActiveSources followed by the
# provided <i>videoAgent</i>.  If <i>list_direction</i> is "horizontal",
# the <i>tile</i> resource option indicates the initial number of rows
# for the grid.  If <i>list_direction</i> is "vertical", the <i>tile</i>
# resource option indicates the initial number of columns for the grid.
#
ActiveSourceManager public init {ui w videoAgent list_direction localChannel {autoplace 0}} {
	$self next

	$self instvar ui_ videoAgent_ localChannel_ autoplace_
	set ui_ $ui
	set videoAgent_ $videoAgent
	set localChannel_ $localChannel
	set autoplace_ $autoplace

	$self instvar curcol_ currow_ ncol_ nrow_ list_direction_
	set curcol_ 0
	set currow_ 0
	set list_direction_ $list_direction
	set ncol_ [$self get_option tile]
	set nrow_ [$self get_option tile]

	# let this ActiveSourceManager be an observer of the VideoAgent so it can catch "trigger_format" and "activate" and "deactivate"
	$videoAgent_ attach $self
}

#
ActiveSourceManager public init_grid {w} {
	$self instvar grid_ label_

	frame $w
	set grid_ $w.grid
	frame $grid_
	set label_ $w.label
	label $label_ -text "Waiting for video..."

	pack $label_ -anchor c -expand 1 -side left -fill both
}

#
# The provided ActiveSource, <i>as</i>, is added to the active_() instvar (an array of active senders indexed by <i>src</i>).
# If the array was previously empty, the $label_ is removed and the $grid_ is packed in its place.
#
ActiveSourceManager public add_active { as src } {
	$self instvar active_ grid_ label_
	set active_($src) $as
	if { [array size active_] == 1 } {
		pack forget $label_
		pack $grid_ -expand 1 -fill x -anchor n
	}
}

#
# The specified <i>src</i> is removed from the active_() instvar (an array of active senders indexed by src).
# If this was the last element of the array, the $grid_ is removed and the $label_ is packed in its place.
#
ActiveSourceManager public rm_active src {
	$self instvar active_ grid_ label_
	unset active_($src)
	if { [array size active_] == 0 } {
		pack forget $grid_
		pack $label_ -anchor c -expand 1 -side left
	}
}

#
# Return a list of the active sources.
#
ActiveSourceManager public active-sources {} {
	$self instvar active_
	return [array names active_]
}

#
# Update the row and column variables to indicate an addition to the
# grid of thumbnails of ActiveSources.  If the <i>list_direction_</i> is
# "vertical", additions to the grid are placed at the end of the first
# non-full column.  When a column becomes full, a new column is begun,
# filled from top to bottom.  If the <i>list_direction_</i> is
# "horizontal", additions to the grid are placed at the end of the first
# non-full row.  When a row becomes full, a new row is begun, filled
# from left to right.
#
ActiveSourceManager public bump { } {
	$self instvar curcol_ currow_ list_direction_

	if { $list_direction_ == "vertical" } {
		$self instvar ncol_
		incr curcol_
		if { $curcol_ == $ncol_ } {
			set curcol_ 0
			incr currow_
		}
	} else {
		$self instvar nrow_
		incr currow_
		if { $currow_ == $nrow_ } {
			set currow_ 0
			incr curcol_
		}
	}
}

#
# If the <i>list_direction_</i> is "vertical", reformat the grid of
# video thumbnails into <i>n</i> columns.  If the <i>list_direction_</i>
# is "horizontal", reformat the grid of video thumbnails into <i>n</i>
# rows.
#
ActiveSourceManager public redecorate n {
	$self instvar curcol_ currow_ list_direction_
	set curcol_ 0
	set currow_ 0

	if { $list_direction_ == "vertical" } {
		$self instvar ncol_
		set ncol_ $n
	} else {
		$self instvar nrow_
		set nrow_ $n
	}

	$self instvar grid_
	if ![info exists grid_] {
		return
	}

	foreach src [$self active-sources] {
		grid $grid_.$src -row $currow_ -column $curcol_ -sticky we
		if { $list_direction_ == "vertical" } {
			grid columnconfigure $grid_ $curcol_ -weight 1
		} else {
			grid rowconfigure $grid_ $currow_ -weight 1
		}
		$self bump
	}
}


#
# Add a <i>src</i> to the active senders list.  E.g., make a postage
# stamp window appear, stats, etc. so that the user can select
# the video stream.
#
ActiveSourceManager public activate src {
	#
	# give the VideoAgent a chance to create and install
	# a decoder and for that decoder to see a packet so it can
	# determine the output geometry and color decimation.
	# we shouldn't have to do this (e.g., resize will
	# take care of a geometry change), but currently
	# decoders can't trigger a renderer realloation
	# when the decimation changes.FIXME fix this
	# Note that trigger_format allocates the decoder object
	# is called before the event loop goes idle
	#
	after idle "$self really_activate $src"
}

#
# Create an ActiveSource, $as, within the widget, $grid_.$src, out of the provided <i>src</i>.
# (This entails dropping a postage-stamp-sized VideoWidget decoding the $src into the $grid_.$src widget.)
# The widget, $grid.$src, is subsequently positioned within $grid_ by the grid manager in
# the row specified by instvar $currow_ and the column specified by instvar $curcol_.
# The column and row variables are updated by the bump method to reflect this addition to the grid.
# Add this ActiveSource, $as, to the instvar array active_() (the array of active senders).
# Update the decoder for the specified <i>src</i>.
#
ActiveSourceManager public really_activate src {
	$self instvar grid_ curcol_ currow_ ui_ list_direction_ localChannel_ autoplace_

	set as [new ActiveSource $self $ui_ $grid_.$src $src $localChannel_]

	# If autoplace is active, get the x,y coordinates and scale factor
	# for the new ActiveSource. After creating the UserWindow, register
	# it so its movements, etc. can be tracked.
	if { $autoplace_ } {
		set coords_scale [$self autoplace add $as]
		$as place [lindex $coords_scale 0] \
			[lindex $coords_scale 1] \
			[lindex $coords_scale 2]
	}

	grid $grid_.$src -row $currow_ -column $curcol_ -sticky we

	if { $list_direction_ == "vertical" } {
		grid columnconfigure $grid_ $curcol_ -weight 1
	} else {
		grid rowconfigure $grid_ $currow_ -weight 1
	}

	$ui_ update_decoder $src
	$self bump

	#
	# Someone became activated, so we have to change
	# the switchable menu to include this participant
	#
	Switcher rebuild_switch_list_menu

	# FIXME this is a temporary hack while cindy works on the VideoBox class...
	if {[$ui_ info class] == "MuiUI"} {
		$ui_ maybe_switch_in $src
	}
}

#
# Remove a <i>src</i> from the active senders list.
#
ActiveSourceManager public deactivate {src} {
	$self instvar active_
	if [info exists active_($src)] {
		set as $active_($src)
		set L [$as user-windows]
		foreach uw $L {
			#FIXME should check if we're voice-switched
			# and if so, bump window
			delete $uw
		}
		# detach any we missed by deleting user-windows
		$as detach-windows
		$as delete-decoder-window
	}

	$self instvar grid_
	set w $grid_.$src
	if [winfo exists $w] {
		grid forget $w
		destroy $w
		$self rm_active $src
	}

	$src handler ""

	#
	# Someone became de-activated, so we have to change
	# the switchable menu to un-include this participant
	#
	Switcher rebuild_switch_list_menu

	#FIXME
	global ftext btext ltext fpshat bpshat lhat shat
	unset ftext($src)
	unset btext($src)
	unset ltext($src)
	unset fpshat($src)
	unset bpshat($src)
	unset lhat($src)
	unset shat($src)
}

#
# Dispatch focus method on switcher windows.
#
ActiveSourceManager public focus_speaker { infoname msg } {
	foreach s [$self active-sources] {
		if { [$s sdes cname] == $msg } {
		        Switcher focus $s
		}
        }
}

#
# Update the name of the <i>src</i> for the UserWindows in which it appears.
#
ActiveSourceManager public change_name {src} {
	set name [$src sdes name]
	# update viewing window names to reflect new name
	$self instvar active_
	if [info exists active_($src)] {
		set as $active_($src)
		foreach uw [$as user-windows] {
			if { [$uw attached-source] == "$as" } {
				$uw set-name $name
			}
		}
	}

	# Someone's name has been altered , so we have to change
	# the switchable menu to include this participant
	#
	Switcher rebuild_switch_list_menu
}

#
ActiveSourceManager public trigger_format {src} {
	$self instvar active_ videoAgent_ ui_

	if ![info exists active_($src)] {
		#
		# if we get a change format before really_activate
		# was called (i.e., so we don't even have a thumbnail yet),
		# don't do anything
		#
		return
	}

	set as $active_($src)

	set L [$as user-windows]
	$as detach-windows
	#FIXME
	set extoutList [extout_detach_src $src]

	set d [$videoAgent_ reactivate $src]

	$ui_ update_decoder $src
	global colorbutton
	$d color $colorbutton($src)

	foreach uw $L {
		$as attach-window $uw
		[$uw video-widget] redraw
	}
	$as attach-thumbnail
	#FIXME
	extout_attach_src $src $extoutList
}

#
# Called when the video stream state changes in a way that would
# affect the choice of renderer.  For example, when a jpeg stream
# changes from type-0 to type-1 we might have to revert from
# hardware to software decoding, or we might have to reallocate
# a 422 renderer as a 411 renderer.  This never needs to happen
# for most stream types (i.e., because the decimation factor is fixed).
#
ActiveSourceManager public decoder_changed {src} {
	$self instvar active_
	if ![info exists active_($src)] {
		return
	}
	#FIXME redundant with trigger_format
	set as $active_($src)

	set L [$as user-windows]
	$as detach-windows
	#FIXME
	set extoutList [extout_detach_src $src]
	foreach uw $L {
		$as attach-window $uw
		[$uw video-widget] redraw
	}
	#FIXME
	extout_attach_src $src $extoutList
	return
}


ActiveSourceManager public trigger_format_all { } {
	foreach s [$self active-sources] {
		$self trigger_format $s
	}
}


ActiveSourceManager public get_activesource src {
	$self instvar active_
	if [info exists active_($src)] {
		return $active_($src)
	} else {
		return ""
	}
}

#
# switches the VideoAgent object.
# This method should be called always from VicUI::switch-agent
#
ActiveSourceManager instproc switch-agent {new_agent} {
	$self instvar videoAgent_

	# the attachement is already done by VicUI::switch-agent
	#$new_agent attach $self

	# set the new videoAgent
	set videoAgent_ $new_agent
}

#
# Outside hook via tkvar autoplaceNewSources to turn on/off use of the
# autoplace function.
#
ActiveSourceManager instproc set_autoplace {} {
    $self tkvar autoplaceNewSources
    if { $autoplaceNewSources } {
	$self autoplace_on
    } else {
	$self autoplace_off
    }
}

#
# Toggle the autoplace function on and off.
#
ActiveSourceManager instproc toggle_autoplace {} {
    $self instvar autoplace_
    if { $autoplace_ } {
	$self autoplace_off
    } else {
	$self autoplace_on
    }
}

#
# Turn off autoplace.
#
ActiveSourceManager instproc autoplace_off {} {
    $self instvar autoplace_
    set autoplace_ false
}

#
# Turn on autoplace. Close all windows and replace them.
#
ActiveSourceManager instproc autoplace_on {} {
    $self instvar autoplace_ active_

    # Don't do anything if autoplace is already on.
    if {$autoplace_} return

    # Close all windows. We can't just call detach-windows because that
    # also detaches the thumbnail.
    foreach src [$self active-sources] {
	set as $active_($src)
	foreach uw [$as user-windows] {
	    # We have to check that the user windows this ActiveSource
	    # knows about are actually still attached to this ActiveSource
	    # because voice-switched or browse-mode windows change their
	    # ActiveSources.
	    if {[$uw attached-source] == "$as"} {
		$uw destroy
	    }
	}
    }

    # Replace all windows.
    foreach src [$self active-sources] {
	set as $active_($src)
	set coords_scale [$self autoplace add $as]
	$as place [lindex $coords_scale 0] \
		[lindex $coords_scale 1] \
		[lindex $coords_scale 2]
    }

    # Set new autoplace_ value.
    set autoplace_ true
}

#
# Default autoplace function.
#
# {add $as} is called when a new ActiveSource is to be placed. It
#           should return virtual screen coordinates and image size
#           formatted as "x y scale". If scale is 0, the ActiveSource
#           will not be placed.
# {remove $as} is called when an ActiveSource's display window is
#              destroyed (e.g. user closes the window).
# {register $as $uw} is called when the ActiveSource's display window
#                    has been created to associate a specific
#                    UserWindow with the ActiveSource.
ActiveSourceManager instproc autoplace {cmd as {uw false}} {
    global mutebutton
    $self instvar asources_ windows_ mon_ xbase_ ybase_ nextx_ nexty_ maxy_ xoffset_ yoffset_ xincr_ yincr_ mon_list_ screen_width_ screen_height_ minx_ miny_ maxx_ maxy_ framesize_ titlebarsize_

    # Should we place windows across monitors or only within monitors?
    set cross_monitors 0

    # Handle add commands.
    if {$cmd == "add"} {
	# Get the monitor coordinates.
	if {![info exists mon_list_]} {
	    # get_monitorinfo returns a list of lists. Each element of the
	    # list is a list formatted as "minx miny maxx maxy". The first
	    # element contains the total virtual screen resolution. All
	    # subsequent elements contain the physical screen resolutions
	    # in the virtual screen coordinate space. e.g. the second
	    # monitor might return "1024 0 1824 600" if its resolution is
	    # 800x600.
	    set mon_list_ [get_monitorinfo]
	    if { $mon_list_ == "" } {
		set mon_list_ [list [list 0 0 [winfo screenwidth .] [winfo screenheight .]] [list 0 0 [winfo screenwidth .] [winfo screenheight .]]]
	    }

	    # Grab the virtual screen coordinates.
	    set v_coords [lindex $mon_list_ 0]
	    set minx_ [lindex $v_coords 0]
	    set miny_ [lindex $v_coords 1]
	    set maxx_ [lindex $v_coords 2]
	    set maxy_ [lindex $v_coords 3]

	    # Remove the virtual screen coordinates from the monitor list,
	    # and sort left-most, then top-most.
	    set mon_list_ [lsort [lrange $mon_list_ 1 end]]

	    # Compute the virtual screen width and height.
	    set screen_width_ [expr $maxx_ - $minx_ + 1]
	    set screen_height_ [expr $maxy_ - $miny_ + 1]

        # Compute the window frame and titlebar sizes.
        set wm_geom [split [wm geometry .] {+}]
        set winfo_geom [split [winfo geometry .] {+}]
        if {[lindex $wm_geom 1] != [lindex $winfo_geom 1] && \
            [lindex $wm_geom 2] != [lindex $winfo_geom 2]} \
        {
            set framesize_ [expr [lindex $winfo_geom 1] - [lindex $wm_geom 1]]
            set titlebarsize_ [expr [lindex $winfo_geom 2] - [lindex $wm_geom 2]]
        } else {
            set framesize_ [expr [winfo rootx .] - [winfo x .]]
            set titlebarsize_ [expr [winfo rooty .] - [winfo y .]]
        }
	}

	# If there are no ActiveSources placed, initialize values.
	#
	# mon_     the list index of the current monitor being populated.
	# xbase_   the left-most pixel value of the last window placed.
	# nextx_   the right-most pixel value of the last window placed.
	# nexty_   the bottom-most pixel value of the last completed row.
	# maxy_    the bottom-most pixel value of the current row.
	# xincr_   the pixel amount to increase the xbase by each time.
	# yincr_   the pixel amount to increase the ybase by each time.
	# xoffset_ the horizontal pixel spacing between windows.
	# yoffset_ the vertical pixel spacing between windows.
	if {![array size asources_]} {
	    set mon_   0
	    set xbase_ [expr $minx_ - $framesize_]
	    set ybase_ [expr $miny_ - $titlebarsize_]
	    set nextx_ $xbase_
	    set nexty_ $ybase_
	    set maxy_ $nexty_
	    set xincr_ 10
	    set yincr_ 20
	    set xoffset_ $framesize_
	    set yoffset_ $titlebarsize_
	}
	
	# Initialize the return values.
	set retx [expr $nextx_ + $xoffset_]
	set rety [expr $nexty_ + $yoffset_]
	set scale 1

    # If the ActiveSource is muted, just return 0 for the scale.
    if {$mutebutton([$as source])} {
        return "$retx $rety 0"
    }

	# If the ActiveSource was already placed, just return the previously
	# calculated coordinates.
	if {[info exists asources_($as)]} {
	    return $asources_($as)
	}
	
	# Get the video stream dimensions.
	set src [$as source]
	set decoder [$src handler]
	set width [expr int([$decoder width] * $scale)]
	set height [expr int([$decoder height] * $scale)]

	# If we're crossing monitors or there is only one monitor, use
	# a simple left-right, top-down placement. If we run out of room
	# on the bottom of the virtual screen, jump back to the top.
	if {$cross_monitors || [llength $mon_list_] == 1} {
	    # If there is no room in the current row, try the next row.
	    if {[expr $retx + $width] >= $screen_width_} {
		set retx [expr $xbase_ + $xoffset_]
		set rety [expr $maxy_ + $yoffset_]
		
		# Update nexty_.
		set nexty_ $maxy_
	    }
	    
	    # If there is no room for the bottom of the window in this
	    # row, go back to the top-left corner. Increase the base
	    # coordinates so we don't cover up the previous windows.
	    if {[expr $rety + $height] >= $screen_height_} {
		set xbase_ [expr $xbase_ + $xincr_]
		set ybase_ [expr $ybase_ + $yincr_]
		set retx [expr $xbase_ + $xoffset_]
		set rety [expr $ybase_ + $yoffset_]
		
			# Update nexty_ and maxy_.
		set nexty_ $ybase_
		set maxy_ $ybase_
	    }
	} else {
	    # If we're not crossing monitors, place on one monitor after
	    # another, looping back to the first monitor if all monitors
	    # have been fully covered. If we're starting on the first
	    # monitor, then there is no need to loop later (set looped to
	    # true).
	    set placed 0
	    set looped [expr $mon_ == 0]
	    while {!$placed} {
		# If the current monitor index is less than zero or larger
		# than the number of monitors, reset it to zero.
		if {$mon_ < 0 || $mon_ >= [llength $mon_list_]} {
		    set mon_ 0
		}
		
		# Get the virtual space coordinates for the current
		# monitor.
		set v_coords [lindex $mon_list_ $mon_]
		set minx [lindex $v_coords 0]
		set miny [lindex $v_coords 1]
		set maxx [lindex $v_coords 2]
		set maxy [lindex $v_coords 3]
		
		# Assume that we've placed the window. If it doesn't fit
		# on this monitor, then we know it can't be placed.
		set placed 1
		
		# If we're on the last monitor and we've looped here, then
		# just use these coordinates.
		if {[expr $mon_ + 1] == [llength $mon_list_] && $looped} {
		    break;
		}
		
		# If there is no room in the current row, try the next row.
		if {[expr $retx + $width] >= $maxx} {
		    set retx [expr $minx + $xbase_ + $xoffset_]
		    set rety [expr $maxy_ + $yoffset_]
		    
		    # Update nexty_.
		    set nexty_ $maxy_
		    
		    # Try the new coordinates.
		    set placed 0
		}
		
		# If there is no room for the bottom of the window in this
		# row, go to the top-left corner of the next monitor.
		if {[expr $rety + $height] >= $maxy} {
		    # If the next monitor would be the first monitor
		    # (looping around) and we have not already looped
		    # around, increase the base coordinates so we don't
		    # cover up the previous windows.
		    incr mon_
		    if {$mon_ == [llength $mon_list_]} {
			set mon_ 0
			set xbase_ [expr $xbase_ + $xincr_]
			set ybase_ [expr $ybase_ + $yincr_]
			set looped 1
		    }
		    
		    # Set the window coordinates to the top-left corner
		    # of the next monitor.
		    set v_coords [lindex $mon_list_ $mon_]
		    set minx [lindex $v_coords 0]
		    set miny [lindex $v_coords 1]
		    set retx [expr $minx + $xbase_ + $xoffset_]
		    set rety [expr $miny + $ybase_ + $yoffset_]
		    
		    # Update nexty_ and maxy_.
		    set nexty_ [expr $miny + $ybase_]
		    set maxy_ [expr $miny + $ybase_]
		    
		    # Try the new coordinates.
		    set placed 0
		}
	    }
	}
    
	# Update nextx_ and maxy_.
	set nextx_ [expr $retx + $width]
	if {[expr $rety + $height] > $maxy_} {
	    set maxy_ [expr $rety + $height]
	}
	
	# Store and return this window's coordinates.
	set asources_($as) "$retx $rety $scale"
	return $asources_($as)
    }

    # Handle remove commands.
    if {$cmd == "remove"} {
	array unset asources_ $as
	array unset windows_ $as
    }

    # Handle register commands.
    if {$cmd == "register" && $uw != "false"} {
	set windows_($as) $uw
    }
}
