/*
 * transducer-spoof.cc --
 *
 *      Reads from mic log file and calls slave transducer with the
 *      value from the file instead of the current value from the card.
 *
 * Copyright (c) 2000-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "transducer-spoof.h"
#include "tclcl.h"
#include "transducer-log.h"
#include "qm-files.h"

#include <stdlib.h> // for atoi

static class SpoofTransducerClass : public TclClass {
public:
	SpoofTransducerClass() : TclClass("Transducer/Spoof") {}
	TclObject* create(int argc, const char*const* argv) {
		return(new SpoofTransducer(argc, argv));
	}
} spoof_transducer_class;

SpoofTransducer::SpoofTransducer(int argc, const char* const* argv)
{
  struct MicInputHeader hdr;
  // args passed in tcl start at argv[4]

  enabled = 0;
  index = 0;
  maxIndex = 0;

  // get the filename from argument 5
  infile = NULL;
  if(argc > 4)
  {
    sprintf(filename, "%s", argv[4]);
    infile = fopen(argv[4], "r");
    if(infile == NULL)
    {
      fprintf(stderr, "SpoofTransducer::SpoofTransducer: file %s could not be opened for reading\n", argv[4]);
    }
    else
    {
      fread(&hdr, sizeof(struct MicInputHeader), 1, infile);
      maxIndex = hdr.numSamples - 1;
    }
  }
  else
  {
    fprintf(stderr, "SpoofTransducer::SpoofTransducer: no filename specified; will not spoof\n");
  }
}

SpoofTransducer::~SpoofTransducer()
{
  if(infile != NULL)
  {
    fflush(infile);
    fclose(infile);
  }
}

// by experimentation, this gets called every ~60 ms => ~16.6667 samples/sec
//    => 50 samples every 3 seconds
//
// by experimentation, it also appears that this "double" is always of the
//    form 60.00000, 32.0000, etc, i.e., converting it to an int loses no
//    precision
//
// for logging, 50 samples/3 seconds => 1000 samples/minute
//    1000 samples/min * 5 bytes/sample (one for \n) = 5000 bytes/min
//    5000 bytes/min = 300,000 bytes/hour
//
void SpoofTransducer::set(double d)
{
  int result;
  struct MicInputInfo info;

  if(enabled)
  {
    if(infile != NULL)
    {
      result = fread(&info, sizeof(struct MicInputInfo), 1, infile);
      index++;
      if(result <= 0)
      {
	// we couldn't read from the file, perhaps we ran out of input?
	//
	// at this point, we don't do anything and just pass on the parameter
	//    to the slave transducer
//	fprintf(stderr, "SpoofTransducer::set: unable to read from file %s, index = %d\n", filename, index);
      }
      else
      {
	d = (double)info.val;
      }
    }
  }

  if(slaveTransducer_ != NULL)
  {
    slaveTransducer_->set(d);
  }
}

void SpoofTransducer::attachTransducer(const char* trans)
{
  slaveTransducer_ = (Transducer*)TclObject::lookup(trans);
}

int SpoofTransducer::setIndex(int newIndex)
{
  if(infile == NULL)
  {
    return -1;
  }
  if(newIndex > maxIndex)
  {
    return(-1);
  }

  if(index != newIndex)
  {
    index = newIndex;
    // seek to be ready to read from this index in the file the next
    //   time set() is called
    int offset = sizeof(struct MicInputHeader) + (index * sizeof(struct MicInputInfo));
    fseek(infile, offset, SEEK_SET);
    // file position indicator is now at the start of the record corresponding
    //    to the specified index
  }
  return(1);
}

int SpoofTransducer::command(int argc, const char* const* argv)
{
  const char* cmd = argv[1];
  Tcl& tcl = Tcl::instance();

  if(argc == 2)
  {
    if(!strcmp(cmd, "enableSpoofing"))
    {
      enabled = 1;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "disableSpoofing"))
    {
      enabled = 0;
      return(TCL_OK);
    }
    if(!strcmp(cmd, "getIndex"))
    {
      tcl.resultf("%d", index);
      return(TCL_OK);
    }
  }
  if(argc == 3)
  {
    // use this to manually call set() from Tcl
    if(!strcmp(cmd, "setCall"))
    {
      int val = atoi(argv[2]);
      set((double)val);
      return(TCL_OK);
    }
    if(!strcmp(cmd, "setIndex"))
    {
      int val = atoi(argv[2]);
      setIndex(val);
      return(TCL_OK);
    }
    if(!strcmp(cmd, "attachTransducer"))
    {
      attachTransducer(argv[2]);
      return(TCL_OK);
    }
  }

  // default to TclObject
  return(TclObject::command(argc, argv));
}

