/*
 * Copyright (c) 1997 Carnegie Mellon University. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation is hereby granted (including for commercial or
 * for-profit use), provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative
 * works, or modified versions, and any portions thereof, and that
 * both notices appear in supporting documentation, and that credit
 * is given to Carnegie Mellon University in all publications reporting
 * on direct or indirect use of this code or its derivatives.
 *
 * THIS SOFTWARE IS EXPERIMENTAL AND IS KNOWN TO HAVE BUGS, SOME OF
 * WHICH MAY HAVE SERIOUS CONSEQUENCES.  CARNEGIE MELLON PROVIDES THIS
 * SOFTWARE IN ITS ``AS IS'' CONDITION, AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * Carnegie Mellon encourages (but does not require) users of this
 * software to return any improvements or extensions that they make,
 * and to grant Carnegie Mellon the rights to redistribute these
 * changes without encumbrance.
 */

#include <stdio.h>

#ifdef BSD_KERNEL
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#else
#include <stdlib.h>
#endif

#include "hfsc_err.h"
#include "hfsc_def.h"
#include "hfsc_decl.h"
#include "hfsc_fun.h"

void restoreHeap(Heap *, u_int);
void checkHeap(Heap *);
/*
#define HEAP_DEBUG
*/

/****************************************************************
 *
 *  NAME: newHeap
 *
 *  DESCRIPTION:
 *    Create new heap
 *
 *  PARAMETERS:
 *
 *  RETURN:
 *    the new heap
 *
 ************************************************************************/

Heap *newHeap()
{
  Heap     *ph;

#ifdef BSD_KERNEL
  if (!(ph = (Heap *)malloc(sizeof(Heap), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(ph, sizeof(Heap));
#else
  if (!(ph = (Heap *)calloc(1, sizeof(Heap))))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif

  /* this is the first item */
#ifdef BSD_KERNEL
  if (!(ph->a = 
       (ReqItem **)malloc(DEF_HEAP_SIZE*sizeof(ReqItem *), M_TEMP, M_NOWAIT)))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
  bzero(ph->a, DEF_HEAP_SIZE*sizeof(ReqItem *));
#else
  if (!(ph->a = (ReqItem **)calloc(DEF_HEAP_SIZE, sizeof(ReqItem *))))
    hfsc_panic(ErrMsg[ERR_MEM_ALLOC]);
#endif
  ph->size = DEF_HEAP_SIZE;

  return ph;
}


/****************************************************************
 *
 *  NAME: enqueueReq
 *
 *  DESCRIPTION:
 *    Add a new item to the heap
 *
 *  PARAMETERS:
 *    ph    - the heap
 *    pitem - the new item to be added
 *
 *  RETURN:
 *    the head of the list
 *
 ************************************************************************/

void enqueueReq(ph, pReq)
Heap    *ph;
ReqItem *pReq;
{
#ifdef BSD_KERNEL
  ReqItem **temp;
  int       i;
#endif

#ifdef REQ_DEBUG
  if (pReq->status == IN_HEAP)
    hfsc_panic("Error in enqueueReq !\n");
  pReq->status = IN_HEAP;
#endif  

  if (ph->num == ph->size) {
    /* the heap is full; double the allocated space */
    ph->size = ph->size << 1;
#ifdef BSD_KERNEL
    if (!(temp = 
      (ReqItem **)malloc(ph->size*sizeof(ReqItem*), M_TEMP, M_NOWAIT)))
      hfsc_panic(ErrMsg[ERR_MEM_REALLOC]);
    bzero(temp, ph->size*sizeof(ReqItem*));
    for (i = 0; i < ph->size/2; i++)
      temp[i] = ph->a[i];
    free((caddr_t)ph->a, M_TEMP);
    ph->a = temp;
#else
    if (!(ph->a = (ReqItem **)realloc(ph->a, ph->size*sizeof(ReqItem*))))
      hfsc_panic(ErrMsg[ERR_MEM_REALLOC]);
#endif
  }

  ph->a[ph->num] = pReq;
  pReq->idx = ph->num;
  (ph->num)++;

  /* restore the heap */
  restoreHeap(ph, ph->num - 1);

#ifdef HEAP_DEBUG
  checkHeap(ph);
#endif
}


/****************************************************************
 *
 *  NAME: dequeueReq
 *
 *  DESCRIPTION:
 *    Get the item with the lowest key (and remove it from the heap)
 *
 *  PARAMETERS:
 *    ph    - the heap
 *
 *  RETURN:
 *    the minimum item from the heap
 *
 ************************************************************************/

ReqItem *dequeueReq(ph) 
Heap     *ph;
{
  register ReqItem **reqVec = ph->a;
  ReqItem *pReq;

  if (!ph->num) {
    return NULL;
  }

  pReq = reqVec[0];
#ifdef REQ_DEBUG
  pReq->status = INACTIVE;
#endif

  ph->num--;
  if (!ph->num) {
    return pReq;
  }

  reqVec[0] = reqVec[ph->num];
  reqVec[0]->idx = 0;

  /* restore the heap */
  restoreHeap(ph, 0);

#ifdef HEAP_DEBUG
  checkHeap(ph);
#endif

  return pReq;
}


/****************************************************************
 *
 *  NAME: changeReq
 *
 *  DESCRIPTION:
 *    Update the request's key
 *
 *  PARAMETERS:
 *    ph    - the heap
 *
 *  RETURN:
 *
 ************************************************************************/

void changeReq(ph, idx, newKey)  
  Heap  *ph;
  u_int  idx;
  key_t  newKey;
{
  ph->a[idx]->d = newKey;

  /* restore the heap */
  restoreHeap(ph, idx);

#ifdef HEAP_DEBUG
  checkHeap(ph);
#endif
}


/****************************************************************
 *
 *  NAME: removeReq
 *
 *  DESCRIPTION:
 *    Delete a specified request from heap
 *
 *  PARAMETERS:
 *    ph  - the heap
 *    idx - index of the item to be deleted
 *
 *  RETURN:
 *    the minimum item from the heap
 *
 ************************************************************************/

void removeReq(ph, idx)
  Heap  *ph; 
  u_int idx;
{
  register ReqItem **reqVec = ph->a;

  if (!ph->num) {
    printf("ERROR: empty heap!\n");
    return;
  }

  ph->num--;
#ifdef REQ_DEBUG
  reqVec[idx]->status = INACTIVE;
#endif

  if (ph->num) {
    reqVec[idx] = reqVec[ph->num];
    reqVec[idx]->idx = idx;
  }

  /* restore the heap */
  restoreHeap(ph, idx);

#ifdef HEAP_DEBUG
  checkHeap(ph);
#endif

}


/****************************************************************
 *
 *  NAME: restoreHeap
 *
 *  DESCRIPTION:
 *    Push up or down the given item in order to restore the heap
 *
 *  PARAMETERS:
 *    ph  - the heap
 *    idx - the heap item
 *
 *  RETURN:
 *
 ************************************************************************/

void restoreHeap(ph, idx)
  Heap     *ph;
  u_int     idx; 
{
  register ReqItem **reqVec = ph->a;
  register u_int    idx1;

  if (ph->num < 2)
    return; /* nothing to do */

  if (idx && reqVec[idx]->d < reqVec[idx1 = ((idx - 1) >> 1)]->d) {
    do {
      SWAP_HEAP_ITEMS(reqVec, idx, idx1);
      idx = idx1;
    } while (idx && reqVec[idx]->d < reqVec[idx1 = ((idx - 1) >> 1)]->d);
    return;
  }
  
  while ((idx1 = (idx << 1) + 1) < ph->num) {
    if (idx1 == ph->num - 1) {
       /* no left children */
       if (reqVec[idx]->d > reqVec[idx1]->d) {
         SWAP_HEAP_ITEMS(reqVec, idx, idx1);
         return;
       }
       idx = idx1 + 1;
     } else { /* curent item has two children */
       if ((reqVec[idx]->d <= reqVec[idx1]->d) && 
         (reqVec[idx]->d <= reqVec[idx1 + 1]->d)) 
         return;
         
       if (reqVec[idx1]->d < reqVec[idx1 + 1]->d) {
         SWAP_HEAP_ITEMS(reqVec, idx, idx1);
         idx = idx1;
       } else {
         idx1++;
         SWAP_HEAP_ITEMS(reqVec, idx, idx1);
         idx = idx1;
       }
     }
  }
}



void checkHeap(ph)
  Heap     *ph;
{
  ReqItem **reqVec = ph->a;
  u_int    idx, flag = 0; 

  /* check for integrity */
  for (idx = 0; idx < ph->num/2; idx++) {
    if (2*idx + 1 < ph->num && (reqVec[idx])->d > (reqVec[2*idx + 1])->d)
      flag = 1;
    if (2*idx + 2 < ph->num && (reqVec[idx])->d > (reqVec[2*idx + 2])->d)
      flag = 1;
    if (reqVec[idx]->idx != idx)
      flag = 1;
  }

  if (flag == 1) {
    for (idx = 0; idx < ph->num; idx++) 
      printf("| %d ", (reqVec[idx])->d);
    printf("\n");
    hfsc_panic("Heap Error !\n");
  }
}


