#!/bin/sh
#
#  Shell script to build and install the zfcpdump ramdisk
#
#  > create_rd.sh (ARCH) : build ramdisk
#  > create_rd.sh -r     : cleanup
#  > create_rd.sh -i     : install files
#
#  Copyright (C) 2003 IBM Deutschland Entwicklung GmbH, IBM Corporation
#  Author(s): Michael Holzheu <holzheu@de.ibm.com>
#

. ../config

RD_FREE_SIZE=200
BBOX_VERSION=0.60.5
BBOX="../extern/$BBOX_TARBALL"
BBOX_OPTIONS="DOSTATIC=true"
E2FS_PROGS="../extern/$E2FS_PROGS_TARBALL"

RD_TMP=rd
RD_IMAGE=ramdisk.dump
DIR_LIST="bin \
	  sbin \
          usr \
          dev \
          proc \
          etc \
          mnt"

#
# check() function to check error codes
#
function check ()
{
        if [ $? != 0 ]
        then
                echo "failed"
                exit 1
        else
                echo "ok"
        fi
}

#
# build(): build the ramdisk
#

function build ()
{
	echo "build: `date`"

	echo "==============================================================="
	echo "Removing old installation"
	echo "==============================================================="
	rm -rf $RD_TMP
	echo "ok"

	echo "==============================================================="
	echo "Creating RD dir structure"
	echo "==============================================================="
	mkdir $RD_TMP 
	for i in $DIR_LIST
	do
		mkdir "$RD_TMP/$i"
	done
	check
	
	#
	# Build BusyBox
	#

	echo "==============================================================="	
	echo "Extracting BBox"
	echo "==============================================================="
	tar xfjv $BBOX
	check

	echo "==============================================================="
	echo "Compiling BBox"
	echo "==============================================================="
	cp BBOX_Config.h "busybox-$BBOX_VERSION/Config.h"
	cd "busybox-$BBOX_VERSION"
	make $BBOX_OPTIONS
	check
	echo "==============================================================="
	echo "Installing BBox"
	echo "==============================================================="
	make PREFIX=../$RD_TMP install
	check
	cd ..
	
	#
	# Build e2fsprogs
	#

	echo "==============================================================="
	echo "Extracting E2FSPROGS"
	echo "==============================================================="
	tar xfzv $E2FS_PROGS
	check

	echo "==============================================================="
	echo "Compiling E2FSPROGS"
	echo "==============================================================="
	cd e2fsprogs-$E2FS_PROGS_VERSION
	patch -p1 < ../patches/e2fsprogs-1.32.gcc3.4.diff
	./configure
	make
	check
	echo "==============================================================="
	echo "Installing e2fsck"
	echo "==============================================================="
	cp e2fsck/e2fsck ../$RD_TMP/sbin
	check
	cd ..

	#
	# copy additional files
	#

	cp linuxrc $RD_TMP 
	cp zfcp_dumper $RD_TMP 
	ln -s /proc/mounts "$RD_TMP/etc/mtab"

	#
	# create ramdisk
	#

	echo "==============================================================="
	echo "Creating ramdisk ($RD_FS)"
	echo "==============================================================="

	if [ "$RD_FS" == "romfs" ]
	then
		printf "%-30s: " "Creating romfs"
		genromfs -f $ZFCPDUMP_RD -d $RD_TMP
		check
		printf "%-30s: " "zip romfs"
		gzip $ZFCPDUMP_RD
		check
		mv "$ZFCPDUMP_RD.gz" $ZFCPDUMP_RD
	elif [ "$RD_FS" == "ext2" ]
	then
		RD_MIN_SIZE=`du -k -s $RD_TMP | awk '{print $1}'`
		RD_SIZE=`expr $RD_MIN_SIZE + $RD_FREE_SIZE`

		printf "creating rd with size $RD_SIZE KB (used $RD_MIN_SIZE):"
		dd if=/dev/zero of=$RD_IMAGE bs=1k count=$RD_SIZE 
		check

		printf "%-30s: " "Creating ext2"
		yes | mke2fs $RD_IMAGE
		check
		printf "%-30s: " "Mounting RD"
		if [ ! -d mnt ]
		then
			mkdir mnt
		fi
		mount $RD_IMAGE "`pwd`/mnt" -o loop
		check

		printf "%-30s: " "Copy RD"
		cp -R $RD_TMP/* mnt
		if [ $? != 0 ]
		then
			umount mnt
			exit
		fi
		echo "ok"

		printf "%-30s: " "Umount RD"
		umount mnt
		check

		printf "%-30s: " "packing RD"
		gzip $RD_IMAGE
		check
		mv "$RD_IMAGE.gz" $ZFCPDUMP_RD
		check
	else
		echo "ERROR: Invalid ramdisk filesystem '$RD_FS'"
		exit
	fi
	echo "=============================================================="
	echo "SUCCESS: built ramdisk '$ZFCPDUMP_RD'"
	echo "=============================================================="
}

#
# cleanup(): function to remove build files
#
function cleanup()
{
	rm -rf $RD_TMP
	rm -rf busybox*
	rm -rf e2fs*
	rm -f $RD_IMAGE
	rm -f "$RD_IMAGE.gz"
        rm -f $ZFCPDUMP_RD
}
 
#
# install(): function to install zfcpdump kernel
#
function install()
{
        cp $ZFCPDUMP_RD "$INSTROOT/$ZFCPDUMP_DIR"; chmod 644 "$INSTROOT/$ZFCPDUMP_DIR/$ZFCPDUMP_RD"
}

#
# main
#
 
if [ "$1" == "-r" ]
then
        cleanup
elif [ "$1" == "-i" ]
then
        install
else
        ARCH="$1"
        build
fi
