
// Copyright (c) 1996-2001 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES
// SUFFERED BY LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING
// OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found
// in the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Krishnan Subramani 
//          Umesh Kumar V. Rajasekaran
//          Malolan Chetlur    
//          Narayanan Thondugulam

//---------------------------------------------------------------------------

#include "IIRScram_ReturnStatement.hh"
#include "IIR_Identifier.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_Label.hh"
#include "IIR_TypeDefinition.hh"
#include "error_func.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "savant.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"


IIRScram_ReturnStatement::~IIRScram_ReturnStatement(){
}


void 
IIRScram_ReturnStatement::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);

  _vhdl_out << "return ";
    
  if (get_return_expression() != NULL) {
    ASSERT(get_return_expression()->_is_resolved());
    get_return_expression()->_publish_vhdl(_vhdl_out);
  }
}


void 
IIRScram_ReturnStatement::_publish_cc( published_file &_cc_out ) {
  IIR_TypeDefinition *ret_type = NULL;

  if( _get_allocate_return_value() == TRUE) {
    _publish_cc_resolution_fn_return( _cc_out );
  } 
  else {
    if (get_label() != NULL) {
      get_label()->_publish_cc_lvalue( _cc_out );
      _cc_out << ": ";
    } // if

    _cc_out << "return ";

    if (get_return_expression() != NULL) {
      ASSERT( get_return_expression()->_is_resolved() );
      ret_type = get_enclosing_subprogram()->_get_return_type();
      
      IIR_TypeDefinition *common_base_type = NULL;
      if (ret_type != get_return_expression()->get_subtype()) {
	common_base_type = ret_type;
      }
      
      if (common_base_type != NULL) {
	// Okay. An explicit type conversion is needed.
	_cc_out << ret_type->_get_cc_type_name();
	_cc_out << OS("(ObjectBase::VARIABLE");
	
	if (((ret_type->_is_array_type() == TRUE) ||
	     (ret_type->_is_record_type() == TRUE)) &&
	    (ret_type->_is_access_type() == FALSE)) {
	  ret_type->_publish_cc_object_type_info(_cc_out, TRUE, "", TRUE);
	  _cc_out << "," << NL();
	  ret_type->_publish_cc_resolution_function_id(_cc_out);
	}

	if (ret_type->_is_unconstrained_array_type() == TRUE) {
	  get_return_expression()->get_subtype()->_publish_cc_constructor_args( _cc_out );
	}
	_cc_out << "," << NL();
      }
      
      get_return_expression()->_publish_cc_lvalue(_cc_out);

      if (common_base_type != NULL) {
	if (ret_type->_is_scalar_type() == TRUE) {
	  ret_type->_publish_cc_object_type_info(_cc_out);
	}
	
	_cc_out << CS(")");
      }
      
    } 
    else {
      ASSERT( get_enclosing_subprogram()->_is_iir_procedure_declaration() == TRUE );
      _cc_out << "NORMAL_RETURN";
    }
    _cc_out << ";" << NL();
  } // else
}

void 
IIRScram_ReturnStatement::_publish_cc_resolution_fn_return( published_file &_cc_out ){
  // Open a temporary scope.
  _cc_out << OS("{") << "VHDLType *retval = new ";
  IIR_TypeDefinition *retType = get_enclosing_subprogram()->_get_return_type();
  ASSERT ( retType != NULL );

  _cc_out << retType->_get_cc_type_name();
  _cc_out << OS("(") << "ObjectBase::VARIABLE";
  
  if ((retType->_is_array_type() == TRUE) ||
      (retType->_is_record_type() == TRUE)) {
    retType->_publish_cc_object_type_info( _cc_out );
    _cc_out << "," << NL();
    retType->_publish_cc_resolution_function_id( _cc_out );
  }
  
  if(get_return_expression()->get_subtype()->_is_scalar_type() == TRUE) {
    _cc_out << ", " << NL() << "(const ";
    get_return_expression()->get_subtype()->_publish_cc_kernel_type( _cc_out );
    _cc_out << " &)";
  }
  else {
    _cc_out << "," << NL();
  }
  
  get_return_expression()->_publish_cc_lvalue( _cc_out );

  if (get_return_expression()->get_subtype()->_is_scalar_type() == TRUE) {
    get_return_expression()->get_subtype()->_publish_cc_object_type_info ( _cc_out );
  }
  
  _cc_out << CS(");")
	  << "return retval;" << NL()
	  << CS("}");
}

void 
IIRScram_ReturnStatement::_type_check(){
  // This needs to actually be type-checked within the context of the
  // return type.
  // See next method...
}

void 
IIRScram_ReturnStatement::_type_check( set<IIR_TypeDefinition> *context_set ){
  ASSERT( get_enclosing_subprogram() != NULL );

  if( get_enclosing_subprogram()->get_kind() == IIR_FUNCTION_DECLARATION  ){
    // Then this return statement appeared in a function declaration.
    // 1) The expression must not be NULL.
    // 2) If the context_set is NULL, something is broken.
    ASSERT( context_set != NULL );

    if( get_return_expression() != NULL ){
      // Then we need to type_check based on the context_set passed in...
      set_return_expression( get_return_expression()->_semantic_transform( context_set ));
      get_return_expression()->_type_check( context_set );

      set<IIR_TypeDefinition> *expression_types = get_return_expression()->_get_rval_set();
      if( expression_types == NULL ){
	report_undefined_symbol( get_return_expression() );
	return;
      }

      reconcile_sets( context_set, expression_types );
      
      switch( expression_types->num_elements() ){
      case 0:{
	ostringstream err;
	err << "Expression |" << *get_return_expression() << "| must have a return type "
	    << "compatible with that of its function, |" 
	    <<  *get_enclosing_subprogram()->get_declarator() << "|.";
	report_error( this, err.str() );
	break;
      }
      case 1:{
	IIR_TypeDefinition *my_type = expression_types->get_element() ;
	set_return_expression( get_return_expression()->_rval_to_decl( my_type ));
	break;
      }
      default:{
	report_ambiguous_error( get_return_expression(), expression_types );
	break;
      }
      }

      delete expression_types;
    }
    else{
      ostringstream err;
      err << "|" << *get_enclosing_subprogram()->get_declarator() << "| " 
	  << "was declared as a function, and should return a value.";
      report_error( this, err.str(), WARNING );
    }
  }
  else{
    // Then this return statement appeared in a procedure declaration.
    // 1) The expression must be NULL.
    ASSERT( get_enclosing_subprogram()->get_kind() == IIR_PROCEDURE_DECLARATION );

    if( get_return_expression() != NULL ){
      ostringstream err;
      err << "|" << get_enclosing_subprogram()->get_declarator() << "| " 
	  << "was declared as a procedure, and may not return a value.";
      report_error( this, err.str() );
    }
  }
}

IIR_Boolean 
IIRScram_ReturnStatement::_is_resolved(){
  ASSERT( get_enclosing_subprogram() != NULL );

  if( get_return_expression() != NULL ){
    return get_return_expression()->_is_resolved();
  }
  else {
    return TRUE;
  }
}

visitor_return_type *
IIRScram_ReturnStatement::_accept_visitor( node_visitor *visitor, 
					   visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ReturnStatement(this, arg);
}
