/*
 * Copyright (c) 2003-2005 Erez Zadok
 * Copyright (c) 2003-2005 Charles P. Wright
 * Copyright (c) 2003-2005 Mohammad Nayyer Zubair
 * Copyright (c) 2003-2005 Puja Gupta
 * Copyright (c) 2003-2005 Harikesavan Krishnan
 * Copyright (c) 2003-2005 Stony Brook University
 * Copyright (c) 2003-2005 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */
/*
 *  $Id: inode.c,v 1.189 2005/03/02 13:30:13 cwright Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include "fist.h"
#include "unionfs.h"


STATIC int
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
unionfs_create(inode_t *parent, dentry_t *dentry, int mode)
#else
unionfs_create(inode_t *parent, dentry_t *dentry, int mode, struct nameidata *nd)
#endif
{
    int err = 0;
    dentry_t *hidden_dentry = NULL;
    dentry_t *whiteout_dentry = NULL;
    dentry_t *new_hidden_dentry;
    dentry_t *hidden_parent_dentry = NULL;
    int bindex = 0, bstart;
    char *name = NULL;

    print_entry_location();
    fist_print_dentry("IN unionfs_create :", dentry);

    /* We start out in the leftmost branch. */
    bstart = dbstart(dentry);
    hidden_dentry = dtohd(dentry);

    /* check if whiteout exists in this branch, i.e. lookup .wh.foo first */
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
	err = -ENOMEM;
	goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[4 + dentry->d_name.len] = '\0';

    whiteout_dentry = lookup_one_len(name, hidden_dentry->d_parent, dentry->d_name.len + 4);
    if (IS_ERR(whiteout_dentry)) {
	err = PTR_ERR(whiteout_dentry);
	goto out;
    }
    PASSERT(whiteout_dentry);

    if (whiteout_dentry->d_inode) {
	/* .wh.foo has been found. */
	/* First truncate it and then rename it to foo (hence having the same
	 * overall effect as a normal create.
	 *
	 * XXX: This is not strictly correct.  If we have unlinked the file and
	 * XXX: it still has a reference count, then we should actually unlink
	 * XXX: the whiteout so that user's data isn't hosed over.
	 *
	 */
	dentry_t *hidden_dir_dentry;
	struct iattr newattrs;

	down(&whiteout_dentry->d_inode->i_sem);
	newattrs.ia_valid = ATTR_CTIME;
        if (whiteout_dentry->d_inode->i_size != 0) {
	    newattrs.ia_valid |= ATTR_SIZE;
	    newattrs.ia_size = 0;
        }
	err = notify_change(whiteout_dentry, &newattrs);
	up(&whiteout_dentry->d_inode->i_sem);

        new_hidden_dentry = dtohd(dentry);
	dget(new_hidden_dentry);

	hidden_dir_dentry = get_parent(whiteout_dentry);
	/* They are in the same directory, but we need to get it twice. */
	ASSERT(get_parent(whiteout_dentry) == hidden_dir_dentry);
	double_lock(hidden_dir_dentry, hidden_dir_dentry);

        if (!(err = is_robranch_super(dentry->d_sb, bstart))) {
            err = vfs_rename(hidden_dir_dentry->d_inode, whiteout_dentry,
                             hidden_dir_dentry->d_inode, new_hidden_dentry);
	}
	if (!err) {
        	fist_copy_attr_timesizes(parent, new_hidden_dentry->d_parent->d_inode);
        	parent->i_nlink = get_nlinks(parent);
	}

	/* This will dput our double gotten parent. */
	double_unlock(hidden_dir_dentry, hidden_dir_dentry);

	dput(new_hidden_dentry);

	if (err) {
            if (IS_SET(parent->i_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* exit if the error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                goto out;
            }
            /* We were not able to create the file in this branch,
             * so, we try to create it in one branch to left
             */
	    bstart--;
	} else {
	    /* reset the unionfs dentry to point to the .wh.foo entry. */

	    /* Discard any old reference. */
            PASSERT(dtohd(dentry));
            dput(dtohd(dentry));

	    /* Trade one reference to another. */
	    set_dtohd_index(dentry, bstart, whiteout_dentry);
	    whiteout_dentry = NULL;

            err = unionfs_interpose(dentry, parent->i_sb, 0);
            goto out;
	}
    }

    for (bindex = bstart; bindex >= 0; bindex--) {
	hidden_dentry = dtohd_index(dentry, bindex);
	if (!hidden_dentry) {
	    /* if hidden_dentry is NULL, create the entire
	     * dentry directory structure in branch 'bindex'. hidden_dentry will NOT be null when
	     * bindex == bstart because lookup passed as a negative unionfs dentry pointing to a
	     * lone negative underlying dentry */
	    hidden_dentry = unionfs_create_dirs(parent, dentry, bindex);
	    if (!hidden_dentry || IS_ERR(hidden_dentry)) {
		if (IS_ERR(hidden_dentry)) {
		    err = PTR_ERR(hidden_dentry);
		}
		continue;
	    }
	}

        PASSERT(hidden_dentry);
	fist_checkinode(parent, "unionfs_create");

	hidden_parent_dentry = lock_parent(hidden_dentry);
	if (IS_ERR(hidden_parent_dentry)) {
	    err = PTR_ERR(hidden_parent_dentry);
	    goto out;
	}
	/* We shouldn't create things in a read-only branch. */
	if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
	    //DQ: vfs_create has a different prototype in 2.6
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	    err = vfs_create(hidden_parent_dentry->d_inode, hidden_dentry, mode);
#else
	    err = vfs_create(hidden_parent_dentry->d_inode, hidden_dentry, mode, nd);
#endif

	}
	/* XXX this could potentially return a negative hidden_dentry! */
        if (err || !hidden_dentry->d_inode) {
            unlock_dir(hidden_parent_dentry);

            if (IS_SET(parent->i_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* break out of for loop if error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                break;
            }
        } else {
            err = unionfs_interpose(dentry, parent->i_sb, 0);
            if (!err) {
                fist_copy_attr_timesizes(parent, hidden_parent_dentry->d_inode);
                /* update number of links on parent directory */
                parent->i_nlink = get_nlinks(parent);
            }
            unlock_dir(hidden_parent_dentry);
            break;
        }
    } // end for

 out:
    if (whiteout_dentry) {
	fist_dprint(8, "whiteout_dentry: %p\n", whiteout_dentry);
	dput(whiteout_dentry);
    }
    if (name) {
	fist_dprint(8, "name: %p\n", name);
	KFREE(name);
    }

    fist_print_dentry("OUT unionfs_create :", dentry);
    print_exit_status(err);
    return err;
}

dentry_t * unionfs_lookup_backend(inode_t *parent, dentry_t *dentry, int interpose_flag)
{
    int err = 0;
    dentry_t *hidden_dentry = NULL;
    dentry_t *wh_hidden_dentry = NULL;
    dentry_t *hidden_dir_dentry = NULL;
    dentry_t *parent_dentry;
    unsigned int namelen;
    int bindex, bstart, bend;
    int dentry_count;
    int is_first_negative = 0;
    int first_dentry_offset = -1;
    dentry_t *first_hidden_dentry;
    char *name = NULL;

    print_entry_location();
    PASSERT(dentry);
    /* No dentries should get created for possible whiteout names. */
    if (!strncmp(dentry->d_name.name, ".wh.", 4)) {
	err = -EPERM;
	goto out;
    }

    parent_dentry = dentry->d_parent;
    namelen = dentry->d_name.len;
    PASSERT(parent_dentry);
    bstart = dbstart(parent_dentry);
    bend = dbend(parent_dentry);
    ASSERT(bstart >= 0);

    fist_print_dentry("IN unionfs_lookup:", dentry);
    fist_print_inode("IN unionfs_lookup:", parent);

    /* must initialize dentry operations */
    dentry->d_op = &unionfs_dops;

    /* allocate space for hidden dentries */
    if (!dtopd_lhs(dentry)) {
    	dtopd_lhs(dentry) =  (struct unionfs_dentry_info *) KMALLOC(sizeof(struct unionfs_dentry_info), GFP_UNIONFS);
        if (!dtopd_lhs(dentry)) {
	   err = -ENOMEM;
	   goto out;
        }
    	init_dtopd(dentry, parent->i_sb);
    } else {
	reinit_dtopd(dentry, parent->i_sb);
    }
    dtohd_ptr(dentry) = KMALLOC(sizeof(dentry_t *) * sbmax(parent->i_sb), GFP_UNIONFS);
    if (!dtohd_ptr(dentry)) {
	err = -ENOMEM;
	goto out_free_dtopd;
    }
    init_dtohd_ptr(dentry, sbmax(parent->i_sb));

    /* these fields are useful when all dentries are negative */
    dentry_count = 0;      /* stores the number of positive dentries found */
    is_first_negative = 0;   /* boolean value, to indicate if first dentry was negative */
    first_hidden_dentry = NULL;  /* pointer to first dentry, either positive or negative */

    for (bindex = bstart; bindex <= bend; bindex++) {
	hidden_dir_dentry = dtohd_index(parent_dentry, bindex);

	/* if the parent hidden dentry doesn't exists skip this */
	if (!(hidden_dir_dentry && hidden_dir_dentry->d_inode)) {
	    continue;
	}

	PASSERT(hidden_dir_dentry->d_inode);
	/* also skip it if it isn't a directory. */
	if (!S_ISDIR(hidden_dir_dentry->d_inode->i_mode)) {
	    continue;
	}

	/* we reuse the whiteout name for the entire function because its value doesn't change. */
	if (!name) {
	    name = KMALLOC(namelen + 5, GFP_UNIONFS);
	    if (!name) {
		err = -ENOMEM;
		goto out_free;
	    }
	    strcpy(name, ".wh.");
	    strncat(name, dentry->d_name.name, dentry->d_name.len);
	    name[4 + dentry->d_name.len] = '\0';
	}

	/* check if whiteout exists in this branch; lookup .wh.foo first */
	wh_hidden_dentry = lookup_one_len(name, hidden_dir_dentry, namelen + 4);
	if (IS_ERR(wh_hidden_dentry)) {
	    err = PTR_ERR(wh_hidden_dentry);
	    goto out_free;
	}

	if (!wh_hidden_dentry->d_inode) {
	    dput(wh_hidden_dentry);
	    wh_hidden_dentry = NULL;
	} else {
	    /* check to see if whiteout dentry is a directory */
	    if (S_ISREG(wh_hidden_dentry->d_inode->i_mode)) {
		/* We found a whiteout so lets give up. */
		dput(wh_hidden_dentry);
		break;
	    } else if (S_ISLNK(wh_hidden_dentry->d_inode->i_mode)) {
		/* This buffer should only be for an integer
		 * which means it can only be 10 characters. */
		char buf[12];
		int grayout;
		int size;

		if (!wh_hidden_dentry->d_inode->i_op || !wh_hidden_dentry->d_inode->i_op->readlink) {
			err = -ENOSYS;
			dput(wh_hidden_dentry);
			goto out_free;
		}

    		size = wh_hidden_dentry->d_inode->i_op->readlink(wh_hidden_dentry, buf, sizeof(buf));
		dput(wh_hidden_dentry);
		if (size < 0) {
			err = size;
			goto out_free;
		}

		if (size <= 10) {
			char *end;
			buf[err] = '\0';

			grayout = simple_strtoul(buf, &end, 0);

			if (*end) {
				printk("<0>Improperly formatted grayout.\n");
				err = -EIO;
			}
		} else {
			printk("<0>Improperly formatted grayout.\n");
			err = -EIO;
			goto out_free;
		}

		bindex += grayout;
		continue;
	    } else {
		err = -EIO;
		printk("<0>EIO: A whiteout entry is an invalid unknown type (%d)!\n", wh_hidden_dentry->d_inode->i_mode);
		dput(wh_hidden_dentry);
		goto out_free;
	    }
	}

	/* Now do regular lookup; lookup foo */
	hidden_dentry = lookup_one_len(dentry->d_name.name, hidden_dir_dentry, dentry->d_name.len);
	if (IS_ERR(hidden_dentry)) {
	    err = PTR_ERR(hidden_dentry);
	    goto out_free;
	}

	if (hidden_dentry->d_inode) {
	    /* number of positive dentries */
	    dentry_count++;

	    /* store underlying dentry */
	    set_dtohd_index(dentry, bindex,  hidden_dentry);
	    if (dbstart(dentry) == -1) {
		set_dbstart(dentry, bindex);
	    }
	    set_dbend(dentry, bindex);

	    /* update parent directory's atime with the bindex */
	    fist_copy_attr_atime(parent, hidden_dir_dentry->d_inode);

	    /* if the first hidden dentry is a file, stop the lookup now */
	    if (!S_ISDIR(hidden_dentry->d_inode->i_mode) && (dentry_count == 1)) {
		goto out_interpose;
	    }
	} else {
	    /* Store the first negative dentry specially, because if they are all negative
	     * we need this for future creates. */
	    if (!is_first_negative && (dbstart(dentry) == -1)) {
		/* entering only if its _first_ ever negative dentry */
		first_hidden_dentry = hidden_dentry;
		first_dentry_offset = bindex;
		is_first_negative = 1;
	    } else {
		dput(hidden_dentry);
	    }

	    /* If we've only got negative dentries, then lets use the leftmost one. */
	    if (is_first_negative && (bindex == bend) && (dentry_count == 0)) {
		set_dtohd_index(dentry, first_dentry_offset, first_hidden_dentry);
		set_dbstart(dentry, first_dentry_offset);
		set_dbend(dentry, first_dentry_offset);
		if (interpose_flag == INTERPOSE_REVAL) {
			if (dentry->d_inode) {
				PASSERT(itopd(dentry->d_inode));
	    			itopd(dentry->d_inode)->uii_stale = 1;
			}
		} else if (interpose_flag == INTERPOSE_REVAL_NEG) {
			/* We don't need to do anything. */
			ASSERT(dentry->d_inode == NULL);
		} else {
			d_add(dentry, NULL);
		}
		goto out;
	    }
	} // else negative dentry
    } // end for

out_interpose:
    /* If we're holding onto the first negative dentry we can throw it out now. */
    if (is_first_negative) {
	dput(first_hidden_dentry);
    }

    /* If we have found a positive dentry. */
    if (dentry_count) {
	err = unionfs_interpose(dentry, parent->i_sb, interpose_flag);
	if (err)
	    goto out_drop;

	fist_checkinode(dentry->d_inode, "unionfs_lookup OUT: dentry->d_inode:");
	fist_checkinode(parent, "unionfs_lookup OUT: dir");
    } else {
	if (interpose_flag == INTERPOSE_REVAL) {
	    PASSERT(dentry);
	    PASSERT(dentry->d_inode);
	    PASSERT(itopd(dentry->d_inode));
	    itopd(dentry->d_inode)->uii_stale = 1;
	} else {
	    /* If no positive dentries were found, and we didn't go to out because of the
	     * negative entry path, then we must have found a whiteout */
	    hidden_dentry = lookup_one_len(dentry->d_name.name, hidden_dir_dentry, dentry->d_name.len);
	    if (IS_ERR(hidden_dentry)) {
		err = PTR_ERR(hidden_dentry);
		goto out;
	    }

	    set_dtohd_index(dentry, bindex, hidden_dentry);
	    set_dbstart(dentry, bindex);
	    set_dbend(dentry, bindex);
	    /* We don't want to hash our dentry if it is already hashed. */
	    if (interpose_flag != INTERPOSE_REVAL_NEG) {
	    	d_add(dentry, NULL);
	    }
	}
    }
    ASSERT(dbstart(dentry) >= 0);
    goto out;

out_drop:
    d_drop(dentry);		/* so that our bad dentry will get destroyed */

out_free:
    /* should dput all the underlying dentries on error condition */
    bstart = dbstart(dentry);
    if (bstart >= 0) {
    	bend = dbend(dentry);
    	for (bindex = bstart; bindex <= bend; bindex++) {
		hidden_dentry = dtohd_index(dentry, bindex);
		if (hidden_dentry) {
	    		dput(hidden_dentry);
		}
    	}
    }
    KFREE(dtohd_ptr(dentry));
    dtohd_ptr(dentry) = NULL;

out_free_dtopd:
    KFREE(dtopd(dentry));
    dtopd_lhs(dentry) = NULL;	/* be safe */

out:
    if (dtopd(dentry)) {
	ASSERT(dbend(dentry) <= dtopd(dentry)->udi_bcount);
	ASSERT(dbend(dentry) <= sbmax(dentry->d_sb));
	ASSERT(dbstart(dentry) >= 0);
    }
    if (name) {
	KFREE(name);
    }
    fist_print_dentry("OUT unionfs_lookup:", dentry);
    fist_print_inode("OUT unionfs_lookup:", parent);
    print_exit_status(err);
    return ERR_PTR(err);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
dentry_t * unionfs_lookup(inode_t *parent, dentry_t *dentry)
#else
dentry_t * unionfs_lookup(inode_t *parent, dentry_t *dentry, struct nameidata* nd)
#endif
{
	return unionfs_lookup_backend(parent, dentry, INTERPOSE_LOOKUP);
}


STATIC int
unionfs_link(struct dentry *old_dentry, struct inode *dir, struct dentry *new_dentry)
{
    int err = 0;
    struct dentry *hidden_old_dentry = NULL;
    struct dentry *hidden_new_dentry = NULL;
    struct dentry *hidden_dir_dentry = NULL;
    struct dentry *whiteout_dentry;
    char *name = NULL;

    print_entry_location();

    hidden_new_dentry = dtohd(new_dentry);

    /* check if whiteout exists in the branch of new dentry, i.e. lookup .wh.foo first. If present, delete it */
    name = KMALLOC(sizeof(char) * (new_dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
	err = -ENOMEM;
	goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, new_dentry->d_name.name, new_dentry->d_name.len);
    name[4 + new_dentry->d_name.len] = '\0';

    whiteout_dentry = lookup_one_len(name, hidden_new_dentry->d_parent, new_dentry->d_name.len + 4);
    if (IS_ERR(whiteout_dentry)) {
	err = PTR_ERR(whiteout_dentry);
	goto out;
    }

    PASSERT(whiteout_dentry);

    if (!whiteout_dentry->d_inode) {
	dput(whiteout_dentry);
	whiteout_dentry = NULL;
    } else {
	/* found a .wh.foo entry, unlink it and then call vfs_link() */
	hidden_dir_dentry = lock_parent(whiteout_dentry);
	if (!(err = is_robranch_super(new_dentry->d_sb,dbstart(new_dentry)))) {
	    err = vfs_unlink(hidden_dir_dentry->d_inode, whiteout_dentry);
	}
	dput(whiteout_dentry);

	if (!err) {
	    d_delete(whiteout_dentry);
	} else {
	    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	    unlock_dir(hidden_dir_dentry);
	    goto out;
	}

	fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	/* propagate number of hard-links */
	dir->i_nlink = get_nlinks(dir);
	unlock_dir(hidden_dir_dentry);
    }

    if (dbstart(old_dentry) != dbstart(new_dentry)) {
	if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP)) {
	    err = -EXDEV;
	    goto out;
	}

	/* if GLOBAL_ERR_PASSUP is set then create destination path in source's branch */
	hidden_new_dentry = unionfs_create_dirs(dir, new_dentry, dbstart(old_dentry));
	err = PTR_ERR(hidden_new_dentry);
	if (IS_COPYUP_ERR(err)) {
	    goto docopyup;
	}
	if (!hidden_new_dentry || IS_ERR(hidden_new_dentry)) {
	    goto out;
	}
    }
    hidden_new_dentry = dtohd(new_dentry);
    hidden_old_dentry = dtohd(old_dentry);

    ASSERT (dbstart(old_dentry) == dbstart(new_dentry));
    dget(hidden_old_dentry);
    hidden_dir_dentry = lock_parent(hidden_new_dentry);

    if (!(err = is_robranch(old_dentry))) {
	err = vfs_link(hidden_old_dentry, hidden_dir_dentry->d_inode, hidden_new_dentry);
    }
docopyup:
    if (IS_COPYUP_ERR(err)) {
	int old_bstart = dbstart(old_dentry);
	int bindex;

    	if (hidden_dir_dentry) {
		unlock_dir(hidden_dir_dentry);
		hidden_dir_dentry = NULL;
	}
    	if (hidden_old_dentry) {
		dput(hidden_old_dentry);
		hidden_old_dentry = NULL;
	}

	for (bindex = old_bstart - 1; bindex >= 0; bindex--) {
	    err = unionfs_copyup_dentry_len(old_dentry->d_parent->d_inode, old_dentry, old_bstart, bindex, NULL, old_dentry->d_inode->i_size);
	    if (!err) {
		hidden_new_dentry = unionfs_create_dirs(dir, new_dentry, bindex);
		hidden_old_dentry = dtohd(old_dentry);
		dget(hidden_old_dentry);
		hidden_dir_dentry = lock_parent(hidden_new_dentry);
		/* do vfs_link */
		err = vfs_link(hidden_old_dentry, hidden_dir_dentry->d_inode, hidden_new_dentry);
		goto check_link;
	    }
	}
	goto out_lock;
    }
check_link:
    if (err || !hidden_new_dentry->d_inode) {
	goto out_lock;
    }

    /* Its a hard link, so use the same inode */
    new_dentry->d_inode=old_dentry->d_inode;
    atomic_inc(&new_dentry->d_inode->i_count);
    /* FIXME: Update c_time & co ? */

    err = unionfs_interpose(new_dentry, dir->i_sb, INTERPOSE_LINK);

    /*  if (err)
	goto out_lock;*/
    fist_copy_attr_timesizes(dir, hidden_new_dentry->d_inode);
    /* propagate number of hard-links */
    old_dentry->d_inode->i_nlink = get_nlinks(old_dentry->d_inode);

out_lock:
    if (hidden_dir_dentry) unlock_dir(hidden_dir_dentry);
    if (hidden_old_dentry) dput(hidden_old_dentry);

out:
    if (!new_dentry->d_inode)
	d_drop(new_dentry);

    if (name)
	KFREE(name);

    print_exit_status(err);
    return err;
}

STATIC int
unionfs_unlink_first(inode_t *dir, dentry_t *dentry)
{
	int err;
	dentry_t *hidden_dentry;
	dentry_t *hidden_dir_dentry = NULL;

	print_entry_location();

	hidden_dentry = dtohd(dentry);
	PASSERT(hidden_dentry);

	hidden_dir_dentry = lock_parent(hidden_dentry);

        /* avoid destroying the hidden inode if the file is in use */
        dget(hidden_dentry);
	if (!(err = is_robranch(dentry))) {
        	err = vfs_unlink(hidden_dir_dentry->d_inode, hidden_dentry);
	}
        dput(hidden_dentry);

        if (!err) {	/* vfs_unlink does that */
	    d_delete(hidden_dentry);
	}

	fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

        if (hidden_dir_dentry) {
		unlock_dir(hidden_dir_dentry);
	}
	print_exit_status(err);
	return err;
}

STATIC int
unionfs_unlink_all(inode_t *dir, dentry_t *dentry)
{
	struct dentry *hidden_dentry;
	struct dentry *hidden_dir_dentry;
 	int bstart, bend, bindex;
	int err = 0;
	int global_err = 0;

	print_entry_location();

	err = unionfs_partial_lookup(dentry);
	if (err) {
	    fist_dprint(8, "Error in partial lookup\n");
	    goto out;
	}

	bstart = dbstart(dentry);
	bend = dbend(dentry);

	for (bindex = bend; bindex >= bstart; bindex--) {
	    hidden_dentry = dtohd_index(dentry, bindex);
	    if (!hidden_dentry)
		continue;

	    hidden_dir_dentry = lock_parent(hidden_dentry);

	    /* avoid destroying the hidden inode if the file is in use */
	    dget(hidden_dentry);
	    if(!(err = is_robranch_super(dentry->d_sb, bindex))) {
	    	err = vfs_unlink(hidden_dir_dentry->d_inode, hidden_dentry);
	    }
	    dput(hidden_dentry);
	    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	    if (err) {
	    	if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || !IS_COPYUP_ERR(err)) {
			/* passup the last error we got */
	                unlock_dir(hidden_dir_dentry);
                	goto out;
		}
		global_err = err;
	    } else {		/* since it was OK, we kill the hidden dentry. */
		d_delete(hidden_dentry);
		if (bindex != bstart) {
			dput(hidden_dentry);
			set_dtohd_index(dentry, bindex, NULL);
		}
	    }

	    unlock_dir(hidden_dir_dentry);
        }

	/* check if encountered error in the above loop */
	if (global_err) {
		/* If we failed in the leftmost branch, then err will be set and we should
		 * move one over to create the whiteout.  Otherwise, we should try in the
		 * leftmost branch.
		 */
		if (err) {
		    if (dbstart(dentry) == 0) {
			goto out;
		    }
		    err = create_whiteout(dentry, dbstart(dentry) - 1);
		} else {
		    err = create_whiteout(dentry, dbstart(dentry));
		}
	}

out:
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

	print_exit_status(err);
	return err;
}

STATIC int
unionfs_unlink_whiteout(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    struct dentry *hidden_old_dentry;
    struct dentry *hidden_wh_dentry = NULL;
    struct dentry *hidden_old_dir_dentry, *hidden_new_dir_dentry;
    char *name = NULL;
//    struct iattr newattrs;

    print_entry_location();
    fist_print_dentry("IN unionfs_unlink_whiteout: ", dentry);

    /* create whiteout, get the leftmost underlying dentry and rename it */
    hidden_old_dentry = dtohd(dentry);

    /* lookup .wh.foo first, MUST NOT EXIST */
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
        err = -ENOMEM;
        goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[4 + dentry->d_name.len] = '\0';

    hidden_wh_dentry = lookup_one_len(name, hidden_old_dentry->d_parent, dentry->d_name.len + 4);
    if (IS_ERR(hidden_wh_dentry)) {
        err = PTR_ERR(hidden_wh_dentry);
        goto out;
    }
    ASSERT(hidden_wh_dentry->d_inode == NULL);

    dget(hidden_old_dentry);

    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_wh_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    if(!(err = is_robranch(dentry))) {
    	err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
			 hidden_new_dir_dentry->d_inode, hidden_wh_dentry);
    }

    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_old_dentry);
    dput(hidden_wh_dentry);

    if (err) {
	if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || (dbstart(dentry) == 0)) {
	    goto out;
	}
        /* exit if the error returned was NOT -EROFS */
        if (!IS_COPYUP_ERR(err)) {
            goto out;
        }
	err = create_whiteout(dentry, dbstart(dentry) - 1);
    } else {
        fist_copy_attr_all(dir, hidden_new_dir_dentry->d_inode);
    }

out:
    if (name) {
        KFREE(name);
    }
    print_exit_status(err);
    return err;
}

STATIC int
unionfs_unlink(inode_t *dir, dentry_t *dentry)
{
    int err = 0;

    print_entry_location();
    fist_print_dentry("IN unionfs_unlink: ", dentry);

    dget(dentry);

    if (IS_SET(dir->i_sb, DELETE_WHITEOUT)) {
        /* create whiteout */
        err = unionfs_unlink_whiteout(dir, dentry);
    } else if (IS_SET(dir->i_sb, DELETE_FIRST)) {
        /* delete only first file */
	err = unionfs_unlink_first(dir, dentry);
	/* The VFS will kill this dentry now, and it will end up being recreated on lookup. */
    } else {
	/* delete all. */
	err = unionfs_unlink_all(dir, dentry);
    }

    /* call d_drop so the system "forgets" about us */
    if (!err) {
	d_drop(dentry);
    }
    dput(dentry);

    print_exit_status(err);
    return err;
}

STATIC int
unionfs_rmdir_first(struct inode *dir, struct dentry *dentry, struct unionfs_dir_state *namelist)
{
	int err;
	dentry_t *hidden_dentry;
	dentry_t *hidden_dir_dentry = NULL;

	print_entry_location();
	fist_print_dentry("IN unionfs_rmdir_first: ", dentry);

	/* Here we need to remove whiteout entries. */
	err = delete_whiteouts(dentry, dbstart(dentry), namelist);
	if (err) {
	    goto out;
	}

	hidden_dentry = dtohd(dentry);
	PASSERT(hidden_dentry);

	hidden_dir_dentry = lock_parent(hidden_dentry);

        /* avoid destroying the hidden inode if the file is in use */
        dget(hidden_dentry);
	if (!(err = is_robranch(dentry))) {
        	err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
	}
        dput(hidden_dentry);
        if (!err) {	/* vfs_rmdir does that */
	    d_delete(hidden_dentry);
	}

	fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

out:
        if (hidden_dir_dentry) {
		unlock_dir(hidden_dir_dentry);
	}
	fist_print_dentry("OUT unionfs_rmdir_first: ", dentry);
	print_exit_status(err);
	return err;
}

STATIC int
unionfs_rmdir_all(inode_t *dir, dentry_t *dentry, struct unionfs_dir_state *namelist)
{
	struct dentry *hidden_dentry;
	struct dentry *hidden_dir_dentry;
 	int bstart, bend, bindex;
	int err = 0;
	int global_err = 0;
        int try_flag = 0;

	print_entry_location();
	fist_print_dentry("IN unionfs_rmdir_all: ", dentry);

	err = unionfs_partial_lookup(dentry);
	if (err) {
	    fist_dprint(8, "Error in partial lookup\n");
	    goto out;
	}

	bstart = dbstart(dentry);
	bend = dbend(dentry);

	for (bindex = bend; bindex >= bstart; bindex--) {

	    hidden_dentry = dtohd_index(dentry, bindex);
	    if (!hidden_dentry)
		continue;

try_rmdir_again:
	    hidden_dir_dentry = lock_parent(hidden_dentry);
	    /* avoid destroying the hidden inode if the file is in use */
	    dget(hidden_dentry);
	    if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
	    	err = vfs_rmdir(hidden_dir_dentry->d_inode, hidden_dentry);
	    }
	    dput(hidden_dentry);
	    fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	    unlock_dir(hidden_dir_dentry);
	    if (err) {
		/* passup the last error we got if GLOBAL_ERR_PASSUP is set
                 or exit if error is (NOT -EROFS and NOT -ENOTEMPTY) */
		if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || (!IS_COPYUP_ERR(err) && err != -ENOTEMPTY)) {
                    goto out;
		}

                global_err = err;

                /* If we got ENOTEMPTY in the zeroth branch, we try to remove
                 * whiteouts, and rmdir again.
                 */
                if ((bindex == 0) && (err == -ENOTEMPTY) && !try_flag) {
                    try_flag = 1;
                    err = delete_whiteouts(dentry, 0, namelist);
                    if (!err) {
                        goto try_rmdir_again;
                    }
                }
	    }

	    if (!err) {		/* since it was OK, we kill the hidden dentry. */
		d_delete(hidden_dentry);
	    }
        }

	/* check if encountered error in the above loop */
	if (global_err) {
		/* If we failed in the leftmost branch, then err will be set and we should
		 * move one over to create the whiteout.  Otherwise, we should try in the
		 * leftmost branch.
		 */
		if (err) {
		    if (dbstart(dentry) == 0) {
			goto out;
		    }
		    err = create_whiteout(dentry, dbstart(dentry) - 1);
		} else {
		    err = create_whiteout(dentry, dbstart(dentry));
		}
	}

out:
	/* propagate number of hard-links */
	dentry->d_inode->i_nlink = get_nlinks(dentry->d_inode);

	fist_print_dentry("OUT unionfs_rmdir_all: ", dentry);
	print_exit_status(err);
	return err;
}

STATIC int
unionfs_rmdir(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    struct unionfs_dir_state *namelist = NULL;

    print_entry_location();
    fist_print_dentry("IN unionfs_rmdir: ", dentry);

    dget(dentry);

    /* check if this unionfs directory is empty or not */
    err = check_empty(dentry, &namelist);
    if (err) {
	goto out;
    }

    if (IS_SET(dir->i_sb, DELETE_WHITEOUT)) {
	/* delete just like if we were doing DELETE_FIRST. */
	err = unionfs_rmdir_first(dir, dentry, namelist);
        /* create whiteout */
	if (!err) {
	    err = create_whiteout(dentry, dbstart(dentry));
	} else {
	    int new_err;

	    if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP) || (dbstart(dentry) == 0)) {
		goto out;
	    }
            /* exit if the error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                goto out;
            }
	    new_err = create_whiteout(dentry, dbstart(dentry) - 1);
	    if (new_err != -EEXIST) {
		err = new_err;
	    }
	}
    } else if (IS_SET(dir->i_sb, DELETE_FIRST)) {
        /* delete only first directory */
	err = unionfs_rmdir_first(dir, dentry, namelist);
	/* The VFS will kill this dentry now, and it will end up being recreated on lookup. */
    } else {
	/* delete all. */
	err = unionfs_rmdir_all(dir, dentry, namelist);
    }

 out:
    /* call d_drop so the system "forgets" about us */
    if (!err) {
	d_drop(dentry);
    }
    dput(dentry);


    if (namelist) {
	free_rdstate(namelist);
    }

    print_exit_status(err);
    return err;
}

STATIC int
unionfs_symlink(inode_t *dir, dentry_t *dentry, const char *symname)
{
    int err = 0;
    dentry_t *hidden_dentry = NULL;
    dentry_t *whiteout_dentry = NULL;
    dentry_t *hidden_dir_dentry = NULL;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,0)
	umode_t	mode;
#endif
    int bindex = 0, bstart;
    char *name = NULL;

    print_entry_location();
    fist_print_dentry("IN unionfs_symlink :", dentry);

    /* We start out in the leftmost branch. */
    bstart = dbstart(dentry);

    hidden_dentry = dtohd(dentry);

    /* check if whiteout exists in this branch, i.e. lookup .wh.foo first. If present, delete it */
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
	err = -ENOMEM;
	goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[4 + dentry->d_name.len] = '\0';

    whiteout_dentry = lookup_one_len(name, hidden_dentry->d_parent, dentry->d_name.len + 4);
    if (IS_ERR(whiteout_dentry)) {
	err = PTR_ERR(whiteout_dentry);
	goto out;
    }

    PASSERT(whiteout_dentry);
    if (!whiteout_dentry->d_inode) {
        dput(whiteout_dentry);
        whiteout_dentry = NULL;
    } else {
	/* found a .wh.foo entry, unlink it and then call vfs_symlink() */
	hidden_dir_dentry = lock_parent(whiteout_dentry);

        if (!(err = is_robranch_super(dentry->d_sb, bstart))) {
            err = vfs_unlink(hidden_dir_dentry->d_inode, whiteout_dentry);
	}
	dput(whiteout_dentry);

	if (!err)			// vfs_unlink does that
	   d_delete(whiteout_dentry);

	fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);
	/* propagate number of hard-links */
	dir->i_nlink = get_nlinks(dir);

	unlock_dir(hidden_dir_dentry);

	if (err) {
	    if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP)) {
		goto out;
            }
	    /* exit if the error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                goto out;
            }
            /* should now try to create symlink in the another branch */
	    bstart--;
	}
    }

    /* deleted whiteout if it was present, now do a normal vfs_symlink() with
       possible recursive directory creation */
    for (bindex = bstart; bindex >= 0; bindex--) {
	hidden_dentry = dtohd_index(dentry, bindex);
	if (!hidden_dentry) {
	    /* if hidden_dentry is NULL, create the entire
	     * dentry directory structure in branch 'bindex'. hidden_dentry will NOT be null when
	     * bindex == bstart because lookup passed as a negative unionfs dentry pointing to a
	     * lone negative underlying dentry */
	    hidden_dentry = unionfs_create_dirs(dir, dentry, bindex);
	    if (!hidden_dentry || IS_ERR(hidden_dentry)) {
		if (IS_ERR(hidden_dentry)) {
		    err = PTR_ERR(hidden_dentry);
		}
		fist_dprint(8, "hidden dentry NULL (or error) for bindex = %d\n", bindex);
		continue;
	    }
	}

        PASSERT(hidden_dentry);

	hidden_dir_dentry = lock_parent(hidden_dentry);

	if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		err = vfs_symlink(hidden_dir_dentry->d_inode, hidden_dentry, symname);
#else
		mode = S_IALLUGO;
		err = vfs_symlink(hidden_dir_dentry->d_inode, hidden_dentry, symname, mode);
#endif
	}
        unlock_dir(hidden_dir_dentry);

	if (err || !hidden_dentry->d_inode) {
            if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP)) {
		goto out;
            }
            /* break out of for loop if error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                break;
            }
        } else {
            err = unionfs_interpose(dentry, dir->i_sb, 0);
            if (!err) {
                fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
                /* update number of links on parent directory */
                dir->i_nlink = get_nlinks(dir);
            }
            break;
        }
    }

out:
    if (!dentry->d_inode)
	d_drop(dentry);

    if (name) {
	KFREE(name);
    }
    fist_print_dentry("OUT unionfs_symlink :", dentry);
    print_exit_status(err);
    return err;
}



STATIC int
unionfs_mkdir(inode_t *parent, dentry_t *dentry, int mode)
{
    int err = 0;
    dentry_t *hidden_dentry = NULL, *whiteout_dentry = NULL;
    dentry_t *hidden_parent_dentry = NULL;
    int bindex = 0, bstart;
    char *name = NULL;
    int whiteout_unlinked = 0;

    print_entry_location();
    fist_print_dentry("IN unionfs_mkdir :", dentry);
    bstart = dbstart(dentry);

    hidden_dentry = dtohd(dentry);

    // check if whiteout exists in this branch, i.e. lookup .wh.foo first
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
	err = -ENOMEM;
	goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[4 + dentry->d_name.len] = '\0';

    whiteout_dentry = lookup_one_len(name, hidden_dentry->d_parent, dentry->d_name.len + 4);
    if (IS_ERR(whiteout_dentry)) {
	err = PTR_ERR(whiteout_dentry);
	goto out;
    }
    PASSERT(whiteout_dentry);

    if (!whiteout_dentry->d_inode) {
        dput(whiteout_dentry);
        whiteout_dentry = NULL;
    } else {
        /* found .wh.foo, unlink it */
	hidden_parent_dentry = lock_parent(whiteout_dentry);

	//found a.wh.foo entry, remove it then do vfs_mkdir
    	if (!(err = is_robranch_super(dentry->d_sb, bstart))) {
	    err = vfs_unlink(hidden_parent_dentry->d_inode, whiteout_dentry);
	}
	dput(whiteout_dentry);

        unlock_dir(hidden_parent_dentry);

        if (err) {
            /* error in unlink of whiteout */
            if (IS_SET(parent->i_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* exit if the error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                goto out;
            }
	    bstart--;
        } else {
            whiteout_unlinked = 1;
        }
    }

    for (bindex = bstart; bindex >= 0; bindex--) {
	hidden_dentry = dtohd_index(dentry, bindex);
	if (!hidden_dentry) {
	    hidden_dentry = unionfs_create_dirs(parent, dentry, bindex);
	    if (!hidden_dentry || IS_ERR(hidden_dentry)) {
		fist_dprint(8, "hidden dentry NULL for bindex = %d\n", bindex);
		continue;
	    }
        }

	PASSERT(hidden_dentry);

	hidden_parent_dentry = lock_parent(hidden_dentry);
	if (IS_ERR(hidden_parent_dentry)) {
	    err = PTR_ERR(hidden_parent_dentry);
	    goto out;
	}
	if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
	    err = vfs_mkdir(hidden_parent_dentry->d_inode, hidden_dentry, mode);
	}
	/* XXX this could potentially return a negative hidden_dentry! */
        if (err || !hidden_dentry->d_inode) {
            unlock_dir(hidden_parent_dentry);

            if (IS_SET(parent->i_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* break out of for loop if error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                break;
            }
        } else {
            err = unionfs_interpose(dentry, parent->i_sb, 0);
            if (!err) {
                fist_copy_attr_timesizes(parent, hidden_parent_dentry->d_inode);
                /* update number of links on parent directory */
                parent->i_nlink = get_nlinks(parent);
            }
            unlock_dir(hidden_parent_dentry);

            if (whiteout_unlinked == 1) {
                /* create whiteout entries for directories named foo present to the right */
	        err = create_dir_whs(dentry, bstart);
	        if (err) {
                    fist_dprint(8, "Error creating whiteouts after mkdir success\n");
                    goto out;
                }
            }
            break;
        }
    } // end for

out:
    if (!dentry->d_inode)
	d_drop(dentry);

    if (name) {
	KFREE(name);
    }

    fist_print_dentry("OUT unionfs_mkdir :", dentry);
    print_exit_status(err);
    return err;
}


//DQ: Change of prototype in 2.6
STATIC int
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
unionfs_mknod(inode_t *dir, dentry_t *dentry, int mode, int dev)
#else
unionfs_mknod(inode_t *dir, dentry_t *dentry, int mode, dev_t dev)
#endif
{
    int err = 0;
    dentry_t *hidden_dentry = NULL, *whiteout_dentry = NULL;
    dentry_t *hidden_parent_dentry = NULL;
    int bindex = 0, bstart;
    char *name = NULL;
    int whiteout_unlinked = 0;

    print_entry_location();
    fist_print_dentry("IN unionfs_mknod :", dentry);
    bstart = dbstart(dentry);

    hidden_dentry = dtohd(dentry);

    // check if whiteout exists in this branch, i.e. lookup .wh.foo first
    name = KMALLOC(sizeof(char) * (dentry->d_name.len + 5), GFP_UNIONFS);
    if (!name) {
	err = -ENOMEM;
	goto out;
    }
    strcpy(name, ".wh.");
    strncat(name, dentry->d_name.name, dentry->d_name.len);
    name[4 + dentry->d_name.len] = '\0';

    whiteout_dentry = lookup_one_len(name, hidden_dentry->d_parent, dentry->d_name.len + 4);
    if (IS_ERR(whiteout_dentry)) {
	err = PTR_ERR(whiteout_dentry);
	goto out;
    }

    if (!whiteout_dentry->d_inode) {
        dput(whiteout_dentry);
        whiteout_dentry = NULL;
    } else {
        /* found .wh.foo, unlink it */
	hidden_parent_dentry = lock_parent(whiteout_dentry);

	//found a.wh.foo entry, remove it then do vfs_mkdir
        if (!(err = is_robranch_super(dentry->d_sb, bstart))) {
	   err = vfs_unlink(hidden_parent_dentry->d_inode, whiteout_dentry);
	}
	dput(whiteout_dentry);

        unlock_dir(hidden_parent_dentry);

        if (err) {
            /* error in unlink of whiteout */
            if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* exit if the error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                goto out;
            }

            bstart--;
        } else {
            whiteout_unlinked = 1;
        }
    }

    for (bindex = bstart; bindex >= 0; bindex--) {
	hidden_dentry = dtohd_index(dentry, bindex);
	if (!hidden_dentry) {
	    hidden_dentry = unionfs_create_dirs(dir, dentry, bindex);
	    if (!hidden_dentry || IS_ERR(hidden_dentry)) {
		fist_dprint(8, "hidden dentry NULL for bindex = %d\n", bindex);
		continue;
	    }
        }

	PASSERT(hidden_dentry);

	hidden_parent_dentry = lock_parent(hidden_dentry);
	if (IS_ERR(hidden_parent_dentry)) {
	    err = PTR_ERR(hidden_parent_dentry);
	    goto out;
	}
	if (!(err = is_robranch_super(dentry->d_sb, bindex))) {
	     err = vfs_mknod(hidden_parent_dentry->d_inode, hidden_dentry, mode, dev);
	}
	/* XXX this could potentially return a negative hidden_dentry! */
        if (err || !hidden_dentry->d_inode) {
            unlock_dir(hidden_parent_dentry);

            if (IS_SET(dir->i_sb, GLOBAL_ERR_PASSUP)) {
                goto out;
            }
            /* break out of for loop if error returned was NOT -EROFS */
            if (!IS_COPYUP_ERR(err)) {
                break;
            }
        } else {
            err = unionfs_interpose(dentry, dir->i_sb, 0);
            if (!err) {
                fist_copy_attr_timesizes(dir, hidden_parent_dentry->d_inode);
                /* update number of links on parent directory */
                dir->i_nlink = get_nlinks(dir);
            }
            unlock_dir(hidden_parent_dentry);

	    break;
        }
    } // end for

out:
    if (!dentry->d_inode)
	d_drop(dentry);

    if (name) {
	KFREE(name);
    }

    fist_print_dentry("OUT unionfs_mknod :", dentry);
    print_exit_status(err);
    return err;
}

STATIC int
do_rename(inode_t *old_dir, dentry_t *old_dentry, inode_t *new_dir, dentry_t *new_dentry, int bindex)
{
    int err = 0;
    dentry_t *hidden_old_dentry;
    dentry_t *hidden_new_dentry;
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;
    dentry_t *hidden_wh_dentry;
    dentry_t *hidden_wh_dir_dentry;
    char *wh_name = NULL;

    print_entry(" bindex=%d", bindex);

    fist_print_dentry("IN: do_rename, old_dentry:", old_dentry);
    fist_print_dentry("IN: do_rename, new_dentry:", new_dentry);
    fist_dprint(7, "do_rename for bindex = %d\n", bindex);

    hidden_new_dentry = dtohd_index(new_dentry, bindex);
    hidden_old_dentry = dtohd_index(old_dentry, bindex);
    PASSERT(hidden_old_dentry);

    if(!hidden_new_dentry) {
	hidden_new_dentry = unionfs_create_dirs(new_dentry->d_parent->d_inode, new_dentry, bindex);
        if (IS_ERR(hidden_new_dentry)) {
            fist_dprint(7, "error creating directory tree for rename\n");
	    err = PTR_ERR(hidden_new_dentry);
            goto out;
        }
    }

    wh_name = KMALLOC(new_dentry->d_name.len + 5, GFP_UNIONFS);
    if (!wh_name) {
	err = -ENOMEM;
	goto out;
    }
    strcpy(wh_name, ".wh.");
    strncat(wh_name, new_dentry->d_name.name, new_dentry->d_name.len);
    wh_name[4 + new_dentry->d_name.len] = '\0';

    hidden_wh_dentry = lookup_one_len(wh_name, hidden_new_dentry->d_parent, new_dentry->d_name.len + 4);
    if (IS_ERR(hidden_wh_dentry)) {
	err = PTR_ERR(hidden_wh_dentry);
        goto out;
    }

    if (hidden_wh_dentry->d_inode) {
        /* get rid of the whiteout that is existing */
        if (hidden_new_dentry->d_inode) {
	    printk(KERN_WARNING "Both a whiteout and a dentry exist when doing a rename!\n");
	    err = -EIO;

	    dput(hidden_wh_dentry);
	    goto out;
	}

        fist_print_dentry("hidden_wh_dentry->d_parent:", hidden_wh_dentry->d_parent);
        hidden_wh_dir_dentry = lock_parent(hidden_wh_dentry);
    	if (!(err = is_robranch_super(old_dentry->d_sb, bindex))) {
            err = vfs_unlink(hidden_wh_dir_dentry->d_inode, hidden_wh_dentry);
	}
	dput(hidden_wh_dentry);
        unlock_dir(hidden_wh_dir_dentry);
        if (err) {
            goto out;
        }
        d_delete(hidden_wh_dentry);
    } else {
        dput(hidden_wh_dentry);
    }

    dget(hidden_old_dentry);
    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);

    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    if (!(err = is_robranch_super(old_dentry->d_sb, bindex))) {
	PASSERT(hidden_old_dir_dentry->d_inode);
	PASSERT(hidden_old_dentry);
	PASSERT(hidden_old_dentry->d_inode);
	PASSERT(hidden_new_dir_dentry->d_inode);
	PASSERT(hidden_new_dentry);
    	err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
            hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    }

    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_old_dentry);

out:
    if (!err) {
        /* Fixup the newdentry. */
    	if (bindex < dbstart(new_dentry)) {
	   set_dbstart(new_dentry, bindex);
    	} else if (bindex > dbend(new_dentry)) {
	    set_dbend(new_dentry, bindex);
        }
    }

    if (wh_name) {
	KFREE(wh_name);
    }

    fist_print_dentry("OUT: do_rename, old_dentry:", old_dentry);
    fist_print_dentry("OUT: do_rename, new_dentry:", new_dentry);

    print_exit_status(err);
    return err;
}

STATIC int
unionfs_rename_whiteout(inode_t *old_dir, dentry_t *old_dentry,
                        inode_t *new_dir, dentry_t *new_dentry)
{
    int err = 0;
    int bindex;
    int old_bstart, old_bend;
    int new_bstart, new_bend;
    int do_copyup = -1;
    dentry_t *parent_dentry = NULL;
    int local_err = 0;
    int eio = 0;
    int revert = 0;

    print_entry_location();

    old_bstart = dbstart(old_dentry);
    old_bend = dbend(old_dentry);
    parent_dentry = old_dentry->d_parent;

    new_bstart = dbstart(new_dentry);
    new_bend = dbend(new_dentry);

    /* Rename source to destination. */
    err = do_rename(old_dir, old_dentry, new_dir, new_dentry, old_bstart);
    if (err) {
	if (!IS_COPYUP_ERR(err)) {
	    goto out;
	}
	do_copyup = old_bstart -1;
    } else {
	revert = 1;
    }

    /* Unlink all instances of destination that exist to the left of
     * bstart of source. On error, revert back, goto out.
     */
    for (bindex = old_bstart - 1; bindex >= new_bstart; bindex--) {
        struct dentry *unlink_dentry;
        struct dentry *unlink_dir_dentry;

        unlink_dentry = dtohd_index(new_dentry, bindex);
        if (!unlink_dentry) {
            continue;
        }

        unlink_dir_dentry = lock_parent(unlink_dentry);
        dget(unlink_dentry);
    	if (!(err = is_robranch_super(old_dir->i_sb, bindex))) {
            err = vfs_unlink(unlink_dir_dentry->d_inode, unlink_dentry);
	}
        dput(unlink_dentry);

        fist_copy_attr_times(new_dentry->d_parent->d_inode, unlink_dir_dentry->d_inode);
        /* propagate number of hard-links */
        new_dentry->d_parent->d_inode->i_nlink = get_nlinks(new_dentry->d_parent->d_inode);

        unlock_dir(unlink_dir_dentry);
        if (!err) {
	    d_delete(unlink_dentry);
	    if (bindex != new_bstart) {
		dput(unlink_dentry);
		set_dtohd_index(new_dentry, bindex, NULL);
	    }
	} else if (IS_COPYUP_ERR(err)) {
		do_copyup = bindex - 1;
	} else if (revert) {
	    goto revert;
	}
    }

    if (do_copyup != -1) {
	for (bindex = do_copyup; bindex >= 0; bindex--) {
	    /* copyup the file into some left directory, so that you can rename it */
	    err = unionfs_copyup_dentry_len(old_dentry->d_parent->d_inode, old_dentry, old_bstart, bindex, NULL, old_dentry->d_inode->i_size);
	    if (!err) {
		parent_dentry = old_dentry->d_parent;
		err = do_rename(old_dir, old_dentry, new_dir, new_dentry, bindex);
	    }
	}
    }

    /* Create whiteout for source, only if:
     * (1) There is more than one underlying instance of source.
     * (2) We did a copy_up
     */
    if ((old_bstart != old_bend) || (do_copyup != -1)) {
	int start = (do_copyup == -1) ? old_bstart : do_copyup;
        /* we want to create a whiteout for name in  this parent dentry */
	local_err = create_whiteout_parent(parent_dentry, old_dentry->d_name.name, start);
        if (local_err) {
            /* We can't fix anything now, so we cop-out and use -EIO. */
	    printk("<0>We can't create a whiteout for the source in rename!\n");
            err = -EIO;
            goto out;
        }
    }

out:

    print_exit_status(err);
    return err;

revert:
    /* Do revert here. */
    local_err = unionfs_refresh_hidden_dentry(new_dentry, old_bstart);
    if (local_err) {
	printk(KERN_WARNING "Revert failed in rename: the new refresh failed.\n");
	eio = -EIO;
    }

    local_err = unionfs_refresh_hidden_dentry(old_dentry, old_bstart);
    if (local_err) {
	printk(KERN_WARNING "Revert failed in rename: the old refresh failed.\n");
	eio = -EIO;
	goto revert_out;
    }

    if (!dtohd_index(new_dentry, bindex) || !dtohd_index(new_dentry, bindex)->d_inode) {
	printk(KERN_WARNING "Revert failed in rename: the object disappeared from under us!\n");
	eio = -EIO;
	goto revert_out;
    }

    if (dtohd_index(old_dentry, bindex) && dtohd_index(old_dentry, bindex)->d_inode) {
	printk(KERN_WARNING "Revert failed in rename: the object was created underneath us!\n");
	eio = -EIO;
	goto revert_out;
    }

    local_err = do_rename(new_dir, new_dentry, old_dir, old_dentry, old_bstart);

    /* If we can't fix it, then we cop-out with -EIO. */
    if (local_err) {
	printk(KERN_WARNING "Revert failed in rename!\n");
	eio = -EIO;
    }

    local_err = unionfs_refresh_hidden_dentry(new_dentry, bindex);
    if (local_err) {
	eio = -EIO;
    }
    local_err = unionfs_refresh_hidden_dentry(old_dentry, bindex);
    if (local_err) {
	eio = -EIO;
    }

revert_out:
    if (eio) {
	err = eio;
    }
    print_exit_status(err);
    return err;
}

/*
 * The function is nasty, nasty, nasty, but so is rename. :(
 *
 * This psuedo-code describes what the function should do.  Essentially we move from
 * right-to-left, renaming each item.  We skip the leftmost destination (this is so
 * we can always undo the rename if the reverts work), and then the very last thing
 * we do is fix up the leftmost destination (either through renaming it or unlinking it).
 *
 * for i = S_R downto S_L
 *	if (i != L_D && exists(S[i]))
 *		err = rename(S[i], D[i])
 *		if (err == COPYUP)  {
 *			do_whiteout = i - 1;
 *			if (i == S_R) {
 *				do_copyup = i - 1
 *			}
 *		}
 *		else if (err)
 *			goto revert;
 *		else
 *			ok = i;
 *
 *
 * If we get to the leftmost source (S_L) and it is EROFS, we should do copyup
 *
 * if (err == COPYUP) {
 * 	do_copyup = i - 1;
 * }
 *
 * while (i > new_bstart) {
 * 	err = unlink(D[i]);
 *	if (err == COPYUP) {
 *		do_copyup = i - 1;
 *	} else if (err) {
 *		goto revert;
 *	}
 * }
 *
 * if (exists(S[L_D])) {
 *	err = rename(S[L_D], D[L_D]);
 *	if (err = COPYUP) {
 *		if (ok > L_D) {
 *			do_copyup = min(do_copyup, L_D - 1);
 *		}
 *		do_whiteout = min(do_whiteout, L_D - 1);
 *	} else {
 *		goto revert;
 *	}
 * } else {
 *	err = unlink(D[L_D]);
 *	if (err = COPYUP) {
 *		do_copyup = min(do_copyup, L_D - 1);
 *	} else {
 *		goto revert;
 *	}
 * }
 *
 * out:
 *	if (do_whiteout != -1) {
 *		create_whiteout(do_whiteout);
 *	}
 *	if (do_copyup != -1) {
 *		copyup(source to do_copyup)
 *		rename source to destination
 *	}
 *	return err;
 * out_revert:
 *	do the reverting;
 *	return err;
 * }
 */
STATIC int
unionfs_rename_all(inode_t *old_dir, dentry_t *old_dentry,
                   inode_t *new_dir, dentry_t *new_dentry)
{
    int old_bstart, old_bend;
    int new_bstart, new_bend;
    dentry_t *parent_dentry = NULL;
    int bindex;
    int err = 0;
    int eio = 0; /* Used for revert. */

    /* These variables control error handling. */
    int rename_ok = FD_SETSIZE; /* The last rename that is ok. */
    int do_copyup = -1; /* Where we should start copyup. */
    int do_whiteout = -1; /* Where we should start whiteouts of the source. */
    int clobber;	/* Are we clobbering the destination. */
    fd_set success_mask;

    print_entry_location();

    old_bstart = dbstart(old_dentry);
    old_bend = dbend(old_dentry);
    parent_dentry = old_dentry->d_parent;
    new_bstart = dbstart(new_dentry);
    new_bend = dbend(new_dentry);
    ASSERT(new_bstart >= 0);
    ASSERT(old_bstart >= 0);

    /* The failure mask only can deal with FD_SETSIZE entries. */
    ASSERT(old_bend <= FD_SETSIZE);
    ASSERT(new_bend <= FD_SETSIZE);
    FD_ZERO(&success_mask);

    /* Life is simpler if the dentry doesn't exist. */
    clobber = (dtohd_index(new_dentry, new_bstart)->d_inode) ? 1 : 0;

    /* Loop through all the branches from right to left and rename all
     * instances of old dentry to new dentry, except if they are
     */
    for (bindex = old_bend; bindex >= old_bstart; bindex--) {
	/* We don't rename if there is no source. */
        if (dtohd_index(old_dentry, bindex) == NULL) {
            continue;
        }

        /* we rename the bstart of destination only at the last of
         * all operations, so that we don't lose it on error
         */
	if (clobber && (bindex == new_bstart)) {
	    continue;
	}

	/* We shouldn't have a handle on this if there is no inode. */
	PASSERT(dtohd_index(old_dentry, bindex)->d_inode);

        err = do_rename(old_dir, old_dentry, new_dir, new_dentry, bindex);
	if (!err) {
	    /* For reverting. */
	    FD_SET(bindex, &success_mask);
	    /* So we know not to copyup on failures the right */
	    rename_ok = bindex;
        } else if (IS_COPYUP_ERR(err)) {
	    do_whiteout = bindex - 1;
	    if (bindex == old_bstart) {
		do_copyup = bindex - 1;
            }
	} else {
	    goto revert;
        }
    }

    while (bindex > new_bstart) {
        struct dentry *unlink_dentry;
        struct dentry *unlink_dir_dentry;

        unlink_dentry = dtohd_index(new_dentry, bindex);
        if (!unlink_dentry) {
            bindex--;
            continue;
        }

        unlink_dir_dentry = lock_parent(unlink_dentry);
        dget(unlink_dentry);
    	if (!(err = is_robranch_super(old_dir->i_sb, bindex))) {
            err = vfs_unlink(unlink_dir_dentry->d_inode, unlink_dentry);
	}
        dput(unlink_dentry);

        fist_copy_attr_times(new_dentry->d_parent->d_inode, unlink_dir_dentry->d_inode);
        new_dentry->d_parent->d_inode->i_nlink = get_nlinks(new_dentry->d_parent->d_inode);

        unlock_dir(unlink_dir_dentry);

	if (!err) {
            d_delete(unlink_dentry);
	    if (bindex != new_bstart) {
		dput(unlink_dentry);
		set_dtohd_index(new_dentry, bindex, NULL);
	    }
	}

        if (IS_COPYUP_ERR(err)) {
	    do_copyup = bindex - 1;
	} else if (err) {
            goto revert;
	}

        bindex--;
    } // while bindex

    /* Now we need to handle the leftmost of the destination. */
    if (clobber && dtohd_index(old_dentry, new_bstart)) {
        err = do_rename(old_dir, old_dentry, new_dir, new_dentry, new_bstart);
        if (IS_COPYUP_ERR(err)) {
		if (rename_ok > new_bstart) {
			if ((do_copyup == -1) || (new_bstart - 1 < do_copyup)) {
				do_copyup = new_bstart - 1;
			}
		}
		if ((do_whiteout == -1) || (new_bstart - 1 < do_whiteout)) {
			do_whiteout = new_bstart - 1;
		}
	} else if (err) {
		goto revert;
	}
    } else if (clobber && (new_bstart < old_bstart)) {
        struct dentry *unlink_dentry;
        struct dentry *unlink_dir_dentry;

        unlink_dentry = dtohd_index(new_dentry, new_bstart);
	PASSERT(unlink_dentry);
	PASSERT(unlink_dentry->d_inode);

        unlink_dir_dentry = lock_parent(unlink_dentry);
        dget(unlink_dentry);
    	if (!(err = is_robranch_super(old_dir->i_sb, new_bstart))) {
            err = vfs_unlink(unlink_dir_dentry->d_inode, unlink_dentry);
	}
        dput(unlink_dentry);

        fist_copy_attr_times(new_dentry->d_parent->d_inode, unlink_dir_dentry->d_inode);
        new_dentry->d_parent->d_inode->i_nlink = get_nlinks(new_dentry->d_parent->d_inode);

        unlock_dir(unlink_dir_dentry);
        if (!err) {
             d_delete(unlink_dentry);
        }

        if (IS_COPYUP_ERR(err)) {
	    if ((do_copyup == -1) || (new_bstart - 1 < do_copyup)) {
		do_copyup = new_bstart - 1;
	    }
	} else if (err) {
            goto revert;
	}
    }

    /* Create a whiteout for the source. */
    if (do_whiteout != -1) {
	ASSERT(do_whiteout >= 0);
	err = create_whiteout_parent(parent_dentry, old_dentry->d_name.name, do_whiteout);
        if (err) {
            /* We can't fix anything now, so we cop-out and use -EIO. */
	    printk("<0>We can't create a whiteout for the source in rename!\n");
            err = -EIO;
            goto out;
        }
    }

    if (do_copyup != -1) {
	for (bindex = do_copyup; bindex >= 0; bindex--) {
	   err = unionfs_copyup_dentry_len(old_dentry->d_parent->d_inode, old_dentry, old_bstart, bindex, NULL, old_dentry->d_inode->i_size);
	   if (!err) {
		err = do_rename(old_dir, old_dentry, new_dir, new_dentry, bindex);
	   }
	}
    }


    /* We are at the point where reverting doesn't happen. */
    goto out;

revert:
    for (bindex = old_bstart; bindex <= old_bend; bindex++) {
	int local_err;

	if (FD_ISSET(bindex, &success_mask)) {
	    local_err = unionfs_refresh_hidden_dentry(new_dentry, bindex);
	    if (local_err) {
		printk(KERN_WARNING "Revert failed in rename: the new refresh failed.\n");
		eio = -EIO;
	    }

	    local_err = unionfs_refresh_hidden_dentry(old_dentry, bindex);
	    if (local_err) {
		printk(KERN_WARNING "Revert failed in rename: the old refresh failed.\n");
		eio = -EIO;
		continue;
	    }

    	    if (!dtohd_index(new_dentry, bindex) || !dtohd_index(new_dentry, bindex)->d_inode) {
		printk(KERN_WARNING "Revert failed in rename: the object disappeared from under us!\n");
		eio = -EIO;
		continue;
    	    }

    	    if (dtohd_index(old_dentry, bindex) && dtohd_index(old_dentry, bindex)->d_inode) {
		printk(KERN_WARNING "Revert failed in rename: the object was created underneath us!\n");
		eio = -EIO;
		continue;
    	    }

	    local_err = do_rename(new_dir, new_dentry, old_dir, old_dentry, bindex);

	    /* If we can't fix it, then we cop-out with -EIO. */
	    if (local_err) {
		printk(KERN_WARNING "Revert failed in rename!\n");
		eio = -EIO;
	    }


	    local_err = unionfs_refresh_hidden_dentry(new_dentry, bindex);
	    if (local_err) {
		eio = -EIO;
            }
	    local_err = unionfs_refresh_hidden_dentry(old_dentry, bindex);
	    if (local_err) {
		eio = -EIO;
            }
	}
    }
    if (eio) {
	err = eio;
    }

out:
    print_exit_status(err);
    return err;
}


STATIC int
unionfs_rename_first(inode_t *old_dir, dentry_t *old_dentry,
                     inode_t *new_dir, dentry_t *new_dentry)
{
    int err;
    dentry_t *hidden_old_dentry;
    dentry_t *hidden_new_dentry;
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;

    print_entry_location();

    hidden_old_dentry = dtohd(old_dentry);
    hidden_new_dentry = dtohd(new_dentry);

    fist_checkinode(old_dir, "unionfs_rename-old_dir");
    fist_checkinode(new_dir, "unionfs_rename-new_dir");

    dget(hidden_old_dentry);
    dget(hidden_new_dentry);
    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    if (!(err = is_robranch(old_dentry))) {
        err = vfs_rename(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
                         hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    }
    if (err)
	goto out_lock;

    fist_copy_attr_all(new_dir, hidden_new_dir_dentry->d_inode);
    if (new_dir != old_dir)
	fist_copy_attr_all(old_dir, hidden_old_dir_dentry->d_inode);

out_lock:
    // double_unlock will dput the new/old parent dentries whose refcnts
    // were incremented via get_parent above.
    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
    dput(hidden_new_dentry);
    dput(hidden_old_dentry);

    fist_checkinode(new_dir, "post unionfs_rename-new_dir");
    print_exit_status(err);
    return err;
}


STATIC int
unionfs_rename(inode_t *old_dir, dentry_t *old_dentry,
	       inode_t *new_dir, dentry_t *new_dentry)
{
    int err = 0;
    dentry_t *hidden_old_dentry;
    dentry_t *hidden_new_dentry;

    print_entry_location();

    fist_checkinode(old_dir, "unionfs_rename-old_dir");
    fist_checkinode(new_dir, "unionfs_rename-new_dir");
    fist_print_dentry("IN: unionfs_rename, old_dentry", old_dentry);
    fist_print_dentry("IN: unionfs_rename, new_dentry", new_dentry);

    err = unionfs_partial_lookup(old_dentry);
    if (err) {
        fist_dprint(8, "Error in partial lookup\n");
        goto out;
    }
    err = unionfs_partial_lookup(new_dentry);
    if (err) {
        fist_dprint(8, "Error in partial lookup\n");
        goto out;
    }

    hidden_new_dentry = dtohd(new_dentry);
    hidden_old_dentry = dtohd(old_dentry);

    if (new_dentry->d_inode) {
        if(S_ISDIR(old_dentry->d_inode->i_mode) != S_ISDIR(new_dentry->d_inode->i_mode)) {
            err = S_ISDIR(old_dentry->d_inode->i_mode) ? -ENOTDIR : -EISDIR;
            goto out;
        }

    	if (S_ISDIR(old_dentry->d_inode->i_mode)) {
    	    /* check if this unionfs directory is empty or not */
    	    err = check_empty(new_dentry, NULL);
    	    if (err) {
		goto out;
    	    }
	    /* Handle the case where we are overwriting directories that are not
	     * really empty because of whiteout or non-whiteout entries.
	     */
        }
    }

    if (IS_SET(old_dir->i_sb, DELETE_WHITEOUT)) {
        /* create whiteout */
        err = unionfs_rename_whiteout(old_dir, old_dentry, new_dir, new_dentry);
    } else if (IS_SET(old_dir->i_sb, DELETE_FIRST)) {
        /* rename only first source entry */
        err = unionfs_rename_first(old_dir, old_dentry, new_dir, new_dentry);
    } else {
	/* delete all. */
	err = unionfs_rename_all(old_dir, old_dentry, new_dir, new_dentry);
    }

out:
    fist_checkinode(new_dir, "post unionfs_rename-new_dir");
    fist_print_dentry("OUT: unionfs_rename, old_dentry", old_dentry);
    fist_print_dentry("OUT: unionfs_rename, new_dentry", new_dentry);
    print_exit_status(err);
    return err;
}




STATIC int
unionfs_readlink(dentry_t *dentry, char *buf, int bufsiz)
{
    int err;
    dentry_t *hidden_dentry;

    print_entry_location();
    hidden_dentry = dtohd(dentry);
//    fist_print_dentry("unionfs_readlink IN", dentry);

    if (!hidden_dentry->d_inode->i_op ||
	!hidden_dentry->d_inode->i_op->readlink) {
	err = -EINVAL;
	goto out;
    }

    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 buf,
						 bufsiz);
    if (err > 0)
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);

 out:
    print_exit_status(err);
    return err;
}


STATIC int
unionfs_follow_link(dentry_t *dentry, struct nameidata *nd)
{
    char *buf;
    int len = PAGE_SIZE, err;
    mm_segment_t old_fs;

    print_entry_location();
//    fist_print_dentry("unionfs_follow_link dentry IN", dentry);

    buf = KMALLOC(len, GFP_UNIONFS);
    if (!buf) {
	err = -ENOMEM;
	goto out;
    }

    /* read the symlink, and then we will follow it */
    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = dentry->d_inode->i_op->readlink(dentry, buf, len);
    set_fs(old_fs);
    if (err < 0)
	goto out_free;

    buf[err] = 0;	// terminate the buffer -- XXX still needed?

    // XXX: FIXME w/ unionfs_encode_filename()
    /*
     * vfs_follow_link will increase the nd's mnt refcnt
     * we assume that some other VFS code decrements it.
     */
    err = vfs_follow_link(nd, buf);

 out_free:
    KFREE(buf);
 out:
#if 0
    if (err < 0) {
	dput(nd->dentry);
	printk("EZK follow_link() mnt_cnt %d\n", atomic_read(&nd->mnt->mnt_count));
	mntput(nd->mnt);
    }
#endif

    print_exit_status(err);
    return err;
}


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
static int unionfs_permission(inode_t *inode, int mask, struct nameidata *nd)
#else
static int unionfs_permission(inode_t *inode, int mask)
#endif
{
    inode_t *hidden_inode=NULL;
    int err = 0;
    int bindex, bstart, bend;
    int is_file = 0;

    print_entry_location();

    bstart = ibstart(inode);
    bend = ibend(inode);

    fist_print_inode("IN unionfs_permission: unionfs inode=", inode);

    /* set if check is for file */
    if (!S_ISDIR(inode->i_mode)) {
        is_file = 1;
    }

    for (bindex = bstart; bindex <= bend; bindex++) {

        hidden_inode = itohi_index(inode, bindex);
	if (!hidden_inode) {
	    continue;
	}

        /* check the condition for D-F-D underlying files/directories,
         * we dont have to check for files, if we are checking for
         * directories.
         */
        if (!S_ISDIR(hidden_inode->i_mode) && (!is_file)) {
            continue;
        }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	err = permission(hidden_inode, mask);
#else
	err = permission(hidden_inode, mask, nd);
#endif

	/* Our permission needs to return OK, even if it is a copy-up. */
	if ((mask & MAY_WRITE) && (!(stopd(inode->i_sb)->usi_branchperms[bindex] & MAY_WRITE))) {
		fist_dprint(8, "Checking permission on read-only branch: need to force copy-up later.\n");
	}

        /* any error in read/execute should be returned back immediately
         * since read/exec is an intersection.
         */
	if ((mask & MAY_READ) || (mask & MAY_EXEC)) {
	    if (!IS_COPYUP_ERR(err)) {
		goto out;
            }
        }

        /* Any error, except for COPYUP_ERR should be passed up */
	if (mask & MAY_WRITE) {
	    if (err) {
                /* If leftmost is RO, return error */
                if (IS_COPYUP_ERR(err) && bindex) {
                    /* This success will finally trigger copyup */
                    err = 0;
                    continue;
		}
	    	goto out;
            }
	}

        /* check for only leftmost file */
        if (is_file) {
            break;
        }
    } // end for

 out:
    print_exit_status(err);
    return err;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
STATIC int unionfs_inode_revalidate(dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry;
    inode_t *hidden_inode;
    inode_t *inode;
    int bindex, bstart, bend;
    int sbgen, igen;
    mode_t savemode;
    int locked = 0;

    print_entry_location();
    PASSERT(dentry);
    inode = dentry->d_inode;
    PASSERT(inode);

    /* Before anything else, we should check if the generation number is valid. */
restart:
    PASSERT(inode->i_sb);
    PASSERT(itopd(inode));
    sbgen = atomic_read(&stopd(inode->i_sb)->usi_generation);
    igen = atomic_read(&itopd(inode)->uii_generation);
    if (sbgen != igen) {
	struct dentry *result;

	lock_super(inode->i_sb);
	lock_dpd(dentry);
	locked = 1;

        savemode = inode->i_mode;

	/* The root entry should always be valid */
	fist_dprint(8, "revalidate: sbgen = %d, igen = %d\n", sbgen, igen);

	ASSERT(!IS_ROOT(dentry));
	PASSERT(dentry->d_parent);
	PASSERT(dentry->d_parent->d_inode);

	/* We can't work correctly if our parent isn't valid. */
	if(atomic_read(&stopd(inode->i_sb)->usi_generation) != atomic_read(&dtopd(dentry->d_parent)->udi_generation)) {
		PASSERT(dentry->d_parent->d_inode);
		PASSERT(dentry->d_parent->d_inode->i_op);
		PASSERT(dentry->d_parent->d_inode->i_op->revalidate);
		unlock_super(inode->i_sb);
		unlock_dpd(dentry);

		err = dentry->d_parent->d_inode->i_op->revalidate(dentry->d_parent);
		if (err) {
			goto out;
		}

		goto restart;
	}

	/* Free the pointers for our inodes and this dentry. */
	bstart = dbstart(dentry);
	bend = dbend(dentry);
	if (bstart >= 0) {
		struct dentry *hidden_dentry;
		for (bindex = bstart; bindex <= bend; bindex++) {
			hidden_dentry = dtohd_index(dentry, bindex);
			if (!hidden_dentry) continue;
			dput(hidden_dentry);
		}
	}
	KFREE(dtohd_ptr(dentry));
	dtohd_ptr(dentry) = NULL;
	set_dbstart(dentry, -1);
	set_dbend(dentry, -1);

	bstart = ibstart(inode);
	bend = ibend(inode);
	if (bstart >= 0) {
		struct inode *hidden_inode;
		for (bindex = bstart; bindex <= bend; bindex++) {
			hidden_inode = itohi_index(inode, bindex);
			if (!hidden_inode) continue;
			iput(hidden_inode);
		}
	}
	KFREE(itohi_ptr(dentry->d_inode));
	itohi_ptr(dentry->d_inode) = NULL;
	ibstart(dentry->d_inode) = -1;
	ibend(dentry->d_inode) = -1;



	result = unionfs_lookup_backend(dentry->d_parent->d_inode, dentry, INTERPOSE_REVAL);
	if (result && IS_ERR(result)) {
		err = PTR_ERR(result);
		goto out;
	}

	if (itopd(dentry->d_inode)->uii_stale) {
		make_stale_inode(dentry->d_inode);
		d_drop(dentry);
  	        err = -ESTALE;
		goto out;
	}
    }

    bstart = dbstart(dentry);
    bend = dbend(dentry);
    for (bindex = bstart; bindex <= bend; bindex++) {
	hidden_dentry = dtohd_index(dentry, bindex);
	if (!hidden_dentry) {
	    continue;
	}

	hidden_inode = hidden_dentry->d_inode;
	PASSERT(hidden_inode);

	/* Now revalidate the lower level. */
	if (hidden_inode->i_op && hidden_inode->i_op->revalidate) {
	    err = hidden_inode->i_op->revalidate(hidden_dentry);
	}
    }

    hidden_inode = NULL;
    for (bindex = bstart; bindex <= bend && !hidden_inode; bindex++) {
	hidden_inode = itohi_index(inode, bindex);
    }

    if (hidden_inode && !err) {
	fist_copy_attr_all(inode, hidden_inode);
    }

out:
    if (locked) {
        unlock_dpd(dentry);
	unlock_super(inode->i_sb);
    }
    print_exit_status(err);
    return err;
}
#endif /* End of revalidate */

STATIC int
unionfs_setattr(dentry_t *dentry, struct iattr *ia)
{
    int err = 0;
    dentry_t *hidden_dentry;
    inode_t *inode = NULL;
    inode_t *hidden_inode = NULL;
    int bstart, bend, bindex;
    int i;
    int copyup = 0;
    int size = 0;
    int locked = 0;

    print_entry_location();
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    if (!(ia->ia_valid & ATTR_SIZE)) {
		locked = 1;
		PASSERT(dentry->d_inode);
		down(&dentry->d_inode->i_sem);
    }
#endif
	if (IS_SET(dentry->d_sb, SETATTR_ALL)) {
		err = unionfs_partial_lookup(dentry);
		if (err) {
			fist_dprint(8, "Error in partial lookup\n");
			goto out;
		}
    }
    bstart = dbstart(dentry);
    bend = dbend(dentry);
    inode = dentry->d_inode;

	for (bindex = bstart; (bindex <= bend) || (bindex == bstart); bindex++) {
		hidden_dentry = dtohd_index(dentry, bindex);
		if (!hidden_dentry) {
			continue;
		}
		ASSERT(hidden_dentry->d_inode != NULL);

		/* If the file is on a read only branch */
		if (is_robranch_super(dentry->d_sb, bindex) || IS_RDONLY(hidden_dentry->d_inode)) {
            if (copyup || (bindex != bstart)) {
                continue;
            }

            /* Only if its the leftmost file, copyup the file */
	    for (i = bstart - 1; i >= 0; i--) {
		    if (ia->ia_valid & ATTR_SIZE) {
			    size = ia->ia_size;
			    err = unionfs_copyup_dentry_len(dentry->d_parent->d_inode, dentry, bstart, i, NULL, size);
		    } else {
			    err = unionfs_copyup_dentry_len(dentry->d_parent->d_inode, dentry, bstart, i, NULL, dentry->d_inode->i_size);
		    }
		    if (err) {
			    /* if error is in the leftmost f/s, stop and passup the error */
			    if (i == 0) {
				    goto out;
			    }
		    } else {
			    copyup = 1;
			    hidden_dentry = dtohd(dentry);
			    break;
		    }
	    }

	}
	err = notify_change(hidden_dentry, ia);
	if (err) {
	    goto out;
	}

	if (!IS_SET(dentry->d_sb, SETATTR_ALL)) {
            break;
	}
    }

    /* get the size from the first hidden inode */
    hidden_inode = itohi(dentry->d_inode);
    fist_checkinode(inode, "unionfs_setattr");
    fist_copy_attr_all(inode, hidden_inode);

 out:
    if (locked) {
	up(&dentry->d_inode->i_sem);
    }
    fist_checkinode(inode, "post unionfs_setattr");
    print_exit_status(err);
    return err;
}
#if NOT_USED_YET
STATIC int
unionfs_getattr(dentry_t *dentry, struct iattr *ia)
{
    return -ENOSYS;
}
#endif /* NOT_USED_YET */


struct inode_operations unionfs_symlink_iops =
{
    readlink:	unionfs_readlink,
    follow_link: unionfs_follow_link,
    permission:	unionfs_permission,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    revalidate:	unionfs_inode_revalidate,
#endif
    setattr:	unionfs_setattr,
};


struct inode_operations unionfs_dir_iops =
{
    create:	unionfs_create,
    lookup:	unionfs_lookup,
    link:	unionfs_link,
    unlink:	unionfs_unlink,
    symlink:	unionfs_symlink,
    mkdir:	unionfs_mkdir,
    rmdir:	unionfs_rmdir,
    mknod:	unionfs_mknod,
    rename:	unionfs_rename,
    /* no readlink/follow_link for non-symlinks */
    // off because we have setattr
    //    truncate:	unionfs_truncate,
    permission:	unionfs_permission,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    revalidate:	unionfs_inode_revalidate,
#endif
    setattr:	unionfs_setattr,
// If you have problems with these lines, try defining FIST_SETXATTR_CONSTVOID
# if defined(UNIONFS_XATTR) && LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
    setxattr:   unionfs_setxattr,
    getxattr:   unionfs_getxattr,
    removexattr:unionfs_removexattr,
    listxattr:  unionfs_listxattr,
# endif
};

struct inode_operations unionfs_main_iops =
{
    permission:	unionfs_permission,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    revalidate:	unionfs_inode_revalidate,
#endif
	setattr:	unionfs_setattr,
// If you have problems with these lines, try defining FIST_SETXATTR_CONSTVOID
# if defined(UNIONFS_XATTR) && LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20)
    setxattr:   unionfs_setxattr,
    getxattr:   unionfs_getxattr,
    removexattr:unionfs_removexattr,
    listxattr:  unionfs_listxattr,
# endif
};

/*
 * vim:shiftwidth=4
 * vim:tabstop=8
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
