/****************************************************************************
 *
 * 			camera.cc: Camera implementation
 *      This is part of the yafray package
 *      Copyright (C) 2002  Alejandro Conty Estvez
 *
 *      This library is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU Lesser General Public
 *      License as published by the Free Software Foundation; either
 *      version 2.1 of the License, or (at your option) any later version.
 *
 *      This library is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *      Lesser General Public License for more details.
 *
 *      You should have received a copy of the GNU Lesser General Public
 *      License along with this library; if not, write to the Free Software
 *      Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "camera.h"

__BEGIN_YAFRAY

camera_t::camera_t(const point3d_t &pos, const point3d_t &look,
		 const point3d_t &up, int _resx, int _resy, PFLOAT df, PFLOAT ap, PFLOAT dofd, bool useq, cameraType ct)
{
	_eye = _position = pos;
	aperture = ap;
	dof_distance = dofd;
	camtype = ct;
	resx = _resx;
	resy = _resy;
	vup = up - pos;
	vto = look - pos;
	vright = vup ^ vto;
	vup = vright ^ vto;
	vup.normalize();
	vright.normalize();
	vright *= -1.0; // vright is negative here
	fdist = vto.normLen();

	dof_rt = vright * aperture; // for dof, premul with aperture
	dof_up = vup * aperture;

	vup *= (PFLOAT)resy / (PFLOAT)resx; // vup*yratio

	// for orthocam
	dir_O = vto;
	PFLOAT idf = fdist / df;
	eye_O = _eye - 0.5 * idf * (vup + vright);

	vto = (vto * df) - 0.5 * (vup + vright);
	vup /= (PFLOAT)resy;
	vright /= (PFLOAT)resx;	

	vright_O = vright * idf;
	vup_O = vup * idf;

	focal_distance = df;
	HSEQ1.setBase(2);
	HSEQ2.setBase(3);
	use_qmc = useq;


}

camera_t::~camera_t() 
{
}

vector3d_t  camera_t::shootRay(PFLOAT px, PFLOAT py)
{
	vector3d_t ray;
	if (camtype==CAM_ORTHO) {
		_position = vright_O*px + vup_O*py + eye_O;
		ray = dir_O;
	}
	else {
	  _position = _eye;
		ray = vright*px + vup*py + vto;
		ray.normalize();
	}

	if (aperture!=0) {
		PFLOAT u, v;
		if (use_qmc)
			ShirleyDisc(HSEQ1.getNext(), HSEQ2.getNext(), u, v);
		else
			ShirleyDisc(ourRandom(), ourRandom(), u, v);
		vector3d_t LI = dof_rt * u + dof_up * v;
		_position += point3d_t(LI);
		ray = (ray * dof_distance) - LI;
		ray.normalize();
	}

	return ray;
}
__END_YAFRAY
