#include "decl.h"
#include "process.h"
#include "format.h"

static gboolean handle_redefinition(Namespace *ns, const char *ident);

Namespace *decl_get_root_namespace(void)
{
  static Namespace *root_ns = NULL;
  
  if (!root_ns)
  {
    root_ns = g_new(Namespace, 1);
    root_ns->common.lineno = 0;
    root_ns->common.filename = "<undefined>";
    root_ns->parent = NULL;
    root_ns->identifier = "/";
    root_ns->members = g_hash_table_new(&g_str_hash, &g_str_equal);
  }

  return root_ns;
}

Namespace *decl_namespace(Namespace *parent, const char *ident,
                          const char *fname, int lineno)
{
  Namespace *ns;
  
  if (!parent)
    parent = decl_get_root_namespace();

  if (handle_redefinition(parent, ident))
    return NULL;
  
  ns = g_new(Namespace, 1);
  ns->common.id = DECL_NAMESPACE;
  ns->common.lineno = lineno;
  ns->common.filename = fname;
  ns->parent = parent;
  ns->identifier = ident;
  ns->members = g_hash_table_new(&g_str_hash, &g_str_equal);

  g_hash_table_insert(parent->members, (gpointer)ident, ns);

  return ns;
}

Class *decl_class(Namespace *ns, const char *ident, Tree defined_by,
                  gboolean definition, const char *fname, int lineno)
{
  Class *klass;
  
  if (!ns)
    ns = decl_get_root_namespace();

  if (handle_redefinition(ns, ident))
    return NULL;
  
  klass = g_new(Class, 1);
  klass->common.id = DECL_CLASS;
  klass->common.lineno = lineno;
  klass->common.filename = fname;
  klass->defined_by = defined_by;
  klass->scope = ns;
  klass->identifier = ident;
  klass->definition = definition;
  
  g_hash_table_insert(ns->members, (gpointer)ident, klass);

  return klass;
}

static gboolean superscope(Namespace *super, Namespace *scope)
{
  while (scope)
  {
    if (scope == super)
      return TRUE;
    scope = scope->parent;
  }
  return FALSE;
}

Class *decl_find_class(Namespace *scope, Tree typname, char **scope_name)
{
  Tree node = typname;
  Namespace *curscope = scope;
  int exact = FALSE;
  
  if (TREE_VALUE(node) == NULL)
  {
    exact = TRUE;
    curscope = decl_get_root_namespace();
    node = TREE_CHAIN(node);
  }

  while (1)
  {
    Tree id = TREE_TYPENAME_IDENTIFIER(TREE_VALUE(node));
    Member *member =
      (Member *)g_hash_table_lookup(curscope->members,
                                    TREE_IDENTIFIER_STR(id));
    if (!member)
    {
      if (exact || !curscope->parent)
      {
        process_error("unknown identifier '%s'", format_typename(typname));
        return NULL;
      }
      /* go into outer scope */
      curscope = curscope->parent;
      continue;
    }
    
    if (member->common.id == DECL_NAMESPACE)
    {
      if (TREE_CHAIN(node) == NULL_TREE)
      {
        process_error("'%s' is not a class", format_typename(typname));
        return NULL;
      }
      /* go into inner scope */
      node = TREE_CHAIN(node);
      curscope = (Namespace *)member;
      exact = TRUE;
    }
    else if (member->common.id == DECL_CLASS)
    {
      if (TREE_CHAIN(node) != NULL_TREE)
      {
        process_error("invalid typename '%s'", format_typename(typname));
        return NULL;
      }
      if (scope_name)
      {
        if (((Class *)member)->definition && superscope(curscope, scope))
        {
          Tree id = TREE_TYPENAME_IDENTIFIER(TREE_VALUE(node));
          *scope_name = (char *)TREE_IDENTIFIER_STR(id);
        }
        else
          *scope_name = NULL;
      }
      return (Class *)member;
    }
    else
      process_internal_error("unknown declaration");
  }
}


static gboolean handle_redefinition(Namespace *ns, const char *ident)
{
  Member *member = (Member *)g_hash_table_lookup(ns->members, ident);
  
  if (member != NULL)
  {
    process_error("redefinition of '%s'", ident);
    process_error_cont(member->common.filename, member->common.lineno,
                       "previous definition here");
    return TRUE;
  }
  return FALSE;
}
