#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif
#include "../include/string.h"
#include "../include/disk.h"
#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "obj.h"
#include "win.h"    
#include "wincb.h"
#include "windnd.h"
#include "winlist.h"
#include "windb.h"
#include "winopcb.h"
#include "core.h"
#include "cfglist.h"

#include "images/icon_avscan_32x32.xpm"
#include "images/icon_biohazard_32x32.xpm"


static const gchar *SEEK_PAST_ARG(const gchar *s);
static const gchar *SEEK_PAST_SPACES(const gchar *s);
static const gchar *GET_ARG(
	const gchar *s, const gchar end_c, 
	gchar **arg_rtn
);
static const gchar *GET_NAME_FROM_PATH(const gchar *path);
static gchar *SHORTEN_STRING(const gchar *s, const gint m);

void WinScanListItemDestroyCB(gpointer data);
void WinResultsListItemDestroyCB(gpointer data);

gint WinEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint WinOPIDEnterEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint WinOPIDLeaveEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void WinToolBarItemCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void WinToolBarItemEnterCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void WinToolBarItemLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
);

void WinNotebookSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
);

gint WinEntryCompletePathKeyCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);

gint WinListEventCB(GtkWidget *widget, GdkEvent *event, gpointer data);
void WinListResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
);
void WinListClickColumnCB(
	GtkCList *clist, gint column, gpointer data             
);
void WinListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void WinListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

void WinTreeSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
void WinTreeUnselectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
void WinTreeExpandCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);
void WinTreeCollapseCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);

static GtkCTreeNode *WinDBFindIterate(
	GtkCTree *ctree,
	const gchar *needle, GtkCTreeNode *node,
	GtkCTreeNode **sel_node
);
void WinDBFindEntryCB(GtkWidget *widget, gpointer data);
gint WinDBTreeLoadStartIdleCB(gpointer data);

gint WinScanMonitorTOCB(gpointer data);
gint WinUpdateMonitorTOCB(gpointer data);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Seeks past non-blank characters until a blank character or
 *	null character is encountered.
 */
static const gchar *SEEK_PAST_ARG(const gchar *s)
{
	if(s == NULL)
	    return(NULL);

	while(!ISBLANK(*s) && (*s != '\0'))
	    s++;

	return(s);
}

/*
 *	Seeks past blank characters.
 */
static const gchar *SEEK_PAST_SPACES(const gchar *s)
{
	if(s == NULL)
	    return(NULL);

	while(ISBLANK(*s))
	    s++;

	return(s);
}

/*
 *	Gets a dynamically allocated string *arg_rtn containing the
 *	string from s to the next encountered character specified by
 *	end_c.
 *
 *	If end_c is not encountered then a copy of s is returned as
 *	*arg_rtn.
 *
 *	Returns the pointer in s past end_c if end_c was encountered or
 *	NULL if end_c was not encountered.
 */
static const gchar *GET_ARG(
	const gchar *s, const gchar end_c,
	gchar **arg_rtn
)
{
	gint len;
	const gchar *s_end;

	if(arg_rtn != NULL)
	    *arg_rtn = NULL;

	if(s == NULL)
	    return(NULL);

	s_end = strchr(s, end_c);
	len = (s_end != NULL) ? (s_end - s) : STRLEN(s);

	if(arg_rtn != NULL)
	{
	    gchar *arg = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(len > 0)
		memcpy(arg, s, len * sizeof(gchar));
	    arg[len] = '\0';
	    *arg_rtn = arg;
	}

	return((s_end != NULL) ? (s_end + 1) : NULL);
}

/*
 *      Returns the name portion of the path (if any).
 */
static const gchar *GET_NAME_FROM_PATH(const gchar *path)
{
	const gchar *s;

	if(STRISEMPTY(path))
	    return(path);
			 
	s = strrchr(path, G_DIR_SEPARATOR);
	return((s != NULL) ? (s + 1) : path);
}

/*
 *	Returns a copy of the string that is no longer than m
 *	characters with appropriate shortened notation where
 *	appropriate.
 */
static gchar *SHORTEN_STRING(const gchar *s, const gint m)
{
	gint len;

	if(s == NULL)
	    return(NULL);

	len = STRLEN(s);
	if((len > m) && (m > 3))
	{
	    /* Need to shorten string */
	    const gint i = len - m + 3;
	    return(g_strdup_printf(
		"...%s", (const gchar *)(s + i)
	    ));
	}
	else
	{
	    return(STRDUP(s));
	}
}


/*
 *	Win Scan List item destroy signal callback.
 */
void WinScanListItemDestroyCB(gpointer data)
{
	ObjDelete(OBJ(data));
}

/*
 *	Win Results List item destroy signal callback.
 */
void WinResultsListItemDestroyCB(gpointer data)
{
	ObjDelete(OBJ(data));
}


/*
 *	Win toplevel GtkWindow event signal callback.
 */
gint WinEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean press;
	guint keyval, state;
	GdkEventKey *key;
	win_struct *win = WIN(data);
	if((widget == NULL) || (event == NULL) || (win == NULL))
	    return(status);


	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    /* If there a scan, database load, or virus database update
	     * in progress then call the stop callback instead of
	     * the close callback
	     */
	    if(WinScanProcessIsRunning(win) ||
	       (win->db_ctree_load_idleid > 0) ||
	       WinUpdateProcessIsRunning(win)
	    )
		WinStopCB(win->stop_mi, win);
	    else
		WinCloseCB(win->close_mi, win);
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    press = (event->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
	    state = key->state;
#define DO_STOP_KEY_SIGNAL_EMIT {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?                                       \
   "key_press_event" : "key_release_event"      \
 );						\
}
	    switch(keyval)
	    {
	      case GDK_Escape:
		if(press)
		    WinStopCB(win->stop_mi, win);
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	    }
#undef DO_STOP_KEY_SIGNAL_EMIT
	    break;
	}

	return(status);
}

/*
 *	Win OPID "enter_notify_event" signal callback.
 */
gint WinOPIDEnterEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	win_opid_struct *opid = WIN_OPID(data);
	if(opid == NULL)
	    return(FALSE);

	WinStatusMessage(opid->win, opid->tooltip, FALSE);
	return(TRUE);
}

/*
 *	Win OPID "leave_notify_event" signal callback.
 */
gint WinOPIDLeaveEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	win_opid_struct *opid = WIN_OPID(data);
	if(opid == NULL)
	    return(FALSE);

	WinStatusMessage(opid->win, NULL, FALSE);
	return(TRUE);
}


/*
 *	Win Tool Bar Item callback.
 */
void WinToolBarItemCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	win_opid_struct *opid = WIN_OPID(data);
	if(opid == NULL)
	    return;

	if(opid->func_cb != NULL)
	    opid->func_cb(
		NULL,		/* GtkWidget */
		opid->win	/* Win */
	    );
}

/*
 *	Win Tool Bar Item enter callback.
 */
void WinToolBarItemEnterCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	WinOPIDEnterEventCB(NULL, NULL, data);
}

/*
 *	Win Tool Bar Item leave callback.
 */
void WinToolBarItemLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	WinOPIDLeaveEventCB(NULL, NULL, data);
}


/*
 *	Win notebook "switch_page" signal callback.
 */
void WinNotebookSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if(win->page_num != (win_page_num)page_num)
	{
	    win->page_num = (win_page_num)page_num;
	    switch(page_num)
	    {
	      case WIN_PAGE_NUM_SCAN:
		break;

	      case WIN_PAGE_NUM_RESULTS:
		break;

	      case WIN_PAGE_NUM_DB:
		WinDBQueueUpdate(win);
		break;
	    }
	    WinUpdate(win);
	}
}


/*
 *	Entry complete path "key_press_event" or "key_release_event"
 *	signal callback.
 */
gint WinEntryCompletePathKeyCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean press;
	GtkEntry *entry = GTK_ENTRY(widget);
	win_struct *win = WIN(data);
	if((entry == NULL) || (key == NULL) || (win == NULL))
	    return(status);

	press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;

#define SIGNAL_EMIT_STOP	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	switch(key->keyval)
	{
	  case GDK_Tab:
	    if(press)
	    {
		gchar *s = STRDUP(gtk_entry_get_text(entry));
		if(s != NULL)
		{
		    gint status;

		    s = CompletePath(s, &status);
		    gtk_entry_set_text(entry, (s != NULL) ? s : "");
		    gtk_entry_set_position(entry, -1);
		    g_free(s);

		    switch(status)
		    {
		      case COMPLETE_PATH_NONE:
		      case COMPLETE_PATH_AMBIGUOUS:
			gdk_beep();
			break;
		    }
		}
	    }
	    SIGNAL_EMIT_STOP
	    status = TRUE;
	    break;
	}

#undef SIGNAL_EMIT_STOP

	return(status);
}


/*
 *	Win list event signal callback.
 */
gint WinListEventCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gint status = FALSE;
	gboolean press;
	guint keyval, state;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	win_struct *win = WIN(data);
	if((widget == NULL) || (event == NULL) || (win == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    press = (event->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
	    state = key->state;
#define DO_STOP_KEY_SIGNAL_EMIT {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?                                       \
   "key_press_event" : "key_release_event"      \
 );						\
}
	    switch(keyval)
	    {
	      case GDK_Insert:
		if(press)
		{
		    if(state & GDK_CONTROL_MASK)
			WinCopyCB(win->copy_mi, win);
		    else if(state & GDK_SHIFT_MASK)
			WinPasteCB(win->paste_mi, win);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	      case GDK_Delete:
		if(press)
		    WinRemoveCB(win->delete_mi, win);
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;
	    }
#undef DO_STOP_KEY_SIGNAL_EMIT
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case 1:
		break;

	      case 2:
		break;

	      case 3:
		if(widget == win->scan_clist)
		{
		    GtkMenu *menu = GTK_MENU(win->scan_list_menu);
		    gtk_menu_popup(
			menu, NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
 		}
		else if(widget == win->results_clist)
		{
		    GtkMenu *menu = GTK_MENU(win->results_list_menu);
		    gtk_menu_popup(
			menu, NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		}
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_2BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case 1:
		if(widget == win->scan_clist)
		{
		    WinStartCB(NULL, win);
		}
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;

	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;

	    break;
	}

	return(status);
}


/*
 *	Win GtkCList "resize_column" signal callback.
 */
void WinListResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;


}

/*
 *	Win GtkCList "click_column" signal callback.
 */
void WinListClickColumnCB(
	GtkCList *clist, gint column, gpointer data             
)
{
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;

	gtk_clist_freeze(clist);

	gtk_clist_set_sort_column(clist, column);
	gtk_clist_set_sort_type(
	    clist,
	    (clist->sort_type == GTK_SORT_ASCENDING) ?
		GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
	);
	gtk_clist_sort(clist);

	gtk_clist_thaw(clist);
}


/*
 *	Win GtkCList "select_row" signal callback.
 */
void WinListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	core_struct *core;
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;

	core = CORE(win->core);
	if(core == NULL)
	    return;

	/* If the row is not visible then scroll to make it visible */
	if(gtk_clist_row_is_visible(clist, row) !=
	    GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column */
		0.5f, 0.0f	/* Row, column */
	    );

	/* Scan List */
	if(clist == (GtkCList *)win->scan_clist)
	{
	    const obj_struct *obj = OBJ(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
	    {
		gchar *buf;
		const gint nobjs = g_list_length(clist->selection);
 
		/* Update scan settings */
		WinScanSettingsUpdate(win, obj);

		/* Update status bar message */
		if(nobjs > 1)
		    buf = g_strdup_printf(
			"%i scan items selected",
			nobjs
		    );
		else if(!STRISEMPTY(obj->name))
		    buf = g_strdup_printf(
			"Scan item \"%s\" selected",
			obj->name
		    );
		else
		    buf = STRDUP("Scan item (Untitled) selected");
		WinStatusMessage(win, buf, FALSE);
		g_free(buf);
	    }

	    WinListDNDSetIcon(clist, row, column);
	    WinUpdate(win);
	}
	/* Results List */
	else if(clist == (GtkCList *)win->results_clist)
	{
	    const obj_struct *obj = OBJ(
		gtk_clist_get_row_data(clist, row)
	    );
	    if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	    {
		gchar *buf;
		const gint nobjs = g_list_length(clist->selection);
		const gchar	*path = obj->path,
				*name = GET_NAME_FROM_PATH(path);
		struct stat stat_buf;

		if(nobjs > 1)
		    buf = g_strdup_printf(
			"%i objects selected",
			nobjs
		    );
		else if(lstat(path, &stat_buf))
		    buf = g_strdup_printf(
			"Object \"%s\" selected",
			name
		    );
#ifdef S_ISREG
		else if(S_ISREG(stat_buf.st_mode))
#else
		else if(FALSE)
#endif
		{
#ifdef HAVE_EDV2
		    gchar *size_str = STRDUP(EDVSizeStrDelim(
			(gulong)stat_buf.st_size
		    ));
#else
		    gchar *size_str = g_strdup_printf(
			"%ld",
			(gulong)stat_buf.st_size
		    );
#endif
		    buf = g_strdup_printf(
			"File \"%s\" selected (%s byte%s)", 
			name,
			size_str,
			(stat_buf.st_size == 1) ? "" : "s"
		    );
		}
#ifdef S_ISDIR
		else if(S_ISDIR(stat_buf.st_mode))
#else
		else if(FALSE)
#endif
		    buf = g_strdup_printf(
			"Directory \"%s\" selected",
			name
		    );
#ifdef S_ISLNK
		else if(S_ISLNK(stat_buf.st_mode))
#else
		else if(FALSE)
#endif
		    buf = g_strdup_printf(
			"Link \"%s\" selected",
			name
		    );
		else
		    buf = g_strdup_printf(
			"Object \"%s\" selected",
			name
		    );

		WinStatusMessage(win, buf, FALSE);
		g_free(buf);
	    }
	    WinListDNDSetIcon(clist, row, column);
	    WinUpdate(win);
	}
}

/*
 *	Win GtkCList "unselect_row" signal callback.
 */
void WinListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;

	/* Scan List */
	if(clist == (GtkCList *)win->scan_clist)
	{
	    WinStatusMessage(win, NULL, FALSE);
	    WinUpdate(win);
	}
	/* Results List */
	else if(clist == (GtkCList *)win->results_clist)
	{
	    WinStatusMessage(win, NULL, FALSE);
	    WinUpdate(win);
	}
}


/*
 *      Win GtkCTree "tree_select_row" signal callback.
 */
void WinTreeSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;

	/* If the node is not visible then scroll to make it visible */
	if(gtk_ctree_node_is_visible(ctree, node)
	   != GTK_VISIBILITY_FULL
	)
	    gtk_ctree_node_moveto(
		ctree,
		node, -1,
		0.5f, 0.0f	/* Row align, column align */
	    );

	/* Virus Database Tree */
	if(ctree == (GtkCTree *)win->db_ctree)
	{
	    WinUpdate(win);
	}
}

/*
 *      Win GtkCTree "tree_unselect_row" signal callback.
 */
void WinTreeUnselectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data                                    
)
{
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;

	/* Virus Database Tree */
	if(ctree == (GtkCTree *)win->db_ctree)
	{
	    WinUpdate(win);
	}
}
 
/*
 *      Win GtkCTree "tree_expand" signal callback.
 */
void WinTreeExpandCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;

}

/*
 *      Win GtkCTree "tree_collapse" signal callback.
 */
void WinTreeCollapseCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{ 
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;


}


/*
 *	Searches for a node who's text matches the specified text
 *	in the specified node and the node's child nodes.
 */
static GtkCTreeNode *WinDBFindIterate(
	GtkCTree *ctree,
	const gchar *needle, GtkCTreeNode *node,
	GtkCTreeNode **sel_node
)
{
	gchar *text;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkCTreeRow *row;

	while(node != NULL)
	{
	    row = GTK_CTREE_ROW(node);
	    if(row == NULL)
		break;

	    /* Past selected node? */
	    if(*sel_node == NULL)
	    {
		/* Get this node's text and see if it matches the
		 * specified text
		 */
		gtk_ctree_node_get_pixtext(
		    ctree, node, ctree->tree_column,
		    &text, &spacing, &pixmap, &mask
		);
		if(text != NULL)
		{
		    if(strcasestr(text, needle) != NULL)
			return(node);
		}
	    }
	    /* Reached the selected node? */
	    else if(*sel_node == node)
	    {
		/* Set the selected node to NULL to indicate that it has
		 * been reached
		 */
		*sel_node = NULL;
	    }

	    /* Does this node have child nodes? */
	    if(row->children != NULL)
	    {
		/* Check all of this node's child nodes */
		GtkCTreeNode *matched_node = WinDBFindIterate(
		    ctree, needle, row->children, sel_node
		);
		if(matched_node != NULL)
		    return(matched_node);
	    }

	    /* Get next sibling */
	    node = row->sibling;
	}

	return(NULL);
}

/*
 *	Win Virus Database Find GtkEntry "activate" signal callback.
 */
void WinDBFindEntryCB(GtkWidget *widget, gpointer data)
{
	gchar *needle;
	GList *glist;
	GtkWidget *toplevel;
	GtkCTreeNode *node, *sel_node;
	GtkCList *clist;
	GtkCTree *ctree;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;

	/* Get copy of search string */
	needle = STRDUP(
	    gtk_entry_get_text(GTK_ENTRY(win->db_find_entry))
	);
	if(needle == NULL)
	    return;

	ctree = GTK_CTREE(win->db_ctree);
	clist = GTK_CLIST(ctree);

	/* Get last selected node or root node */
	glist = clist->selection_end;
	sel_node = (GtkCTreeNode *)((glist != NULL) ? glist->data : NULL);
	if(sel_node != NULL)
	{
	    node = WinDBFindIterate(
		ctree, needle, win->db_root_node, &sel_node
	    );
	    /* If no node was matched then start from the beginning */
	    if(node == NULL)
	    {
		sel_node = NULL;
		node = WinDBFindIterate(
		    ctree, needle, win->db_root_node, &sel_node
		);
	    }
	}
	else
	{
	    /* Search for node starting from the root node */
	    node = WinDBFindIterate(
		ctree, needle, win->db_root_node, &sel_node
	    );
	}

	/* Got match? */
	if(node != NULL)
	{
	    gtk_ctree_select(ctree, node);
	}
	else
	{
	    gchar *buf = g_strdup_printf(
		"The text \"%s\" was not found",
		needle
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Find Results",
		buf,
		NULL,
		CDIALOG_ICON_SEARCH,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	}

	g_free(needle);
}

/*
 *	Win Virus Database Tree load start idle callback.
 */
gint WinDBTreeLoadStartIdleCB(gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	/* Update the Virus Database Tree as needed */
	if(win->db_root_node == NULL)
	{
	    WinDBUpdate(win, NULL, TRUE);
	}

	win->db_ctree_load_idleid = 0;
	WinUpdate(win);

	return(FALSE);
}

/*
 *	Win scan monitoring timeout callback.
 */
gint WinScanMonitorTOCB(gpointer data)
{
	gint p;
	gchar buf[1024];
	FILE *fp;
	GtkWidget *toplevel;
	GtkCList *clist;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	p = win->scan_pid;
	toplevel = win->toplevel;
	clist = (GtkCList *)win->results_clist;
	core = CORE(win->core);

	/* User clicked on the stop button? */
	if(win->stop_count > 0)
	{
	    gint new_row;

	    /* Stop scanning
	     *
	     * Mark that this timeout is no longer being called
	     */
	    win->scan_toid = 0;

	    new_row = WinResultsListAppend(win, NULL, "Interrupted");
	    if(new_row > -1)
	    {
		gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
	    }

	    /* Tell the scanning process to stop scanning, delete
	     * all the tempory output files, and clean up
	     */
	    WinScanProcessStop(win);

	    WinStatusMessage(
		win,
"Scan interrupted",
		FALSE
	    );

	    return(FALSE);
	}

	/* Read stderr */
	fp = win->stderr_fp;
	if(fp != NULL)
	{
	    /* Format for each line from stderr is either:
	     *
	     *     "<path>" <virus|problem>
	     *
	     * or:
	     *
	     *     <problem>
	     *
	     * Example:
	     *
	     *     "/home/ftp/incoming/runme" Some virus detected!
	     *
	     *     The virus engine is out of date, upgrade!
	     */
	    gint new_row;
	    gchar *s_end, *path, *virus_name;
	    const gchar *s;

	    while(fgets(buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		s_end = strpbrk(buf, "\n\r");
		if(s_end != NULL)
		    *s_end = '\0';

		path = NULL;			/* Path */
		virus_name = NULL;		/* Virus|problem */

		/* Seek past initial spaces */
		s = SEEK_PAST_SPACES(buf);

		/* Check format, if the first character is a "
		 * then it implies the format is:
		 *
		 *     "path" <virus|problem>
		 */
		if(*s == '"')
		{
		    s++;	/* Seek past initial '"' character */

		    /* Get the path and increment s past the next '"'
		     * character
		     */
		    s = GET_ARG(s, '"', &path);

		    s = SEEK_PAST_SPACES(s);

		    /* Get the virus/problem */
		    virus_name = STRDUP(s);
		}
		else
		{
		    /* No path, just get the whole line as the virus
		     * name
		     */
		    virus_name = STRDUP(s);
		}

		new_row = WinResultsListAppend(win, path, virus_name);
		if(new_row > -1)
		{
		    gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
		}

		g_free(path);
		g_free(virus_name);
	    }
	}

	/* Read stdout */
	fp = win->stdout_fp;
	if(fp != NULL)
	{
	    /* Each line from stdout must have its prefix checked and
	     * handled accordingly
	     */
	    gchar *s, *s2;

	    while(fgets(buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		s = strpbrk(buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		/* Scanning file? */
		if(g_strcasepfx(s, "Scanning:") ||
		   g_strcasepfx(s, "Scanning file:")
		)
		{
		    /* Format is:
		     *
		     * "<pfx>: "<path>" <files_scanned> <total_files>"
		     */
		    const gchar *s2 = s;
		    gchar *pfx = NULL, *path = NULL, *msg;
		    gulong i = 0l, n = 0l;

		    /* Get the prefix */
		    if(s2 != NULL)
		    {
			s2 = GET_ARG(s2, ':', &pfx);
		    }

		    /* Get the path */
		    if(s2 != NULL)
		    {
			gchar *path2;

			s2 = strchr(s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);

			/* Shorten the path as needed */
			path2 = SHORTEN_STRING(path, 40);
			g_free(path);
			path = path2;
		    }

		    /* Get the number of files scanned */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_SPACES(s2);
			win->scan_files_scanned = i = (gulong)ATOL(s2);
		    }

		    /* Get the total files */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			win->scan_files_total = n = (gulong)ATOL(s2);
		    }

		    msg = g_strdup_printf(
			"%s (%ld/%ld): %s",
			pfx, i + 1l, n, path
		    );
		    WinStatusMessage(win, msg, FALSE);
		    g_free(msg);

		    WinStatusProgress(
			win,
			(n > 0l) ? ((gfloat)i / (gfloat)n) : 0.0f,
			TRUE
		    );

		    g_free(path);
		    g_free(pfx);
		}
		/* Scanning directory? */
		else if(g_strcasepfx(s, "Scanning directory:"))
		{
#if 0
/* We ignore the scanning directory because it usually appears very
 * brief and makes subsequent messages difficult to read in the cases
 * of scanning multiple subdirectories
 */
		    /* Format is:
		     *
		     * "<pfx>: "<path>""
		     */
		    const gchar *s2 = s;
		    gchar *pfx = NULL, *path = NULL, *msg;

		    /* Get the prefix */
		    if(s2 != NULL)
		    {
			s2 = GET_ARG(s2, ':', &pfx);
		    }

		    /* Get the path */
		    if(s2 != NULL)
		    {
			gchar *path2;

			s2 = strchr(s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);

			/* Shorten the path as needed */
			path2 = SHORTEN_STRING(path, 45);
			g_free(path);
			path = path2;
		    }

		    msg = g_strdup_printf(
			"%s: %s",
			pfx, path
		    );
		    WinStatusMessage(win, msg, TRUE);
		    g_free(msg);

		    g_free(path);
		    g_free(pfx);
#endif
		}
		/* Report */
		else if(g_strcasepfx(s, "Report:"))
		{
		    /* Seek past prefix */
		    s = strchr(s, ':');
		    s++;
		    while(ISBLANK(*s))
			s++;

		    /* Replace all ';' with newline characters */
		    s2 = strchr(s, ';');
		    while(s2 != NULL)
		    {
			*s2 = '\n';
			s2 = strchr(s, ';');
		    }

#ifdef HAVE_EDV2
		    /* Play "completed" sound */
		    EDVPlaySoundCompleted(core->edv_ctx);
#endif
		    /* Raise Win */
		    WinMap(win);

		    WinStatusProgress(win, 1.0f, FALSE);
		    WinStatusMessage(win, "Scan done", FALSE);

		    /* Display message in dialog */
		    if(!CDialogIsQuery())
		    {
			CDialogSetTransientFor(toplevel);
			CDialogGetResponseIconData(
			    "Scan Results", s, NULL,
			    (guint8 **)icon_avscan_32x32_xpm,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		    }
		}
		/* Report infected or errors */
		else if(g_strcasepfx(s, "ReportInfected:") ||
		        g_strcasepfx(s, "ReportError:")
		)
		{
		    /* Seek past prefix */
		    s = strchr(s, ':');
		    s++;
		    while(ISBLANK(*s))
			s++;

		    /* Replace all ';' with newline characters */
		    s2 = strchr(s, ';');
		    while(s2 != NULL)
		    {
			*s2 = '\n';
			s2 = strchr(s, ';');
		    }

#ifdef HAVE_EDV2
		    /* Play "warning" sound */
		    EDVPlaySoundWarning(core->edv_ctx);
#endif
		    /* Raise Win */
		    WinMap(win);

		    WinStatusProgress(win, 1.0f, FALSE);
		    WinStatusMessage(win, "Scan done", FALSE);

		    /* Display message in dialog */
		    if(!CDialogIsQuery())
		    {
			CDialogSetTransientFor(toplevel);
			CDialogGetResponseIconData(
			    "Scan Results", s, NULL,
			    (guint8 **)icon_biohazard_32x32_xpm,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		    }
		}
		/* All else print to the status bar's message */
		else
		{
		    /* Print this line to the status bar's message */
		    WinStatusMessage(win, s, TRUE);
		}
	    }
	}

	/* Process has exited? */
	if(!WinScanProcessIsRunning(win))
	{
	    /* Stop scanning */
	    win->scan_toid = 0;
	    WinScanProcessStop(win);

	    WinStatusMessage(
		win,
"Scan done",
		FALSE
	    );

	    return(FALSE);
	}

	/* Update the progress only if the total number of files has
	 * not been received yet
	 */
	if(win->scan_files_total == 0l)
	    WinStatusProgress(win, -1.0f, FALSE);

	/* Update the time */
	if(win->scan_start_time > 0l)
	{
	    const gulong t = (gulong)time(NULL);
	    WinStatusTime(
		win,
		(t > win->scan_start_time) ? (t - win->scan_start_time) : 0l,
		FALSE
	    );
	}

	return(TRUE);
}

/*
 *	Win virus database update monitoring timeout callback.
 *
 *	Monitors the virus database update process and prints its
 *	stdout and stderr messages to the status bar.
 */
gint WinUpdateMonitorTOCB(gpointer data)
{
	gint p;
	gchar buf[1024];
	FILE *fp;
	GtkWidget *toplevel;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	p = win->update_pid;
	toplevel = win->toplevel;
	core = CORE(win->core);

	/* Need to stop? */
	if(win->stop_count > 0)
	{
	    /* Stop update */
	    win->update_toid = 0;
	    WinUpdateProcessStop(win);

	    WinStatusMessage(win, "Virus database update interrupted", FALSE); 
	    return(FALSE);
	}

	/* Read stdout */
	fp = win->stdout_fp;
	if(fp != NULL)
	{
	    gchar *s;

	    while(fgets(buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		/* Strip newline characters */
		s = strpbrk(buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		WinStatusMessage(win, s, TRUE);
	    }
	}

	/* Read stderr */
	fp = win->stderr_fp;
	if(fp != NULL)
	{
	    gint new_row;
	    gchar *s;
	    GtkCList *clist = GTK_CLIST(win->results_clist);

	    while(fgets(buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		/* Strip newline characters */
		s = strpbrk(buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		new_row = WinResultsListAppend(
		    win,
		    NULL,			/* No path */
		    s				/* Virus/Problem */
		);
		if(new_row > -1)
		{
		    gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
		}
	    }
	}


	/* Process has exited? */ 
	if(!WinUpdateProcessIsRunning(win))
	{
	    /* Mark the virus database update time */
	    CFGItemListSetValueL(
		core->cfg_list, CFG_PARM_DB_LAST_UPDATE,
		(gulong)time(NULL), FALSE
	    );

#ifdef HAVE_EDV2
	    /* Play "completed" sound */
	    EDVPlaySoundCompleted(core->edv_ctx);
#endif

	    /* Stop the virus database update */
	    win->update_toid = 0;
	    WinUpdateProcessStop(win);

	    WinStatusMessage(win, "Virus database update done", TRUE);


	    /* Clear Virus Database Tree due to virus database update */
	    WinDBClear(win);

	    /* Currently viewing the Virus Database Page? */
	    if(win->page_num == WIN_PAGE_NUM_DB)
	    {
		/* Need to update the Virus Database Tree since it is
		 * currently being viewed
		 *
		 * Note that the Win will be raised after the update
		 */
		WinDBQueueUpdate(win);
	    }

	    return(FALSE);
	}

	/* Update the progress */
	WinStatusProgress(win, -1.0f, FALSE);

	/* Update the time */
	if(win->update_start_time > 0l)
	{
	    const gulong t = (gulong)time(NULL);
	    WinStatusTime(
		win,
		(t > win->update_start_time) ? (t - win->update_start_time) : 0l,
		FALSE
	    );
	}

	return(TRUE);
}
