/*-
 * Copyright (c) 2000-2001 GOTOU YUUZOU <gotoyuzo@notwork.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $IPR: ssl.c,v 1.22 2001/09/21 17:35:51 gotoyuzo Exp $
 */
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <ruby.h>
#include <rubyio.h>
#include <rubysig.h>

#define numberof(ary) (sizeof(ary)/sizeof((ary)[0]))

#define ssl_def_const(x) rb_define_const(mSSL, #x, INT2FIX(SSL_##x))
#define x509_def_const(x) rb_define_const(cX509, #x, INT2FIX(X509_V_ERR_##x))

#define ssl_get_io(o)            rb_ivar_get((o),rb_intern("@io"))
#define ssl_get_cert_file(o)     rb_ivar_get((o),rb_intern("@cert_file"))
#define ssl_get_key_file(o)      rb_ivar_get((o),rb_intern("@key_file"))
#define ssl_get_ca_file(o)       rb_ivar_get((o),rb_intern("@ca_file"))
#define ssl_get_ca_path(o)       rb_ivar_get((o),rb_intern("@ca_path"))
#define ssl_get_timeout(o)       rb_ivar_get((o),rb_intern("@timeout"))
#define ssl_get_verify_mode(o)   rb_ivar_get((o),rb_intern("@verify_mode"))
#define ssl_get_verify_dep(o)    rb_ivar_get((o),rb_intern("@verify_depth"))
#define ssl_get_verify_cb(o)     rb_ivar_get((o),rb_intern("@verify_callback"))

#define ssl_set_io(o,v)          rb_ivar_set((o),rb_intern("@io"),(v))
#define ssl_set_cert_file(o,v)   rb_ivar_set((o),rb_intern("@cert_file"),(v))
#define ssl_set_key_file(o,v)    rb_ivar_set((o),rb_intern("@key_file"),(v))
#define ssl_set_ca_file(o,v)     rb_ivar_set((o),rb_intern("@ca_file"),(v))
#define ssl_set_ca_path(o,v)     rb_ivar_set((o),rb_intern("@ca_path"),(v))
#define ssl_set_timeout(o,v)     rb_ivar_set((o),rb_intern("@timeout"),(v))
#define ssl_set_verify_mode(o,v) rb_ivar_set((o),rb_intern("@verify_mode"),(v))
#define ssl_set_verify_dep(o,v)  rb_ivar_set((o),rb_intern("@verify_depth"),(v))
#define ssl_set_verify_cb(o,v)   rb_ivar_set((o),rb_intern("@verify_callback"),(v))

char *ssl_attrs[] = {
  /* "io", */ "cert_file", "key_file", "ca_file", "ca_path", 
  "timeout", "verify_mode", "verify_depth", "verify_callback"
}; 

VALUE mSSL;
VALUE eSSLError;
VALUE cSSLSocket;
VALUE cX509;
VALUE cX509_STORE_CTX;

typedef struct ssl_st_t{
    SSL     *ssl;
    SSL_CTX *ctx;
} ssl_st;

static void
ssl_shutdown(p)
    ssl_st *p;
{
    if(p->ssl){
        SSL_shutdown(p->ssl);
        SSL_clear(p->ssl);
    }
}

static void
ssl_free(p)
    ssl_st *p;
{
    ssl_shutdown(p);
    SSL_free(p->ssl);
    SSL_CTX_free(p->ctx);
    free(p);
}

typedef struct x509_st_t{
    X509 *cert;
} x509_st;

static void
x509_free(p)
    x509_st *p;
{
    X509_free(p->cert);
    free(p);
}

typedef struct x509_store_ctx_st_t{
    X509_STORE_CTX  *ctx;
} x509_store_ctx_st;

static const char *
ssl_err_str(void)
{
    return ERR_error_string(ERR_get_error(), NULL);
}

static VALUE ssl_verify_callback_proc;

static int
ssl_verify_callback(int ok, X509_STORE_CTX *ctx)
{
    if(!NIL_P(ssl_verify_callback_proc)){
        x509_store_ctx_st *p;
        VALUE x509stc, ret;
        x509stc = Data_Make_Struct(cX509_STORE_CTX,x509_store_ctx_st,0,free,p);
        p->ctx = ctx;
        ret = rb_funcall(ssl_verify_callback_proc, rb_intern("call"), 2,
                         ok ? Qtrue : Qfalse, x509stc);
        p->ctx = NULL;
        ok = (ret == Qtrue) ? 1 : 0;
    }

    return ok;
}

static void
ssl_ctx_setup(self)
    VALUE self;
{
    ssl_st *p;
    char *cert_file, *key_file, *ca_file, *ca_path;
    int verify_mode;
    VALUE val;

    Data_Get_Struct(self, ssl_st, p);

    /* private key may be bundled in certificate file. */
    val = ssl_get_cert_file(self);
    cert_file = NIL_P(val) ? NULL : STR2CSTR(val);
    val = ssl_get_key_file(self);
    key_file = NIL_P(val) ? cert_file : STR2CSTR(val);
    if(cert_file){
        if(SSL_CTX_use_certificate_file(p->ctx,cert_file,SSL_FILETYPE_PEM) < 1)
            rb_raise(eSSLError,"SSL_CTX_use_certificate_file:%s",ssl_err_str());
        if(SSL_CTX_use_PrivateKey_file(p->ctx,key_file,SSL_FILETYPE_PEM) < 1)
            rb_raise(eSSLError,"SSL_CTX_use_PrivateKey_file:%s",ssl_err_str());
        if(!SSL_CTX_check_private_key(p->ctx))
            rb_raise(eSSLError,"SSL_CTX_check_private_key:%s",ssl_err_str());
    }

    val = ssl_get_ca_file(self);
    ca_file = NIL_P(val) ? NULL : STR2CSTR(val);
    val = ssl_get_ca_path(self);
    ca_path = NIL_P(val) ? NULL : STR2CSTR(val);
    if(!SSL_CTX_load_verify_locations(p->ctx, ca_file, ca_path) ||
       !SSL_CTX_set_default_verify_paths(p->ctx) && ruby_verbose){
        rb_warning("can't set verify locations:%s", ssl_err_str());
    }

    val = ssl_get_verify_mode(self);
    verify_mode = NIL_P(val) ? SSL_VERIFY_NONE : NUM2INT(val);
    SSL_CTX_set_verify(p->ctx, verify_mode, ssl_verify_callback);

    val = ssl_get_timeout(self);
    if(!NIL_P(val)) SSL_CTX_set_timeout(p->ctx, NUM2LONG(val));

    val = ssl_get_verify_dep(self);
    if(!NIL_P(val)) SSL_CTX_set_verify_depth(p->ctx, NUM2LONG(val));
}

ssl_setup(self)
    VALUE self;
{
    ssl_st *p;
    VALUE io;
    OpenFile *fptr;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ssl){
        io = ssl_get_io(self);
        GetOpenFile(io, fptr);
        rb_io_check_readable(fptr);
        rb_io_check_writable(fptr);
        if((p->ssl = SSL_new(p->ctx)) == NULL)
            rb_raise(eSSLError, "SSL_new:%s", ssl_err_str());
        SSL_set_fd(p->ssl, fileno(fptr->f));
    }
}

static VALUE
ssl_s_new(argc, argv, klass)
    int argc;
    VALUE *argv;
    VALUE klass;
{
    VALUE obj;
    ssl_st *p;

    obj = Data_Make_Struct(klass, ssl_st, 0, ssl_free, p);
    memset(p, 0, sizeof(ssl_st));
    if((p->ctx = SSL_CTX_new(SSLv23_method())) == NULL)
        rb_raise(eSSLError, "SSL_CTX_new:%s", ssl_err_str());
    SSL_CTX_set_options(p->ctx, SSL_OP_ALL);

    rb_obj_call_init(obj, argc, argv);

    return obj;
}

static VALUE
ssl_initialize(argc, argv, self)
    int argc;
    VALUE *argv;
    VALUE self;
{
    VALUE io, key, cert;

    rb_scan_args(argc, argv, "12", &io, &cert, &key);

    if(!NIL_P(io)) Check_Type(io, T_FILE);
    ssl_set_io(self, io);
    ssl_set_cert_file(self, cert);
    ssl_set_key_file(self, key);

    return self;
}

static VALUE
ssl_connect(self)
    VALUE self;
{
    ssl_st *p;

    Data_Get_Struct(self, ssl_st, p);
    ssl_ctx_setup(self);
    ssl_setup(self);
    rb_thread_critical = 1;
    ssl_verify_callback_proc = ssl_get_verify_cb(self);
    if(SSL_connect(p->ssl) <= 0){
        rb_thread_critical = 0;
        rb_raise(eSSLError, "SSL_connect:%s", ssl_err_str());
    }
    rb_thread_critical = 0;

    return self;
}

static VALUE
ssl_accept(self)
    VALUE self;
{
    ssl_st *p;

    Data_Get_Struct(self, ssl_st, p);
    ssl_ctx_setup(self);
    ssl_setup(self);
    rb_thread_critical = 1;
    ssl_verify_callback_proc = ssl_get_verify_cb(self);
    if(SSL_accept(p->ssl) <= 0){
        rb_thread_critical = 0;
        rb_raise(eSSLError, "SSL_accept:%s", ssl_err_str());
    }
    rb_thread_critical = 0;

    return self;
}

static VALUE
ssl_read(self, len)
    VALUE self, len;
{
    ssl_st *p;
    size_t ilen, nread = 0;
    char *buf;
    VALUE str;
    int iostate = 1;

    Data_Get_Struct(self, ssl_st, p);
    ilen = NUM2INT(len);
    str = rb_str_new(0, ilen);
    
    if(p->ssl){
        nread = SSL_read(p->ssl, RSTRING(str)->ptr, RSTRING(str)->len);
        if(nread < 0) rb_raise(eSSLError, "SSL_read:%s", ssl_err_str());
    }
    else{
        OpenFile *fptr;
        if(ruby_verbose) rb_warning("SSL session is not started yet.");
        GetOpenFile(ssl_get_io(self), fptr);
        rb_io_check_readable(fptr);
        TRAP_BEG;
        nread = read(fileno(fptr->f), RSTRING(str)->ptr, RSTRING(str)->len);
        TRAP_END;
        if(nread < 0) rb_raise(eSSLError, "read:%s", strerro(errno));
    }
    if(nread == 0) rb_raise(rb_eEOFError, "End of file reached");

    RSTRING(str)->len = nread;
    RSTRING(str)->ptr[nread] = 0;
    OBJ_TAINT(str);

    return str;
}

static VALUE
ssl_write(self, str)
    VALUE self, str;
{
    ssl_st *p;
    size_t nwrite = 0;

    Data_Get_Struct(self, ssl_st, p);
    if(TYPE(str) != T_STRING)
        str = rb_obj_as_string(str);

    if(p->ssl){
        nwrite = SSL_write(p->ssl, RSTRING(str)->ptr, RSTRING(str)->len);
        if(nwrite < 0) rb_raise(eSSLError, "SSL_write:%s", ssl_err_str());
    }
    else{
        OpenFile *fptr;
        FILE *fp;
        if(ruby_verbose) rb_warning("SSL session is not started yet.");
        GetOpenFile(ssl_get_io(self), fptr);
        rb_io_check_writable(fptr);
        fp = GetWriteFile(fptr);
        nwrite = write(fileno(fp), RSTRING(str)->ptr, RSTRING(str)->len);
        if(nwrite < 0) rb_raise(eSSLError, "write:%s", strerro(errno));
    }

    return INT2NUM(nwrite);
}

static VALUE
ssl_close(self)
    VALUE self;
{
    ssl_st *p;

    Data_Get_Struct(self, ssl_st, p);
    ssl_shutdown(p);
    return Qnil;
}

static VALUE
ssl_get_cert(self)
    VALUE self;
{
    ssl_st *p;
    X509 *cert;
    x509_st *x509p;
    VALUE ret;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ssl){
        if(ruby_verbose) rb_warning("SSL session is not started yet.");
        return Qnil;
    }

    if((cert = SSL_get_certificate(p->ssl)) == NULL) return Qnil;
    ret = Data_Make_Struct(cX509, x509_st, 0, x509_free, x509p);
    x509p->cert = X509_dup(cert);

    return ret;
}

static VALUE
ssl_get_peer_cert(self)
    VALUE self;
{
    ssl_st *p;
    X509 *cert;
    x509_st *x509p;
    VALUE ret;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ssl){
        if(ruby_verbose) rb_warning("SSL session is not started yet.");
        return Qnil;
    }

    if((cert = SSL_get_peer_certificate(p->ssl)) == NULL) return Qnil;
    ret = Data_Make_Struct(cX509, x509_st, 0, x509_free, x509p);
    x509p->cert = X509_dup(cert);

    return ret;
}

static VALUE
ssl_cipher_to_ary(cipher)
    SSL_CIPHER *cipher;
{
    VALUE ary;
    int bits, alg_bits;

    ary = rb_ary_new2(4);
    rb_ary_push(ary, rb_str_new2(SSL_CIPHER_get_name(cipher)));
    rb_ary_push(ary, rb_str_new2(SSL_CIPHER_get_version(cipher)));
    bits = SSL_CIPHER_get_bits(cipher, &alg_bits);
    rb_ary_push(ary, INT2FIX(bits));
    rb_ary_push(ary, INT2FIX(alg_bits));

    return ary;
}

static VALUE
ssl_get_cipher(self)
    VALUE self;
{
    ssl_st *p;
    SSL_CIPHER *cipher;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ssl){
        if(ruby_verbose) rb_warning("SSL session is not started yet.");
        return Qnil;
    }
    cipher = SSL_get_current_cipher(p->ssl);

    return ssl_cipher_to_ary(cipher);
}

static VALUE
ssl_get_ciphers(self)
    VALUE self;
{
    ssl_st *p;
    STACK_OF(SSL_CIPHER) *ciphers;
    SSL_CIPHER *cipher;
    VALUE ary;
    int i;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ctx){
        if(ruby_verbose) rb_warning("SSL_CTX is not initialized.");
        return Qnil;
    }
    ciphers = p->ctx->cipher_list;
    ary = rb_ary_new();
    if(ciphers){
        for(i = 0; i < sk_num((STACK*)ciphers); i++){
            cipher = (SSL_CIPHER*)sk_value((STACK*)ciphers, i);
            rb_ary_push(ary, ssl_cipher_to_ary(cipher));
        }
    }
    return ary;
}

static VALUE
ssl_set_ciphers(self, v)
    VALUE self, v;
{
    ssl_st *p;
    VALUE str, s, elem;
    int i;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ctx){
        rb_raise(eSSLError, "SSL_CTX is not initialized.");
        return Qnil;
    }

    if(TYPE(v) == T_STRING) str = v;
    else if(TYPE(v) == T_ARRAY){
        str = rb_str_new2("");
        for(i = 0; i < RARRAY(v)->len; i++){
            elem = rb_ary_entry(v, i);
            if(TYPE(elem) == T_ARRAY) elem = rb_ary_entry(elem, 0);
            elem = rb_obj_as_string(elem);
            rb_str_append(str, elem);
            if(i < RARRAY(v)->len-1) rb_str_cat2(str, ":");
        }
    }
    else str = rb_obj_as_string(v);

    if(!SSL_CTX_set_cipher_list(p->ctx, STR2CSTR(str)))
        rb_raise(eSSLError, "SSL_CTX_set_ciphers:%s", ssl_err_str());

    return Qnil;
}

static VALUE
ssl_get_state(self)
    VALUE self;
{
    ssl_st *p;
    VALUE ret;

    Data_Get_Struct(self, ssl_st, p);
    if(!p->ssl){
        if(ruby_verbose) rb_warning("SSL session is not started yet.");
        return Qnil;
    }
    ret = rb_str_new2(SSL_state_string(p->ssl));
    if(ruby_verbose){
        rb_str_cat2(ret, ": ");
        rb_str_cat2(ret, SSL_state_string_long(p->ssl));
    }
    return ret;
}

static VALUE
x509_init(self, v)
    VALUE self, v;
{
    FILE *fp;
    x509_st *p;
    X509 *ret;

    Data_Get_Struct(self, x509_st, p);
    if((fp = fopen(STR2CSTR(v), "r")) == NULL) return self;
    ret = PEM_read_X509(fp, &p->cert, NULL, NULL);
    fclose(fp);
    if(!ret) rb_raise(eSSLError, "PEM_read_X509:%s", ssl_err_str());

    return self;
}

static VALUE
x509_s_new(argc, argv, klass)
    int argc;
    VALUE *argv, klass;
{
    VALUE obj;
    x509_st *p;

    obj = Data_Make_Struct(klass, x509_st, 0, x509_free, p);
    p->cert = NULL;
    rb_obj_call_init(obj, argc, argv);
    return obj;
}

static VALUE
x509_version(self)
    VALUE self;
{
    x509_st *p;

    Data_Get_Struct(self, x509_st, p);
    if(p->cert == NULL) return Qnil;
    return INT2NUM(X509_get_version(p->cert));
}

static VALUE
x509_serialNumber(self)
    VALUE self;
{
    x509_st *p;
    BIGNUM *bn;

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    bn = ASN1_INTEGER_to_BN(X509_get_serialNumber(p->cert), NULL);
    return rb_cstr2inum(BN_bn2dec(bn), 10);
}

static VALUE
x509_sigAlgor(self)
    VALUE self;
{
    x509_st *p;
    int i;

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    i = OBJ_obj2nid(p->cert->cert_info->signature->algorithm);
    return (i == NID_undef) ? Qnil : rb_str_new2(OBJ_nid2ln(i));
}

static VALUE
x509_subject(self)
    VALUE self;
{
    x509_st *p;
    char buf[256];

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    X509_NAME_oneline(X509_get_subject_name(p->cert), buf, sizeof(buf));
    return rb_str_new2(buf);
}

static VALUE
x509_issuer(self)
    VALUE self;
{
    x509_st *p;
    char buf[256];

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    X509_NAME_oneline(X509_get_issuer_name(p->cert), buf, sizeof(buf));
    return rb_str_new2(buf);
}

static VALUE
x509_notBefore(self)
{
    x509_st *p;
    VALUE ret;
    BUF_MEM *buf;
    BIO *bio;

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    bio = BIO_new(BIO_s_mem());
    ASN1_TIME_print(bio, X509_get_notBefore(p->cert));
    BIO_get_mem_ptr(bio, &buf);
    ret = rb_str_new(buf->data, buf->length);
    BIO_free(bio);
    return ret;
}

static VALUE
x509_notAfter(self)
{
    x509_st *p;
    VALUE ret;
    BUF_MEM *buf;
    BIO *bio;

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    bio = BIO_new(BIO_s_mem());
    ASN1_TIME_print(bio, X509_get_notAfter(p->cert));
    BIO_get_mem_ptr(bio, &buf);
    ret = rb_str_new(buf->data, buf->length);
    BIO_free(bio);
    return ret;
}

static VALUE
x509_key_type(self)
    VALUE self;
{
    x509_st *p;
    int i;

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;
    i = OBJ_obj2nid(p->cert->cert_info->key->algor->algorithm);
    return (i == NID_undef) ? Qnil : rb_str_new2(OBJ_nid2ln(i));
}

static VALUE
x509_extension(self)
    VALUE self;
{
    VALUE ary;
    x509_st *p;
    int i;

    Data_Get_Struct(self, x509_st, p);
    if(!p->cert) return Qnil;

    ary = rb_ary_new();
    for(i = 0; i < X509_get_ext_count(p->cert); i++){
        X509_EXTENSION *ext;
        ASN1_OBJECT *object;
        ASN1_OCTET_STRING *value;
        BIO *bio;
        BUF_MEM *bmem;
        char buf[80];
        VALUE tmp, ext_str;

        tmp = rb_ary_new2(2);
        ext = X509_get_ext(p->cert, i);

        object = X509_EXTENSION_get_object(ext);
        i2t_ASN1_OBJECT(buf, sizeof(buf), object);
        rb_ary_push(tmp, rb_str_new2(buf));
 
        ext_str = Qnil;
        bio = BIO_new(BIO_s_mem());
        if(X509V3_EXT_print(bio, ext, 0, 0)){
            BIO_get_mem_ptr(bio, &bmem);
            ext_str = rb_str_new(bmem->data, bmem->length);
        }
        else if(value = X509_EXTENSION_get_data(ext)){
            ext_str = rb_str_new(value->data, value->length);
        }
        BIO_free(bio);
        rb_ary_push(tmp, ext_str);

        rb_ary_push(ary, tmp); 
    }

    return ary;
}

static VALUE
x509_verify(self, other)
    VALUE self, other;
{
    x509_st *p0, *p1;
    EVP_PKEY *pkey;
    int result;

    if(TYPE(other) == T_STRING)
      other = rb_funcall(cX509, rb_intern("new"), 1, other);
    else if(!rb_obj_is_kind_of((other), cX509))
      rb_raise(rb_eArgError, "argument must be a X509 or String");

    Data_Get_Struct(self, x509_st, p0);
    if(!p0->cert) return Qfalse;
    Data_Get_Struct(other, x509_st, p1);
    if(!p1->cert) return Qfalse;
    if((pkey = X509_get_pubkey(p1->cert)) == NULL) return Qfalse;
    result = X509_verify(p0->cert, pkey);
    EVP_PKEY_free(pkey);

    return result ? Qtrue : Qfalse;
}

static VALUE
x509_inspect(self)
    VALUE self;
{
    VALUE str;
    struct {
        const char *name; VALUE (*func)_((VALUE));
    } field_tab[] = {
        { "version=",        x509_version },
        { "serialNumber=",   x509_serialNumber },
        { "sigAlgor=",       x509_sigAlgor },
        { "issuer=",         x509_issuer },
        { "notBefore=",      x509_notBefore },
        { "notAfter=",       x509_notAfter },
        { "subject=",        x509_subject },
        { "key_type=",       x509_key_type },
    };
    int nfields = numberof(field_tab);
    int i;

    str = rb_str_new2("#<");
    rb_str_cat2(str, rb_class2name(CLASS_OF(self)));
    rb_str_cat2(str, " ");
    for(i = 0; i < nfields; i++){
        if(i > 0) rb_str_cat2(str, ", ");
        rb_str_cat2(str, field_tab[i].name);
        rb_str_append(str, rb_inspect(field_tab[i].func(self)));
    }
    rb_str_cat2(str, ">");

    return str;
}

static VALUE
x509_to_s(self)
    VALUE self;
{
    x509_st *p;
    VALUE ret;
    BUF_MEM *buf;
    BIO *bio;

    Data_Get_Struct(self, x509_st, p);
    bio = BIO_new(BIO_s_mem());
    PEM_write_bio_X509(bio, p->cert);
    BIO_set_mem_eof_return(bio, 0);
    BIO_get_mem_ptr(bio, &buf);
    ret = rb_str_new(buf->data, buf->length);
    BIO_free(bio);

    return ret;
}

static VALUE
x509str_get_error(self)
    VALUE self;
{
    x509_store_ctx_st *p;
    Data_Get_Struct(self, x509_store_ctx_st, p);

    if(p->ctx == NULL){
        rb_raise(eSSLError,
            "X509_STORE_CTX cannot be referd out of verify callback clause");
    }
    return INT2FIX(X509_STORE_CTX_get_error(p->ctx));
}

static VALUE
x509str_get_err_msg(self)
    VALUE self;
{
    int err;

    err = FIX2INT(x509str_get_error(self));
    return rb_str_new2(X509_verify_cert_error_string(err));
}

static VALUE
x509str_get_curr_cert(self)
    VALUE self;
{
    x509_store_ctx_st *p;
    x509_st *x509p;
    VALUE ret;

    Data_Get_Struct(self, x509_store_ctx_st, p);
    if(p->ctx == NULL){
        rb_raise(eSSLError,
            "X509_STORE_CTX cannot be referd out of verify callback clause");
    }
    ret = Data_Make_Struct(cX509, x509_st, 0, x509_free, x509p);
    x509p->cert = X509_dup(X509_STORE_CTX_get_current_cert(p->ctx));

    return ret;
}

static VALUE
x509str_get_err_depth(self)
    VALUE self;
{
    x509_store_ctx_st *p;

    Data_Get_Struct(self, x509_store_ctx_st, p);
    if(p->ctx == NULL){
        rb_raise(eSSLError,
            "X509_STORE_CTX cannot be referd out of verify callback clause");
    }
    return INT2FIX(X509_STORE_CTX_get_error_depth(p->ctx));
}

void
Init_ssl()
{
    int i;

    SSLeay_add_all_algorithms();
    SSL_load_error_strings();

    /* module SSL */
    mSSL = rb_define_module("SSL");
    rb_define_const(mSSL, "VERSION", rb_str_new2("0.3.3"));
    rb_define_const(mSSL, "OPENSSL_VERSION", rb_str_new2(OPENSSL_VERSION_TEXT));

    /* class SSLError */
    eSSLError = rb_define_class_under(mSSL, "SSLError", rb_eStandardError);

    /* class SSLSocket */
    cSSLSocket = rb_define_class_under(mSSL, "SSLSocket", rb_cObject);
    rb_define_singleton_method(cSSLSocket, "new", ssl_s_new, -1);
    rb_define_method(cSSLSocket, "initialize",   ssl_initialize, -1);
    rb_define_method(cSSLSocket, "connect",      ssl_connect, 0);
    rb_define_method(cSSLSocket, "accept",       ssl_accept, 0);
    rb_define_method(cSSLSocket, "sysread",      ssl_read, 1);
    rb_define_method(cSSLSocket, "syswrite",     ssl_write, 1);
    rb_define_method(cSSLSocket, "sysclose",     ssl_close, 0);
    rb_define_method(cSSLSocket, "cert",         ssl_get_cert, 0);
    rb_define_method(cSSLSocket, "peer_cert",    ssl_get_peer_cert, 0);
    rb_define_method(cSSLSocket, "cipher",       ssl_get_cipher, 0);
    rb_define_method(cSSLSocket, "ciphers",      ssl_get_ciphers, 0);
    rb_define_method(cSSLSocket, "ciphers=",     ssl_set_ciphers, 1);
    rb_define_method(cSSLSocket, "state",        ssl_get_state, 0);
    for(i = 0; i < numberof(ssl_attrs); i++)
        rb_attr(cSSLSocket, rb_intern(ssl_attrs[i]), 1, 1, Qfalse);
    rb_attr(cSSLSocket, rb_intern("io"), 1, 0, Qfalse);
    rb_define_alias(cSSLSocket, "to_io", "io");

    ssl_def_const(VERIFY_NONE);
    ssl_def_const(VERIFY_PEER);
    ssl_def_const(VERIFY_FAIL_IF_NO_PEER_CERT);
    ssl_def_const(VERIFY_CLIENT_ONCE);

    /* class X509 */
    cX509 = rb_define_class_under(mSSL, "X509", rb_cObject);
    rb_define_singleton_method(cX509, "new", x509_s_new, -1);
    rb_define_method(cX509, "initialize",    x509_init, 1);
    rb_define_method(cX509, "version",       x509_version, 0);
    rb_define_method(cX509, "serialNumber",  x509_serialNumber, 0);
    rb_define_method(cX509, "sigAlgor",      x509_sigAlgor, 0);
    rb_define_method(cX509, "subject",       x509_subject, 0);
    rb_define_method(cX509, "issuer",        x509_issuer, 0);
    rb_define_method(cX509, "notBefore",     x509_notBefore, 0);
    rb_define_method(cX509, "notAfter",      x509_notAfter, 0);
    rb_define_method(cX509, "key_type",      x509_key_type, 0);
    rb_define_method(cX509, "extension",     x509_extension, 0);
    rb_define_method(cX509, "verify",        x509_verify, 1);
    rb_define_method(cX509, "inspect",       x509_inspect, 0);
    rb_define_method(cX509, "to_s",          x509_to_s, 0);

    x509_def_const(UNABLE_TO_GET_ISSUER_CERT);
    x509_def_const(UNABLE_TO_GET_CRL);
    x509_def_const(UNABLE_TO_DECRYPT_CERT_SIGNATURE);
    x509_def_const(UNABLE_TO_DECRYPT_CRL_SIGNATURE);
    x509_def_const(UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY);
    x509_def_const(CERT_SIGNATURE_FAILURE);
    x509_def_const(CRL_SIGNATURE_FAILURE);
    x509_def_const(CERT_NOT_YET_VALID);
    x509_def_const(CERT_HAS_EXPIRED);
    x509_def_const(CRL_NOT_YET_VALID);
    x509_def_const(CRL_HAS_EXPIRED);
    x509_def_const(ERROR_IN_CERT_NOT_BEFORE_FIELD);
    x509_def_const(ERROR_IN_CERT_NOT_AFTER_FIELD);
    x509_def_const(ERROR_IN_CRL_LAST_UPDATE_FIELD);
    x509_def_const(ERROR_IN_CRL_NEXT_UPDATE_FIELD);
    x509_def_const(OUT_OF_MEM);
    x509_def_const(DEPTH_ZERO_SELF_SIGNED_CERT);
    x509_def_const(SELF_SIGNED_CERT_IN_CHAIN);
    x509_def_const(UNABLE_TO_GET_ISSUER_CERT_LOCALLY);
    x509_def_const(UNABLE_TO_VERIFY_LEAF_SIGNATURE);
    x509_def_const(CERT_CHAIN_TOO_LONG);
    x509_def_const(CERT_REVOKED);
    x509_def_const(INVALID_CA);
    x509_def_const(PATH_LENGTH_EXCEEDED);
    x509_def_const(INVALID_PURPOSE);
    x509_def_const(CERT_UNTRUSTED);
    x509_def_const(CERT_REJECTED);
    x509_def_const(SUBJECT_ISSUER_MISMATCH);
    x509_def_const(AKID_SKID_MISMATCH);
    x509_def_const(AKID_ISSUER_SERIAL_MISMATCH);
    x509_def_const(KEYUSAGE_NO_CERTSIGN);
    x509_def_const(APPLICATION_VERIFICATION);

    /* class X509_STORE_CTX */
    cX509_STORE_CTX = rb_define_class_under(mSSL, "X509_STORE_CTX", rb_cObject);
    rb_undef_method(cX509_STORE_CTX, "new");
    rb_define_method(cX509_STORE_CTX, "error",        x509str_get_error, 0);
    rb_define_method(cX509_STORE_CTX, "current_cert", x509str_get_curr_cert, 0);
    rb_define_method(cX509_STORE_CTX, "error_depth",  x509str_get_err_depth, 0);
    rb_define_method(cX509_STORE_CTX, "error_message", x509str_get_err_msg, 0);
}
