// Copyright (c) 2000 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "connection.h"
#include "mq3d.h"
#include "collections.h"

#include <qsocket.h>
#include <qsocketnotifier.h>
#include <qfile.h>
#include <qtimer.h>


Connection::Connection(int sockfd, MQ3d *parent)
    : QObject(parent)
{
    mq3d = parent;

    socket = new QSocket(this);
    socket->setSocket(sockfd);

    connect(socket, SIGNAL(readyRead()), SLOT(readyRead()));
    connect(socket, SIGNAL(connectionClosed()), SLOT(otherEndClosed()));
    connect(socket, SIGNAL(delayedCloseFinished()), SLOT(myEndClosed()));

    st = Connected;
    file = 0;
    timer = 0;
}


Connection::~Connection()
{
    if (socket) {
	socket->close();
	delete socket;
    }
    if (file)
	delete file;
    if (timer)
	delete timer;
}


void Connection::readyRead()
{
    switch (st) {
    case Connected:
	{
	    const char *reply = 0;

	    QCString line(socket->bytesAvailable() + 1);
	    socket->readLine(line.data(), line.size());

	    if (line.isNull() || line.isEmpty()) {
		reply = "*NOPE Invalid command\r\n";
		st = Error;
	    } else {
		// remove newlines
		int pos = line.length() - 1;
		while (pos >= 0 && (line[pos] == '\n' ||
				    line[pos] == '\r'))
		    line[pos--] = 0;

		// parse the command and argument
		QString command, argument;

		if ((pos = line.find(' ')) != -1) {
		    // we have a space, take the first one to be the space
		    // after the command, the rest is the argument
		    command = line.left(pos);
		    argument = line.mid(pos + 1);
		} else
		    command = line;

		if (command == ".list") {
		    if (file) {
			qDebug("file nonzero!");
			delete file;
		    }

		    file = new QFile(mq3d->collectionsFileName());
		    if (! file->open(IO_ReadOnly)) {
			// EH?
			reply = "*NOPE Server error: collections file not found\r\n";
			st = Error;
		    } else {
			// send the collection list
			reply = "*GOOD Server listing follows \\r\\n\r\n";
			st = Reply;

			// start 0 timer to handle writes
			if (timer) {
			    if (timer->isActive())
				timer->stop();
			} else {
			    timer = new QTimer(this);
			    connect(timer, SIGNAL(timeout()), SLOT(readyWrite()));
			}
			timer->start(0);
			// bytesWritten = 0;
			// startTime = QDateTime::currentDateTime();
			// qDebug("transfer started at %s",
			// startTime.toString().latin1());
		    }
		} else if (command == ".song") {
		    if (file) {
			qDebug("file nonzero!");
			delete file;
		    }

		    Collections *c = mq3d->collections();
		    QString path = c->fileName(argument);
		    if (path.isNull() || path.isEmpty()) {
			reply = "*NOPE Error parsing path\r\n";
			st = Error;
		    } else {
			file = new QFile(path);
			if (! file->open(IO_ReadOnly)) {
			    // EH?
			    reply = "*NOPE Server error: song file not found\r\n";
			    st = Error;
			} else {
			    // send the file
			    reply = "*GOOD Song data follows \\r\\n\r\n";
			    st = Reply;

			    // start 0 timer to handle writes
			    if (timer) {
				if (timer->isActive())
				    timer->stop();
			    } else {
				timer = new QTimer(this);
				connect(timer, SIGNAL(timeout()), SLOT(readyWrite()));
			    }
			    timer->start(0);
			    // bytesWritten = 0;
			    // startTime = QDateTime::currentDateTime();
			    // qDebug("transfer started at %s",
			    // startTime.toString().latin1());
			}
		    }
		} else if (command == ".quit") {
		    delete this;
		    return;
		} else {
		    // we have no *clue* what they're talking about
		    reply = "*NOPE Unknown command\r\n";
		    st = Error;
		}
	    }

	    if (! reply)
		reply = "*NOPE Unknown error\r\n";
	    socket->writeBlock(reply, qstrlen(reply));

	    st = Connected;

	    break;
	}

    case Error:
	{
	    const char *reply = "*NOPE Connection in ERROR state\n";
	    socket->writeBlock(reply, qstrlen(reply));

	    break;
	}

    default:
	// do nothing
	;
    }
}


void Connection::readyWrite()
{
    if (! socket || ! file) {
	qDebug("WHOA! nothing exists");
	return;
    }

    uint count = QMAX((unsigned) (file->size() - file->at()),
		      (unsigned) socket->bytesToWrite());
    count = QMIN(count, 32768);
    bool done = FALSE;

    if (count) {
	QByteArray ba(count);
	uint r = file->readBlock(ba.data(), ba.size());

	// short file read?
	if (r != count)
	    count = r;

	socket->writeBlock(ba.data(), count);
	// bytesWritten += count;

	if (file->size() - file->at() == 0)
	    done = TRUE;
    } else
	done = TRUE;

    if (done) {
	// int totalsecs = startTime.secsTo(QDateTime::currentDateTime());
	// qDebug("transfer done at %d b/s", (bytesWritten / 1024) / totalsecs);

	socket->close();

	if (socket->state() == QSocket::Idle)
	    delete this;
    }
}


void Connection::otherEndClosed()
{
    delete this;
}


void Connection::myEndClosed()
{
    delete this;
}

