/**
 * Siege, http regression tester / benchmark utility
 *
 * Copyright (C) 2000 Jeffrey Fulmer <jdfulmer@armstrong.com>
 * This file is part of Siege
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *
 */
#define  INTERN  1

#ifdef  HAVE_CONFIG_H
# include <config.h>
#endif/*HAVE_CONFIG_H*/

#ifdef  HAVE_SIGNAL_H
# include <signal.h>
#endif/*HAVE_SIGNAL_H*/

#include <setup.h>
#include <getopt.h>
#include <util.h>

int  fd        =    0;    /* tmp file, for user defined URL          */
int signaled   =    0;    /* SIGTERM, SIGHUP, SIGINT from user.      */
char temp[32];            /* container for tmp file name.            */
int status;               /* waitpid status from the children        */
int qid;                  /* msg queue ID return by create_queue     */
struct joemsg message;    /* message queue, print messages to stdout */
int count      =    0;    /* count messages processes, ie. transacts */
clock_t start, stop;      /* process start and stop times.           */

struct tms t_start, t_stop;
int  i;

#ifdef HAVE_SYS_RESOURCE_H
  struct rlimit rlp;
#endif /* HAVE_SYS_RESOURCE_H */


/**
 * long options, std options struct
 */
static struct option long_options[] =
{
  { "version",    no_argument,       NULL, 'V' },
  { "help",       no_argument,       NULL, 'h' },
  { "verbose",    no_argument,       NULL, 'v' },
  { "config",     no_argument,       NULL, 'C' },
  { "debug",      no_argument,       NULL, 'D' },
  { "concurrent", required_argument, NULL, 'c' },
  { "internet",   no_argument,       NULL, 'i' },
  { "benchmark",  no_argument,       NULL, 'b' },
  { "reps",       required_argument, NULL, 'r' },
  { "time",       required_argument, NULL, 't' },
  { "delay",      required_argument, NULL, 'd' },
  { "log",        no_argument,       NULL, 'l' },
  { "file",       required_argument, NULL, 'f' },
  { "url",        required_argument, NULL, 'u' },
  { "mark",       required_argument, NULL, 'm' }
};

/**
 * display_version   
 * displays the version number and exits on boolean false. 
 * continue running? TRUE=yes, FALSE=no
 * return void
 */
void 
display_version( int i )
{
  /**
   * version_string is defined in version.c 
   * adding it to a separate file allows us
   * to parse it in configure.  
   */
  if( my.debug )
    printf( "Siege %s: debugging enabled\n", version_string );
  else 
    printf( "Siege %s\n", version_string );
  /**
   * if TRUE, exit 
   */
  if( i == 1 ){ exit( EXIT_SUCCESS ); }
}  /* end of display version */

/**
 * display_help 
 * displays the help section to STDOUT and exits
 */ 
void 
display_help()
{
  /**
   * call display_version, but do not exit 
   */
  display_version( FALSE ); 
  printf("Usage: siege [options]\n");
  printf("Options:\n"                    );
  puts("  -V, --version         VERSION, prints version number to screen.");
  puts("  -h, --help            HELP, prints this section.");
  puts("  -C, --config          CONFIGURATION, show the current configuration.");
  puts("  -v, --verbose         VERBOSE, prints notification to screen.");
  puts("  -c, --concurrent=NUM  CONCURRENT users, default is 10");
  puts("  -u, --url=\"URL\"       URL, a single user defined URL for stress testing." );
  puts("  -i, --internet        INTERNET user simulation, hits the URLs randomly." );
  puts("  -b, --benchmark       BENCHMARK, signifies no delay for time testing." );
  puts("  -t, --time=NUMm       TIME based testing where \"m\" is the modifier S, M, or H" );
  puts("                        no space between NUM and \"m\", ex: --time=1H, one hour test." );
  puts("  -r, --reps=NUM        REPS, number of times to run the test, default is 25" );
  puts("  -f, --file=FILE       FILE, change the configuration file to file." );
  puts("  -l, --log             LOG, logs the transaction to PREFIX/var/siege.log");
  puts("  -m, --mark=\"text\"     MARK, mark the log file with a string separator." );
  puts("  -d, --delay=NUM       Time DELAY, random delay between 1 and num designed" );
  puts("                        to simulate human activity. Default value is 3" );
  /**
   * our work is done, exit nicely
   */
  exit( EXIT_SUCCESS );
}

/**
 * ulimit_help
 * explains the ulimit to the user, displays proc number
 */
void
ulimit_help( int proc )
{
  puts( "* * * * * * * * * * FATAL * * * * * * * * * *" );
  puts( "The number of concurrent users  that you have" );
  puts( "selected exceeds the number of processes that" );
  puts( "you are allowed to run on this system." );
  puts( "This system is currently  configured to allow" );
  printf( "you to run %d processes.\n", proc );
  puts( "In order to exceed that number, you will have" );
  puts( "to run siege as root user or have your system" );
  puts( "administrator increase your nproc limit." );
  puts( "* * * * * * * * * * FATAL * * * * * * * * * *" );
}

/**
 * sig_handler  method to handle signals,
 * interrupts and alarms.
 */
void
sig_handler( int signum )
{
  /**
   * currently all signals use this
   * handler, need a separate handler
   * for SIGCHILD, see TODO list
   */
  signaled = 1;
} /** end of sig_handler **/ 

/**
 * alarm_handler
 * sigalarm handler, kills the process group, i.e., 
 * all my children.
 */
void
alarm_handler( int signum )
{
  kill( - getpgrp(), 1 ); 
  signaled = 1; 
}

/**
 * parses command line arguments and assigns
 * values to run time variables. relies on GNU
 * getopts included with this distribution.  
 */ 
void 
parse_cmdline( int argc, char *argv[] )
{
  int c = 0;
  while ((c = getopt_long (argc, argv, "VhvCDlibr:t:f:d:c:u:m:", long_options, (int *)0)) != EOF){
  switch( c ){
      case 'V':
        display_version( TRUE );
        break;
      case 'h':
        display_help();
        exit( EXIT_SUCCESS );
      case 'D':
        my.debug = TRUE;
        break;
      case 'C':
        my.config = TRUE;
        break;
      case 'c':
        my.cusers  = atoi( optarg );
        break;
      case 'i':
        my.internet = TRUE;
        break;
      case 'b':
        my.bench    = TRUE;
        break;
      case 'd':
        my.delay   = atoi( optarg );
        break;
      case 'l':
        my.logging = TRUE;
        break;
      case 'm':
        my.mark    = TRUE;
        my.markstr = optarg;
        my.logging = TRUE; 
        break;
      case 'v':
        my.verbose = TRUE;
        break;
      case 'r':
        my.reps = atoi( optarg );
        break;
      case 't':
        parse_time( optarg );
        break;
      case 'f':
        memset( my.file, '\0', sizeof( my.file ));
        strncpy( my.file, optarg, sizeof( my.file ));
        break;
      case 'u':
        my.url = optarg;
        memset( my.file, '\0', sizeof( my.file ));
        strcpy( temp, "/tmp/siegeXXXXXX" );
        strcpy( my.file, mktemp(temp)); 	
        if((fd = open( my.file, O_CREAT | O_WRONLY, 0644 )) < 0 ) {
            joe_fatal( "Unable to write to /tmp" );
        }
        write( fd, my.url, strlen( my.url )); 	
        break;
    } /** end of switch( c )           **/
  }   /** end of while c = getopt_long **/
}     /** end of parse_cmdline         **/

/**
 * siege main
 */  
int 
main( int argc, char *argv[] )
{
  LINES       *lines;
  URL         *urls;
  pid_t       pid[MXCHLD];
  pid_t       mypid;
  struct sigaction action;
  int x, y  = 0, length;
  int rnd   = 0; 
  int inc   = 0;
  int code  = 0;
  int fail  = 0;
  float elapsed; 
  float ttime = 0.0;	
  unsigned long int bytes = 0;
  lines = (LINES*) xmalloc(sizeof(LINES));
  lines->index   = 0;
  lines->line    = NULL;
  urls  = (URL*)   xmalloc(sizeof( URL ));
  urls->index    = 0;
  urls->protocol = NULL;
  urls->hostname = NULL;
  urls->port     = 0;
  urls->pathname = NULL;
  urls->calltype = URL_GET;
  urls->postdata = NULL;
 
  init_config();
  parse_cmdline(argc, argv);
  
  if( my.config ) show_config( TRUE );
  length = read_cfg_file( lines, my.file );

  /** KLUDGE **/
  if( length <= 2 ) my.internet = TRUE;
  if( length == 0 ) display_help();

  /**
   * Living with my woman 'cause she 
   * wants to help me with my mind...
   */
  #ifdef HAVE_SYS_RESOURCE_H 
    getrlimit( RLIMIT_NPROC, &rlp );
    if( my.cusers >= ( rlp.rlim_cur - 10 )){
      ulimit_help( rlp.rlim_cur );
      exit( EXIT_FAILURE );
    }  
  #endif /* HAVE_SYS_RESOURCE_H */

  for( x = 0; x < lines->index; x ++ ){
    parse_url( urls, lines->line[x]  );
  }
	
  /*
    This was changed in order to assist people
    who are not familiar with IPC messaging. 
    Prior to this change, when the program was
    terminated improperly, it left an IPC message
    queue behind which prevented future invocations
    of siege. This fix removes any old message queues 
    in the event that one is found on start up.
  */	
  MAKE_QUEUE:
  if(( qid = create_queue()) > 0 );
  else{
    #ifdef DEBUG
    joe_warning( "removed an old IPC message queue!\n" );
    #endif 
    if(( qid = get_queue()) > 0 ){
      remove_queue( qid );
      goto MAKE_QUEUE;
    }
  }

  /**
   * record start time
   */
  start = times( &t_start );
  write( 1, "** ", 3 ); 
  display_version( FALSE );
  printf( "** Preparing %d concurrent users for battle.\n", my.cusers );
  write( 1, "The server is now under siege...", 32 );
  if( my.verbose ){ write( 1, "\n", 1 ); }
  
  /**
   * set the process group in case we
   * lose control of the system...
   */ 
  #ifdef SETPGRP_VOID
    setpgrp();
  #else  /* you are BSD */
    setpgrp( 0, getgid()); 
  #endif /* SETPGRP_VOID */
  
  /**
   * for each concurrent user, fork a
   * child process and loop forever...
   */
  for( x = 0; x < my.cusers; x++ ){
    if( x <= my.cusers )
    switch( pid[x] = fork() ){
      case -1:
        joe_error( "system resources exhausted!!!" );
        /**
         * remove IPC msg queue
         */
        remove_queue( qid ); 
        /**
         * clean up all existing processes.
         */
        kill( - getpgrp(), 1 );
        /**
         * shouldn't get here.
         */
        joe_fatal( "system resources exhausted!" );
      case  0:
        x = my.cusers;
        y = 1;
        mypid = getpid();
        insert_childid( urls, mypid );
        srand((unsigned)time( NULL ));

        while(( y <= my.reps ) || ( my.reps == -1 )){
          if( !my.bench ){
            sleep(
              (int)((double)rand() / ((double)RAND_MAX + 1) * my.delay ) +1
            ); 
          }
          if( urls->posttemp[inc] ){
            /* between 0 and (1<<15)-1 .. call it twice to get 8 digits */
            rnd = my_random( 9999, mypid )*10000 + my_random( 9999, mypid );
            build_from_template( urls, inc, rnd );
          }
          if( my.internet ){
            rnd = my_random( length, mypid );
            http_connect(
              urls->protocol[rnd], urls->hostname[rnd], 
              urls->port,          urls->pathname[rnd], 
              urls->calltype[rnd], urls->postdata[rnd], urls->postlen[rnd],
              qid
            );
          }
          else{
            http_connect(
              urls->protocol[inc], urls->hostname[inc], 
              urls->port,          urls->pathname[inc], 
              urls->calltype[inc], urls->postdata[inc], urls->postlen[inc],
              qid
            );

            /* inc is the array index for regression tests. 
               length is the size of the configuration file
               in lines.  if the array index is one less then
               the config structure, then reset it to zero */ 
            if( inc < length -1 ) inc  ++;
            else                  inc = 0;

          }
          y = (( my.secs < 0 ) || ( my.reps > 0 )) ? y++ : 0;
        } /* end of while loop */
        send_message2( qid );
        _exit( EXIT_SUCCESS );
    }     /* end switch PID    */
  }       /* end of for my.cusers */

  signal( SIGALRM, alarm_handler );
  alarm( my.secs );
  memset( &action, 0, sizeof( action ));
  action.sa_handler = sig_handler;	
  if( sigaction( SIGHUP, &action, NULL ))
    joe_error( "sigaction" );
  if( sigaction( SIGINT, &action, NULL))
    joe_error( "sigaction" );
  if( sigaction( SIGTERM, &action, NULL))
    joe_error( "sigaction" );

  y = 0;
  while((i = msgrcv(qid, &message, MSQSIZ, 0, MSG_NOERROR )) > 0){
    switch( message.type ){	
    case 1:	
      bytes += ( message.bytes );
      message.code <  400 ? code ++ : code;
      message.code >= 400 ? fail ++ : fail;
      ttime += ( message.time );
      memset( &message, '\0', sizeof( message ));
      count ++;
      break;	
    case 2:
      y ++;
      break;
    } /** end switch **/
    if( y == my.cusers ) break;	
    if( signaled == 1  ) break;
  }   /** end of while **/
		
  /**
   * wait for child processes.
   */
  for( x = 0; x < my.cusers; x ++ ){
    if(( waitpid( pid[x], &status, 0 )) < 0 ) 
      joe_error( "Waiting for child processes" );
  }  /** end of for loop **/	

  /**
   * remove IPC msg queue
   */	
  remove_queue( qid );
	
  /**
   * record stop time
   */
  stop = times( &t_stop );
  write( 1, "done", 4 );

  /**
   * take a short nap  for  cosmetic  effect
   * this does NOT affect performance stats.
   */
  sleep( 1 ); 

  /**
   * prepare and print statistics.
   */
  elapsed = elapsed_time( stop - start);
  printf( "\n" );
  printf( "Transactions:\t\t%12d hits\n", count );
  printf( "Elapsed time:\t\t%12.2f secs\n", elapsed );
  printf( "Data transferred:\t%12u bytes\n", bytes );
  printf( "Response time:\t\t%12.2f secs\n", ttime / count );
  printf( "Transaction rate:\t%12.2f trans/sec\n", count / elapsed );
  printf( "Throughput:\t\t%12.2f bytes/sec\n", bytes / elapsed );
  printf( "Concurrency:\t\t%12.2f\n", ttime / elapsed );
  printf( "Successful transactions:%12d\n", code ); 
  printf( "Failed transactions:\t%12d\n", fail );
  puts  ( " " );

  if( my.mark )    mark_log_file( my.markstr );
  if( my.logging ) write_to_log( count, elapsed, bytes, ttime, code );

  /**
   * exit program.
   */
  exit( EXIT_SUCCESS );	
} /* end of int main **/





