// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/google/google_update_policy_fetcher_win_util.h"

#include <OleCtl.h>

#include "base/check.h"
#include "base/strings/string_piece.h"
#include "base/values.h"
#include "base/win/scoped_bstr.h"
#include "components/policy/core/common/policy_map.h"
#include "components/policy/core/common/policy_types.h"

namespace {

policy::PolicySource GetPolicySource(BSTR source_bstr) {
  constexpr base::WStringPiece kCloudSource = L"Device Management";
  constexpr base::WStringPiece kDefaultSource = L"Default";
  const auto source =
      base::WStringPiece(source_bstr, ::SysStringLen(source_bstr));
  if (source == kCloudSource)
    return policy::POLICY_SOURCE_CLOUD;
  if (source == kDefaultSource)
    return policy::POLICY_SOURCE_ENTERPRISE_DEFAULT;
  DCHECK_EQ(source, base::WStringPiece(L"Group Policy"));
  return policy::POLICY_SOURCE_PLATFORM;
}

}  // namespace

std::unique_ptr<policy::PolicyMap::Entry> ConvertPolicyStatusValueToPolicyEntry(
    IPolicyStatusValue* policy,
    const PolicyValueOverrideFunction& value_override_function) {
  DCHECK(policy);

  base::win::ScopedBstr value;
  if (FAILED(policy->get_value(value.Receive())) || value.Length() == 0) {
    return nullptr;
  }

  base::win::ScopedBstr source;
  if (FAILED(policy->get_source(source.Receive())))
    return nullptr;

  auto entry = std::make_unique<policy::PolicyMap::Entry>(
      policy::POLICY_LEVEL_MANDATORY, policy::POLICY_SCOPE_MACHINE,
      GetPolicySource(source.Get()),
      value_override_function ? value_override_function.Run(value.Get())
                              : base::Value(value.Get()),
      nullptr);
  VARIANT_BOOL has_conflict = VARIANT_FALSE;
  base::win::ScopedBstr conflict_value;
  base::win::ScopedBstr conflict_source;
  if (SUCCEEDED(policy->get_hasConflict(&has_conflict)) &&
      has_conflict == VARIANT_TRUE &&
      SUCCEEDED(policy->get_conflictValue(conflict_value.Receive())) &&
      SUCCEEDED(policy->get_conflictSource(conflict_source.Receive()))) {
    entry->AddConflictingPolicy(policy::PolicyMap::Entry(
        policy::POLICY_LEVEL_MANDATORY, policy::POLICY_SCOPE_MACHINE,
        GetPolicySource(conflict_source.Get()),
        value_override_function
            ? value_override_function.Run(conflict_value.Get())
            : base::Value(conflict_value.Get()),
        nullptr));
  }
  if (entry->source == policy::POLICY_SOURCE_ENTERPRISE_DEFAULT)
    entry->SetIsDefaultValue();
  return entry;
}
