/* packet-esis.c
 * Routines for ISO/OSI End System to Intermediate System  
 * Routing Exchange Protocol ISO 9542.
 *
 * $Id: packet-esis.c,v 1.25 2002/04/30 23:56:58 guy Exp $
 * Ralf Schneider <Ralf.Schneider@t-online.de>
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@ethereal.com>
 * Copyright 1998 Gerald Combs
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <epan/packet.h>
#include "nlpid.h"
#include "packet-osi.h"
#include "packet-osi-options.h"
#include "packet-esis.h"


/* esis base header */
static int  proto_esis        = -1;

static int  hf_esis_nlpi      = -1;
static int  hf_esis_length    = -1;
static int  hf_esis_version   = -1;
static int  hf_esis_reserved  = -1;
static int  hf_esis_type      = -1;
static int  hf_esis_holdtime  = -1;
static int  hf_esis_checksum  = -1;

static gint ett_esis              = -1;
static gint ett_esis_area_addr    = -1;

static const value_string esis_vals[] = {
  { ESIS_ESH_PDU, "ES HELLO"},
  { ESIS_ISH_PDU, "IS HELLO"},
  { ESIS_RD_PDU,  "RD REQUEST"},
  { 0,             NULL} };

/* internal prototypes */

static void esis_dissect_esh_pdu( u_char len, tvbuff_t *tvb,
                           proto_tree *treepd);
static void esis_dissect_ish_pdu( u_char len, tvbuff_t *tvb,
                           proto_tree *tree);
static void esis_dissect_redirect_pdu( u_char len, tvbuff_t *tvb,
                           proto_tree *tree);

/* ################## Descriptions ###########################################*/
/* Parameters for the ESH PDU
 * Source Address Parameter:
 *
 * Octet:    Length:   Parameter Type:
 *     10          1   Number of Source Adresses ( NSAPs served by this Network
 *     11          1   Source Address Length Indicator ( SAL )     #    Entity )
 * 12-m-1   variable   Source Address ( NSAP )
 *      m              Options, dissected in osi.c
 *
 *
 * Parameter for the ISH PDU:
 * Network Entity Title Parameter:
 *
 * Octet:    Length:   Parameter Type:
 *     10          1   Network Entity Title Length Indicator ( NETL )
 * 11-m-1   variable   Network Entity Title ( NET )
 *      m              Options, dissected in osi.c
 *
 *
 * Parameter for the RD PDU:
 * When re-directed to an IS:
 *
 *  Octet:   Length:   Parameter Type:
 *      10         1   Destination Address Length Indicator ( DAL )
 *  11>m-1  variable   Destination Address ( DA )
 *       m         1   Subnetwork Address Length Indicator ( BSNPAL )
 * m+1>n-1  variable   Subnetwork Address ( BSNPA )
 *       n         1   Network Entity Title Length Indicator ( NETL )
 * n+1>p-1  variable   Network Entity Title ( NET )
 *       p             Options, dissected in osi.c
 *
 *
 * Parameter for the RD PDU:
 * When re-directed to an ES:
 *
 *  Octet:   Length:   Parameter Type:
 *      10         1   Destination Address Length Indicator ( DAL )
 *  11>m-1  variable   Destination Address ( DA )
 *       m         1   Subnetwork Address Length Indicator ( BSNPAL )
 * m+1>n-1  variable   Subnetwork Address ( BSNPA )
 *       n         1   Network Entity Title Length Indicator ( NETL ) == 0
 *     n+1             Options, dissected in osi.c
 *
 */

/* ############################ Tool Functions ############################## */


/* ############################## Dissection Functions ###################### */
/*
 * Name: dissect_esis_unknown()
 *
 * Description:
 *   There was some error in the protocol and we are in unknown space
 *   here.  Add a tree item to cover the error and go on.  Note
 *   that we make sure we don't go off the end of the bleedin packet here!
 *
 *   This is just a copy of isis.c and isis.h, so I keep the stuff also
 *   and adapt the names to cover possible protocol errors! Ive really no
 *   idea wether I need this or not.
 *  
 * Input
 *   tvbuff_t *      : tvbuff with packet data.
 *   proto_tree *    : tree of display data.  May be NULL.
 *   char *          : format text
 *   subsequent args : arguments to format
 *
 * Output:
 *   void (may modify proto tree)
 */
static void
esis_dissect_unknown( tvbuff_t *tvb, proto_tree *tree, char *fmat, ...){
  va_list ap;

  va_start(ap, fmat);
  proto_tree_add_text_valist(tree, tvb, 0, -1, fmat, ap);
  va_end(ap);
}


static void
esis_dissect_esh_pdu( u_char len, tvbuff_t *tvb, proto_tree *tree) {
  proto_tree *esis_area_tree;
  int         offset  = 0;
  int         no_sa   = 0;
  int         sal     = 0;
  
  proto_item  *ti;
  
  if (tree) {
    offset += ESIS_HDR_FIXED_LENGTH;

    no_sa  = tvb_get_guint8(tvb, offset);
    len   -= 1;

    ti = proto_tree_add_text( tree, tvb, offset, -1,
            "Number of Source Addresses (SA, Format: NSAP) : %u", no_sa );
    offset++;
    
    esis_area_tree = proto_item_add_subtree( ti, ett_esis_area_addr );
    while ( no_sa-- > 0 ) {
       sal = (int) tvb_get_guint8(tvb, offset);
       offset++;
       proto_tree_add_text(esis_area_tree, tvb, offset, 1, "SAL: %2u Octets", sal);
       proto_tree_add_text(esis_area_tree, tvb, offset + 1, sal,
                           " SA: %s",
			   print_nsap_net( tvb_get_ptr(tvb, offset, sal), sal ) );
       offset += sal;
       len    -= ( sal + 1 );
    }
    dissect_osi_options( len, tvb, offset, tree );
  }  
} /* esis_dissect_esh_pdu */ ;

static void
esis_dissect_ish_pdu( u_char len, tvbuff_t *tvb, proto_tree *tree) {
  
  int   offset  = 0;
  int   netl    = 0;

  if (tree) {
    offset += ESIS_HDR_FIXED_LENGTH;

    netl = (int) tvb_get_guint8(tvb, offset);
    proto_tree_add_text( tree, tvb, offset, netl + 1, 
                         "### Network Entity Title Section ###");
    proto_tree_add_text( tree, tvb, offset++, 1, "NETL: %2u Octets", netl);
    proto_tree_add_text( tree, tvb, offset, netl,
                           " NET: %s",
			   print_nsap_net( tvb_get_ptr(tvb, offset, netl), netl ) );
    offset += netl;
    len    -= ( netl + 1 );

    dissect_osi_options( len, tvb, offset, tree );
  }
};

static void
esis_dissect_redirect_pdu( u_char len, tvbuff_t *tvb, proto_tree *tree) {

  int   offset  = 0;
  int   tmpl    = 0;

  if (tree) {
    offset += ESIS_HDR_FIXED_LENGTH;

    tmpl = (int) tvb_get_guint8(tvb, offset);
    proto_tree_add_text( tree, tvb, offset, tmpl + 1, 
                         "### Destination Address Section ###" );
    proto_tree_add_text( tree, tvb, offset++, 1, "DAL: %2u Octets", tmpl);
    proto_tree_add_text( tree, tvb, offset, tmpl,
                         " DA : %s",
			 print_nsap_net( tvb_get_ptr(tvb, offset, tmpl), tmpl ) );
    offset += tmpl;
    len    -= ( tmpl + 1 );
    tmpl    = (int) tvb_get_guint8(tvb, offset);

    proto_tree_add_text( tree, tvb, offset, tmpl + 1, 
                         "###  Subnetwork Address Section ###");
    proto_tree_add_text( tree, tvb, offset++, 1, "BSNPAL: %2u Octets", tmpl);
    proto_tree_add_text( tree, tvb, offset, tmpl,
                           " BSNPA: %s",
			   print_system_id( tvb_get_ptr(tvb, offset, tmpl), tmpl ) );
    offset += tmpl;
    len    -= ( tmpl + 1 );
    tmpl    = (int) tvb_get_guint8(tvb, offset);

    if ( 0 == tmpl ) {
      proto_tree_add_text( tree, tvb, offset, 1, 
                           "### No Network Entity Title Section ###" );
      offset++;
      len--;
    }
    else {
      proto_tree_add_text( tree, tvb, offset, 1,
                           "### Network Entity Title Section ###" );
      proto_tree_add_text( tree, tvb, offset++, 1, "NETL: %2u Octets", tmpl );
      proto_tree_add_text( tree, tvb, offset, tmpl,
                           " NET: %s",
			   print_nsap_net( tvb_get_ptr(tvb, offset, tmpl), tmpl ) );
      offset += tmpl;
      len    -= ( tmpl + 1 );
    }
    dissect_osi_options( len, tvb, offset, tree );
  }
}


/*
 * Name: dissect_esis()
 * 
 * Description:
 *   Main entry area for esis de-mangling.  This will build the
 *   main esis tree data and call the sub-protocols as needed.
 *
 * Input:
 *   tvbuff *      : tvbuff referring to packet data
 *   packet_info * : info for current packet
 *   proto_tree *  : tree of display data.  May be NULL.
 *
 * Output:
 *   void, but we will add to the proto_tree if it is not NULL.
 */
static void
dissect_esis(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree) {
   char       *pdu_type_string        = NULL;
   char       *pdu_type_format_string = "PDU Type      : %s (R:%s%s%s)";   
   esis_hdr_t  ehdr;
   proto_item *ti;
   proto_tree *esis_tree    = NULL;
   int         variable_len = 0;
   u_int       tmp_uint     = 0;
   char       *cksum_status;

   if (check_col(pinfo->cinfo, COL_PROTOCOL))
     col_set_str(pinfo->cinfo, COL_PROTOCOL, "ESIS");
   if (check_col(pinfo->cinfo, COL_INFO))
     col_clear(pinfo->cinfo, COL_INFO);

   tvb_memcpy(tvb, (guint8 *)&ehdr, 0, sizeof ehdr);
   
   if (tree) {
     ti = proto_tree_add_item(tree, proto_esis, tvb, 0, -1, FALSE);
     esis_tree = proto_item_add_subtree(ti, ett_esis);

     if (ehdr.esis_version != ESIS_REQUIRED_VERSION){
       esis_dissect_unknown(tvb, esis_tree,
                          "Unknown ESIS version (%u vs %u)",
                           ehdr.esis_version, ESIS_REQUIRED_VERSION );
       return;
     }

     proto_tree_add_uint( esis_tree, hf_esis_nlpi, tvb, 0, 1, ehdr.esis_nlpi );
     proto_tree_add_uint( esis_tree, hf_esis_length, tvb,
                          1, 1, ehdr.esis_length );
     proto_tree_add_uint( esis_tree, hf_esis_version, tvb, 2, 1, 
                          ehdr.esis_version );
     proto_tree_add_uint( esis_tree, hf_esis_reserved, tvb, 3, 1, 
                          ehdr.esis_reserved );

     pdu_type_string = val_to_str(ehdr.esis_type&OSI_PDU_TYPE_MASK,
                                  esis_vals, "Unknown (0x%x)");

     proto_tree_add_uint_format( esis_tree, hf_esis_type, tvb, 4, 1, 
                                 ehdr.esis_type, 
                                 pdu_type_format_string,
                                 pdu_type_string,
                                 (ehdr.esis_type&BIT_8) ? "1" : "0",
                                 (ehdr.esis_type&BIT_7) ? "1" : "0",
                                 (ehdr.esis_type&BIT_6) ? "1" : "0");

     tmp_uint = pntohs( ehdr.esis_holdtime );
     proto_tree_add_uint_format(esis_tree, hf_esis_holdtime, tvb, 5, 2, 
                                tmp_uint, "Holding Time  : %u seconds",
                                tmp_uint );

     tmp_uint = pntohs( ehdr.esis_checksum );
     
     switch (calc_checksum( tvb, 0, ehdr.esis_length, tmp_uint )) {

     case NO_CKSUM:
	cksum_status = "Not Used";
	break;

     case DATA_MISSING:
	cksum_status = "Not checkable - not all of packet was captured";
	break;

     case CKSUM_OK:
	cksum_status = "Is good";
	break;

     case CKSUM_NOT_OK:
	cksum_status = "Is wrong";
	break;

     default:
	cksum_status = NULL;
	g_assert_not_reached();
     }
     proto_tree_add_uint_format( esis_tree, hf_esis_checksum, tvb, 7, 2,
                                 tmp_uint, "Checksum      : 0x%x ( %s )", 
                                 tmp_uint, cksum_status );
   }


   /*
    * Let us make sure we use the same names for all our decodes
    * here.  First, dump the name into info column, and THEN
    * dispatch the sub-type.
    */
   if (check_col(pinfo->cinfo, COL_INFO)) {
     col_add_str(pinfo->cinfo, COL_INFO, 
                 val_to_str( ehdr.esis_type&OSI_PDU_TYPE_MASK, esis_vals,
                             "Unknown (0x%x)" ) );
   } 

   variable_len = ehdr.esis_length - ESIS_HDR_FIXED_LENGTH;

   switch (ehdr.esis_type & OSI_PDU_TYPE_MASK) {
     case ESIS_ESH_PDU:
          esis_dissect_esh_pdu( variable_len, tvb, esis_tree);
     break;
     case ESIS_ISH_PDU:
          esis_dissect_ish_pdu( variable_len, tvb, esis_tree);
     break;
     case ESIS_RD_PDU:
          esis_dissect_redirect_pdu( variable_len, tvb, esis_tree);
     break;
     default:
         esis_dissect_unknown(tvb, esis_tree,
			      "Unknown ESIS packet type 0x%x",
			      ehdr.esis_type & OSI_PDU_TYPE_MASK );
   }
} /* dissect_esis */


/*
 * Name: proto_register_esis()
 *
 * Description:
 *	main register for esis protocol set.  We register some display
 *	formats and the protocol module variables.
 *
 * 	NOTE: this procedure to autolinked by the makefile process that
 *	builds register.c
 *
 * Input: 
 *	void
 *
 * Output:
 *	void
 */
void 
proto_register_esis(void) {
  static hf_register_info hf[] = {
    { &hf_esis_nlpi,
      { "Network Layer Protocol Identifier", "esis.nlpi",	
        FT_UINT8, BASE_HEX, VALS(nlpid_vals), 0x0, "", HFILL }},
    { &hf_esis_length,
      { "PDU Length    ", "esis.length", FT_UINT8, BASE_DEC, NULL, 0x0, "", HFILL }},
    { &hf_esis_version,
      { "Version (==1) ", "esis.ver",    FT_UINT8, BASE_DEC, NULL, 0x0, "", HFILL }},
    { &hf_esis_reserved,
      { "Reserved(==0) ", "esis.res",    FT_UINT8, BASE_DEC, NULL, 0x0, "", HFILL }},
    { &hf_esis_type,
      { "PDU Type      ", "esis.type",   FT_UINT8, BASE_DEC, VALS(esis_vals),
         0xff, "", HFILL }},
    { &hf_esis_holdtime,
      { "Holding Time  ", "esis.htime",  FT_UINT16, BASE_DEC, NULL, 0x0, " s", HFILL }},
    { &hf_esis_checksum,
      { "Checksum      ", "esis.chksum", FT_UINT16, BASE_HEX, NULL, 0x0, "", HFILL }}
  };
  /*
   * 
   * 
   */
  static gint *ett[] = {
    &ett_esis,
    &ett_esis_area_addr,
  };

  proto_esis = proto_register_protocol( PROTO_STRING_ESIS, "ESIS", "esis");
  proto_register_field_array(proto_esis, hf, array_length(hf));
  proto_register_subtree_array(ett, array_length(ett));
};

void
proto_reg_handoff_esis(void)
{
  dissector_handle_t esis_handle;

  esis_handle = create_dissector_handle(dissect_esis, proto_esis);
  dissector_add("osinl", NLPID_ISO9542_ESIS, esis_handle);
}
