/* GKrellM Newsticker
|
|  Copyright (c) 2001 Tilman "Trillian" Sauerbeck
|
|  Author: Tilman "Trillian" Sauerbeck <tilman.sauerbeck@stud.tu-ilmenau.de>
|
|  Contributors: Daniel McNulty <dan.mcnulty@choicepoint.net>: Basic proxy code
|
|  Uses code from gkrellmwho by Ernesto Novillo (scroll code) and
|  seti@home by Xavier Serpaggi (various small snippets).
|  Implementation of the linked list was heavily inspired by GKrellM's inet.c
|  LED idea and pixmaps are from GKrellM's net.c
|
|  The panel's background pixmap was made by Peter Smith <psmith@twcny.rr.com>
|  
|
|	You will need GKrellM 1.2.x, libcurl 7.8+ and pthreads for this to work!
|
|
|  Version: 0.3
|
|  The latest version can be found at http://www.tu-ilmenau.de/~tisa-in/
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
|  
|  You should have received a copy of the GNU General Public License
|  along with this program; if not, write to the Free Software
|  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


#include <gkrellm/gkrellm.h>
#include <pthread.h>

#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>

#include "pixmaps/bg_panel.xpm"
#include "pixmaps/decal_leds.xpm"

#if (GKRELLM_VERSION_MAJOR<1) || (GKRELLM_VERSION_MAJOR==1 && GKRELLM_VERSION_MINOR<2)
#error This plugin requires GKrellM version >= 1.2.0
#endif

#define	CONFIG_NAME	"Newsticker"
#define	STYLE_NAME	"newsticker"


static void apply_plugin_config();


typedef struct
{
    Panel		*panel;
    Decal		*decal;
    gchar		*label;
    gchar		*headline;
    gchar		*url;
    gchar		*file;
    gchar		*link;
    gchar		*tooltip_text;
    gshort		headline_xoff;
    gshort		x_scroll;
    gshort		failures;
    GtkTooltips		*tooltip;
} Newsticker;

static Monitor		*monitor;

static gshort		counter;
static gshort		num_headlines;
static gshort		num_headlines;
static gshort		scroll_speed;
static gshort		selected_row;
static gshort		style_id;
static gshort		update_freq;
static gshort		led_value[2];

static gboolean		delete_list;
static gboolean		downloading;
static gboolean		download_files;
static gboolean		list_modified;
static gboolean		read_files;
static gboolean		use_proxy;
static gboolean		save_password;
static gboolean		purpose;

static Decal		*led[2];

static GList		*browser_list;
static GList		*newsticker_list;

static GtkWidget	*plugin_vbox;
static GtkWidget	*label_entry;
static GtkWidget	*url_entry;
static GtkWidget	*newsticker_clist;
static GtkWidget	*browser_combo;
static GtkWidget	*num_headlines_spin_button;
static GtkWidget	*update_freq_spin_button;
static GtkWidget	*scroll_speed_spin_button;
static GtkWidget	*proxy_entry;
static GtkWidget	*proxy_username_entry;
static GtkWidget	*proxy_password_entry;
static GtkWidget	*proxy_button;
static GtkWidget	*save_password_button;

static GdkImlibImage	*decal_leds_image;
static GdkImlibImage	*bg_panel_image;

static GdkPixmap	*decal_leds_pixmap;

static GdkBitmap	*decal_leds_mask;

static gchar		*browser;
static gchar		*proxy;
static gchar		*proxy_auth;
static gchar		*proxy_username;
static gchar		*proxy_password;
static gchar		*plugin_info_text[] = {
"<b>Newsticker:\n\n",
"This plugin scrolls news items retrieved from internet news services or similar sites.\n\n",
"Use the configuration dialog to integrate some news sites into the newsticker.\n",
"You have to supply a label and a URL for each item.\n",
"The URL should point to a *.RDF file, because only this format is supported\n",
"at the moment!\n\n",
"You can also change an entries position in GKrellM by selecting the entry in the list\n",
"and clicking the arrows.\n\n",
"When you need to download through a proxy, set those options in the ",
"<i>Setup ",
"tab.\n\n",
"The first time you start this plugin, three example entries are created.\n",
"I guess you have noticed this already ;)\n\n",
"For each entry you configure, a panel will be created. In this panel the three\n",
"latest headlines are scrolled continuously.\n\n",
"Every panel contains a tooltip which shows the label you set for the panel/news\n",
"service and the three headlines.\n\n",
"When you click a panel with the left mouse button, your browser will start and\n",
"show the site belonging to the panel you clicked.\n",
"Use the config dialog to set which browser is used (you have to use %s once in the\n",
"command, which will be replaced with the URL belonging to the panel).\n\n",
"Click a panel with the right mouse button to make the plugin update the headlines.\n",
"Anyway, every 10 minutes (default, you can change this of course) the headlines\n",
"are updated automatically.\n\n",
"Using the middle mouse button, you can directly open the config dialog for the\n",
"newsticker.\n\n",
"While downloading, the first LED in the top panel will blink.\n",
"When there's a new news item, the second LED will blink.\n\n",
"The speed the headlines are scrolled with can also be adjusted, as well as\n",
"the number of headlines that are scrolled.\n\n",
"You can also export your RDF sources list. When you import a list, the URLs\n",
"are appended to your list. Use the buttons in the config dialog for this.\n\n",
"Use the ",
"<i>Sources ",
"and ",
"<i>Setup ",
"tab in the config dialog to customize the plugin\n",
"according to your needs and wishes.\n"
};
static gchar		plugin_about_text[] =
"GKrellM Newsticker version 0.3\n\n\n"\
"Copyright (C) 2001 by Tilman \"Trillian\" Sauerbeck\n"\
"tilman.sauerbeck@stud.tu-ilmenau.de\n"\
"http://www.tu-ilmenau.de/~tisa-in/\n\n"\
"Released under the GPL.\n";



/*  Replaces all occurences of char one with char two  */
static gchar *convert_string(gchar *label, gchar one, gchar two)
{
    gchar		    *pt;
    gchar		    *tmp;

    
    tmp = g_strdup(label);
    for (pt = tmp; *pt; pt++)
	if (*pt == one)
	    *pt = two;


    return tmp;
}

	    
/* Needed by libcurl */
size_t write_data(void *ptr, size_t size, size_t nmemb, FILE *stream)
{
    size_t		written;

    
    written = fwrite(ptr, size, nmemb, stream);

    return written;
}


/* Downloads the headline-files */
static void get_headline()
{
    CURL		*curl;
    CURLcode		 res;
    FILE		*outfile;
    GList		*list;
    Newsticker		*nt;

   
#ifdef DEBUG
    if (GK.debug)
	if ((downloading) || (read_files))
	    g_warning("attempting to download while files are open!\n");
#endif
    
    downloading = TRUE;

    if ((curl = curl_easy_init()))
    {
	if (use_proxy)
        {
            curl_easy_setopt(curl, CURLOPT_PROXY, proxy);
            if (strcmp(proxy_auth, "none"))
		curl_easy_setopt(curl, CURLOPT_PROXYUSERPWD, proxy_auth);
        }
	curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
	curl_easy_setopt(curl, CURLOPT_TIMEOUT, 10);
        curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
        curl_easy_setopt(curl, CURLOPT_MUTE, 1);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_data);

	for (list = newsticker_list; list; list = list->next)
	{
	    nt =  (Newsticker *) list->data;
	    outfile = fopen(nt->file, "w");
            curl_easy_setopt(curl, CURLOPT_FILE, outfile);
	    curl_easy_setopt(curl, CURLOPT_URL, nt->url);
	    
	    if (!(res = curl_easy_perform(curl)))
	    {
		nt->failures = 0;
		led_value[0] = 3;
	    }
	    else
	    {
#ifdef DEBUG
		if (GK.debug)
		    g_warning("Error %i occured while downloading!\n", res);
#endif
		nt->failures++;
	    }
	    fclose(outfile);
	}
    }

    curl_easy_cleanup(curl);
    
    downloading = FALSE;
    read_files = TRUE;


    return;
}


/* Reads the downloaded headline-files and parses the headlines out of it */
static void read_headline(Newsticker *nt)
{
FILE		*f;
gchar		*pt;
gchar		*pt2;
gchar		*headline_old;
gchar		buf[256];
gshort		flag;
    

if ((f = fopen(nt->file, "r")))
{ 
    headline_old = g_strdup(nt->headline);
    flag = 0;
    nt->link = "NULL";
    while ((fgets(buf, sizeof(buf), f)))
    {
	if (((pt = strstr(buf, "<link>")) != NULL) && (!flag))
	{
	    pt = strchr(pt, '>');
	    pt++;
	    pt2 = strstr(buf, "</link>");
	    nt->link = g_strndup(pt, (pt2 - pt));
	    flag++;
	    continue;
	}
	if ((strstr(buf, "<item")) && (flag < 2))
	{
	    flag++;
	    continue;
	}
	if (((pt = strstr(buf, "<title>")) != NULL) && (flag >= 2) && (flag <= (num_headlines+1)))
	{
	    pt = strchr(pt, '>');
	    pt++;
	    pt2 = strstr(buf, "</title>");
	    if (flag == 2)
		nt->headline = g_strndup(pt, (pt2 - pt));
	    else
		nt->headline = g_strconcat(nt->headline, " --- ", g_strndup(pt, (pt2 - pt)), NULL);
	    flag++;
	    if (flag > (num_headlines+1))
		break;
	    continue;

	}
    }
    fclose(f);
	    
    if (strcmp(nt->headline, headline_old))
	    led_value[1] = 1;
    g_free(headline_old);
}
    
return;
}


/* Replaces HTML chars with their ASCII versions */
static void beautify_headline(Newsticker *nt)
{
    gshort		i;
    gchar		*pt;
    gchar		*strings[11] = {"&lt;", "&gt;", "&amp;", "&quot;", "&Auml;", "&auml;", "&Ouml;", "&ouml;", "&Uuml;", "&uuml;", "&szlig;"};
    gchar		*chars[11] = {"<", ">", "&", "\"", "", "", "", "", "", "", ""};
    
    
    for (i = 0; i < 11; i++)
	while ((pt = strstr(nt->headline, strings[i])) != NULL)
        {
	    nt->headline = g_strndup(nt->headline, (pt - nt->headline));
	    nt->headline = g_strconcat(nt->headline, chars[i], pt+(strlen(strings[i])), NULL);
        }
    

    return;
}


/* Compute the xoff value for a headline */
static gint compute_xoff(Newsticker *nt)
{
    gshort		xoff;
    
    
    xoff = -gdk_string_width(gkrellm_default_font(1), nt->headline);
    
    return xoff;
}


/* Updates the tooltips */
static void update_tooltip(Newsticker *nt)
{
    if ((update_freq * nt->failures) > 60)
	nt->tooltip_text = g_strdup_printf("%s: %s (outdated)", nt->label, nt->headline);
    else
	nt->tooltip_text = g_strdup_printf("%s: %s", nt->label, nt->headline);
    
    gtk_tooltips_set_tip(nt->tooltip, GTK_WIDGET(nt->panel->drawing_area), nt->tooltip_text, NULL);
    gtk_tooltips_enable(nt->tooltip);
   
    g_free(nt->tooltip_text);

    return;
}


static void update_plugin()
{
    static gshort	w;
    gshort		i;
    GList		*list;
    Newsticker		*nt;
    pthread_t		thread;
    
    
    if (!newsticker_list)
	return;
	    
    if (GK.second_tick)
    {
	counter++;
        if ((counter == (update_freq * 60)) || (download_files))
	{
	    counter = 0;
	    download_files = FALSE;
	    if (!downloading)
		pthread_create(&thread, NULL, (void *)&get_headline, NULL);
	}
    
	for (list = newsticker_list; list; list = list->next)
	{	
	    nt = (Newsticker *) list->data;
	    led_value[0] = 0;
	    led_value[1] = 2;
	}
    }
    
    if ((read_files) && (!downloading))
    {
	for (list = newsticker_list; list; list = list->next)
	{	
	    nt = (Newsticker *) list->data;
	    read_headline(nt);
	    beautify_headline(nt);
	    nt->headline_xoff = compute_xoff(nt);
	    update_tooltip(nt);
	}
	read_files = FALSE;
    }
    
    if (!w)
    {
	w = gkrellm_chart_width();
	for (list = newsticker_list; list; list = list->next)
	{
	    nt = (Newsticker *) list->data;
	    nt->x_scroll = w;
	}
    }

    for (i = 0, list = newsticker_list; list; list = list->next, i++)
    {
	nt = (Newsticker *) list->data;
    
	if (nt->x_scroll > nt->headline_xoff)
	    nt->x_scroll = nt->x_scroll - scroll_speed;
	else
	    nt->x_scroll = w;
		    
	nt->decal->x_off = nt->x_scroll;
	gkrellm_draw_decal_text(nt->panel, nt->decal, nt->headline, w - nt->x_scroll);
	if (!i)
	{
	    gkrellm_draw_decal_pixmap(nt->panel, led[0], led_value[0]);
	    gkrellm_draw_decal_pixmap(nt->panel, led[1], led_value[1]);
	}
	gkrellm_draw_panel_layers(nt->panel);
    }
}


static gint panel_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
    GList		    *list;
    Newsticker		    *nt;

    
    for (list = newsticker_list; list; list = list->next)
    {
	nt = (Newsticker *) list->data;
    
	if (widget == nt->panel->drawing_area)
	{
	    gdk_draw_pixmap(widget->window, widget->style->fg_gc[GTK_WIDGET_STATE (widget)], nt->panel->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y, ev->area.width, ev->area.height);
	    break;
	}
    }

    
    return FALSE;
}


static gint panel_click_event(GtkWidget *widget, GdkEventButton *ev)
{
    gchar		*command;
    GList		*list;
    Newsticker		*nt;
	
	
    for (list = newsticker_list; list; list = list->next)
    {
	nt = (Newsticker *) list->data;
		
	if (widget == nt->panel->drawing_area)
	{
	    if ((ev->button == 1) && (strcmp(nt->link, "NULL")))
	    {
		command = g_strdup_printf(browser, nt->link);
		command = g_strconcat(command, " &", NULL);
		system(command);
		g_free(command);
	    }
	    else if (ev->button == 2)
		download_files = TRUE;
	    else if (ev->button == 3)
		gkrellm_open_config_window(monitor);
			
	    break;
	}
    }
	
    return 1;
}


static gshort map_x(gshort x, gshort width)
{
    gshort		xnew;
    gshort		chart_width;


    xnew = x;
    chart_width = gkrellm_chart_width();
    if (GK.allow_scaling && GK.chart_width_ref != chart_width)
	xnew = xnew * chart_width / GK.chart_width_ref;
    if (x < 0)
	xnew = xnew + chart_width - width;
    
    return xnew;
}


GdkColor *set_text_color(gchar *string)
{
    GdkColormap *cmap;
    static GdkColor text_color;

    
    cmap = gtk_widget_get_colormap(gkrellm_get_top_window());
    gdk_color_parse(string, &text_color);
    gdk_color_alloc(cmap, &text_color);
    gdk_colormap_alloc_color(cmap, &text_color, TRUE, TRUE);
    
    
    return &text_color;
}


static void create_panel(GtkWidget *vbox, Newsticker *nt, gshort first_create, gshort num_panel)
{
    Style		*style;
    TextStyle		*ts;
    gshort		w;
    gint		rx_led_x;
    gint		rx_led_y;
    gint		tx_led_x;
    gint		tx_led_y;
	
	
    style = gkrellm_panel_style(style_id);
    ts = gkrellm_meter_textstyle(style_id);
	
    if (first_create)
	nt->panel = gkrellm_panel_new0();

    
    if (!gkrellm_style_is_themed(style))
    	ts->color = *set_text_color("white");
        
    gkrellm_load_image("bg_panel", bg_panel_xpm, &bg_panel_image, STYLE_NAME);
    gkrellm_panel_bg_image_override(nt->panel, bg_panel_image);
	
    nt->decal = gkrellm_create_decal_text(nt->panel, "Ay", ts, style, -1, -1, -1);
    nt->panel->textstyle = ts;

    if (!num_panel)
    {
	gkrellm_load_image("decal_leds", decal_leds_xpm, &decal_leds_image, STYLE_NAME);
	w = decal_leds_image->rgb_width;
	if (GK.allow_scaling && GK.chart_width_ref != gkrellm_chart_width())
		w = w * gkrellm_chart_width() / GK.chart_width_ref;
	gkrellm_render_to_pixmap(decal_leds_image, &decal_leds_pixmap, &decal_leds_mask, w, 0);

	if (!gkrellm_get_gkrellmrc_integer("newsticker_rx_led_x", &rx_led_x))
	    rx_led_x = -13;
	if (!gkrellm_get_gkrellmrc_integer("newsticker_rx_led_y", &rx_led_y))
	    rx_led_y = 0;
	if (!gkrellm_get_gkrellmrc_integer("newsticker_tx_led_x", &tx_led_x))
	    tx_led_x = -2;
	if (!gkrellm_get_gkrellmrc_integer("newsticker_tx_led_y", &tx_led_y))
	    tx_led_y = 0;

	led[0] = gkrellm_create_decal_pixmap(nt->panel, decal_leds_pixmap, decal_leds_mask, 4, NULL, 0, rx_led_y);
	led[1] = gkrellm_create_decal_pixmap(nt->panel, decal_leds_pixmap, decal_leds_mask, 4, NULL, 0, tx_led_y);
	
	led[0]->x = map_x(rx_led_x, led[0]->w);
	led[1]->x = map_x(tx_led_x, led[1]->w);

	gkrellm_draw_decal_pixmap(nt->panel, led[0], led_value[0]);
	gkrellm_draw_decal_pixmap(nt->panel, led[1], led_value[1]);
    }
	
    gkrellm_panel_configure(nt->panel, NULL, style);
    gkrellm_panel_create(vbox, monitor, nt->panel);
	
    if (first_create)
    {
	gtk_signal_connect(GTK_OBJECT (nt->panel->drawing_area), "expose_event", (GtkSignalFunc) panel_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT (nt->panel->drawing_area), "button_release_event", (GtkSignalFunc) panel_click_event, NULL);
		
	nt->headline = "Retrieving headline...";
	nt->link = "NULL";	
	nt->file = g_strdup_printf("%s/%s/newsticker/%s.rdf", gkrellm_homedir(), GKRELLM_DATA_DIR, convert_string(nt->label, ' ', '_'));
	download_files = TRUE;
    }
	
    nt->headline_xoff = compute_xoff(nt);
	
    if (!nt->tooltip)
    {
	nt->tooltip = gtk_tooltips_new();
	gtk_tooltips_set_delay(nt->tooltip, 500);
    }

    return;
}


static void create_plugin(GtkWidget *vbox, gint first_create)
{
    gshort		i;
    GList		*list;
	

    plugin_vbox = vbox;
	
    for (i = 0, list = newsticker_list; list; list = list->next, i++)
	create_panel(plugin_vbox, (Newsticker *)list->data, first_create, i);
}


static void save_list(FILE *f, gboolean export)
{
    gchar		*label;
    GList		*list;
    Newsticker		*nt;


    for (list = newsticker_list; list; list = list->next)
    {
        nt = (Newsticker *) list->data;

	//when saving, we convert spaces in labels to underscores
	label = convert_string(nt->label, ' ', '_');
                
	if (!export)        
	    fprintf(f, "newsticker sources %s %s\n", label, nt->url);
	else
	    fprintf(f, "%s %s\n", label, nt->url);
    }


    return;
}


static void load_list(FILE *f)
{
    Newsticker		*nt;
    gchar               label[26];
    gchar               url[76];
    gchar		buf[256];
    gchar		*entry[2];


    while ((fgets(buf, sizeof(buf), f)) != NULL)
    {
	nt = g_new0(Newsticker, 1);
        label[0] = '\0';
	url[0] = '\0';
        sscanf(buf, "%25s %75s", label, url);
	//when loading, we convert underscores in labels to spaces
	entry[0] = convert_string(label, '_', ' ');

	entry[1] = url;
	gtk_clist_append(GTK_CLIST(newsticker_clist), entry);
    }
    
    list_modified = TRUE;

    apply_plugin_config();

    return;
}


static void file_ok_sel(GtkWidget *w, GtkFileSelection *fs)
{
    FILE		*f;
   
    
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT (fs));

    if ((purpose) && ((f = fopen(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)), "w"))))
    {
	save_list(f, TRUE);
	fclose(f);
    }
    else if ((!purpose) && ((f = fopen(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)), "r"))))
    {	
	load_list(f);
	fclose(f);
    }
    else
	gkrellm_message_window("An error occured...", "Cannot access file - check the permissions!", NULL);


    return;
}


static void file_select(gchar *title)
{
    GtkWidget		*filesel;


    filesel = gtk_file_selection_new(title);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), g_strconcat(gkrellm_homedir(), "/", NULL));
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked", (GtkSignalFunc) file_ok_sel, filesel);
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(filesel));
    gtk_widget_show(filesel);


    return;
}


static void cb_export(GtkWidget *widget, gpointer drawer)
{
    purpose = TRUE;
    file_select("Save list as...");


    return;
}


static void cb_import(GtkWidget *widget, gpointer drawer)
{
    purpose = FALSE;
    file_select("Load list from...");


    return;
}


static void cb_up(GtkWidget *widget, gpointer drawer)
{
    GtkWidget		*clist;
    gshort		row;

	
    clist = newsticker_clist;
    row = selected_row;
    if (row > 0)
    {
	gtk_clist_row_move(GTK_CLIST(clist), row, row - 1);
	gtk_clist_select_row(GTK_CLIST(clist), row - 1, -1);
	if (gtk_clist_row_is_visible(GTK_CLIST(clist), row - 1)	!= GTK_VISIBILITY_FULL)
	    gtk_clist_moveto(GTK_CLIST(clist), row - 1, -1, 0.0, 0.0);
	selected_row = row - 1;
	list_modified = TRUE;
    }

    
    return;
}


static void cb_down(GtkWidget *widget, gpointer drawer)
{
    GtkWidget		*clist;
    gshort		row;

	
    clist = newsticker_clist;
    row = selected_row;
    if (row >= 0 && row < GTK_CLIST(clist)->rows - 1)
    {
	gtk_clist_row_move(GTK_CLIST(clist), row, row + 1);
	gtk_clist_select_row(GTK_CLIST(clist), row + 1, -1);
	if (gtk_clist_row_is_visible(GTK_CLIST(clist), row + 1) != GTK_VISIBILITY_FULL)
	    gtk_clist_moveto(GTK_CLIST(clist), row + 1, -1, 1.0, 0.0);
	selected_row = row + 1;
	list_modified = TRUE;
    }
}


static void reset_entries()
{
    gtk_entry_set_text(GTK_ENTRY(label_entry), "");
    gtk_entry_set_text(GTK_ENTRY(url_entry), "");


    return;
}


static void cb_selected(GtkWidget *clist, gint row, gint column, GdkEventButton *bevent, gpointer data)
{
    gchar	    *s;

    gtk_clist_get_text(GTK_CLIST(newsticker_clist), row, 0, &s);
    gtk_entry_set_text(GTK_ENTRY(label_entry), s);
    gtk_clist_get_text(GTK_CLIST(newsticker_clist), row, 1, &s);
    gtk_entry_set_text(GTK_ENTRY(url_entry), s);

    selected_row = row;
	
    return;
}


static void cb_unselected(GtkWidget *clist, gint row, gint column, GdkEventButton *bevent, gpointer data)
{
    selected_row = -1;
    reset_entries();
	
    return;
}
	

static void cb_enter(GtkWidget *widget, gpointer data)
{
    gchar	*buf[2];
	

    buf[0] = gkrellm_entry_get_text(&label_entry);
    buf[1] = gkrellm_entry_get_text(&url_entry);
    if ((!strlen(buf[0])) || (!strlen(buf[1])))
	return;
    if (strncmp(buf[1], "http://", 7))
	buf[1] = g_strdup_printf("http://%s", buf[1]);

    if (selected_row >= 0)
    {
	gtk_clist_set_text(GTK_CLIST(newsticker_clist), selected_row, 0, buf[0]);
	gtk_clist_set_text(GTK_CLIST(newsticker_clist), selected_row, 1, buf[1]);
	gtk_clist_unselect_row(GTK_CLIST(newsticker_clist), selected_row, 0);
	selected_row = -1;
    }
    else
	gtk_clist_append(GTK_CLIST(newsticker_clist), buf);

    reset_entries();
    list_modified = TRUE;

    
    return;
}


static void cb_delete(GtkWidget *widget, gpointer data)
{
    reset_entries();
	
    if (selected_row >= 0)
    {
	gtk_clist_remove(GTK_CLIST(newsticker_clist), selected_row);
	list_modified = TRUE;
	selected_row = -1;
    }
	
    
    return;
}


static void cb_use_proxy(GtkWidget *widget, gpointer data)
{
    gboolean		    enabled;


    enabled = GTK_TOGGLE_BUTTON(widget)->active;
    if (!enabled)
    {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(save_password_button), FALSE);
    }
    gtk_widget_set_sensitive(proxy_entry, enabled);
    gtk_widget_set_sensitive(proxy_username_entry, enabled);
    gtk_widget_set_sensitive(proxy_password_entry, enabled);
    gtk_widget_set_sensitive(save_password_button, enabled);
}

	
static void create_plugin_config(GtkWidget *tab_vbox)
{
    GtkWidget		    *tabs;
    GtkWidget		    *vbox;
    GtkWidget		    *vbox2;
    GtkWidget		    *hbox;
    GtkWidget		    *hbox2;
    GtkWidget		    *label;
    GtkWidget		    *info_label;
    GtkWidget		    *text;
    GtkWidget		    *scrolled;
    GtkWidget		    *button;
    GtkWidget		    *arrow;
    GList		    *list;
    Newsticker		    *nt;
    gchar		    *buf[2];
    gchar		    *titles[2] = {"Label", "URL"};
    gshort		    i;



    list_modified = FALSE;
    selected_row = -1;
	
/* Make a couple of tabs */
    tabs = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
    gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* Sources tab */
    vbox = gkrellm_create_framed_tab(tabs, "Sources");
	
    label = gtk_label_new("Label");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
   
    hbox2 = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), hbox2, FALSE, TRUE, 0);
   
    label_entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(label_entry), 25);
    gtk_widget_set_usize(label_entry, 180, 0);
    gtk_box_pack_start(GTK_BOX(hbox2), label_entry, FALSE, TRUE, 0);


    hbox2 = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox2), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox2), 5);
    gtk_box_pack_end(GTK_BOX(hbox), hbox2, FALSE, FALSE, 25);

    button = gtk_button_new_with_label("Import List");
    gtk_signal_connect(GTK_OBJECT(button), "clicked", (GtkSignalFunc) cb_import, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(hbox2), button);
    button = gtk_button_new_with_label("Export List");
    gtk_signal_connect(GTK_OBJECT(button), "clicked", (GtkSignalFunc) cb_export, NULL);
    gtk_box_pack_start_defaults(GTK_BOX(hbox2), button);

    
    label = gtk_label_new("URL");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);
	
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 5);
    url_entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(url_entry), 75);
    gtk_widget_set_usize(url_entry, 475, 0);
    gtk_box_pack_start(GTK_BOX(hbox), url_entry, FALSE, TRUE, 0);
	
    hbox = gtk_hbox_new(TRUE, 100);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 5);
	
    hbox2 = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox2), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox2), 5);
    gtk_box_pack_start(GTK_BOX(hbox), hbox2, FALSE, FALSE, 5);

    button = gtk_button_new_with_label("Enter");
    gtk_signal_connect(GTK_OBJECT(button), "clicked", (GtkSignalFunc) cb_enter, NULL);
    gtk_box_pack_start(GTK_BOX(hbox2), button, TRUE, TRUE, 0);

    button = gtk_button_new_with_label("Delete");
    gtk_signal_connect(GTK_OBJECT(button), "clicked", (GtkSignalFunc) cb_delete, NULL);
    gtk_box_pack_start(GTK_BOX(hbox2), button, TRUE, TRUE, 0);
	
    hbox2 = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox2), GTK_BUTTONBOX_START);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox2), 5);
    gtk_box_pack_start(GTK_BOX(hbox), hbox2, FALSE, FALSE, 5);
	
    button = gtk_button_new();
    arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_ETCHED_OUT);
    gtk_container_add(GTK_CONTAINER(button), arrow);
    gtk_signal_connect(GTK_OBJECT(button), "clicked", (GtkSignalFunc) cb_up, NULL);
    gtk_box_pack_start(GTK_BOX(hbox2), button, TRUE, TRUE, 0);

    button = gtk_button_new();
    arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_ETCHED_OUT);
    gtk_container_add(GTK_CONTAINER(button), arrow);
    gtk_signal_connect(GTK_OBJECT(button), "clicked", (GtkSignalFunc) cb_down, NULL);
    gtk_box_pack_start(GTK_BOX(hbox2), button, TRUE, TRUE, 0);
    

    scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);	
	
    newsticker_clist = gtk_clist_new_with_titles(2, titles);
    gtk_clist_set_shadow_type (GTK_CLIST(newsticker_clist), GTK_SHADOW_OUT);
    gtk_clist_column_titles_passive(GTK_CLIST(newsticker_clist));
    gtk_clist_set_column_justification(GTK_CLIST(newsticker_clist), 0, GTK_JUSTIFY_LEFT);
    gtk_clist_set_column_justification(GTK_CLIST(newsticker_clist), 1, GTK_JUSTIFY_LEFT);

    gtk_clist_set_column_width(GTK_CLIST(newsticker_clist), 0, 100);
    gtk_clist_set_column_width(GTK_CLIST(newsticker_clist), 1, 100);
	
    gtk_signal_connect (GTK_OBJECT(newsticker_clist), "select_row", (GtkSignalFunc) cb_selected, NULL);
    gtk_signal_connect (GTK_OBJECT(newsticker_clist), "unselect_row", (GtkSignalFunc) cb_unselected, NULL);
    gtk_container_add(GTK_CONTAINER(scrolled), newsticker_clist);
	
    for (i = 0, list = newsticker_list; list; i++, list = list->next)
    {
	nt = (Newsticker *) list->data;
	buf[0] = nt->label;
	buf[1] = nt->url;
	gtk_clist_append(GTK_CLIST(newsticker_clist), buf);
	gtk_clist_set_row_data(GTK_CLIST(newsticker_clist), i, nt);
    }
	
/* Setup */
    vbox = gkrellm_create_framed_tab(tabs, "Setup");
    vbox2 = gkrellm_framed_vbox(vbox, "Browser (%s will be replaced with URL)", 2, FALSE, 0, 5);
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);

    browser_combo = gtk_combo_new();
    gtk_widget_set_usize (GTK_WIDGET(browser_combo), 300, 0);
    gtk_box_pack_start(GTK_BOX(hbox), browser_combo, FALSE, TRUE, 0);
    browser_list = NULL;
    browser_list = g_list_append(browser_list, "mozilla '%s'");
    browser_list = g_list_append(browser_list, "netscape '%s'");
    gtk_combo_set_popdown_strings(GTK_COMBO(browser_combo), browser_list);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(browser_combo)->entry), browser);

    
    vbox2 = gkrellm_framed_vbox(vbox, "Misc", 2, FALSE, 0, 5);
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);
    label = gtk_label_new("Scroll");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
    gkrellm_spin_button(hbox, &num_headlines_spin_button, (gfloat) num_headlines, 1.0, 10.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE, "headlines");

    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);
    label = gtk_label_new("Update headlines every");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
    gkrellm_spin_button(hbox, &update_freq_spin_button, (gfloat) update_freq, 1.0, 60.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE, "minutes");

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);
    label = gtk_label_new("Scroll speed:");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
    gkrellm_spin_button(hbox, &scroll_speed_spin_button, (gfloat) scroll_speed, 1.0, 10.0, 1.0, 5.0, 0, 60, NULL, NULL, FALSE, NULL);

/* Proxy tab */
    vbox = gkrellm_create_framed_tab(tabs, "Proxy");
    
    gkrellm_check_button_connected(vbox, &proxy_button, use_proxy, FALSE, FALSE, 2, cb_use_proxy, NULL, "Use Proxy?");
   
    vbox2 = gkrellm_framed_vbox(vbox, NULL, 2, FALSE, 0, 5);
    
    label = gtk_label_new("Proxy (host:port)");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox2), label, FALSE, TRUE, 0);
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);
    
    proxy_entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(proxy_entry), 65);
    gtk_entry_set_text(GTK_ENTRY(proxy_entry), proxy);
    gtk_box_pack_start(GTK_BOX(hbox), proxy_entry, FALSE, TRUE, 0);
	
    label = gtk_label_new("Proxy username");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox2), label, FALSE, TRUE, 0);
				 
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);
						 
    proxy_username_entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(proxy_username_entry), 35);
    gtk_entry_set_text(GTK_ENTRY(proxy_username_entry), proxy_username);
    gtk_entry_set_visibility(GTK_ENTRY(proxy_username_entry), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), proxy_username_entry, FALSE, TRUE, 0);
										 
    label = gtk_label_new("Proxy password");
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_box_pack_start(GTK_BOX(vbox2), label, FALSE, TRUE, 0);
				 
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 5);
															 
    proxy_password_entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(proxy_password_entry), 35);
    gtk_entry_set_text(GTK_ENTRY(proxy_password_entry), proxy_password);
    gtk_entry_set_visibility(GTK_ENTRY(proxy_password_entry), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), proxy_password_entry, FALSE, TRUE, 0);
																					 
    save_password_button = gtk_check_button_new_with_label("Save password? (insecure)");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(save_password_button), save_password);
    gtk_box_pack_end(GTK_BOX(hbox), save_password_button, FALSE, TRUE, 0);

    gtk_widget_set_sensitive(proxy_entry, use_proxy);
    gtk_widget_set_sensitive(proxy_username_entry, use_proxy);
    gtk_widget_set_sensitive(proxy_password_entry, use_proxy);
    gtk_widget_set_sensitive(save_password_button, use_proxy);

/* Info tab */
    vbox = gkrellm_create_framed_tab(tabs, "Info");
    text = gkrellm_scrolled_text(vbox, NULL, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    for (i = 0; i < sizeof(plugin_info_text)/sizeof(gchar *); i++)
	gkrellm_add_info_text_string(text, plugin_info_text[i]);
	
/* About tab */
    info_label = gtk_label_new(plugin_about_text);
    label = gtk_label_new("About");
    gtk_notebook_append_page(GTK_NOTEBOOK(tabs), info_label, label);
}


static void save_plugin_config(FILE *f)
{
    fprintf(f, "newsticker use_proxy %i\n", use_proxy);
    fprintf(f, "newsticker save_password %i\n", save_password);
    fprintf(f, "newsticker proxy %s\n", proxy);
    fprintf(f, "newsticker proxy_username %s\n", proxy_username);
    if (save_password)
	fprintf(f, "newsticker proxy_password %s\n", proxy_password);
    fprintf(f, "newsticker num_headlines %i\n", num_headlines);
    fprintf(f, "newsticker scroll_speed %i\n", scroll_speed);
    fprintf(f, "newsticker update_freq %i\n", update_freq);
    fprintf(f, "newsticker browser %s\n", browser);
    
    save_list(f, FALSE);
}
	
	
static void load_plugin_config(gchar *arg)
{
    Newsticker		*nt;
    gchar		 plugin_config[64];
    gchar		 item[256];
    gchar		 label[26];
    gchar		 url[76];
    gshort		 n;
    gshort		 i;
	

    n = sscanf(arg, "%s %[^\n]", plugin_config, item);

    if (n == 2)
    {
        if (!strcmp(plugin_config, "use_proxy"))
	{
	    use_proxy = 0;
	    use_proxy = atoi(item);
        }
	else if (!strcmp(plugin_config, "save_password"))
        {
	    save_password = 0;
	    save_password = atoi(item);
	}
        else if (!strcmp(plugin_config, "proxy"))
	{
	    gkrellm_dup_string(&proxy, item);
	    if (!strcmp(proxy, "none:0"))
		gkrellm_dup_string(&proxy, "none");
	}
	else if (!strcmp(plugin_config, "proxy_username"))
	{
	    gkrellm_dup_string(&proxy_username, item);
	}
	else if (!strcmp(plugin_config, "proxy_password"))
	{
	    gkrellm_dup_string(&proxy_password, item);
	}
	else if (!strcmp(plugin_config, "num_headlines"))
	{
	    num_headlines = 3;
	    num_headlines = atoi(item);
	}

	else if (!strcmp(plugin_config, "scroll_speed"))
	{
	    scroll_speed = 0;
	    scroll_speed = atoi(item);
	}
	else if (!strcmp(plugin_config, "update_freq"))
	{
	    update_freq = 0;
	    update_freq = atoi(item);
	}
	else if (!strcmp(plugin_config, "browser"))
	{
	    gkrellm_dup_string(&browser, item);
	}
	else if (!strcmp(plugin_config, "sources"))
	{
	    if (delete_list)
	    {
		for (i = 0; i < 3; i++)
		{
		    nt = (Newsticker *) newsticker_list->data;
		    newsticker_list = g_list_remove(newsticker_list, nt);
		}
		delete_list = FALSE;
	    }
			
	    nt = g_new0(Newsticker, 1);
	    label[0] = '\0';
	    url[0] = '\0';
	    sscanf(item, "%25s %75s", label, url);
	    //when loading, we convert underscores in labels to spaces
	    nt->label = convert_string(label, '_', ' ');
	    gkrellm_dup_string(&nt->url, url);
	    newsticker_list = g_list_append(newsticker_list, nt);
	}
    }
    
    if ((proxy_password) && (proxy_username))
	gkrellm_dup_string(&proxy_auth, g_strconcat(proxy_username, ":", proxy_password, NULL));
    else if (proxy_username)
	gkrellm_dup_string(&proxy_auth, proxy_username);
}

static void apply_plugin_config()
{
    gshort	     i;
    gshort	     row;
    gshort	     num_headlines_old;
    gchar	    *s;
    GtkSpinButton   *spin;
    Newsticker	    *nt;
    Newsticker	    *nt_tmp;
    GList	    *list;
    GList	    *new_newsticker_list;


    spin = GTK_SPIN_BUTTON(num_headlines_spin_button);
    num_headlines_old = num_headlines;
    num_headlines = gtk_spin_button_get_value_as_int(spin);
    if (num_headlines != num_headlines_old)
	download_files = TRUE;
    
    spin = GTK_SPIN_BUTTON(scroll_speed_spin_button);
    scroll_speed = gtk_spin_button_get_value_as_int(spin);
	
    spin = GTK_SPIN_BUTTON(update_freq_spin_button);
    update_freq = gtk_spin_button_get_value_as_int(spin);

    gkrellm_dup_string(&browser, gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(browser_combo)->entry)));
    g_strdown(browser);

    use_proxy = GTK_TOGGLE_BUTTON(proxy_button)->active;
    if ((use_proxy) && (strcmp(proxy, "none")))
    {
	gkrellm_dup_string(&proxy, gtk_entry_get_text(GTK_ENTRY(proxy_entry)));
	gkrellm_dup_string(&proxy_username, gtk_entry_get_text(GTK_ENTRY(proxy_username_entry))); 
	gkrellm_dup_string(&proxy_password, gtk_entry_get_text(GTK_ENTRY(proxy_password_entry)));
	save_password = GTK_TOGGLE_BUTTON(save_password_button)->active;
	
	if (strncmp(proxy, "http://", 7))
	    proxy = g_strdup_printf("http://%s", proxy);
	
	if ((proxy_password) && (proxy_username))
	    gkrellm_dup_string(&proxy_auth, g_strconcat(proxy_username, ":", proxy_password, NULL));
	else if (proxy_username)
	    gkrellm_dup_string(&proxy_auth, proxy_username);
    }
    else
    {
	gkrellm_dup_string(&proxy, "none");
	gkrellm_dup_string(&proxy_username, "none");
	gkrellm_dup_string(&proxy_password, "none");
	save_password = 0;
	gtk_entry_set_text(GTK_ENTRY(proxy_entry), proxy);
	gtk_entry_set_text(GTK_ENTRY(proxy_username_entry), proxy_username);
	gtk_entry_set_text(GTK_ENTRY(proxy_password_entry), proxy_password);
    }

    if (list_modified)
    {
	new_newsticker_list = NULL;
		
	for (row = 0; row < GTK_CLIST(newsticker_clist)->rows; row++)
	{
	    nt = g_new0(Newsticker, 1);
	    new_newsticker_list = g_list_append(new_newsticker_list, nt);
	    nt_tmp = (Newsticker *) gtk_clist_get_row_data(GTK_CLIST(newsticker_clist),row);
	    gtk_clist_set_row_data(GTK_CLIST(newsticker_clist), row, nt);
		
	    gtk_clist_get_text(GTK_CLIST(newsticker_clist), row, 0, &s);
	    gkrellm_dup_string(&nt->label, s);
	    gtk_clist_get_text(GTK_CLIST(newsticker_clist), row, 1, &s);
	    gkrellm_dup_string(&nt->url, s);
	}
		
	while (newsticker_list)
	{
	    nt = (Newsticker *) newsticker_list->data;
	    gkrellm_remove_and_destroy_decal(nt->panel, nt->decal);
	    gkrellm_panel_destroy(nt->panel);
	    newsticker_list = g_list_remove(newsticker_list, nt);
	}
		
	newsticker_list = new_newsticker_list;
		
	for (i = 0, list = newsticker_list; list; list = list->next, i++)
	{
	    nt = (Newsticker *) list->data;
	    create_panel(plugin_vbox, nt, 1, i);
	}
		
	list_modified = FALSE;
	read_files = TRUE;
	download_files = TRUE;
    }
}


static Monitor plugin_mon =
{
    CONFIG_NAME,		/* Title for config clist. */
    0,				/* Id,  0 if a plugin */
    create_plugin,		/* The create function */
    update_plugin,		/* The update function */
    create_plugin_config,	/* The config tab create function */
    apply_plugin_config,	/* Apply the config function */

    save_plugin_config,		/* Save user config */
    load_plugin_config,		/* Load user config */
    "newsticker",		/* config keyword */

    NULL,			/* Undefined 2 */
    NULL,			/* Undefined 1 */
    NULL,			/* private */

    MON_CPU,			/* Insert plugin before this monitor */

    NULL,			/* Handle if a plugin, filled in by GKrellM */
    NULL			/* path if a plugin, filled in by GKrellM */
};


/* GKrellM exits, so clean up  */
static void cleanup()
{
    gchar	*command;
	
	
    //delete the downloaded RDF files:
    command = g_strdup_printf("rm -f %s/%s/newsticker/*.rdf", gkrellm_homedir(), GKRELLM_DATA_DIR);
    system(command);
    g_free(command);
	
    curl_global_cleanup();
	
    return;
}


/* Ripped from GKrellM's utils.c :D */
gboolean isdir(gchar *dir)
{
    struct stat		st;
    gint		result = FALSE;

	
    if (stat(dir, &st) == 0 && S_ISDIR(st.st_mode))
	result = TRUE;
    g_free(dir);
	
    return result;
}


/* All GKrellM plugins must have one global routine named init_plugin()
  |  which returns a pointer to a filled in monitor structure.
 */
Monitor * init_plugin()
{
    gchar		*data_dir;
    Newsticker		*nt;
    gshort		i;
	
	
    g_atexit(cleanup);
	
    /* If this next call is made, the background and krell images for this
    |  plugin can be custom themed by putting bg_meter.png or krell.png in the
    |  subdirectory STYLE_NAME of the theme directory.  Text colors (and
    |  other things) can also be specified for the plugin with gkrellmrc
    |  lines like:  StyleMeter  STYLE_NAME.textcolor orange black shadow
    |  If no custom theming has been done, then all above calls using
    |  style_id will be equivalent to style_id = DEFAULT_STYLE_ID.
    */
    style_id = gkrellm_add_meter_style(&plugin_mon, STYLE_NAME);

    scroll_speed = 2;
    update_freq = 10;
    counter = 0;
    num_headlines = 3;
    
    list_modified = FALSE;
    download_files = FALSE;
    read_files = FALSE;
    downloading = FALSE;
    delete_list = TRUE;
    use_proxy = FALSE;
    save_password = FALSE;
    purpose = FALSE;
    
    gkrellm_dup_string(&browser, "mozilla '%s'");
    gkrellm_dup_string(&proxy, "none");
    gkrellm_dup_string(&proxy_username, "none");
    gkrellm_dup_string(&proxy_password, "none");

    data_dir = g_strdup_printf("%s/%s/newsticker", gkrellm_homedir(), GKRELLM_DATA_DIR);
    if (!isdir(data_dir))
	mkdir(data_dir, 0755);

    curl_global_init(CURL_GLOBAL_NOTHING);
	
    for (i = 0; i < 3; i++)
    {
	nt = g_new0(Newsticker, 1);
	switch(i)
	{
	    case 0: nt->label = "Slashdot";
		    nt->url = "http://slashdot.org/slashdot.rdf";
		    break;
	    case 1: nt->label = "Gnotices";
		    nt->url = "http://news.gnome.org/gnome-news/rdf";
		    break;
	    case 2: nt->label = "KDE Dot News";
		    nt->url = "http://www.kde.org/dotkdeorg.rdf";
		    break;
	}
	newsticker_list = g_list_append(newsticker_list, nt);
    }
    
    monitor = &plugin_mon;
	
    return &plugin_mon;
}
