/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* setting-dialog.cpp
 *
 * Copyright (C) 2005 Takuo KITAME.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * Author: Takuo KITAME.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "peercast.h"

#include "utils.h"
#include "setting-dialog.h"
#include "gnome-peercast.h"

#define GLADE_FILE_NAME "setting-dialog.glade"

enum {
	CAT_ID,
	CAT_NAME,
	N_COLUMNS
};

enum {
	FILTER_IP,
	FILTER_EDITABLE,
	FILTER_DIRECT,
	FILTER_NETWORK,
	FILTER_PRIVATE,
	FILTER_BAN,
	FILTER_COLUMNS
};

static GtkTreeModel *
create_model (void)
{
	GtkListStore * store;

	store = gtk_list_store_new (N_COLUMNS,
				    G_TYPE_UINT,   /* ID */
				    G_TYPE_STRING  /* NAME */
		);
	return GTK_TREE_MODEL (store);
}

static GtkTreeModel *
filter_create_model (void)
{
	GtkListStore * store;

	store = gtk_list_store_new (FILTER_COLUMNS,
				    G_TYPE_STRING,   /* IP */
				    G_TYPE_BOOLEAN,  /* EDITABLE */
				    G_TYPE_BOOLEAN,  /* NETWORK */
				    G_TYPE_BOOLEAN,  /* Direct */
				    G_TYPE_BOOLEAN,  /* Private */
				    G_TYPE_BOOLEAN  /* Ban */
		);
	return GTK_TREE_MODEL (store);
}

static void
add_columns (GtkTreeView *tv)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("ID"), renderer,
							"text", CAT_ID,
							NULL);
	gtk_tree_view_column_set_visible (col, false);
	gtk_tree_view_append_column (tv, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Categories"),
							renderer,
							"text", CAT_NAME,
							NULL);
	gtk_tree_view_append_column (tv, col);
}

static void
filter_edited_cb (GtkCellRendererToggle *cell,
		  const gchar *path_string,
		  const gchar *new_text,
		  gpointer data)
{
	GtkTreeModel *model = GTK_TREE_MODEL (data);
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter iter;
	gint *col;

	col = (gint *)g_object_get_data (G_OBJECT (cell), "column");

	gtk_tree_model_get_iter (model, &iter, path);

	gtk_list_store_set (GTK_LIST_STORE (model),
			    &iter, col, new_text, -1);

	/* clean up */
	gtk_tree_path_free (path);
}

static void
filter_toggled_cb (GtkCellRendererToggle *cell,
		   const gchar *path_string,
		   gpointer data)
{
	GtkTreeModel *model = GTK_TREE_MODEL (data);
	GtkTreePath *path = gtk_tree_path_new_from_string (path_string);
	GtkTreeIter iter;
	gboolean ac;
	gint *col;

	col = (gint *)g_object_get_data (G_OBJECT (cell), "column");

	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter,
			    col, &ac,
			    -1);
	ac ^= 1;

	gtk_list_store_set (GTK_LIST_STORE (model),
			    &iter, col, ac, -1);

	/* clean up */
	gtk_tree_path_free (path);
}

static void
filter_add_columns (GtkTreeView *tv)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkTreeModel *model = gtk_tree_view_get_model (tv);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("IP"), renderer,
							"text", FILTER_IP,
							"editable", FILTER_EDITABLE,
							NULL);
	g_signal_connect (G_OBJECT (renderer), "edited",
			  G_CALLBACK (filter_edited_cb), model);
	gtk_tree_view_append_column (tv, col);

	renderer = gtk_cell_renderer_toggle_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Network"),
							renderer,
							"active",
							FILTER_NETWORK,
							NULL);
	g_object_set_data (G_OBJECT (renderer), "column",
			   (gint *)FILTER_NETWORK);
	g_signal_connect (renderer, "toggled",
			  G_CALLBACK (filter_toggled_cb), model);
	gtk_tree_view_append_column (tv, col);

	renderer = gtk_cell_renderer_toggle_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Direct"),
							renderer,
							"active",
							FILTER_DIRECT,
							NULL);
	g_object_set_data (G_OBJECT (renderer), "column",
			   (gint *)FILTER_DIRECT);
	g_signal_connect (renderer, "toggled",
			  G_CALLBACK (filter_toggled_cb), model);
	gtk_tree_view_append_column (tv, col);

	renderer = gtk_cell_renderer_toggle_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Private"),
							renderer,
							"active",
							FILTER_PRIVATE,
							NULL);
	g_object_set_data (G_OBJECT (renderer), "column",
			   (gint *)FILTER_PRIVATE);
	g_signal_connect (renderer, "toggled",
			  G_CALLBACK (filter_toggled_cb), model);
	gtk_tree_view_append_column (tv, col);

	renderer = gtk_cell_renderer_toggle_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Ban"),
							renderer,
							"active",
							FILTER_BAN,
							NULL);
	g_object_set_data (G_OBJECT (renderer), "column",
			   (gint *)FILTER_BAN);
	g_signal_connect (renderer, "toggled",
			  G_CALLBACK (filter_toggled_cb), model);
	gtk_tree_view_append_column (tv, col);
}

static void
cat_changed_cb (GtkTreeSelection *sel, gpointer data)
{
	SettingDialog *sd = (SettingDialog *)data;
	GtkWidget *cat_book = sd->getWidget ("cat_book");
	GtkTreeIter iter;
	gint id;
	gchar *text;
	gchar label[128];
	GtkWidget *cat_label = sd->getWidget ("cat_name_label");
	GtkTreeView *tv = GTK_TREE_VIEW (sd->getWidget ("catlist"));
	GtkTreeModel *model = gtk_tree_view_get_model (tv);

        if (! gtk_tree_selection_get_selected (sel, &model, &iter))
                return;

	gtk_tree_model_get (model, &iter,
			    CAT_ID, &id,
			    CAT_NAME, &text,
			    -1);
	g_snprintf (label, sizeof (label),
		    "<span weight=\"bold\" size=\"larger\">%s</span>",  text);

	gtk_label_set_markup (GTK_LABEL (cat_label), label);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (cat_book), id);
}

static void
list_init (GtkTreeView *tv)
{
	GtkTreeModel *model = create_model ();
	GtkListStore *store = GTK_LIST_STORE (model);
	GtkTreeIter iter;

	gtk_tree_view_set_model (tv, model);

	add_columns (tv);

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 0,
			    CAT_NAME, _("Basic"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 1,
			    CAT_NAME, _("Server"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 2,
			    CAT_NAME, _("Network"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 3,
			    CAT_NAME, _("Filters"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 4,
			    CAT_NAME, _("Security"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 5,
			    CAT_NAME, _("Authentication"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 6,
			    CAT_NAME, _("Log"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 7,
			    CAT_NAME, _("Root mode"),
			    -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 8,
			    CAT_NAME, _("Extend"),
			    -1);

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 9,
			    CAT_NAME, _("GUI"),
			    -1);

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    CAT_ID, 10,
			    CAT_NAME, _("Applications"),
			    -1);
}

static void
filter_init (GtkTreeView *tv)
{
	GtkTreeModel *model = filter_create_model ();

	gtk_tree_view_set_model (tv, model);

	filter_add_columns (tv);
}

static void
add_filter (GtkTreeView *tv, ServFilter *sf)
{
	GtkTreeModel *model = gtk_tree_view_get_model (tv);
	GtkTreeIter iter;
	GtkListStore *store = GTK_LIST_STORE (model);
	gchar ipStr[64];

	gtk_list_store_append (store, &iter);

	if (sf != NULL) {
		sf->host.IPtoStr (ipStr);

		gtk_list_store_set (store, &iter,
				    FILTER_IP, ipStr,
				    FILTER_EDITABLE, TRUE,
				    FILTER_NETWORK, sf->flags & ServFilter::F_NETWORK,
				    FILTER_DIRECT, sf->flags & ServFilter::F_DIRECT,
				    FILTER_PRIVATE, sf->flags & ServFilter::F_PRIVATE,
				    FILTER_BAN, sf->flags & ServFilter::F_BAN,
				    -1);
	} else {
		gtk_list_store_set (store, &iter,
				    FILTER_IP, "0.0.0.0",
				    FILTER_EDITABLE, TRUE,
				    FILTER_NETWORK, FALSE,
				    FILTER_DIRECT, FALSE,
				    FILTER_PRIVATE, FALSE,
				    FILTER_BAN, FALSE,
				    -1);
	}
}

static void
filter_selection_changed_cb (GtkTreeSelection *sel, gpointer data)
{
	gtk_widget_set_sensitive (GTK_WIDGET (data),
				  gtk_tree_selection_count_selected_rows (sel) > 0 ? TRUE : FALSE);
}

static void
filter_add_cb (GtkWidget *w, GtkTreeView *tv)
{
	add_filter (tv, NULL);
}

static void
filter_delete_cb (GtkWidget *w, GtkTreeView *tv)
{
	GtkTreeSelection *sel;
	GtkTreeIter iter;
	GtkTreeModel *model;

	sel = gtk_tree_view_get_selection (tv);
	gtk_tree_selection_get_selected (sel, &model, &iter);
	gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
}

static void
response_cb (GtkDialog *dlg, gint id, gpointer data)
{
	SettingDialog *sd = (SettingDialog *)data;

	switch (id) {
	case GTK_RESPONSE_CANCEL:
		gtk_widget_destroy (GTK_WIDGET (dlg));
		delete sd;
		break;
	case GTK_RESPONSE_OK:
		sd->apply();
		peercastInst->saveSettings();
		gtk_widget_destroy (GTK_WIDGET (dlg));
		delete sd;
		break;
	case GTK_RESPONSE_APPLY:
		sd->apply();
		break;
	}
}

static gboolean
apply_each_filter (GtkTreeModel *model, GtkTreePath *path,
		   GtkTreeIter *iter, gpointer data)
{
	gchar *ipStr;
	gint nw;
	gint di;
	gint pr;
	gint ba;
	ServFilter *curFilter = (ServFilter *)data;

	gtk_tree_model_get (model, iter,
			    FILTER_IP, &ipStr,
			    FILTER_NETWORK, &nw,
			    FILTER_DIRECT, &di,
			    FILTER_PRIVATE, &pr,
			    FILTER_BAN, &ba,
			    -1);
	curFilter = &servMgr->filters[servMgr->numFilters];
	curFilter->init ();
	curFilter->host.fromStrIP (ipStr, DEFAULT_PORT);
	if ((curFilter->host.ip) &&
	    (servMgr->numFilters < (ServMgr::MAX_FILTERS -1 ))) {
		servMgr->numFilters++;
		servMgr->filters[servMgr->numFilters].init();
	}
	if (nw)	curFilter->flags |= ServFilter::F_NETWORK;
	if (di)	curFilter->flags |= ServFilter::F_DIRECT;
	if (pr)	curFilter->flags |= ServFilter::F_PRIVATE;
	if (ba)	curFilter->flags |= ServFilter::F_BAN;
	return FALSE;
}
void
SettingDialog::applyFilters (void)
{
	GtkTreeModel *model = gtk_tree_view_get_model (filter_list);
	ServFilter *curFilter = servMgr->filters;
	servMgr->numFilters = 0;
	gtk_tree_model_foreach (model,
				apply_each_filter,
				curFilter);
}

static void
auto_keep_toggled (GtkToggleButton *b, GtkWidget *child)
{
	gtk_widget_set_sensitive (child, gtk_toggle_button_get_active (b));
	if (! gtk_toggle_button_get_active (b))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (child), FALSE);
}

static void
use_gnome_browser_toggled (GtkToggleButton *b, GtkWidget *child)
{
	gtk_widget_set_sensitive (child, ! gtk_toggle_button_get_active (b));
}

void
SettingDialog::setCurrent (void)
{
	GtkWidget *w;
	gchar buf[1024];
	gchar *c;
	gchar *mstr;
	GdkColor color;
	GnomePeercastApp *ap = (GnomePeercastApp *)peercastApp;

	/* Basic */
	w = getWidget ("entry_port");
	g_snprintf (buf, sizeof (buf), "%d", servMgr->serverHost.port);
	gtk_entry_set_text (GTK_ENTRY (w), buf);
	w = getWidget ("entry_password");
	g_snprintf (buf, sizeof (buf), "%s", servMgr->password);
	gtk_entry_set_text (GTK_ENTRY (w), buf);

	w = getWidget ("spin_max_relays");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), servMgr->maxRelays);
	w = getWidget ("spin_max_direct");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w), servMgr->maxDirect);


	/* Server */
	w = getWidget ("entry_message");
	g_snprintf (buf, sizeof (buf), "%s", chanMgr->broadcastMsg.cstr());
	gtk_entry_set_text (GTK_ENTRY (w), buf);

	w = getWidget ("spin_icy_interval");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   chanMgr->icyMetaInterval);

	w = getWidget ("combo_mode");
	if (servMgr->isRoot) {
		gtk_combo_box_set_active (GTK_COMBO_BOX (w), 1);
	} else {
		gtk_combo_box_set_active (GTK_COMBO_BOX (w), 0);
	}

	w = getWidget ("spin_refresh_html");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->refreshHTML);

	/* Network */
	w = getWidget ("entry_yp_address");
	g_snprintf (buf, sizeof (buf), "%s", servMgr->rootHost.cstr());
	gtk_entry_set_text (GTK_ENTRY (w), buf);

	w = getWidget ("spin_max_output");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->maxBitrateOut);

	w = getWidget ("spin_relays_per_channel");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   chanMgr->maxRelaysPerChannel);

	w = getWidget ("spin_max_control");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->maxControl);

	/* Filters */
	for (gint i = 0; i < servMgr->numFilters; i++) {
		ServFilter *sf = &servMgr->filters[i];
		add_filter (filter_list, sf);
	}

	/* Security */
	w = getWidget ("sec_port_1");
	g_snprintf (buf, sizeof (buf), "%u", servMgr->serverHost.port);
	gtk_label_set_text (GTK_LABEL (w), buf);
	w = getWidget ("sec_port_2");
	g_snprintf (buf, sizeof (buf), "%u", servMgr->serverHost.port + 1);
	gtk_label_set_text (GTK_LABEL (w), buf);

	w = getWidget ("sec_chk_html_1");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      (servMgr->allowServer1 & Servent::ALLOW_HTML));
	w = getWidget ("sec_chk_html_2");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      (servMgr->allowServer2 & Servent::ALLOW_HTML));

	w = getWidget ("sec_chk_broadcast_1");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      (servMgr->allowServer1 & Servent::ALLOW_BROADCAST));
	w = getWidget ("sec_chk_broadcast_2");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      (servMgr->allowServer2 & Servent::ALLOW_BROADCAST));

	w = getWidget ("sec_chk_network");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      (servMgr->allowServer1 & Servent::ALLOW_NETWORK));

	w = getWidget ("sec_chk_direct");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      (servMgr->allowServer1 & Servent::ALLOW_DIRECT));

	/* Authentication */
	w = getWidget ("combo_html_auth");
	gtk_combo_box_set_active (GTK_COMBO_BOX (w), servMgr->authType);

	w = getWidget ("combo_cookie_expire");
	gtk_combo_box_set_active (GTK_COMBO_BOX (w), servMgr->cookieList.neverExpire ? 1 : 0);

	/* Log */
	w = getWidget ("log_chk_debug");
	if ((servMgr->showLog & (1<<LogBuffer::T_DEBUG)) != 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), FALSE);
	w = getWidget ("log_chk_errors");
	if ((servMgr->showLog & (1<<LogBuffer::T_ERROR)) != 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), FALSE);
	w = getWidget ("log_chk_network");
	if ((servMgr->showLog & (1<<LogBuffer::T_NETWORK)) != 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), FALSE);
	w = getWidget ("log_chk_channels");
	if ((servMgr->showLog & (1<<LogBuffer::T_CHANNEL)) != 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), FALSE);

	/* Root mode */
	w = getWidget ("spin_root_hostupdate");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   chanMgr->hostUpdateInterval);
	w = getWidget ("entry_root_message");
	gtk_entry_set_text (GTK_ENTRY (w),
			    servMgr->rootMsg.cstr ());
	/* always TRUE */
	w = getWidget ("chk_root_get_update");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);
	/* always TRUE */
	w = getWidget ("chk_root_send");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), TRUE);

	w = getWidget ("spin_root_min_pgnu");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->minGnuIncoming);
	w = getWidget ("spin_root_max_pgnu");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->maxGnuIncoming);

	/* Extend */
	w = getWidget ("ext_chk_auto_keep");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      servMgr->autoRelayKeep > 0 ? TRUE : FALSE);
	w = getWidget ("ext_chk_auto_keep_opt");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      servMgr->autoRelayKeep == 2 ? TRUE : FALSE);
	gtk_widget_set_sensitive (w, servMgr->autoRelayKeep == 0 ? FALSE: TRUE);
	signalConnect ("ext_chk_auto_keep_toggled",
		       G_CALLBACK (auto_keep_toggled), w);

	w = getWidget ("ext_spin_kick_fw_numrelays");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->kickPushStartRelays);
	w = getWidget ("ext_spin_kick_fw_interval");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->kickPushInterval);

	w = getWidget ("ext_spin_auto_bump");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->autoBumpSkipCount);

	w = getWidget ("ext_chk_resolve_name");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      servMgr->enableGetName);
	w = getWidget ("ext_chk_allow_pcst");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      servMgr->allowConnectPCST);

	w = getWidget ("ext_spin_auto_max_relays");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (w),
				   servMgr->autoMaxRelaySetting);	

	/* GUI */
	w = getWidget ("gui_chk_resolve_connection");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      ap->getConfigBool (CONF_GUI_PREFIX"/name_resolve_connection"));
	w = getWidget ("gui_chk_resolve_hitlist");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      ap->getConfigBool (CONF_GUI_PREFIX"/name_resolve_hitlist"));

	mstr = ap->getConfigString (CONF_GUI_PREFIX"/color_direct");
	gdk_color_parse (mstr,  &color);
	g_free (mstr);
	w = getWidget ("gui_color_direct");
	gtk_color_button_set_color (GTK_COLOR_BUTTON (w), &color);

	mstr = ap->getConfigString (CONF_GUI_PREFIX"/color_firewalled");
	gdk_color_parse (mstr, &color);
	g_free (mstr);
	w = getWidget ("gui_color_firewalled");
	gtk_color_button_set_color (GTK_COLOR_BUTTON (w), &color);

	mstr = ap->getConfigString (CONF_GUI_PREFIX"/color_unfirewalled");
	gdk_color_parse (mstr, &color);
	g_free (mstr);
	w = getWidget ("gui_color_unfirewalled");
	gtk_color_button_set_color (GTK_COLOR_BUTTON (w), &color);

	mstr = ap->getConfigString (CONF_GUI_PREFIX"/color_push_relay");
	gdk_color_parse (mstr, &color);
	g_free (mstr);
	w = getWidget ("gui_color_push_relay");
	gtk_color_button_set_color (GTK_COLOR_BUTTON (w), &color);

	/* Applications */
	w = getWidget ("app_entry_ogg");
	c = ap->getConfigString (CONF_GLOBAL_PREFIX"/app_ogg");
	gtk_entry_set_text (GTK_ENTRY (w), c);
	g_free (c);
	w = getWidget ("app_entry_mp3");
	c = ap->getConfigString (CONF_GLOBAL_PREFIX"/app_mp3");
	gtk_entry_set_text (GTK_ENTRY (w), c);
	g_free (c);
	w = getWidget ("app_entry_nsv");
	c = ap->getConfigString (CONF_GLOBAL_PREFIX"/app_nsv");
	gtk_entry_set_text (GTK_ENTRY (w), c);
	g_free (c);
	w = getWidget ("app_entry_wma");
	c = ap->getConfigString (CONF_GLOBAL_PREFIX"/app_wma");
	gtk_entry_set_text (GTK_ENTRY (w), c);
	g_free (c);
	w = getWidget ("app_entry_wmv");
	c = ap->getConfigString (CONF_GLOBAL_PREFIX"/app_wmv");
	gtk_entry_set_text (GTK_ENTRY (w), c);
	g_free (c);
	w = getWidget ("app_entry_raw");
	c = ap->getConfigString (CONF_GLOBAL_PREFIX"/app_raw");
	gtk_entry_set_text (GTK_ENTRY (w), c);
	g_free (c);

	w = getWidget ("app_list_frame");
	signalConnect ("app_chk_use_gnome_browser_toggled",
		       G_CALLBACK (use_gnome_browser_toggled), w);
	w = getWidget ("app_chk_use_gnome_browser");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				      ap->getConfigBool (CONF_GLOBAL_PREFIX"/use_gnome_browser"));
}

void
SettingDialog::apply (void)
{
	gchar buf[1024];
	GtkWidget *w;
	const gchar *str;
	gint i;
	gboolean flag;
	gint s1 = 0, s2 = 0;
	gboolean brRoot, getUpdate;
	GdkColor color;
	GnomePeercastApp *ap = (GnomePeercastApp *)peercastApp;

	/* Basic */
	w = getWidget ("entry_port");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	i = strtol(str, NULL, 10);
	if (i != servMgr->serverHost.port) {
		servMgr->serverHost.port = i;
		servMgr->restartServer = true;
	}

	w = getWidget ("entry_password");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	strcpy (servMgr->password, str);

	w = getWidget ("spin_max_relays");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->setMaxRelays (i);

	w = getWidget ("spin_max_direct");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->maxDirect = i;

	/* Server */
	w = getWidget ("entry_message");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	{
		String msg;
		msg.set (str, String::T_ESC);
		chanMgr->setBroadcastMsg (msg);
	}

	w = getWidget ("spin_icy_interval");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	{
		if (i < 0) i = 0;
		else if (i > 16384) i = 16384;
		chanMgr->icyMetaInterval = i;
	}

	w = getWidget ("combo_mode");
	i = gtk_combo_box_get_active (GTK_COMBO_BOX (w));
	if (i == 1)
		servMgr->isRoot = true;
	else
		servMgr->isRoot = false;

	w = getWidget ("spin_refresh_html");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->refreshHTML = i;

	/* Network */
	w = getWidget ("entry_yp_address");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	{
		String yp (str, String::T_ESC);
		yp.convertTo (String::T_ASCII);
		servMgr->rootHost = yp;
	}

	w = getWidget ("spin_max_output");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->maxBitrateOut = i;

	w = getWidget ("spin_relays_per_channel");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	chanMgr->maxRelaysPerChannel = i;

	w = getWidget ("spin_max_control");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->maxControl = i;

	/* Filters */
	applyFilters ();

	/* Security */
	w = getWidget ("sec_chk_html_1");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	s1 |= flag ? (Servent::ALLOW_HTML):0;

	w = getWidget ("sec_chk_html_2");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	s2 |= flag ? (Servent::ALLOW_HTML):0;

	w = getWidget ("sec_chk_broadcast_1");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	s1 |= flag ? (Servent::ALLOW_BROADCAST):0;

	w = getWidget ("sec_chk_broadcast_2");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	s2 |= flag ? (Servent::ALLOW_BROADCAST):0;

	w = getWidget ("sec_chk_network");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	s1 |= flag ? (Servent::ALLOW_NETWORK):0;

	w = getWidget ("sec_chk_direct");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	s1 |= flag ? (Servent::ALLOW_DIRECT):0;

	servMgr->allowServer1 = s1;
	servMgr->allowServer2 = s2;

	/* Authentication */
	w = getWidget ("combo_html_auth");
	i = gtk_combo_box_get_active (GTK_COMBO_BOX (w));
	if (i == 0)
		servMgr->authType = ServMgr::AUTH_COOKIE;
	else
		servMgr->authType = ServMgr::AUTH_HTTPBASIC;

	w = getWidget ("combo_cookie_expire");
	i = gtk_combo_box_get_active (GTK_COMBO_BOX (w));
	if (i == 1)
		servMgr->cookieList.neverExpire = true;
	else
		servMgr->cookieList.neverExpire = false;

	/* Log */
	i = 0;
	w = getWidget ("log_chk_debug");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	i |= flag ? (1<<LogBuffer::T_DEBUG):0;
	w = getWidget ("log_chk_errors");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	i |= flag ? (1<<LogBuffer::T_ERROR):0;
	w = getWidget ("log_chk_network");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	i |= flag ? (1<<LogBuffer::T_NETWORK):0;
	w = getWidget ("log_chk_channels");
	flag = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	i |= flag ? (1<<LogBuffer::T_CHANNEL):0;
	servMgr->showLog = i;

	/* Root mode */
	w = getWidget ("spin_root_hostupdate");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	chanMgr->hostUpdateInterval = i;
	w = getWidget ("entry_root_message");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	servMgr->rootMsg.set (str, String::T_ESC);
	servMgr->rootMsg.convertTo (String::T_UNICODE);

	w = getWidget ("chk_root_get_update");
	getUpdate = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	w = getWidget ("chk_root_send");
	brRoot = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	w = getWidget ("spin_root_min_pgnu");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->minGnuIncoming = i;
	w = getWidget ("spin_root_max_pgnu");
	i = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));
	servMgr->maxGnuIncoming = i;

	/* Extend */
	w = getWidget ("ext_chk_auto_keep");
	servMgr->autoRelayKeep =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) ? 1 : 0;

	w = getWidget ("ext_chk_auto_keep_opt");
	servMgr->autoRelayKeep =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)) ? 2 : servMgr->autoRelayKeep;

	w = getWidget ("ext_spin_kick_fw_numrelays");
	servMgr->kickPushStartRelays = 
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));

	w = getWidget ("ext_spin_kick_fw_interval");
	servMgr->kickPushInterval =
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));

	w = getWidget ("ext_spin_auto_bump");
	servMgr->autoBumpSkipCount =
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));

	w = getWidget ("ext_chk_resolve_name");
	servMgr->enableGetName = 
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
	w = getWidget ("ext_chk_allow_pcst");
	servMgr->allowConnectPCST =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	w = getWidget ("ext_spin_auto_max_relays");
	servMgr->autoMaxRelaySetting =
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (w));


	/* GUI */
	w = getWidget ("gui_chk_resolve_connection");
	ap->setConfigBool (CONF_GUI_PREFIX"/name_resolve_connection",
			   gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)));

	w = getWidget ("gui_chk_resolve_hitlist");
	ap->setConfigBool (CONF_GUI_PREFIX"/name_resolve_hitlist",
			   gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)));

	w = getWidget ("gui_color_direct");
	gtk_color_button_get_color (GTK_COLOR_BUTTON (w), &color);
	gdk_color_hex_string (color, buf, sizeof (buf));
	ap->setConfigString (CONF_GUI_PREFIX"/color_direct", buf);

	w = getWidget ("gui_color_firewalled");
	gtk_color_button_get_color (GTK_COLOR_BUTTON (w), &color);
	gdk_color_hex_string (color, buf, sizeof (buf));
	ap->setConfigString (CONF_GUI_PREFIX"/color_firewalled", buf);

	w = getWidget ("gui_color_unfirewalled");
	gtk_color_button_get_color (GTK_COLOR_BUTTON (w), &color);
	gdk_color_hex_string (color, buf, sizeof (buf));
	ap->setConfigString (CONF_GUI_PREFIX"/color_unfirewalled", buf);

	w = getWidget ("gui_color_push_relay");
	gtk_color_button_get_color (GTK_COLOR_BUTTON (w), &color);
	gdk_color_hex_string (color, buf, sizeof (buf));
	ap->setConfigString (CONF_GUI_PREFIX"/color_push_relay", buf);

	/* Root mode */
	if ((servMgr->isRoot) && (brRoot))
		servMgr->broadcastRootSettings(getUpdate);

	/* Applications */
	w = getWidget ("app_entry_ogg");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	ap->setConfigString (CONF_GLOBAL_PREFIX"/app_ogg", (gchar *)str);

	w = getWidget ("app_entry_mp3");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	ap->setConfigString (CONF_GLOBAL_PREFIX"/app_mp3", (gchar *)str);

	w = getWidget ("app_entry_nsv");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	ap->setConfigString (CONF_GLOBAL_PREFIX"/app_nsv", (gchar *)str);

	w = getWidget ("app_entry_wma");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	ap->setConfigString (CONF_GLOBAL_PREFIX"/app_wma", (gchar *)str);

	w = getWidget ("app_entry_wmv");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	ap->setConfigString (CONF_GLOBAL_PREFIX"/app_wmv", (gchar *)str);

	w = getWidget ("app_entry_raw");
	str = gtk_entry_get_text (GTK_ENTRY (w));
	ap->setConfigString (CONF_GLOBAL_PREFIX"/app_raw", (gchar *)str);

	w = getWidget ("app_chk_use_gnome_browser");
	ap->setConfigBool (CONF_GLOBAL_PREFIX"/use_gnome_browser",
			   gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w)));
	ap->gconfInit ();
}

GtkWidget *
SettingDialog::getWidget (const gchar *name)
{
	return glade_xml_get_widget (xml, name);
}

SettingDialog::SettingDialog (GtkWindow *window)
{
	GtkTreeIter iter;
	GtkWidget *dialog;

	debug_print (1, "New object: SettingDialog");

	parent = window;

	xml =load_glade_xml (GLADE_FILE_NAME, "setting_dlg");
	dialog = getWidget ("setting_dlg");
	gtk_widget_hide (dialog);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);

	catlist = GTK_TREE_VIEW (getWidget ("catlist"));
	selection = gtk_tree_view_get_selection (catlist);

	filter_list = GTK_TREE_VIEW (getWidget ("filter_list"));
	filter_sel = gtk_tree_view_get_selection (filter_list);

	list_init (catlist);
	filter_init (filter_list);
	glade_xml_signal_connect_data (xml, "filter_add",
				       G_CALLBACK (filter_add_cb),
				       filter_list);
	glade_xml_signal_connect_data (xml, "filter_delete",
				       G_CALLBACK (filter_delete_cb),
				       filter_list);
	g_signal_connect (G_OBJECT (filter_sel), "changed",
			  G_CALLBACK (filter_selection_changed_cb),
			  getWidget ("button_filter_delete"));
	gtk_widget_set_sensitive (getWidget ("button_filter_delete"), FALSE);

	setCurrent ();

	g_signal_connect (G_OBJECT (selection), "changed",
			  G_CALLBACK (cat_changed_cb), this);

	gtk_tree_model_get_iter_first (gtk_tree_view_get_model (catlist),
				       &iter);
	gtk_tree_selection_select_iter (selection, &iter);

	glade_xml_signal_connect_data (xml, "setting_dlg_response",
				       G_CALLBACK (response_cb), this);
	gtk_widget_show (dialog);
}

