/*!
    @file     SharedSQL_ISQLCache.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    SQLCache for calling SharedSQL from SQLManager
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SharedSQL_ISQLCACHE_HPP
#define SharedSQL_ISQLCACHE_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_IHandles.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandCacheStatistics.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandCacheIterator.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

class SharedSQL_CommandCache;
class SAPDBMem_RawAllocator;

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_ISQLCache
    @brief          Interface to SharedSQL_SQLCache
    @copydoc        SAPDBMem_IRawAllocator
 */
class SharedSQL_ISQLCache : public SAPDBMem_IRawAllocator
{
public:
    /*!
        @brief  Get an interface to a SharedSQL_PrepareHandle for some SharedSQL_CachedCommand
        @param  Alloc    [in]
        @param  Context  [in]
        @param  Stmt     [in]
        @param  Internal [in]
        @return SharedSQL_IPrepareHandle*
    */
	virtual SharedSQL_IPrepareHandle* NewPrepareHandle( SAPDBMem_IRawAllocator& Alloc, SharedSQL_SQLContext& Context, SharedSQL_SQLStmt& Stmt, SAPDB_Bool Internal ) = 0;
    
    /*!
        @brief  Destroy the Handle created with NewPrepareHandle
        @param  PH [in]
    */
    virtual void DestroyPrepareHandle( SharedSQL_IPrepareHandle* PH ) = 0;

    /*!
        @brief  Get an interface to a SharedSQL_ExecuteHandle for some ParseID
        @param  Alloc          [in]
        @param  PID            [in]
        @param  CountExecuting [in]
        @param  Valid           [out]
        @param  M9              [out]
        @return SharedSQL_IExecuteHandle*
    */
	virtual SharedSQL_IExecuteHandle* NewExecuteHandle( SAPDBMem_IRawAllocator& Alloc, const SharedSQL_ParseID& PID, bool CountExecuting, SAPDB_Bool& Valid, SAPDB_Bool& M9 ) = 0;

    /*!
        @brief  Destroy the Handle created with NewExecuteHandle
        @param  EH [in]
    */
	virtual void DestroyExecuteHandle( SharedSQL_IExecuteHandle* EH ) = 0;

    /*!
        @brief  Add parse and modulename for some command identified its prepare handle
        @param  PH          [in]
        @param  PID         [in]
        @param  ModuleName  [in]
        @param  M9          [in]
        @return true, if ok, else false
    */
    virtual SAPDB_Bool PutParseID( const SharedSQL_IPrepareHandle* PH, const SharedSQL_ParseID& PID, const SharedSQL_ModuleName& ModuleName, const SAPDB_Bool M9 ) = 0;

    /*!
        @brief  Drop some parseid previously added with PutParseID
        @param  PID [in]
        @return true, if ok, else false
    */
	virtual SAPDB_Bool DropParseID( const SharedSQL_ParseID& PID ) = 0;

    /*!
        @brief  Get the module name for some command identified by parseid
        @param  PID [in]
        @return const SharedSQL_ModuleName*
    */
    virtual const SharedSQL_ModuleName* GetModuleName( const SharedSQL_ParseID& PID ) = 0;
    //

    /*!
        @brief  Free as much memory as possible by destroying unnecessary elements
    */
    virtual void CleanUpAll( void ) = 0;

    /*!
        @brief  Invalidate all commands within the cache
    */
    virtual void InvalidateAll( void ) = 0;
    //

    /*!
        @brief  Set statistic data from execution for some command identified by parseid
        @param  PID [in]
        @param  ExecuteTimeSec
        @param  ExecuteTimeMicroSec
        @param  ReadRowCount
        @param  QualifiedRowCount
        @param  VirtualReadCount
        @param  PhysicalReadCount
        @param  FetchRowCount
        @param  SuspendCount
        @param  WaitCount
        @return true, if ok, else false
    */
    virtual SAPDB_Bool SetExecuteStatisticData( const SharedSQL_ParseID& PID,
                                        SharedSQL_CommandID& CommandID,
                                        SAPDB_UInt8 ExecuteTimeSec, SAPDB_UInt8 ExecuteTimeMicroSec,
                                        SAPDB_Int8 ReadRowCount, SAPDB_Int8 QualifiedRowCount,
                                        SAPDB_Int8 VirtualReadCount, SAPDB_Int8 PhysicalReadCount,
                                        SAPDB_Int8 FetchRowCount, 
                                        SAPDB_Int8 SuspendCount, SAPDB_Int8 WaitCount ) = 0;
    //

    /*!
        @brief  Get information about the CommandCache
        @param  S [out]
    */
    virtual void GetCommandCacheInfo( SharedSQL_CommandCacheInfo& S ) = 0;

    /*!
        @brief  Get information about the CommandCache (reset version)
        @param  S [out]
    */
    virtual void GetCommandCacheInfoReset( SharedSQL_CommandCacheInfo& S ) = 0;

    /*!
        @brief  Reset statistics information of the cache
    */
    virtual void ResetCommandCacheInfo( void ) = 0;

    /*!
        @brief  Reset statistics information of all stored commands
    */
    virtual void ResetCommandInfos( void ) = 0;

    /*!
        @brief  Get an iterator for the cached commands
    */
    virtual SharedSQL_CommandCacheIterator GetCommandCacheIterator( void ) = 0;
    //

    /// Just for debug purpose
    virtual SAPDBMem_RawAllocator* GetCommandCacheBaseRawAllocator( void ) = 0; // just for Testing !!!

};  // SharedSQL_ISQLCache

#endif  /* SharedSQL_ISQLCACHE_HPP */

