/*
  -----------------------------------------------------------------------------

  module: vls51.cpp

  -----------------------------------------------------------------------------

  responsible:  SteffenS, d033893

  special area: Loader

  description:  Support class for TABLELOAD functionality.

  version:      7.5.

  -----------------------------------------------------------------------------

  Copyright (c) 2000-2004 SAP AG-2003

  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/


#include "gls00.h"
#include "hls07.h"
#include "hls30.h"
#include "hls51.h"
#include "hls98msg.h"
#include "hls99.h"

/*
  -----------------------------------------------------------------------------
  function:     Constructor
  -----------------------------------------------------------------------------
*/
ReadBuffer::ReadBuffer(tls00_VFile *VFile, int SingleBufSize, int SingleBufCount)
{
    ROUTINE_DBG_MEO00 ("ReadBuffer");

    m_FileStruct = VFile;

    m_BufferSize     = SingleBufSize * SingleBufCount;
    m_BufferPartSize = SingleBufSize;
    m_Buffer         = new _TCHAR[m_BufferSize];      // no trailing NULL
    
    m_UpperBound     = SingleBufCount - 1;            // we're in the times of C
    
    m_Index          = m_UpperBound;                  // force first buffering of data pages
    
    m_ErrorCode      = errOK_els00;
    m_ErrorText[0]   = 0;
    m_EOFReached     = false;

    m_notUsed        = false;
}


/*
  -----------------------------------------------------------------------------
  function:     Destructor
  -----------------------------------------------------------------------------
*/
ReadBuffer::~ReadBuffer()
{
    // We need to reinitialize the file position because of some (maybe) unused
    // data that has already been read (the class reads ahead)
    // This can (at this time) only be done for streams of type FILE; all
    // other stream types do not allow seeking backwards.
    if (NULL != m_FileStruct)
    {
        if ( (dtypeFile_ls00 == m_FileStruct->vfFileType_ls00) && (m_Index != m_UpperBound) )
        {
            tsp00_Longint lDistance = -(m_UpperBound - m_Index + 1)*m_BufferPartSize;
            if (false == m_notUsed)
            {
                lDistance += m_BufferPartSize;
            }
            ls30VFSeekLong(*m_FileStruct, lDistance, m_ErrorText);
        }
    }
        
    if (NULL != m_Buffer)
    {
        delete [] m_Buffer;
        m_Buffer = NULL;
    }
}


/*
  -----------------------------------------------------------------------------
  function:     ls51GetNext
  -----------------------------------------------------------------------------
*/
_TCHAR *
ReadBuffer::ls51GetNext()
{
    if (false == m_notUsed)
    {
        if (m_Index == m_UpperBound)
        {
            if  (true == m_EOFReached)
            {
                m_ErrorCode = EOF_LS00;
                return NULL;
            }

            // Not reached eof -> get new data from file
            tsp00_Longint ReadLen = 0;

            ReadLen = ls30VFReadLongBin(m_Buffer, *m_FileStruct, m_BufferSize, &m_FileStruct->vfFilePos_ls00, m_ErrorText);
            if (ReadLen >= 0)
            {
                if (ReadLen < m_BufferSize)
                {
                    m_UpperBound = STATIC_CAST(tsp00_Int4, ReadLen) / m_BufferPartSize - 1;
                    m_EOFReached = true;
                }
                m_Index = 0;        // Reinitialisieren
            }
            else
            {
                m_ErrorCode = STATIC_CAST(tsp00_Int4, ReadLen);      // ReadLen = -1 signals EOF
                return NULL;
            }
        } //if (m_Index == m_UpperBound)
        else
        {
            m_Index++;
        }
    }
    else
    {
        m_notUsed = false;
    }

    // Achtung: eine schwaeche - wir geben hier einen pointer in einen puffer zurueck; damit ist
    // allerdings auch noch der rest des puffers vom aufrufer zugreifbar!!!

    return &m_Buffer[m_Index * m_BufferPartSize];
}
// ls51GetNext()


/*
  -----------------------------------------------------------------------------
  function:     ls51GetError
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ReadBuffer::ls51GetError(_TCHAR *pszErrText)
{
    if (EOF_LS00 != m_ErrorCode)        // error
    {
        size_t Len = _tcslen(m_ErrorText);
        Len = (Len < MAX_REPLY_LENGTH_LS00) ? Len : (MAX_REPLY_LENGTH_LS00 - 1);
        memcpy(pszErrText, m_ErrorText, Len);
        pszErrText[Len] = 0;
    }

    return m_ErrorCode;
}
// ls51GetError()


/*
  -----------------------------------------------------------------------------
  function:     ls51ResetFilePos
  -----------------------------------------------------------------------------
*/
tsp00_Int4
ReadBuffer::ls51ResetFilePos()
{
    ROUTINE_DBG_MEO00 ("ls51ResetFilePos");
    
    tsp00_Int4 rc = errOK_els00;

    tsp00_Longint lDistance  = ls51GetCountOfUnusedPages();

    TRACE_PRNF_MLS99(("ls51ResetFilePos", "Unused pages = %d", lDistance) );

    if (NULL != m_FileStruct)
    {
        if ( (dtypeFile_ls00 == m_FileStruct->vfFileType_ls00) && (0 != lDistance) )
        {
            lDistance *= (-m_BufferPartSize);
            rc =ls30VFSeekLong(*m_FileStruct, lDistance, m_ErrorText);
        
            // Force reloading buffer after resetting file pos with next request of buffer
            m_Index = m_UpperBound;
        }
    }
    return rc;
}


/*
  -----------------------------------------------------------------------------
  function:     ls51RecalculateFilePos
  -----------------------------------------------------------------------------
*/
void
ReadBuffer::ls51RecalculateFilePos(tls00_LargeNumber &NewPos)
{
    ROUTINE_DBG_MEO00 ("ls51RecalculateFilePos");
    
    if (NULL != m_FileStruct)
    {
        NewPos = m_FileStruct->vfFilePos_ls00;

        tsp00_Longint lDistance  = ls51GetCountOfUnusedPages();
        
        if (0 != lDistance)
        {
            TRACE_PRNF_MLS99(("ls51RecalculateFilePos", "Unused pages = %d", lDistance) );
    
            tls00_LargeNumber Subtrahend;
            Subtrahend.lnu2GB_Blocks_ls00 = 0;
            Subtrahend.lnuRemainder_ls00  = lDistance * m_BufferPartSize;

            NewPos = ls07SubtractLargeNumbers(m_FileStruct->vfFilePos_ls00, Subtrahend);
        }
    }
    else
    {
        NewPos.lnu2GB_Blocks_ls00 = -1;
        NewPos.lnuRemainder_ls00  = -1;
    }
}
// ls51RecalculateFilePos(()
