/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: vvector.hxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 08:02:02 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef CSV_VVECTOR_HXX
#define CSV_VVECTOR_HXX

// USED SERVICES
	// BASE CLASSES
	// COMPONENTS
	// PARAMETERS
#include <vector>
#include <cosv/template/tpltools.hxx>


namespace csv
{
                                            
namespace vvector
{
            
            
template <class TYPE>
struct delete_ptrs
{             
    static void         Destruct(
                            std::vector< TYPE* > & 
                                                v) 
                        { csv::erase_container_of_heap_ptrs(v); }    

    /// @precond @->it is a valid iterator within v
    static void         Erase(
                            std::vector< TYPE* > & 
                                                v,
                            std::vector< TYPE* >::iterator
                                                it ) 
                        { delete *it; v.erase(it); }    

    /// @precond @->v.size() > 0
    static void         PopBack(
                            std::vector< TYPE* > & 
                                                v ) 
                        { delete v.back(); v.pop_back(); }    
};

/** One helper class for the ->csv::VirtualVector.
    Implements a 
*/
template <class TYPE>
struct keep_ptrs
{
    static void         Destruct(std::vector< TYPE* > & v) 
                        {}
                            
    static void         Erase(
                            std::vector< TYPE* > & 
                                                v,
                            std::vector< TYPE* >::iterator
                                                it2erase ) 
                        { v.erase(it); }    
                        
    static void         PopBack(
                            std::vector< TYPE* > & 
                                                v ) 
                        { v.pop_back(); }    
};

    
}   // namespace vvector


                                            
/** Implements a vector of different implementations of a base
    class. 
    
    Implementation has to be by pointers to get the polymorphic
    behaviour, however access is by references to the base class.
    
    The class owns its members
*/
template <class XX, class PTRDEL = vvector::delete_ptrs<XX> >
class VirtualVector
{
  public:
	typedef VirtualVector<XX,PTRDEL>    self;
	typedef const XX &                  const_reference;
	typedef XX &                        reference;
	typedef typename std::vector< DYN XX* >::size_type
	                                    size_type;
	               
	class const_iterator
	{
	  public:              
	                        const_iterator(
	                            std::vector< XX* >::const_iterator
	                                                i_implIter );
        const_iterator &    operator++();
        const_reference     operator*() const;         
        
        bool                operator==(
                                const_iterator      i_other ) const
                                { return aImplIterator == i_other.aImplIterator; }
        bool                operator!=(
                                const_iterator      i_other ) const
                                { return aImplIterator != i_other.aImplIterator; }

        const_iterator &    operator+=(
                                intt                i_diff ) 
                                { aImplIterator += i_diff; }
        const_iterator &    operator-=(
                                intt                i_diff ) 
                                { aImplIterator -= i_diff; }
	         
        const_iterator      operator+(
                                intt                i_diff ) const
                                { const_iterator ret(aImplIterator);
                                  return ret += i_diff; }
        const_iterator      operator-(
                                intt                i_diff ) const
                                { const_iterator ret(aImplIterator);
                                  return ret -= i_diff; }
	  private:     
	    friend class VirtualVector<XX,PTRDEL>;
	    typename std::vector< DYN XX* >::const_iterator
	                        aImplIterator;
	};
	
	class iterator
	{
	  public:
	                        iterator(
	                            std::vector< XX* >::iterator
	                                                i_implIter );
        iterator &          operator++();	         
        reference           operator*() const;
	    
        bool                operator==(
                                iterator            i_other ) const 
                                { return aImplIterator == i_other.aImplIterator; }
        bool                operator!=(
                                iterator            i_other ) const 
                                { return aImplIterator != i_other.aImplIterator; }

        iterator &          operator+=(
                                intt                i_diff ) 
                                { aImplIterator += i_diff; }
        iterator &          operator-=(
                                intt                i_diff ) 
                                { aImplIterator -= i_diff; }

        iterator            operator+(
                                intt                i_diff ) const
                                { iterator ret(aImplIterator);
                                  return ret += i_diff; }
        iterator            operator-(
                                intt                i_diff ) const
                                { iterator ret(aImplIterator);
                                  return ret -= i_diff; }
	  private:
	    friend class VirtualVector<XX,PTRDEL>;
	    typename std::vector< DYN XX* >::iterator
	                        aImplIterator;
	};
	
	

						VirtualVector();
    explicit    		VirtualVector(
						    int                 i_size );
						~VirtualVector();
                          
    const_reference     operator[](
                            size_type           i_pos ) const;
    reference           operator[](
                            size_type           i_pos );
                            
    void                push_back(
                            DYN XX &            i_drElement );                            
    void                pop_back(
                            DYN XX &            i_drElement );                            

    void                insert(                             
                            iterator            i_pos,
                            DYN XX &            i_drElement );                            
    void                erase(                             
                            iterator            i_pos );
    void                set(
                            size_type           i_pos,
                            DYN XX &            i_drElement );          
          
    bool                IsEmpty() const;
    size_t              Size() const;
	const_iterator      Begin() const;
	const_iterator      End() const;
    const_reference     Front() const;
    const_reference     Back() const;

	
	iterator            Begin();
	iterator            End();
    reference           Front();
    reference           Back();

  private:
    // Forbidden (until implemented):
                        VirtualVector(const VirtualVector&);    
    VirtualVector &     operator=(const VirtualVector&);    

    // DATA
    std::vector< DYN XX* >  aData;
};



// IMPLEMENTATION

template <class XX, class PTRDEL>
inline 
VirtualVector<XX,PTRDEL>::VirtualVector() 
    :   aData()
{
}

template <class XX, class PTRDEL>
inline
VirtualVector<XX,PTRDEL>::VirtualVector(int i_size)
    :   aData(i_size, 0)
{
}

template <class XX, class PTRDEL>
inline 
VirtualVector<XX,PTRDEL>::~VirtualVector()
{
    PTRDEL::Destruct(aData);            
}
                          
template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::const_reference     
VirtualVector<XX,PTRDEL>::operator[]( size_type i_pos ) const
{
    return *aData[i_pos];    
}

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::reference           
VirtualVector<XX,PTRDEL>::operator[]( size_type i_pos )
{
    return *aData[i_pos];    
}

template <class XX, class PTRDEL>
inline void                
VirtualVector<XX,PTRDEL>::push_back( DYN XX & i_drElement )
{
    aData.push_back(&i_drElement);    
}

template <class XX, class PTRDEL>
inline void                
VirtualVector<XX,PTRDEL>::pop_back( DYN XX & i_drElement )
{          
    if (NOT aData.empty())
        PTRDEL::PopBack(aData);            
}

template <class XX, class PTRDEL>
inline void                
VirtualVector<XX,PTRDEL>::insert( iterator            i_pos,
                           DYN XX &            i_drElement )
{
    aData.insert(i_pos.aImplIterator, &i_drElement);
}   
                           
template <class XX, class PTRDEL>
inline void                
VirtualVector<XX,PTRDEL>::erase( iterator i_pos )
{
    PTRDEL::Erase(aData, i_pos.aImplIterator);
}

template <class XX, class PTRDEL>
inline void                
VirtualVector<XX,PTRDEL>::set( size_type    i_pos,
                               DYN XX &     i_drElement )
{
    aData[i_pos] = &i_drElement;
}
                               
template <class XX, class PTRDEL>
inline bool                
VirtualVector<XX,PTRDEL>::IsEmpty() const
{
    return aData.empty();
}   

template <class XX, class PTRDEL>
inline size_t              
VirtualVector<XX,PTRDEL>::Size() const
{
    return aData.size();
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::const_iterator      
VirtualVector<XX,PTRDEL>::Begin() const
{
    return const_iterator(aData.begin());
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::const_iterator      
VirtualVector<XX,PTRDEL>::End() const
{
    return const_iterator(aData.end());
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::const_reference     
VirtualVector<XX,PTRDEL>::Front() const
{
    return *aData.front();
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::const_reference     
VirtualVector<XX,PTRDEL>::Back() const
{
    return *aData.back();
}  
	
template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::iterator            
VirtualVector<XX,PTRDEL>::Begin()
{
    return iterator(aData.begin());
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::iterator            
VirtualVector<XX,PTRDEL>::End()
{
    return iterator(aData.end());
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::reference           
VirtualVector<XX,PTRDEL>::Front()
{
    return *aData.front();
}  

template <class XX, class PTRDEL>
inline typename VirtualVector<XX,PTRDEL>::reference           
VirtualVector<XX,PTRDEL>::Back()
{
    return *aData.back();
}  


}   // namespace csv


#endif
