/*************************************************************************/
/* module:          Definition of WBXML/XML tags for the en-/decoder     */
/* file:            XLTTags.c                                            */
/* target system:   all                                                  */
/* target OS:       all                                                  */   
/*************************************************************************/

/*
 * Copyright Notice
 * Copyright (c) Ericsson, IBM, Lotus, Matsushita Communication 
 * Industrial Co., LTD,Motorola, Nokia, Palm, Inc., Psion, 
 * Starfish Software (2001).
 * All Rights Reserved.
 * Implementation of all or part of any Specification may require 
 * licenses under third party intellectual property rights, 
 * including without limitation, patent rights (such a third party 
 * may or may not be a Supporter). The Sponsors of the Specification 
 * are not responsible and shall not be held responsible in any 
 * manner for identifying or failing to identify any or all such 
 * third party intellectual property rights.
 * 
 * THIS DOCUMENT AND THE INFORMATION CONTAINED HEREIN ARE PROVIDED 
 * ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND AND ERICSSON, IBM, 
 * LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO. LTD, MOTOROLA, 
 * NOKIA, PALM INC., PSION, STARFISH SOFTWARE AND ALL OTHER SYNCML 
 * SPONSORS DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION 
 * HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF 
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT 
 * SHALL ERICSSON, IBM, LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO., 
 * LTD, MOTOROLA, NOKIA, PALM INC., PSION, STARFISH SOFTWARE OR ANY 
 * OTHER SYNCML SPONSOR BE LIABLE TO ANY PARTY FOR ANY LOSS OF 
 * PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF 
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTAL, 
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH 
 * THIS DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 * 
 * The above notice and this paragraph must be included on all copies 
 * of this document that are made.
 * 
 */

#include "xlttags.h"

#include <libstr.h>
#include <smlerr.h>
#include <libmem.h>
#include <libutil.h>


TagPtr_t getTagTable(void);


/**
 * FUNCTION: getTagTable
 *
 * Returns the tag table - this function is used to avoid a global
 * tag table variable
 *
 * RETURN:             a pointer to the tag table containing tag ids, 
 *                     codepages, wbxml tags and xml tags
 */
TagPtr_t getTagTable(void)
{
  TagPtr_t _tmpTagPtr;
  Tag_t tagtbl[] =
  {
    { TN_ADD,           SML_CP,  0x05,    "Add"           },
    { TN_ALERT,         SML_CP,  0x06,    "Alert"         },
    { TN_ARCHIVE,       SML_CP,  0x07,    "Archive"       },
    { TN_ATOMIC,        SML_CP,  0x08,    "Atomic"        },
    { TN_CHAL,          SML_CP,  0x09,    "Chal"          },
    { TN_CMD,           SML_CP,  0x0A,    "Cmd"           },
    { TN_CMDID,         SML_CP,  0x0B,    "CmdID"         },
    { TN_CMDREF,        SML_CP,  0x0C,    "CmdRef"        },
    { TN_COPY,          SML_CP,  0x0D,    "Copy"          },
    { TN_CRED,          SML_CP,  0x0E,    "Cred"          },
    { TN_DATA,          SML_CP,  0x0F,    "Data"          },
    { TN_DELETE,        SML_CP,  0x10,    "Delete"        },
    { TN_EXEC,          SML_CP,  0x11,    "Exec"          },
    { TN_FINAL,         SML_CP,  0x12,    "Final"         },
    { TN_GET,           SML_CP,  0x13,    "Get"           },
    { TN_ITEM,          SML_CP,  0x14,    "Item"          },
    { TN_LANG,          SML_CP,  0x15,    "Lang"          },
    { TN_LOCNAME,       SML_CP,  0x16,    "LocName"       },
    { TN_LOCURI,        SML_CP,  0x17,    "LocURI"        },
    { TN_MAP,           SML_CP,  0x18,    "Map"           },
    { TN_MAPITEM,       SML_CP,  0x19,    "MapItem"       },
    { TN_META,          SML_CP,  0x1A,    "Meta"          },
    { TN_MSGID,         SML_CP,  0x1B,    "MsgID"         },
    { TN_MSGREF,        SML_CP,  0x1C,    "MsgRef"        },
    { TN_NORESP,        SML_CP,  0x1D,    "NoResp"        },
    { TN_NORESULTS,     SML_CP,  0x1E,    "NoResults"     },
    { TN_PUT,           SML_CP,  0x1F,    "Put"           },
    { TN_REPLACE,       SML_CP,  0x20,    "Replace"       },
    { TN_RESPURI,       SML_CP,  0x21,    "RespURI"       },
    { TN_RESULTS,       SML_CP,  0x22,    "Results"       },
    { TN_SEARCH,        SML_CP,  0x23,    "Search"        },
    { TN_SEQUENCE,      SML_CP,  0x24,    "Sequence"      },
    { TN_SESSIONID,     SML_CP,  0x25,    "SessionID"     },
    { TN_SFTDEL,        SML_CP,  0x26,    "SftDel"        },
    { TN_SOURCE,        SML_CP,  0x27,    "Source"        },
    { TN_SOURCEREF,     SML_CP,  0x28,    "SourceRef"     },
    { TN_STATUS,        SML_CP,  0x29,    "Status"        },
    { TN_SYNC,          SML_CP,  0x2A,    "Sync"          },
    { TN_SYNCBODY,      SML_CP,  0x2B,    "SyncBody"      },
    { TN_SYNCHDR,       SML_CP,  0x2C,    "SyncHdr"       },
    { TN_SYNCML_END,    SML_CP,  0x2D,    "SyncML"        },
    { TN_SYNCML_START,  SML_CP,  0x2D,    "SyncML xmlns=\'SYNCML:SYNCML1.0\'"        },
    { TN_TARGET,        SML_CP,  0x2E,    "Target"        },
    { TN_TARGETREF,     SML_CP,  0x2F,    "TargetRef"     },

    { TN_VERSION,       SML_CP,  0x31,    "VerDTD"        },
    { TN_PROTO,         SML_CP,  0x32,    "VerProto"      },

#ifdef __META_EXT__
    { TN_META_ANCHOR,   META_CP,  0x05,    "Anchor"        },
    { TN_META_EMI,      META_CP,  0x06,    "EMI"           },
    { TN_META_FORMAT,   META_CP,  0x07,    "Format"        },
    { TN_META_FREEID,   META_CP,  0x08,    "FreeID"        },
    { TN_META_FREEMEM,  META_CP,  0x09,    "FreeMem"       },
    { TN_META_LAST,     META_CP,  0x0A,    "Last"          },
    { TN_META_MARK,     META_CP,  0x0B,    "Mark"          },
    { TN_META_MAXMSGSIZE,META_CP, 0x0C,    "MaxMsgSize"    },
    { TN_META_MEM,      META_CP,  0x0D,    "Mem"           },
    { TN_META_METINF,   META_CP,  0x0E,    "MetInf"        },
    { TN_META_NEXT,     META_CP,  0x0F,    "Next"          },
    { TN_META_NEXTNONCE,META_CP,  0x10,    "NextNonce"     },
    { TN_META_SHAREDMEM,META_CP,  0x11,    "SharedMem"     },
    { TN_META_SIZE,     META_CP,  0x12,    "Size"          },
    { TN_META_TYPE,     META_CP,  0x13,    "Type"          },
    { TN_META_VERSION,  META_CP,  0x14,    "Version"       },
#endif
    { TN_UNDEF,       255,  0x00,    NULL            }
  };
  
  _tmpTagPtr = (TagPtr_t)smlLibMalloc(sizeof(tagtbl));
  if (_tmpTagPtr == NULL) return NULL;
  smlLibMemcpy(_tmpTagPtr, &tagtbl, sizeof(tagtbl));
  return _tmpTagPtr;  
}

/**
 * FUNCTION: getTagString
 *
 * Returns a tag string which belongs to a tag ID. 
 * This function is needed for the XML encoding
 *
 * PRE-Condition:   valid tag ID, the tagSring has to be allocated 
 *
 * POST-Condition:  tag string is returned
 *
 * IN:              tagId, the ID for the tag 
 *
 * IN/OUT:          tagString, allocated string into which the XML 
 *                             tag string will be written
 * 
 * RETURN:          0,if OK
 */
#ifdef __SML_XML__ 
Ret_t getTagString(XltTagID_t tagID, String_t tagString)
{
    int i = 0;
    TagPtr_t pTags = getTagTable();
    if (pTags == NULL)
    {
      tagString[0] = '\0';
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pTags+i)->id) != TN_UNDEF)
    {
      if ((((pTags+i)->id) == tagID))
      {
        String_t _tmp = (pTags+i)->xml;
        smlLibStrcpy(tagString, _tmp); 
      	smlLibFree(pTags);
        return SML_ERR_OK;
      }    
      i++;
    }        
    smlLibStrcpy(tagString, NULL); 
    smlLibFree(pTags);
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}
#endif

/**
 * FUNCTION: getTagByte
 *
 * Returns a WBXML byte which belongs to a tag ID in a defined codepage. 
 * This function is needed for the WBXML encoding
 *
 * PRE-Condition:   valid tag ID, valid code page
 *
 * POST-Condition:  tag byte is returned
 *
 * IN:              tagId, the ID for the tag 
 *                  cp, code page group for the tag 
 *                  pTagByte, the byte representation of the tag
 * 
 * RETURN:          0, if OK
 */
Ret_t getTagByte(XltTagID_t tagID, Byte_t cp, Byte_t *pTagByte)
{    
    int i = 0; 
    TagPtr_t pTags = getTagTable();
    if (pTags == NULL)
    {
      smlLibFree(pTags);
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pTags+i)->id) != TN_UNDEF)
    {
      if ((((pTags+i)->id) == tagID) && (((pTags+i)->cp) == cp))
      {
      	*pTagByte = (pTags+i)->wbxml;
      	smlLibFree(pTags);
        return SML_ERR_OK;
      }    
      i++;
    }        
   	smlLibFree(pTags);
    *pTagByte = 0;
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}

/**
 * FUNCTION: getCodePage
 *
 * Returns the code page which belongs to a certain PCDATA extension type. 
 *
 * PRE-Condition:   valid PCDATA extension type
 *
 * POST-Condition:  the code page is returned
 *
 * IN:              ext, the PCDATA extension type
 * 
 * RETURN:          the code page
 */
Byte_t getCodePage(SmlPcdataExtension_t ext)
{
    if (ext == SML_EXT_META)
        return 1;
    return 0;
}

/**
 * FUNCTION: getCodePageById
 *
 * Returns the codepage which belongs to a certain tag ID
 *
 * PRE-Condition:   valid tag ID
 *
 * POST-Condition:  the code page is returned
 *
 * IN:              tagID, the ID of the tag 
 *                  pCp, the codepage of the tag
 *
 * RETURN:          0, if OK
 */
Ret_t getCodePageById(XltTagID_t tagID, Byte_t *pCp)
{
    int i = 0; 
    TagPtr_t pTags = getTagTable();
    if (pTags == NULL)
    {
      smlLibFree(pTags);
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pTags+i)->id) != TN_UNDEF)
    {
      if ((((pTags+i)->id) == tagID))
      {
       	*pCp = (pTags+i)->cp;
      	smlLibFree(pTags);
        return SML_ERR_OK;
      }    
      i++;
    }        
   	smlLibFree(pTags);    
    *pCp = 255;
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}

/**
 * FUNCTION: getTagIDByStringAndCodepage
 *
 * Returns the tag ID which belongs to a tag string in a certain codepage
 *
 * PRE-Condition:   valid tag string, valid code page
 *
 * POST-Condition:  tag id is returned
 *
 * IN:              tag, the string representation of the tag 
 *                  cp, code page group for the tag 
 *                  pTagID, the tag id of the tag
 * 
 * RETURN:          0, if OK
 */

Ret_t getTagIDByStringAndCodepage(String_t tag, Byte_t cp, XltTagID_t *pTagID)
{
    int i = 0; 
    TagPtr_t pTags = getTagTable();
    if (pTags == NULL)
    {
      smlLibFree(pTags);
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pTags+i)->id) != TN_UNDEF)
    {
      if ((smlLibStrcmp(((pTags+i)->xml), tag) == 0) && (((pTags+i)->cp) == cp))
      {
        *pTagID = (pTags+i)->id;
       	smlLibFree(pTags);
        return SML_ERR_OK;
      }    
      i++;
    }        
    smlLibFree(pTags);
    *pTagID = TN_UNDEF;
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}

/**
 * FUNCTION: getTagIDByByteAndCodepage
 *
 * Returns the tag ID which belongs to a tag byte in a certain codepage
 *
 * PRE-Condition:   valid tag byte, valid code page
 *
 * POST-Condition:  tag id is returned
 *
 * IN:              tag, the byte representation of the tag 
 *                  cp, code page group for the tag  
 *                  pTagID, the tag id of the tag
 * 
 * RETURN:          0, if OK
 */
Ret_t getTagIDByByteAndCodepage(Byte_t tag, Byte_t cp, XltTagID_t *pTagID)
{

    int i = 0; 
    TagPtr_t pTags = getTagTable();
    if (pTags == NULL)
    {
      smlLibFree(pTags);
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pTags+i)->id) != TN_UNDEF)
    {
      if ((((pTags+i)->wbxml) == tag) && (((pTags+i)->cp) == cp))
      {
        *pTagID = (pTags+i)->id;
       	smlLibFree(pTags);
        return SML_ERR_OK;
      }    
      i++;
    }        
    smlLibFree(pTags);
    *pTagID = TN_UNDEF;
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}

/**
 * FUNCTION: getTagIDByStringAndNamespace
 *
 * Returns the tag ID which belongs to a tag string in a certain namespace
 *
 * PRE-Condition:   valid tag string, valid namespace
 *
 * POST-Condition:  tag id is returned
 *
 * IN:              tag, the string representation of the tag 
 *                  ns, namespace group for the tag  
 *                  pTagID, the tag id of the tag
 * 
 * RETURN:          0, if OK
 */
#ifdef __SML_XML__ 
Ret_t getTagIDByStringAndNamespace(String_t tag, String_t ns, XltTagID_t *pTagID)
{
    int i = 0; 
    TagPtr_t pTags = getTagTable();
    if (pTags == NULL)
    {
      smlLibFree(pTags);
      return SML_ERR_NOT_ENOUGH_SPACE;
    }
    while (((pTags+i)->id) != TN_UNDEF)
    {
      if ((smlLibStrcmp(((pTags+i)->xml), tag) == 0))
      {
        *pTagID = (pTags+i)->id;
       	smlLibFree(pTags);
        return SML_ERR_OK;
      }    
      i++;
    }        
    smlLibFree(pTags);
    *pTagID = TN_UNDEF;
    return SML_ERR_XLT_INVAL_PROTO_ELEM;
}
#endif
