/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* This file defines all string functions
** Warning: Some string functions doesn't always put and end-null on a String
** (This shouldn't be neaded)
*/

#ifdef __GNUC__
#pragma implementation				// gcc: Class implementation
#endif

#include "mysql_priv.h"
#include <m_ctype.h>
#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif

String empty_string("");

uint nr_of_decimals(const char *str)
{
  if ((str=strchr(str,'.')))
  {
    const char *start= ++str;
    for ( ; isdigit(*str) ; str++) ;
    return (uint) (str-start);
  }
  return 0;
}

double Item_str_func::val()
{
  String *res;
  res=str(&str_value);
  return res ? atof(res->c_ptr()) : 0.0;
}

longlong Item_str_func::val_int()
{
  String *res;
  res=str(&str_value);
  return res ? strtoll(res->c_ptr(),NULL,10) : (longlong) 0;
}


/*
** Concatinate args with the following premissess
** If only one arg which is ok, return value of arg
** Don't reallocate str() if not absolute necessary.
*/

String *Item_func_concat::str(String *str)
{
  String *res,*res2,*use_as_buff;
  uint i;

  null_value=0;
  if (!(res=args[0]->str(str)))
    goto null;
  use_as_buff= &tmp_value;
  for (i=1 ; i < arg_count ; i++)
  {
    if (res->length() == 0)
    {
      if (!(res=args[i]->str(str)))
	goto null;
    }
    else
    {
      if (!(res2=args[i]->str(use_as_buff)))
	goto null;
      if (res2->length() == 0)
	continue;
      if (res->length()+res2->length() > max_allowed_packet)
	goto null;				// Error check
      if (res->alloced_length() >= res->length()+res2->length())
      {						// Use old buffer
	res->append(*res2);
      }
      else if (str->alloced_length() >= res->length()+res2->length())
      {
	str->copy(*res);
	str->append(*res2);
	res=str;
      }
      else if (res == &tmp_value)
      {
	res->append(*res2);			// Must be a blob
      }
      else if (res2 == &tmp_value)
      {						// This can happend only 1 time
	tmp_value.replace(0,0,*res);
	res= &tmp_value;
	use_as_buff=str;			// Put next arg here
      }
      else if (tmp_value.is_alloced() && res2->ptr() >= tmp_value.ptr() &&
	       res2->ptr() <= tmp_value.ptr() + tmp_value.alloced_length())
      {						// This happens real seldom
	tmp_value.copy(*res);			// Move res to start of tmp
	tmp_value.replace(0,0,*res);
	res= &tmp_value;
	use_as_buff=str;			// Put next arg here
      }
      else
      {						// Two big const strings
	if (tmp_value.alloc(max_length))
	  return &empty_string;			/* purecov: inspected */
	tmp_value.copy(*res);
	tmp_value.append(*res2);
	res= &tmp_value;
	use_as_buff=str;
      }
    }
  }
  return res;

null:
  null_value=1;
  return 0;
}


void Item_func_concat::fix_length_and_dec()
{
  max_length=0;
  for (uint i=0 ; i < arg_count ; i++)
    max_length+=args[i]->max_length;
  if (max_length > MAX_BLOB_WIDTH)
  {
    max_length=MAX_BLOB_WIDTH;
    maybe_null=1;
  }
}


String *Item_func_reverse::str(String *str)
{
  String *res = args[0]->str(str);
  char *ptr,*end;

  if ((null_value=args[0]->null_value))
    return 0;
  res=copy_if_not_alloced(*str,*res,res->length());
  ptr = (char *) res->ptr();
  end=ptr+res->length()-1;
  for (; ptr < end ; ptr++,end--)
  {
    char tmp = *ptr;
    *ptr = *end;
    *end=tmp;
  }
  return res;
}


void Item_func_reverse::fix_length_and_dec()
{
  max_length = args[0]->max_length;
}

/*
** Replace all occurences of string2 in string1 with string3.
** Don't reallocate str() if not neaded
*/


String *Item_func_replace::str(String *str)
{
  String *res,*res2,*res3;
  int offset;
  uint from_length,to_length;

  null_value=0;
  res=args[0]->str(str);
  if (args[0]->null_value)
    goto null;
  res2=args[1]->str(&tmp_value);
  if (args[1]->null_value)
    goto null;

  if (res2->length() == 0)
    return res;
  if ((offset=res->strstr(*res2)) < 0)
    return res;
  if (!(res3=args[2]->str(&tmp_value2)))
    goto null;
  from_length= res2->length();
  to_length=   res3->length();

  do
  {
    if (res->length()-from_length + to_length > max_allowed_packet)
      goto null;
    res->replace((uint) offset,from_length,*res3);
    offset+=(int) to_length;
  }
  while ((offset=res->strstr(*res2,(uint) offset)) >0);
  return res;

null:
  null_value=1;
  return 0;
}


void Item_func_replace::fix_length_and_dec()
{
  max_length=args[0]->max_length;
  int diff=(int) (args[2]->max_length - args[1]->max_length);
  if (diff > 0 && args[1]->max_length)
  {						// Calculate of maxreplaces
    max_length= max_length/args[1]->max_length;
    max_length= (max_length+1)*(uint) diff;
  }
  if (max_length > MAX_BLOB_WIDTH)
  {
    max_length=MAX_BLOB_WIDTH;
    maybe_null=1;
  }
}


String *Item_func_insert::str(String *str)
{
  String *res,*res2;
  uint start,length;

  null_value=0;
  res=args[0]->str(str);
  if (args[0]->null_value)
    goto null; /* purecov: inspected */
  res2=args[3]->str(&tmp_value);
  if (args[3]->null_value)
    goto null;
  start=(uint) args[1]->val_int()-1;
  length=(uint) args[2]->val_int();
  if (args[1]->null_value || args[2]->null_value)
    goto null; /* purecov: inspected */
  if (start > res->length()+1)
    return res;					// Wrong param; skipp insert
  if (length > res->length()-start)
    length=res->length()-start;
  if (res->length() - length + res2->length() > max_allowed_packet)
    goto null;					// OOM check
  res->replace(start,length,*res2);
  return res;
null:
  null_value=1;
  return 0;
}


void Item_func_insert::fix_length_and_dec()
{
  max_length=args[0]->max_length+args[3]->max_length;
  if (max_length > MAX_BLOB_WIDTH)
  {
    max_length=MAX_BLOB_WIDTH;
    maybe_null=1;
  }
}


String *Item_func_lcase::str(String *str)
{
  String *res;
  if (!(res=args[0]->str(str)))
  {
    null_value=1; /* purecov: inspected */
    return 0; /* purecov: inspected */
  }
  null_value=0;
  res=copy_if_not_alloced(str_value,*res,res->length());
  res->casedn();
  return res;
}


String *Item_func_ucase::str(String *str)
{
  String *res;
  if (!(res=args[0]->str(str)))
  {
    null_value=1; /* purecov: inspected */
    return 0; /* purecov: inspected */
  }
  null_value=0;
  res=copy_if_not_alloced(str_value,*res,res->length());
  res->caseup();
  return res;
}


String *Item_func_left::str(String *str)
{
  String *res  =args[0]->str(str);
  long length  =(long) args[1]->val_int();

  if ((null_value=args[0]->null_value))
    return 0;
  if (length <= 0)
    return &empty_string;
  if (res->length() > (ulong) length)
    res->length((uint) length);			// Safe even if const arg
  return res;
}


void Item_str_func::left_right_max_length()
{
  max_length=args[0]->max_length;
  if (args[1]->const_item())
  {
    int length=(int) args[1]->val_int();
    if (length <= 0)
      max_length=0;
    else
      set_if_smaller(max_length,(uint) length);
  }
}


void Item_func_left::fix_length_and_dec()
{
  left_right_max_length();
}


String *Item_func_right::str(String *str)
{
  String *res  =args[0]->str(str);
  long length  =(long) args[1]->val_int();

  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */
  if (length <= 0)
    return &empty_string; /* purecov: inspected */
  if (res->length() <= (uint) length)
    return res; /* purecov: inspected */
  tmp_value.set(*res,(res->length()- (uint) length),(uint) length);
  return &tmp_value;
}


void Item_func_right::fix_length_and_dec()
{
  left_right_max_length();
}


String *Item_func_substr::str(String *str)
{
  String *res  =args[0]->str(str);
  long start   =(long) args[1]->val_int();
  long length  =(long) args[2]->val_int();
  long tmp_length;

  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */
  if (start <= 0 || (uint) start > res->length() || length <= 0)
    return &empty_string;

  start--;
  tmp_length=(long) res->length()-start;
  length=min(length,tmp_length);

  if (!start && res->length() == (uint) length)
    return res;
  tmp_value.set(*res,(uint) start,(uint) length);
  return &tmp_value;
}


void Item_func_substr::fix_length_and_dec()
{
  max_length=args[0]->max_length;

  if (args[1]->const_item())
  {
    long start=(long) args[1]->val_int()-1;
    if (start < 0 || start >= (long) max_length)
      max_length=0; /* purecov: inspected */
    else
      max_length-= (uint) start;
  }
  if (args[2]->const_item())
  {
    long length= (long) args[2]->val_int();
    if (length <= 0)
      max_length=0; /* purecov: inspected */
    else
      set_if_smaller(max_length,(uint) length);
  }
}


String *Item_func_substr_index::str(String *str)
{
  String *res =args[0]->str(str);
  String *delimeter =args[1]->str(&tmp_value);
  long count = (long) args[2]->val_int();
  uint length,offset;

  if (args[0]->null_value || args[1]->null_value || args[2]->null_value)
  {					// string and/or delim are null
    null_value=1;
    return 0;
  }
  null_value=0;
  uint delimeter_length=delimeter->length();
  if (!(length = res->length()) || !delimeter_length || !count)
    return &empty_string;		// Wrong parameters

  if (count > 0)
  {					// start counting from the beginning
    for (offset=0 ;; offset+=delimeter_length)
    {
      if ((int) (offset=res->strstr(*delimeter,offset)) < 0)
	return res;			// Didn't find, return org string
      if (!--count)
      {
	tmp_value.set(*res,0,offset);
	break;
      }
    }
  }
  else
  {					// Start counting at end
    for (offset=res->length() ; ; offset-=delimeter_length)
    {
      if ((int) (offset=res->strrstr(*delimeter,offset)) < 0)
	return res;			// Didn't find, return org string
      if (!++count)
      {
	offset+=delimeter_length;
	tmp_value.set(*res,offset,res->length()- offset);
	break;
      }
    }
  }
  return (&tmp_value);
}

/*
** The trim functions are extension to ANSI SQL because they trim substrings
** They ltrim() and rtrim() functions are optimized for 1 byte strings
** They also return the original string if possible, else they return
** a substring that points at the original string.
*/


String *Item_func_ltrim::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0;					/* purecov: inspected */
  char buff[MAX_FIELD_WIDTH];
  String tmp(buff,sizeof(buff));
  String *remove=args[1]->str(&tmp);
  uint remove_length;
  LINT_INIT(remove_length);

  if (!remove || (remove_length=remove->length()) == 0 ||
      remove_length > res->length())
    return res;

  char *ptr=(char*) res->ptr();
  char *end=ptr+res->length();
  if (remove_length == 1)
  {
    char chr=(*remove)[0];
    while (ptr != end && *ptr == chr)
      ptr++;
  }
  else
  {
    const char *r_ptr=remove->ptr();
    end-=remove_length;
    while (ptr < end && !memcmp(ptr,r_ptr,remove_length))
      ptr+=remove_length;
  }
  if (ptr == res->ptr())
    return res;
  tmp_value.set(*res,(uint) (ptr - res->ptr()),(uint) (end-ptr));
  return &tmp_value;
}


String *Item_func_rtrim::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */
  char buff[MAX_FIELD_WIDTH];
  String tmp(buff,sizeof(buff));
  String *remove=args[1]->str(&tmp);
  uint remove_length;
  LINT_INIT(remove_length);

  if (!remove || (remove_length=remove->length()) == 0 ||
      remove_length > res->length())
    return res;

  char *ptr=(char*) res->ptr();
  char *end=ptr+res->length();
  if (remove_length == 1)
  {
    char chr=(*remove)[0];
    while (ptr != end  && end[-1] == chr)
      end--;
  }
  else
  {
    const char *r_ptr=remove->ptr();
    while (ptr + remove_length < end &&
	   !memcmp(end-remove_length,r_ptr,remove_length))
      end-=remove_length;
  }
  if (end == res->ptr()+res->length())
    return res;
  tmp_value.set(*res,0,(uint) (end-ptr));
  return &tmp_value;
}


String *Item_func_trim::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0;					/* purecov: inspected */
  char buff[MAX_FIELD_WIDTH];
  String tmp(buff,sizeof(buff));
  String *remove=args[1]->str(&tmp);
  uint remove_length;
  LINT_INIT(remove_length);

  if (!remove || (remove_length=remove->length()) == 0 ||
      remove_length > res->length())
    return res;

  char *ptr=(char*) res->ptr();
  char *end=ptr+res->length();
  const char *r_ptr=remove->ptr();
  while (ptr+remove_length < end && !memcmp(ptr,r_ptr,remove_length))
    ptr+=remove_length;
  while (ptr + remove_length < end &&
	 !memcmp(end-remove_length,r_ptr,remove_length))
    end-=remove_length;

  if (ptr == res->ptr() && end == ptr+res->length())
    return res;
  tmp_value.set(*res,(uint) (ptr - res->ptr()),(uint) (end-ptr));
  return &tmp_value;
}



String *Item_func_password::str(String *str)
{
  String *res  =args[0]->str(str);
  if ((null_value=args[0]->null_value))
    return 0;
  if (res->length() == 0)
    return &empty_string;
  make_scrambled_password(tmp_value,res->c_ptr());
  str->set(tmp_value,16);
  return str;
}

#define bin_to_ascii(c) ((c)>=38?((c)-38+'a'):(c)>=12?((c)-12+'A'):(c)+'.')

String *Item_func_encrypt::str(String *str)
{
  String *res  =args[0]->str(str);

#ifdef HAVE_CRYPT
  char salt[2];
 if ((null_value=args[0]->null_value))
    return 0;
  if (res->length() == 0)
    return &empty_string;

  if (arg_count == 1)
  {					// generate random salt
    time_t timestamp=current_thd->query_start();
    salt[0] = bin_to_ascii(timestamp & 0x3f);
    salt[1] = bin_to_ascii((timestamp >> 5) & 0x3f);
  }
  else
  {					// obtain salt from the first two bytes
    String *salt_str=args[1]->str(&tmp_value);
    if ((null_value= (args[1]->null_value || salt_str->length() < 2)))
      return 0;
    salt[0]=(*salt_str)[0];
    salt[1]=(*salt_str)[1];
  }
  str->set(crypt(res->c_ptr(),salt),13);
  return str;
#else
  null_value=1;
  return 0;
#endif	/* HAVE_CRYPT */
}


String *Item_func_database::str(String *str)
{
  if (!current_thd->db)
    str->length(0);
  else
    str->set((const char*) current_thd->db,strlen(current_thd->db));
  return str;
}

String *Item_func_user::str(String *str)
{
  str->set((const char*) current_thd->user,strlen(current_thd->user));
  return str;
}

void Item_func_soundex::fix_length_and_dec()
{
  max_length=args[0]->max_length;
  set_if_bigger(max_length,4);
}


  /*
    If alpha, map input letter to soundex code.
    If not alpha and remove_garbage is set then skipp to next char
    else return 0
    */

extern "C" {
extern char *soundex_map;		// In mysys/static.c
}

static char get_scode(char *ptr)
{
  uchar ch=toupper(*ptr);
  if (ch < 'A' || ch > 'Z')
  {
					// Thread extended alfa (country spec)
    return '0';				// as vokal
  }
  return(soundex_map[ch-'A']);
}


String *Item_func_soundex::str(String *str)
{
  String *res  =args[0]->str(str);
  char last_ch,ch;
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */

  if (str_value.alloc(max(res->length(),4)))
    return str; /* purecov: inspected */
  char *to= (char *) str_value.ptr();
  char *from= (char *) res->ptr(), *end=from+res->length();

  while (from != end && isspace(*from)) // Skipp pre-space
    from++; /* purecov: inspected */
  if (from == end)
    return &empty_string;		// No alpha characters.
  *to++ = toupper(*from);		// Copy first letter
  last_ch = get_scode(from);		// code of the first letter
					// for the first 'double-letter check.
					// Loop on input letters until
					// end of input (null) or output
					// letter code count = 3
  for (from++ ; from < end ; from++)
  {
    if (!isalpha(*from))
      continue;
    ch=get_scode(from);
    if ((ch != '0') && (ch != last_ch)) // if not skipped or double
    {
       *to++ = ch;			// letter, copy to output
       last_ch = ch;			// save code of last input letter
    }					// for next double-letter check
  }
  for (end=(char*) str_value.ptr()+4 ; to < end ; to++)
    *to = '0';
  *to=0;				// end string
  str_value.length((uint) (to-str_value.ptr()));
  return &str_value;
}


/*
** Change a number to format '3,333,333,333.000'
** This should be 'internationalized' sometimes.
*/

Item_func_format::Item_func_format(Item *org,int dec) :Item_str_func(org)
{
  decimals=(uint) set_zone(dec,0,30);
}


String *Item_func_format::str(String *str)
{
  double nr	=args[0]->val();
  uint diff,length,dec;
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */
  dec= decimals ? decimals+1 : 0;
  str->set(nr,decimals);
  length=str->length()+(diff=(str->length()- dec-1)/3);
  if (diff)
  {
    char *tmp,*pos;
    str=copy_if_not_alloced(tmp_str,*str,length);
    str->length(length);
    tmp=(char*) str->ptr()+length - dec-1;
    for (pos=(char*) str->ptr()+length ; pos != tmp; pos--)
      pos[0]=pos[- (int) diff];
    while (diff)
    {
      pos[0]=pos[-(int) diff]; pos--;
      pos[0]=pos[-(int) diff]; pos--;
      pos[0]=pos[-(int) diff]; pos--;
      pos[0]=',';
      pos--;
      diff--;
    }
  }
  return str;
}


void Item_func_elt::fix_length_and_dec()
{
  max_length=0;
  decimals=0;
  for (uint i=1 ; i < arg_count ; i++)
  {
    set_if_bigger(max_length,args[i]->max_length);
    set_if_bigger(decimals,args[i]->decimals);
  }
  maybe_null=1;					// NULL if wrong first arg
  used_tables_cache|=item->used_tables();
}


void Item_func_elt::update_used_tables()
{
  Item_func::update_used_tables();
  item->update_used_tables();
  used_tables_cache|=item->used_tables();
}


double Item_func_elt::val()
{
  uint tmp;
  if ((tmp=(uint) item->val_int()) == 0 || tmp > arg_count)
  {
    null_value=1;
    return 0.0;
  }
  null_value=0;
  return args[tmp-1]->val();
}

longlong Item_func_elt::val_int()
{
  uint tmp;
  if ((tmp=(uint) item->val_int()) == 0 || tmp > arg_count)
  {
    null_value=1;
    return 0;
  }
  null_value=0;
  return args[tmp-1]->val_int();
}

String *Item_func_elt::str(String *str)
{
  uint tmp;
  if ((tmp=(uint) item->val_int()) == 0 || tmp > arg_count)
  {
    null_value=1;
    return NULL;
  }
  null_value=0;
  return args[tmp-1]->str(str);
}


String *Item_func_char::str(String *str)
{
  str->length(0);
  for (uint i=0 ; i < arg_count ; i++)
  {
    longlong num= args[i]->val_int();
    if (!args[i]->null_value)
      str->append((char) num);
  }
  str->realloc(str->length());			// Add end 0 (for Purify)
  return str;
}

String *Item_date::str(String *str)
{
  ulong value=(ulong) val_int();
  if (null_value)
    return (String*) 0;
  if (!value)
  {
    return &empty_string;			// zero daynr
  }
  if (str->alloc(32))
    return &empty_string;			/* purecov: inspected */
  sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	  (int) (value/10000L),(int) (value/100)%100,(int) (value%100));
  str->length(strlen(str->ptr()));
  return str;
}


longlong Item_func_from_days::val_int()
{
  longlong value=args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */

  uint year,month,day;
  get_date_from_daynr((long) value,&year,&month,&day);
  return (longlong) (year*10000L+month*100+day);
}


void Item_func_date_add_mm::fix_length_and_dec()
{
  /* Try to guess if arg[0] is a timestamp or a date */
  max_length= (args[0]->max_length <= 10) ? 10 : 19;
}


String *Item_func_date_add_mm::str(String *str)
{
  String *res= args[0]->str(str);
  long count= (long) args[1]->val_int();
  timestamp_type type;
  ulong period;
  TIME l_time;

  /* If some of the arguments was NULL, return NULL */
  if ((null_value=(args[0]->null_value || args[1]->null_value)))
    return 0;					// Return NULL pointer

  /* If not a TIMESTAMP or DATE, return a empty string */
  if ((type=str_to_TIME(res->ptr(),res->length(),&l_time)) == TIMESTAMP_NONE)
    return &empty_string;

  /* As res->ptr() isn't needed anymore, we can return result in str */

  str->alloc(19);				// Safety check
  period=l_time.year*12+l_time.month-1+count;
  if ((long) period < 0)
    bzero(&l_time,sizeof(l_time));
  else if (period >= 120000L)
  {
    l_time.year= 9999;
    l_time.month= 12;
    l_time.day= 31;
    l_time.hour=23;
    l_time.minute=l_time.sec=59;
  }
  else
  {
    l_time.year= (uint) (period/12);
    l_time.month= (uint) (period % 12L)+1;
  }
  /* Return result in DATE or TIMESTAMP format, according to the argument */
  if (type == TIMESTAMP_DATE)
    sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	    l_time.year,l_time.month,l_time.day);
  else
    sprintf((char*) str->ptr(),"%04d-%02d-%02d %02d:%02d:%02d",
	    l_time.year,l_time.month,l_time.day,
	    l_time.hour,l_time.minute,l_time.sec);
  str->length(strlen(str->ptr()));
  return str;
}


void Item_func_curdate::fix_length_and_dec()
{
  struct tm tm_tmp,*start_time;
  time_t query_start=current_thd->query_start();
  decimals=0; max_length=10;
  localtime_r(&query_start,&tm_tmp);
  start_time=&tm_tmp;
  value=(longlong) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		    ((uint) start_time->tm_mon+1)*100+
		    (uint) start_time->tm_mday);
}


longlong Item_func_curdate::val_int()
{
  return value;
}

void Item_func_curtime::fix_length_and_dec()
{
  struct tm tm_tmp,*start_time;
  time_t query_start=current_thd->query_start();
  decimals=0; max_length=8;
  localtime_r(&query_start,&tm_tmp);
  start_time=&tm_tmp;
  value=(longlong) ((ulong) ((uint) start_time->tm_hour)*10000L+
		    (ulong) (((uint) start_time->tm_min)*100L+
			     (uint) start_time->tm_sec));
  sprintf(buff,"%02d:%02d:%02d",
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str_value.set((const char*) buff,strlen(buff));
}


void Item_func_now::fix_length_and_dec()
{
  struct tm tm_tmp,*start_time;
  time_t query_start=current_thd->query_start();
  decimals=0; max_length=19;
  localtime_r(&query_start,&tm_tmp);
  start_time=&tm_tmp;
  value=((longlong) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		     (((uint) start_time->tm_mon+1)*100+
		      (uint) start_time->tm_mday))*(longlong) 1000000L+
	 (longlong) ((ulong) ((uint) start_time->tm_hour)*10000L+
		     (ulong) (((uint) start_time->tm_min)*100L+
			    (uint) start_time->tm_sec)));
  sprintf(buff,"%04d-%02d-%02d %02d:%02d:%02d",
	  ((int) (start_time->tm_year+1900)) % 10000,
	  (int) start_time->tm_mon+1,
	  (int) start_time->tm_mday,
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str_value.set((const char*) buff,strlen(buff));
}


longlong Item_func_now::val_int()
{
  return value;
}


String *Item_func_now::str(String *str)
{
  return &str_value;
}


String *Item_func_sec_to_time::str(String *str)
{
  char buff[23];
  ulonglong time=(ulonglong) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return (String*) 0;
  uint sec= (uint) (time % 3600);
  sprintf(buff,"%02lu:%02u:%02u",(ulong) (time/3600),sec/60, sec % 60);
  str->copy(buff,strlen(buff));
  return str;
}


longlong Item_func_sec_to_time::val_int()
{
  ulonglong time=(ulonglong) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0;
  return (time / 3600)*10000+((time/60) % 60)*100+ (time % 60);
}


/*
** Todo: Move month and days to language files
*/

static char *month_names[] = { "January", "February", "March", "April",
			       "May", "June", "July", "August",
			       "September", "October", "November", "December" };
static char *day_names[] = { "Monday", "Tuesday", "Wednesday",
			     "Thursday", "Friday", "Saturday" ,"Sunday" };

void Item_func_date_format::fix_length_and_dec()
{
  decimals=0;
  if (args[1]->type() == STRING_ITEM)
  {						// Optimize the normal case
    fixed_length=1;
    max_length=format_length(((Item_string*) args[1])->const_string());
  }
  else
  {
    fixed_length=0;
    max_length=args[1]->max_length*10;
    set_if_smaller(max_length,MAX_BLOB_WIDTH);
  }
}


uint Item_func_date_format::format_length(const String *format)
{
  uint size=0;
  const char *ptr=format->ptr();
  const char *end=ptr+format->length();

  for (; ptr != end ; ptr++)
  {
    switch(*ptr) {
    case 'M': /* month, textual */
    case 'W': /* day (of the week), textual */
      size += 9;
      break;
    case 'D': /* day (of the month), numeric plus english suffix */
    case 'Y': /* year, numeric, 4 digits */
      size += 4;
      break;
    case 'y': /* year, numeric, 2 digits */
    case 'm': /* month, numeric */
    case 'd': /* day (of the month), numeric */
    case 'h': /* hour, numeric */
    case 'i': /* minutes, numeric */
    case 's': /* seconds, numeric */
      size += 2;
      break;
    case 'w': /* day (of the week), numeric */
    default:
      size++;
      break;
    }
  }
  return size;
}


String *Item_func_date_format::str(String *str)
{
  String *res,*format;
  TIME l_time;
  char intbuff[15];
  uint size,weekday;

  if (!(res=args[0]->str(str)))
  {
    null_value=1;
    return 0;
  }
  null_value=0;

  if (str_to_TIME(res->ptr(),res->length(),&l_time) == TIMESTAMP_NONE)
    return &empty_string;

  if (!(format = args[1]->str(str)) || !format->length())
  {
    null_value=1;
    return 0;
  }

  if (fixed_length)
    size=max_length;
  else
    size=format_length(format);
  if (format == str)
    str=&str_value;				// Save result here
  if (str->alloc(size))
    return &empty_string;
  str->length(0);

  /* Create the result string */
  const char *ptr=format->ptr();
  const char *end=ptr+format->length();
  for ( ; ptr != end ; ptr++)
  {
    switch(*ptr) {
    case 'M':
      str->append(month_names[l_time.month-1]);
      break;
    case 'W':
      weekday=calc_weekday(calc_daynr(l_time.year,l_time.month,l_time.day));
      str->append(day_names[weekday]);
      break;
    case 'D':
      sprintf(intbuff,"%d",l_time.day);
      str->append(intbuff);
      if (l_time.day >= 10 &&  l_time.day <= 19)
	str->append("th");
      else
      {
	switch (l_time.day %10)
	{
	case 1:
	  str->append("st");
	  break;
	case 2:
	  str->append("nd");
	  break;
	case 3:
	  str->append("rd");
	  break;
	default:
	  str->append("th");
	  break;
	}
      }
      break;
    case 'Y':
      sprintf(intbuff,"%04d",l_time.year);
      str->append(intbuff);
      break;
    case 'y':
      sprintf(intbuff,"%02d",l_time.year%100);
      str->append(intbuff);
      break;
    case 'm':
      sprintf(intbuff,"%02d",l_time.month);
      str->append(intbuff);
      break;
    case 'd':
      sprintf(intbuff,"%02d",l_time.day);
      str->append(intbuff);
      break;
    case 'h':
      sprintf(intbuff,"%02d",l_time.hour);
      str->append(intbuff);
      break;
    case 'i':					/* minutes */
      sprintf(intbuff,"%02d",l_time.minute);
      str->append(intbuff);
      break;
    case 's':
      sprintf(intbuff,"%02d",l_time.sec);
      str->append(intbuff);
      break;
    case 'w':
      weekday=calc_weekday(calc_daynr(l_time.year,l_time.month,l_time.day));
      sprintf(intbuff,"%01d",weekday);
      str->append(intbuff);
      break;
    default:
      str->append(*ptr);
      break;
    }
  }
  return str;
}


String *Item_func_from_unixtime::str(String *str)
{
  struct tm tm_tmp,*start_time;
  time_t tmp=(time_t) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0;
  localtime_r(&tmp,&tm_tmp);
  start_time=&tm_tmp;
  if (str->alloc(32))				// Some extra space for safety
    return str;					/* purecov: inspected */
  sprintf((char*) str->ptr(),"%04d-%02d-%02d %02d:%02d:%02d",
	  (int) ((start_time->tm_year+1900) % 10000),
	  (int) start_time->tm_mon+1,
	  (int) start_time->tm_mday,
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str->length(strlen(str->ptr()));		// Can't trust sprintf
  return str;
}


longlong Item_func_from_unixtime::val_int()
{
  time_t tmp=(time_t) (ulong) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0;
  struct tm tm_tmp,*start_time;
  localtime_r(&tmp,&tm_tmp);
  start_time= &tm_tmp;
  return ((longlong) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		      (((uint) start_time->tm_mon+1)*100+
		       (uint) start_time->tm_mday))*LL(1000000)+
	  (longlong) ((ulong) ((uint) start_time->tm_hour)*10000L+
		      (ulong) (((uint) start_time->tm_min)*100L+
			       (uint) start_time->tm_sec)));
}


void Item_func_repeat::fix_length_and_dec()
{
  if (args[1]->const_item())
  {
    max_length=(long) (args[0]->max_length * args[1]->val_int());
    if (max_length >= MAX_BLOB_WIDTH)
    {
      max_length=MAX_BLOB_WIDTH;
      maybe_null=1;
    }
  }
  else
  {
    max_length=MAX_BLOB_WIDTH;
    maybe_null=1;
  }
}

/*
** Item_func_repeat::str is carefully written to avoid reallocs
** as much as possibly at the cost of a local buffer
*/

String *Item_func_repeat::str(String *str)
{
  uint length,tot_length;
  char *to;
  long count= (long) args[1]->val_int();
  String *res =args[0]->str(str);

  if (args[0]->null_value || args[1]->null_value)
    goto err;				// string and/or delim are null
  null_value=0;
  if (count <= 0)			// For nicer SQL code
    return &empty_string;
  if (count == 1)			// To avoid reallocs
    return res;
  length=res->length();
  if (length > max_allowed_packet/count)	// Safe length check
    goto err;				// Probably an error
  tot_length= length*(uint) count;
  if (res->alloced_length() < tot_length)
  {					// Use local buffer
    if (str->alloced_length() >= tot_length)
    {
      str->copy(*res);
      res=str;
    }
    else
    {					// We have to alloc a local buffer
      if (tmp_value.alloc(tot_length))
	goto err;
      tmp_value.copy(*res);
      res=&tmp_value;
    }
  }
  else if (res->realloc(tot_length))
    goto err;
  res->length(tot_length);

  to=(char*) res->ptr()+length;
  while (--count)
  {
    memcpy(to,res->ptr(),length);
    to+=length;
  }
  return (res);

err:
  null_value=1;
  return 0;

}


void Item_date_add_interval::fix_length_and_dec()
{
  /* Try to guess if arg[0] is a timestamp or a date */
  max_length= (args[0]->max_length <= 10) ? 10 : 19;
}


String *Item_date_add_interval::str(String *str)
{
  /* Here arg[1] is a Item_interval object */
  String *res= args[0]->str(str);
  timestamp_type type;
  ulong period;
  INTERVAL interval;
  TIME ltime;
  Item_interval *interval_item=(Item_interval*) args[1];

  if (interval_item->val_time(&interval))
  {
    null_value=1;
    return 0;
  }
  /* If not a TIMESTAMP or DATE, return a empty string */
  if ((type=str_to_TIME(res->ptr(),res->length(),&ltime)) == TIMESTAMP_NONE)
    return &empty_string;

  str->alloc(32);				// Some extra space for safety
  switch (interval_item->int_type) {
  case YEAR:
    ltime.year += interval.year;
    if ((long) ltime.year < 0)
      bzero(&ltime,sizeof(ltime));
    else if (ltime.year >= 1000L)
    {
      ltime.year= 9999;
      ltime.month= 12;
      ltime.day= 31;
      ltime.hour=23;
      ltime.minute=ltime.sec=59;
    }
    break;
  case YEAR_MONTH:
  case MONTH:
    period=ltime.year*12+interval.year+ltime.month-1+interval.month;
    if ((long) period < 0)
      bzero(&ltime,sizeof(ltime));
    else if (period >= 120000L)
    {
      ltime.year= 9999;
      ltime.month= 12;
      ltime.day= 31;
      ltime.hour=23;
      ltime.minute=ltime.sec=59;
    }
    else
    {
      ltime.year= (uint) (period/12);
      ltime.month= (uint) (period % 12L)+1;
    }
    break;
  default:
    return &empty_string;				// Don't crash yet.
  }
  /* Return result in DATE or TIMESTAMP format, according to the argument */
  if (type == TIMESTAMP_DATE)
    sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	    ltime.year,ltime.month,ltime.day);
  else
    sprintf((char*) str->ptr(),"%04d-%02d-%02d %02d:%02d:%02d",
	    ltime.year,ltime.month,ltime.day,
	    ltime.hour,ltime.minute,ltime.sec);
  str->length(strlen(str->ptr()));
  return str;
}
