/* This file is part of
* ======================================================
* 
*           LyX, the High Level Word Processor
* 	 
*	    Copyright (C) 1995 Matthias Ettrich
*
*======================================================*/

#include "config.h"

#include <ctype.h>

#include "inset.h"
#include "layout.h"
#include "lyxparagraph.h"
#include "lyxtext.h"
#include "textutils.h"
#include "autocor.h"

static const int LYX_PAPER_MARGIN = 20;

extern int mono_video;
extern int fast_selection;

extern Widget toplevel;
extern void WriteAlert(const char* s1, const char* s2, const char* s3);


int LyXText::SingleWidth(LyXParagraph *par, int pos)
{
    int fsn;
   /* need inset support later */
   
   char c;
   LyXFont font;
   
   c = par->GetChar(pos);
   
   if (IsNewlineChar(c))
     return 0; 
   
   if (IsHfillChar(c))
     return 3;			       /* because of the representation
					* as vertical lines */
   
   font = GetFont(par, pos);
   
   if (c == LYX_META_FOOTNOTE ||
       c == LYX_META_MARGIN ||
       c == LYX_META_FIG ||
       c == LYX_META_TAB) {
	  char* fs;
	  switch (c) {
	   case LYX_META_MARGIN:
	     fs = "margin\0";
	     break;
	   case LYX_META_FIG:
	     fs = "fig\0";
	     break;
	   case LYX_META_TAB:
	     fs = "tab\0";
	     break;
	   default:		       /* LYX_META_FOOTNOTE */
	     fs = "foot\0";
	     break;
	  }
	  for (fsn=0; fs[fsn]; fsn++); /* get length of string  */
	  if (font.size > LYX_SIZE_TINY) // we is this done twice? (Lgb)
	    font.size--;
	  if (font.size > LYX_SIZE_TINY)
	    font.size--;
	  return LyXTextWidth(font, fs, fsn);
       }
   
   if (c == LYX_META_INSET) {
      Inset *tmpinset=par->GetInset(pos);
      if (tmpinset)
      	return par->GetInset(pos)->Width(font);
      else
      	return 0;
   }
   
   if (IsSeparatorChar(c))
     c = ' ';
   
   return LyXWidth(font, c);
}


inline
static void DrawLine(Pixmap pm, int baseline, int x, int length)
{
   XDrawLine(fl_display,
	     pm,
	     LyXGetFillGC(),
	     x, baseline,
	     x + length, baseline);
   XFlush(fl_display);
      
}


inline
static void DrawOnOffLine(Pixmap pm, int baseline, int x, int length)
{
   XDrawLine(fl_display,
	     pm,
	     LyXGetOnOffLineGC(),
	     x, baseline,
	     x + length, baseline);
   XFlush(fl_display);
}


inline
static void DrawThickLine(Pixmap pm, int baseline, int x, int length)
{
   XDrawLine(fl_display,
	     pm,
	     LyXGetThickLineGC(),
	     x, baseline,
	     x + length, baseline);
   XFlush(fl_display);
}


/* table stuff -- begin*/ 
inline
static void DrawTableLine(Pixmap pm, int baseline, int x,
			  int length, char on_off)
{
  GC gc;
  if (on_off)
    gc = LyXGetThinOnOffLineGC();
  else
    gc = LyXGetCopyGC();
  XDrawLine(fl_display,
	    pm,
	    gc,
	    x, baseline,
	    x + length, baseline);
  XFlush(fl_display);
}


inline
static void DrawVerticalTableLine(Pixmap pm, int x, int y1,
				  int y2, char on_off)
{
  GC gc;
  if (on_off)
    gc = LyXGetThinOnOffLineGC();
  else
    gc = LyXGetCopyGC();
  XDrawLine(fl_display,
	    pm,
	    gc,
	    x, y1,
	    x, y2);
  XFlush(fl_display);
}
/* table stuff -- end*/ 


inline
static void DrawVerticalLine(Pixmap pm, int x, int y1, int y2)
{
   XDrawLine(fl_display,
	     pm,
	     LyXGetFillGC(),
	     x, y1,
	     x, y2);
   XFlush(fl_display);
}


inline
static void DrawVerticalOnOffLine(Pixmap pm, int x, int y1, int y2)
{
   XDrawLine(fl_display,
	     pm,
	     LyXGetFillGC(),
	     x, y1,
	     x, y2);
   XFlush(fl_display);
}


/* returns the paragraph position of the last character in the 
* specified row */
int LyXText::RowLast(Row *row)
{
   if (row->next == NULL)
     return row->par->Last()-1;
   else if (row->next->par != row->par) 
     return row->par->Last()-1;
   else 
     return row->next->pos - 1;
}


void LyXText::Draw(Row *row, int &pos, unsigned long  pm, int offset, float &x)
{
   char c;
   LyXFont font;
   int tmppos;
   int last;
   int fsn;
   float tmpx = x;
   
   c = row->par->GetChar(pos);
   
   if (IsNewlineChar(c)) {
      pos++;
     return;
   } 
   
   tmppos = pos;
   last = RowLast(row);
   font = GetFont(row->par, pos);
   LyXFont font2 = font;

   if (c == LYX_META_FOOTNOTE ||
       c == LYX_META_MARGIN ||
       c == LYX_META_FIG ||
       c == LYX_META_TAB) {
	  char* fs;
	  switch (c) {
	   case LYX_META_MARGIN:
	     fs = "margin\0";
	     /* draw a sign at the left margin! */ 
	     LyXDrawString(font, "!", pm, offset + row->baseline,
			   LYX_PAPER_MARGIN - LyXWidth(font, '!')/2);
	     break;
	   case LYX_META_FIG:
	     fs = "fig\0";
	     break;
	   case LYX_META_TAB:
	     fs = "tab\0";
	     break;
	   default:		       /*  LYX_META_FOOTNOTE:  */
	     fs = "foot\0";
	     break;
	     
	  }
	  for (fsn=0; fs[fsn]; fsn++); /* get length of string  */
	  if (font.size > LYX_SIZE_TINY)
	    font.size--;
	  if (font.size > LYX_SIZE_TINY)
	    font.size--;
	  
	  /* calculate the position of the footnotemark */
	  int y = (row->baseline - LyXMaxAscent(font2) 
		   + LyXAscent(font, 'f'));
	  
	  /* a footnote environment cannot be plain LaTeX, so just
	   * change the font */
	  
	  // font.latex = LYX_NO_LATEX;
	  // a red footnote-mark might look nicer
	  font.latex  = LYX_LATEX;

	  /* draw it! */ 
	  XDrawString(fl_display,
		      pm,
		      LyXSetFontOfGC(LyXGetFootGC(), font),
		      (int) x, offset + y,
		      fs, fsn);
	  XFlush(fl_display);
	  
	  /* calculate the width and set the new x position */ 
	  x += LyXTextWidth(font, fs, fsn);
	  
	  pos++;
	  return;
       }  
   
   if (c == LYX_META_INSET) {
      Inset *tmpinset = row->par->GetInset(pos);
      if (tmpinset) 
	tmpinset->Draw(font, pm, offset + row->baseline, x);
      pos++;
      return;
   }

   /* usual characters, no insets */
   
   /* select character that you can draw in one command (faster!!)
    * This is not done with smallcaps, of cause! */
   while (pos < last  &&
	  (isalpha(row->par->GetChar(pos + 1)) &&
	   isalpha(row->par->GetChar(pos)) &&
	   FontEqual(font2, GetFont(row->par, pos + 1))))
     pos++;

   pos++; 			       /* the next character,
					* NOT printed this time */
   
   /* copy the selected characters into a vertex */ 
   char * textstring = new char[pos - tmppos];
   int i;
   textstring[0] = c;
   for (i=1; i < pos - tmppos; i++)
     textstring[i] = row->par->GetChar(tmppos + i);
   
   /* draw it! */ 
   if (mono_video && font.latex != LYX_NO_LATEX) {
     int a=LyXAscent(font, 'I'), d=LyXDescent(font, 'g');
     XFillRectangle (fl_display, pm, LyXGetCopyGC(), 
		     (int)x, offset + row->baseline-a, 
		     LyXTextWidth(font, textstring, pos - tmppos), a+d);
     XFlush(fl_display);
   }

   LyXDrawText(font, textstring, pos - tmppos, pm, 
	       offset + row->baseline, 
	       (int) x);

   /* calculate the width and set the new x position */ 
   x +=  LyXTextWidth(font, textstring, pos - tmppos);
   
   /* free it */ 
   delete[] textstring;

   
   /* what about underbars? */
   if (font.bar == LYX_UNDER_BAR
       && font.latex == LYX_NO_LATEX){
     XDrawLine(fl_display,
	       pm,
	       LyXGetCopyGC(),
	       (int)tmpx, offset + row->baseline + 2,
	       (int)x, offset + row->baseline + 2);
     XFlush(fl_display);
   }
}


/* Returns the left beginning of the text. 
* This information cannot be taken from the layouts-objekt, because in 
* LaTeX the beginning of the text fits in some cases (for example sections)
* exactly the label-width. */
int LyXText::LeftMargin(Row* row)
{
   int x;
   char*  parindent;
   LyXLayout *layout;
   LyXFont labelfont;
   LyXParagraph *newpar;
   Row dummyrow;
   layout = lyxstyle.Style(parameters->textclass, row->par->GetLayout());
   
   parindent = layout->parindent; 

   /* table stuff -- begin*/ 
   if (row->par->table)
      parindent = "";
   /* table stuff -- end*/       

   x = LYX_PAPER_MARGIN;
   if (row->par->footnoteflag == LYX_OPEN_FOOTNOTE)  {
	   LyXFont font; // = {
	   font.family = LYX_ROMAN_FAMILY;
	   font.series = LYX_MEDIUM_SERIES;
	   font.shape = LYX_UP_SHAPE;
	   font.size = LYX_SIZE_SMALL; 
	   font.latex = LYX_LATEX;
	   font.bar = LYX_NO_BAR; //};
     x += LyXTextWidth(font, " margin ", 8) + LYX_PAPER_MARGIN/2;
   }


   
   /* this is the way, LyX handles the LaTeX-Environments.
    * I have had this idea very late, so it seems to be a
    * later added hack and this is true */ 
   if (!row->par->GetDepth()) {
      if (!row->par->GetLayout()) {
	 /* find the previous same level paragraph */
	 if (row->par->FirstPhysicalPar()->Previous()) {
	    newpar = row->par->DepthHook(row->par->GetDepth());
	    if (newpar && lyxstyle.Style(parameters->textclass, newpar->GetLayout())->nextnoindent)
	       parindent = "";
	 }
      }
   }
   else {
      /* find the next level paragraph */ 
      
      newpar = row->par->DepthHook(row->par->GetDepth()-1);

      /* make a corresponding row. Needed to call LeftMargin() */
      
      /* check wether it is a sufficent paragraph */ 
      if (newpar && newpar->footnoteflag == row->par->footnoteflag
	  && lyxstyle.Style(parameters->textclass, newpar->GetLayout())->latextype != LATEX_PARAGRAPH
	  && lyxstyle.Style(parameters->textclass, newpar->GetLayout())->latextype != LATEX_COMMAND) {
	     dummyrow.par = newpar;
	     dummyrow.pos = newpar->Last();
	     x = LeftMargin(&dummyrow);
	  }
      else {
	 /* this is no longer an error, because this function is used
	  * to clear impossible depths after changing a layout. Since there
	  * is always a redo, LeftMargin() is always called */ 
	 
	 /* printf("LYX_ERROR (text, LeftMargin()) impossible depth \n");*/ 
	 row->par->FirstPhysicalPar()->depth = 0;
      }
      
      if (newpar && !row->par->GetLayout()) {
	 if (newpar->FirstPhysicalPar()->noindent)
	    parindent = "";
	 else
	    parindent = lyxstyle.Style(parameters->textclass, newpar->GetLayout())->parindent;
      }
      
   }
   
   labelfont = GetFont(row->par, -2);
   switch (layout->margintype) {
    case MARGIN_DYNAMIC:
      if (layout->leftmargin[0]) {
	x += LyXStringWidth(GetFont(row->par, -1),
			    layout->leftmargin);
      }
      if (row->par->GetLabelString()) {
	x += LyXStringWidth(labelfont, layout->labelindent);
	    x += LyXStringWidth(labelfont, row->par->GetLabelString());
	    x += LyXStringWidth(labelfont, layout->labelsep);
	 }
      break;
    case MARGIN_MANUAL:
      x += LyXStringWidth(labelfont, layout->labelindent);
      if (row->pos >= BeginningOfMainBody(row->par)) {
	 if (row->par->GetLabelWidthString()) {
	    x += LyXStringWidth(labelfont, row->par->GetLabelWidthString());
	    x += LyXStringWidth(labelfont, layout->labelsep);
	 }
      }
      break;
    case MARGIN_STATIC:
      x += ( LyXStringWidth(GetFont(row->par, -1), layout->leftmargin) * 4
	    / (row->par->GetDepth() + 4));
      break;
    case MARGIN_FIRST_DYNAMIC:
      if (layout->labeltype == LABEL_MANUAL) {
	 if (row->pos >= BeginningOfMainBody(row->par)) {
	    x += LyXStringWidth(labelfont, layout->leftmargin);
	 }
	 else {
	    x += LyXStringWidth(labelfont, layout->labelindent);
	 }
      }
      else {
	 if (row->pos) {
	    x += LyXStringWidth(labelfont, layout->leftmargin);
	 }
	 else if (layout->labeltype != LABEL_TOP_ENVIRONMENT
		  && layout->labeltype != LABEL_CENTERED_TOP_ENVIRONMENT){
	    x += LyXStringWidth(labelfont, layout->labelindent);
	    x += LyXStringWidth(labelfont, layout->labelsep);
	    x += LyXStringWidth(labelfont, row->par->GetLabelString());
	 }
      }
      break;
      
    case MARGIN_RIGHT_ADDRESS_BOX:
      /* ok, a terrible hack. The left margin depends on the widest row
       * in this paragraph. Do not care about footnotes, they are *NOT*
       * allowed in the LaTeX realisation of this layout. */ 
      
      /* find the first row of this paragraph */ 
      Row *tmprow = row;
      while (tmprow->previous && tmprow->previous->par == row->par)
      	tmprow = tmprow->previous;
      
      int minfill = tmprow->fill;
      while (tmprow-> next && tmprow->next->par == row->par) {
	 tmprow = tmprow->next;
	 if (tmprow->fill < minfill)
	   minfill = tmprow->fill;
      }
      
      x += LyXStringWidth(GetFont(row->par, -1), layout->leftmargin);
      x += minfill;
      break;
   }
   
   int align;
   if (row->par->FirstPhysicalPar()->align == LYX_ALIGN_LAYOUT)
     align = layout->align;
   else
     align = row->par->FirstPhysicalPar()->align;
   
   /* set the correct parindent */
   if (row->pos == 0
       && (layout->labeltype == LABEL_NO_LABEL ||
	   layout->labeltype == LABEL_TOP_ENVIRONMENT ||
	   layout->labeltype == LABEL_CENTERED_TOP_ENVIRONMENT)
       && row->par == row->par->FirstPhysicalPar()
       && align == LYX_ALIGN_BLOCK
       && !row->par->noindent
       && (row->par->layout || parameters->paragraph_separation == LYX_PARSEP_INDENT))
      x += LyXStringWidth(GetFont(row->par, -1), parindent);
   
   return x;
}
    
   
int LyXText::RightMargin(Row *row)
{
   int  x;
   LyXLayout* layout;
   
   LyXParagraph *newpar;
   Row dummyrow;
   layout = lyxstyle.Style(parameters->textclass, row->par->GetLayout());

   x = LYX_PAPER_MARGIN;
   if (row->par->footnoteflag == LYX_OPEN_FOOTNOTE)  {
     x += LYX_PAPER_MARGIN/2;
   }

    /* this is the way, LyX handles the LaTeX-Environments.
    * I have had this idea very late, so it seems to be a
    * later added hack and this is true */ 
   if (row->par->GetDepth()) {
      /* find the next level paragraph */ 
      
      newpar = row->par;
      
      do {
	 newpar = newpar->FirstPhysicalPar()->Previous()->FirstPhysicalPar();
      } while (newpar && newpar->GetDepth() >= row->par->GetDepth()
	       && newpar->footnoteflag == row->par->footnoteflag);
      
      /* make a corresponding row. Needed to call LeftMargin() */

      /* check wether it is a sufficent paragraph */ 
      if (newpar && newpar->footnoteflag == row->par->footnoteflag
	  && lyxstyle.Style(parameters->textclass, newpar->GetLayout())->latextype != LATEX_PARAGRAPH
	  && lyxstyle.Style(parameters->textclass, newpar->GetLayout())->latextype != LATEX_COMMAND) {
	     dummyrow.par = newpar;
	     dummyrow.pos = 0;
	     x = RightMargin(&dummyrow);
	  }
      else {
	 /* this is no longer an error, because this function is used
	  * to clear impossible depths after changing a layout. Since there
	  * is always a redo, LeftMargin() is always called */ 
	 
	 /* printf("LYX_ERROR (text, LeftMargin()) impossible depth \n");*/ 
	 row->par->FirstPhysicalPar()->depth = 0;
      }
   }

   x += (LyXStringWidth(GetFont(row->par, -1), layout->rightmargin) * 4
	 / (row->par->GetDepth() + 4));
   return x;
   
}


int LyXText::LabelEnd (Row *row)
{
   if (lyxstyle.Style(parameters->textclass, row->par->GetLayout())->margintype == MARGIN_MANUAL) {
      Row tmprow;
      tmprow = *row;
      tmprow.pos = row->par->Last();
      return LeftMargin(&tmprow);      /* just the beginning 
					* of the main body */
   }
   else
     return 0;			       /* LabelEnd is only needed, if the  
					* layout fills a flushleft
					* label. */
}


/* table stuff -- begin*/ 
int LyXText::NumberOfCell(LyXParagraph *par, int pos)
{
   int cell = 0;
   int tmp_pos = 0;
   while (tmp_pos < pos) {
      if (par->IsNewline(tmp_pos))
      	 cell++;
      tmp_pos++;
   }
   return cell;
}


int LyXText::WidthOfCell(LyXParagraph *par, int &pos)
{
   int w = 0;
   while (pos < par->Last() && !par->IsNewline(pos)) {
      w += SingleWidth(par, pos);
      pos++;
   }
   if (par->IsNewline(pos))
      pos++;
   return w;
}

char LyXText::HitInTable(Row* row, int x){
  float tmpx;
  float fill_separator, fill_hfill, fill_label_hfill;
  if (!row->par->table)
    return 0;
  PrepareToPrint(row, tmpx, fill_separator, fill_hfill, fill_label_hfill);
  return (x > tmpx && x < tmpx + row->par->table->WidthOfTable());
}



/* table stuff -- end*/


/* get the next breakpoint in a given paragraph */
int LyXText::NextBreakPoint(Row* row, int width)
{
   int x = 0;
   int last_separator = -1; /* position of the last possible breakpoint 
			    * -1 isn't a suitable value, but a flag */
   int i = 0;
   int left_margin;
   
   LyXParagraph *par = row->par;
   int pos = row->pos;
   
   /* table stuff -- begin*/ 
   if (par->table) {
      while (pos<par->last 
	     && (!par->IsNewline(pos) 
		 || !par->table->IsFirstCell(NumberOfCell(par, pos+1)))) {
	if (par->GetChar(pos) == LYX_META_INSET &&
	    par->GetInset(pos) && par->GetInset(pos)->Display()){
	  par->GetInset(pos)->SetDisplay(false);
	}
	pos++;
      }
      return pos;
   }
   /* table stuff -- end*/ 

   left_margin = LabelEnd(row);
   width -= RightMargin(row);
   int main_body = BeginningOfMainBody(par);
   LyXLayout* layout = lyxstyle.Style(parameters->textclass, par->GetLayout());
   i = pos;
   char c;

   if (layout->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
      /* special code for right address boxes, only newlines count */
      while (i < par->Last()) {
	 if (par->IsNewline(i)) {
	   last_separator = i;
	   i = par->Last() - 1;/* this means break  */
	   x = width;
	 }
	 else if (par->GetChar(i) == LYX_META_INSET &&
		  par->GetInset(i) && par->GetInset(i)->Display()){
	   par->GetInset(i)->SetDisplay(false);
	 }
	 i++;
      }
   }
   else {
      /* this is the usual handling */ 
      x = LeftMargin(row);
      while (x < width && i < par->Last()) {
	c = par->GetChar(i);
	 if (IsNewlineChar(c)) {
	   last_separator = i;
	   x = width;		       /* this means break  */
	 }
	 else if (c == LYX_META_INSET &&
		 par->GetInset(i) && par->GetInset(i)->Display()){
	   /* check wether a Display() inset is valid here .
	      if not, change it to non-display*/ 
	   if (layout->latextype == LATEX_COMMAND
	       || (layout->labeltype == LABEL_MANUAL
		   && i < BeginningOfMainBody(par))){
	     /* display istn't allowd */ 
	     par->GetInset(i)->SetDisplay(false);
	     x += SingleWidth(par, i);
	   }
	   else {
	     /* inset is display. So break the line here */ 
	     if (i==pos){
	       if (pos < par->Last()-1)
		 last_separator = i;
	       else
		 last_separator = par->Last(); // to avaoid extra rows
	     }
	     else
	       last_separator = i - 1;
	     x = width;		       /* this means break  */
	   }
	 }
	 else  {
	    if (IsLineSeparatorChar(c))
	      last_separator = i;
	    x += SingleWidth(par, i);
	 }
	 i++;
	 if (i == main_body) {
	    x += LyXStringWidth(GetFont(par, -2),
				layout->labelsep);
	    if (par->IsLineSeparator(i - 1))
	      x-= SingleWidth(par, i - 1);
	    if (x < left_margin)
	      x = left_margin;
	 }
      }
      /* end of paragraph is always a suitable separator */
     if (i == par->Last() && x < width)
       last_separator = i;
   }
   
   /* well, if last_separator is still 0, the line isn't breakable. 
    * don't care and cut simply at the end */
   if (last_separator < 0) {
     last_separator = i;
   }
   
   /* manual labels cannot be broken in LaTeX, do not care  */
   if (main_body && last_separator < main_body)
     last_separator = main_body - 1;

   return last_separator;
}


/* returns the minimum space a row needs on the screen in pixel */
int LyXText::Fill(Row *row, int paperwidth)
{
   int w,i, last, fill, left_margin;
   /* get the pure distance */
   last = RowLast(row);
   
   /* table stuff -- begin*/
   if (row->par->table) {
      /* for tables FILL does calculate the widthes of each cell in 
       * the row */ 
      int pos = row->pos;
      int cell = NumberOfCell(row->par, pos);
      w = 0;
      do {
	 row->par->table->SetWidthOfCell(cell, 
					 WidthOfCell(row->par, pos));
	 cell++;
      } while (pos <= last && !row->par->table->IsFirstCell(cell));
      
      return 0; /* width of table cannot be returned since
		 * we cannot guarantee its correct value at
		 * this point. */ 
   }
   /* table stuff -- end*/ 

   left_margin = LabelEnd(row);

   if (last >= 0){
     /* if the row ends with newline, this newline will not be relevant */ 
     if (row->par->IsNewline(last))
       last--;
     
     /* if the row ends with a space, this space will not be relevant */ 
     if (row->par->IsLineSeparator(last))
       last--;
   }
   /* special handling of the right address boxes */ 
   if (lyxstyle.Style(parameters->textclass, row->par->GetLayout())->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
      int tmpfill = row->fill;
      row->fill = 0; 		       /* the minfill in MarginLeft()  */
      w = LeftMargin(row);
      row->fill = tmpfill;
      
      /* the old way, impossible when using environments: */ 
      /*  w = LyXStringWidth(lyxstyle.Style(parameters->textclass, row->par->GetLayout())->font, */ 
      /* 			 lyxstyle.Style(parameters->textclass, row->par->GetLayout())->leftmargin); */
   }
   else
     w = LeftMargin(row);
   
   int main_body = BeginningOfMainBody(row->par);
   LyXLayout *layout = lyxstyle.Style(parameters->textclass,
				      row->par->GetLayout());
   i = row->pos;
   while (i<= last) {
      w += SingleWidth(row->par, i);
      i++;
      if (i == main_body) {
	 w += LyXStringWidth(GetFont(row->par, -2),
			     layout->labelsep);
	 if (row->par->IsLineSeparator(i - 1))
	   w-= SingleWidth(row->par, i - 1);
	 if (w < left_margin)
	   w = left_margin;
      }
   }
   
   fill = paperwidth - w - RightMargin(row);
   return fill;
}


/* returns the minimum space a manual label needs on the screen in pixel */ 
int LyXText::LabelFill(Row *row)
{
   int w,i, last;
   int fill=0;
   
   last = BeginningOfMainBody(row->par) - 1;
   /* -1 because a label ends either with a space that is in the label, 
    * or with the beginnig of a footnote that is outside the label. */ 
   
   if (row->par->IsLineSeparator(last))    /* a sepearator at this end 
					* does not count */
     last--;
   
   w = 0;
   i = row->pos;
   while (i<= last) {
      w += SingleWidth(row->par, i);
      i++;
   }
   
   if (row->par->labelwidthstring) {
      fill = LyXStringWidth(GetFont(row->par, -2),
			    row->par->labelwidthstring) - w;
   }
   
   if (fill < 0)
     fill = 0;
   
   return fill;
}


/* returns the number of separators in the specified row. The separator 
* on the very last column doesnt count */ 
int LyXText::NumberOfSeparators(Row *row)
{
   int n,p,last;
   
   last = RowLast(row);
   n = 0;
   p = row->pos;
   int main_body = BeginningOfMainBody(row->par);
   if (p < main_body)
     p = main_body;
   for (; p < last; p++) {
      if (row->par->IsSeparator(p)) {
	 n++;
      }
   }
   return n;
}


/* returns the number of hfills in the specified row. The LyX-Hfill is
* a LaTeX \hfill so that the hfills at the beginning and at the end were 
* ignored. This is *MUCH* more usefull than not to ignore!  */
int LyXText::NumberOfHfills(Row *row)
{
   int n,p,last, first;
   
   last = RowLast(row);
   first = row->pos;
   if (first) {			       /* hfill *DO* count at the beginning 
					* of paragraphs! */
     while(first < last && row->par->IsHfill(first))
       first++;
   }

   n = 0;
   int main_body = BeginningOfMainBody(row->par);
   if (first < main_body)
     first = main_body;
   for (p = first; p < last; p++) {    /* last, because the end is ignored!  */
      if (row->par->IsHfill(p)) {
	 n++;
      }
   }
   return n;
}


/* like NumberOfHfills, but only those in the manual label! */ 
int LyXText::NumberOfLabelHfills(Row *row)
{
   int n,p,last, first;
   
   last = RowLast(row);
   first = row->pos;
   if (first) {			       /* hfill *DO* count at the beginning 
					* of paragraphs! */
      while(first < last && row->par->IsHfill(first))
      	first++;
   }

   n = 0;
   int main_body = BeginningOfMainBody(row->par);
   
   if (last > main_body)
     last = main_body;
   
   for (p = first; p < last; p++) {    /* last, because the end is ignored!  */
      if (row->par->IsHfill(p)) {
	 n++;
      }
   }
   return n;
}


/* returns true, if a expansion is needed.
 * Rules are given by LaTeX */ 
bool LyXText::HfillExpansion(Row *row_ptr, int pos)
{
   /* by the way, is it a hfill? */ 
   if (!row_ptr->par->IsHfill(pos))
     return false;
   
   /* at the end of a row it does not count */ 
   if (pos >= RowLast(row_ptr))
     return false;
   
   /* at the beginning of a row it does not count, if it is not 
    * the first row of a paragaph */ 
   if (!row_ptr->pos)
     return true;
   
   /* in some labels  it does not count */ 
   if ( lyxstyle.Style(parameters->textclass, row_ptr->par->GetLayout())->margintype != MARGIN_MANUAL &&
       pos < BeginningOfMainBody(row_ptr->par))
     return false; 
   
   /* if there is anything between the first char of the row and
    * the sepcified position that is not a newline and not a hfill,
    * the hfill will count, otherwise not */ 
   
   int i = row_ptr->pos;
   while (i < pos && (row_ptr->par->IsNewline(i)
		      || row_ptr->par->IsHfill(i)))
     i++;
   
   return (i != pos);
}


void LyXText::SetHeightOfRow(Row *row_ptr)
{
    /* get the maximum ascent and the maximum descent */
   int asc, maxasc, desc, maxdesc, pos_end, pos, labeladdon;
   float layoutasc = 0;
   float layoutdesc = 0;
   float tmptop = 0;
   LyXParagraph *par, *firstpar;
   LyXFont font, labelfont, tmpfont;
   Inset *tmpinset;
   char c;
   char size;

   /* this must not happen before the currentrow for clear reasons.
      so the trick is just to set the current row onto this row */
   long unused_y;
   GetRow(row_ptr->par, row_ptr->pos, unused_y);

   /* ok , let us initialize the maxasc and maxdesc value. 
    * This depends in LaTeX of the font of the last character
    * in the paragraph. The hack below is necessary because
    * of the possibility of open footnotes */

   /* corection: only the fontsize does count. The other properties
      are taken from the layoutfont. Nicer on the screen :) */
   
   par = row_ptr->par->LastPhysicalPar();
   firstpar = row_ptr->par->FirstPhysicalPar();
   
   LyXLayout *layout = lyxstyle.Style(parameters->textclass, firstpar->GetLayout());
   
   font = GetFont(par, par->Last()-1);
   size = font.size;
   
   font = GetFont(par, -1);
   labelfont = GetFont(par, -2);
   font.size = size;

   maxasc = (int) (LyXMaxAscent(font) *
		   layout->baselinestretch * parameters->baseline);
   maxdesc = (int) (LyXMaxDescent(font) *
		    layout->baselinestretch * parameters->baseline);

   pos_end = RowLast(row_ptr);
   
   asc = 0;
   desc = 0;
   labeladdon = 0;
   
   for (pos = row_ptr->pos; pos <= pos_end; pos++) {
      /* this check is, that I do not 
       * calculate unneccesary things. A X Call like Ascent is
       * too expensive! */
      tmpfont = GetFont(row_ptr->par, pos);
      c = row_ptr->par->GetChar(pos);
      if (c == LYX_META_INSET) {
         tmpinset = row_ptr->par->GetInset(pos);
         if (tmpinset) {
            asc = tmpinset->Ascent(tmpfont);
            desc = tmpinset->Descent(tmpfont);
         }
	 if (asc > maxasc) 
	   maxasc = asc;
	 if (desc > maxdesc)
	   maxdesc = desc;
      } else
         if (tmpfont.size > font.size) {
	    asc = LyXAscent(tmpfont, c);
	    desc = LyXDescent(tmpfont, c);
	    if (asc > maxasc) 
	      maxasc = asc;
	    if (desc > maxdesc)
	      maxdesc = desc;
         }
   }

   /* table stuff -- begin*/
   if (row_ptr->par->table){
     // stretch the rows a bit
      maxasc += 1;
      maxdesc +=1;
   }
   /* table stuff -- end*/

   // This is nicer with box insets:
   maxasc++;
   maxdesc++;

   row_ptr->ascent_of_text = maxasc;
   
   /* is it a top line? */ 
   if (row_ptr->pos == 0
       && row_ptr->par == firstpar) {
      
      /* some parksips VERY EASY IMPLEMENTATION */ 
      if (parameters->paragraph_separation == LYX_PARSEP_SKIP) {
	 if (layout->latextype == LATEX_PARAGRAPH
	     && firstpar->GetDepth() == 0
	     && firstpar->Previous())
	    maxasc += (int)(layout->parskip * DefaultHeight());
	 else if (firstpar->Previous()
		  && lyxstyle.Style(parameters->textclass,
			   firstpar->Previous()->GetLayout())->latextype ==
		              LATEX_PARAGRAPH
		  && firstpar->Previous()->GetDepth() == 0)
	   maxasc += (int)(lyxstyle.Style(parameters->textclass,
				 firstpar->Previous()->GetLayout())->parskip *
			         DefaultHeight());
      }
      
      /* the paper margins */ 
      if (!row_ptr->par->previous)
      	 maxasc += LYX_PAPER_MARGIN;
      
      /* add the vertical spaces, that the user added */
      if (firstpar->added_space_top)
      	 maxasc += (int) (firstpar->added_space_top * 37);
      
      /* do not forget the DTP-lines! 
       * there height depends on the font of the nearest character */
      if (firstpar->line_top)
      	 maxasc += 2 * LyXAscent(GetFont(firstpar, 0), 'x');
      
      /* and now the pagebreaks */ 
      if (firstpar->pagebreak_top)
      	 maxasc += 3 * DefaultHeight();
      
      /* and the vertical fills */ 
      if (firstpar->fill_top)
      	 maxasc += 3 * DefaultHeight();
      
      /*  this is special code for the chapter, since the label of this
       * layout is printed in an extra row */ 
      if (layout->labeltype == LABEL_COUNTER_CHAPTER
	  && parameters->secnumdepth>=0) {
	labeladdon = (int) (LyXMaxDescent(labelfont) *
			    layout->baselinestretch * parameters->baseline)
	             + (int) (LyXMaxAscent(labelfont) *
			      layout->baselinestretch * parameters->baseline);
	//	 + (int) layout->parsep * DefaultHeight();
      }
      
      /* sepcial code for the top label */ 
      if ((layout->labeltype == LABEL_TOP_ENVIRONMENT
	   || layout->labeltype == LABEL_CENTERED_TOP_ENVIRONMENT)
	  && (row_ptr->par == row_ptr->par->DepthHook(row_ptr->par->GetDepth())
	      || (row_ptr->par->DepthHook(row_ptr->par->GetDepth())->GetLayout() != row_ptr->par->GetLayout()
		  || row_ptr->par->DepthHook(row_ptr->par->GetDepth())->GetDepth() != row_ptr->par->GetDepth()
		  )
	      )
	  ){
	 labeladdon = (int)(
	   (LyXMaxAscent(labelfont) *
	    layout->baselinestretch * parameters->baseline)
	   +(LyXMaxDescent(labelfont) *
	     layout->baselinestretch * parameters->baseline)
	   + layout->topsep * DefaultHeight()
	   + layout->labelbottomsep *  DefaultHeight());
      }
   
      /* and now the layout spaces, for example before and after a section, 
       * or between the items of a itemize or enumerate environment */ 
      
      if (!firstpar->pagebreak_top) {
	 LyXParagraph *prev = row_ptr->par->Previous();
	 if (prev)
	    prev = row_ptr->par->DepthHook(row_ptr->par->GetDepth());
	 if (prev && prev->GetLayout() == firstpar->GetLayout()
	     && prev->GetDepth() == firstpar->GetDepth()
	     && StringEqual(prev->GetLabelWidthString(),
			    firstpar->GetLabelWidthString()))
	   {
	      layoutasc = (layout->itemsep * DefaultHeight());
	   }
	 else if (row_ptr->previous) {
	    tmptop = layout->topsep;
	    
	    if (row_ptr->previous->par->GetDepth() >= row_ptr->par->GetDepth())
	       tmptop-=lyxstyle.Style(parameters->textclass, row_ptr->previous->par->GetLayout())->bottomsep;
	    
	    if (tmptop > 0)
	       layoutasc = (tmptop * DefaultHeight());
	 }
	 else if (row_ptr->par->line_top){
	    tmptop = layout->topsep;
	    
	    if (tmptop > 0)
	       layoutasc = (tmptop * DefaultHeight());
	 }
	 
	 prev = row_ptr->par->DepthHook(row_ptr->par->GetDepth()-1);
	 if (prev)  {
	    maxasc += (int) (lyxstyle.Style(parameters->textclass, prev->GetLayout())->parsep * DefaultHeight());
	 }
	 else {
		if (firstpar->Previous()
		    && firstpar->Previous()->GetDepth() == 0
		    && firstpar->Previous()->GetLayout() != firstpar->GetLayout()) {
		   /* avoid parsep */ 
		}
	    else if (firstpar->Previous()){
	       maxasc += (int) (layout->parsep * DefaultHeight());
	    }
	 }
      }
   }
   
   /* is it a bottom line? */ 
   if (row_ptr->par->ParFromPos(RowLast(row_ptr) + 1) == par
       && (!row_ptr->next || row_ptr->next->par != row_ptr->par)) {     
	  
	  /* the paper margins */ 
	  if (!par->next)
	    maxdesc += LYX_PAPER_MARGIN;
	
	  /* add the vertical spaces, that the user added */
	  if (firstpar->added_space_bottom)
	    maxdesc += (int) (firstpar->added_space_bottom * 37);
	  
	  /* do not forget the DTP-lines! 
	   * there height depends on the font of the nearest character */
	  if (firstpar->line_bottom)
	    maxdesc += 2 * LyXAscent(GetFont(par, par->Last()-1), 'x');
	  
	  /* and now the pagebreaks */
	  if (firstpar->pagebreak_bottom)
	    maxdesc += 3 * DefaultHeight();
	  
	  /* and the vertical fills */
	  if (firstpar->fill_bottom)
	    maxdesc += 3 * DefaultHeight();
	  
	  /* and now the layout spaces, for example before and after a section, 
	   * or between the items of a itemize or enumerate environment */
	  if (!firstpar->pagebreak_bottom && row_ptr->par->Next()) {
	     LyXParagraph *nextpar = row_ptr->par->Next();
	     LyXParagraph *comparepar = row_ptr->par;
	     float usual = 0;
	     float  unusual = 0;
	     
	     if (comparepar->GetDepth() > nextpar->GetDepth()) {
		usual = (lyxstyle.Style(parameters->textclass, comparepar->GetLayout())->bottomsep * DefaultHeight());
		comparepar = comparepar->DepthHook(nextpar->GetDepth());
		if (comparepar->GetLayout()!=nextpar->GetLayout()
		    ||(!StringEqual(nextpar->GetLabelWidthString(),
				    comparepar->GetLabelWidthString())))
		  unusual = (lyxstyle.Style(parameters->textclass, comparepar->GetLayout())->bottomsep * DefaultHeight());
		
		if (unusual > usual)
		  layoutdesc = unusual;
		else
		  layoutdesc = usual;
	     }
	     else if (comparepar->GetDepth() ==  nextpar->GetDepth()) {
		
		if (comparepar->GetLayout()!= nextpar->GetLayout()
		    || (!StringEqual(nextpar->GetLabelWidthString(),
				     comparepar->GetLabelWidthString())))
		  layoutdesc = (int)(lyxstyle.Style(parameters->textclass, comparepar->GetLayout())->bottomsep * DefaultHeight());
	     }
	  }
       }
   
   /* incalculate the layout spaces */ 
   maxasc += (int) (layoutasc * 2 / (2 + firstpar->GetDepth()));
   maxdesc += (int) (layoutdesc * 2 / (2 + firstpar->GetDepth()));

   /* table stuff -- begin*/
   if (row_ptr->par->table){
      maxasc += row_ptr->par->table->
	AdditionalHeight(NumberOfCell(row_ptr->par, row_ptr->pos));
   }
   /* table stuff -- end*/
   
   /* calculate the new height of the text */ 
   height -= row_ptr->height;
   
   row_ptr->height=maxasc+maxdesc+labeladdon;
   row_ptr->baseline=maxasc+labeladdon;
   
   height += row_ptr->height;
}


/* appends  the implizit specified paragraph behind the specified row,
* start at the implizit given position */
void LyXText::AppendParagraph(Row *row)
{
   int z;
   Row *tmprow;
   char not_ready;
   
   not_ready = 1;
   
   /* get the width of the row */
   z = row->pos;

   do {
      /* get the next breakpoint */
      z = NextBreakPoint(row, paperwidth);
      
      
      tmprow = row;
      /* inset the new row */
      if (z < row->par->Last()) {
	 z++;
	 InsertRow(row, row->par, z);
	 row = row->next;
	 row->height = 0;
      }
      else
      	not_ready = 0;
      
      /* set the dimensions of the row */ 
      tmprow->fill = Fill(tmprow, paperwidth);
      SetHeightOfRow(tmprow);
      
   } while (not_ready);
}


void LyXText::BreakAgain(Row *row)
{
   int z;
   char not_ready;
   z = row->pos;
   Row *tmprow, *tmprow2;
   
   not_ready = 1;
   
   do  {
      /* get the next breakpoint */
      z = NextBreakPoint(row, paperwidth);
      
      tmprow = row;
      
      if (z < row->par->Last() ) {
	 if (!row->next || (row->next && row->next->par != row->par)) {
	    /* insert a new row */ 
	    z++;
	    InsertRow(row, row->par, z);
	    row = row->next;
	    row->height = 0;
	 }
	 else  {
	    row=row->next;
	    z++;
	    if (row->pos == z)
	      not_ready = 0;	       /* the rest will not change  */
	    else {
	       row->pos = z;
	    }
	 }
      }
      else {
	 /* if there are some rows too much, delete them */
	 /* only if you broke the whole paragraph! */ 
	 tmprow2 = row;
	 while (tmprow2->next && tmprow2->next->par == row->par) {
	    tmprow2 = tmprow2->next;
	 }
	 while (tmprow2 != row) {
	    tmprow2 = tmprow2->previous;
	    RemoveRow(tmprow2->next);
	 }
	 not_ready = 0;
      }
       
      /* set the dimensions of the row */ 
      tmprow->fill = Fill(tmprow, paperwidth);
      SetHeightOfRow(tmprow);
   } while (not_ready);
}


/* this is just a little changed version of break again */ 
void LyXText::BreakAgainOneRow(Row *row)
{
   int z;
   char not_ready;
   z = row->pos;
   Row *tmprow, *tmprow2;
   
   not_ready = 1;
   
   /* get the next breakpoint */
   z = NextBreakPoint(row, paperwidth);
   
   tmprow = row;
   
   if (z < row->par->Last() ) {
      if (!row->next || (row->next && row->next->par != row->par)) {
	 /* insert a new row */ 
	 z++;
	 InsertRow(row, row->par, z);
	 row = row->next;
	 row->height = 0;
      }
      else  {
	 row=row->next;
	 z++;
	 if (row->pos == z)
	   not_ready = 0;	       /* the rest will not change  */
	 else {
	    row->pos = z;
	 }
      }
   }
   else {
      /* if there are some rows too much, delete them */
      /* only if you broke the whole paragraph! */ 
      tmprow2 = row;
      while (tmprow2->next && tmprow2->next->par == row->par) {
	 tmprow2 = tmprow2->next;
      }
      while (tmprow2 != row) {
	 tmprow2 = tmprow2->previous;
	 RemoveRow(tmprow2->next);
      }
      not_ready = 0;
   }
   
   /* set the dimensions of the row */ 
   tmprow->fill = Fill(tmprow, paperwidth);
   SetHeightOfRow(tmprow);
}


void LyXText::BreakParagraph(char keep_layout)
{
   LyXLayout *layout = lyxstyle.Style(parameters->textclass,
				      cursor.par->GetLayout());
   
   /* table stuff -- begin*/
   if (cursor.par->table) {
     // breaking of tables is only allowed at the beginning or the end */
     if (cursor.pos && cursor.pos < cursor.par->last)
      return; /* no breaking of tables allowed */ 
   }
   /* table stuff -- end*/

   /* this is only allowed, if the current paragraph is not empty or caption*/ 
   if ((cursor.par->Last() <= 0 && !cursor.par->IsDummy())
       && 
       layout->labeltype!=LABEL_SENSITIVE)
     return;

   SetUndo(LYX_UNDO_INSERT, 
	   cursor.par->ParFromPos(cursor.pos)->previous, 
	   cursor.par->ParFromPos(cursor.pos)->next); 

   /* please break alway behind a space */ 
   if (cursor.pos < cursor.par->Last()
       && cursor.par->IsLineSeparator(cursor.pos))
     cursor.pos++;
   
   /* break the paragraph */
   if (keep_layout)
     keep_layout = 2;
   else	
     keep_layout = (layout->latextype >= LATEX_ENVIRONMENT);
   cursor.par->BreakParagraph(cursor.pos, keep_layout);

   /* table stuff -- begin*/
   if (cursor.par->table){
     // the table should stay with the contents
     if (!cursor.pos){
       cursor.par->Next()->table = cursor.par->table;
       cursor.par->table = NULL;
     }
   }
   /* table stuff -- end*/

   /* well this is the caption hack since one caption is really enough */
   if (layout->labeltype == LABEL_SENSITIVE){
     if (!cursor.pos)
       cursor.par->SetLayout(0); /* set the new paragraph to standard-layout */
     else
       cursor.par->Next()->SetLayout(0); /* set the new paragraph to standard-layout */
	 
   }
   
   /* if the cursor is at the beginning of a row without prior newline, 
    *  move one row up! 
    * This touches only the screen-update. Otherwise we would may have
    * an empty row on the screen */
   if (cursor.pos && !cursor.row->par->IsNewline(cursor.row->pos -1) &&
       cursor.row->pos == cursor.pos) {
     CursorLeft();
   } 
   
   status = LYX_NEED_MORE_REFRESH;
   refresh_row = cursor.row;
   refresh_y = cursor.y - cursor.row->baseline;
   
   /* Do not forget the special right address boxes */
   if (layout->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
      while (refresh_row->previous &&
	     refresh_row->previous->par == refresh_row->par) {
		refresh_row = refresh_row->previous;
		refresh_y -= refresh_row->height;
	     }
   }
   RemoveParagraph(cursor.row);
   
   /* set the dimensions of the cursor row */
   cursor.row->fill = Fill(cursor.row, paperwidth);

   SetHeightOfRow(cursor.row);
   
   while (cursor.par->Next()->Last()
	  && cursor.par->Next()->IsNewline(0))
     cursor.par->Next()->Erase(0);
   
   InsertParagraph(cursor.par->Next(), cursor.row);

   UpdateCounters(cursor.row->previous);
   
   /* this check is necessary. Otherwise the new empty paragraph will
    * be deleted automatically. And it is more friendly for the user! */ 
   if (cursor.pos)
     SetCursor(cursor.par->Next(), 0);
   else
     SetCursor(cursor.par, 0);
   
   if (cursor.row->next)
     BreakAgain(cursor.row->next);
}


void LyXText::OpenFootnote()
{
   LyXParagraph *par, *endpar,*tmppar;
   Row *row;
   
   par = cursor.par->ParFromPos(cursor.pos);
   
   /* if there is no footnote in this paragraph, just return. */ 
   if (!par->next
       || par->next->footnoteflag != LYX_CLOSED_FOOTNOTE)
     return;
   
   /* ok, move the cursor right before the footnote */ 
   
   /* just a little faster than using CursorRight() */
   for (cursor.pos=0; cursor.par->ParFromPos(cursor.pos)!=par; cursor.pos++);
   /* now the cursor is at the beginning of the physical par */
   SetCursor(cursor.par, cursor.pos + cursor.par->ParFromPos(cursor.pos)->last);
   
   /* the cursor must be exactly before the footnote */ 
   par = cursor.par->ParFromPos(cursor.pos);
   
   status = LYX_NEED_MORE_REFRESH;
   refresh_row = cursor.row;
   refresh_y = cursor.y - cursor.row->baseline;
   
   tmppar = cursor.par;
   endpar = cursor.par->Next();
   row = cursor.row;
   
   tmppar->OpenFootnote(cursor.pos);
   RemoveParagraph(row);
   /* set the dimensions of the cursor row */
   row->fill = Fill(row, paperwidth);
   SetHeightOfRow(row);
   tmppar = tmppar->Next();
   
   while (tmppar != endpar) {
      if (tmppar) {
	 InsertParagraph(tmppar, row);
	 while (row->next && row->next->par == tmppar)
	   row = row->next;
	 tmppar = tmppar->Next();
      }
   }
   SetCursor(par->next, 0);
   sel_cursor = cursor;
}
   

/* table stuff -- begin*/

void LyXText::TableFeatures(int feature)
{ 
	if (!cursor.par->table)
		return; /* this should never happen */
  
	SetUndo(LYX_UNDO_FINISH, 
		cursor.par->ParFromPos(cursor.pos)->previous, 
		cursor.par->ParFromPos(cursor.pos)->next); 
	
	switch (feature){
	case LYX_TABLE_APPEND_ROW:
	{
		int pos = cursor.pos;
		/* move to the next row */
		int cell_org = NumberOfCell(cursor.par, cursor.pos);
		int cell = cell_org;
		while (pos < cursor.par->Last() && 
		(cell == cell_org
		 || !cursor.par->table->IsFirstCell(cell))){
			while (pos < cursor.par->Last() && !cursor.par->IsNewline(pos))
				pos++;
			if (pos < cursor.par->Last())
			  pos++;
			cell++;
		}
		
		/* insert the new cells */ 
		int number = cursor.par->table->NumberOfCellsInRow(cell_org);
		int i;
		for (i=0; i<number; i++)
			cursor.par->InsertChar(pos, LYX_META_NEWLINE);
		
		/* append the row into the table */ 
		cursor.par->table->AppendRow(cell_org);
		
		RedoParagraph();
		return;
	}
	break;
	
	case LYX_TABLE_APPEND_COLUMN: 
	{
		int pos = 0;
		int cell_org = NumberOfCell(cursor.par, cursor.pos);
		int cell = 0;
		do{
			if (pos && (cursor.par->IsNewline(pos-1))){
				if (cursor.par->table->AppendCellAfterCell(cell_org, cell)){
					cursor.par->InsertChar(pos, LYX_META_NEWLINE);
					if (pos<=cursor.pos)
						cursor.pos++;
					pos++;
				}
				cell++;
			}
			pos++;
		} while (pos<= cursor.par->Last());
		/* remember that the very last cell doesn't end with a newline.
		   This saves one byte memory per table ;-) */ 
		if (cursor.par->table->AppendCellAfterCell(cell_org, cell))
			cursor.par->InsertChar(cursor.par->Last(), LYX_META_NEWLINE);
		
		/* append the column into the table */ 
		cursor.par->table->AppendColumn(cell_org);
		
		RedoParagraph();
		return;
	}
	break;

	case LYX_TABLE_DELETE_ROW:
	{
		/* move to the previous row */
		int cell = NumberOfCell(cursor.par, cursor.pos);
		while (cursor.pos && !cursor.par->IsNewline(cursor.pos-1))
		  cursor.pos--;
		while (cursor.pos && 
		       !cursor.par->table->IsFirstCell(cell)){
		  cursor.pos--;
		  while (cursor.pos && !cursor.par->IsNewline(cursor.pos-1))
		    cursor.pos--;
		  cell--;
		}

		/* delete up to the next row */
		int cell_org = cell;
		while (cursor.pos < cursor.par->Last() && 
		(cell == cell_org
		 || !cursor.par->table->IsFirstCell(cell))){
			while (cursor.pos < cursor.par->Last() && !cursor.par->IsNewline(cursor.pos))
				cursor.par->Erase(cursor.pos);
			cell++;
			if (cursor.pos < cursor.par->Last())
			  cursor.par-> Erase(cursor.pos);
		}
		if (cursor.pos && cursor.pos == cursor.par->Last()){
		  cursor.pos--;
		  cursor.par->Erase(cursor.pos); // no newline at the very end!
		}
		
		cursor.par->table->DeleteRow(cell_org);
		RedoParagraph();
		return;
	}
	break;
	
	case LYX_TABLE_DELETE_COLUMN: 
	{
		int pos = 0;
		int cell_org = NumberOfCell(cursor.par, cursor.pos);
		int cell = 0;
		do{
			if (!pos || (cursor.par->IsNewline(pos-1))){
			  if (cursor.par->table->DeleteCellIfColumnIsDeleted(cell, cell_org)){
			    // delete one cell
			    while (pos < cursor.par->Last() && !cursor.par->IsNewline(pos))
			      cursor.par->Erase(pos);
			    if (pos < cursor.par->Last())
			      cursor.par->Erase(pos);
			    else 
			      cursor.par->Erase(pos - 1); // the missing newline at the end of a table
			    pos--; // because of pos++ below
			  }   
			  cell++;
			}
			pos++;
		} while (pos<= cursor.par->Last());
		
		/* delete the column from the table */ 
		cursor.par->table->DeleteColumn(cell_org);
		
		/* set the cursor to the beginning of the table, where else? */ 
		cursor.pos = 0;
		RedoParagraph();
		return;
	}
	break;

	case LYX_TABLE_TOGGLE_LINE_TOP:
		cursor.par->table->SetTopLine(NumberOfCell(cursor.par, cursor.pos), 
					      !(cursor.par->table->TopLine(NumberOfCell(cursor.par, cursor.pos)))
			);
		RedoParagraph();
		return;
		break;
    
	case LYX_TABLE_TOGGLE_LINE_BOTTOM:
		cursor.par->table->SetBottomLine(NumberOfCell(cursor.par, cursor.pos), 
						 !(cursor.par->table->BottomLine(NumberOfCell(cursor.par, cursor.pos)))
			);
		RedoParagraph();
		return;
		break;
		
	case LYX_TABLE_TOGGLE_LINE_LEFT:
		cursor.par->table->SetLeftLine(NumberOfCell(cursor.par, cursor.pos), 
					       !(cursor.par->table->LeftLine(NumberOfCell(cursor.par, cursor.pos)))
			);
		RedoParagraph();
		return;
		break;

	case LYX_TABLE_TOGGLE_LINE_RIGHT:
		cursor.par->table->SetRightLine(NumberOfCell(cursor.par, cursor.pos), 
						!(cursor.par->table->RightLine(NumberOfCell(cursor.par, cursor.pos)))
			);
		RedoParagraph();
		return;
		break;
    
	case LYX_TABLE_ALIGN_LEFT:
		cursor.par->table->SetAlignment(NumberOfCell(cursor.par, cursor.pos),
						LYX_ALIGN_LEFT);
		RedoParagraph();
		return;
		break;
		
	case LYX_TABLE_ALIGN_RIGHT:
		cursor.par->table->SetAlignment(NumberOfCell(cursor.par, cursor.pos),
						LYX_ALIGN_RIGHT);
		RedoParagraph();
		return;
		break;
		
	case LYX_TABLE_ALIGN_CENTER:
		cursor.par->table->SetAlignment(NumberOfCell(cursor.par, cursor.pos),
						LYX_ALIGN_CENTER);
		RedoParagraph();
		return;
		break;
		
	case LYX_TABLE_DELETE_TABLE:
	  {
	    SetCursorIntern(cursor.par, 0);
	    delete cursor.par->table;
	    cursor.par->table = NULL;
	    // temporary: Should put table in simple_cut_buffer (with before and after
	    // dummy-paragraph !! 
	    // not necessar anymore with UNDO :)
	    int i;
	    for (i = cursor.par->last-1; i>=0; i--)
	      cursor.par->Erase(i);
	    RedoParagraph();
	    return;
	  }
	  break;
		
	case LYX_TABLE_MULTICOLUMN:
	  {
	    int number = 0;
	    // check wether we are completly in a multicol
	    int multicol = cursor.par->table->IsMultiColumn(NumberOfCell(cursor.par, cursor.pos));
	    if (multicol && selection && sel_start_cursor.row == sel_end_cursor.row){
	      multicol = NumberOfCell(sel_start_cursor.par, sel_start_cursor.pos)
		== NumberOfCell(sel_end_cursor.par, sel_end_cursor.pos);
	    }

	    if (multicol){
	      int newlines = cursor.par->table->UnsetMultiColumn(NumberOfCell(cursor.par, cursor.pos));
	      int pos = cursor.pos;
	      while (pos<cursor.par->Last() && !cursor.par->IsNewline(pos))
		pos++;
	      for (;newlines;newlines--)
		cursor.par->InsertChar(pos, LYX_META_NEWLINE);
	      RedoParagraph();
	      return;
	    }
	    else {
	      // selection must be in one row (or no selection)
	      if (!selection){
		cursor.par->table->SetMultiColumn(NumberOfCell(cursor.par,
							       cursor.pos),
						  1);
		RedoParagraph();
		return;
	      }
	      else {
		if (sel_start_cursor.row == sel_end_cursor.row){
		  int i;
		  number = 1;
		  for (i=sel_start_cursor.pos; i<sel_end_cursor.pos; i++){
		    if (sel_start_cursor.par->IsNewline(i)){
		      sel_start_cursor.par->Erase(i);
		      // check for double-blanks
		      if ((i && !sel_start_cursor.par->IsLineSeparator(i-1))
			  &&
			  (i<sel_start_cursor.par->Last() 
			   && !sel_start_cursor.par->IsLineSeparator(i)))
			sel_start_cursor.par->InsertChar(i, ' ');
		      else {
			sel_end_cursor.pos--;
			i--;
		      }
		      number++;
		    }
		  }
		  cursor.par->table->
			  SetMultiColumn(NumberOfCell(sel_start_cursor.par,
						      sel_start_cursor.pos),
					 number);
		  cursor.pos = sel_start_cursor.pos;
		  RedoParagraph();
		  return;
		}
		else {
		  WriteAlert("Impossible Operation!", 
			     "Multicolumns can only be horizontally.", 
			     "Sorry.");
		}
	      }
	    }
	  }
	  break;
	}
}
	

void LyXText::InsertCharInTable(char c)
{
	Row *row;
	Row *tmprow;
	long y;
	LyXFont tmpfont, realtmpfont, reference_font1, reference_font2;
	bool jumped_over_space;
	
	/* first check, if there will be two blanks together or a blank at 
	 * the beginning of a paragraph. 
	 * I decided to handle blanks like normal characters, the main 
	 * difference are the special checks when calculating the row.fill
	 * (blank does not count at the end of a row) and the check here */ 
	
	realtmpfont = real_current_font;
	tmpfont = current_font;	       /* store the current font.
					* This is because of the use
					* of cursor movements. The moving
					* cursor would refresh the 
					* current font */
	
	/* get the font that is used to calculate the baselineskip */ 
	reference_font1 = GetFont(cursor.par, cursor.par->Last() - 1);
	reference_font2 = cursor.par->GetFontSettings(cursor.par->Last() - 1);
	jumped_over_space = false;
	if (IsLineSeparatorChar(c)) {
		
		/* avoid double blanks but insert the new blank because
		 * of a possible font change */
		if (cursor.pos < cursor.par->Last() &&
		cursor.par->IsLineSeparator(cursor.pos))
		{
			cursor.par->Erase(cursor.pos);
			jumped_over_space = true;
		}
		else if ((cursor.pos > 0 && 
			  cursor.par->IsLineSeparator(cursor.pos - 1))
			 || (cursor.pos > 0 && cursor.par->IsNewline(cursor.pos - 1))
			  || (cursor.pos == 0 &&
			      !(cursor.par->Previous()
			      && cursor.par->Previous()->footnoteflag
			      == LYX_OPEN_FOOTNOTE)))
			return;
	}
	else if (IsNewlineChar(c)) {
	  /* the newline character is the separator of the cells */
	  // cursor.par->InsertChar(cursor.pos, c);
	  // SetCharFont(cursor.par, cursor.pos, tmpfont);
	  // RedoParagraphs(cursor, cursor.par->Next());
	  // SetCursor(cursor.par, cursor.pos+1);
	  return;
	}
   
	row = cursor.row;
	y = cursor.y - row->baseline;
	if (c != LYX_META_INSET)	       /* in this case LyXText::InsertInset 
						* already insertet the character */
		cursor.par->InsertChar(cursor.pos, c);
	SetCharFont(cursor.par, cursor.pos, tmpfont);
	
	if (!jumped_over_space) {
		/* refresh the positions */
		tmprow = row;
		while (tmprow->next && tmprow->next->par == row->par) {
			tmprow = tmprow->next;
			tmprow->pos++;
		}
	}

	// auto correction 
	if (IsSeparatorChar(c) || IsKommaChar(c)){
	  LyXParagraph* tmppar = cursor.par->ParFromPos(cursor.pos);
	  int tmpposend = cursor.par->PositionInParFromPos(cursor.pos)-1;
	  int tmppos = tmpposend;
	  int old_tmpposend = tmpposend;
	  int old_cursorpos = cursor.pos-1;
	  // int i; // unused
	  while (tmppos >  0 &&  
		 !tmppar->IsSeparator(tmppos-1) &&
		 !tmppar->IsKomma(tmppos-1)
		 ){
	    tmppos--;
	    old_cursorpos--;
	  }

	  if (AutoCorrect(tmppar, tmppos, tmpposend)){
	    Row* tmprow = cursor.row;
	    while (tmprow->next && tmprow->next->par == cursor.row->par) {
	      tmprow = tmprow->next;
	      tmprow->pos += (tmpposend - old_tmpposend);
	    }
	    cursor.pos = old_cursorpos + tmpposend - tmppos + 2;// for the character c inserted above
	    CheckParagraph(cursor.par, old_cursorpos);
	    current_font = tmpfont; real_current_font = realtmpfont;
	    /* check, wether the last characters font has changed. */ 
	    if (cursor.pos && cursor.pos == cursor.par->Last()
		&& !(FontEqual(reference_font1, tmpfont)
		     || FontEqual(reference_font2, tmpfont)))
	      RedoHeightOfParagraph(cursor);
	    else {
	      /* now the special right address boxes */
	      if (lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
		RedoDrawingOfParagraph(cursor); 
	      }
	    }
	    return;
	  }
	}

	cursor.pos++;

	CheckParagraphInTable(cursor.par, cursor.pos);
	
	current_font = tmpfont; real_current_font = realtmpfont;
	
	/* check, wether the last characters font has changed. */ 
	if (cursor.pos && cursor.pos == cursor.par->Last()
	&& !(FontEqual(reference_font1, tmpfont)
	     || FontEqual(reference_font2, tmpfont)))
		RedoHeightOfParagraph(cursor);
}


void LyXText::CheckParagraphInTable(LyXParagraph* par, int pos)
{
	Row *row;
	long y;
	
	if (par->GetChar(pos) == LYX_META_INSET &&
	    par->GetInset(pos) && par->GetInset(pos)->Display()){
	  par->GetInset(pos)->SetDisplay(false);
	}

	row = GetRow(par, pos, y);
	
	int tmpheight = row->height;
	SetHeightOfRow(row);
	
	int tmp_pos = pos;
	/* update the table information */
	while (tmp_pos && !par->IsNewline(tmp_pos - 1))
		tmp_pos--;
	if (par->table->SetWidthOfCell(NumberOfCell(par, pos),
				       WidthOfCell(par, tmp_pos))) {
		LyXCursor tmpcursor = cursor;
		SetCursorIntern(par, pos);
		/* make a complete redraw */
		RedoDrawingOfParagraph(cursor);
		cursor = tmpcursor;
	}
	else {
		/* redraw only the row */
		LyXCursor tmpcursor = cursor;
		SetCursorIntern(par, pos);
		refresh_y = y;
		refresh_x = cursor.x;
		refresh_row = row;
		refresh_pos = cursor.pos;
		cursor = tmpcursor;
		
		if (tmpheight == row->height)
			status = LYX_NEED_VERY_LITTLE_REFRESH;
		else
			status = LYX_NEED_MORE_REFRESH;
	}
	SetCursorIntern(cursor.par, cursor.pos);
}


void LyXText::BackspaceInTable()
{
	Row *tmprow, *row;
	long y;
	LyXFont tmpfont, realtmpfont, reference_font1, reference_font2;
	
	tmpfont = current_font;
	realtmpfont = real_current_font;
	
	reference_font1 = GetFont(cursor.par, cursor.par->Last() - 1);
	reference_font2 = cursor.par->GetFontSettings(cursor.par->Last() - 1);
	
	if (cursor.pos == 0) {
		/* no pasting of table paragraphs */
		
		CursorLeft();
	}
	else {
		/* this is the code for a normal backspace, not pasting
		 * any paragraphs */ 
	  SetUndo(LYX_UNDO_DELETE, 
		  cursor.par->ParFromPos(cursor.pos)->previous, 
		  cursor.par->ParFromPos(cursor.pos)->next); 
	  
		CursorLeftIntern();
		
		/* some insets are undeletable here */
		if (cursor.par->GetChar(cursor.pos)==LYX_META_INSET) {
			if (!cursor.par->GetInset(cursor.pos)->Deletable())
				return;
		}
		
		row = cursor.row;
		y = cursor.y - row->baseline;
		// int z; // unused
		
		/* some special code when deleting a newline. */
		if (cursor.par->IsNewline(cursor.pos)) {
			/* nothing :-) */
			return;
		}
		else {
			cursor.par->Erase(cursor.pos);
			
			/* refresh the positions */
			tmprow = row;
			while (tmprow->next && tmprow->next->par == row->par) {
				tmprow = tmprow->next;
				tmprow->pos--;
			}
			
			/* delete superfluous blanks */ 
			if (cursor.pos < cursor.par->Last() - 1 &&
			(cursor.par->IsLineSeparator(cursor.pos))) {
				
				if (cursor.pos == BeginningOfMainBody(cursor.par)
				|| !cursor.pos 
				|| cursor.par->IsLineSeparator(cursor.pos - 1)) {
					cursor.par->Erase(cursor.pos);
					/* refresh the positions */
					tmprow = row;
					while (tmprow->next && 
					       tmprow->next->par == row->par) {
						tmprow = tmprow->next;
						tmprow->pos--;
					}
					if (cursor.pos)   /* move one character left */
						cursor.pos--;
				}
			}
		}
      
		CheckParagraphInTable(cursor.par, cursor.pos);
      
		/* check, wether the last characters font has changed. */ 
		if (cursor.pos && cursor.pos == cursor.par->Last()
		&& !(FontEqual(reference_font1, tmpfont)
		     || FontEqual(reference_font2, tmpfont)))
			RedoHeightOfParagraph(cursor);
		
		/* restore the current font 
		 * That is what a user
		 * expects! */
		current_font = tmpfont; real_current_font = realtmpfont;
	}
}

/* table stuff -- end*/


/* just a macro to make some thing easier. */ 
void LyXText::RedoParagraph()
{
  LyXCursor tmpcursor = cursor;
  ClearSelection();
  RedoParagraphs(cursor, cursor.par->Next());;
  SetCursorIntern(tmpcursor.par, tmpcursor.pos);
}


/* insert a character, moves all the following breaks in the 
 * same Paragraph one to the right and make a rebreak */
void  LyXText::InsertChar(char c)
{
   Row *row;
   Row *tmprow;
   int z;
   long y;
   LyXFont tmpfont, realtmpfont, reference_font1, reference_font2;
   bool jumped_over_space;
   
   SetUndo(LYX_UNDO_INSERT, 
	   cursor.par->ParFromPos(cursor.pos)->previous, 
	   cursor.par->ParFromPos(cursor.pos)->next);

   /* When the free-spacing option is set for the current layout,
    * all spaces are converted to protected spaces. */
   bool freeSpacingBo =
	   lyxstyle.Style(parameters->textclass,
			  cursor.row->par->GetLayout())->free_spacing;
   //
   // Is this wanted? There cannot be a line break between protected separators. Therefore
   // I suggest the way implemented below (Matthias)
   //
   //   if ( freeSpacingBo && IsLineSeparatorChar(c) )
   //	   c = LYX_META_PROTECTED_SEPARATOR;
   
   if (freeSpacingBo && IsLineSeparatorChar(c) &&(!cursor.pos || cursor.par->IsLineSeparator(cursor.pos-1))) 
     c = LYX_META_PROTECTED_SEPARATOR;
   
/* table stuff -- begin*/
   
   if (cursor.par->table) {
      InsertCharInTable(c);
      return;
   }
/* table stuff -- end*/
   
   /* first check, if there will be two blanks together or a blank at 
    * the beginning of a paragraph. 
    * I decided to handle blanks like normal characters, the main 
    * difference are the special checks when calculating the row.fill
    * (blank does not count at the end of a row) and the check here */ 

   realtmpfont = real_current_font;
   tmpfont = current_font;	       /* store the current font.
					* This is because of the use
					* of cursor movements. The moving
					* cursor would refresh the 
					* current font */
   
   /* get the font that is used to calculate the baselineskip */ 
   reference_font1 = GetFont(cursor.par, cursor.par->Last() - 1);
   reference_font2 = cursor.par->GetFontSettings(cursor.par->Last() - 1);
   jumped_over_space = false;
   
   if (IsLineSeparatorChar(c)) {
	   
      if (cursor.pos < cursor.par->Last() &&
	       cursor.par->IsLineSeparator(cursor.pos))
 	{
	   /* the user inserted a space before a space. So we
	    * will just make a CursorRight. BUT: The font of this
	    * space should be set to current font. That is why
	    * we need to rebreak perhaps. If there is a protected
	    * blank at the end of a row we have to force a rebreak.*/ 
	   
	   if (cursor.pos == RowLast(cursor.row)
	       && !IsLineSeparatorChar(c))
	     cursor.row->fill = -1;    /* force rebreak  */
	   
	   cursor.par->Erase(cursor.pos);
	   jumped_over_space = true;
	   
	}
      else if ((cursor.pos > 0 && 
		     cursor.par->IsLineSeparator(cursor.pos - 1))
		    || (cursor.pos > 0 && cursor.par->IsNewline(cursor.pos - 1))
		    || (cursor.pos == 0 &&
			!(cursor.par->Previous()
			  && cursor.par->Previous()->footnoteflag
			  == LYX_OPEN_FOOTNOTE)))
      	return;
   }

   else if (IsNewlineChar(c)) {
     if (cursor.par->FirstPhysicalPar() == cursor.par
	 && cursor.pos <= BeginningOfMainBody(cursor.par))
       return;			       /* no newline at first position 
					* of a paragraph or behind labels. 
					* TeX does not allow that. */
      
      if (cursor.pos < cursor.par->Last() &&
	  cursor.par->IsLineSeparator(cursor.pos))
      	CursorRightIntern();		       /* newline always after a blank! */
      cursor.row->fill = -1;	       /* to force a new break  */
   }
   
   /* the display inset stuff */ 
   if (cursor.row->par->GetChar(cursor.row->pos) == LYX_META_INSET
       && cursor.row->par->GetInset(cursor.row->pos)
       && cursor.row->par->GetInset(cursor.row->pos)->Display())
     cursor.row->fill = -1; /* to force a new break */  

   /* get the cursor row fist */
   /* this is a dump solution, i will try to hold the cursor.row in future */ 
   /* row = GetRow(cursor.par, cursor.pos, y);*/
   /* ok, heres a better way: */ 
   row = cursor.row;
   y = cursor.y - row->baseline;
   if (c != LYX_META_INSET)	       /* in this case LyXText::InsertInset 
					* already insertet the character */
     cursor.par->InsertChar(cursor.pos, c);
   SetCharFont(cursor.par, cursor.pos, tmpfont);
   
   if (!jumped_over_space) {
      /* refresh the positions */
      tmprow = row;
      while (tmprow->next && tmprow->next->par == row->par) {
	 tmprow = tmprow->next;
	 tmprow->pos++;
      }
   }
   
   // auto correction 
   if (IsSeparatorChar(c) || IsKommaChar(c)){
     LyXParagraph* tmppar = cursor.par->ParFromPos(cursor.pos);
     int tmpposend = cursor.par->PositionInParFromPos(cursor.pos)-1;
     int tmppos = tmpposend;
     int old_tmpposend = tmpposend;
     int old_cursorpos = cursor.pos-1;
     // int i; //unused
     while (tmppos >  0 &&  
	    !tmppar->IsSeparator(tmppos-1) &&
	    !tmppar->IsKomma(tmppos-1)
	    ){
       tmppos--;
       old_cursorpos--;
     }

     if (AutoCorrect(tmppar, tmppos, tmpposend)){
       Row* tmprow = cursor.row;
       while (tmprow->next && tmprow->next->par == cursor.row->par) {
	 tmprow = tmprow->next;
	 tmprow->pos += (tmpposend - old_tmpposend);
       }
       cursor.pos = old_cursorpos + tmpposend - tmppos + 2;// for the character c inserted above
       CheckParagraph(cursor.par, old_cursorpos);
       current_font = tmpfont; real_current_font = realtmpfont;
       /* check, wether the last characters font has changed. */ 
       if (cursor.pos && cursor.pos == cursor.par->Last()
	   && !(FontEqual(reference_font1, tmpfont)
		|| FontEqual(reference_font2, tmpfont)))
	 RedoHeightOfParagraph(cursor);
       else {
	 /* now the special right address boxes */
	 if (lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
	   RedoDrawingOfParagraph(cursor); 
	 }
       }
       return;
     }
   }

   /* Is there a break one row above */ 
   if ((cursor.par->IsLineSeparator(cursor.pos)
 	|| cursor.par->IsNewline(cursor.pos)
 	|| cursor.row->fill == -1)&& 
       row->previous && row->previous->par == row->par) {
	  z = NextBreakPoint(row->previous, paperwidth);
	  if ( z >= row->pos) {
	     row->pos = z + 1;
	          	     
	     /* set the dimensions of the row above  */ 
	     row->previous->fill = Fill(row->previous, paperwidth);

	     SetHeightOfRow(row->previous);
	     
	     y -= row->previous->height;
	     refresh_y = y;
	     refresh_row = row->previous;
	     status = LYX_NEED_MORE_REFRESH;
	     
	     BreakAgainOneRow(row);
	     SetCursor(cursor.par, cursor.pos + 1);
	     /* cursor MUST be in row now */
	     
	     if (row->next && row->next->par == row->par)
	       need_break_row = row->next;
	     else
	       need_break_row = NULL;
	     current_font = tmpfont; real_current_font = realtmpfont;
	     
	     /* check, wether the last characters font has changed. */ 
	     if (cursor.pos && cursor.pos == cursor.par->Last()
		 && !(FontEqual(reference_font1, tmpfont)
		      || FontEqual(reference_font2, tmpfont)))
	       RedoHeightOfParagraph(cursor);
	     
   	     return;
	  }
       }
   
   /* recalculate the fill of the row */ 
   if (row->fill >= 0)		       /* needed because a newline
					* will set fill to -1. Otherwise
					* we would not get a rebreak! */
     row->fill = Fill(row, paperwidth);
   if (row->fill < 0 ) {
      refresh_y = y;
      refresh_row = row; 
      refresh_x = cursor.x;
      refresh_pos = cursor.pos;
      status = LYX_NEED_MORE_REFRESH;
      BreakAgainOneRow(row); 
      /* will the cursor be in another row now? */ 
      if (RowLast(row) <= cursor.pos + 1 && row->next) {
	 if (row->next && row->next->par == row->par)   /* this should
							 * always be true */
	   row = row->next;
	 BreakAgainOneRow(row);
      }
      SetCursor(cursor.par, cursor.pos + 1);
      if (row->next && row->next->par == row->par)
      	need_break_row = row->next;
      else
      	need_break_row = NULL;
      current_font = tmpfont; real_current_font = realtmpfont;
     
   }  
   else  {
      
      refresh_y = y;
      refresh_x = cursor.x;
      refresh_row = row;
      refresh_pos = cursor.pos;
      
      int tmpheight = row->height;
      SetHeightOfRow(row);
      if (tmpheight == row->height)
      	status = LYX_NEED_VERY_LITTLE_REFRESH;
      else
      	status = LYX_NEED_MORE_REFRESH;
            
      SetCursor(cursor.par, cursor.pos + 1);
      current_font = tmpfont; real_current_font = realtmpfont;
   }
   
   /* check, wether the last characters font has changed. */ 
   if (cursor.pos && cursor.pos == cursor.par->Last()
       && !(FontEqual(reference_font1, tmpfont)
	    || FontEqual(reference_font2, tmpfont)))
     RedoHeightOfParagraph(cursor);
   else {
      /* now the special right address boxes */
      if (lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
	 RedoDrawingOfParagraph(cursor); 
      }
   }
}
   
   
void LyXText::PrepareToPrint(Row *row, float &x, float &fill_separator, 
			     float &fill_hfill, float &fill_label_hfill)
     {
	float w, nh, nlh, ns;
	
	w = row->fill;
	fill_hfill = 0;
	fill_label_hfill = 0;
	fill_separator = 0;
	fill_label_hfill = 0;
	
	x = LeftMargin(row);
	
	/* is there a manual margin with a manual label */ 
	if (lyxstyle.Style(parameters->textclass, row->par->GetLayout())->margintype == MARGIN_MANUAL &&
	    lyxstyle.Style(parameters->textclass, row->par->GetLayout())->labeltype == LABEL_MANUAL) {
	       
	       nlh = NumberOfLabelHfills(row) + 1; /* one more since labels 
						    * are left aligned*/ 
	       if (nlh && row->par->GetLabelWidthString()) {
		  fill_label_hfill = LabelFill(row) / nlh;
	       }
	    }
		
	/* are there any hfills in the row? */ 
	nh = NumberOfHfills(row);
	
/* table stuff -- begin*/
	if (row->par->table) {
	   w = paperwidth - row->par->table->WidthOfTable()
	   - x - RightMargin(row);
	   nh = 0; /* ignore hfills in tables */ 
	}
/* table stuff -- end*/

	if (nh)
	  fill_hfill = w /nh;
	else  {
	   /* is it block, flushleft or flushright? 
	    * set x how you need it */
	   int align;
	   if (row->par->FirstPhysicalPar()->align == LYX_ALIGN_LAYOUT)
	     align = lyxstyle.Style(parameters->textclass, row->par->GetLayout())->align;
	   else
	     align = row->par->FirstPhysicalPar()->align;
	   
	   /* center displayed insets */ 
	   if (row->par->GetChar(row->pos) == LYX_META_INSET
	       && row->par->GetInset(row->pos)
	       && row->par->GetInset(row->pos)->Display())
	     align = LYX_ALIGN_CENTER;

	   switch (align) {
	    case LYX_ALIGN_BLOCK:
	      ns = NumberOfSeparators(row);
	      if (ns && row->next && row->next->par == row->par &&
		  !(row->next->par->IsNewline(row->next->pos-1))
		  && !(row->next->par->GetChar(row->next->pos) == LYX_META_INSET
		       && row->next->par->GetInset(row->next->pos)
		       && row->next->par->GetInset(row->next->pos)->Display())
		  )
	      	fill_separator = w / ns;
	      break;
	    case LYX_ALIGN_RIGHT:
	      x += w;
	      break;
	    case LYX_ALIGN_FORMULA:
	      if (row->previous && row->previous->par == row->par &&
		  !(row->par->IsNewline(row->pos-1)))
		{
		   /* avoid centering */ 
		}
	      else
	      	x += w / 2;
	      break;
	    case LYX_ALIGN_CENTER:
	      x += w / 2;
	      break;
	   }
	}
     }

      
/* important for the screen */


/* the cursor set functions have a special mechanism. When they
* realize, that you left an empty paragraph, they will delete it.
* They also delet the corresponding row */


void LyXText::CursorRightOneWord()
{
  LyXCursor tmpcursor = cursor;
   while (tmpcursor.pos < tmpcursor.par->Last()
	  && !tmpcursor.par->IsSeparator(tmpcursor.pos)
	  && !tmpcursor.par->IsKomma(tmpcursor.pos))
      tmpcursor.pos++;
     while ((tmpcursor.par->Last() > tmpcursor.pos
	     && (tmpcursor.par->IsSeparator(tmpcursor.pos)
		 || tmpcursor.par->IsKomma(tmpcursor.pos)))
	    || (tmpcursor.par->Last() == tmpcursor.pos
		&& tmpcursor.par->Next())){
       if (tmpcursor.pos == tmpcursor.par->Last()) {
	 tmpcursor.par = tmpcursor.par->Next();
	 tmpcursor.pos = 0;
       }
       else
	 tmpcursor.pos++;
     }
     
     SetCursor(tmpcursor.par, tmpcursor.pos);
}


void LyXText::CursorTab()
{
  LyXCursor tmpcursor = cursor;
   while (tmpcursor.pos < tmpcursor.par->Last()
	  && !tmpcursor.par->IsNewline(tmpcursor.pos))
     tmpcursor.pos++;
   
   if (tmpcursor.pos == tmpcursor.par->Last()){
     if (tmpcursor.par->Next()) {
       tmpcursor.par = tmpcursor.par->Next();
       tmpcursor.pos = 0;
     }
   }
   else
     tmpcursor.pos++;
   
     SetCursor(tmpcursor.par, tmpcursor.pos);
}


/* -------> Skip initial whitespace at end of word and move cursor to *start* of prior
			word, not to end of next prior word. */

void LyXText::CursorLeftOneWord() 
{
  LyXCursor tmpcursor = cursor;
	while (tmpcursor.pos && (tmpcursor.par->IsSeparator(tmpcursor.pos - 1) 
			 || tmpcursor.par->IsKomma(tmpcursor.pos - 1)))
      tmpcursor.pos--;

   if (!tmpcursor.pos) {
     if (tmpcursor.par->Previous()){
       tmpcursor.par = tmpcursor.par->Previous();
       tmpcursor.pos = tmpcursor.par->Last();
     }
     while (tmpcursor.pos && (tmpcursor.par->IsSeparator(tmpcursor.pos - 1) 
			    || tmpcursor.par->IsKomma(tmpcursor.pos - 1)))
       tmpcursor.pos--;
   }

   while (tmpcursor.pos > 0 
	  && !tmpcursor.par->IsSeparator(tmpcursor.pos-1)
	  && !tmpcursor.par->IsKomma(tmpcursor.pos-1))
     tmpcursor.pos--;

   SetCursor(tmpcursor.par, tmpcursor.pos);
}

/* -------> Select current word. This depeds on behaviour of CursorLeftOneWord(), so it is
			patched as well. */

void LyXText::SelectWord() 
{
	/* Move cursor to the beginning, when not already there. */
	if ( cursor.pos
	     && !cursor.par->IsSeparator(cursor.pos-1)
	     && !cursor.par->IsKomma(cursor.pos-1) )
		CursorLeftOneWord();

	/* set the sel cursor */
	sel_cursor = cursor;

	while ( cursor.pos < cursor.par->Last()
			&& !cursor.par->IsSeparator(cursor.pos)
			&& !cursor.par->IsKomma(cursor.pos) )
		cursor.pos++;
	SetCursor( cursor.par, cursor.pos );
	
	/* finally set the selection */ 
	SetSelection();
}


/* -------> Select the word currently under the cursor when:
			1: no selection is currently set,
			2: the cursor is not at the borders of the word. */

int LyXText::SelectWordWhenUnderCursor( void ) 
{
	if ( selection ) return 0;
	if ( cursor.pos < cursor.par->Last()
		 && !cursor.par->IsSeparator(cursor.pos)
		 && !cursor.par->IsKomma(cursor.pos)
		 && cursor.pos 
		 && !cursor.par->IsSeparator(cursor.pos -1)
		 && !cursor.par->IsKomma(cursor.pos -1) ) {
		SelectWord();
		return 1;
	}
	return 0;
}

// This function is only used by the spellchecker for NextWord().
// It doesn't handle LYX_ACCENTs and probably never will.
char* LyXText::SelectNextWord(float &value)
{
  LyXParagraph* tmppar = cursor.par;

  // If this is not the very first word, skip rest of current word because
  // we are probably in the middle of a word if there is text here.
  if (cursor.pos || cursor.par->previous) {
     while (cursor.pos < cursor.par->Last()
	  && cursor.par->IsLetter(cursor.pos))
       cursor.pos++;
  }
  // Now, skip until we have real text (will jump paragraphs)
  while ((cursor.par->Last() > cursor.pos
	  && !cursor.par->IsLetter(cursor.pos))
	 || (cursor.par->Last() == cursor.pos
	     && cursor.par->Next())){
     if (cursor.pos == cursor.par->Last()) {
       cursor.par = cursor.par->Next();
       cursor.pos = 0;
     }
     else
       cursor.pos++;
  }
  
  // Update the value if we changed paragraphs
  if (cursor.par != tmppar){
    SetCursor(cursor.par, cursor.pos);
    value = (float)(cursor.y)/height;
  }

  /* Start the selection from here */
  sel_cursor = cursor;
   
  /* and find the end of the word */
  while (cursor.pos < cursor.par->Last()
	 && cursor.par->IsLetter(cursor.pos))
    cursor.pos++;

  // Finally, we copy the word to a string and return it
  char* string = NULL;

  if (sel_cursor.pos < cursor.pos) {
    string = new char [cursor.pos - sel_cursor.pos + 2];
    int i,j;

    for (i=sel_cursor.pos, j=0; i<cursor.pos; i++)
      string[j++] = cursor.par->GetChar(i);
    string[j] = '\0';
  }
  return string;
}
   
// This one is also only for the spellchecker
void LyXText::SelectSelectedWord()
{
  /* move cursor to the beginning */
  SetCursor(sel_cursor.par, sel_cursor.pos);
  
  /* set the sel cursor */
  sel_cursor = cursor;
   
   /* now find the end of the word */
   while (cursor.pos < cursor.par->Last()
	  && cursor.par->IsLetter(cursor.pos))
     cursor.pos++;
   
   SetCursor(cursor.par, cursor.pos);

   /* finally set the selection */ 
   SetSelection();
   
}


/* -------> Delete from cursor up to the end of the current or next word. */
void LyXText::DeleteWordForward( void )
{
	LyXCursor tmpcursor = cursor;
        
	if (!cursor.par->Last())
		CursorRight();
	else {
		/* -------> Skip initial non-word stuff. */
		while ( cursor.pos < cursor.par->Last() 
			&& (cursor.par->IsSeparator(cursor.pos)
			    || cursor.par->IsKomma(cursor.pos)) )
			cursor.pos++;
		
		SetCursorIntern(cursor.par, cursor.pos);
		selection = True; // to avoid deletion 
		CursorRightOneWord();
		sel_cursor = cursor;
		cursor = tmpcursor;
		SetSelection(); 
		
		/* -----> Great, CutSelection() gets rid of multiple spaces. */
		CutSelection();
	}
}


/* -------> Delete from cursor to start of current or prior word. */
void LyXText::DeleteWordBackward( void )
{
       LyXCursor tmpcursor = cursor;
       if (!cursor.par->Last())
         CursorLeft();
       else{
         selection = True; // to avoid deletion 
         CursorLeftOneWord();
         sel_cursor = cursor;
         cursor = tmpcursor;
         SetSelection();
         CutSelection();
       }
}


/* -------> Kill to end of line. */
void LyXText::DeleteLineForward( void )
{
	LyXCursor tmpcursor = cursor;
	if (!cursor.par->Last())
		CursorRight();
	else{
		CursorEnd();
		sel_cursor = cursor;
		cursor = tmpcursor;
		SetSelection();
		if (selection == False) {
			DeleteWordForward();
		} else {
			CutSelection();
		}
	}
}


/* -------> Upcase characters from cursor to end of word. */
void LyXText::UpcaseWord(void) 
{
  SetUndo(LYX_UNDO_FINISH, 
	  cursor.par->ParFromPos(cursor.pos)->previous,
	  cursor.par->ParFromPos(cursor.pos)->next); 

	while (cursor.pos < cursor.par->Last()
	&& !cursor.par->IsSeparator(cursor.pos) 
	&& !cursor.par->IsKomma(cursor.pos) ) {
		 if (cursor.par->GetChar(cursor.pos) != LYX_META_INSET)
		    cursor.par->text[cursor.pos] = toupper(cursor.par->text[cursor.pos]);
		CursorRight();
	}
	CheckParagraph(cursor.par, cursor.pos);
	CursorRightOneWord();
}


/* -------> Lowcase characters from cursor to end of word. */
void LyXText::LowcaseWord(void) 
{
  SetUndo(LYX_UNDO_FINISH, 
	  cursor.par->ParFromPos(cursor.pos)->previous,
	  cursor.par->ParFromPos(cursor.pos)->next); 

	while ( cursor.pos < cursor.par->Last()
	&& !cursor.par->IsSeparator(cursor.pos) 
	&& !cursor.par->IsKomma(cursor.pos) ) {
		 if (cursor.par->GetChar(cursor.pos) != LYX_META_INSET)
		   cursor.par->text[cursor.pos] = tolower(cursor.par->text[cursor.pos]);
		CursorRight();
	}
	CheckParagraph(cursor.par, cursor.pos);
	CursorRightOneWord();
}


/* -------> Capitalize characters from cursor to end of word. */
void LyXText::CapitalizeWord(void) 
{
  SetUndo(LYX_UNDO_FINISH, 
	  cursor.par->ParFromPos(cursor.pos)->previous,
	  cursor.par->ParFromPos(cursor.pos)->next); 

	bool firstBo = true;
	while ( cursor.pos < cursor.par->Last()
	&& !cursor.par->IsSeparator(cursor.pos) 
	&& !cursor.par->IsKomma(cursor.pos) ) {
		if ( firstBo ){
			  if (cursor.par->GetChar(cursor.pos) != LYX_META_INSET)
				cursor.par->text[cursor.pos]
					= toupper(cursor.par->text[cursor.pos]);
		}
		else{
			  if (cursor.par->GetChar(cursor.pos) != LYX_META_INSET)
			    cursor.par->text[cursor.pos]
			      = tolower(cursor.par->text[cursor.pos]);
		 }
		firstBo = false;
		CursorRight();
	}
	CheckParagraph(cursor.par, cursor.pos);
	CursorRightOneWord();
}


void  LyXText::Delete()
{
   LyXCursor old_cursor = cursor;
   LyXCursor tmpcursor;
   /* this is a very easy implementation*/ 
   
   /* just move to the right */ 
   CursorRightIntern();
   
   if (cursor.par->previous == old_cursor.par->previous
       && cursor.par != old_cursor.par)
     return; // delete-emty-paragraph-mechanism has done it
   
   /* if you had success make a backspace */ 
   if (old_cursor.par != cursor.par || old_cursor.pos != cursor.pos) {
     tmpcursor = cursor;
     cursor = old_cursor; // to make sure undo gets the right cursor position
     SetUndo(LYX_UNDO_DELETE, 
	     cursor.par->ParFromPos(cursor.pos)->previous, 
	     cursor.par->ParFromPos(cursor.pos)->next); 
     cursor = tmpcursor;
     Backspace();
   }
}


void  LyXText::Backspace()
{
   LyXParagraph *tmppar;
   Row *tmprow, *row;
   long y;
   int tmpheight;
   LyXFont tmpfont, realtmpfont, reference_font1, reference_font2;

   /* table stuff -- begin*/
   
   if (cursor.par->table) {
      BackspaceInTable();
      return;
   }
   /* table stuff -- end*/

   tmpfont = current_font;
   realtmpfont = real_current_font;
   
   reference_font1 = GetFont(cursor.par, cursor.par->Last() - 1);
   reference_font2 = cursor.par->GetFontSettings(cursor.par->Last() - 1);
    
   if (cursor.pos == 0) {
      /* we may paste some paragraphs */
      
      /* is it an empty paragraph? */
      
      if ((cursor.par->Last() == 0
	   || (cursor.par->Last() == 1
	       && cursor.par->IsSeparator(0)))
	  && !(cursor.par->Next() 
               && cursor.par->footnoteflag == LYX_NO_FOOTNOTE
	       && cursor.par->Next()->footnoteflag == LYX_OPEN_FOOTNOTE)) {
		  
		  if (cursor.par->previous) {
		     tmppar = cursor.par->previous->FirstPhysicalPar();
		     if (cursor.par->GetLayout() == tmppar->GetLayout()
			 && cursor.par->footnoteflag == tmppar->footnoteflag
			 && cursor.par->GetAlign() == tmppar->GetAlign()) {
			    
			    tmppar->line_bottom = cursor.par->line_bottom;
			    tmppar->added_space_bottom = cursor.par->added_space_bottom;
			    tmppar->fill_bottom = cursor.par->fill_bottom;
			    tmppar->pagebreak_bottom = cursor.par->pagebreak_bottom;
			    
			 }
		     
		     CursorLeftIntern();
		     
		     /* the layout things can change the height of a row ! */ 
		     tmpheight = cursor.row->height;
		     SetHeightOfRow(cursor.row);
		     if (cursor.row->height != tmpheight) {
			refresh_y = cursor.y - cursor.row->baseline;
			refresh_row = cursor.row;
			status = LYX_NEED_MORE_REFRESH;
		     }
		     return;
		  }
	       }
      if (cursor.par->ParFromPos(cursor.pos)->previous){
	SetUndo(LYX_UNDO_DELETE, 
		cursor.par->ParFromPos(cursor.pos)->previous->previous, 
		cursor.par->ParFromPos(cursor.pos)->next); 
      }
      tmppar = cursor.par;
      tmprow = cursor.row;
      CursorLeftIntern();
       /* Pasting is not allowed, if the paragraphs have different layout.
	* I think it is a real bug of all other word processors to allow
	* it. It confuses the user. Even so with a footnote paragraph and
	* a non-footnote paragraph. I will not allow pasting in this case, 
	* because the user would be confused if the footnote behaves 
	* different wether it is open or closed.
	* 
	* Correction: Pasting is always allowed with standard-layout */
      if (cursor.par != tmppar
	  && (cursor.par->GetLayout() == tmppar->GetLayout()
	      || !tmppar->GetLayout())
	  && cursor.par->footnoteflag == tmppar->footnoteflag
	  /* table stuff -- begin*/
	  && !cursor.par->table /* no pasting of tables */ 
	  /* table stuff -- end*/
	  && cursor.par->GetAlign() == tmppar->GetAlign()) {
	     
	     cursor.par->PasteParagraph();
	     
	     if (!(cursor.pos &&
		   cursor.par->IsSeparator(cursor.pos - 1)))
	       cursor.par->InsertChar(cursor.pos, ' ');
	     else
	       if (cursor.pos)
	       cursor.pos--;
	     
	     status = LYX_NEED_MORE_REFRESH;
	     refresh_row = cursor.row;
	     refresh_y = cursor.y - cursor.row->baseline;
	     
	     /* remove the lost paragraph */
	     RemoveParagraph(tmprow);
	     RemoveRow(tmprow);  
	     
	     /* break the paragraph again */ 
	     /* BreakAgain(cursor.row); */ 
	     
	     AppendParagraph(cursor.row);
	     UpdateCounters(cursor.row);
	     
	     /* the row may have changed, block, hfills etc. */ 
	     SetCursor(cursor.par, cursor.pos);
	  }
   }
   else  {
      /* this is the code for a normal backspace, not pasting
       * any paragraphs */ 
     SetUndo(LYX_UNDO_DELETE, 
	     cursor.par->ParFromPos(cursor.pos)->previous, 
	     cursor.par->ParFromPos(cursor.pos)->next); 
      CursorLeftIntern();

      /* some insets are undeletable here */
      if (cursor.par->GetChar(cursor.pos)==LYX_META_INSET) {
	if (!cursor.par->GetInset(cursor.pos)->Deletable())
	  return; 
	/* force complete redo when erasing display insets */ 
	/* this is a cruel mathod but save..... Matthias */ 
	if (cursor.par->GetInset(cursor.pos)->Display()){
	  cursor.par->Erase(cursor.pos);
	  RedoParagraph();
	  return;
	}
      }
      
      row = cursor.row;
      y = cursor.y - row->baseline;
      int z;
     
      /* remember that a space at the end of a row doesnt count
       * when calculating the fill */ 
      if (cursor.pos < RowLast(row) ||
	  !cursor.par->IsLineSeparator(cursor.pos)) {
	     row->fill += SingleWidth(cursor.par, cursor.pos);
	  }
      
      /* some special code when deleting a newline. This is similar
       * to the behavior when pasting paragraphs */ 
      if (cursor.pos && cursor.par->IsNewline(cursor.pos)) {
	 cursor.par->Erase(cursor.pos);
	 /* refresh the positions */
	 tmprow = row;
	 while (tmprow->next && tmprow->next->par == row->par) {
	    tmprow = tmprow->next;
	    tmprow->pos--;
	 }
	 if (cursor.par->IsLineSeparator(cursor.pos - 1))
	    cursor.pos--;
	 
	 if (cursor.pos < cursor.par->Last() && !cursor.par->IsSeparator(cursor.pos)) {
	    cursor.par->InsertChar(cursor.pos, ' ');
	    /* refresh the positions */
	    tmprow = row;
	    while (tmprow->next && tmprow->next->par == row->par) {
	       tmprow = tmprow->next;
	       tmprow->pos++;
	    }
	 }
      }
      else {
	 cursor.par->Erase(cursor.pos);
	 
	 /* refresh the positions */
	 tmprow = row;
	 while (tmprow->next && tmprow->next->par == row->par) {
	    tmprow = tmprow->next;
	    tmprow->pos--;
	 }
	 
	 /* delete superfluous blanks */ 
	 if (cursor.pos < cursor.par->Last() - 1 &&
	     (cursor.par->IsLineSeparator(cursor.pos))) {

		if (cursor.pos == BeginningOfMainBody(cursor.par)
		    || !cursor.pos 
		    || cursor.par->IsLineSeparator(cursor.pos - 1)) {
			   cursor.par->Erase(cursor.pos);
			   /* refresh the positions */
			   tmprow = row;
			   while (tmprow->next && 
				  tmprow->next->par == row->par) {
				     tmprow = tmprow->next;
				     tmprow->pos--;
				  }
			   if (cursor.pos)   /* move one character left */
			     cursor.pos--;
			}
	     }
	 
	 /* delete newlines at the beginning of paragraphs */ 
	 while (cursor.par->Last() &&
		cursor.par->IsNewline(cursor.pos) &&
		cursor.pos == BeginningOfMainBody(cursor.par)
		) {
		   cursor.par->Erase(cursor.pos);
		   /* refresh the positions */
		   tmprow = row;
		   while (tmprow->next && 
			  tmprow->next->par == row->par) {
			     tmprow = tmprow->next;
			     tmprow->pos--;
			  }
		}
      }
			   
      /* is there a break one row above */ 
      if (row->previous && row->previous->par == row->par) {
	 z = NextBreakPoint(row->previous, paperwidth);
	 if ( z >= row->pos) {
	    row->pos = z + 1;
	    
	    tmprow = row->previous;
	    
	    /* maybe the current row is now empty */ 
	    if (row->pos >= row->par->Last()) {
	       /* remove it */ 
	       RemoveRow(row);
	       need_break_row = NULL;
	    }
	    else  {
	       BreakAgainOneRow(row);
	       if (row->next && row->next->par == row->par)
	       	 need_break_row = row->next;
	       else
	       	 need_break_row = NULL;
	    }
	    
	    /* set the dimensions of the row above  */ 
	    y -= tmprow->height;
	    tmprow->fill = Fill(tmprow, paperwidth);
	    SetHeightOfRow(tmprow);
	    	    
	    refresh_y = y;
	    refresh_row = tmprow;
	    status = LYX_NEED_MORE_REFRESH;
	    SetCursor(cursor.par, cursor.pos);
	    current_font = tmpfont; real_current_font = realtmpfont;
	    /* check, wether the last characters font has changed. */
	    tmpfont = GetFont(cursor.par, cursor.par->Last() - 1);
	    if (!(FontEqual(reference_font1, tmpfont)
		  || FontEqual(reference_font2, tmpfont)))
	      RedoHeightOfParagraph(cursor);
	    return;
	 }
      }
      
      /* break the cursor row again */ 
      z = NextBreakPoint(row, paperwidth);
      
      if ( z != RowLast(row) || 
	  (row->next && row->next->par == row->par &&
	   RowLast(row) == row->par->Last() - 1)){
	      
	      /* it can happen that a paragraph loses one row
	       * without a real breakup. This is when a word
	       * is to long to be broken. Well, I don t care this 
	       * hack ;-) */ 
	      if (row->next && row->next->par == row->par &&
		  RowLast(row) == row->par->Last() - 1)
	      	RemoveRow(row->next);
	      
	      refresh_y = y;
	      refresh_row = row;
	      status = LYX_NEED_MORE_REFRESH;
	      
	      BreakAgainOneRow(row);
	      
	      SetCursor(cursor.par, cursor.pos);
	      /* cursor MUST be in row now */
	      
	      if (row->next && row->next->par == row->par)
	      	need_break_row = row->next;
	      else
	      	need_break_row = NULL;
	   }
      else  {
	 /* set the dimensions of the row */ 
	 row->fill = Fill(row, paperwidth);
	 int tmpheight = row->height;
	 SetHeightOfRow(row);
	 if (tmpheight == row->height)
	   status = LYX_NEED_VERY_LITTLE_REFRESH;
	 else
	   status = LYX_NEED_MORE_REFRESH;
   	 refresh_y = y;
	 refresh_row = row;
	 SetCursor(cursor.par, cursor.pos);
      }
   }
   
   /* restore the current font 
    * That is what a user
    * expects! */
   current_font = tmpfont; real_current_font = realtmpfont;

   /* check, wether the last characters font has changed. */
   tmpfont = GetFont(cursor.par, cursor.par->Last() - 1);
   if (!(FontEqual(reference_font1, tmpfont)
	 || FontEqual(reference_font2, tmpfont)))
     RedoHeightOfParagraph(cursor);
   else {
      /* now the special right address boxes */
      if (lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
	 RedoDrawingOfParagraph(cursor); 
      }
   }
}
   
   
void LyXText::GetVisibleRow(unsigned long pixmap, int offset, 
			    Row *row_ptr, long y)
{
	/* returns a printed row */
   
	int pos, pos_end;
	float x, tmpx;
	int y_top, y_bottom;
	float fill_separator, fill_hfill, fill_label_hfill;
	LyXParagraph *par, *firstpar;
	int left_margin;
	LyXFont font;
	char* tmpstring;
	int maxdesc;
	if (row_ptr->height <= 0) {
		fprintf(stderr, "LYX_ERROR: row.height: %d \n",
			row_ptr->height);
		return;
	}
	left_margin = LabelEnd(row_ptr);
	PrepareToPrint(row_ptr, x, fill_separator,
		       fill_hfill, fill_label_hfill);

	int main_body = BeginningOfMainBody(row_ptr->par);
   
	/* initialize the pixmap */

	XFillRectangle(fl_display,pixmap,
		       LyXGetClearGC(),
		       0, offset, paperwidth, row_ptr->height);

	// check for NOT FAST SELECTION
	if (!fast_selection && !mono_video){
	if (selection) {
         
		/* selection code */ 
		if (sel_start_cursor.row == row_ptr && sel_end_cursor.row == row_ptr) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectionGC(),
				       sel_start_cursor.x, offset, 
				       sel_end_cursor.x - sel_start_cursor.x, row_ptr->height);
		}
		else if (sel_start_cursor.row == row_ptr) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectionGC(),
				       sel_start_cursor.x, offset, 
				       paperwidth - sel_start_cursor.x, row_ptr->height);
		}
		else if (sel_end_cursor.row == row_ptr) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectionGC(),
				       0, offset, 
				       sel_end_cursor.x, row_ptr->height);
		}
		else if (y > sel_start_cursor.y && y < sel_end_cursor.y) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectionGC(),
				       0, offset, paperwidth, row_ptr->height);
	 
		}
	}
	} // end of NOT FAST SELECTION  code
   
	if (row_ptr->par->footnoteflag == LYX_OPEN_FOOTNOTE) {
//       if (row_ptr->previous && 
// 	  row_ptr->previous->par->footnoteflag != LYX_OPEN_FOOTNOTE) {
// 	XDrawLine(fl_display,pixmap,LyXGetFootGC(), // use GetCopyGC for black lines
// 		  0, offset, paperwidth, offset);
// 	  }
		LyXFont font;// = {
		font.family = LYX_ROMAN_FAMILY;
		font.series = LYX_MEDIUM_SERIES;
		font.shape = LYX_UP_SHAPE;
		font.size = LYX_SIZE_SMALL; 
		font.latex = LYX_LATEX;
		font.bar = LYX_NO_BAR;//};
		
		int box_x = LYX_PAPER_MARGIN;
		box_x += LyXTextWidth(font, " margin ", 8);
		if (row_ptr->previous && 
		    row_ptr->previous->par->footnoteflag != LYX_OPEN_FOOTNOTE) {
			char* fs;
			int fsn;
			switch (row_ptr->par->footnotekind) {
			case LYX_MARGIN:
				fs = " margin\0";
				break;
			case LYX_FIG:
				fs = " fig\0";
				break;
			case LYX_TAB:
				fs = " tab\0";
				break;
			default:                      /* LYX_META_FOOTNOTE */
				fs = " foot\0";
				break;
			}
			for (fsn=0; fs[fsn]; fsn++); /* get length of string  */
 
			XFillRectangle(fl_display,
				       pixmap,
				       LyXGetLightedGC(),
				       LYX_PAPER_MARGIN, offset,
				       box_x - LYX_PAPER_MARGIN, 
				       (int)(LyXMaxAscent(font)*1.2)+(int)(LyXMaxDescent(font)*1.2));
			XFlush(fl_display);
 
//			XDrawLine(fl_display,pixmap,LyXGetRedGC(), // use GetCopyGC for black lines
			XDrawLine(fl_display,pixmap,LyXGetFootGC(), // use GetCopyGC for black lines
				  LYX_PAPER_MARGIN, offset, paperwidth - LYX_PAPER_MARGIN, offset);
			XFlush(fl_display);
        
			XDrawString(fl_display,
				    pixmap,
				    LyXGetGC(font),
				    LYX_PAPER_MARGIN, offset + (int)(LyXMaxAscent(font)*1.2),
				    fs, fsn);
			XFlush(fl_display);
			XDrawLine(fl_display,
				  pixmap,
				  //LyXGetRedGC(),
				  LyXGetFootGC(),
				  LYX_PAPER_MARGIN, offset,
				  LYX_PAPER_MARGIN, offset + (int)(LyXMaxAscent(font)*1.2)+(int)(LyXMaxDescent(font)*1.2));
			XFlush(fl_display);
			XDrawLine(fl_display,
				  pixmap,
				  //LyXGetRedGC(),
				  LyXGetFootGC(),
				  LYX_PAPER_MARGIN, offset + (int)(LyXMaxAscent(font)*1.2)+(int)(LyXMaxDescent(font)*1.2),
				  box_x, offset + (int)(LyXMaxAscent(font)*1.2)+(int)(LyXMaxDescent(font)*1.2));
			XFlush(fl_display);
		}
       
            
		/* draw the open floats in a red box */
		{
			XDrawLine(fl_display,
				  pixmap,
				  //LyXGetRedGC(),
				  LyXGetFootGC(),
				  box_x, offset,
				  box_x, offset + row_ptr->height);
			XFlush(fl_display);
			XDrawLine(fl_display,
				  pixmap,
				  //LyXGetRedGC(),
				  LyXGetFootGC(),
				  paperwidth - LYX_PAPER_MARGIN, offset,
				  paperwidth - LYX_PAPER_MARGIN, offset + row_ptr->height);
			XFlush(fl_display);
		}

	}
	else  {
//       if (row_ptr->previous && 
// 	  row_ptr->previous->par->footnoteflag == LYX_OPEN_FOOTNOTE) {
// 	XDrawLine(fl_display,pixmap,LyXGetFootGC(), // use GetCopyGC for black lines
// 		  0, offset, paperwidth, offset);
// 	  }
		if (row_ptr->previous && 
		    row_ptr->previous->par->footnoteflag == LYX_OPEN_FOOTNOTE) {
			LyXFont font;// = {
			font.family = LYX_ROMAN_FAMILY;
			font.series = LYX_MEDIUM_SERIES;
			font.shape = LYX_UP_SHAPE;
			font.size = LYX_SIZE_SMALL; 
			font.latex = LYX_LATEX;
			font.bar = LYX_NO_BAR;//};
			
			int box_x = LYX_PAPER_MARGIN;
			box_x += LyXTextWidth(font, " margin ", 8);
			//XDrawLine(fl_display,pixmap,LyXGetRedGC(), // use GetCopyGC for black lines
			XDrawLine(fl_display,pixmap,LyXGetFootGC(), // use GetCopyGC for black lines
				  box_x, offset, paperwidth - LYX_PAPER_MARGIN, offset);
			XFlush(fl_display);
		}
   
	}
   
	LyXLayout* layout = lyxstyle.Style(parameters->textclass,
					   row_ptr->par->GetLayout());
	firstpar = row_ptr->par->FirstPhysicalPar();
   
	y_top = 0;
	y_bottom = row_ptr->height;
   
	/* is it a first row? */ 
	if (row_ptr->pos == 0
	    && row_ptr->par == firstpar) {
	   
		/* think about the margins */ 
		if (!row_ptr->previous)
			y_top += LYX_PAPER_MARGIN;
       
		if (row_ptr->par->pagebreak_top){ /* draw a top pagebreak  */
			DrawOnOffLine(pixmap, 
				      offset + y_top + 2 * DefaultHeight(),
				      0, paperwidth);
			y_top += 3 * DefaultHeight();
		}
       
		if (row_ptr->par->fill_top) {    /* draw a vfill top  */
			DrawLine(pixmap, 
				 offset + 2 + y_top,
				 0, LYX_PAPER_MARGIN);
			DrawLine(pixmap, 
				 offset + y_top + 3 * DefaultHeight(),
				 0, LYX_PAPER_MARGIN);
			DrawVerticalOnOffLine(pixmap, LYX_PAPER_MARGIN / 2, 
					      offset + 2 + y_top,
					      offset + y_top + 3 * DefaultHeight()
				);
			y_top += 3 * DefaultHeight();
		}
    
		/* think about user added space */ 
		y_top += int (row_ptr->par->added_space_top * 37);
       
		/* think about the parskip */ 
		/* some parksips VERY EASY IMPLEMENTATION */ 
		if (parameters->paragraph_separation == LYX_PARSEP_SKIP) {
			if (layout->latextype == LATEX_PARAGRAPH
			    && firstpar->GetDepth() == 0
			    && firstpar->Previous())
				y_top += (int)(layout->parskip * DefaultHeight());
			else if (firstpar->Previous()
				 && lyxstyle.Style(parameters->textclass,
						   firstpar->Previous()->GetLayout())->latextype == LATEX_PARAGRAPH
				 && firstpar->Previous()->GetDepth() == 0)
				y_top += (int)(lyxstyle.Style(parameters->textclass, firstpar->Previous()->GetLayout())->parskip * DefaultHeight());
	  
		}
	      
		if (row_ptr->par->line_top) {      /* draw a top line  */
			y_top +=  LyXAscent(GetFont(row_ptr->par, 0),'x');
			DrawThickLine(pixmap,
				      offset + y_top,
				      0, paperwidth);
			y_top +=  LyXAscent(GetFont(row_ptr->par, 0),'x');
		}
       
		/* should we print a label? */ 
		if (layout->labeltype >= LABEL_STATIC) {
			font = GetFont(row_ptr->par, -2);
			if (row_ptr->par->GetLabelString()) {
				tmpx = x;
				tmpstring = row_ptr->par->GetLabelString();
	     
				if (layout->labeltype == LABEL_COUNTER_CHAPTER) {
					if (parameters->secnumdepth >=0){
						/* this is spacel code for the chapter layout. This is printed in
						 * an extra row and has a pagebreak at the top. */ 
						maxdesc = (int) (LyXMaxDescent(font) * layout->baselinestretch * parameters->baseline)
							+ (int) layout->parsep * DefaultHeight();
						LyXDrawString(font, tmpstring, 
							      pixmap, 
							      offset + row_ptr->baseline
							      - row_ptr->ascent_of_text - maxdesc,
							      (int) x);
					}
				}
				else {
					x -= LyXStringWidth(font, layout->labelsep);
					x -= LyXStringWidth(font, tmpstring);
					/* draw it! */
					LyXDrawString(font, tmpstring, 
						      pixmap, 
						      offset + row_ptr->baseline, 
						      (int) x);
				}
		 
				x = tmpx;
			}
		}
		/* the labels at the top of an environment. More or less for bibliography */ 
		else if (layout->labeltype == LABEL_TOP_ENVIRONMENT
			 || layout->labeltype == LABEL_CENTERED_TOP_ENVIRONMENT) {
			if (row_ptr->par == row_ptr->par->DepthHook(row_ptr->par->GetDepth())
			    || (row_ptr->par->DepthHook(row_ptr->par->GetDepth())->GetLayout() != row_ptr->par->GetLayout()
				|| row_ptr->par->DepthHook(row_ptr->par->GetDepth())->GetDepth() != row_ptr->par->GetDepth()
				    )
				){
				font = GetFont(row_ptr->par, -2);
				if (row_ptr->par->GetLabelString()) {
					tmpstring = row_ptr->par->GetLabelString();
		
					maxdesc = (int) (LyXMaxDescent(font) * layout->baselinestretch * parameters->baseline
							 + (layout->labelbottomsep * DefaultHeight()));
		
					int top_label_x = (int)x;
					if (layout->labeltype == LABEL_CENTERED_TOP_ENVIRONMENT){
						top_label_x = (int) (x + (paperwidth - RightMargin(row_ptr) - x) / 2); 
						top_label_x -= (LyXStringWidth(font, tmpstring)/2);
					}

					LyXDrawString(font, tmpstring,
						      pixmap,
						      offset + row_ptr->baseline
						      - row_ptr->ascent_of_text - maxdesc,  
						      top_label_x);
				}
			}
		}
	}
   
	/* is it a last row? */
	par = row_ptr->par->LastPhysicalPar();
	if (row_ptr->par->ParFromPos(RowLast(row_ptr) + 1) == par
	    && (!row_ptr->next || row_ptr->next->par != row_ptr->par)) {     

		/* think about the margins */ 
		if (!row_ptr->next)
			y_bottom -= LYX_PAPER_MARGIN;
      
      /* draw a bottom pagebreak */ 
		if (firstpar->pagebreak_bottom) {
			DrawOnOffLine(pixmap, 
				      offset + y_bottom - 2 * DefaultHeight(),
				      0, paperwidth);
			y_bottom -= 3 * DefaultHeight();
		}
      
		if (firstpar->fill_bottom) {    /* draw a vfill bottom  */
			DrawLine(pixmap, 
				 offset + y_bottom - 3 * DefaultHeight(),
				 0, LYX_PAPER_MARGIN);
			DrawLine(pixmap, offset + y_bottom - 2,
				 0, LYX_PAPER_MARGIN);
			DrawVerticalOnOffLine(pixmap, LYX_PAPER_MARGIN / 2, 
					      offset + y_bottom - 3 * DefaultHeight(),
					      offset + y_bottom - 2
				);
			y_bottom -= 3* DefaultHeight();
		}
      
		/* think about user added space */ 
		y_bottom -= int (firstpar->added_space_bottom * 37);
      
		if (firstpar->line_bottom) {
			/* draw a bottom line */
			y_bottom -= LyXAscent(GetFont(par, par->Last() - 1),'x');
			DrawThickLine(pixmap,
				      offset + y_bottom,
				      0, paperwidth);
			y_bottom -= LyXAscent(GetFont(par, par->Last() - 1),'x');
		}
	}
   
	/* draw the text in the pixmap */  
	pos_end = RowLast(row_ptr);
   
	pos = row_ptr->pos;
/* table stuff -- begin*/
	if (row_ptr->par->table) {
		char on_off;
		int cell = NumberOfCell(row_ptr->par, row_ptr->pos);
		float x_old = x;
		x += row_ptr->par->table->GetBeginningOfTextInCell(cell);
      
		while (pos <= pos_end)  {
			if (row_ptr->par->IsNewline(pos)) {

				x = x_old + row_ptr->par->table->WidthOfColumn(cell);

				/* draw the table lines, still very simple */
				on_off = !row_ptr->par->table->TopLine(cell);
				if (!on_off || !row_ptr->par->table->TopAlreadyDrawed(cell))
					DrawTableLine(pixmap,
						      offset + row_ptr->baseline - row_ptr->ascent_of_text,
						      (int)x_old, (int)(x - x_old), on_off);
				on_off = !row_ptr->par->table->BottomLine(cell);
				if (!on_off || row_ptr->par->table->VeryLastRow(cell))
					DrawTableLine(pixmap,
						      offset + y_bottom - 1,
						      (int)x_old, (int)(x - x_old), on_off);
				on_off = !row_ptr->par->table->LeftLine(cell);
				DrawVerticalTableLine(pixmap,
						      (int)x_old, 
						      offset + row_ptr->baseline - row_ptr->ascent_of_text,
						      offset + y_bottom - 1,
						      on_off
					);
				on_off = !row_ptr->par->table->RightLine(cell);
				DrawVerticalTableLine(pixmap,
						      (int)x - row_ptr->par->table->AdditionalWidth(cell),
						      offset + row_ptr->baseline - row_ptr->ascent_of_text,
						      offset + y_bottom - 1,
						      on_off
					);
				x_old = x;
				/* take care about the alignment and other spaces */
				cell++;
				x += row_ptr->par->table->GetBeginningOfTextInCell(cell);
				if (row_ptr->par->table->IsFirstCell(cell))
					cell--; // little hack, sorry
				pos++;
			}
	 
			else if (row_ptr->par->IsHfill(pos)) {
				x += 1;
				DrawVerticalLine(pixmap, (int)x,  
						 offset + row_ptr->baseline - DefaultHeight()/2, 
						 offset + row_ptr->baseline);
				x += 2;
				pos++;
			}
			else {
				if (row_ptr->par->IsSeparator(pos)) {
					tmpx = x;
					x+=SingleWidth(row_ptr->par, pos);
					/* -------> Only draw protected spaces when not in
					 * free-spacing mode. */
					if (row_ptr->par->GetChar(pos)==LYX_META_PROTECTED_SEPARATOR && !layout->free_spacing) {
						DrawVerticalLine(pixmap, (int) tmpx,
								 offset + row_ptr->baseline - 3,
								 offset + row_ptr->baseline - 1);
						DrawLine(pixmap, 
							 offset + row_ptr->baseline - 1,
							 (int)tmpx,
							 (int)(x-tmpx-2));
						DrawVerticalLine(pixmap, (int) (x-2),
								 offset + row_ptr->baseline - 3,
								 offset + row_ptr->baseline - 1);
						/* what about underbars? */
						font = GetFont(row_ptr->par, pos); 
						if (font.bar == LYX_UNDER_BAR
						    && font.latex == LYX_NO_LATEX){
							XDrawLine(fl_display,
								  pixmap,
								  LyXGetCopyGC(),
								  (int)tmpx, offset + row_ptr->baseline + 2,
								  (int)x, offset + row_ptr->baseline + 2);
						}
					}
					pos++;
				}
				else
					Draw(row_ptr, pos, pixmap, offset, x);
			}
		}
      
		/* do not forget the very last cell. This has no NEWLINE so 
		 * ignored by the code above*/ 
		if (cell == row_ptr->par->table->GetNumberOfCells()-1){
			x = x_old + row_ptr->par->table->WidthOfColumn(cell);
			on_off = !row_ptr->par->table->TopLine(cell);
			if (!on_off || !row_ptr->par->table->TopAlreadyDrawed(cell))
				DrawTableLine(pixmap,
					      offset + row_ptr->baseline - row_ptr->ascent_of_text,
					      (int)x_old, (int)(x - x_old), on_off);
			on_off = !row_ptr->par->table->BottomLine(cell);
			if (!on_off || row_ptr->par->table->VeryLastRow(cell))
				DrawTableLine(pixmap,
					      offset + y_bottom - 1,
					      (int)x_old, (int)(x - x_old), on_off);
			on_off = !row_ptr->par->table->LeftLine(cell);
			DrawVerticalTableLine(pixmap,
					      (int)x_old, 
					      offset + row_ptr->baseline - row_ptr->ascent_of_text,
					      offset + y_bottom - 1,
					      on_off
				);
			on_off = !row_ptr->par->table->RightLine(cell);
			DrawVerticalTableLine(pixmap,
					      (int)x - row_ptr->par->table->AdditionalWidth(cell),
					      offset + row_ptr->baseline - row_ptr->ascent_of_text,
					      offset + y_bottom - 1,
					      on_off
				);
		}
	}
	else
		/* table stuff -- end*/
   
		while (pos <= pos_end)  {

			if (row_ptr->par->IsHfill(pos)) {
				x += 1;
				DrawVerticalLine(pixmap, (int)x,  
						 offset + row_ptr->baseline - DefaultHeight()/2, 
						 offset + row_ptr->baseline);
				if (HfillExpansion(row_ptr,pos)) {
					if (pos >= main_body) {
						DrawOnOffLine(pixmap,
							      offset + row_ptr->baseline - DefaultHeight()/4,
							      (int)x,
							      (int)fill_hfill);
						x += fill_hfill;
					}
					else {
						DrawOnOffLine(pixmap,
							      offset + row_ptr->baseline - DefaultHeight()/4,
							      (int)x,
							      (int)fill_label_hfill);
						x += fill_label_hfill;
					}
					DrawVerticalLine(pixmap, (int)x,
							 offset + row_ptr->baseline - DefaultHeight()/2, 
							 offset + row_ptr->baseline);
				}
				x += 2;
				pos++;
			}
			else {
				if (row_ptr->par->IsSeparator(pos)) {
					tmpx = x;
					x+=SingleWidth(row_ptr->par, pos);
					if (pos >= main_body)
						x+= fill_separator;
					/* -------> Only draw protected spaces when not in
					 * free-spacing mode. */
					if (row_ptr->par->GetChar(pos)==LYX_META_PROTECTED_SEPARATOR && !layout->free_spacing) {
						DrawVerticalLine(pixmap, (int) tmpx,
								 offset + row_ptr->baseline - 3,
								 offset + row_ptr->baseline - 1);
						DrawLine(pixmap, 
							 offset + row_ptr->baseline - 1,
							 (int)tmpx,
							 (int)(x-tmpx-2));
						DrawVerticalLine(pixmap, (int) (x-2),
								 offset + row_ptr->baseline - 3,
								 offset + row_ptr->baseline - 1);
						/* what about underbars? */
						font = GetFont(row_ptr->par, pos); 
						if (font.bar == LYX_UNDER_BAR
						    && font.latex == LYX_NO_LATEX){
							XDrawLine(fl_display,
								  pixmap,
								  LyXGetCopyGC(),
								  (int)tmpx, offset + row_ptr->baseline + 2,
								  (int)x, offset + row_ptr->baseline + 2);
						}
					}
					pos++;
				}
				else
					Draw(row_ptr, pos, pixmap, offset, x);
			}
			if (pos == main_body) {
				x += LyXStringWidth(GetFont(row_ptr->par, -2),
						    layout->labelsep);
				if (row_ptr->par->IsLineSeparator(pos - 1))
					x-= SingleWidth(row_ptr->par, pos - 1);
				if (x < left_margin)
					x = left_margin;
			}
		}
   

	// check for FAST SELECTION
	if (fast_selection || mono_video){
	if (selection) {
         
		/* selection code */ 
		if (sel_start_cursor.row == row_ptr && sel_end_cursor.row == row_ptr) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectGC(),
				       sel_start_cursor.x, offset, 
				       sel_end_cursor.x - sel_start_cursor.x, row_ptr->height);
		}
		else if (sel_start_cursor.row == row_ptr) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectGC(),
				       sel_start_cursor.x, offset, 
				       paperwidth - sel_start_cursor.x, row_ptr->height);
		}
		else if (sel_end_cursor.row == row_ptr) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectGC(),
				       0, offset, 
				       sel_end_cursor.x, row_ptr->height);
		}
		else if (y > sel_start_cursor.y && y < sel_end_cursor.y) {
			XFillRectangle(fl_display,pixmap,
				       LyXGetSelectGC(),
				       0, offset, paperwidth, row_ptr->height);
	 
		}
	}
	} // end of FAST SELECTION code
	/* thats it */ 
}


int LyXText::DefaultHeight()
{
	static LyXFont font;// = {
	font.family = LYX_ROMAN_FAMILY;
	font.series = LYX_MEDIUM_SERIES;
	font.shape = LYX_UP_SHAPE;
	font.size = LYX_SIZE_NORMAL; 
	font.latex = LYX_NO_LATEX;
	font.bar = LYX_NO_BAR;//};
   
   return (int) (LyXMaxAscent(font)
		 + LyXMaxDescent(font) * 1.5);
}

   
/* returns the column near the specified x-coordinate of the row 
* x is set to the real beginning of this column  */ 
int  LyXText::GetColumnNearX(Row *row, int& x)
{
   int c; 
   float tmpx;
   float fill_separator, fill_hfill, fill_label_hfill;
   int left_margin;
   
   left_margin = LabelEnd(row);
   PrepareToPrint(row, tmpx, fill_separator, fill_hfill, fill_label_hfill);
   int main_body = BeginningOfMainBody(row->par);
   
   c = row->pos;

   int last = RowLast(row);
   if (row->par->IsNewline(last))
     last--;
   
   LyXLayout *layout = lyxstyle.Style(parameters->textclass, row->par->GetLayout());
   /* table stuff -- begin*/
   if (row->par->table) {
     if (!row->next || row->next->par != row->par)
       last = RowLast(row); /* the last row doesn't need a newline at the end*/
      int cell = NumberOfCell(row->par, row->pos);
      float x_old = tmpx;
      int ready = 0;
      tmpx += row->par->table->GetBeginningOfTextInCell(cell);
      while (c <= last
	     && tmpx + SingleWidth(row->par, c)  <= x
	     && !ready){
	 if (row->par->IsNewline(c)) {
	    if (x_old + row->par->table->WidthOfColumn(cell) <= x){
	       tmpx = x_old + row->par->table->WidthOfColumn(cell);
	       x_old = tmpx;
	       cell++;
	       tmpx += row->par->table->GetBeginningOfTextInCell(cell);
	       c++;
	    }
	    else
	      ready = 1;
	 }
	 else {
	    tmpx += SingleWidth(row->par, c);
	    c++;
	 }
      }
   }
   else
      /* table stuff -- end*/

   while (c <= last
	  && tmpx + SingleWidth(row->par, c)  <= x) {

	     if (c && c == main_body
		 && !row->par->IsLineSeparator(c - 1)) {
		    tmpx += LyXStringWidth(GetFont(row->par, -2),
					   layout->labelsep);
		    if (tmpx < left_margin)
		      tmpx = left_margin;
		 }
	     
	     tmpx += SingleWidth(row->par, c);
	     if (HfillExpansion(row, c)) {
		if (c >= main_body)
		  tmpx += fill_hfill;
		else
		  tmpx += fill_label_hfill;
	     }
	     else if (c >= main_body && row->par->IsSeparator(c)) {
		tmpx+= fill_separator;  
	     }
	     c++;
	     if (c == main_body
		 && row->par->IsLineSeparator(c - 1)) {
		    tmpx += LyXStringWidth(GetFont(row->par, -2),
					   layout->labelsep);
		    tmpx-= SingleWidth(row->par, c - 1);
		    if (tmpx < left_margin)
		      tmpx = left_margin;
		 }
	  }
   /* make shure that a last space in a row doesnt count */
   if (c >= last 
       && row->par->IsLineSeparator(c - 1)
       && !(!row->next || row->next->par != row->par)) {
	  tmpx -= SingleWidth(row->par, c - 1);
	  tmpx -= fill_separator;
       }
   c-=row->pos;
   
   x = (int)tmpx;
   return c;
}

   
/* turn the selection into a new environment. If there is no selection,
* create an empty environment */ 
void LyXText::InsertFootnoteEnvironment(char kind)
{
   /* no footnoteenvironment in a footnoteenvironment */ 
   if (cursor.par->footnoteflag != LYX_NO_FOOTNOTE) {
      WriteAlert("Impossible operation", "You can't insert a float in a float!", "sorry.");
     return;
   }
   
   /* this doesnt make sense, if there is no selection */ 
   bool dummy_selection = false;
   if (!selection) {
      sel_start_cursor = cursor;       /* dummy selection  */
      sel_end_cursor = cursor;
      dummy_selection = true;
   }
   
   LyXParagraph *tmppar;

   if (sel_start_cursor.par->table || sel_end_cursor.par->table){
      WriteAlert("Impossible operation", "Cannot cut table.", "sorry.");
      return;
   }
     
   if (sel_start_cursor.par->table || sel_end_cursor.par->table){
      WriteAlert("Impossible operation", "Cannot cut table.", "sorry.");
      return;
   }
     
   /* a test to make shure there is not already a footnote
    * in the selection. */
   
   tmppar = sel_start_cursor.par->ParFromPos(sel_start_cursor.pos);
   
   while (tmppar != sel_end_cursor.par->ParFromPos(sel_end_cursor.pos) && 
	  tmppar->footnoteflag == LYX_NO_FOOTNOTE)
     tmppar = tmppar->next;
   
   if (tmppar != sel_end_cursor.par->ParFromPos(sel_end_cursor.pos)
       || tmppar->footnoteflag != LYX_NO_FOOTNOTE) {
      WriteAlert("Impossible operation", "Float  would include float!", "sorry.");
      return;
   }
   
   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */

   SetUndo(LYX_UNDO_FINISH, 
	   sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
	   sel_end_cursor.par->ParFromPos(sel_end_cursor.pos)->next); 
   
   if (sel_end_cursor.pos > 0 
       && sel_end_cursor.par->IsLineSeparator(sel_end_cursor.pos - 1))
     sel_end_cursor.pos--;	       /* please break before a space at
					* the end */
   if (sel_start_cursor.par == sel_end_cursor.par
       && sel_start_cursor.pos > sel_end_cursor.pos)
     sel_start_cursor.pos--;

   sel_end_cursor.par->BreakParagraphConservative(sel_end_cursor.pos);
   
   sel_end_cursor.par = sel_end_cursor.par->Next();
   sel_end_cursor.pos = 0;
   
   // don't forget to insert a dummy layout paragraph if necessary
   if (sel_start_cursor.par->GetLayout() != sel_end_cursor.par->layout){
     sel_end_cursor.par->BreakParagraphConservative(0);
     sel_end_cursor.par->layout = LYX_DUMMY_LAYOUT;
     sel_end_cursor.par = sel_end_cursor.par->next;
   }
   else
     sel_end_cursor.par->layout = LYX_DUMMY_LAYOUT;

   cursor = sel_end_cursor;

   /* please break behind a space, if there is one. The space should
    * be erased too */ 
   if (sel_start_cursor.pos > 0 
       && sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos - 1))
     sel_start_cursor.pos--;
   if (sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos)) {
      sel_start_cursor.par->Erase(sel_start_cursor.pos);
   }
   
   sel_start_cursor.par->BreakParagraphConservative(sel_start_cursor.pos);
   tmppar = sel_start_cursor.par->Next();
   
   if (dummy_selection) {
     tmppar->Clear();
     if (kind == LYX_TAB
	 || kind == LYX_FIG)
       tmppar->SetLayout(lyxstyle.NumberOfLayout(parameters->textclass,"Caption"));
   }
   else {
     if (sel_start_cursor.pos > 0) {
       /* the footnote-environment should beginn with standard layout.
	* Imaging you insert a footnote withing a enumeration, you 
	* certainly do not want a enumerated footnote! */ 
       tmppar->Clear();
     }
     else {
       /* this is a exception the user would sometimes expect, I hope */
       sel_start_cursor.par->Clear();
     }
   }
   
   while (tmppar != sel_end_cursor.par) {
      tmppar->footnoteflag = LYX_OPEN_FOOTNOTE;
      tmppar->footnotekind = kind;
      tmppar = tmppar->Next();
   } 

   RedoParagraphs(sel_start_cursor, sel_end_cursor.par->Next());
   
   SetCursor(sel_start_cursor.par->Next(), 0);

   ClearSelection();
}
   

/* returns pointer to a specified row */
Row* LyXText::GetRow(LyXParagraph *par, int pos, long &y)
{
   Row* tmprow;

   if (currentrow){
     if (par == currentrow->par || par == currentrow->par->Previous()){
       while (currentrow->previous &&
	      currentrow->previous->par != par->previous){
	 currentrow = currentrow->previous;
	 currentrow_y -= currentrow->height;
       }
     }
     tmprow = currentrow;
     y = currentrow_y;
     /* find the first row of the specified paragraph */ 
     while (tmprow->next && (tmprow->par != par)) {
       y += tmprow->height;
       tmprow = tmprow->next;
     }

     if (tmprow->par == par){
       /* now find the wanted row */ 
       while (tmprow->pos < pos && tmprow->next && tmprow->next->par == par && 
	      tmprow->next->pos <= pos) {
	 y += tmprow->height;
       tmprow = tmprow->next;
       }
       currentrow = tmprow;
       currentrow_y = y;
       return tmprow;
     }
   }
   tmprow = firstrow;
   y = 0;
   /* find the first row of the specified paragraph */ 
   while (tmprow->next && (tmprow->par != par)) {
      y += tmprow->height;
      tmprow = tmprow->next;
   }
 
   /* now find the wanted row */ 
   while (tmprow->pos < pos && tmprow->next && tmprow->next->par == par && 
	  tmprow->next->pos <= pos) {
	     y += tmprow->height;
	     tmprow = tmprow->next;
   }
   
   currentrow = tmprow;
   currentrow_y = y;

   return tmprow;
}
