/*
 * picasm -- config.c
 *
 * handles configuration fuse setting with the CONFIG-directive
 *
 */

#include <stdio.h>
#include <string.h>

#include "picasm.h"

/*
 * Check Yes/No (or Disable(d)/Enable(d) or On/Off) strings for CONFIG
 *
 * returns: 0=no, 1=yes, -1=error
 *
 */
static int
config_yes_no(char *s)
{
  if(strcasecmp(s, "yes") == 0 || strcasecmp(s, "on") == 0 ||
     strncasecmp(s, "enable", 6) == 0)
    return 1;
  else if(strcasecmp(s, "no") == 0 || strcasecmp(s, "off") == 0 ||
	  strncasecmp(s, "disable", 7) == 0)
    return 0;
  else
    return -1; /* error */
}

/*
 * selstr consists of NUL-terminated strings with two NULs in the end
 * If str matches one of the components of selstr, the index of
 * that component (starting from zero) is returned,
 * if no match is found, -1 is returned.
 */
static int
strsel(char *selstr, char *str)
{
  int sel = 0;

  while(*selstr != '\0') {
    if(strcasecmp(selstr, str) == 0)
      return sel;

    selstr += strlen(selstr)+1;
    sel++;
  }

  return -1;
}

/*
 * Parse CONFIG-directive and set the value
 * of the configuration fuse register
 */
void
parse_config(void)
{
  static char symname[256];
  int t;

  for(;;) {
    if(token_type != TOK_IDENTIFIER) {
cfg_error:
      error(1, "CONFIG syntax error");
      return;
    }
    strcpy(symname, token_string);
    get_token();

    /* hmm... this is a little kludge, but as
       the tokenizer now makes 'local id's from
       the valid config strings, this must be used... */
    if(token_type != TOK_LOCAL_ID) {
      if(token_type != TOK_EQUAL) {
	error(1, "'=' expected");
	return;
      }
      get_token();
      if(token_type != TOK_IDENTIFIER)
	goto cfg_error;
    }

    if(instr_set == PIC12BIT) {
      switch(strsel("CP\0WDT\0OSC\0", symname)) {
        case 0: /* CP -- code protect */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0xff7) | (t ? 0 : 0x8);
	  break;

	case 1: /* WDT -- watchdog timer */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0xffb) | (t ? 4 : 0);
	  break;

	case 2: /* OSC -- oscillator select */
	  if((t = strsel("LP\0XT\0HS\0RC\0", token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0xffc) | t;
	  break;

	default:
	  goto cfg_error;
      }
    } else if(strcmp(pic_type->name, "14000") == 0) {
      /*
       * PIC14000 config flags are a special case,
       * as they are quite different from the other 14-bit PICs
       */
      switch(strsel("CPC\0CPU\0CPP\0PWRT\0WDT\0OSC\0", symname)) {
	case 0: /* CPC -- calibr. space code protection (bit #7, 0==enabled) */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = ((config_fuses & 0x3f7f) | (t ? 0 : 0x80));
	  break;

        case 1: /* CPU -- user space code protection (bit #5, 0==enabled) */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = ((config_fuses & 0x3fdf) | (t ? 0 : 0x20));
	  break;

	case 2: /* CPP -- program space code protection (bit #4, 0==enabled) */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = ((config_fuses & 0x3fef) | (t ? 0 : 0x10));
	  break;

	case 3: /* PWRT -- powerup timer (bit #3, 0==enabled) */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0x3ff7) | (t ? 0 : 8);
	  break;

	case 4: /* WDT -- watchdog timer (bit #2, 1==enabled) */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0x3ffb) | (t ? 4 : 0);
	  break;

	case 5: /* OSC -- oscillator select (bit #0, 0==HS, 1==IN) */
	  if((t = strsel("HS\0IN\0", token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0x3ffe) | t;
	  break;

	default:
	  goto cfg_error;
      }
      /* end of PIC14000 config handling */
    } else { /* normal 14bit PIC */
      switch(strsel("CP\0PWRT\0WDT\0BOD\0OSC\0", symname)) {
        case 0: /* CP -- code protect */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  if(pic_type->instr_flags & PIC_CP2) {
	    /* two code protect bits (16C64/65/74) */
	    /* (partial protection currently not supported) */
	    config_fuses = (config_fuses & 0x3fcf) | (t ? 0 : 0x30);
	  } else if(pic_type->instr_flags & PIC_CP3) {
	    /* replicated code-protect bits (PIC16C62x) */
	    /* (partial protection currently not supported) */
	    config_fuses = (config_fuses & 0x00cf) | (t ? 0 : 0x3f30);
	  } else {
	    /* one code protect bit (16C61/84/71) */
	    config_fuses = (config_fuses & 0x3fef) | (t ? 0 : 0x10);
	  }
	  break;

	case 1: /* PWRT -- powerup timer */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  if(pic_type->instr_flags & PIC_PWRT_INV)
	    t = !t;

	  config_fuses = (config_fuses & 0x3ff7) | (t ? 8 : 0);
	  break;

	case 2: /* WDT -- watchdog timer */
	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0x3ffb) | (t ? 4 : 0);
	  break;

	case 3: /* BOD -- brown-out detect */
	  if((pic_type->instr_flags & PIC_BOD) == 0)
	    goto cfg_error;

	  if((t = config_yes_no(token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0x3fbf) | (t ? 0x40 : 0);
	  break;

	case 4: /* OSC -- oscillator select */
	  if((t = strsel("LP\0XT\0HS\0RC\0", token_string)) < 0)
	    goto cfg_error;

	  config_fuses = (config_fuses & 0x3ffc) | t;
	  break;

	default:
	  goto cfg_error;
      }
    }

    get_token();
    if(token_type != TOK_COMMA)
      break;

    get_token();
  }
}
