/***************************************************************************
**    xIrcSocket.cpp  $Revision: 1.12 $ - $Name: V1-17B $ 
**    xSocket Class to parse IRC Messages
**
** Copyright (C) 1995, 1996  Joseph Croft <jcroft@unicomp.net>  
** 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
 ***************************************************************************/
#include <ctype.h>
#include "xIrcCommands.h"
#include "xIrcSocket.h"

static int dbg = 0;

extern xIrcCommands ircResponses;

xIrcSocket::xIrcSocket(xWidgetResInfo *pPRes, QObject *pParent, const char *pHost,
                       const char *pService, QObject *pSocketDialog, 
                       const char *pSlotStatus, const char *pSlotProgress, 
                       int &err) :
            xSocketTCP(pPRes, pParent, pHost, pService, pSocketDialog,
                       pSlotStatus, pSlotProgress, serr)
{
   if ((err = serr) == 0)
   {
      connect(this, SIGNAL(socketDataOut(const char *)), 
              this, SLOT(socketDataIn(const char *)));
      connect(this, SIGNAL(readFromSocket(char *)), 
              this, SLOT(rawSocketDataIn(char *)));
      connect(this, SIGNAL(sendToServer(const char *)), 
              this, SLOT(sendToSocket(const char *)));
   }
}

xIrcSocket::~xIrcSocket()
{
//   QObject::disconnect(this);
}

//
// This member function parses the Private messages is they
// are special, ei, ones that begin with a ^A. On exiting, if
// pRtn is not NULL, it will be set past the command found unless
// it is not recognized,  in which case pRtn will be set to the
// original value of cp.
//
// pMsg->pmsgType will be set to one of the values of the 
// enum xIrcPrvMsgType to indicate the type of private message
// found.
//
void xIrcSocket::parseMsgSpecial(xIrcMessage *pMsg, const char *pStr,
                     const char **pRtn)
{
   QString  tmpStr = "";
   const char *pSave, *pSave1;

   pSave = pStr;
   while (*pStr && !isspace(*pStr) && *pStr >= ' ')
      tmpStr += toupper(*(pStr++));

   if (strcmp(tmpStr, "PING") == 0)
      pMsg->pmsgTyp = ipmPing;
   else if (strcmp(tmpStr, "ACTION") == 0) 
      pMsg->pmsgTyp = ipmAction;
   else if (strcmp(tmpStr, "VERSION") == 0) 
      pMsg->pmsgTyp = ipmVersion;
   else if (strcmp(tmpStr, "USERINFO") == 0) 
      pMsg->pmsgTyp = ipmUserInfo;
   else if (strcmp(tmpStr, "CLIENTINFO") == 0) 
      pMsg->pmsgTyp = ipmClientInfo;
   else if (strcmp(tmpStr, "SOURCE") == 0) 
      pMsg->pmsgTyp = ipmSource;
   else if (strcmp(tmpStr, "FINGER") == 0) 
      pMsg->pmsgTyp = ipmFinger;
   else if (strcmp(tmpStr, "ERRMSG") == 0) 
      pMsg->pmsgTyp = ipmErrMsg;
   else if (strcmp(tmpStr, "TIME") == 0) 
      pMsg->pmsgTyp = ipmTime;
   else if (strcmp(tmpStr, "DCC") == 0)
   {
      pSave1 = pStr;
      while (*pStr && isspace(*pStr)) pStr++;
      tmpStr = "";
      while (*pStr && !isspace(*pStr))
         tmpStr += toupper(*(pStr++));
      if (strcmp(tmpStr, "SEND") == 0)
         pMsg->pmsgTyp = ipmDCCSend;
      else if (strcmp(tmpStr, "CHAT") == 0)
         pMsg->pmsgTyp = ipmDCCChat;
      else
      {
         pStr = pSave1;
         pMsg->pmsgTyp = ipmDCC;
      }
   }
   else
      pMsg->pmsgTyp = ipmUnknown;

   if (pMsg->pmsgTyp == ipmUnknown)
      *pRtn = pSave;
   else
      *pRtn = pStr;
}

//
// This member function parses the string pointed to by
// pText and emits a ircServerMessage signal passing the pointer
// to a xIrcMessage Structure to be handled by the higher levels.
//
void xIrcSocket::socketDataIn(const char *pText)
{
   xIrcMessage msg;
   QString strTmp;
   bool isNumber;
   const char *cp;
   int x;
   
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Enter\n");
   if (dbg) fflush(stdout);
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn():Have Raw message: |%s|\n",
                             pText);
   if (dbg) fflush(stdout);
   msg.srcNick = "";
   msg.srcAddr = "";
   msg.msgStr  = "";
   msg.rawMsg = pText;
   msg.rspCode = -1;
   //
   // Start by getting the nick, if present, then the address of the messages
   // source
   //
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Getting Nick\n");
   if (dbg) fflush(stdout);
   cp = pText;
   if (*pText != ':')
   {
      for (; *cp && *cp != '!' && *cp != ' '; cp++)
         strTmp += *(cp);
      msg.rspStr = strTmp;
   
      x = strTmp.toInt(&isNumber);
      if (isNumber)
      {
         if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Response was a code\n");
         if (dbg) fflush(stdout);
         msg.rspCode = x;
      }
      else
      {
         if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Response was a string\n");
         if (dbg) fflush(stdout);
         msg.rspCode = ircResponses.code(strTmp);
      }
      strTmp = "";
   }
   
   for (cp++; *cp && *cp != '!' && *cp != ' '; cp++)
      strTmp += *(cp);
   if (*cp == '!')
   {
      if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Getting Address\n");
      if (dbg) fflush(stdout);
      msg.srcNick = strTmp;
      strTmp = "";
      for (cp = cp+1; *cp && *cp != ' '; cp++)
         strTmp += *(cp);
   }      
   msg.srcAddr = strTmp;
   strTmp = "";
   
   //
   // Next either a numeric or text response will follow
   // Gather it, if its numeric, get its value and stuff it in msg.rspCode
   // Otherwise, get its text
   //
   if (msg.rspCode < 0)
   {
      if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Getting Response\n");
      if (dbg) fflush(stdout);
      for (cp++; *cp && *cp != ' '; cp++)
         strTmp += *(cp);
      msg.rspStr = strTmp;
      
      x = strTmp.toInt(&isNumber);
      if (isNumber)
      {
         if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Response was a code\n");
         if (dbg) fflush(stdout);
         msg.rspCode = x;
      }
      else
      {
         if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Response was a string\n");
         if (dbg) fflush(stdout);
         msg.rspCode = ircResponses.code(strTmp);
      }
      strTmp = "";
   }
   
   //
   // Now get the destination for the message, it should either be
   // a nick, or a channel name
   //
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Getting Destination\n");
   if (dbg) fflush(stdout);
   if (msg.rspCode >= 300 && msg.rspCode <= 369)
   {
      //
      // If this is a response in the range of 300 - 369, assume that the
      // Destination is the first word of the message and skip the destination
      // specified in the destination field
      //
      for (cp++; *cp && *cp != ':' && *cp != ' '; cp++);
      while (isspace(*cp) || *cp == ':' || *cp == '=' || *cp == '*') cp++;
      for (; *cp && *cp != ':' && *cp != ' '; cp++)
      {
         if (*cp != '\r' && *cp != '\n')
            strTmp += *(cp);
      }
      msg.dstStr = strTmp;
   }
   else
   {
      //
      // Otherwise, just use the destination as specified
      //
      while (*cp == ' ' || *cp == ':' || *cp == '=') cp++;
      for (; *cp && *cp != ':' && *cp != ' '; cp++)
      {
         if (*cp != '\r' && *cp != '\n')
            strTmp += *(cp);
      }
      msg.dstStr = strTmp;
   }
   strTmp = "";
   
   //
   // Just gather the remainder of the line and assume that it is the
   // message. This way it can be parsed as needed by the higher levels.
   // In any case, drop and '\r' encountered
   //
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Getting Message\n");
   if (dbg) fflush(stdout);
   while (isspace(*cp)) cp++;
   if (*cp == ':')
      while (isspace(*(++cp)));
   
   /*
   ** If this is a special private message, go parse it
   ** and set the pointer past the command. If its not a special
   ** private message, or even a private message for that matter,
   ** the pmsgTyp field of pMsg will be set to ipmMessage.
   */
   if ((ircResponses.is(msg.rspCode, "PRIVMSG")  ||
       ircResponses.is(msg.rspCode, "NOTICE")) &&
       *cp == 0x01)
      parseMsgSpecial(&msg, ++cp, &cp);
   else 
      msg.pmsgTyp = ipmMessage;

   /*
   ** Strip away any characters we don't want the calling code to have
   ** to mess with!
   */
   for (; *cp; cp++)
   {
      if (*cp != '\r' && *cp != 0x01)
         strTmp += *cp;
   }

   /*
   ** Set the message and go send it to the receiver!
   */
   msg.msgStr = strTmp;
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Sending data on via ircServerMessage\n");
   if (dbg) fflush(stdout);
   emit ircServerMessage(&msg);
   if (dbg) fprintf(stdout, "xIrcSocket::socketDataIn(): Exit\n");
   if (dbg) fflush(stdout);
}

void xIrcSocket::rawSocketDataIn(char *pText)
{
   int crFlag;

   if (dbg > 3) fprintf(stdout, "xIrcSocket::rawSocketDataIn(): Enter\n");
   if (dbg > 3) fflush(stdout);
   for (crFlag = 0; *pText; pText++)
   {
      if (crFlag)
      {
         if (*pText != '\n')
         {
            socketBuffer += '\n';
            emit socketDataOut(socketBuffer);
            socketBuffer = "";
            socketBuffer += *pText;
         }
         else
         {
            socketBuffer += *pText;
            emit socketDataOut(socketBuffer);
            socketBuffer = "";
         }
         if (*pText != '\r')
            crFlag = 0;
      }
      else
      {
         if (*pText == '\r')
            crFlag = 1;
         else
            socketBuffer += *pText;
         if (*pText == '\n')
         {
            emit socketDataOut(socketBuffer);
            socketBuffer = "";
         }
      }
   }
   if (dbg > 3) fprintf(stdout, "xIrcSocket::rawSocketDataIn(): Exit\n");
   if (dbg > 3) fflush(stdout);
}

void xIrcSocket::sendIrcServerMessage(xIrcMessage *pMsg)
{
   QString  msgStr = "";
   
   if (dbg > 2) fprintf(stdout, "xIrcSocket::sendIrcServerMessage(): Sending message\n");
   if (dbg > 2) fprintf(stdout, "xIrcSocket::sendIrcServerMessage(): Response code = %d\n", pMsg->rspCode);
   if (dbg > 2) fprintf(stdout, "xIrcSocket::sendIrcServerMessage(): Destination =  |%s|\n",
                                (const char *)pMsg->dstStr);
   if (dbg > 2) fflush(stdout);
   
   msgStr = ircResponses.name(pMsg->rspCode);
   if (strlen(msgStr) == 0)
   {
      msgStr.setNum(pMsg->rspCode);
   }
   msgStr += ' ';
   if (strlen((const char *)pMsg->dstStr) > 0)
   {
      msgStr += pMsg->dstStr;
      if (strlen((const char *)pMsg->msgStr) > 0)
         msgStr += " :";
   }
   if (strlen((const char *)pMsg->msgStr) > 0)
      msgStr += pMsg->msgStr;
   msgStr += '\n';
   if (dbg > 2) fprintf(stdout, "xIrcSocket::sendIrcServerMessage(): Sending message %s\n", (const char *)msgStr);
   if (dbg > 2) fflush(stdout);
   emit sendToServer((const char *)msgStr);
   if (dbg > 2) fprintf(stdout, "xIrcSocket::sendIrcServerMessage(): Message sent\n");
   if (dbg > 2) fflush(stdout);
}

void xIrcSocket::socketError(int err)
{
   err = 0;
}

#include "xIrcSocket.moc"
