//  
//  Copyright (c) 1994 by Mike Romberg (  romberg@md.fsl.noaa.gov )
//
//  This file may be distributed under terms of the GPL
//

#include "xosview.h"
#include "cpumeter.h"
#include "memmeter.h"
#include "swapmeter.h"
#include "netinmeter.h"
#include "netoutmeter.h"
#include "intmeter.h"
#include "serialmeter.h"
#include <iostream.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>


static const char NAME[] = "xosview@";


XOSView::XOSView( int argc, char *argv[] ) : XWin( argc, argv, NULL ){
  xoff_ = 5;
  yoff_ = 0;
  nummeters_ = 0;
  meters_ = NULL;
  name_ = "xosview";

  //  set up the X events
  addEvent( new Event( this, ConfigureNotify, 
		      (EventCallBack)&XOSView::resizeEvent ) );
  addEvent( new Event( this, Expose, 
		      (EventCallBack)&XOSView::exposeEvent ) );
  addEvent( new Event( this, KeyPress, 
		      (EventCallBack)&XOSView::keyPressEvent ) );

  // add or change the Resources
  addResource( new Parm( "labels", "-l", "False", 1 ) );
  addResource( new Parm( "usedlabels", "-ul", "False", 1 ) );
  addResource( new Parm( "network", "-net", "0" ) );
  addResource( new Parm( "term", "-term", "0" ) );
  addResource( new Parm( "interrupts", "-int", "False", 1 ) );
  addResource( new Parm( "serialone", "-cua0", "False", 1 ) );
  addResource( new Parm( "serialtwo", "-cua1", "False", 1 ) );
  addResource( new Parm( "serialthree", "-cua2", "False", 1 ) );
  addResource( new Parm( "serialfour", "-cua3", "False", 1 ) );
  changeResource( "background", "navy" );
  changeResource( "foreground", "cyan" );
  changeResource( "font", "7x13bold" );

  // see if legends are to be used
  parseargs( argc, argv );
  if ( legend_ ){
    if ( !usedlabels_ )
      xoff_ = textWidth( "XXXXX" );
    else
      xoff_ = textWidth( "XXXXXXXXX" );

    yoff_ = textHeight() + textHeight() / 4;
  }
  
  // add in the standard meters
  addmeter(new CPUMeter( this, xoff_, newypos(), width_ - 10, 10 ) );
  addmeter(new MemMeter( this, xoff_, newypos(), width_ - 10, 10 ) );
  addmeter(new SwapMeter( this, xoff_, newypos(), width_ - 10, 10 ) );
  
  // add in the net meters
  if ( netmax_ > 0 ){
    addmeter( new NetInMeter( this, xoff_, newypos(),
			      width_ - 10, 10, netmax_ ) );
    addmeter( new NetOutMeter( this, xoff_, newypos(),
			       width_ - 10, 10, netmax_ ) );
  }

  // add in the interrupts meter
  if ( ints_ )
    addmeter( new IntMeter( this, xoff_, newypos(), width_ - 10, 10 ) );

  // add in the serial meters
  if ( serial_[0] )
    addmeter( new SerialMeter( this, xoff_, newypos(), width_ - 10, 10,
			       "/dev/cua0", "cua0" ) );
  if ( serial_[1] )
    addmeter( new SerialMeter( this, xoff_, newypos(), width_ - 10, 10,
			       "/dev/cua1", "cua1" ) );
  if ( serial_[2] )
    addmeter( new SerialMeter( this, xoff_, newypos(), width_ - 10, 10,
			       "/dev/cua2", "cua2" ) );
  if ( serial_[3] )
    addmeter( new SerialMeter( this, xoff_, newypos(), width_ - 10, 10,
			       "/dev/cua3", "cua3" ) );

//  // add in the term meters
//  if ( doterm_ ){
//    addmeter( new TermInMeter( this, xoff_, newypos(),
//			       width_ - 10, 10, 100 ) );
//    addmeter( new TermOutMeter( this, xoff_, newypos(),
//				width_ - 10, 10, 100 ) );
//  }

  // check for new resources that the meters may have added
  parseargs( argc, argv );
//  dumpResources( cerr );
  checkMeterResources();
//  dumpResources( cerr );

  // determine the width and height of the window then create it
  width_ = findx();
  height_ = findy();
  init( argc, argv );
  title( winname() );
  iconname( winname() );
  dolegends();
  resize();
}

void XOSView::checkMeterResources( void ){
  MeterNode *tmp = meters_;

  while ( tmp != NULL ){
    tmp->meter_->checkResources();
    tmp = tmp->next_;
  }
}

int XOSView::newypos( void ){
  return 15 + 25 * nummeters_;
}

void XOSView::dolegends( void ){
  MeterNode *tmp = meters_;
  while ( tmp != NULL ){
    tmp->meter_->dolegends( legend_ );
    tmp->meter_->dousedlegends( usedlabels_ );
    tmp = tmp->next_;
  }
}

void XOSView::addmeter( Meter *fm ){
  MeterNode *tmp = meters_;

  if ( meters_ == NULL )
    meters_ = new MeterNode( fm );
  else {
    while ( tmp->next_ != NULL )
      tmp = tmp->next_;
    tmp->next_ = new MeterNode( fm );
  }
  nummeters_++;
}

int XOSView::findx( void ){
  if ( legend_ ){
    if ( !usedlabels_ )
      return textWidth( "XXXXXXXXXXXXXXXXXXXXXXXX" );
    else
      return textWidth( "XXXXXXXXXXXXXXXXXXXXXXXXXXXXX" );
  }
  return 80;
}

int XOSView::findy( void ){
  if ( legend_ )
    return 10 + textHeight() * nummeters_ * 2;

  return 15 * nummeters_;
}

void XOSView::parseargs( int argc, char *argv[] ) {
  netmax_ = usedlabels_ = legend_ = doterm_ = ints_ = 0;

  for ( int i = 0 ; i < 4 ; i++ )
    serial_[i] = 0;

  getResources();
  parseArgs( argc, argv );
  setFont();
  
  // use labels
  if ( !strcmp( getResource( "labels" ), "True" ) )
      legend_ = 1;

  // use "free" labels
  if ( !strcmp( getResource( "usedlabels" ), "True" ) )
    usedlabels_ = 1;

  // use "interrupts" meter
  if ( !strcmp( getResource( "interrupts" ), "True" ) )
    ints_ = 1;

  // use "serial" meters
  if ( !strcmp( getResource( "serialone" ), "True" ) )
    serial_[0] = 1;
  if ( !strcmp( getResource( "serialtwo" ), "True" ) )
    serial_[1] = 1;
  if ( !strcmp( getResource( "serialthree" ), "True" ) )
    serial_[2] = 1;
  if ( !strcmp( getResource( "serialfour" ), "True" ) )
    serial_[3] = 1;


  // use net meters
  if ( strcmp( getResource( "network" ), "0" ) )
    netmax_ = atof( getResource( "network" ) );

  // use term meters
  if ( strcmp( getResource( "term" ), "0" ) )
      doterm_ = 1;
}

const char *XOSView::winname( void ){
  static char name[100];
  char host[100];
  strcpy( name, NAME );
  gethostname( host, 99 );
  strcat( name, host );
  return name;
}

void  XOSView::resize( void ){
  int newwidth = width_ - xoff_ - 5;
  int newheight = (height_ - (10 + 5 * (nummeters_ - 1) + 
			      nummeters_ * yoff_)) / nummeters_;

  int counter = 1;
  MeterNode *tmp = meters_;
  while ( tmp != NULL ) {
    tmp->meter_->resize( xoff_, 5 * counter + counter * yoff_ +
			 (counter - 1) * newheight, newwidth, newheight );
    tmp = tmp->next_;
    counter++;
  }
}

XOSView::~XOSView( void ){
  MeterNode *tmp = meters_;
  while ( tmp != NULL ){
    MeterNode *save = tmp->next_;
    delete tmp->meter_;
    delete tmp;
    tmp = save;
  }
}

void XOSView::draw( void ){
  clear();
  MeterNode *tmp = meters_;
  while ( tmp != NULL ){
    tmp->meter_->draw();
    tmp = tmp->next_;
  }
}

void XOSView::keyrelease( char *ch ){
  if ( (*ch == 'q') || (*ch == 'Q') )
    done_ = 1;
}

void XOSView::run( void ){
  int counter = 0;

  while( !done_ ){
    checkevent();

    MeterNode *tmp = meters_;
    while ( tmp != NULL ){
      if ( tmp->meter_->requestevent() )
	tmp->meter_->checkevent();
      tmp = tmp->next_;
    }

    flush();
    usleep( 100000 );
    counter = (counter + 1) % 5;    
  }
}

void XOSView::usleep( unsigned long usec ){
  struct timeval time;

  time.tv_sec = usec / 1000000;
  time.tv_usec = usec - time.tv_sec * 1000000;

  select( 0, 0, 0, 0, &time );
}

void XOSView::keyPressEvent( XKeyEvent &event ){
  char c;
  KeySym key;
  XComposeStatus cs;

  XLookupString( &event, &c, 1, &key, &cs );

  if ( (c == 'q') || (c == 'Q') )
    done_ = 1;
}

