/*---------------------------------------------------------------------*/
/*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \    /  '                              */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome Send them to                                           */
/*        <Manuel.Serrano@inria.fr>                                    */
/*        Manuel Serrano                                               */
/*        INRIA -- Rocquencourt                                        */
/*        Domaine de Voluceau, BP 105                                  */
/*        78153 Le Chesnay Cedex                                       */
/*        France                                                       */
/*---------------------------------------------------------------------*/


/*---------------------------------------------------------------------*/
/*    serrano/prgm/project/bigloo/runtime1.9/Clib/writer.c             */
/*                                                                     */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Tue Dec 17 09:44:20 1991                          */
/*    Last change :  Fri Mar 29 10:42:09 1996 (serrano)                */
/*                                                                     */
/*    On imprime les objets (non recursifs)                            */
/*---------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    Les recuperations externes                                       */
/*---------------------------------------------------------------------*/
extern obj_t c_constant_string_to_string();
extern obj_t write_object( obj_t, obj_t );

/*---------------------------------------------------------------------*/
/*    Les noms des caracateres                                         */
/*---------------------------------------------------------------------*/
static char *char_name[] =
{
   "","","","","","","","",
   "",  "tab", "newline", "", "", "return", "", "",
   "", "","","","","","","",
   "", "", "","","", "", "", "",
   "space", "!", "\"","#","$","%","&","'",
   "(", ")", "*", "+", ",", "-", ".", "/",
   "0", "1", "2", "3", "4", "5", "6", "7",
   "8", "9", ":", ";", "<", "=", ">", "?",
   "@", "A", "B", "C", "D", "E", "F", "G",
   "H", "I", "J", "K", "L", "M", "N", "O",
   "P", "Q", "R", "S", "T", "U", "V", "W",
   "X", "Y", "Z", "[", "\\", "]", "^", "_",
   "`", "a", "b", "c", "d", "e", "f", "g",
   "h", "i", "j", "k", "l", "m", "n", "o",
   "p", "q", "r", "s", "t", "u", "v", "w",
   "x", "y", "z", "{", "|", "}", "~", ""
};


/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strputc ...                                                      */
/*---------------------------------------------------------------------*/
obj_t
strputc( c, p )
char  c;
obj_t p;
{
   long offset;
   
   if( END_OF_STRING_PORTP( p ) )
      strport_grow( p );

   offset = OUTPUT_STRING_PORT( p ).offset;

   OUTPUT_STRING_PORT( p ).buffer[ offset ] = c;
   OUTPUT_STRING_PORT( p ).offset = offset + 1;

   return p;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    lstrputs ...                                                     */
/*---------------------------------------------------------------------*/
static obj_t
lstrputs( s, p, len )
char *s;
obj_t p;
long  len;
{
   long offset;
   
   offset = OUTPUT_STRING_PORT( p ).offset;

   while((OUTPUT_STRING_PORT( p ).offset+len) > OUTPUT_STRING_PORT( p ).size)
      strport_grow( p );

   memcpy( &(OUTPUT_STRING_PORT(p).buffer[offset] ), s, len);

   OUTPUT_STRING_PORT( p ).offset = offset + len;

   return p;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strputs ...                                                      */
/*---------------------------------------------------------------------*/
obj_t
strputs( s, p )
char *s;
obj_t p;
{
   return lstrputs( s, p, strlen( s ) );
}
   
/*---------------------------------------------------------------------*/
/*    We catch the `escape_char_found' variable from Clib/string.c     */
/*---------------------------------------------------------------------*/
extern int escape_char_found;

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_string ...                                               */
/*---------------------------------------------------------------------*/
obj_t
display_string( o, port )
obj_t o, port;
{
   if( OUTPUT_STRING_PORTP( port ) )
      lstrputs( BSTRING_TO_STRING( o ),
	        port,
	        STRING_LENGTH( o ) );
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
      int len = STRING_LENGTH( o );
      char *aux = &STRING_REF( o, 0 );
      
      while( len-- )
         fputc( *aux++, fout );
   }
   
   return o;
}

/*---------------------------------------------------------------------*/
/*    write_string ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
write_string( string, port )
obj_t string, port;
{
   char *aux = BSTRING_TO_STRING( string );
   long len  = STRING_LENGTH( string );
   
   if( OUTPUT_STRING_PORTP( port ) )
   {
      if( escape_char_found )
         strputc( '#', port );

      strputc( '"', port );
      lstrputs( aux, port,len );
      strputc( '"', port );
   }
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;

      if( escape_char_found )
         fputc( '#', fout );
   
      fputc( '"', fout );
      fwrite( aux, 1, len, fout );
      fputc( '"', fout );
      return string;
   }

   return string;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_fixnum ...                                               */
/*---------------------------------------------------------------------*/
obj_t
display_fixnum( o, port )
obj_t o, port;
{
   if( OUTPUT_STRING_PORTP( port ) )
   {
      char new[ 100 ];
      
      sprintf( new, "%ld", CINT( o ) );
      strputs( new, port );
   }
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
      
      fprintf( fout, "%ld", CINT( o ) );
   }
   
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_flonum ...                                               */
/*    -------------------------------------------------------------    */
/*    Many thanks to Raj Manandhar <raj@droid.msfc.nasa.gov> for       */
/*    providing this code.                                             */
/*---------------------------------------------------------------------*/
obj_t
display_flonum( o, port )
obj_t o, port;
{
#if( defined( __STDC__ ) )
#   define FLONUM_PREC 12
#   define STRINGIZE(x) stringize(x)
#   define stringize(x) #x
#   define FLONUM_LEN  (FLONUM_PREC + 8)
#   define FORMAT      "%#." STRINGIZE( FLONUM_PREC ) "g"
#else
#   define FLONUM_LEN  20
#   define FORMAT      "%#.12g"
#endif
   
   char new[ FLONUM_LEN + 1 ];
      
   memset( new, '\0', FLONUM_LEN );
   sprintf( new, FORMAT, REAL( o ).real );
   new[ FLONUM_LEN ] = '\0';

   {
       char *newp;
       int i, simplenum;

       for( newp = new, simplenum = 1; newp; newp++ )
	 if( !isdigit( *newp ) && (*newp != '.') && (*newp != '-') )
	 {
	     /* No exponents, infinity, underflow, etc */
	     simplenum = 0;
	     break;
	 }

       if( simplenum )
       {
	  /* Chop trailing zeros. */
	  for( i = FLONUM_LEN - 1;
	      (i > 0) && ((new[ i ] == '0') || (new[ i ] == '\0'));
	      i-- );
	  new[ i ] = '\0';
       }
   }

   if( OUTPUT_STRING_PORTP( port ) )
      strputs( new, port );
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;

      fputs( new, fout );
   }
   return o;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_cell ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
display_cell( o, port )
obj_t o, port;
{
   if( OUTPUT_STRING_PORTP( port ) )
   {
      lstrputs( "#<cell:", port, 7 );
      write_object( CELL_REF( o ), port );
      lstrputs( ">", port, 7 );
   }
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
      
      fputs( "#<cell:", fout );
      write_object( CELL_REF( o ), port );
      fputs( ">", fout);
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    display_char ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
display_char( o, port )
obj_t o, port;
{
   if( OUTPUT_STRING_PORTP( port ) )
      strputc( CCHAR( o ), port );
   else
   {
      FILE *fout = OUTPUT_PORT( port ).file;
         
      fputc( CCHAR( o ), fout );
   }
      
   return o;
}

/*---------------------------------------------------------------------*/
/*    write_char ...                                                   */
/*---------------------------------------------------------------------*/
obj_t
write_char( c, port )
obj_t c, port;
{
   int  cc = CCHAR( c );
   
   if( OUTPUT_STRING_PORTP( port ) )
   {
      if( (cc > 0) && (cc < 128) && (char_name[ cc ][ 0 ]) )
      {
      
         lstrputs( "#\\", port, 2 );
         strputs( char_name[ cc ], port );
      }
      else
      {
         char aux[ 10 ];
         
         sprintf( aux, "#a%03d", (unsigned char)(cc) );
         strputs( aux, port );
      }
   }
   else
   {
      FILE *f = OUTPUT_PORT( port ).file;
   
      if( (cc > 0) && (cc < 128) && (char_name[ cc ][ 0 ]) )
         fprintf( f, "#\\%s", char_name[ CCHAR( c ) ] );
      else
         fprintf( f, "#a%03d", (unsigned char)(cc) );
   }
   return c;
}

/*---------------------------------------------------------------------*/
/*    ill_char_rep ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
ill_char_rep( c )
obj_t c;
{
   char aux[ 10 ];

   sprintf( aux, "#a%03d", CCHAR( c ) );

   return c_constant_string_to_string( aux );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    write_object ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
write_object( o, port )
obj_t o, port;
{
   if( INTEGERP( o ) )
      return display_fixnum( o, port );
   
   if( CHARP( o ) )
      return display_char( o, port );

#if defined( TAG_STRING )
   if( STRINGP( o ) )
      return display_string( o, port );
#endif  
      
#if defined( TAG_REAL )
   if( REALP( o ) )
      return display_flonum( o, port );   
#endif

#if defined( TAG_CELL )
   if( CELLP( o ) )
      return display_cell( o, port );
#endif
   
   switch( (long)o )
   {
#if !(defined( ALLOCATE_CONSTANT ))
    case (long)BNIL :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "()", port, 2 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
               fputs( "()", fout );
            }

            return o;

    case (long)BUNSPEC :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#unspecified", port, 12 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
	       fputs( "#unspecified", fout );
            }
         
            return o;
    
    case (long)BFALSE :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#f", port, 2 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;

               fputs( "#f", fout );
            }

            return o;
    
    case (long)BTRUE :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#t", port, 2 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;

               fputs( "#t", fout );
            }

            return o;

    case (long)BEOF :
            if( OUTPUT_STRING_PORTP( port ) )
	       lstrputs( "#eof-object", port, 14 );
            else
            {
               FILE *fout = OUTPUT_PORT( port ).file;
               
	       fputs( "#eof-object", fout);
            }
         
            return o;
#endif
       
    default :
            if( CNSTP( o ) )
            {
               if( OUTPUT_STRING_PORTP( port ) )
               {
                  char aux[ 7 ];
               
                  sprintf( aux, "#<%04x>", (int)CCNST( o ) );
                  lstrputs( aux, port, 7 );
               
               }
               else
               {
                  FILE *fout = OUTPUT_PORT( port ).file;
               
                  fprintf( fout, "#<%04x>", (int)CCNST( o ) );
               }
               
               return o;
            }
                
            if( !POINTERP( o ) )
            {
               if( OUTPUT_STRING_PORTP( port ) )
               {
                  char aux[ 16 ];
               
                  sprintf( aux, "#<???:%08x>", (unsigned long)o );
                  strputs( aux, port );
               }
               else
               {
                  FILE *fout = OUTPUT_PORT( port ).file;
                  
                  fprintf( fout, "#<???:%08x>", (unsigned long)o );
               }
               
               return o;
            }
	    else 
	    switch( TYPE( o ) )
	    {
#if( !defined( TAG_STRING ) )
               case STRING_TYPE :
	             return display_string( o, port );
#endif
                        
	       case SYMBOL_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
		        strputs( SYMBOL( o ).name, port );
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fputs( SYMBOL( o ).name, fout );
                     }
                     
		     return o;

#if( !defined( TAG_REAL ) )
	       case REAL_TYPE :
	             return display_flonum( o, port );
#endif
                        
	       case PROCEDURE_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 100 ];
                  
                        sprintf( new, "#<procedure:%x.%d>",
                                 (unsigned long) o,
                                 (long)PROCEDURE( o ).arity );
                        strputs( new, port );
                     }
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fprintf( fout, "#<procedure:%x.%d>", (unsigned long)o,
				 (long)PROCEDURE( o ).arity );
                     }
                     
                     return o;
        
	       case OUTPUT_PORT_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 100 ];
                        
			sprintf( new, "#<output_port:%s>",
				 OUTPUT_PORT( o ).name ); 
                        strputs( new, port );
                     }
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                     
                        fprintf( fout, "#<output_port:%s>",
                                 OUTPUT_PORT( o ).name );
                     }
                        
                     return o;
                  
	       case OUTPUT_STRING_PORT_TYPE :
                     if( OUTPUT_STRING_PORTP( port ) )
		        lstrputs( "#<output_string_port>", port, 21 );
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fputs( "#<output_string_port>", fout );
                     }
                        
                     return o;
                  
	       case INPUT_PORT_TYPE : 
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 500 ];
                        
			sprintf( new, "#<input_port:%s.%d>",
				 INPUT_PORT( o ).name,
				 (long)INPUT_PORT( o ).bufsiz );
                        strputs( new, port );
                     } 
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fprintf( fout, "#<input_port:%s.%d>",
                                 INPUT_PORT( o ).name,
                                 (long)INPUT_PORT( o ).bufsiz );

                     }
                     
		     return o;
      
	       case BINARY_PORT_TYPE : 
                     if( OUTPUT_STRING_PORTP( port ) )
                     {
                        char new[ 500 ];
                        
			sprintf( new, "#<binary_input_port:%s.%s>",
				 BINARY_PORT( o ).name,
                                 BINARY_PORT_INP( o ) ? "in" : "out" );
                        strputs( new, port );
                     } 
                     else
                     {
                        FILE *fout = OUTPUT_PORT( port ).file;
                        
                        fprintf( fout, "#<binary_input_port:%s.%s>",
                                 BINARY_PORT( o ).name,
                                 BINARY_PORT_INP( o ) ? "in" : "out" );
                     }
                                
		     return o;

#if( !defined( TAG_CELL ) )                    
	       case CELL_TYPE :
	             return display_cell( o, port );
#endif

	       case ELONG_TYPE:
		     if( OUTPUT_STRING_PORTP( port ) )
		     {
			char new[ 100 ];
			
			sprintf( new, "#e%ld", BELONG_TO_LONG( o ) );
			strputs( new, port );
		     }
		     else
		     {
			FILE *fout = OUTPUT_PORT( port ).file;
			
			fprintf( fout, "#e%ld", BELONG_TO_LONG( o ) );
		     }
		     
		     return o;

	       case LLONG_TYPE:
		     if( OUTPUT_STRING_PORTP( port ) )
		     {
			char new[ 100 ];
			
			sprintf( new, "#l%ld", BLLONG_TO_LLONG( o ) );
			strputs( new, port );
		     }
		     else
		     {
			FILE *fout = OUTPUT_PORT( port ).file;
			
			fprintf( fout, "#l%ld", BLLONG_TO_LLONG( o ) );
		     }
		     
		     return o;
                  
	       case FOREIGN_TYPE :
	          if( OUTPUT_STRING_PORTP( port ) )
		  {
		     char new[ 500 ];
		     
		     lstrputs( "#<foreign:", port, 10 );
		     write_object( FOREIGN_ID( o ), port );

		     sprintf( new, ":%x>", FOREIGN_TO_COBJ( o ) );
		     strputs( new, port );
	 	  }
		  else
		  {
		     FILE *fout = OUTPUT_PORT( port ).file;

		     fputs( "#<foreign:", fout );
		     write_object( FOREIGN_ID( o ), port );
		     fprintf( fout, ":%x>", FOREIGN_TO_COBJ( o ) );
		  }
		  return o;

	       default :
                  if( OUTPUT_STRING_PORTP( port ) )
                  {
                     char aux[ 20 ];
                  
                     sprintf( aux,
			      "#<???:%d:%08x>",
			      TYPE( o ),
			      (unsigned long)o );
                     strputs( aux, port );
                  }                            
                  else
                  {
                     FILE *fout = OUTPUT_PORT( port ).file;

                     fprintf( fout,
			      "#<???:%d:%08x>",
			      TYPE( o ),
			      (unsigned long)o );
                  }
                        
		  return o;
            }
   }
}


