/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
**  File  : getstring.c
**  Datum : 26-4-92
**  Doel  : De invoer van strings mogelijk maken
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "keymap.h"
#include "output.h"
#include "message.h"
#include "getstring.h"
#include "helpfile.h"

#define INPUTCURSOR XC_xterm
#define TEXTX        3
#define TEXTY        3
#define STACKD      20

typedef struct STRINGINFO STRINGINFO;
struct STRINGINFO {
           Window win_id;
           unsigned int width, height;
           Bool is_integer, is_mapped;
           int charcurs, pixcurs;
           int maxlen, maxpix, firstpix;
           char *text;
           STRINGINFO *prev, *next;
        } ;

static unsigned long string_mask;
static XSetWindowAttributes string_attr;
static STRINGINFO *strinput = NULL;
static STRINGINFO *stackstr[STACKD];
static int stackdpt = 0;
static int key_map=0;
static int next_field =0;


#define compens       (font_height_max -line_height())/2 + \
                       font_ascent(TEXTFONT,0) - font_ascent_max

#define INPUT_OK(A)   if (strinput && strinput->is_mapped && (A))

static void switch_cursor(STRINGINFO *sinfo)
{
    INPUT_OK(sinfo->pixcurs>=0) {
	set_output_window((void *) &sinfo->win_id);
	set_x_y(0,TEXTY+compens);
	set_drawstyle(INVISIBLE);
	thinspace(sinfo->pixcurs);
	out_cursor(CURSOR);
	unset_output_window();
    }
}

static void remove_old_cursor(STRINGINFO *sinfo)
{
    if (strinput) {
	switch_cursor(strinput);
	strinput->pixcurs = strinput->charcurs= -1;
	strinput = NULL;
    }
    if (strinput!=sinfo )
	strinput = sinfo;
}

static void push_stack(void)
{
    if (stackdpt<STACKD)
	stackstr[stackdpt++] = strinput;
}

static void pop_stack(void)
{
    if (stackdpt)
	strinput = stackstr[--stackdpt];
    else
	strinput = NULL;
}

static void remove_from_stack(STRINGINFO *sinfo)
{
    int i, d=0;

    for (i=0; i<stackdpt; i++) {
	if (stackstr[i]== sinfo)
	    d++;
	else
	    stackstr[i-d] = stackstr[i];
    }
    stackdpt -= d;
}

#define char2width(A) (2 + TEXTX*2 + (A)*char_width(Font2Char(TEXTFONT,'x'),0))
#define width2char(A) (((A)-2-TEXTX*2)/char_width(Font2Char(TEXTFONT,'x'),0))
#define max_curspix(A) ((A) - TEXTX - 1)
#define min_curspix(A) (TEXTX+1)

unsigned int string_height(void)
{
    int i;
    push_fontgroup(POPUPFONT);
    i = (TEXTY * 2 + font_height(TEXTFONT,0));
    pop_fontgroup();
    return i;
}

unsigned int string_window_width(int nchars)
{
    int i;
    push_fontgroup(POPUPFONT);
    if (nchars>MAX_GETSTRING) nchars = MAX_GETSTRING;
    i = char2width(nchars);
    pop_fontgroup();
    return i;
}

static void string_draw(void *data)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;
    int i;

    if (sinfo && sinfo->is_mapped) {
	push_fontgroup(POPUPFONT);
	set_output_window( (void *) &sinfo->win_id);
	out_clear();
	set_x_y(sinfo->firstpix, TEXTY+compens);
	for (i=0; sinfo->text[i]; i++) {
	    if (i==sinfo->charcurs) out_cursor(CURSOR);
	    out_char(sinfo->text[i]);
	}
	if (sinfo->charcurs==i) out_cursor(CURSOR);
	out_char(Newline);
	unset_output_window();
	pop_fontgroup();
    }
}

static Bool adjust_firstpix(STRINGINFO *sinfo)
{
    int i;

    i=0;
    if (sinfo->pixcurs < min_curspix(sinfo->width))
	i = min_curspix(sinfo->width);
    else
	if (sinfo->pixcurs > max_curspix(sinfo->width))
	    i = max_curspix(sinfo->width);
    if (i) {
	sinfo->firstpix += i-sinfo->pixcurs;
	sinfo->pixcurs = i;
	return True;
    } else
	return False;
}

static void get_pos(STRINGINFO *sinfo, int x)
{
    int i;

    /* bepaal positie bij x en verplaats cursor */

    push_fontgroup(POPUPFONT);
    remove_old_cursor(sinfo);
    INPUT_OK(True) {
	set_output_window(&sinfo->win_id);
	set_drawstyle(INVISIBLE);
	set_x_y(sinfo->firstpix, TEXTY+compens);
	for (i=0; x>where_x() && sinfo->text[i]; i++)
	    out_char(Font2Char(TEXTFONT, sinfo->text[i]));
	if (sinfo->charcurs !=i) {
	    sinfo->charcurs = i;
	    sinfo->pixcurs = where_x();
	}
	if (adjust_firstpix(sinfo)) {
	    unset_output_window();
	    string_draw( (void *) sinfo);
	} else {
	    if (sinfo->pixcurs>=0) {
		set_x_y(0,TEXTY+compens);
		thinspace(sinfo->pixcurs);
		out_cursor(CURSOR);
	    }
	    unset_output_window();
	}
    }
    pop_fontgroup();
}

static void string_press(void *data, XButtonEvent *event)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;
    
    get_pos(sinfo,event->x);
    get_motion_hints(sinfo->win_id, -1);
}

static void string_release(void *data, XButtonEvent *event)
{
    stop_motion_hints();
    use_map(0, key_map);
}

static void string_motion(void *data, int x_pos, int y_pos)
{
    get_pos( (STRINGINFO *) data, x_pos);
}

void string_destroy(void *data)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    /*
    **  als niet hele lijst verdwijnt opnieuw koppelen
    */
    remove_from_stack(sinfo);
    if (sinfo == strinput) {
	pop_stack();
	if (strinput) {
	    strinput->charcurs = strlen(strinput->text);
	    strinput->firstpix = min_curspix(strinput->width);
	    push_fontgroup(POPUPFONT);
	    strinput->pixcurs = strinput->firstpix +
		string_width(TEXTFONT, strinput->text, strinput->charcurs);
	    adjust_firstpix(strinput);
	    string_draw( (void *) strinput);
	    pop_fontgroup();
	}
    }
    myfree(sinfo->text);
    destroy_window(sinfo->win_id);
}

FUNCTIONS stringfuncs = {
    string_destroy, string_draw, NULL, string_press, string_release,
    string_motion };

void string_init(void)
{
    string_mask =
	(CWBackPixel | CWBorderPixel | CWEventMask |
	 CWCursor | CWColormap);

    string_attr.background_pixel = white_pixel;
    string_attr.border_pixel = black_pixel;
    string_attr.colormap = colormap;
    string_attr.event_mask = (ExposureMask | ButtonPressMask |
			      ButtonReleaseMask | KeyPressMask |
			      ButtonMotionMask | PointerMotionHintMask |
			      StructureNotifyMask | VisibilityChangeMask);
    string_attr.cursor = XCreateFontCursor(display, INPUTCURSOR);
}

void *string_make(Window parent, char *txt, int maxlen, unsigned int width,
		  char *helpfile, int x_offset, int y_offset,
		  Bool is_integer)
{
    STRINGINFO *sinfo = 0;

    if ( !(sinfo = (STRINGINFO *) malloc( sizeof(STRINGINFO))) ||
	 !(sinfo->text =  (char *) malloc( maxlen+1)))  {
	message(ERROR, "Out of memory in string. ");
	if (sinfo) myfree(sinfo);
	return NULL;
    }
    push_fontgroup(POPUPFONT);
    sinfo->width  = char2width(width2char(width));
    sinfo->is_integer = is_integer;
    sinfo->is_mapped = False;
    if (txt)
	strncpy(sinfo->text, txt, maxlen);
    else
	sinfo->text[0] = '\0';
    sinfo->text[maxlen] = '\0';
    sinfo->maxlen = maxlen;
    sinfo->maxpix = max_curspix(sinfo->width);
    sinfo->firstpix = min_curspix(sinfo->width);
    sinfo->charcurs = sinfo->pixcurs = -1;
    sinfo->prev = sinfo;
    sinfo->next = sinfo;
    sinfo->win_id = XCreateWindow(display, parent, x_offset, y_offset,
				 sinfo->width, string_height(),
				 0, CopyFromParent, InputOutput,
				 visual,
				 string_mask, &string_attr);
    pop_fontgroup();
    if (!helpfile) helpfile=helpname[GETSTRINGHELP];
    if (add_window(sinfo->win_id, STRINGWINDOW, parent, (void *) sinfo,
		   helpfile))
	return (void *) sinfo;
    XDestroyWindow(display, sinfo->win_id);
    myfree(sinfo->text);
    myfree(sinfo);
    return NULL;
}

void string_relation(void *data, void *prev, void *next)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    sinfo->prev = (STRINGINFO *) prev;
    sinfo->next = (STRINGINFO *) next;
}

void  string_resize(void *data, int new_width)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    push_fontgroup(POPUPFONT);
    sinfo->width = char2width(width2char(new_width));
    XResizeWindow(display, sinfo->win_id, sinfo->width, string_height());
    if (strinput == sinfo) {
	sinfo->pixcurs  = min_curspix(sinfo->width);
	sinfo->charcurs = 0;
    }
    sinfo->firstpix = min_curspix(sinfo->width);
    string_draw(data);
    pop_fontgroup();
}

void string_move(void *data, int newx, int newy)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    XMoveWindow(display, sinfo->win_id, newx, newy);
}

void string_refresh(void *data, char *txt)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    strncpy(sinfo->text, txt, sinfo->maxlen);
    if (strinput == sinfo) {
	sinfo->pixcurs  = min_curspix(sinfo->width);
	sinfo->charcurs = 0;
    }
    sinfo->firstpix = min_curspix(sinfo->width);
    string_draw(data);
}

char *string_text(void *data)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;
    char *temp;

    temp = (char *) malloc(strlen(sinfo->text)+1);
    strcpy(temp, sinfo->text);
    return temp;
}

void string_get_input(void *data)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    push_stack();
    push_fontgroup(POPUPFONT);
    remove_old_cursor(sinfo);
    use_map(0, key_map);
    sinfo->charcurs = strlen(sinfo->text);
    sinfo->pixcurs = sinfo->firstpix +
	string_width(TEXTFONT, sinfo->text, -1);
    adjust_firstpix(sinfo);
    pop_fontgroup();
}

static Bool move_left(char *str, int pos, int delta)
{
    int j;

    if (delta<=0) return False;
    for (j=0; j<delta && str[pos+j]; j++);
    if (j<delta)
	str[pos] = '\0';
    else
	do {
	    str[pos] = str[pos+delta];
	} while (str[pos++]);
    return True;
}

static Bool move_right(char *str, int pos, int delta, int max)
{
    int i;

    if (delta<=0) return False;
    i=strlen(str);
    if (i+delta-1==max)
	return False;
    for ( ; i>=pos; i--)
	str[i+delta] = str[i];
    return True;
}

static void str_delete(KEYCODE keycode, Index arg)
{
    INPUT_OK( move_left(strinput->text, strinput->charcurs, arg))
	string_draw( (void *) strinput);
}

static void str_m_delete(KEYCODE keycode, Index arg)
{
    INPUT_OK( strinput->text[strinput->charcurs]) {
	strinput->text[strinput->charcurs] = '\0';
	string_draw( (void *) strinput);
    }
}

static void do_backspace(int delta)
{
    int i;

    if (delta<=0) return;
    INPUT_OK(strinput->charcurs) {
	if ((i = (strinput->charcurs -= delta))<0) {
	    delta -=i;
	    i = strinput->charcurs =0;
	}
	push_fontgroup(POPUPFONT);
	strinput->pixcurs -= string_width( TEXTFONT, strinput->text+i, delta);
	(void) adjust_firstpix(strinput);
	(void) move_left(strinput->text, i, delta);
	string_draw((void *) strinput);
	pop_fontgroup();
    }
}

static void str_backspace(KEYCODE keycode, Index arg)
{
    do_backspace(arg);
}

static void str_m_backspace(KEYCODE keycode, Index arg)
{
    do_backspace(strinput->charcurs);
}

static void str_left(KEYCODE keycode, Index arg)
{
    INPUT_OK(strinput->charcurs) {
	push_fontgroup(POPUPFONT);
	switch_cursor(strinput);
	strinput->charcurs--;
	strinput->pixcurs -=
	    char_width(Font2Char(TEXTFONT,strinput->text[strinput->charcurs]),0);
	if (adjust_firstpix(strinput))
	    string_draw( (void *) strinput);
	else
	    switch_cursor(strinput);
	pop_fontgroup();
    }
}

static void str_right(KEYCODE keycode, Index arg)
{
    INPUT_OK(strinput->text[strinput->charcurs]) {
	push_fontgroup(POPUPFONT);
	switch_cursor(strinput);
	strinput->pixcurs +=
	    char_width(Font2Char(TEXTFONT,strinput->text[strinput->charcurs]),0);
	strinput->charcurs++;
	if (adjust_firstpix(strinput))
	    string_draw(strinput);
	else
	    switch_cursor(strinput);
	pop_fontgroup();
    }
}

static void str_up(KEYCODE keycode, Index arg)
{
    INPUT_OK(True) get_pos(strinput->prev, strinput->pixcurs);
}

static void str_down(KEYCODE keycode, Index arg)
{
    INPUT_OK(True) get_pos(strinput->next, strinput->pixcurs);
}

static void str_home(KEYCODE keycode, Index arg)
{
    INPUT_OK(True) get_pos(strinput, strinput->firstpix -1);
}

static void str_end(KEYCODE keycode, Index arg)
{
    INPUT_OK(True) 
	get_pos(strinput, string_width(TEXTFONT, strinput->text, -1) +
		min_curspix(strinput->width));
}

static void str_next_field(KEYCODE keycode, Index arg)
{
    INPUT_OK(True) 
	get_pos(strinput->next, string_width(TEXTFONT,strinput->next->text,-1)+
		min_curspix(strinput->next->width));
}

static void str_prev_field(KEYCODE keycode, Index arg)
{
    INPUT_OK(True) 
	get_pos(strinput->prev, string_width(TEXTFONT,strinput->prev->text,-1)+
		min_curspix(strinput->prev->width));
}

static void str_insert_char(KEYCODE keycode, Index arg)
{
    INPUT_OK(((strinput->is_integer && (K_0<= keycode && keycode<=K_9)) ||
	      (!strinput->is_integer && (K_space<=keycode && keycode<=K_tilde))) &&
	     move_right(strinput->text,strinput->charcurs, arg,
			strinput->maxlen)) {
	int i;
        for (i=0; i<arg; i++)
	    strinput->text[strinput->charcurs+i]= (char) keycode;
	strinput->charcurs+= arg;
	push_fontgroup(POPUPFONT);
	strinput->pixcurs += arg*char_width(Font2Char(TEXTFONT, keycode),0);
	(void) adjust_firstpix(strinput);
	string_draw((void *) strinput);
	pop_fontgroup();
    }
}

static void str_insert_string(KEYCODE keycode, Index arg)
{
    INPUT_OK(wmselection) {
	char *p = wmselection;
	push_fontgroup(POPUPFONT);
	for (; *p; p++) {
	    if ((strinput->is_integer && *p>='0' && *p<='9') ||
		(!strinput->is_integer && *p>=' ' && *p<='~')) {
		if (move_right(strinput->text, strinput->charcurs,
			       1, strinput->maxlen)) {
		    strinput->text[strinput->charcurs] = *p;
		    strinput->charcurs++;
		    strinput->pixcurs+= char_width(Font2Char(TEXTFONT,*p),0);
		}
	    }
	}
	(void) adjust_firstpix(strinput);
	string_draw((void*) strinput);
	pop_fontgroup();
    }
}

static void ask_selection(KEYCODE keycode, Index arg)
{
    get_wm_selection();
}

void string_change_return(int funcnr)
{
    remove_key(key_map, K_return);
    add_key(key_map, K_return, funcnr, 1);
}

void string_reset_return(void)
{
    remove_key(key_map, K_return);
    add_key(key_map, K_return, next_field, 1);
}

void string_keyboard(void)
{
    key_map = new_map("String");
    next_field = add_func( str_insert_string, NULL);
    add_key(key_map, WMSELECT, next_field, 1);
    next_field = add_func( str_next_field, "S-next-field");
    add_func( str_prev_field, "S-previous-field");
    add_func(str_delete, "S-delete-char");
    add_func( str_m_delete, "S-kill-line");
    add_func( str_backspace, "S-backward-delete-char");
    add_func( str_m_backspace, "S-backward-kill-line");
    add_func( str_left, "S-backward-char");
    add_func( str_right, "S-forward-char");
    add_func( str_up, "S-previous-line");
    add_func( str_down, "S-next-line");
    add_func( str_home, "S-beginning-of-line");
    add_func( str_end, "S-end-of-line");
    add_func( ask_selection, "insert-selection");
    add_func( str_insert_char, "S-self-insert");
}

void string_map(void *data)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    sinfo->is_mapped = True;
}

void string_unmap(void *data)
{
    STRINGINFO *sinfo = (STRINGINFO *) data;

    sinfo->is_mapped = False;
}
