/*
 *
 *  (c) COPYRIGHT MIT and INRIA, 1996.
 *  Please first read the full copyright statement in file COPYRIGHT.
 *
 */

/*
 *
 * Author: S Bonhomme
 *
 */

/* Included headerfiles */
#define THOT_EXPORT extern
#include "amaya.h"
#include "trans.h"
#include "html2thot_f.h"
#include "transparse_f.h"
 
/*
 * pattern matching stack associates a node of source structure tree to a 
 * list of pat.
 */
static struct _mStack
  {
     strNode            *Node;
     strListSymb        *Symbols;
  }
matchStack[MAX_STACK];

static int          maxMatchDepth;

static int          topMatchStack;

/*  HTML generation */
/* stack of generated tags */
typedef struct _gStack
  {
     char               *Tag;
     strAttrDesc        *Attributes;
     int                 Idf;
     int                 Nbc;
  }
strGenStack;

static strGenStack *generationStack[MAX_STACK];

static int          topGenerStack;

static char        *bufHTML;		/* HTML text buffer */
static int          szHTML;		/* size of generated HTML code */
static int          lastRulePlace;	/* pointer to the stack*/
/* identifies the last node generated by the previous rule applied*/
static int          idfCounter;		/* element identifier generator */
static boolean      isClosed;

/* correspondance table between the transformations proposed to the user */
/* and the matching descriptors */
static strMatch    *menuTrans[20];
/* did the transformation succed ? */
static boolean      resultTrans;

/* pointer to the selected elements */
static int          ffc, flc, lfc, llc;
static int          maxSelDepth;
static Element      myFirstSelect, origFirstSelect;
static Element      myLastSelect, origLastSelect;
static Element      mySelect;

#ifdef __STDC__
void		    TransCallbackDialog(int ref, int typedata, char *data);
#else
void                TransCallbackDialog (/*ref, typedata, data*/);
#endif
/*----------------------------------------------------------------------
   FreeStructTree: frees the structure tree and matching relations. 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         FreeStructTree (StructureTree t)
#else
static void         FreeStructTree (t)
structureTree       t;

#endif
{
   strMatch           *m, *m2;
   strMatchChildren   *mc, *mc2;
   strNode            *c, *n;

   if (t != NULL)
     {
	m = t->Matches;
	while (m != NULL)
	  {
	     mc = m->MatchChildren;
	     while (mc != NULL)
	       {
		  mc2 = mc->Next;
		  TtaFreeMemory ((char *) mc);
		  mc = mc2;
	       }
	     m2 = m->Next;
	     TtaFreeMemory ((char *) m);
	     m = m2;
	  }

	c = t->Child;
	while (c != NULL)
	  {
	     n = c->Next;
	     FreeStructTree (c);
	     c = n;
	  }
	TtaFreeMemory (t->Tag);
	TtaFreeMemory ((char *) t);
     }
}


/*----------------------------------------------------------------------
  FreeMatchEnv : frees the transformation context . 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         FreeMatchEnv ()
#else
static void         FreeMatchEnv ()
#endif
{
   strListElem           *l, *l1;

   /* liberation de la liste des sous-arbres transferes */
   l = strMatchEnv.ListSubTrees;
   strMatchEnv.ListSubTrees = NULL;
   while (l != NULL)
     {
	l1 = l;
	l = l->Next;
	TtaFreeMemory ((char *) l1);
     }
   strMatchEnv.ListSubTrees = NULL;

   /* liberation de l'arbre des tags */
   FreeStructTree (strMatchEnv.SourceTree);
   strMatchEnv.SourceTree = NULL;
}

/*----------------------------------------------------------------------
   InitTransform: initialisation, called during Amaya initialisation
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                InitTransform ()
#else
void                InitTransform ()
#endif
{
   TRANSDIAL = TtaGetMessageTable ("transdialogue", TRANS_MSG_MAX);
   TransBaseDialog = TtaSetCallback (TransCallbackDialog, MAX_TRANS_DLG);
   timeLastWrite = (time_t) 0;
}

/*----------------------------------------------------------------------
   NewNode: allocation of a Structure tree node.     
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static strNode *NewNode (char *tag)
#else
static strNode * NewNode (tag)
char               *tag;

#endif
{
   StructureTree res;

   res = (StructureTree) TtaGetMemory (sizeof (strNode));
   res->Tag = (char *) TtaGetMemory (NAME_LENGTH);
   strcpy (res->Tag, tag);
   res->Matches = NULL;
   res->Candidates = NULL;
   res->Elem = NULL;
   res->MatchSymb = NULL;
   res->Parent = NULL;
   res->Child = NULL;
   res->Next = NULL;
   res->Previous = NULL;
   res->IsTrans = FALSE;
   res->NodeDepth = 0;
   return res;
}


/*----------------------------------------------------------------------
    thot tree -> structure tree
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         BuildStructureTree (Element elem, Document doc, StructureTree father, int maxdepth, int depth)
#else
static void         BuildStructureTree (elem, doc, father, maxdepth, depth)
Element             elem;
Document            doc;
strNode            *father;
int                 maxdepth;
int                 depth;

#endif
{
   Element             elemCour;
   Attribute           attr;
   ElementType         elemType;
   char               *tag;
   strNode            *new, *child;

   if (depth > maxdepth)
      return;
   tag = TtaGetMemory (NAME_LENGTH);
   elemType = TtaGetElementType (elem);
   strcpy (tag, GITagNameByType (elemType.ElTypeNum));
   attr = NULL;
   TtaNextAttribute (elem, &attr);
   if (strcmp (tag, "???") && (TtaGetFirstChild (elem) != NULL || attr != NULL || TtaIsLeaf (elemType)))
     {
	new = NewNode (tag);
	new->Elem = elem;
	new->Parent = father;
	new->NodeDepth = depth;
	if (father->Child == NULL)
	  {
	     father->Child = new;
	     new->Previous = NULL;
	  }
	else
	  {
	     child = father->Child;
	     while (child->Next != NULL)
		child = child->Next;
	     child->Next = new;
	     new->Previous = child;
	  }
	depth++;
     }
   else
      new = father;
   TtaFreeMemory (tag);
   if (elemType.ElTypeNum != HTML_EL_Comment_ && elemType.ElTypeNum != HTML_EL_Invalid_element)
     {
	elemCour = TtaGetFirstChild (elem);
	while (elemCour != NULL)
	  {
	     BuildStructureTree (elemCour, doc, new, maxdepth, depth);
	     TtaNextSibling (&elemCour);
	  }
     }
}

/*----------------------------------------------------------------------
    pattern matching functions
  ----------------------------------------------------------------------*/

/*----------------------------------------------------------------------
   IntersectMatch: builds the symbol list res, intersection of symbol list LS with the 
    symbols contained in matching relation list MS
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         IntersectMatch (strListSymb ** res, strListSymb * LS, strMatch * MS)
#else
static void         IntersectMatch (res, LS, MS)
strListSymb          **res;
strListSymb           *LS;
strMatch              *MS;

#endif
{
   strListSymb        *pLS, *pRes;
   strMatch           *pMS;
   boolean             found;

   if (*res != NULL)
     {
	FreeList (*res);
	*res = NULL;
     }
   pRes = NULL;
   pLS = LS;
   while (pLS != NULL)
     {
	pMS = MS;
	found = FALSE;
	while (!found && pMS != NULL)
	  {
	     found = (pMS->MatchSymb == pLS->Symbol);
	     pMS = pMS->Next;
	  }
	if (found)
	  {
	     if (pRes == NULL)
	       {
		  *res = (strListSymb *) TtaGetMemory (sizeof (strListSymb));
		  pRes = *res;
	       }
	     else
	       {
		  pRes->Next = (strListSymb *) TtaGetMemory (sizeof (strListSymb));
		  pRes = pRes->Next;
	       }
	     pRes->Next = NULL;
	     pRes->Symbol = pLS->Symbol;
	  }
	pLS = pLS->Next;
     }
}




/*----------------------------------------------------------------------
  initialistion of pattern matching stack
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void InitMatchStack (void)
#else
static void InitMatchStack ()
#endif
{
   int i;

   for (i = 0; i < MAX_STACK; i++)
     {
	matchStack[i].Node = NULL;
	matchStack[i].Symbols = NULL;
     }
    topMatchStack = 0;
}
/*----------------------------------------------------------------------
  push a node of source tree with the symbols it matches in  pattern matching stack
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void PushMatchStack (strNode * t, strListSymb * v)
#else
static void PushMatchStack (t, v)
strNode     *t;
strListSymb *v;
#endif
{
   if (topMatchStack < MAX_STACK)
     {
	matchStack[topMatchStack].Node = t;
	matchStack[topMatchStack++].Symbols = v;
     }
   else
     {
	printf ("Matching stack overflow \n");
     }
}


#ifdef __STDC__
static void PopMatchStack (strNode ** t, strListSymb ** v)
#else
static void PopMatchStack (t, v)
strNode     **t;
strListSymb **v;
#endif
{
   if (topMatchStack > 0)
     {
	*t = matchStack[--topMatchStack].Node;
	*v = matchStack[topMatchStack].Symbols;
	matchStack[topMatchStack].Node = NULL;
	matchStack[topMatchStack].Symbols = NULL;
     }
   else
     {
	*t = NULL;
	*v = NULL;
     }
}

/*----------------------------------------------------------------------
  ConstListMatch: adds the symbol symb to the list of symbols matched with node.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void ConstListMatch (strNode * node, strSymbDesc * symb)
#else
static void ConstListMatch (node, symb)
strNode            *node;
strSymbDesc        *symb;
#endif
{
   strMatch           *sm;
   strMatchChildren   *smc;
   strNode            *n;
   strListSymb           *dl;

   sm = node->Matches;
   if (sm == NULL)
     {
	node->Matches = (strMatch *) TtaGetMemory (sizeof (strMatch));
	sm = node->Matches;
     }
   else
     {
	while (sm->Next != NULL && sm->MatchSymb != symb)
	   sm = sm->Next;
	if (sm->MatchSymb != symb)
	  {
	     sm->Next = (strMatch *) TtaGetMemory (sizeof (strMatch));
	     sm = sm->Next;
	  }
	else
	   return;
     }
   sm->MatchSymb = symb;
   sm->MatchNode = node;
   sm->Next = NULL;
   sm->MatchChildren = NULL;
   PopMatchStack (&n, &dl);
   while (n != NULL)
     {
	smc = (strMatchChildren *) TtaGetMemory (sizeof (strMatchChildren));
	smc->Next = sm->MatchChildren;
	sm->MatchChildren = smc;
	smc->MatchSymb = dl->Symbol;
	smc->MatchNode = n;
	PopMatchStack (&n, &dl);
     }
}

/*----------------------------------------------------------------------
  ChildrenMatch: return TRUE if the chidren of node n are matched with the possible 
   children of the pattern symbol p.
   ----------------------------------------------------------------------*/

#ifdef __STDC__
static boolean ChildrenMatch (strNode * n, strSymbDesc * p)
#else
static boolean ChildrenMatch (n, p)
strNode     *n;
strSymbDesc *p;
#endif
{
   strNode            *child;
   strListSymb        *candidate, *ms;
   boolean             matchFound, matchFailed;

   matchFound = FALSE;
   matchFailed = FALSE;

   candidate = p->Children;
   if (candidate == NULL)
     /* there is no possible children for p, returns true */
     {
	ConstListMatch (n, p);
	return TRUE;
     }
   if (n->Child == NULL)
     /* n is a leaf */
     {
	if (p->IsOptChild)
	  { /* possible children of p are all optional: returns true */
	     ConstListMatch (n, p);
	     return TRUE;
	  }
	else
	   return FALSE;
     }
   /* looking for a matching of each child of symbol p (candidate) with the children */
   /* of the source node n */ 
   child = n->Child;
   IntersectMatch (&(child->Candidates), p->Children, child->Matches);
   candidate = child->Candidates;
   if (candidate == NULL)
     {
	return FALSE;
     }
   InitMatchStack ();
   while (!matchFailed && !matchFound)
     {
	if (child == NULL && (candidate != NULL && candidate->Symbol == NULL))
	  {
	    /* all the children of n are matched, add a matching for n and p */
	     ConstListMatch (n, p);
	     matchFound = TRUE;
	  }
	else if (child == NULL || candidate == NULL || candidate->Symbol == NULL)
	  {  
	    /* the children are not all matched, pop the matching stack to see if an other */
	    /* symbol can be matched with a node already explored */
	     PopMatchStack (&child, &ms);
	     while (child != NULL &&
		    ms->Next == NULL)
		PopMatchStack (&child, &ms);
	     if (child == NULL)
	       {
		  matchFailed = TRUE;
		  /* no matching of n has been found, exiting */
	       }
	     else
	       {/* a new candidate to matching is aviable */
		  candidate = ms->Next;
	       }
	  }
	else
	  {			/* here:  child!=NULL && candidate != -1 */
	    /* searches if the next siblings of child can be matched with successors */
	    /* of candidate */

	     /* push child and candidate */
	     PushMatchStack (child, candidate);
	     if (child->Next != NULL)
	       {
		  /* build the candidate list for matching of the next sibling */
		  IntersectMatch (&(child->Next->Candidates),
				  candidate->Symbol->Followings,
				  child->Next->Matches);
		  /* try the first candidate */
		  candidate = child->Next->Candidates;
	       }
	     else
	       {		/* child->Next==NULL */
		  ms = candidate->Symbol->Followings;
		  while (ms != NULL && ms->Symbol != NULL)
		     ms = ms->Next;
		  if (ms != NULL)
		     candidate = ms;
	       }
	     child = child->Next;
	  }
     }
   
   if (!matchFound && p->IsOptChild)
     {
	ConstListMatch (n, p);
	matchFound = TRUE;
     }
   return matchFound;
}

/*----------------------------------------------------------------------
  check if elem actually has mandatory attributes of the pattern symbol pSymb
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean      MatchAttributes (strSymbDesc * pSymb, Element elem)
#else
static boolean      MatchAttributes (pSymb, elem)
strSymbDesc           *pSymb;
Element             elem;

#endif
{
   boolean             result;
   strAttrDesc        *pAttr;
   AttributeType       AttrTyp;
   Attribute           attr;
   char               *buf;
   int                 AttrKind, length;

   buf = TtaGetMemory (MAX_LENGTH);
   AttrTyp.AttrSSchema = TtaGetElementType (elem).ElSSchema;
   result = TRUE;
   pAttr = pSymb->Attributes;
   while (pAttr != NULL && result)
     {
	AttrTyp.AttrTypeNum = pAttr->ThotAttr;
	if (AttrTyp.AttrTypeNum != 0)
	  {
	     attr = TtaGetAttribute (elem, AttrTyp);
	     result = (attr != (Attribute) NULL);
	  }
	if (result)
	  {
	     TtaGiveAttributeType (attr, &AttrTyp, &AttrKind);
	     if (AttrKind == 2 && !pAttr->IsInt)
	       {
		  length = MAX_LENGTH;
		  TtaGiveTextAttributeValue (attr, buf, &length);
		  result = !strcmp (pAttr->TextVal, buf);
	       }
	     else if (AttrKind != 2 && pAttr->IsInt)
	       {
		  result = (TtaGetAttributeValue (attr) == pAttr->IntVal);
	       }
	     else
		result = FALSE;
	  }
	pAttr = pAttr->Next;
     }
   TtaFreeMemory (buf);
   return result;
}


/*----------------------------------------------------------------------
   MatchNode : builds a pointer list to symbols that match with the node n
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean      MatchNode (strNode * n)
#else
static boolean      MatchNode (n)
strNode            *n;

#endif
{
   strTransDesc          *td;
   strSymbDesc           *sd;

   td = strMatchEnv.Transformations;
   while (td != NULL)
     {
	if (td->IsActiveTrans)
	  {
	     sd = td->Symbols;
	     /* for each symbol having the same tag than the node n, seaches if */
	     /* the children of the node can be matched with those of the pattern symbol */
	     while (sd != NULL)
	       {
		  if (sd->IsActiveSymb)
		     if ((n->NodeDepth - sd->Depth <= maxSelDepth)
			 && (n->NodeDepth - sd->Depth >= 0))
			if (!strcmp (sd->Tag, "*") || !strcmp (sd->Tag, n->Tag))
			   if (sd->Attributes == NULL || MatchAttributes (sd, n->Elem))
			      ChildrenMatch (n, sd);
		  sd = sd->Next;
	       }
	     ChildrenMatch (n, td->RootDesc);
	  }
	td = td->Next;
     }
   return TRUE;
}

/*----------------------------------------------------------------------
  PostfixSearch:  Postfix search in source tree t applying a binary function f to every node . 
  the search end if f(n) = False, or if the entiere tree has been traversed.
  returns the node beeing tested if failure, NULL if the entiere tree has been searched.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static strNode *RecursivePostfixSearch (StructureTree t, boolean (*f) (strNode *))
#else
static strNode *RecursivePostfixSearch (t, f)
StructureTree t;
boolean       (*f) (strNode *);
#endif
{
   strNode *n, *res;

   n = t->Child;
   while (n != NULL)
     {
	res = RecursivePostfixSearch (n, f);
	if (res == NULL)
	  {
	     if (f (n))
		n = n->Next;
	     else
		return n;
	  }
	else
	   return res;
     }
   return NULL;
}

#ifdef __STDC__
static strNode *PostfixSearch (StructureTree t, boolean (*f) (strNode *))
#else
static strNode *PostfixSearch (t, f)
StructureTree t;
boolean       (*f) (strNode *);
#endif
{
   strNode *res;

   res = RecursivePostfixSearch (t, f);
   if (res == NULL)
     {
	if (f (t))
	   return NULL;
	else
	   return t;
     }
   else
      return res;
}

/*----------------------------------------------------------------------
 AddListSubTree:  inserts the reference to a structure subtree, with its rank between its sibling,
 and its identifier in the environment's subtrees list
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void AddListSubTree (Element elem, int idf, int rank)
#else
static void AddListSubTree (elem, idf, rank)
Element             elem;
int                 idf;
int                 rank;

#endif
{
   strListElem           *pcour, *pprec;

   pcour = strMatchEnv.ListSubTrees;
   pprec = NULL;
   while (pcour != NULL && (pcour->Id < idf || (pcour->Id == idf && pcour->Rank < rank)))
     {
	pprec = pcour;
	pcour = pcour->Next;
     }
   if (pprec != NULL)
     {
	pprec->Next = (strListElem *) TtaGetMemory (sizeof (strListElem));
	pprec = pprec->Next;
     }
   else
     {
	strMatchEnv.ListSubTrees = (strListElem *) TtaGetMemory (sizeof (strListElem));
	pprec = strMatchEnv.ListSubTrees;
     }
   pprec->Id = idf;
   pprec->Rank = rank;
   pprec->Elem = elem;
   pprec->Next = pcour;
}

/*----------------------------------------------------------------------
  Searches an element identified with a given label in the environement subtrees list.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static Element      FindListSTreeByLabel (char *label)
#else
static Element      FindListSTreeByLabel (label)
char               *label;

#endif
{
   strListElem        *pcour;
   boolean             found;

   found = FALSE;
   pcour = strMatchEnv.ListSubTrees;
   while (!found && pcour != NULL)
     {
	found = !strcmp (TtaGetElementLabel (pcour->Elem), label);
	if (!found)
	   pcour = pcour->Next;
     }
   if (!found)
      return NULL;
   else
      return pcour->Elem;
}

/*----------------------------------------------------------------------
   search for the next element with the identifier id in the ListSubTrees, searches the first one if 
   *elem =NULL 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static int          FindListSubTree (int id, Element * elem)
#else
static int          FindListSubTree (id, elem)
int                 id;
Element            *elem;

#endif
{
   strListElem        *pcour;
   boolean             found;
   int                 result;

   found = FALSE;
   pcour = strMatchEnv.ListSubTrees;
   while (!found && pcour != NULL)
     {
	found = (pcour->Id == id && (*elem == NULL || *elem == pcour->Elem));
	if (!found)
	   pcour = pcour->Next;
     }
   if (found && *elem == NULL)
     {
	*elem = pcour->Elem;
	result = pcour->Rank;
     }
   else
     {
	*elem = NULL;
	result = 0;
	if (found)
	  {
	     pcour = pcour->Next;
	     found = FALSE;
	     while (!found && pcour != NULL)
	       {
		  found = pcour->Id == id;
		  if (!found)
		     pcour = pcour->Next;
	       }
	     if (found)
	       {
		  *elem = pcour->Elem;
		  result = pcour->Rank;
	       }
	  }
     }
   return result;
}

/*----------------------------------------------------------------------
   StartHtmlParser initializes  parsing environement in order to parse
   the HTML fragment in buffer in the context of a last descendance of the
   element ElFather 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean StartHtmlParser (strMatchChildren * sMatch, Document doc)
#else
static boolean StartHtmlParser (sMatch, doc)
strMatch           *sMatch;
Document            doc;

#endif
{
   strMatchChildren   *prevMatch, *DMatch;
   Element             lastEl, courEl, invEl;
   ElementType         typeEl;
   boolean             res;


   prevMatch = NULL;
   DMatch = sMatch;
   myFirstSelect = DMatch->MatchNode->Elem;
   TtaPreviousSibling (&myFirstSelect);
   isClosed = TRUE;
   res = TRUE;
   if (myFirstSelect == NULL)
     {
	myFirstSelect = DMatch->MatchNode->Parent->Elem;
	isClosed = FALSE;
     }
   while (DMatch != NULL)
     {
	prevMatch = DMatch;
	DMatch = DMatch->Next;
	if (DMatch != NULL)
	   TtaRemoveTree (prevMatch->MatchNode->Elem, doc);
     }
   if (prevMatch != NULL)
     {
	myLastSelect = prevMatch->MatchNode->Elem;
	TtaNextSibling (&myLastSelect);
	TtaRemoveTree (prevMatch->MatchNode->Elem, doc);
     }
   if (strcmp (bufHTML, ""))
     {
	TtaSetStructureChecking (0, doc);
	InitializeParser (myFirstSelect, isClosed, doc);
	HTMLparse (NULL, bufHTML);
	TtaSetStructureChecking (1, doc);
	typeEl = TtaGetElementType (myFirstSelect);
	typeEl.ElTypeNum = HTML_EL_Invalid_element;
	invEl = NULL;
	if (isClosed)
	  {
	     courEl = myFirstSelect;
	     TtaNextSibling (&courEl);
	     invEl = TtaSearchTypedElement (typeEl, SearchForward, courEl);
	     if (invEl != NULL)
	       {
		  if ((!myLastSelect && TtaIsAncestor (invEl, TtaGetParent (myFirstSelect)))
		   || ((myLastSelect && TtaIsBefore (invEl, myLastSelect))))
		    {		/* if the transformation failed */
		       res = FALSE;
		       while (courEl != NULL && courEl != myLastSelect)
			 {	/* deleting the generated structure */
			    lastEl = courEl;
			    TtaNextSibling (&courEl);
			    TtaDeleteTree (lastEl, doc);
			 }
		       DMatch = sMatch;
		       lastEl = myFirstSelect;
		       while (DMatch != NULL)
			 {	/* restoring the source elements */
			    TtaInsertSibling (DMatch->MatchNode->Elem, lastEl, FALSE, doc);
			    lastEl = DMatch->MatchNode->Elem;
			    DMatch = DMatch->Next;
			 }
		    }
	       }
	  }
	else
	  {
	     invEl = TtaSearchTypedElement (typeEl, SearchInTree, myFirstSelect);
	     if (invEl != NULL)
	       {
		  if (myLastSelect == NULL || TtaIsBefore (invEl, myLastSelect))
		    {		/* if the transformation failed */
		       res = FALSE;
		       /* deleting the generated structure */
		       courEl = TtaGetFirstChild (myFirstSelect);
		       while (courEl != NULL && courEl != myLastSelect)
			 {
			    TtaDeleteTree (courEl, doc);
			    courEl = TtaGetFirstChild (lastEl);
			 }
		       /* restoring the source elements */
		       DMatch = sMatch;
		       TtaInsertFirstChild (&(DMatch->MatchNode->Elem), myFirstSelect, doc);
		       lastEl = DMatch->MatchNode->Elem;
		       DMatch = DMatch->Next;
		       while (DMatch != NULL)
			 {
			    TtaInsertSibling (DMatch->MatchNode->Elem, lastEl, FALSE, doc);
			    lastEl = DMatch->MatchNode->Elem;
			    DMatch = DMatch->Next;
			 }
		    }
	       }
	  }
     }
   return res;
}

/*----------------------------------------------------------------------
  functions for transfert of non-matched descendants of matched nodes for source structure
  tree to the result HTML instance.
  ----------------------------------------------------------------------*/
/*----------------------------------------------------------------------
  FlattenAndInsertElement: tries to fllaten the structure sub-tree rooted elSource in order to 
  insert it as the first child of elDest (if sibling is false) or as its immediate right sibling (if 
  sibling is true) returns the number of higer-level nodes created in nbCreated and the 
  rightmost of them in lastCreated.
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean    FlattenAndInsertElement (Element elSource, Element elDest, boolean sibling, Document doc, Element * lastCreated, int *nbCreated)
#else
static boolean    FlattenAndInsertElement  (elSource, elDest, sibling, doc, lastCreated, nbCreated)
Element             elSource;
Element             elDest;
boolean             sibling;
Document            doc;
Element            *lastCreated;
int                *nbCreated;

#endif
{
   Element             elCour, elPreced, elNext;
   int                 nbc;
   boolean             res;

   *nbCreated = 0;
   if (TtaIsLeaf (TtaGetElementType (elSource)))
     {			
	/* if the source element is a leaf, it is impossible to flatten it more */
	lastCreated = NULL;
	return FALSE;
     }
   res = TRUE;
   elCour = TtaGetFirstChild (elSource);
   elPreced = NULL;
   if (sibling)
      elPreced = elDest;
   while (elCour != NULL && res)
     {		
       	/* for each child of the source element */
	elNext = elCour;
	TtaNextSibling (&elNext);
	TtaRemoveTree (elCour, doc);
	if (elPreced != NULL)
	   TtaInsertSibling (elCour, elPreced, FALSE, doc);
	else
	   TtaInsertFirstChild (&elCour, elDest, doc);
	res = !(boolean) TtaGetErrorCode ();
	if (res)
	  {
	     elPreced = elCour;
	     *lastCreated = elCour;
	     (*nbCreated)++;
	  }
	else
	  {
	     if (elPreced != NULL)
		res =  FlattenAndInsertElement(elCour, elPreced, TRUE, doc, lastCreated, &nbc);
	     else
		res =  FlattenAndInsertElement(elCour, elDest, FALSE, doc, lastCreated, &nbc);
	     elPreced = *lastCreated;
	     (*nbCreated) += nbc;
	  }
	elCour = elNext;
     }
   if (!res && TtaIsConstant (TtaGetElementType (elSource)))
      /* we don't mind about failure when inserting constants */
      return TRUE;
   else
      return res;
}
/*----------------------------------------------------------------------
   RankedInsertion


   Inserts an element as a given child or sibling of an element, creating a descendance 
   between the two elements if necesary. 

   Parameters:
   newEl: the element to be inserted
   parent: the element in which newEl will be inserted
   prevEl : the element after which newEl will be inserted
   rank:  rank of newEl 
   doc: document in which newwEl have to be inserted
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static boolean      RankedInsertion (Element newEl, Element parent, Element prevEl, int *rank, Document doc)
#else
static boolean      RankedInsertion (newEl, parent, prevEl, *rank, doc)
Element             newEl;
Element             parent;
Element             prevEl;
int                 rank;
Document            doc;

#endif
{
   ElementType         elemTypeNew;
   Element             elFirst, elLast, elPrevious, elCour, elem;
   int                 i;
   boolean             result;


   if (prevEl == NULL && parent == NULL)
      return FALSE;
   result = TRUE;
   TtaSetErrorMessages (0);

   elCour = NULL;
   if (*rank > 1 || prevEl != NULL)
     {
	/* if the element have not to be inserted as the first child of parent */
	if (prevEl == NULL)
	  {
	     elCour = TtaGetFirstChild (parent);
	     i = 2;
	  }
	else
	  {
	     elCour = prevEl;
	     i = 1;
	  }
	/* search for the previous sibling of the inserted element */
	for (; i < *rank && elCour != NULL; i++)
	   TtaNextSibling (&elCour);
	/* if there is not enough children of parent, */
	/*insert element as the next sibling of the last child of parent */
	if (elCour == NULL && parent != NULL)
	   elCour = TtaGetLastChild (parent);
	else if (elCour == NULL)
	   elCour = TtaGetLastChild (TtaGetParent (prevEl));
     }
   *rank = 1;
   if (elCour != NULL)
     {
	/* the newEl has to be inserted as the next sibling of elCour */
	TtaInsertSibling (newEl, elCour, FALSE, doc);
     }
   else
      /* the newEl has to be inserted as the first child of parent */
      TtaInsertFirstChild (&newEl, parent, doc);

   if (TtaGetErrorCode ())
     {
	/* the newEl has not been inserted */
	/* create a descendance between the parent and the newEl */
	elemTypeNew = TtaGetElementType (newEl);
	if (parent == NULL)
	   parent = TtaGetParent (elCour);
	elLast = TtaCreateDescent (doc, parent, elemTypeNew);
	if (elLast != NULL && !TtaGetErrorCode ())
	  {
	     /* A descendance has been created */
	     elFirst = TtaGetFirstChild (parent);
	     while (elFirst != NULL && !TtaIsAncestor (elLast, elFirst))
	       {
		  TtaNextSibling (&elFirst);
	       }
	     if (elFirst != NULL)
	       {
		  TtaRemoveTree (elFirst, doc);
		  if (elCour != NULL)
		     TtaInsertSibling (elFirst, elCour, FALSE, doc);
		  else
		     TtaInsertFirstChild (&elFirst, parent, doc);
		  if (TtaGetErrorCode ())
		    {
		       TtaDeleteTree (elFirst, doc);
		       result = FALSE;
		    }
		  else
		    {
		       elPrevious = TtaGetParent (elLast);
		       TtaDeleteTree (elLast, doc);
		       TtaInsertFirstChild (&newEl, elPrevious, doc);
		       if (TtaGetErrorCode ())
			 {
			    TtaDeleteTree (elFirst, doc);
			    result = FALSE;
			 }
		    }
	       }
	  }
	else
	   result = FALSE;
     }
   if (result == FALSE)
      /* impossible to create a descendance to the element */
      /* trying to flatten the element to be transferred */
      /* this is the worst case */
      if (elCour == NULL)
	 result = FlattenAndInsertElement (newEl, parent, FALSE, doc, &elem, rank);
      else
	 result = FlattenAndInsertElement (newEl, elCour, TRUE, doc, &elem, rank);
   return result;
}

/*----------------------------------------------------------------------
 CopySubTreeChildren: copies the descendants of matched nodes from source tree to result 
  instance
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CopySubTreeChildren (Element father, Document doc)
#else
static void         CopySubTreeChildren (father, doc)
Element             father;
Document            doc;

#endif
{
   Element             elCour, elOriginal;
   AttributeType       attrType;
   Attribute           attrFound;
   char                label[10];
   int                 l, rank, idf, delta;

   attrType.AttrSSchema = TtaGetDocumentSSchema (doc);
   attrType.AttrTypeNum = HTML_ATTR_Ghost_restruct;
   elCour = TtaGetFirstChild (father);
   while (elCour != NULL)
     {
	CopySubTreeChildren (elCour, doc);
	attrFound = TtaGetAttribute (elCour, attrType);
	if (attrFound != NULL)
	  {
	     l = 10;
	     TtaGiveTextAttributeValue (attrFound, label, &l);
	     TtaRemoveAttribute (elCour, attrFound, doc);
	     idf = atoi (label);
	     elOriginal = NULL;
	     rank = FindListSubTree (idf, &elOriginal);
	     delta = 0;
	     while (elOriginal != NULL)
	       {
		  TtaRemoveTree (elOriginal, doc);
		  RankedInsertion (elOriginal, elCour, NULL, &rank, doc);
		  delta += rank - 1;
		  rank = delta + FindListSubTree (idf, &elOriginal);
	       }
	  }
	TtaNextSibling (&elCour);
     }
}

/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         CopySubTrees (strNode * Root, Document doc)
#else
static void         CopySubTrees (Root, doc)
strNode            *Root;
Document            doc;

#endif
{
   char                label[10];
   int                 l, idf, rank, delta;
   AttributeType       attrType;
   Attribute           attrFound;
   Element             elCour, elLast, elOriginal;

   elCour = myFirstSelect;	
   elLast = myLastSelect;	
   if (isClosed)
      TtaNextSibling (&elCour);
   else
      elCour = TtaGetFirstChild (myFirstSelect);
   attrType.AttrSSchema = TtaGetDocumentSSchema (doc);
   attrType.AttrTypeNum = HTML_ATTR_Ghost_restruct;
   while (elCour != NULL && elCour != elLast)
     { 	/* Copies the children of elCour into the result instance */		
	CopySubTreeChildren (elCour, doc);
	/* searches for a ZZGHOST ettribute */
	attrFound = TtaGetAttribute (elCour, attrType);	
	if (attrFound != NULL)
	  {
	     l = 10;
	     TtaGiveTextAttributeValue (attrFound, label, &l);
	     TtaRemoveAttribute (elCour, attrFound, doc);
	     idf = atoi (label);
	     elOriginal = NULL;
	     /* serches and inserts the children of element elOriginal */
	     rank = FindListSubTree (idf, &elOriginal);
	     delta = 0;
	     while (elOriginal != NULL)
	       {
		  TtaRemoveTree (elOriginal, doc);
		  RankedInsertion (elOriginal, elCour, NULL, &rank, doc);
		  delta += rank - 1;
		  rank = delta + FindListSubTree (idf, &elOriginal);
	       }
	  }
	TtaNextSibling (&elCour);
     }
   /* insert the subtrees as children of the root */
   elOriginal = NULL;
   elCour = myFirstSelect;
   rank = FindListSubTree (0, &elOriginal);
   delta = 0;
   while (elOriginal != NULL)
     {
	TtaRemoveTree (elOriginal, doc);
	if (isClosed)
	   RankedInsertion (elOriginal, NULL, elCour, &rank, doc);
	else
	   RankedInsertion (elOriginal, elCour, NULL, &rank, doc);
	delta += rank - 1;
	rank = delta + FindListSubTree (0, &elOriginal);
     }
}

/*----------------------------------------------------------------------
   PutBeginTag,PutEndTag,TransfertNode 
   fill the HTML buffer with the generated nodes 
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static boolean      PutInHtmlBuffer (char *s)
#else
static boolean      PutInHtmlBuffer (s)
char               *s;

#endif
{
   if ((szHTML += strlen (s)) > BUFFER_LEN )
     {
	fprintf (stderr, "increase BUFFER_LEN");
	return FALSE;
     }
   else
     {
	strcat (bufHTML, s);
	return TRUE;
     }
}

/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         PutBeginTag (strNodeDesc * ND, strNode * TN)
#else
static void         PutBeginTag (ND, TN)
strNodeDesc           *ND;
strNode            *TN;

#endif
{
   strAttrDesc           *AD;
   strGenStack          *NS;
   char               *attrValue, *tag;
   strNode            *ancestor;
   boolean             found;
   AttributeType       attrType;
   Attribute           attr;
   int                 l, attrKind;

   attrType.AttrSSchema = TtaGetDocumentSSchema (TransDoc);
   attrValue = TtaGetMemory (NAME_LENGTH);
   tag = TtaGetMemory (NAME_LENGTH);
   /* push the new tag on the generation stack */
   generationStack[topGenerStack]->Nbc++;
   NS = (strGenStack *) TtaGetMemory (sizeof (strGenStack));
   NS->Tag = TtaGetMemory (NAME_LENGTH);
   strcpy (NS->Tag, ND->Tag);
   /* create a ghost attribute with the identifier of the node */
   NS->Idf = idfCounter++;
   NS->Nbc = 0;
   NS->Attributes = (strAttrDesc *) TtaGetMemory (sizeof (strAttrDesc));
   NS->Attributes->NameAttr = TtaGetMemory (NAME_LENGTH);
   strcpy (NS->Attributes->NameAttr, "ZZGHOST");
   NS->Attributes->IsInt = TRUE;
   NS->Attributes->IsTransf = FALSE;
   NS->Attributes->IntVal = NS->Idf;
   NS->Attributes->Next = ND->Attributes;
   generationStack[++topGenerStack] = NS;

   /* writing the tag name */
   PutInHtmlBuffer ("<");
   PutInHtmlBuffer (NS->Tag);

   AD = NS->Attributes;
   /* wrting the attributes */
   while (AD != NULL)
     {
	if (AD->IsTransf)
	  {			/* transfer attribute */
	     ancestor = TN;
	     found = FALSE;
	     while (!found && ancestor != NULL)
	       {		/* searching for source element (in current element ancestors) */
		  found = (!strcmp (ancestor->MatchSymb->SymbolName, 
				    AD->AttrTag)
			|| !strcmp (ancestor->MatchSymb->Tag, AD->AttrTag));
		  if (!found)
		     ancestor = ancestor->Parent;
	       }
	     if (found)
	       {		/* searching for an ancestor of the source element which have the wanted attribute  */
		  if (ancestor != NULL)
		    {
		       strcpy (tag, GITagNameByType ((int) (TtaGetElementType (ancestor->Elem).ElTypeNum)));
		       attrType.AttrTypeNum = MapThotAttr (AD->AttrAttr, tag);
		    }
		  attr = NULL;
		  found = FALSE;
		  while (!found && ancestor != NULL)
		    {
		       if (attrType.AttrTypeNum != -1)
			  attr = TtaGetAttribute (ancestor->Elem, attrType);
		       found = (attr != NULL);
		       if (!found)
			 {
			    ancestor = ancestor->Parent;
			    if (ancestor != NULL)
			      {
				 strcpy (tag, GITagNameByType ((int) (TtaGetElementType (ancestor->Elem).ElTypeNum)));
				 attrType.AttrTypeNum = MapThotAttr (AD->AttrAttr, tag);
			      }
			 }
		    }
		  if (found)
		    {		/* the attribute has been found, writing the attribute name */
		       PutInHtmlBuffer (" ");
		       PutInHtmlBuffer (AD->AttrAttr);
		       PutInHtmlBuffer ("=");
		       /* writing the attribute value */
		       TtaGiveAttributeType (attr, &attrType, &attrKind);
		       if (attrKind == 2)
			 {	/* text attribute */
			    l = TtaGetTextAttributeLength (attr);
			    TtaGiveTextAttributeValue (attr, attrValue, &l);
			    PutInHtmlBuffer (attrValue);
			 }
		       else
			 {	/* int attribute */
			    sprintf (attrValue, "%d", TtaGetAttributeValue (attr));
			    PutInHtmlBuffer (attrValue);
			 }
		    }
	       }
	     if (!found)
	       {
		  fprintf (stderr, "can't transfer attribute %s\n", AD->AttrAttr);
	       }
	  }
	else
	  {			/* creation of an attribute */
	     PutInHtmlBuffer (" ");
	     PutInHtmlBuffer (AD->NameAttr);
	     PutInHtmlBuffer ("=");
	     if (AD->IsInt)
	       {		/* int attribute */
		  sprintf (attrValue, "%d", AD->IntVal);
		  PutInHtmlBuffer (attrValue);
	       }
	     else
	       {		/* text attribute */
		  l = strlen (bufHTML);
		  bufHTML[l] = '"';
		  bufHTML[l + 1] = EOS;
		  szHTML++;
		  PutInHtmlBuffer (AD->TextVal);
		  l = strlen (bufHTML);
		  bufHTML[l] = '"';
		  bufHTML[l + 1] = EOS;
		  szHTML++;
	       }
	  }
	AD = AD->Next;
     }
   /* closing the tag */
   PutInHtmlBuffer (">");
   /*free the ZZGHOST attribute */
   TtaFreeMemory ((char *) NS->Attributes->NameAttr);
   TtaFreeMemory ((char *) NS->Attributes);
   NS->Attributes = NULL;
   TtaFreeMemory (attrValue);
   TtaFreeMemory (tag);
}


/*----------------------------------------------------------------------
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void PutEndTag (strGenStack * ND)
#else
static void PutEndTag (ND)
strGenStack *ND;
#endif
{
   if (strcmp (ND->Tag, "HR") && strcmp (ND->Tag, "BR"))
     {
	PutInHtmlBuffer ("</");
	PutInHtmlBuffer (ND->Tag);
	PutInHtmlBuffer (">");
     }
}

/*----------------------------------------------------------------------
   TransfertChildren : copies the children of node into the result instance 
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static void         TransfertChildren (strNode * node)
#else
static void         TransfertChildren (strNode * node)
strNode            *node;

#endif
{
   strNode            *child;

   child = node->Child;
   while (child != NULL)
     {
	if (TtaGetElementVolume (child->Elem) != 0)
	  {			/* if the element is empty: no transfert */
	     generationStack[topGenerStack]->Nbc++;
	     AddListSubTree (child->Elem,
			     generationStack[topGenerStack]->Idf,
			     generationStack[topGenerStack]->Nbc);
	  }
	child = child->Next;
     }
}

/*----------------------------------------------------------------------
   TransfertNode : copies a node and its content into the result instance 
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static void         TransfertNode (strNode * node, boolean inplace)
#else
static void         TransfertNode (node, inplace)
strNode            *node;
boolean             inplace;

#endif
{
   if (TtaGetElementVolume (node->Elem) != 0)
     {				/* if the element is empty: no transfert */
	if (!inplace)
	  /* closing previously generated elements */
	   while (topGenerStack >= lastRulePlace)
	     {
		PutEndTag (generationStack[topGenerStack]);
		TtaFreeMemory (generationStack[topGenerStack]->Tag);
		TtaFreeMemory ((char *) generationStack[topGenerStack]);
		topGenerStack--;
	     }
	generationStack[topGenerStack]->Nbc++;
	AddListSubTree (node->Elem,
			generationStack[topGenerStack]->Idf,
			generationStack[topGenerStack]->Nbc);
     }
}


/*----------------------------------------------------------------------
   Fonctions de transformation par regles            
  ----------------------------------------------------------------------*/

#ifdef __STDC__
static void         TransformNode (strMatchChildren * sm);
#else
static void         TransformNode (sm);
#endif


#ifdef __STDC__
static void         ApplyTransChild (strMatchChildren * smc)
#else
static void         ApplyTransChild (smc)
strMatchChildren   *smc;

#endif
{
   strMatchChildren   *smc2;
   strMatch           *sm;
   boolean             found;

   smc2 = smc;
   while (smc2 != NULL)
     {
	smc2->MatchNode->IsTrans = TRUE;
	if (smc2->MatchSymb->Rule == NULL)
	  {
	    /* no transformation rule for the node, searches if a child has been matched */
	     sm = smc2->MatchNode->Matches;
	     found = FALSE;
	     while (!found && sm != NULL)
	       {
		  found = (sm->MatchSymb == smc2->MatchSymb);
		  if (!found)
		     sm = sm->Next;
	       }
	     if (found)
	       {
		 /* at least one child has been matched, applying the transformation */
		 /* to the children */
		  ApplyTransChild (sm->MatchChildren);
	       }
	     else
	       { /* there is no matching: transferring the node to destination instance */
		  TransfertNode (sm->MatchNode, FALSE);
	       }
	  }
	else
	  {  /* there is a transformation rule relative to the matched symbol */
	    /* applying the rule */
	     TransformNode (smc2);
	  }
	smc2 = smc2->Next;
     }
}


/*----------------------------------------------------------------------
  TransformNode : transforms a matched node acconding to the rule relative to the 
  matched symbol
  ---------------------------------------------------------------------*/
#ifdef __STDC__
static void         TransformNode (strMatchChildren * sm)
#else
static void         TransformNode (sm)
strMatchChildren   *sm;

#endif
{
   int                 courNode;
   strMatch           *sm2;
   strNodeDesc        *RNodeCour;
   boolean             stop, sonsMatch;

   sm2 = sm->MatchNode->Matches;
   sonsMatch = FALSE;
   while (sm2 != NULL && !sonsMatch)
     {
	sonsMatch = (sm2->MatchSymb == sm->MatchSymb && sm2->MatchChildren != NULL);
	if (!sonsMatch)
	   sm2 = sm2->Next;
     }
   /* sonsMatch is true if there is at least one matched node in the children of source node */
   sm->MatchNode->MatchSymb = sm->MatchSymb;
   courNode = 1;
   RNodeCour = sm->MatchSymb->Rule->OptionNodes;
   stop = (RNodeCour == NULL || courNode > topGenerStack);
   while (!stop)
     { /* for each optional tag in the rule */
	if (!strcmp (generationStack[courNode]->Tag, RNodeCour->Tag))
	  {  /* does nothing if the tag is already present in the destination instance */
	     RNodeCour = RNodeCour->Next;
	     courNode++;
	     stop = (RNodeCour == NULL || courNode > topGenerStack);
	  }
	else
	  {
	    /* a new branch have to be created in the destination */
	     stop = TRUE;
	  }
     }

   while (topGenerStack >= courNode)
     { /* closes the opened tags (on generation stack) */
	PutEndTag (generationStack[topGenerStack]);
	TtaFreeMemory (generationStack[topGenerStack]->Tag);
	TtaFreeMemory ((char *) generationStack[topGenerStack]);
	topGenerStack--;
     }

   while (RNodeCour != NULL)
     {/* generates optional nodes not already present */
	PutBeginTag (RNodeCour, sm->MatchNode);
	courNode++;
	RNodeCour = RNodeCour->Next;
     }

   lastRulePlace = courNode;
   RNodeCour = sm->MatchSymb->Rule->NewNodes;

   while (RNodeCour != NULL && strcmp (RNodeCour->Tag, "*"))
     { /* generates the new nodes */
	PutBeginTag (RNodeCour, sm->MatchNode);
	courNode++;
	RNodeCour = RNodeCour->Next;
     }
   if (RNodeCour != NULL && !strcmp (RNodeCour->Tag, "*"))
     {
	TransfertNode (sm->MatchNode, TRUE);
     }
   /* process the children */
   else if (sonsMatch)
     {			
	ApplyTransChild (sm2->MatchChildren);
     }
   else
     {			
	TransfertChildren (sm->MatchNode);
     }
}

/*----------------------------------------------------------------------
  ApplyTransformation: applies the transformation based on sm matching descriptors
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean      ApplyTransformation (strMatch * sm, Document doc)
#else
static boolean      ApplyTransformation (sm, doc)
strMatch           *sm;
Document            doc;

#endif
{
   strGenStack          *ND;
   strMatchChildren   *DMatch;
   boolean             res;

   res = FALSE;
   idfCounter = 0;
   ND = (strGenStack *) TtaGetMemory (sizeof (strGenStack));
   ND->Tag = TtaGetMemory (NAME_LENGTH);
   ND->Attributes = NULL;
   ND->Idf = idfCounter++;
   ND->Nbc = 0;

   if (sm->MatchChildren != NULL)
     {
	/* initialize the transformation stack */
	DMatch = sm->MatchChildren;
	topGenerStack = 0;
	strcpy (ND->Tag, DMatch->MatchNode->Parent->Tag);
	generationStack[0] = ND;
	lastRulePlace = 1;
	szHTML = 0;
	bufHTML = TtaGetMemory (BUFFER_LEN);
	strcpy (bufHTML, "");
	/* applying the transformation */
	ApplyTransChild (sm->MatchChildren);
	while (topGenerStack > 0)
	  {
	     PutEndTag (generationStack[topGenerStack]);
	     TtaFreeMemory (generationStack[topGenerStack]->Tag);
	     TtaFreeMemory ((char *) generationStack[topGenerStack]);
	     topGenerStack--;
	  }
	/* parsing the produced structure */
	res = StartHtmlParser (sm->MatchChildren, doc);

	TtaFreeMemory (bufHTML);
     }
   if (res)
     {	
       /* if the html parsing was succesful */
       /* transfers the unchanged subtrees */
       CopySubTrees (strMatchEnv.SourceTree, doc);
       /* deletes the source structure elements */
	DMatch = sm->MatchChildren;
	TtaSetErrorMessages (0);
	while (DMatch != NULL)
	  {
	     if (DMatch->MatchNode->Elem != NULL &&
		 FindListSTreeByLabel (TtaGetElementLabel (DMatch->MatchNode->Elem)) == NULL)
		TtaDeleteTree (DMatch->MatchNode->Elem, doc);
	     DMatch = DMatch->Next;
	  }
	TtaSetErrorMessages (1);
     }
   return res;
}

/*----------------------------------------------------------------------
   CheckSelection : checks if all the selected element are at the same level. Extends the selction  
   to an element if all its children are selected                                       
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean      CheckSelectionLevel (Document doc)
#else
static boolean      CheckSelectionLevel (doc)
Document            doc;

#endif
{
   Element             prevFirst, parentFirst, nextLast, parentLast;
   boolean             result;

   TtaGiveFirstSelectedElement (doc, &origFirstSelect, &ffc, &flc);
   TtaGiveLastSelectedElement (doc, &origLastSelect, &lfc, &llc);
   myFirstSelect = origFirstSelect;
   myLastSelect = origLastSelect;
   parentFirst = NULL;
   maxSelDepth = 0;
   if (myFirstSelect != myLastSelect)
     {
	if (myFirstSelect != NULL && ffc <= 1)
	  {			/* searching for the first selected element */
	     prevFirst = myFirstSelect;
	     TtaPreviousSibling (&prevFirst);
	     parentFirst = TtaGetParent (myFirstSelect);
	     while (parentFirst != NULL && prevFirst == NULL && TtaIsBefore (parentFirst, myLastSelect))
	       {
		  myFirstSelect = parentFirst;
		  prevFirst = myFirstSelect;
		  TtaPreviousSibling (&prevFirst);
		  parentFirst = TtaGetParent (myFirstSelect);
	       }
	  }
	if (myLastSelect != NULL && (llc == 0 || (llc > 0 && llc >= TtaGetTextLength (myLastSelect))))
	  {			/* searching for the last selected element */
	     nextLast = myLastSelect;
	     TtaNextSibling (&nextLast);
	     parentLast = TtaGetParent (myLastSelect);
	     while (parentLast != NULL && nextLast == NULL && TtaIsBefore (myFirstSelect, parentLast))
	       {
		  myLastSelect = parentLast;
		  nextLast = myLastSelect;
		  TtaNextSibling (&nextLast);
		  parentLast = TtaGetParent (myLastSelect);
	       }
	  }
     }
   else
     {
	prevFirst = TtaGetFirstChild (myFirstSelect);
	nextLast = TtaGetLastChild (myFirstSelect);
	while (prevFirst != NULL && prevFirst == nextLast)
	  {
	     myFirstSelect = prevFirst;
	     prevFirst = TtaGetFirstChild (myFirstSelect);
	     nextLast = TtaGetLastChild (myFirstSelect);
	  }
	if (prevFirst != NULL)
	  {
	     myFirstSelect = prevFirst;
	     myLastSelect = nextLast;
	  }
	else
	  {
	     myLastSelect = myFirstSelect;
	  }
	parentFirst = parentLast = TtaGetParent (myFirstSelect);
     }
   mySelect = NULL;
   result = myFirstSelect != NULL && (parentFirst == parentLast);
   if (result && parentFirst != NULL)
     {		
       /* if all selected elements are at the same level, */
       /* checking if ancestors have any sibling */
       /* if it is not the case, they become the first selected element */
	nextLast = myLastSelect;
	prevFirst = myFirstSelect;
	do
	   TtaNextSibling (&nextLast);
	while (nextLast != NULL && GITagName (nextLast) == NULL);
	do
	   TtaPreviousSibling (&prevFirst);
	while (prevFirst != NULL && GITagName (prevFirst) == NULL);
	while (parentFirst != NULL &&
	       TtaGetElementType (parentFirst).ElTypeNum != HTML_EL_BODY &&
	       nextLast == NULL && prevFirst == NULL)
	  {
	     maxSelDepth++;
	     mySelect = parentFirst;
	     parentFirst = TtaGetParent (parentFirst);
	     if (parentFirst != NULL)
	       {
		  nextLast = mySelect;
		  prevFirst = mySelect;
		  do
		     TtaNextSibling (&nextLast);
		  while (nextLast != NULL && GITagName (nextLast) == NULL);
		  do
		     TtaPreviousSibling (&prevFirst);
		  while (prevFirst != NULL && GITagName (prevFirst) == NULL);
	       }
	  }
     }
   return result;
}

/*----------------------------------------------------------------------
   Give the next selected element, accordingly  to extension given by CheckSelectionLevel 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static void         MyNextSelectedElement (Document doc, Element * elSelect)
#else
static void         MyNextSelectedElement (doc, elSelect)
Document            doc;
Element            *elSelect;

#endif
{
   Element             elFirst;
   int                 fc, lc;

   if (*elSelect == NULL || *elSelect == mySelect || *elSelect == myLastSelect)
     {				/* if the selection is an unique element, or elSelect is the last */
	*elSelect = NULL;
     }
   else
     {
	if (*elSelect == myFirstSelect)
	  {
	     TtaGiveFirstSelectedElement (doc, &elFirst, &fc, &lc);
	     if (elFirst == myFirstSelect)
		TtaGiveNextSelectedElement (doc, &elFirst, &fc, &lc);
	     else
		while (elFirst != NULL && TtaIsAncestor (elFirst, myFirstSelect))
		   TtaGiveNextSelectedElement (doc, &elFirst, &fc, &lc);
	  }
	else
	  {
	     elFirst = *elSelect;
	     TtaGiveNextSelectedElement (doc, &elFirst, &fc, &lc);
	  }
	if (elFirst != NULL && TtaIsAncestor (elFirst, myLastSelect))
	   *elSelect = myLastSelect;
	else
	   *elSelect = elFirst;
     }
}
/*----------------------------------------------------------------------
   IsValidHtmlChild(element, tag)                                       
   returns TRUE if the tag is valid as a direct descendant of an element of type elType 
   
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean      IsValidHtmlChild (ElementType elemType, char *tag, char *prevtag)
#else
static boolean      IsValidHtmlChild (elemType, tag, prevtag)
Element             element;
char               *tag;
char               *prevtag;

#endif
{

   ElementType         elemTypeChild;
   int                 cardinal, prevTypeNum, typeNum, i;
   ElementType 	      *subTypes;
   boolean             result, found;
   Construct           constOfType;

   result = FALSE;
   elemTypeChild.ElSSchema = elemType.ElSSchema;
   cardinal = TtaGetCardinalOfType(elemType);
   subTypes = (ElementType *)TtaGetMemory(cardinal * sizeof(ElementType));
   TtaGiveConstructorsOfType(&subTypes,&cardinal,elemType);
   constOfType = TtaGetConstructOfType(elemType);
   GIType (tag, &typeNum);
   if (typeNum == 0)
      return FALSE;
   switch (constOfType)
      {
            case ConstructIdentity:
               if (subTypes[0].ElTypeNum == typeNum)
		  result = TRUE;
	       else if (!strcmp (GITagNameByType (subTypes[0].ElTypeNum), "???"))
		  /* search if tag can beinserted as a child of the identity */
		  result = IsValidHtmlChild (subTypes[0], tag, "");
	       break;

	    case ConstructList:
	       if (subTypes[0].ElTypeNum == typeNum)
		  result = TRUE;
	       else if (!strcmp (GITagNameByType (subTypes[0].ElTypeNum), "???"))
                  result = IsValidHtmlChild (subTypes[0], tag, "");
	       break;

	    case ConstructChoice:
	       for (i = 0; !result && i < cardinal; i++)
		 {
		    if (subTypes[i].ElTypeNum == typeNum)
		       result = TRUE;
		    else if (!strcmp (GITagNameByType (subTypes[i].ElTypeNum),"???"))
 		       result = IsValidHtmlChild (subTypes[i], tag, "");
		 }
	       break;

	    case ConstructOrderedAggregate:
	       found = (!strcmp (prevtag, ""));
	       GIType (prevtag, &prevTypeNum);
	       found = (prevTypeNum == 0);
	       /* searches the rule of previous sibling */
	       for (i = 0; !found && i < cardinal; i++)
		 {
		    if (prevTypeNum == subTypes[i].ElTypeNum)
		       found = TRUE;
		    else if (strcmp (GITagNameByType (subTypes[i].ElTypeNum),"???"))
		       i = cardinal;
		 }
	       if (found)
		 {
		    while (!result && i < cardinal)
		      {
			 if (typeNum == subTypes[i].ElTypeNum)
			    result = TRUE;
			 else if (!strcmp (GITagNameByType (subTypes[i].ElTypeNum), "???"))
			    result = IsValidHtmlChild (subTypes[i], tag, "");
			 if (!result)
			    if (TtaIsOptionalInAggregate(i,elemType)) 
			       i++;
			    else
			       i = cardinal;
		      }
		 }
	       break;
	    case ConstructUnorderedAggregate:
	       while (!result && i < cardinal)
		 {
		    if (typeNum == subTypes[i].ElTypeNum)
		       result = TRUE;
		    else if (!strcmp (GITagNameByType (subTypes[i].ElTypeNum), "???"))
		       result = IsValidHtmlChild (subTypes[i], tag, "");
		    if (!result)
	               if (TtaIsOptionalInAggregate(i,elemType)) 
			  i++;
		       else
			  i = cardinal;
		 }
	    case ConstructConstant:
	    case ConstructReference:
	    case ConstructBasicType:
	    case ConstructNature:
	    case ConstructPair:
            case ConstructError:
	       break;
	 }
   TtaFreeMemory((char *)subTypes);
   return result;
}

/*----------------------------------------------------------------------
   CheckValidTransRoot                                                  
   checks if the higher-level generated elements are possible children of the   
   transformation root element                                          
  ----------------------------------------------------------------------*/
#ifdef __STDC__
static boolean      CheckValidTransRoot (strMatch * sm, ElementType elemTypeRoot, char *prevTag)
#else
static boolean      CheckValidTransRoot (sm, elemTypeRoot, prevTag)
strMatch           *sm;
ElementType         elemTypeRoot;
char               *prevTag;

#endif
{
   strMatchChildren   *smc;
   strMatch           *sm2;
   strNodeDesc           *node;
   boolean             result, sonsMatch;
   char               *curTag;

   curTag = TtaGetMemory (NAME_LENGTH);
   result = TRUE;
   smc = sm->MatchChildren;
   while (result && smc != NULL)
     {
	if (smc->MatchSymb->Rule == NULL)
	  {			/* there is no rule for the current node */
	     sm2 = smc->MatchNode->Matches;
	     sonsMatch = FALSE;
	     while (sm2 != NULL && !sonsMatch)
	       {		/* checks if the children of the node have been matched */
		  sonsMatch = (sm2->MatchSymb == smc->MatchSymb && sm2->MatchChildren != NULL);
		  if (!sonsMatch)
		     sm2 = sm2->Next;
	       }
	     if (!sonsMatch)
	       {		/* if the children of the node have not been matched */
		  /* checks if the node can be transferred in the destination */
		  if (TtaGetElementVolume (smc->MatchNode->Elem) != 0)
		    {		/* if the element is empty, it is ignored in transformation */
		       if (strcmp (prevTag, smc->MatchNode->Tag))
			  result = IsValidHtmlChild (elemTypeRoot,
						     smc->MatchNode->Tag,
						     prevTag);
		       strcpy (prevTag, smc->MatchNode->Tag);
		    }
	       }
	     else
	       {		/* if they have been, checks the elements generated by these children */
		  result = CheckValidTransRoot (sm2, elemTypeRoot, prevTag);
	       }
	  }
	else
	  {			/* there is a rule for the current node */
	     node = smc->MatchSymb->Rule->OptionNodes;
	     if (node != NULL)
	       {		/* if there is at least one place node */
		  if (strcmp (prevTag, node->Tag))
		    {
		       result = IsValidHtmlChild (elemTypeRoot,
						  node->Tag,
						  prevTag);
		       strcpy (prevTag, smc->MatchNode->Tag);
		    }
	       }
	     else
	       {
		  node = smc->MatchSymb->Rule->NewNodes;
		  if (node != NULL)
		    {
		       if (!strcmp (node->Tag, "*"))
			  strcpy (curTag, smc->MatchNode->Tag);
		       else
			  strcpy (curTag, node->Tag);
		       result = IsValidHtmlChild (elemTypeRoot,
						  curTag,
						  prevTag);
		       strcpy (prevTag, curTag);
		    }
		  else		/*error */
		     result = FALSE;
	       }
	  }
	if (result)
	   smc = smc->Next;
     }
   TtaFreeMemory (curTag);
   return result;
}

/*----------------------------------------------------------------------
   callback of the transformation selection menu 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                TransCallbackDialog (int ref, int typedata, char *data)
#else
void                TransCallbackDialog (ref, typedata, data)
int                 ref;
int                 typedata;
char               *data;

#endif
{
   int                 val;
   DisplayMode         oldDisplayMode;

   val = (int) data;
   switch (ref - TransBaseDialog)
	 {
	    case TransMenu:
	       oldDisplayMode = TtaGetDisplayMode (TransDoc);
	       TtaSetDisplayMode (TransDoc, NoComputedDisplay);
	       TtaSelectElement (TransDoc, NULL);
	       resultTrans = ApplyTransformation (menuTrans[val], TransDoc);
	       if (!resultTrans)
		 {
		    /* transformation has failed, restoring the old selection */
		    if (ffc == 0 && flc == 0)
		       TtaSelectElement (TransDoc, origFirstSelect);
		    else
		       TtaSelectString (TransDoc, origFirstSelect, ffc, flc);
		    TtaExtendSelection (TransDoc, origLastSelect, llc);
		    /* display an error message */
		    TtaSetStatus (TransDoc, 1, TtaGetMessage (AMAYA, AM_TRANS_FAILED), NULL);
		 }
	       else
		 {
		    /* transformation was succesful, checking the thot tree */
		    InitializeParser (TtaGetMainRoot (TransDoc), TRUE, TransDoc);
		    TtaSetStructureChecking (0, TransDoc);
		    CheckAbstractTree (NULL);
		    TtaSetStructureChecking (1, TransDoc);
		    /* selecting the new elements */
		    if (myLastSelect == NULL)
		       if (!isClosed)
			  myLastSelect = TtaGetLastChild (myFirstSelect);
		       else
			  myLastSelect = TtaGetLastChild (TtaGetParent (myFirstSelect));
		    else
		       TtaPreviousSibling (&myLastSelect);
		    if (isClosed)
		       TtaNextSibling (&myFirstSelect);
		    else
		       myFirstSelect = TtaGetFirstChild (myFirstSelect);
		    TtaSelectElement (TransDoc, myFirstSelect);
		    if (myLastSelect != NULL && TtaIsBefore (myFirstSelect, myLastSelect))
		       TtaExtendSelection (TransDoc, myLastSelect, 0);
		 }
	       TtaSetDisplayMode (TransDoc, oldDisplayMode);
	       FreeMatchEnv ();
	       break;
	 }
}

/*----------------------------------------------------------------------
   callback of  the tranform entry of type menu 
  ----------------------------------------------------------------------*/
#ifdef __STDC__
void                TransformType (Document doc, View view)
#else
void                TransformType (doc, view)
Document            doc;
View                view;
#endif
{
   Element             elemSelect;
   int                 i, j, k;
   char               *menuBuf, *tag;
   strMatch           *sm;
   StructureTree             node;

   strMatchEnv.SourceTree = NULL;
   strMatchEnv.ListSubTrees = NULL;
   resultTrans = FALSE;
   TransDoc = doc;
   /* context initialisation -- parses the transformation file */
   if (CheckSelectionLevel (TransDoc) && ppStartParser ("HTML"))
     {
	/* allocates temporary text buffer */
	menuBuf = TtaGetMemory (MAX_LENGTH);
	/* builds the source structure tree */
	maxMatchDepth = strMatchEnv.MaxDepth + maxSelDepth;
	strMatchEnv.SourceTree = (StructureTree) NewNode ("Root");
	if (mySelect != NULL)
	  {
	     (strMatchEnv.SourceTree)->Elem = TtaGetParent (mySelect);
	     BuildStructureTree (mySelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
	  }
	else
	  {
	     (strMatchEnv.SourceTree)->Elem = TtaGetParent (myFirstSelect);
	     elemSelect = myFirstSelect;
	     while (elemSelect != NULL)
	       {
		  BuildStructureTree (elemSelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
		  MyNextSelectedElement (TransDoc, &elemSelect);
	       }
	  }
	/* pattern matching */
	PostfixSearch (strMatchEnv.SourceTree, MatchNode);

	/* construct the popup menu with the result of matching*/
	node = strMatchEnv.SourceTree;
	i = 0;
	j = 0;
	tag = TtaGetMemory (NAME_LENGTH);
	do
	  {			/* for each node above the first selected */
	     sm = node->Matches;
	     while (sm != NULL)
	       {		/* for each matching of the node */
		  if (!strcmp (sm->MatchSymb->Tag, "pattern_root"))
		    {		/* if it is matching a pattern root : insert the transformation name */
		       /* in the menu buffer */
		       strcpy (tag, "");
		       if (CheckValidTransRoot (sm,
				      TtaGetElementType (sm->MatchNode->Elem),
						tag))
			 {
			    for (k = 0; k < i && strcmp (menuTrans[k]->MatchSymb->SymbolName, sm->MatchSymb->SymbolName); k++) ;
			    if (k == i)
			      {
				 sprintf (&menuBuf[j], "%s%s", "B", sm->MatchSymb->SymbolName);
				 j += strlen (&menuBuf[j]) + 1;
				 menuTrans[i++] = (strMatch *) sm;
			      }
			 }
		    }
		  sm = sm->Next;
	       }
	     node = node->Child;
	  }
	while (node != NULL &&
	       (!TtaIsAncestor (node->Elem, myFirstSelect)));
	TtaFreeMemory (tag);
	if (i > 0)
	  {			/* if some transformations have been matched, shows the menu */
	     TtaNewPopup (TransBaseDialog + TransMenu, 0, TtaGetMessage (TRANSDIAL, TR_TRANSFORM), i, menuBuf, NULL, 'L');
	     TtaSetDialoguePosition ();
	     TtaShowDialogue (TransBaseDialog + TransMenu, TRUE);
	  }
	else
	   /* display an status message */
	   TtaSetStatus (TransDoc, 1, TtaGetMessage (AMAYA, AM_NO_TRANS), NULL);
	TtaFreeMemory (menuBuf);
     }
   else
      /* display an status message */
      TtaSetStatus (TransDoc, 1, TtaGetMessage (AMAYA, AM_NO_TRANS), NULL);
}



/*----------------------------------------------------------------------
   TransformIntoType:  Changes the type of a the selected elements into the given DestType
   Selects the transformations that produces the givem type, matches teir patterns, and
   applies the first matched transformation
  ----------------------------------------------------------------------*/
#ifdef __STDC__
boolean             TransformIntoType (ElementType resultType, Document doc)
#else
boolean             TransformIntoType (resultType, doc)
ElementType         resultType;
Document            doc;

#endif
{
   char                DestTag[20];
   boolean             ok;
   int                 i;
   Element             elemSelect;
   strTransDesc          *td;
   strSymbDesc           *sd;
   strMatch           *sm;
   StructureTree             node;
   char               *tag;

   strMatchEnv.SourceTree = NULL;
   strMatchEnv.ListSubTrees = NULL;
   resultTrans = FALSE;
   TransDoc = doc;

   /* context initialisation -- parses the transformation file */
   if (CheckSelectionLevel (TransDoc) && ppStartParser ("HTML"))
     {
	ok = FALSE;
	strcpy (DestTag, GITagNameByType (resultType.ElTypeNum));
	/* selects the transformation producing the given element type */
	td = strMatchEnv.Transformations;
	strMatchEnv.MaxDepth = 0;
	while (td != NULL)
	  {
	     if (td->DestinationTag == NULL || strcmp (td->DestinationTag, DestTag))
	       {		
		 /* the transformation does not produce the given type, it is desactived */
		  td->IsActiveTrans = FALSE;
		  sd = td->Symbols;
		  while (sd != NULL)
		    {
		       sd->IsActiveSymb = FALSE;
		       sd = sd->Next;
		    }
	       }
	     else
	       {
		 /* at least 1 transformation produces the given type */		 
		  ok = TRUE;	
		  if (td->PatDepth > strMatchEnv.MaxDepth)
		     strMatchEnv.MaxDepth = td->PatDepth;
	       }
	     td = td->Next;
	  }

	if (ok)
	  {
	     maxMatchDepth = strMatchEnv.MaxDepth + maxSelDepth;
	     /* Builds the source structure tree */

	     strMatchEnv.SourceTree = (StructureTree) NewNode ("Root");
	     if (mySelect != NULL)
	       {
		  (strMatchEnv.SourceTree)->Elem = TtaGetParent (mySelect);
		  BuildStructureTree (mySelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
	       }
	     else
	       {
		  (strMatchEnv.SourceTree)->Elem = TtaGetParent (myFirstSelect);
		  elemSelect = myFirstSelect;
		  while (elemSelect != NULL)
		    {
		       BuildStructureTree (elemSelect, TransDoc, strMatchEnv.SourceTree, maxMatchDepth, 0);
		       MyNextSelectedElement (TransDoc, &elemSelect);
		    }
	       }
	     /* pattern matching */

	     PostfixSearch (strMatchEnv.SourceTree, MatchNode);
	     /* construct the result list of matching */
	     node = strMatchEnv.SourceTree;
	     /* sets node to the selected node */
	     while (node->Child != NULL && TtaIsAncestor (myFirstSelect, node->Elem))
		node = node->Child;
	     tag = TtaGetMemory (NAME_LENGTH);
	     i = 0;
	     do
	       {		/* for each node above the first selected */
		  node = node->Parent;
		  sm = node->Matches;
		  while (sm != NULL)
		    {		/* for each matching of the node */
		       if (!strcmp (sm->MatchSymb->Tag, "pattern_root"))
			 {	/* if it is matching a pattern root : insert the transformation */
			    /* in the matched transformations list */

			    strcpy (tag, "");
			    if (CheckValidTransRoot (sm,
				      TtaGetElementType (sm->MatchNode->Elem),
						     tag))
			       menuTrans[i++] = (strMatch *) sm;
			 }
		       sm = sm->Next;
		    }
	       }
	     while (node != strMatchEnv.SourceTree);
	     TtaFreeMemory (tag);
	     if (i > 0)
		/* if at least one transformation have been matched, apply the first one */
		TransCallbackDialog (TransBaseDialog + TransMenu, 1 + i, (char *) 0);

	  }
     }
   return resultTrans;
}
