/* This file is part of
* ======================================================
* 
*           LyX, the High Level Word Processor
* 	 
*	    Copyright (C) 1995 Matthias Ettrich
*
*======================================================*/

#include "config.h"

#include <ctype.h>
#include "forms.h"

#include "lyxparagraph.h"
#include "lyxtext.h"
#include "inset.h"
#include "layout.h"
#include "lyx.h"
#include "textutils.h"
#include "lyx_cb.h"
#include "undo.h"
#include "minibuffer.h"

extern FD_form_main *fd_form_main;

extern void WriteAlert(const char* s1, const char* s2, const char* s3);

#define LYX_PAPER_MARGIN 15


LyXFont LyXText::GetFont(LyXParagraph* par, int pos)
{
   LyXFont layoutfont, tmpfont;
   LyXLayout *layout = lyxstyle.Style(parameters->textclass, par->GetLayout());
   


   if (pos >= 0){
     if (pos < BeginningOfMainBody(par))
       layoutfont = layout->labelfont;
     else
       layoutfont = layout->font;
     tmpfont = par->GetFontSettings(pos);
     if (tmpfont.family == LYX_LAYOUT_DEFAULT)
       tmpfont.family = layoutfont.family;
     if (tmpfont.series == LYX_LAYOUT_DEFAULT)
       tmpfont.series = layoutfont.series;
     if (tmpfont.shape == LYX_LAYOUT_DEFAULT)
       tmpfont.shape = layoutfont.shape;
     if (tmpfont.size == LYX_LAYOUT_DEFAULT)
       tmpfont.size = layoutfont.size;
     if (tmpfont.bar == LYX_LAYOUT_DEFAULT)
       tmpfont.bar = layoutfont.bar;
     if (tmpfont.latex == LYX_LAYOUT_DEFAULT)
       tmpfont.latex = layoutfont.latex;
   }
   else{
     // process layoutfont for pos == -1 and labelfont for pos < -1
     if (pos == -1)
       tmpfont = layout->font;
     else
       tmpfont = layout->labelfont;
   }
   

   // check for environment font information
   if (par->GetDepth()){
     if (tmpfont.family == LYX_ENVIRONMENT_DEFAULT){
       while (par && par->GetDepth()
	      && lyxstyle.Style(parameters->textclass,
				par->GetLayout())->font.family == LYX_ENVIRONMENT_DEFAULT)
	 par = par->DepthHook(par->GetDepth()-1);
       if (par)
	 tmpfont.family =  lyxstyle.Style(parameters->textclass,
					par->GetLayout())->font.family;       
     }
     if (tmpfont.series == LYX_ENVIRONMENT_DEFAULT){
       while (par && par->GetDepth()
	      && lyxstyle.Style(parameters->textclass,
				par->GetLayout())->font.series == LYX_ENVIRONMENT_DEFAULT)
	 par = par->DepthHook(par->GetDepth()-1);
       if (par)
	 tmpfont.series =  lyxstyle.Style(parameters->textclass,
					par->GetLayout())->font.series;       
     }
     if (tmpfont.shape == LYX_ENVIRONMENT_DEFAULT){
       while (par && par->GetDepth()
	      && lyxstyle.Style(parameters->textclass,
				par->GetLayout())->font.shape == LYX_ENVIRONMENT_DEFAULT)
	 par = par->DepthHook(par->GetDepth()-1);
       if (par)
	 tmpfont.shape =  lyxstyle.Style(parameters->textclass,
					par->GetLayout())->font.shape;       
     }
     if (tmpfont.size == LYX_ENVIRONMENT_DEFAULT){
       while (par && par->GetDepth()
	      && lyxstyle.Style(parameters->textclass,
				par->GetLayout())->font.size == LYX_ENVIRONMENT_DEFAULT)
	 par = par->DepthHook(par->GetDepth()-1);
       if (par)
	 tmpfont.size =  lyxstyle.Style(parameters->textclass,
					par->GetLayout())->font.size;       
     }
   }
     
     if (tmpfont.family == LYX_ENVIRONMENT_DEFAULT)
       tmpfont.family =  lyxstyle.TextClass(parameters->textclass)->defaultfamily;
     if (tmpfont.series == LYX_ENVIRONMENT_DEFAULT)
       tmpfont.series =  lyxstyle.TextClass(parameters->textclass)->defaultseries;
     if (tmpfont.shape == LYX_ENVIRONMENT_DEFAULT)
       tmpfont.shape =  lyxstyle.TextClass(parameters->textclass)->defaultshape;
     if (tmpfont.size == LYX_ENVIRONMENT_DEFAULT)
       tmpfont.size =  lyxstyle.TextClass(parameters->textclass)->defaultsize;
     
     
   /* if this paragraph is an open footnote, just make the font smaller */ 
   if (par->footnoteflag == LYX_OPEN_FOOTNOTE
       && par->footnotekind == LYX_FOOTNOTE) {
	  if (tmpfont.size > LYX_SIZE_TINY) tmpfont.size--;
       }
   
   return tmpfont;
}


void LyXText::SetCharFont(LyXParagraph *par, int pos, LyXFont font)
{
   LyXFont layoutfont;

   /* let the insets convert their font */ 
   if (par->GetChar(pos) == LYX_META_INSET) {
      if (par->GetInset(pos))
      	 font = par->GetInset(pos)->ConvertFont(font);
   }
   
   LyXLayout *layout = lyxstyle.Style(parameters->textclass, par->GetLayout());
   
   if (pos < BeginningOfMainBody(par))
     layoutfont = layout->labelfont;
   else
     layoutfont = layout->font;

   if (font.family == layoutfont.family)
     font.family = LYX_LAYOUT_DEFAULT;
   if (font.series == layoutfont.series)
     font.series = LYX_LAYOUT_DEFAULT;
   if (font.shape == layoutfont.shape)
     font.shape = LYX_LAYOUT_DEFAULT;
   if (font.size == layoutfont.size)
     font.size = LYX_LAYOUT_DEFAULT;
   if (font.bar == layoutfont.bar)
     font.bar = LYX_LAYOUT_DEFAULT;
   if (font.latex == layoutfont.latex)
     font.latex = LYX_LAYOUT_DEFAULT;
   par->SetFont(pos, font);
}


/* inserts a new row behind the specified row, increments
* the touched counters */
void LyXText::InsertRow(Row *row, LyXParagraph *par, int pos)
{
   Row *tmprow = new Row;
   if (!row) {
      tmprow->previous = NULL;
      tmprow->next = firstrow;
      firstrow = tmprow;
   }
   else {
      tmprow->previous = row;
      tmprow->next = row->next;
      row->next = tmprow;
   }
   
   if (tmprow->next)
     tmprow->next->previous = tmprow;
   
   if (tmprow->previous)
     tmprow->previous->next = tmprow;
   
   
   tmprow->par = par;
   tmprow->pos = pos;
   
   if (row == lastrow)
     lastrow = tmprow;
   number_of_rows++;			       /* one more row  */
}


/* removes the row and reset the touched counters */
void LyXText::RemoveRow(Row *row)
{
   /* this must not happen before the currentrow for clear reasons.
      so the trick is just to set the current row onto the previous
      row of this row */
   long unused_y;
   GetRow(row->par, row->pos, unused_y);
   currentrow = currentrow->previous;
   if (currentrow)
     currentrow_y -= currentrow->height;
   else
     currentrow_y = 0;
   
   if (row->next)
     row->next->previous = row->previous;
   if (!row->previous) {
      firstrow = row->next;
   }
   else  {
      row->previous->next = row->next;
   }
   if (row == lastrow)
     lastrow = row->previous;
   
   height -= row->height;	       /* the text becomes smaller  */
   
   delete row;
   number_of_rows--;			       /* one row less  */
}
     

/* remove all following  rows of the paragraph of the specified row. */
void LyXText::RemoveParagraph(Row *row)
{
   LyXParagraph *tmppar;
   Row *tmprow;

   tmppar = row->par;
   row = row->next;
    
   while (row && row->par == tmppar) {
      tmprow = row->next;
      RemoveRow(row);
      row = tmprow;
   }
}
   
  
/* insert the specified paragraph behind the specified row */
void LyXText::InsertParagraph(LyXParagraph *par, Row *row)
{
   InsertRow(row, par, 0);	       /* insert a new row, starting 
					* at postition 0 */

   SetCounter(par);		       /* set the counters  */
   
   /* and now append the whole paragraph behind the new row */
   if (!row) {
      firstrow->height = 0;
      AppendParagraph(firstrow);
   }
   else {
      row->next->height = 0;
      AppendParagraph(row->next);
   }
}
    

/* konstruktor */ 
LyXText::LyXText(int pw, LyXTextParameters *p)
{
   firstrow = NULL;
   lastrow = NULL;
   currentrow = NULL;
   currentrow_y = 0;
   paperwidth = pw;
   parameters = p;
   number_of_rows = 0;
   refresh_y= 0;
   status = LYX_UNCHANGED;
   LyXParagraph *par = p->paragraph;
   current_font = GetFont(par, 0);
   
   height = 0;
   
   while (par) {
      InsertParagraph(par, lastrow);
      par = par->Next();
   }
   /* set cursor at the very top position */
   selection = True;		       /* these setting is necessary 
    					* because of the delete-empty-
					* paragraph mechanism in
					* SetCursor */
   SetCursor(firstrow->par, 0);
   sel_cursor = cursor;
   selection = False;
   mark_set = False;
   
   /* no rebreak necessary */ 
   need_break_row = NULL;
   
   undo_finished = True;
   undo_frozen = False;
}


/* Destruktor */ 
LyXText::~LyXText()
{
   Row *tmprow;
   
   /* SetCursor(firstrow->par, 0); */ 
   
   tmprow = firstrow;
   
   while (firstrow) {		       /* delete all rows, this does 
					* not touch the paragraphs! */
      tmprow = firstrow->next;
      delete firstrow;
      firstrow = tmprow;
   }
}


void LyXText::ToggleFootnote()
{
   LyXParagraph *par;
   
   par = cursor.par->ParFromPos(cursor.pos);
   if (par->next && par->next->footnoteflag == LYX_CLOSED_FOOTNOTE){
     OpenFootnote();
     minibuffer.Set("Opened float");
   }
   else {
     minibuffer.Set("Closed float");
     CloseFootnote();
   }
}


void LyXText::OpenStuff()
{
   if (cursor.pos < cursor.par->Last() 
       && cursor.par->GetChar(cursor.pos) == LYX_META_INSET
       && cursor.par->GetInset(cursor.pos)->Editable()) {
     minibuffer.Set(cursor.par->GetInset(cursor.pos)->EditMessage());
     if (cursor.par->GetInset(cursor.pos)->Editable() != 2)
       SetCursorParUndo();
     cursor.par->GetInset(cursor.pos)->Edit(0,0);
   }
   else {
      ToggleFootnote();
   }
}


void LyXText::CloseFootnote()
{
   LyXParagraph *par, *endpar,*tmppar;
   Row *row;
   
   par = cursor.par->ParFromPos(cursor.pos);
   
   /* if the cursor is not in an open footnote, or 
    * there is no open footnote in this paragraph, just return. */ 
   if (cursor.par->footnoteflag != LYX_OPEN_FOOTNOTE) {
      
      if (!par->next
	  || par->next->footnoteflag != LYX_OPEN_FOOTNOTE) {
	 minibuffer.Set("Nothing to do");
	 return;
      }
   
      /* ok, move the cursor right before the footnote */ 
 
      /* just a little faster than using CursorRight() */
      for (cursor.pos=0; cursor.par->ParFromPos(cursor.pos)!=par; cursor.pos++);
      /* now the cursor is at the beginning of the physical par */
      SetCursor(cursor.par, cursor.pos + cursor.par->ParFromPos(cursor.pos)->last);
   }
   else  {
       /* we are in a footnote, so let us move at the beginning */ 
       /*      while (cursor.par->footnoteflag == LYX_OPEN_FOOTNOTE)
	       cursor.par = cursor.par->Previous();
	       
	       SetCursor(cursor.par, cursor.par->Last()); */
       /* this is just faster than using just CursorLeft() */ 
       
     tmppar = cursor.par;
     while (tmppar->footnoteflag == LYX_OPEN_FOOTNOTE) {
       /* just a little bit faster than movin the cursor */
	  tmppar = tmppar->Previous();
     }
      SetCursor(tmppar, tmppar->Last());
   }
   
   /* the cursor must be exactly before the footnote */ 
   par = cursor.par->ParFromPos(cursor.pos);
   
   status = LYX_NEED_MORE_REFRESH;
   refresh_row = cursor.row;
   refresh_y = cursor.y - cursor.row->baseline;
   
   tmppar = cursor.par;
   endpar = par->NextAfterFootnote()->Next();
   row = cursor.row;
   
   tmppar->CloseFootnote(cursor.pos);

   /* set the dimensions of the cursor row */
   /* row->fill = Fill(row, paperwidth);
   SetHeightOfRow(row);  */ 
   
   while (tmppar != endpar) {
      RemoveRow(row->next);
      if (row->next)
      	tmppar = row->next->par;
      else
      	tmppar = NULL;
   }
   
   AppendParagraph(cursor.row);
   
   SetCursor(cursor.par, cursor.pos);
   sel_cursor = cursor;
   
   /* just necessary */
   if (cursor.row->next)
       SetHeightOfRow(cursor.row->next);
}


/* used in setlayout */
void LyXText::MakeFontEntriesLayoutSpecific(LyXParagraph *par)
{
   LyXFont layoutfont, tmpfont;
   
   int pos;
   
   LyXLayout* layout = lyxstyle.Style(parameters->textclass, par->GetLayout());
   
   for (pos = 0; pos < par->Last(); pos++) {
      if (pos < BeginningOfMainBody(par))
      	layoutfont = layout->labelfont;
      else
      	layoutfont = layout->font;
      
      tmpfont = par->GetFontSettings(pos);
   
      if (tmpfont.family == layoutfont.family)
      	tmpfont.family = LYX_LAYOUT_DEFAULT;
      if (tmpfont.series == layoutfont.series)
      	tmpfont.series = LYX_LAYOUT_DEFAULT;
      if (tmpfont.shape == layoutfont.shape)
      	tmpfont.shape = LYX_LAYOUT_DEFAULT;
      if (tmpfont.size == layoutfont.size)
      	tmpfont.size = LYX_LAYOUT_DEFAULT;
      if (tmpfont.bar == layoutfont.bar)  
      	tmpfont.bar = LYX_LAYOUT_DEFAULT;
      if (tmpfont.latex == layoutfont.latex)  
      	tmpfont.latex = LYX_LAYOUT_DEFAULT;
      par->SetFont(pos, tmpfont);
   }
}


/* set layout over selection and make a total rebreak of those  paragraphs */
void  LyXText::SetLayout(char layout)
{
   LyXCursor tmpcursor;

   /* if there is no selection just set the layout of the current paragraph  */
   if (!selection) {
      sel_start_cursor = cursor;       /* dummy selection  */
      sel_end_cursor = cursor;
   }

   LyXParagraph *endpar = sel_end_cursor.par->LastPhysicalPar()->Next();
   LyXParagraph *undoendpar = endpar;

   if (endpar && endpar->GetDepth()) {
     while (endpar && endpar->GetDepth()) {
       //endpar->depth = 0;
       endpar = endpar->LastPhysicalPar()->Next();
       undoendpar = endpar;
     }
   }
   else if (endpar) {
     endpar = endpar->Next();	       /* because of parindents etc.  */
   }
   
   SetUndo(LYX_UNDO_EDIT, 
	   sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
	   undoendpar);

   tmpcursor = cursor;		       /* store the current cursor  */

   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */ 
   cursor = sel_start_cursor;
   
   LyXLayout * lyxlayout = lyxstyle.Style(parameters->textclass, layout);
   
   while (cursor.par != sel_end_cursor.par) {
      if (cursor.par->footnoteflag ==
	  sel_start_cursor.par->footnoteflag) {
	     cursor.par->SetLayout(layout);
	     MakeFontEntriesLayoutSpecific(cursor.par);
	     cursor.par->FirstPhysicalPar()->fill_top = lyxlayout->fill_top;
	     cursor.par->FirstPhysicalPar()->fill_bottom = lyxlayout->fill_bottom;
	     if (lyxlayout->margintype == MARGIN_MANUAL)
	       cursor.par->SetLabelWidthString(lyxlayout->labelstring);
	  }
      cursor.par = cursor.par->Next();
   }
   if (cursor.par->footnoteflag ==
       sel_start_cursor.par->footnoteflag) {
	  cursor.par->SetLayout(layout);
	  MakeFontEntriesLayoutSpecific(cursor.par);
	  cursor.par->FirstPhysicalPar()->fill_top = lyxlayout->fill_top;
	  cursor.par->FirstPhysicalPar()->fill_bottom = lyxlayout->fill_bottom;
	  if (lyxlayout->margintype == MARGIN_MANUAL)
	    cursor.par->SetLabelWidthString(lyxlayout->labelstring);
       }
   
   RedoParagraphs(sel_start_cursor, endpar);
   
   /* we have to reset the selection, because the
    * geometry could have changed */ 
   SetCursor(sel_start_cursor.par, sel_start_cursor.pos);
   sel_cursor = cursor;
   SetCursor(sel_end_cursor.par, sel_end_cursor.pos);
   UpdateCounters(cursor.row);
   ClearSelection();
   SetSelection();
   SetCursor(tmpcursor.par, tmpcursor.pos);
}


/* increment depht over selection and
 * make a total rebreak of those paragraphs */
void  LyXText::IncDepth()
{
   LyXCursor tmpcursor;
   
   /* if there is no selection just set the layout of the current paragraph  */
   if (!selection) {
      sel_start_cursor = cursor;       /* dummy selection  */
      sel_end_cursor = cursor;
   }
   
   LyXParagraph *endpar = sel_end_cursor.par->LastPhysicalPar()->Next();
   LyXParagraph *undoendpar = endpar;

   if (endpar && endpar->GetDepth()) {
     while (endpar && endpar->GetDepth()) {
       endpar = endpar->LastPhysicalPar()->Next();
       undoendpar = endpar;
     }
   }
   else if (endpar) {
     endpar = endpar->Next();	       /* because of parindents etc.  */
   }
   
   SetUndo(LYX_UNDO_EDIT, 
	   sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
	   undoendpar);

   tmpcursor = cursor;		       /* store the current cursor  */

   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */ 
   cursor = sel_start_cursor;
   
   bool anything_changed = false;
   
   while (cursor.par != sel_end_cursor.par) {
      if (cursor.par->footnoteflag ==
	  sel_start_cursor.par->footnoteflag) {
	     if (cursor.par->FirstPhysicalPar()->Previous()
		 && (cursor.par->FirstPhysicalPar()->Previous()->GetDepth() -
		     cursor.par->GetDepth() > 0
		     ||
		     (cursor.par->FirstPhysicalPar()->Previous()->GetDepth()
		      ==  cursor.par->GetDepth()
		      && lyxstyle.Style(parameters->textclass, cursor.par->FirstPhysicalPar()->Previous()->GetLayout())->latextype != LATEX_PARAGRAPH
		      && lyxstyle.Style(parameters->textclass, cursor.par->FirstPhysicalPar()
			       ->Previous()->GetLayout())->latextype != LATEX_COMMAND))) {
		cursor.par->FirstPhysicalPar()->depth++;
		anything_changed = true;
	     }
	  }
      cursor.par = cursor.par->Next();
   }
   if (cursor.par->footnoteflag ==
       sel_start_cursor.par->footnoteflag) {
	  if (cursor.par->FirstPhysicalPar()->Previous()
	      && (cursor.par->FirstPhysicalPar()->Previous()->GetDepth() -
		  cursor.par->GetDepth() > 0
		  ||
		  (cursor.par->FirstPhysicalPar()->Previous()->GetDepth()
		   ==  cursor.par->GetDepth()
		   && lyxstyle.Style(parameters->textclass, cursor.par->FirstPhysicalPar()->Previous()->GetLayout())->latextype != LATEX_PARAGRAPH
		   && lyxstyle.Style(parameters->textclass, cursor.par->FirstPhysicalPar()
			    ->Previous()->GetLayout())->latextype != LATEX_COMMAND))) {
	     cursor.par->FirstPhysicalPar()->depth++;
	     anything_changed = true;
	  }
   }
   
   /* if nothing changed set all depth to 0 */ 
   if (!anything_changed) {
       cursor = sel_start_cursor;
      while (cursor.par != sel_end_cursor.par) {
	 cursor.par->FirstPhysicalPar()->depth = 0;
	 cursor.par = cursor.par->Next();
      }
      if (cursor.par->footnoteflag == sel_start_cursor.par->footnoteflag)
      	 cursor.par->FirstPhysicalPar()->depth = 0;
   }
   
   RedoParagraphs(sel_start_cursor, endpar);
   
   /* we have to reset the selection, because the
    * geometry could have changed */ 
   SetCursor(sel_start_cursor.par, sel_start_cursor.pos);
   sel_cursor = cursor;
   SetCursor(sel_end_cursor.par, sel_end_cursor.pos);
   UpdateCounters(cursor.row);
   ClearSelection();
   SetSelection();
   SetCursor(tmpcursor.par, tmpcursor.pos);
}


/* decrement depht over selection and
 * make a total rebreak of those paragraphs */
void  LyXText::DecDepth()
{
   LyXCursor tmpcursor;
   
   /* if there is no selection just set the layout of the current paragraph  */
   if (!selection) {
      sel_start_cursor = cursor;       /* dummy selection  */
      sel_end_cursor = cursor;
   }
   
   LyXParagraph *endpar = sel_end_cursor.par->LastPhysicalPar()->Next();
   LyXParagraph *undoendpar = endpar;

   if (endpar && endpar->GetDepth()) {
     while (endpar && endpar->GetDepth()) {
       endpar = endpar->LastPhysicalPar()->Next();
       undoendpar = endpar;
     }
   }
   else if (endpar) {
     endpar = endpar->Next();	       /* because of parindents etc.  */
   }
   
   SetUndo(LYX_UNDO_EDIT, 
	   sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
	   undoendpar);

   tmpcursor = cursor;		       /* store the current cursor  */

   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */ 
   cursor = sel_start_cursor;
   
   while (cursor.par != sel_end_cursor.par) {
      if (cursor.par->footnoteflag ==
	  sel_start_cursor.par->footnoteflag) {
	     if (cursor.par->FirstPhysicalPar()->depth)
	       cursor.par->FirstPhysicalPar()->depth--;
	  }
      cursor.par = cursor.par->Next();
   }
   if (cursor.par->footnoteflag ==
       sel_start_cursor.par->footnoteflag) {
	  if (cursor.par->FirstPhysicalPar()->depth)
	    cursor.par->FirstPhysicalPar()->depth--;
       }
   
   RedoParagraphs(sel_start_cursor, endpar);
   
   /* we have to reset the selection, because the
    * geometry could have changed */ 
   SetCursor(sel_start_cursor.par, sel_start_cursor.pos);
   sel_cursor = cursor;
   SetCursor(sel_end_cursor.par, sel_end_cursor.pos);
   UpdateCounters(cursor.row);
   ClearSelection();
   SetSelection();
   SetCursor(tmpcursor.par, tmpcursor.pos);
}


/* set font over selection and make a total rebreak of those paragraphs */
void  LyXText::SetFont(LyXFont font)
{
   LyXCursor tmpcursor;
  
   /* if there is no selection just set the current_font */
   if (!selection) {
      if (font.family != LYX_NO_FONT_CHANGE)
      	current_font.family = font.family;
      if (font.series != LYX_NO_FONT_CHANGE)
      	current_font.series = font.series;
      if (font.shape != LYX_NO_FONT_CHANGE)
      	current_font.shape = font.shape;
      if (font.size != LYX_NO_FONT_CHANGE)
      	current_font.size = font.size;
      if (font.latex != LYX_NO_FONT_CHANGE)
      	current_font.latex = font.latex;
      if (font.bar != LYX_NO_FONT_CHANGE)
      	current_font.bar = font.bar;
      
      /* set the real current font */ 
      LyXLayout *layout = lyxstyle.Style(parameters->textclass, cursor.par->GetLayout());
      LyXFont layoutfont;
      if (cursor.pos < BeginningOfMainBody(cursor.par))
      	 layoutfont = GetFont(cursor.par, -2);
      else
      	 layoutfont = GetFont(cursor.par, -1);
      
      real_current_font = current_font;
      
      if (real_current_font.family == LYX_LAYOUT_DEFAULT)
      	 real_current_font.family = layoutfont.family;
      if (real_current_font.series == LYX_LAYOUT_DEFAULT)
      	 real_current_font.series = layoutfont.series;
      if (real_current_font.shape == LYX_LAYOUT_DEFAULT)
      	 real_current_font.shape = layoutfont.shape;
      if (real_current_font.size == LYX_LAYOUT_DEFAULT)
      	 real_current_font.size = layoutfont.size;
      if (real_current_font.latex == LYX_LAYOUT_DEFAULT)
      	 real_current_font.latex = layoutfont.latex;
      if (real_current_font.bar == LYX_LAYOUT_DEFAULT)
      	 real_current_font.bar = layoutfont.bar;
      return;
   }
   
   tmpcursor = cursor;		       /* store the current cursor  */
   
   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */ 
   
   SetUndo(LYX_UNDO_EDIT, 
	   sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
	   sel_end_cursor.par->ParFromPos(sel_end_cursor.pos)->next); 
   cursor = sel_start_cursor;
   while (cursor.par != sel_end_cursor.par ||
	  (cursor.par->footnoteflag == sel_start_cursor.par->footnoteflag
	   &&cursor.pos < sel_end_cursor.pos)) {
     if (cursor.pos < cursor.par->Last()
	 && cursor.par->footnoteflag
	 == sel_start_cursor.par->footnoteflag) {   /* an open footnote
						     * should behave
						     * like a closed */
       SetCharFont(cursor.par, cursor.pos, font);
       cursor.pos++;
     }
     else {
       cursor.pos = 0;
       cursor.par = cursor.par->Next();
     }
   }
   
   RedoParagraphs(sel_start_cursor, sel_end_cursor.par->Next());
   
   /* we have to reset the selection, because the
    * geometry could have changed */ 
   SetCursor(sel_start_cursor.par, sel_start_cursor.pos);
   sel_cursor = cursor;
   SetCursor(sel_end_cursor.par, sel_end_cursor.pos);
   ClearSelection();
   SetSelection();
   SetCursor(tmpcursor.par, tmpcursor.pos);
}


void LyXText::RedoHeightOfParagraph(LyXCursor cursor)
{
   Row *tmprow;
   LyXParagraph *first_phys_par;
   long y;
   
   tmprow = cursor.row;
   
   y = cursor.y - tmprow->baseline;
   SetHeightOfRow(tmprow);
   first_phys_par = tmprow->par->FirstPhysicalPar();
   /* find the first row of the paragraph */
   if (first_phys_par != tmprow->par)
     while (tmprow->previous && tmprow->previous->par != first_phys_par)  {
	tmprow = tmprow->previous;
	y -= tmprow->height;
	SetHeightOfRow(tmprow);
     }
   while (tmprow->previous && tmprow->previous->par == first_phys_par)  {
      tmprow = tmprow->previous;
      y -= tmprow->height;
      SetHeightOfRow(tmprow);
   }
   
   /* we can set the refreshing parameters now */
   status = LYX_NEED_MORE_REFRESH;
   refresh_y = y;
   refresh_row = tmprow;
   SetCursor(cursor.par, cursor.pos);
}


void LyXText::RedoDrawingOfParagraph(LyXCursor cursor)
{
   Row *tmprow;
   LyXParagraph *first_phys_par;
   long y;
   
   tmprow = cursor.row;
   
   y = cursor.y - tmprow->baseline;
   SetHeightOfRow(tmprow);
   first_phys_par = tmprow->par->FirstPhysicalPar();
   /* find the first row of the paragraph */
   if (first_phys_par != tmprow->par)
     while (tmprow->previous && tmprow->previous->par != first_phys_par)  {
	tmprow = tmprow->previous;
	y -= tmprow->height;
     }
   while (tmprow->previous && tmprow->previous->par == first_phys_par)  {
      tmprow = tmprow->previous;
      y -= tmprow->height;
      
   }
   
   /* we can set the refreshing parameters now */
   if (status == LYX_UNCHANGED || y < refresh_y) {
     refresh_y = y;
     refresh_row = tmprow;
   }
   status = LYX_NEED_MORE_REFRESH;
   SetCursor(cursor.par, cursor.pos);
}


/* deletes and inserts again all paragaphs between the cursor
* and the specified par 
* This function is needed after SetLayout and SetFont etc. */
void LyXText::RedoParagraphs(LyXCursor cursor, LyXParagraph *endpar)
{
   Row *tmprow, *tmprow2;
   LyXParagraph *tmppar, *first_phys_par;
   long y;
   
   tmprow = cursor.row;
   
   y = cursor.y - tmprow->baseline;
   
   if (!tmprow->previous){
     first_phys_par = FirstParagraph();   // a trick/hack for UNDO
   }
   else {
     first_phys_par = tmprow->par->FirstPhysicalPar();
     /* find the first row of the paragraph */
     if (first_phys_par != tmprow->par)
       while (tmprow->previous && tmprow->previous->par != first_phys_par)  {
	 tmprow = tmprow->previous;
	 y -= tmprow->height;
       }
     while (tmprow->previous && tmprow->previous->par == first_phys_par)  {
       tmprow = tmprow->previous;
       y -= tmprow->height;
     }
   }
   
   /* we can set the refreshing parameters now */
   status = LYX_NEED_MORE_REFRESH;
   refresh_y = y;
   refresh_row = tmprow->previous;	       /* the real refresh row will
						* be deleted, so I store
						* the previous here */ 
   /* remove it */
   if (tmprow->next)
     tmppar = tmprow->next->par;
   else
     tmppar = NULL;
   while (tmppar != endpar) {
      RemoveRow(tmprow->next);
      if (tmprow->next)
      	tmppar = tmprow->next->par;
      else
      	tmppar = NULL;
   }  
   
   /* remove the first one */
   tmprow2 = tmprow;		       /* this is because tmprow->previous
					* can be NULL */
   tmprow = tmprow->previous;
   RemoveRow(tmprow2);
   
   tmppar = first_phys_par;

   do {
      if (tmppar) {
	 InsertParagraph(tmppar, tmprow);
	 if (!tmprow)
	   tmprow = firstrow;
	 while (tmprow->next && tmprow->next->par == tmppar)
	   tmprow = tmprow->next;
	 tmppar = tmppar->Next();
	 
      }
   }
   while (tmppar != endpar);
   
   /* this is because of layout changes */ 
   if (refresh_row) {
     refresh_y -= refresh_row->height;
     SetHeightOfRow(refresh_row);   
   }
   else {
       refresh_row = firstrow;
       refresh_y = 0;
       SetHeightOfRow(refresh_row);   
   }
   
   if (tmprow && tmprow->next)
     SetHeightOfRow(tmprow->next);
   
   		       /* restore the correct refresh row  */
/*    if (refresh_row)
     refresh_row = refresh_row->next;
   else
     refresh_row = firstrow;*/ 
}


int LyXText::FullRebreak()
{
   if (need_break_row) {
      BreakAgain(need_break_row);
      need_break_row = NULL;
      return 1;
   }
   return 0;
}


/* important for the screen */


/* the cursor set functions have a special mechanism. When they
* realize, that you left an empty paragraph, they will delete it.
* They also delet the corresponding row */
   
/* need the selection cursor: */ 
void LyXText::SetSelection()
{
   if (!selection) {
      last_sel_cursor = sel_cursor;
      sel_start_cursor = sel_cursor;
      sel_end_cursor = sel_cursor;
   }
   
   selection = True;
   
   /* first the toggling area */ 
   if (cursor.y < last_sel_cursor.y ||
       (cursor.y == last_sel_cursor.y && cursor.x < last_sel_cursor.x)) {
	  toggle_end_cursor = last_sel_cursor;
	  toggle_cursor = cursor;
       }
   else {
      toggle_end_cursor = cursor;
      toggle_cursor = last_sel_cursor;
   }
   
   last_sel_cursor = cursor;
   
   /* and now the whole selection */ 
   
   if (sel_cursor.y < cursor.y ||
       (sel_cursor.y == cursor.y && sel_cursor.x < cursor.x)) {
	  sel_end_cursor = cursor;
	  sel_start_cursor = sel_cursor;
       }
   else {
      sel_end_cursor = sel_cursor; 
      sel_start_cursor = cursor;
   }
   
   /* a selection with no contents is not a selection */ 
   if (sel_start_cursor.x == sel_end_cursor.x && 
       sel_start_cursor.y == sel_end_cursor.y)
     selection = False;
}


void LyXText::ClearSelection()
{
   selection = False;
   mark_set = False;
}


void  LyXText::CursorHome()
{
  SetCursor(cursor.par, cursor.row->pos);
}


void  LyXText::CursorEnd()
{
   if (!cursor.row->next || cursor.row->next->par != cursor.row->par)
     SetCursor(cursor.par, RowLast(cursor.row) + 1);
   else {
      if (cursor.par->Last() && 
	  (cursor.par->GetChar(RowLast(cursor.row)) == ' '
	   || cursor.par->IsNewline(RowLast(cursor.row))))
      	SetCursor(cursor.par, RowLast(cursor.row));
      else
      	SetCursor(cursor.par, RowLast(cursor.row) + 1);
   }
}


void  LyXText::CursorTop()
{
   while (cursor.par->Previous())
     cursor.par = cursor.par->Previous();
   SetCursor(cursor.par, 0);
}


void  LyXText::CursorBottom()
{
   while (cursor.par->Next())
     cursor.par = cursor.par->Next();
   SetCursor(cursor.par, cursor.par->Last());
}
   
   
/* returns a pointer to the row near the specified y-coordinate
* (relative to the whole text). y is set to the real beginning
* of this row */ 
Row* LyXText::GetRowNearY(long& y)
{
   Row* tmprow;
   long tmpy;
   
   if (currentrow){
     tmprow = currentrow;
     tmpy = currentrow_y;
   }
   else {
     tmprow = firstrow;
     tmpy = 0;
   }

   if (tmpy<=y)
     while (tmprow->next && tmpy + tmprow->height <= y) {
       tmpy += tmprow->height;
       tmprow = tmprow->next;
     }
   else
     while (tmprow->previous && tmpy > y) {
       tmprow = tmprow->previous;
       tmpy -= tmprow->height;
     }

   currentrow = tmprow;
   currentrow_y = tmpy;

   y = tmpy;			       /* return the real y  */
   return tmprow;
}
   

void LyXText::ToggleFree(LyXFont font, const char* stylename)
{
   LyXFont non_free_font = { 
      LYX_NO_FONT_CHANGE,
      LYX_NO_FONT_CHANGE,
      LYX_NO_FONT_CHANGE,
      LYX_NO_FONT_CHANGE,
      LYX_NO_FONT_CHANGE,
      LYX_NO_FONT_CHANGE
   };

   if (font.family == LYX_NO_FONT_CHANGE &&
       font.series == LYX_NO_FONT_CHANGE &&
       font.shape == LYX_NO_FONT_CHANGE &&
       font.size == LYX_NO_FONT_CHANGE &&
       font.latex == LYX_NO_FONT_CHANGE &&
       font.bar == LYX_NO_FONT_CHANGE){
      // Could only happen with user style
     minibuffer.Set("No user style defined. Use menu entry layout->character to define.");
     return;
   }

   if (font.family != LYX_NO_FONT_CHANGE)
     non_free_font.family = LYX_LAYOUT_DEFAULT;
   if (font.series != LYX_NO_FONT_CHANGE){
     if (font.series == LYX_BOLD_SERIES)
       non_free_font.series = LYX_MEDIUM_SERIES;
     else
       non_free_font.series = LYX_LAYOUT_DEFAULT;
   }
   if (font.shape != LYX_NO_FONT_CHANGE)
     non_free_font.shape = LYX_LAYOUT_DEFAULT;
   if (font.size != LYX_NO_FONT_CHANGE)
     non_free_font.size = LYX_LAYOUT_DEFAULT;
   if (font.latex != LYX_NO_FONT_CHANGE)
     non_free_font.latex = LYX_LAYOUT_DEFAULT;
   if (font.bar != LYX_NO_FONT_CHANGE)
     non_free_font.bar = LYX_LAYOUT_DEFAULT;

   bool all_free;
       
      /* if the whole selection is italic, we will do non-italic,
       * otherwise we do italic. The paragraph himself will 
       * change it to layout-default if possible , however it 
       * looks like */

	/* -------> Try for implicit word selection. */
	LyXCursor resetCursor = cursor;
	int implicitSelection = SelectWordWhenUnderCursor();

   /* if there is no selection just set the current font  */
   
   if (!selection) {
      if ((font.family != LYX_NO_FONT_CHANGE && real_current_font.family != font.family)
	  ||
	  (font.series != LYX_NO_FONT_CHANGE && real_current_font.series != font.series)
	  ||
	  (font.shape != LYX_NO_FONT_CHANGE && real_current_font.shape != font.shape)
	  ||
	  (font.size != LYX_NO_FONT_CHANGE && real_current_font.size != font.size)
	  ||
	  (font.bar != LYX_NO_FONT_CHANGE && real_current_font.bar != font.bar)
	  ||
	  (font.latex != LYX_NO_FONT_CHANGE && real_current_font.latex != font.latex)
	  )
	{
	   SetFont(font);
	   minibuffer.Set(stylename, "style set");
	}
      else{
	 SetFont(non_free_font);
	 minibuffer.Set(stylename, "style removed");
      }
      return;
   }
   
   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */ 
   LyXCursor tmpcursor = sel_start_cursor;
   all_free = true;
   while (all_free && 
	  (tmpcursor.par != sel_end_cursor.par ||
	   tmpcursor.pos < sel_end_cursor.pos)) {
	      if (tmpcursor.pos < tmpcursor.par->Last()) {
		 if (tmpcursor.par->footnoteflag 
		     == sel_start_cursor.par->footnoteflag
		     &&
		     (
		      (font.family != LYX_NO_FONT_CHANGE && GetFont(tmpcursor.par, tmpcursor.pos).family != font.family) ||
		      (font.series != LYX_NO_FONT_CHANGE && GetFont(tmpcursor.par, tmpcursor.pos).series != font.series) ||
		      (font.shape != LYX_NO_FONT_CHANGE && GetFont(tmpcursor.par, tmpcursor.pos).shape != font.shape) ||
		      (font.size != LYX_NO_FONT_CHANGE && GetFont(tmpcursor.par, tmpcursor.pos).size != font.size) ||
		      (font.latex != LYX_NO_FONT_CHANGE && GetFont(tmpcursor.par, tmpcursor.pos).latex != font.latex) ||
		      (font.bar != LYX_NO_FONT_CHANGE && GetFont(tmpcursor.par, tmpcursor.pos).bar != font.bar)
		      )
		     )
		   all_free = false;
		 tmpcursor.pos++;
	      }
	      else {
		 tmpcursor.pos = 0;
		 tmpcursor.par = tmpcursor.par->Next();
	      }
	   }
   if (all_free){
     SetFont(non_free_font);
     minibuffer.Set(stylename, "style removed");
   }
   else{
     SetFont(font);
     minibuffer.Set(stylename, "style set");
   }

   /* -------> Implicit selections are cleared afterwards and cursor is set to the
			   original position. */

   if ( implicitSelection ) {
	   ClearSelection();
	   cursor = resetCursor;
	   SetCursor( cursor.par, cursor.pos );
	   sel_cursor = cursor;
   }
}


int LyXText::BeginningOfMainBody(LyXParagraph *par)
{
   if (lyxstyle.Style(parameters->textclass, par->GetLayout())->labeltype != LABEL_MANUAL)
     return 0;
   else
     return par->BeginningOfMainBody();
}


/* if there is a selection, reset every environment you can find
* in the selection, otherwise just the environment you are in */ 
void LyXText::MeltFootnoteEnvironment()
{
   LyXParagraph *tmppar, *firsttmppar;
   
   ClearSelection();
   
   /* is is only allowed, if the cursor is IN an open footnote.
    * Otherwise it is too dangerous */ 
   if (cursor.par->footnoteflag != LYX_OPEN_FOOTNOTE)
     return;
   
   SetUndo(LYX_UNDO_FINISH, 
	   cursor.par->PreviousBeforeFootnote()->previous,
	   cursor.par->NextAfterFootnote()->next);

   /* ok, move to the beginning of the footnote. */ 
   while (cursor.par->footnoteflag == LYX_OPEN_FOOTNOTE)
     cursor.par = cursor.par->Previous();
   
   SetCursor(cursor.par, cursor.par->Last());
   /* this is just faster than using CursorLeft(); */ 
   
   firsttmppar = cursor.par->ParFromPos(cursor.pos);
   tmppar = firsttmppar;
   /* tmppar is now the paragraph right before the footnote */
   
   char first_footnote_par_is_not_empty = tmppar->next->last;
   
   while (tmppar->next && tmppar->next->footnoteflag == LYX_OPEN_FOOTNOTE) {
      tmppar = tmppar->next;	       /* I use next instead of Next(),
					* because there cannot be any
					* footnotes in a footnote
					* environment */
      tmppar->footnoteflag = LYX_NO_FOOTNOTE;
      
      /* remember the captions and empty paragraphs */
      if (lyxstyle.Style(parameters->textclass, tmppar->GetLayout())->labeltype == LABEL_SENSITIVE
	  || !tmppar->Last())
	tmppar->SetLayout(0);
   }
   
   /* now we will paste the ex-footnote, if the layouts allow it */
   /* first restore the layout of the paragraph right behind the footnote*/ 
   if (tmppar->next) 
     tmppar->next->MakeSameLayout(cursor.par);

   /* first the end */ 
   if ((!tmppar->GetLayout() && !tmppar->table)
       || (tmppar->Next() && (!tmppar->Next()->Last()
			      || tmppar->Next()->HasSameLayout(tmppar)))) {
      if (tmppar->Next()->Last() && tmppar->Next()->IsLineSeparator(0))
      	 tmppar->Next()->Erase(0);
      tmppar->PasteParagraph();
   }

   tmppar = tmppar->Next();	       /* make shure tmppar cannot be touched
					* by the pasting of the beginning */

   /* then the beginning */ 
   /* if there is no space between the text and the footnote, so we insert
    * a blank 
    * (only if the previous par and the footnotepar are not empty!) */
   if ((!firsttmppar->next->GetLayout() && !firsttmppar->next->table)
       || firsttmppar->HasSameLayout(firsttmppar->next)) {
      if (firsttmppar->last
	  && !firsttmppar->IsSeparator(firsttmppar->last - 1)
	  && first_footnote_par_is_not_empty) {
	 firsttmppar->next->InsertChar(0, ' ');
      }
      firsttmppar->PasteParagraph();
   }
   
   /* now redo the paragaphs */
   RedoParagraphs(cursor, tmppar);
   
   SetCursor(cursor.par, cursor.pos);
   
   /* sometimes it can happen, that there is a counter change */ 
   Row *row = cursor.row;
   while (row->next && row->par != tmppar && row->next->par != tmppar)
     row = row->next;
   UpdateCounters(row);
   
   
   ClearSelection();
}


/* the DTP switches for paragraphs. LyX will store them in the 
* first physicla paragraph. When a paragraph is broken, the top settings 
* rest, the bottom settings are given to the new one. So I can make shure, 
* they do not duplicate themself and you cannnot make dirty things with 
* them!  */ 

void LyXText::SetParagraph(char line_top, char line_bottom,
			   char pagebreak_top, char pagebreak_bottom,
			   float space_top, float space_bottom,
			   char fill_top, char fill_bottom,
			   char align, 
			   const char* labelwidthstring,
			   char noindent) 
{
	LyXCursor tmpcursor;
	tmpcursor = cursor;
	LyXParagraph *tmppar;
	if (!selection) {
		sel_start_cursor = cursor;
		sel_end_cursor = cursor;
	}

	// make sure that the depth behind the selection are restored, too
	LyXParagraph *endpar = sel_end_cursor.par->LastPhysicalPar()->Next();
	LyXParagraph *undoendpar = endpar;

	if (endpar && endpar->GetDepth()) {
	  while (endpar && endpar->GetDepth()) {
	    endpar = endpar->LastPhysicalPar()->Next();
	    undoendpar = endpar;
	  }
	}
	else if (endpar) {
	  endpar = endpar->Next();	       /* because of parindents etc.  */
	}
   
	SetUndo(LYX_UNDO_EDIT, 
		sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
		undoendpar);

	
	tmppar = sel_end_cursor.par;
	while (tmppar != sel_start_cursor.par->FirstPhysicalPar()->Previous())
	{
	   SetCursor(tmppar->FirstPhysicalPar(), 0);
	   status = LYX_NEED_MORE_REFRESH;
	   refresh_row = cursor.row;
	   refresh_y = cursor.y - cursor.row->baseline;
	   if (cursor.par->footnoteflag ==
	       sel_start_cursor.par->footnoteflag) {
	      cursor.par->line_top = line_top;
	      cursor.par->line_bottom = line_bottom;
	      cursor.par->pagebreak_top = pagebreak_top;
	      cursor.par->pagebreak_bottom = pagebreak_bottom;
	      cursor.par->added_space_top = space_top;
	      cursor.par->added_space_bottom = space_bottom;
	      cursor.par->fill_top = fill_top;
	      cursor.par->fill_bottom = fill_bottom;
	      /* does the layout allow the new alignment? */ 
	      if (align == LYX_ALIGN_LAYOUT)
	      	 align = lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->align;
	      if (align & lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->alignpossible) {
		 if (align == lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->align)
		    cursor.par->align = LYX_ALIGN_LAYOUT;
		 else
		    cursor.par->align = align;
	      }
	      cursor.par->SetLabelWidthString((char *)labelwidthstring);
	      cursor.par->noindent = noindent;
	   }
	   
	   /* 
	    tmprow = cursor.row;
	    while (tmprow->next && tmprow->next->par->previous != cursor.par->LastPhysicalPar())
	       tmprow = tmprow->next;
	    SetHeightOfRow(tmprow);
	    SetHeightOfRow(cursor.row); */ 
	   tmppar = cursor.par->FirstPhysicalPar()->Previous();
	}
	
	RedoParagraphs(sel_start_cursor, endpar);
	
	ClearSelection();
	SetCursor(sel_start_cursor.par, sel_start_cursor.pos);
	sel_cursor = cursor;
	SetCursor(sel_end_cursor.par, sel_end_cursor.pos);
	SetSelection();
	SetCursor(tmpcursor.par, tmpcursor.pos);
}


/* set the counter of a paragraph. This includes the labels */ 
void LyXText::SetCounter(LyXParagraph *par)
{
   int i;
   char *s;
   
   /* this is only relevant for the beginning of paragraph */ 
   par = par->FirstPhysicalPar();

   /* copy the prev-counters to this one */
   if (par->Previous()) {
      for (i=0; i<10; i++) {
	 par->counter[i]=par->Previous()->GetCounter(i);
      }
      par->enumdepth = par->Previous()->FirstPhysicalPar()->enumdepth;
      par->itemdepth = par->Previous()->FirstPhysicalPar()->itemdepth;
   }
   else {
      for (i=0; i<10; i++) {
	 par->counter[i]=0;
      }  
      par->enumdepth = 0;
      par->itemdepth = 0;
   }
   
   LyXLayout* layout = lyxstyle.Style(parameters->textclass, par->GetLayout());
   
   LyXTextClass *textclass = lyxstyle.TextClass(parameters->textclass);
   
   /* maybe we have to increment the enumeration depth */ 
   if (par->Previous() &&
       par->Previous()->GetDepth() < par->GetDepth() &&
       lyxstyle.Style(parameters->textclass, par->Previous()->GetLayout())->labeltype == LABEL_COUNTER_ENUMI
       && par->enumdepth < 3)
     par->enumdepth++;

   /* maybe we have to decrement the enumeration depth */ 
   if (par->Previous() &&
       par->Previous()->GetDepth() > par->GetDepth()) {
	  par->enumdepth = par->DepthHook(par->GetDepth())->enumdepth;
	  par->counter[6 + par->enumdepth] = par->DepthHook(par->GetDepth())->counter[6 + par->enumdepth];
	  /* reset the counters. A depth change is like a breaking layout */ 
	  for (i=6 + par->enumdepth + 1; i<10;i++)
	    par->counter[i]=0;
	  
       }
   
   if (par->labelstring) {
      delete[] par->labelstring;
      par->labelstring = NULL;
   }
   
   if (layout->margintype == MARGIN_MANUAL) {
      if (!par->labelwidthstring) {
	 par->SetLabelWidthString(layout->labelstring);
      }
   }
   else {
      par->SetLabelWidthString(NULL);
   }
   
   /* is it a layout that has an automatic label ? */ 
   if (layout->labeltype >=  LABEL_FIRST_COUNTER) {
      
      i = layout->labeltype - LABEL_FIRST_COUNTER;
      if (i>=0 && i<=parameters->secnumdepth) {
	 par->counter[i]++;	       /* increment the counter  */
	 
	 s = new char[50];
	 
	 if (LABEL_FIRST_COUNTER + i == LABEL_COUNTER_CHAPTER) {
	    sprintf(s, "Chapter %d", par->counter[i]);
	 }
	 else {
	    switch (2 * LABEL_FIRST_COUNTER - textclass->maxcounter + i) {
	     case LABEL_COUNTER_CHAPTER:
	       sprintf(s, "%d", par->counter[i]);
	       break;
	     case LABEL_COUNTER_SECTION:
	       sprintf(s, "%d.%d", par->counter[i - 1], par->counter[i]);
	       break;
	     case LABEL_COUNTER_SUBSECTION:
	       sprintf(s, "%d.%d.%d", par->counter[i-2], par->counter[i-1],par->counter[i]);
	       break;
	     case LABEL_COUNTER_SUBSUBSECTION:
	       sprintf(s, "%d.%d.%d.%d", par->counter[i-3], par->counter[i-2],par->counter[i-1],par->counter[i]);
	       break;
	     case LABEL_COUNTER_PARAGRAPH:
	       sprintf(s, "%d.%d.%d.%d.%d", par->counter[i-4], par->counter[i-3], par->counter[i-2],par->counter[i-1],par->counter[i]);
	       break;
	     case LABEL_COUNTER_SUBPARAGRAPH:
	       sprintf(s, "%d.%d.%d.%d.%d.%d", par->counter[i-5], par->counter[i-4], par->counter[i-3], par->counter[i-2],par->counter[i-1],par->counter[i]);
	       break;
	     default:
	       sprintf(s, "%d.", par->counter[i]);
	       break;
	    }
	 }
	 
	 par->labelstring = StringCopy(s);
	 delete[] s;
	 
	 for (i++;i<10;i++)
	   par->counter[i]=0;      /* reset the following counters  */
      }
      else if (layout->labeltype == LABEL_COUNTER_ENUMI) {
	 par->counter[i + par->enumdepth]++;
	 s = new char[25];
	 int number = par->counter[i + par->enumdepth];
	 switch (par->enumdepth) {
	  case 1:
	    sprintf(s, "(%c)", (number % 27) + 'a' - 1);
	    break;
	  case 2:
	    switch (number) {
	     case 1: sprintf(s, "i."); break;
	     case 2: sprintf(s, "ii."); break;
	     case 3: sprintf(s, "iii."); break;
	     case 4: sprintf(s, "iv."); break;
	     case 5: sprintf(s, "v."); break;
	     case 6: sprintf(s, "vi."); break;
	     case 7: sprintf(s, "vii."); break;
	     case 8: sprintf(s, "viii."); break;
	     case 9: sprintf(s, "ix."); break;
	     case 10: sprintf(s, "x."); break;
	     case 11: sprintf(s, "xi."); break;
	     case 12: sprintf(s, "xii."); break;
	     case 13: sprintf(s, "xiii."); break;
	     default:
	       sprintf(s, "\\roman{%d}.", number);
	       break;
	    }
	    break;
	  case 3:
	    sprintf(s, "%c.", (number % 27) + 'A' - 1);
	    break;
	  default:
	    sprintf(s, "%d.", number);
	    break;
	 }
	 par->labelstring = StringCopy(s);
	 delete[] s;
	 
	 for (i += par->enumdepth + 1;i<10;i++)
	    par->counter[i]=0;      /* reset the following counters  */
	 
      }
   }
   else  {
      s = layout->labelstring;
      
      /* the caption hack: */
      
      if (layout->labeltype == LABEL_SENSITIVE) {
	 if (par->footnoteflag != LYX_NO_FOOTNOTE
	     && par->footnotekind == LYX_FIG)
	   s = "Figure:";
	 else if (par->footnoteflag != LYX_NO_FOOTNOTE
		  && par->footnotekind == LYX_TAB)
	   s = "Table:";
	 else {
	   /* par->SetLayout(0); 
	   s = layout->labelstring;  */
	   s = "Senseless: "; 
	   
	 }
      }
      par->labelstring = StringCopy(s);
      
      /* reset the enumeration counter. They are always resetted
       * when there is any other layout between */ 
      for (i=6 + par->enumdepth; i<10;i++)
      	par->counter[i]=0;
   }
}


/* Updates all counters BEHIND the row. Changed paragraphs
* with a dynamic left margin will be rebroken. */ 
void LyXText::UpdateCounters(Row *row)
{
   LyXParagraph *par;
   if (!row) {
      row = firstrow;
      par = row->par;
   }
   else
     par = row->par->LastPhysicalPar()->Next();
   while (par) {
     while (row->par != par)
      	row = row->next;
      
      SetCounter(par);
      
      /* now  check for the headline layouts. remember that they
       * have a dynamic left margin */ 
      if (!par->IsDummy()
	  && ( lyxstyle.Style(parameters->textclass, par->layout)->margintype == MARGIN_DYNAMIC
	      || lyxstyle.Style(parameters->textclass, par->layout)->labeltype == LABEL_SENSITIVE)
	  ){
	 
       /* Rebreak the paragraph */ 
       RemoveParagraph(row);
       AppendParagraph(row);
       
       /* think about the damned open footnotes! */ 
       while (par->Next() &&
	      (par->Next()->footnoteflag == LYX_OPEN_FOOTNOTE
	       || par->Next()->IsDummy())){
	 par = par->Next();
	 if (par->IsDummy()) {
	     while (row->par != par)
	       row = row->next;
	     RemoveParagraph(row);
	     AppendParagraph(row);
	   }
       }
     }
     
     par = par->LastPhysicalPar()->Next();
     
   }
}


/* insets an inset. */ 
void LyXText::InsertInset(Inset *inset)
{
   SetUndo(LYX_UNDO_INSERT, 
	   cursor.par->ParFromPos(cursor.pos)->previous, 
	   cursor.par->ParFromPos(cursor.pos)->next);
   cursor.par->InsertChar(cursor.pos, LYX_META_INSET);
   cursor.par->InsertInset(cursor.pos, inset);
   InsertChar(LYX_META_INSET);	       /* just to rebreak and refresh correctly.
					* The character will not be inserted a
					* second time */
}


/* this is for the simple cut and paste mechanism */ 
static LyXParagraph *simple_cut_buffer = NULL;
static char simple_cut_buffer_textclass = 0;

void DeleteSimpleCutBuffer()
{
  if (!simple_cut_buffer)
    return;
  LyXParagraph *tmppar;

  while (simple_cut_buffer) {
    tmppar =  simple_cut_buffer;
    simple_cut_buffer = simple_cut_buffer->next;
    delete tmppar;
  }
  simple_cut_buffer = NULL;
}


void LyXText::CutSelection()
{
  /* this doesnt make sense, if there is no selection */ 
   if (!selection) {
      return;
   }
   
   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */
   LyXParagraph *tmppar;
   int i;
   
   /* check wether there are half footnotes in the selection */
   if (sel_start_cursor.par->footnoteflag != LYX_NO_FOOTNOTE
       || sel_end_cursor.par->footnoteflag != LYX_NO_FOOTNOTE){
     tmppar = sel_start_cursor.par;
     while (tmppar != sel_end_cursor.par){
       if (tmppar->footnoteflag != sel_end_cursor.par->footnoteflag){
	 WriteAlert("Impossible operation", "Don't know what to do with half floats.", "sorry.");
	 return;
       }
       tmppar = tmppar->Next();
     }
   }

   /* table stuff -- begin*/
   if (sel_start_cursor.par->table || sel_end_cursor.par->table){
     if ( sel_start_cursor.par != sel_end_cursor.par){
       WriteAlert("Impossible operation", "Don't know what to do with half tables.", "sorry.");
       return;
     }
     sel_start_cursor.par->table->Reinit();
   }
   /* table stuff -- end*/

   // make sure that the depth behind the selection are restored, too
   LyXParagraph *endpar = sel_end_cursor.par->LastPhysicalPar()->Next();
   LyXParagraph *undoendpar = endpar;

   if (endpar && endpar->GetDepth()) {
     while (endpar && endpar->GetDepth()) {
       endpar = endpar->LastPhysicalPar()->Next();
       undoendpar = endpar;
     }
   }
   else if (endpar) {
     endpar = endpar->Next();	       /* because of parindents etc.  */
   }
   
   SetUndo(LYX_UNDO_DELETE, 
	   sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->previous, 
	   undoendpar);
   
   /* delete the simple_cut_buffer */ 
   DeleteSimpleCutBuffer();
   
   /* set the textclass */
   simple_cut_buffer_textclass = parameters->textclass;

   char space_wrapped =
	   sel_end_cursor.par->IsLineSeparator(sel_end_cursor.pos);
   if (sel_end_cursor.pos > 0
       && sel_end_cursor.par->IsLineSeparator(sel_end_cursor.pos - 1)) {
	  sel_end_cursor.pos--;	       /* please break before a space at
					* the end */
	  space_wrapped = True;
       }

   // cut behind a space if there is one
   while (sel_start_cursor.par->Last() > sel_start_cursor.pos
	  && sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos)
	  && (sel_start_cursor.par != sel_end_cursor.par
	      || sel_start_cursor.pos < sel_end_cursor.pos))
     sel_start_cursor.pos++; 
   
   /* there are two cases: cut only within one paragraph or
    * more than one paragraph */
   
   if (sel_start_cursor.par->ParFromPos(sel_start_cursor.pos) 
       == sel_end_cursor.par->ParFromPos(sel_end_cursor.pos)) {
     /* only within one paragraph */
     simple_cut_buffer = new LyXParagraph();
     for (i=sel_start_cursor.pos; i< sel_end_cursor.pos; i++){
       /* table stuff -- begin*/
       if (sel_start_cursor.par->table
	   && sel_start_cursor.par->IsNewline(sel_start_cursor.pos)){
	 sel_start_cursor.par->CopyIntoMinibuffer(sel_start_cursor.pos);
	 sel_start_cursor.pos++;
       } else {
       /* table stuff -- end*/
	 sel_start_cursor.par->CutIntoMinibuffer(sel_start_cursor.pos);
	 sel_start_cursor.par->Erase(sel_start_cursor.pos);
       }
       simple_cut_buffer->InsertFromMinibuffer(simple_cut_buffer->Last());
     }
     /* check for double spaces */
     if (sel_start_cursor.pos &&
	 sel_start_cursor.par->Last()>sel_start_cursor.pos &&
	 sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos - 1) &&
	 sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos)){
       sel_start_cursor.par->Erase(sel_start_cursor.pos);
     }
     if (space_wrapped)
       simple_cut_buffer->InsertChar(i - sel_start_cursor.pos, ' ');
     endpar = sel_end_cursor.par->Next();
   }
   else {
     /* cut more than one paragraph */ 
   
     sel_end_cursor.par->BreakParagraphConservative(sel_end_cursor.pos);
     /* insert a space at the end if there was one */
     if (space_wrapped)
       sel_end_cursor.par->InsertChar(sel_end_cursor.par->Last(), ' ');
   
     sel_end_cursor.par = sel_end_cursor.par->Next();
     sel_end_cursor.pos = 0;
   
     cursor = sel_end_cursor;
   
     /* please break behind a space, if there is one. The space should
      * be copied too */ 
     if (sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos))
       sel_start_cursor.pos++;
   
     sel_start_cursor.par->BreakParagraphConservative(sel_start_cursor.pos);
     if (!sel_start_cursor.pos
	 || sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos - 1)
	 || sel_start_cursor.par->IsNewline(sel_start_cursor.pos - 1)) {
       sel_start_cursor.par->Next()->InsertChar(0, ' ');
     }
   
     /* store the endparagraph for redoing later */
     endpar = sel_end_cursor.par->Next();   /* needed because
							   * the sel_end_
							   * cursor.par
							   * will be pasted!*/
   
     /*store the selection */ 
     simple_cut_buffer = sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->next;
     simple_cut_buffer->previous = NULL;
     sel_end_cursor.par->previous->next = NULL;

     /* cut the selection */ 
     sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->next 
       = sel_end_cursor.par;
   
     sel_end_cursor.par->previous 
       = sel_start_cursor.par->ParFromPos(sel_start_cursor.pos);

     /* care about footnotes */
     if (simple_cut_buffer->footnoteflag) {
       LyXParagraph *tmppar = simple_cut_buffer;
       while (tmppar){
        tmppar->footnoteflag = LYX_NO_FOOTNOTE;
        tmppar = tmppar->next;
       }
     }

     /* the cut selection should begin with standard layout */
     simple_cut_buffer->Clear(); 
   
   /* paste the paragraphs again, if possible  */
     sel_start_cursor.par->Next()->ClearParagraph();
     if (sel_start_cursor.par->FirstPhysicalPar()->HasSameLayout(sel_start_cursor.par->Next())
	 || 
	 !sel_start_cursor.par->Next()->Last())
       sel_start_cursor.par->ParFromPos(sel_start_cursor.pos)->PasteParagraph();

   
   /* maybe a forgotten blank */
     if (sel_start_cursor.pos 
	 && sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos)
	 && sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos - 1)) {
       sel_start_cursor.par->Erase(sel_start_cursor.pos);
     }
   }   


   /* sometimes necessary */
   sel_start_cursor.par->ClearParagraph();

   RedoParagraphs(sel_start_cursor, endpar);
   
   ClearSelection();
   cursor = sel_start_cursor;
   SetCursor(cursor.par, cursor.pos);
   sel_cursor = cursor;
   UpdateCounters(cursor.row);
}

    
void LyXText::CopySelection()
{
  int i=0;
   
   /* this doesnt make sense, if there is no selection */ 
   if (!selection) {
      return;
   }

   /* ok we have a selection. This is always between sel_start_cursor
    * and sel_end cursor */
   LyXParagraph *tmppar;
   
   /* check wether there are half footnotes in the selection */
   if (sel_start_cursor.par->footnoteflag != LYX_NO_FOOTNOTE
       || sel_end_cursor.par->footnoteflag != LYX_NO_FOOTNOTE){
     tmppar = sel_start_cursor.par;
     while (tmppar != sel_end_cursor.par){
       if (tmppar->footnoteflag != sel_end_cursor.par->footnoteflag){
	 WriteAlert("Impossible operation", "Don't know what to do with half floats.", "sorry.");
	 return;
       }
       tmppar = tmppar->Next();
     }
   }

   /* table stuff -- begin*/
   if (sel_start_cursor.par->table || sel_end_cursor.par->table){
     if ( sel_start_cursor.par != sel_end_cursor.par){
       WriteAlert("Impossible operation", "Don't know what to do with half tables.", "sorry.");
       return;
     }
   }
   /* table stuff -- end*/
   
   /* delete the simple_cut_buffer */ 
   DeleteSimpleCutBuffer();

   /* set the textclass */
   simple_cut_buffer_textclass = parameters->textclass;

   // copy behind a space if there is one
   while (sel_start_cursor.par->Last() > sel_start_cursor.pos
	  && sel_start_cursor.par->IsLineSeparator(sel_start_cursor.pos)
	  && (sel_start_cursor.par != sel_end_cursor.par
	      || sel_start_cursor.pos < sel_end_cursor.pos))
     sel_start_cursor.pos++; 

   /* there are two cases: copy only within one paragraph or more than one paragraph */
   if (sel_start_cursor.par->ParFromPos(sel_start_cursor.pos) 
       == sel_end_cursor.par->ParFromPos(sel_end_cursor.pos)) {
     /* only within one paragraph */
     simple_cut_buffer = new LyXParagraph();
     for (i=sel_start_cursor.pos; i< sel_end_cursor.pos; i++){
       sel_start_cursor.par->CopyIntoMinibuffer(i);
       simple_cut_buffer->InsertFromMinibuffer(i - sel_start_cursor.pos);
}
   }
   else {
     /* copy more than one paragraph */ 
     /* clone the paragraphs within the selection*/
     tmppar = sel_start_cursor.par->ParFromPos(sel_start_cursor.pos);
     simple_cut_buffer = tmppar->Clone();
     LyXParagraph *tmppar2 = simple_cut_buffer;
     
     while (tmppar != sel_end_cursor.par->ParFromPos(sel_end_cursor.pos)
	    && tmppar->next) {
       tmppar = tmppar->next;
       tmppar2->next = tmppar->Clone();
       tmppar2->next->previous = tmppar2;
       tmppar2=tmppar2->next;
     }
     tmppar2->next = NULL;

     /* care about footnotes */
     if (simple_cut_buffer->footnoteflag) {
       tmppar = simple_cut_buffer;
       while (tmppar){
        tmppar->footnoteflag = LYX_NO_FOOTNOTE;
        tmppar = tmppar->next;
       }
     }
     
     /* the simple_cut_buffer paragraph is too big */
     int tmpi2;

     tmpi2 = sel_start_cursor.par->PositionInParFromPos(sel_start_cursor.pos);
     for (;tmpi2;tmpi2--)
       simple_cut_buffer->Erase(0);

     /* now tmppar 2 is too big, delete all after sel_end_cursor.pos */
     
     tmpi2 = sel_end_cursor.par->PositionInParFromPos(sel_end_cursor.pos);
     while (tmppar2->last > tmpi2) {
       tmppar2->Erase(tmppar2->last-1);
     }

   }
}
          

void LyXText::PasteSelection()
{
   /* this does not make sense, if there is nothing to paste */ 
   if (!simple_cut_buffer)
     return;

   LyXParagraph *tmppar;
   LyXParagraph *endpar;

   LyXCursor tmpcursor;

   /* be carefull with footnotes in footnotes */ 
   if (cursor.par->footnoteflag != LYX_NO_FOOTNOTE) {
      
      /* check wether the cut_buffer includes a footnote */
      tmppar = simple_cut_buffer;
      while (tmppar && tmppar->footnoteflag == LYX_NO_FOOTNOTE)
      	tmppar = tmppar->next;
      
      if (tmppar) {
	WriteAlert("Impossible operation",
		   "Float would include float!", "sorry.");
	return;
      }
   }

   /* table stuff -- begin*/
   if (cursor.par->table){
     if (simple_cut_buffer->next){
       WriteAlert("Impossible operation",
		  "Table cell cannot include more than one paragraph!",
		  "sorry.");
       return;
     }
   }
   /* table stuff -- end*/
   
   SetUndo(LYX_UNDO_INSERT, 
	   cursor.par->ParFromPos(cursor.pos)->previous, 
	   cursor.par->ParFromPos(cursor.pos)->next); 

   tmpcursor = cursor;

   /* there are two cases: cutbuffer only one paragraph or many */
   if (!simple_cut_buffer->next) {
     /* only within a paragraph */
     
       /* please break behind a space, if there is one */
     while (tmpcursor.par->Last() > tmpcursor.pos
     && tmpcursor.par->IsLineSeparator(tmpcursor.pos))
       tmpcursor.pos++; 

     tmppar = simple_cut_buffer->Clone();
     /* table stuff -- begin*/
     char table_too_small = 0;
     if (tmpcursor.par->table) {
       while (simple_cut_buffer->last && !table_too_small){
	 if (simple_cut_buffer->IsNewline(0)){
	   while(tmpcursor.pos < tmpcursor.par->Last() && !tmpcursor.par->IsNewline(tmpcursor.pos))
	     tmpcursor.pos++;
	   simple_cut_buffer->Erase(0);
	   if (tmpcursor.pos < tmpcursor.par->Last())
	     tmpcursor.pos++;
	   else
	     table_too_small = 1;
	 } else {
	   simple_cut_buffer->CutIntoMinibuffer(0);
	   simple_cut_buffer->Erase(0);
	   tmpcursor.par->InsertFromMinibuffer(tmpcursor.pos);
	   tmpcursor.pos++;
	 }
       }
     } else {
       /* table stuff -- end*/
       while (simple_cut_buffer->last){
	 simple_cut_buffer->CutIntoMinibuffer(0);
	 simple_cut_buffer->Erase(0);
	 tmpcursor.par->InsertFromMinibuffer(tmpcursor.pos);
	 tmpcursor.pos++;
       }
     }

     delete simple_cut_buffer;
     simple_cut_buffer = tmppar;
     endpar = tmpcursor.par->Next();
   }
   else {
     /* many paragraphs */

     /* make a copy of the simple cut_buffer */
     tmppar = simple_cut_buffer;
     LyXParagraph *simple_cut_clone = tmppar->Clone();
     LyXParagraph *tmppar2 = simple_cut_clone;
     if (cursor.par->footnoteflag){
       tmppar->footnoteflag = cursor.par->footnoteflag;
       tmppar->footnotekind = cursor.par->footnotekind;
     }
     while (tmppar->next) {
       tmppar = tmppar->next;
       tmppar2->next = tmppar->Clone();
       tmppar2->next->previous = tmppar2;
       tmppar2=tmppar2->next;
       if (cursor.par->footnoteflag){
        tmppar->footnoteflag = cursor.par->footnoteflag;
        tmppar->footnotekind = cursor.par->footnotekind;
       }
     }
     
     /* make sure there is no class difference */ 
     SwitchLayoutsBetweenClasses(simple_cut_buffer_textclass, parameters->textclass, simple_cut_buffer);
     
     /* make the simple_cut_buffer exactly the same layout than
	the cursor paragraph */
     simple_cut_buffer->MakeSameLayout(cursor.par);
     
     /* find the end of the buffer */ 
     LyXParagraph *lastbuffer = simple_cut_buffer;
     while (lastbuffer->Next())
     lastbuffer=lastbuffer->Next();
     
     /* find the physical end of the buffer */ 
     lastbuffer = simple_cut_buffer;
     while (lastbuffer->Next())
       lastbuffer=lastbuffer->Next();
     
     /* please break behind a space, if there is one. The space should
      * be copied too */ 
     if (cursor.par->Last() > cursor.pos && cursor.par->IsLineSeparator(cursor.pos))
       cursor.pos++; 
     
     char paste_the_end = 0;

     /* open the paragraph for inserting the simple_cut_buffer if necessary */
     if (cursor.par->Last() > cursor.pos || !cursor.par->Next()){
       cursor.par->BreakParagraphConservative(cursor.pos);
       paste_the_end = 1;
     }
     
     /* be careful with double spaces */ 
     if ((!cursor.par->Last()
	|| cursor.par->IsLineSeparator(cursor.pos - 1)
	|| cursor.par->IsNewline(cursor.pos - 1))
	 && simple_cut_buffer->last
	 && simple_cut_buffer->IsLineSeparator(0))
       simple_cut_buffer->Erase(0);
     
     /* set the end for redoing later */ 
     endpar = cursor.par->ParFromPos(cursor.pos)->next->Next();
     
   /* paste it! */ 
     lastbuffer->ParFromPos(lastbuffer->Last())->next = cursor.par->ParFromPos(cursor.pos)->next;
     cursor.par->ParFromPos(cursor.pos)->next->previous = lastbuffer->ParFromPos(lastbuffer->Last());
     
     cursor.par->ParFromPos(cursor.pos)->next = simple_cut_buffer;
     simple_cut_buffer->previous = cursor.par->ParFromPos(cursor.pos);
   
     if (cursor.par->ParFromPos(cursor.pos)->Next() == lastbuffer)
       lastbuffer = cursor.par;
     
     cursor.par->ParFromPos(cursor.pos)->PasteParagraph();
     
     /* store the new cursor position  */
     tmpcursor.par = lastbuffer;
     tmpcursor.pos = lastbuffer->Last();
     
     /* maybe some pasting */ 
     if (lastbuffer->Next() && paste_the_end) {
       if (lastbuffer->Next()->HasSameLayout(lastbuffer)) {
	 
	 /* be careful witth double spaces */ 
	 if ((!lastbuffer->Last()
	      || lastbuffer->IsLineSeparator(lastbuffer->Last() - 1)
	      || lastbuffer->IsNewline(lastbuffer->Last() - 1))
	     && lastbuffer->Next()->Last()
	     && lastbuffer->Next()->IsLineSeparator(0))
	   lastbuffer->Next()->Erase(0);
	 
	 lastbuffer->ParFromPos(lastbuffer->Last())->PasteParagraph();
	 
       }
       else if (!lastbuffer->Next()->Last()) {
	 lastbuffer->Next()->MakeSameLayout(lastbuffer);
	 
	 /* be careful witth double spaces */ 
	 if ((!lastbuffer->Last()
	      || lastbuffer->IsLineSeparator(lastbuffer->Last() - 1)
	      || lastbuffer->IsNewline(lastbuffer->Last() - 1))
	     && lastbuffer->Next()->Last()
	     && lastbuffer->Next()->IsLineSeparator(0))
	   lastbuffer->Next()->Erase(0);
	 
	 lastbuffer->ParFromPos(lastbuffer->Last())->PasteParagraph();
	 
       }
       else if (!lastbuffer->Last()) {
	 lastbuffer->MakeSameLayout(lastbuffer->next);
	 
	 /* be careful witth double spaces */ 
	 if ((!lastbuffer->Last()
	      || lastbuffer->IsLineSeparator(lastbuffer->Last() - 1)
	      || lastbuffer->IsNewline(lastbuffer->Last() - 1))
	     && lastbuffer->Next()->Last()
	     && lastbuffer->Next()->IsLineSeparator(0))
	   lastbuffer->Next()->Erase(0);
	 
	 lastbuffer->ParFromPos(lastbuffer->Last())->PasteParagraph();
	 
       }
       else lastbuffer->Next()->ClearParagraph();
     }

     /* restore the simple cut buffer */
     simple_cut_buffer = simple_cut_clone;
   }

   RedoParagraphs(cursor, endpar);
    
  SetCursor(cursor.par, cursor.pos);
   ClearSelection();
   
   sel_cursor = cursor;
   SetCursor(tmpcursor.par, tmpcursor.pos);
   SetSelection();
   UpdateCounters(cursor.row);
}
   

/* returns a pointer to the very first LyXParagraph */ 
LyXParagraph* LyXText::FirstParagraph()
{
   return parameters->paragraph;
}


/* returns true if the specified string is at the specified position */
bool LyXText::IsStringInText(LyXParagraph *par, int pos, char* string)
{
  int i;

  if (par) {
    i = 0;
    while (pos+i < par->Last() && string[i] && string[i]==par->GetChar(pos+i))
    {
      i++;
    }
    
    if (!string[i])
      return true;
  }
  return false;
}


/* sets the selection over the number of characters of string, no check!! */
void LyXText::SetSelectionOverString(char* string)
{
  sel_cursor = cursor;
  int i;
  for (i=0; string[i]; i++)
    CursorRight();
  SetSelection();
}


/* simple replacing. The font of the first selected character is used */
void LyXText::ReplaceSelectionWithString(char* string)
{
  int i=0;
  
  SetCursorParUndo();
  FreezeUndo();

  if (!selection) { /* create a dummy selection */
    sel_end_cursor = cursor;
    sel_start_cursor = cursor;
  }
 
  /* insert the new string behind the selection */
  LyXFont font = sel_start_cursor.par->GetFontSettings(sel_start_cursor.pos);
  for (i=0; string[i];i++); /* the end of the string */
  while (i) {
    i--;
    sel_end_cursor.par->InsertChar(sel_end_cursor.pos, string[i]);
    sel_end_cursor.par->SetFont(sel_end_cursor.pos, font);
  }

  /* cut the selection*/
  CutSelection();
  UnFreezeUndo();
}


/* if the string can be found: return true and set the cursor to
 * the new position */
bool LyXText::SearchForward(char* string)
{
  LyXParagraph *par = cursor.par;
  int pos = cursor.pos;

  while (par && !IsStringInText(par,pos,string)) {
    if (pos<par->Last()-1)
      pos++;
    else {
      pos = 0;
      par = par->Next();
    }
  }
  if (par) {
    SetCursor(par,pos);
    return true;
  }
  else
    return false;
}


bool LyXText::SearchBackward(char* string)
{
  LyXParagraph *par = cursor.par;
  int pos = cursor.pos;

  do {
    if (pos>0)
      pos--;
    else {
      par = par->Previous();
      if (par)
	pos = par->Last()-1;
    }
  } while (par && !IsStringInText(par,pos,string));
  
  if (par) {
    SetCursor(par,pos);
    return true;
  }
  else
    return false;
}


/* needed to insert the selection */
void LyXText::InsertStringA(char* string)
{
  LyXParagraph *par = cursor.par;
  int pos = cursor.pos;
   int a = 0;
  LyXParagraph *endpar = cursor.par->Next();

  SetCursorParUndo();
   
   char flag = (lyxstyle.Style(parameters->textclass, cursor.par->GetLayout())->latextype >= LATEX_ENVIRONMENT);
   /* only to be sure, should not be neccessary */ 
   ClearSelection();
   
   /* insert the string, don't insert doublespace */ 
  int i=0;
  int i2 = 0;
   /* the old idea that was more strict 
  while (string[i]) {
    if (string[i]!='\n') {
       if (string[i]!=' ' || pos > 0) {
	  if ((i<1 && string[i]!=' ')  
	      || (i>0 && (string[i]!=' ' || string[i-1]!=' '))) {
		 if (string[i] != 13) {
		    par->InsertChar(pos,string[i]);
		    pos++;
		 }
	      }
       }
    }
    else {
      if (i>0 && pos>0 && string[i-1]!='\n') {
	 par->BreakParagraph(pos, flag);
	 par = par->Next();
	 pos = 0;
      }
    }  
     i++;
  }
   */ 
   
  for (i2=i;string[i2]&&string[i2]!='\n';i2++);
  par->Enlarge(pos, i2 - i);
  while (string[i]) {
    if (string[i]!='\n') {
       if (string[i]==' ' && (string[i+1]!=' ')
	   && pos && par->GetChar(pos-1)!=' ') {
	  par->InsertChar(pos,' ');
	  pos++;
       }
       else if (string[i]==' ') {
	  par->InsertChar(pos,LYX_META_PROTECTED_SEPARATOR);
	  pos++;
       }
       else if (string[i]=='\t') {
	  for (a=pos; a<(pos/8+1)*8 ; a++) {
	     par->InsertChar(a,LYX_META_PROTECTED_SEPARATOR);
	  }
	  pos = a;
       }
       else if (string[i]!=13 && 
                ((unsigned char) string[i] & 127) >= ' ' // Ignore unprintables
                ) {
	  par->InsertChar(pos,string[i]);
	  pos++;
       }
    }
      else {
	 if (!par->last) {
	    par->InsertChar(pos,LYX_META_PROTECTED_SEPARATOR);
	    pos++;
	 }
	 par->BreakParagraph(pos, flag);
	 par = par->Next();
	 pos = 0;
	 for (i2=i;string[i2]&&string[i2]!='\n';i2++);
	 par->Enlarge(pos, i2 - i);
      }
      
      i++;
   }
   
   RedoParagraphs(cursor,endpar);
   SetCursor(cursor.par, cursor.pos);
   sel_cursor = cursor;
   SetCursor(par, pos);
   SetSelection();
}


/* turns double-CR to single CR, others where converted into one blank and 13s 
 * that are ignored .Double spaces are also converted into one. Spaces at
 * the beginning of a paragraph are forbidden. tabs are converted into one
 * space. then InsertStringA is called */ 
void LyXText::InsertStringB(char* string)
{
   int i=1;
   while (string[i]) {
      if (string[i]=='\t')
      	 string[i] = ' ';
      if (string[i]==' ' && string[i+1]==' ')
      	 string[i] = 13;
      if (string[i]=='\n' && string[i+1]){
	 if (string[i+1]!='\n') {
	    if (string[i-1]!=' ')
	       string[i]=' ';
	    else
	       string[i]= 13;
	 }
	 while (string[i+1] && (string[i+1]==' '
				|| string[i+1]=='\t'
				|| string[i+1]=='\n'
				|| string[i+1]==13)) {
	    string[i+1]=13;
	    i++;
	 }
      }
      i++;
   }
   InsertStringA(string);
}


bool LyXText::GotoNextError()
{
   LyXCursor res=cursor;
   do {
      if (res.pos < res.par->Last()-1) {
	 res.pos++;
      }
      else  {
	 res.par=res.par->Next();
	 res.pos = 0;
      }
      
   } while (res.par && 
	    !(res.par->GetChar(res.pos)==LYX_META_INSET
	      && res.par->GetInset(res.pos)->AutoDelete()));
   
   if (res.par) {
      SetCursor(res.par, res.pos);
      return true;
   }
   
   return false;
}


bool LyXText::GotoNextNote()
{
   LyXCursor res=cursor;
   do {
      if (res.pos < res.par->Last()-1) {
	 res.pos++;
      }
      else  {
	 res.par=res.par->Next();
	 res.pos = 0;
      }
      
   } while (res.par && 
	    !(res.par->GetChar(res.pos)==LYX_META_INSET
	      && res.par->GetInset(res.pos)->LyxCode()==LYX_IGNORE_CODE));
   
   if (res.par) {
      SetCursor(res.par, res.pos);
      return true;
   }
   
   return false;
}


int LyXText::SwitchLayoutsBetweenClasses(char class1, char class2,
					 LyXParagraph *par)
{
   char* name = NULL;
   char* s;
   InsetError * new_inset = NULL;
   int ret = 0;
   if (!par || class1 == class2)
      return ret;
   par = par->FirstPhysicalPar();
   while (par) {
      name = lyxstyle.NameOfLayout(class1, par->layout);
      par->layout = lyxstyle.NumberOfLayout(class2, name);
      
      if (!StringEqual(name, lyxstyle.NameOfLayout(class2, par->layout))) {
	 ret++;
	 s = new char[300];
	 sprintf(s, "Layout had to be changed from\n%s to %s\nbecause of class conversion from\n%s to %s", 
		 name, lyxstyle.NameOfLayout(class2, par->layout), lyxstyle.NameOfClass(class1), lyxstyle.NameOfClass(class2));
	 new_inset = new InsetError(s);
	 par->InsertChar(0, LYX_META_INSET);
	 par->InsertInset(0, new_inset);
	 delete[] s;
      }
      
      par = par->next;
   }
   return ret;
}


void LyXText::CheckParagraph(LyXParagraph* par, int pos)
{
  ClearSelection();

  FullRebreak();

   /* table stuff -- begin*/
   
   if (par->table) {
     CheckParagraphInTable(par, pos);
     return;
   }
   /* table stuff -- end*/
  
  long y = 0;
  int z;

  Row* row = GetRow(par, pos, y);

  LyXCursor tmpcursor;

  /* is there a break one row above */ 
   if (row->previous && row->previous->par == row->par) {
     z = NextBreakPoint(row->previous, paperwidth);
     if ( z >= row->pos) {
       /* set the dimensions of the row above  */ 
       y -= row->previous->height;
       refresh_y = y;
       refresh_row = row->previous;
       status = LYX_NEED_MORE_REFRESH;
       
       BreakAgain(row->previous);

       /* set the cursor again. Otherwise dungling pointers are possible */
       SetCursor(cursor.par, cursor.pos);
       sel_cursor = cursor;
       return;
     }
   }

   int tmpheight = row->height;
   int tmplast = RowLast(row);
   refresh_y = y;
   refresh_row = row;
   BreakAgain(row);
   if (row->height == tmpheight && RowLast(row) == tmplast)
     status = LYX_NEED_VERY_LITTLE_REFRESH;
   else
     status = LYX_NEED_MORE_REFRESH; 
   
   /* check the special right address boxes */
   if (lyxstyle.Style(parameters->textclass, par->GetLayout())->margintype == MARGIN_RIGHT_ADDRESS_BOX) {
     tmpcursor.par = par;
     tmpcursor.row = row;
     tmpcursor.y = y;
     tmpcursor.x = 0;
     tmpcursor.pos = pos;
     RedoDrawingOfParagraph(tmpcursor); 
   }
   
   /* set the cursor again. Otherwise dungling pointers are possible */
   SetCursor(cursor.par, cursor.pos);
   sel_cursor = cursor; 
}


/* returns 0 if inset wasn't found */
int LyXText::UpdateInset(Inset* inset)
{
  int pos;
  LyXParagraph *par;

  /* first check the current paragraph */
  pos = cursor.par->GetPositionOfInset(inset);
  if (pos != -1){
    CheckParagraph(cursor.par, pos);
    return 1;
  }
  
  /* check every paragraph */
  
  par = FirstParagraph();
  do {
    /* make sure the paragraph is open */
    if (par->footnoteflag != LYX_CLOSED_FOOTNOTE){
      pos = par->GetPositionOfInset(inset);
      if (pos != -1){
	CheckParagraph(par, pos);
	return 1;
      }
    }
    par = par->Next();
  } while (par);
  
  return 0;
}


void LyXText::SetCursor(LyXParagraph *par, int pos)
{
  LyXCursor old_cursor;
  old_cursor = cursor;

  SetCursorIntern(par, pos);
  DeleteEmptyParagraphMechanism(old_cursor);
}


void LyXText::SetCursorIntern(LyXParagraph *par, int pos)
{
   long y;
   Row *row;
   int left_margin;
   LyXParagraph *tmppar;
   
   /* correct the cursor position if impossible */
   if (pos > par->Last()){
     tmppar = par->ParFromPos(pos);
     pos = par->PositionInParFromPos(pos);
     par = tmppar;
   }
   if (par->IsDummy()
       && par->previous
       && par->previous->footnoteflag == LYX_CLOSED_FOOTNOTE){
     while (par->previous && 
	    par->previous->footnoteflag == LYX_CLOSED_FOOTNOTE){
       par = par->previous ;
     }
     par = par->previous;
     pos += par->last + 1;
   }

   cursor.par = par;
   cursor.pos = pos;

   /* get the cursor y position in text  */
   row = GetRow(par, pos, y);
   /* y is now the beginning of the cursor row */ 
   y += row->baseline;
   /* y is now the cursor baseline */ 
   cursor.y = y;
   
   /* now get the cursors x position */
   
   float x;
   float fill_separator, fill_hfill, fill_label_hfill;
   left_margin = LabelEnd(row);
   PrepareToPrint(row, x, fill_separator, fill_hfill, fill_label_hfill);
   int main_body = BeginningOfMainBody(row->par);
      
   /* table stuff -- begin*/
   if (row->par->table) {
      int cell = NumberOfCell(row->par, row->pos);
      float x_old = x;
      x += row->par->table->GetBeginningOfTextInCell(cell);
      for (pos = row->pos; pos < cursor.pos; pos++)  {
	 if (row->par->IsNewline(pos)) {
	    x = x_old + row->par->table->WidthOfColumn(cell);
	    x_old = x;
	    cell++;
	    x += row->par->table->GetBeginningOfTextInCell(cell);
	 }
	 else {
	    x += SingleWidth(row->par, pos);
	 }
      }
   }
   else
      /* table stuff -- end*/

   for (pos = row->pos; pos < cursor.pos; pos++)  {
      if (pos && pos == main_body
	  && !row->par->IsLineSeparator(pos - 1)) {
	x += LyXStringWidth(GetFont(row->par, -2),
			    lyxstyle.Style(parameters->textclass, row->par->GetLayout())->labelsep);
	     if (x < left_margin)
	       x = left_margin;
	  }
      
      x += SingleWidth(row->par, pos);
      if (HfillExpansion(row, pos)) {
	 if (pos >= main_body)
	   x += fill_hfill;
	 else 
	   x += fill_label_hfill;
      }
      else if (pos >= main_body && row->par->IsSeparator(pos)) {
	 x+= fill_separator;
      }
      
      if (pos + 1 == main_body
	  && row->par->IsLineSeparator(pos)) {
	     x += LyXStringWidth(GetFont(row->par, -2),
				 lyxstyle.Style(parameters->textclass, row->par->GetLayout())->labelsep);
	     if (row->par->IsLineSeparator(pos))
	       x-= SingleWidth(row->par, pos);
	     if (x < left_margin)
	       x = left_margin;
	  }
   }
   
   cursor.x = (int)x;
   
   cursor.x_fix = cursor.x;
   cursor.row = row;
   
   if (cursor.pos && 
       (cursor.pos == cursor.par->Last() || cursor.par->IsSeparator(cursor.pos)
	|| (cursor.pos && cursor.pos == BeginningOfMainBody(cursor.par)
	    && !cursor.par->IsSeparator(cursor.pos))
	)){
	   current_font = cursor.par->GetFontSettings(cursor.pos - 1);
	   real_current_font = GetFont(cursor.par, cursor.pos - 1);
	}
   else {
      current_font = cursor.par->GetFontSettings(cursor.pos);
      real_current_font = GetFont(cursor.par, cursor.pos);
   }
}


void LyXText::SetCursorFromCoordinates(int x, long y)
{
   Row *row;
   int column;

   LyXCursor old_cursor;
   
   old_cursor = cursor;
   
   /* get the row first */ 
   
   row = GetRowNearY(y);
   
   cursor.par = row->par;
   
   column = GetColumnNearX(row, x);
   cursor.pos = row->pos + column;
   cursor.x = x;
   cursor.y = y + row->baseline;
   
   cursor.row = row;
    
   if (cursor.pos && 
       (cursor.pos == cursor.par->Last() || cursor.par->IsSeparator(cursor.pos)
	|| (cursor.pos && cursor.pos == BeginningOfMainBody(cursor.par)
	    && !cursor.par->IsSeparator(cursor.pos))
	)){
	   current_font = cursor.par->GetFontSettings(cursor.pos - 1);
	   real_current_font = GetFont(cursor.par, cursor.pos - 1);
	}
   else {
      current_font = cursor.par->GetFontSettings(cursor.pos);
      real_current_font = GetFont(cursor.par, cursor.pos);
   }
   DeleteEmptyParagraphMechanism(old_cursor);
}


void LyXText::CursorLeft()
{
  CursorLeftIntern();
}


void LyXText::CursorLeftIntern()
{
   if (cursor.pos > 0) {
      SetCursor(cursor.par, cursor.pos - 1);
   }
   else if (cursor.par->Previous()) {
      SetCursor(cursor.par->Previous(), cursor.par->Previous()->Last());
   }
}


void LyXText::CursorRight()
{
  CursorRightIntern();
}


void LyXText::CursorRightIntern()
{
	if (cursor.pos < cursor.par->Last()) {
		SetCursor(cursor.par, cursor.pos + 1);
	}
	else if (cursor.par->Next()) {
		SetCursor(cursor.par->Next(), 0);
	}
}


void LyXText::CursorUp()
{
	SetCursorFromCoordinates(cursor.x_fix, 
				 cursor.y - cursor.row->baseline - 1);
}


void LyXText::CursorDown()
{
	SetCursorFromCoordinates(cursor.x_fix, 
				 cursor.y - cursor.row->baseline
				 + cursor.row->height + 1);
}


void LyXText::DeleteEmptyParagraphMechanism(LyXCursor old_cursor)
{
	bool deleted = false;
	
	/* this is the delete-empty-paragraph-mechanism. */ 
	if (selection)
		return;

	LyXCursor tmpcursor;

	if (old_cursor.par != cursor.par) {
		if ( (old_cursor.par->Last() == 0
		|| (old_cursor.par->Last() == 1
		&& (old_cursor.par->IsLineSeparator(0))))
		     && old_cursor.par->FirstPhysicalPar()
		     == old_cursor.par->LastPhysicalPar()
		     && (lyxstyle.Style(parameters->textclass,
					old_cursor.par->GetLayout())->labeltype!=LABEL_SENSITIVE
		     || (old_cursor.par->footnoteflag == LYX_NO_FOOTNOTE
			 || (old_cursor.par->footnotekind != LYX_FIG
			     && old_cursor.par->footnotekind != LYX_TAB)))) {
	     
			/* ok, we will delete anything */ 
			
			// make sure that you do not delete any environments
			if ((old_cursor.par->footnoteflag != LYX_OPEN_FOOTNOTE &&
			     !(old_cursor.row->previous 
			       && old_cursor.row->previous->par->footnoteflag == LYX_OPEN_FOOTNOTE)
			&& !(old_cursor.row->next 
			     && old_cursor.row->next->par->footnoteflag == LYX_OPEN_FOOTNOTE))
			    || 
			    (old_cursor.par->footnoteflag == LYX_OPEN_FOOTNOTE &&
			     ((old_cursor.row->previous 
			       && old_cursor.row->previous->par->footnoteflag == LYX_OPEN_FOOTNOTE)
			      || 
			      (old_cursor.row->next
			       && old_cursor.row->next->par->footnoteflag == LYX_OPEN_FOOTNOTE))
				    )){
				status = LYX_NEED_MORE_REFRESH;
				deleted = true;
				
				if (old_cursor.row->previous) {
					refresh_row = old_cursor.row->previous;
					refresh_y = old_cursor.y - old_cursor.row->baseline - refresh_row->height;
					tmpcursor = cursor;
					cursor = old_cursor; // that undo can restore the right cursor position
					LyXParagraph *endpar = old_cursor.par->next;
					if (endpar && endpar->GetDepth()) {
					  while (endpar && endpar->GetDepth()) {
					    endpar = endpar->LastPhysicalPar()->Next();
					  }
					}
					SetUndo(LYX_UNDO_DELETE,
						old_cursor.par->previous,
						endpar);
					cursor = tmpcursor;

					/* delete old row */ 
					RemoveRow(old_cursor.row);
					if (parameters->paragraph == old_cursor.par) {
					  parameters->paragraph = parameters->paragraph->next;
					}
					/* delete old par */ 
					delete old_cursor.par;
					
					/* Breakagain the next par. Needed
					 * because of the parindent that
					 * can occur or dissappear. The
					 * next row can change its height,
					 * if there is another layout before */
					if (refresh_row->next) {
						BreakAgain(refresh_row->next);
						UpdateCounters(refresh_row);
					}
					SetHeightOfRow(refresh_row);
				}
				else {
					refresh_row = old_cursor.row->next;
					refresh_y = old_cursor.y - old_cursor.row->baseline;
					
					tmpcursor = cursor;
					cursor = old_cursor; // that undo can restore the right cursor position
					LyXParagraph *endpar = old_cursor.par->next;
					if (endpar && endpar->GetDepth()) {
					  while (endpar && endpar->GetDepth()) {
					    endpar = endpar->LastPhysicalPar()->Next();
					  }
					}
					SetUndo(LYX_UNDO_DELETE,
						old_cursor.par->previous,
						endpar);
					cursor = tmpcursor;

					/* delete old row */ 
					RemoveRow(old_cursor.row);
					/* delete old par */ 
					if (parameters->paragraph == old_cursor.par) {
					  parameters->paragraph = parameters->paragraph->next;
					}
					delete old_cursor.par;
					
					/* Breakagain the next par. Needed because of
					 * the parindent that can occur or dissappear.
					 * The next row can change its height, if there
					 * is another layout before */ 
					if (refresh_row) {
						BreakAgain(refresh_row);
						UpdateCounters(refresh_row->previous);
					}
				}
				
				/* correct cursor y */
				SetCursor(cursor.par, cursor.pos);
		     
				/* if (cursor.y > old_cursor.y)
				   cursor.y -= old_cursor.row->height; */ 
	 
				if (sel_cursor.par  == old_cursor.par
				    && sel_cursor.pos == sel_cursor.pos) {
					/* correct selection*/ 
					sel_cursor = cursor;
				}
			}
       
		}
		if (!deleted){
			if (old_cursor.par->ClearParagraph()){
				RedoParagraphs(old_cursor, old_cursor.par->Next());
				/* correct cursor y */
				SetCursor(cursor.par, cursor.pos);
				sel_cursor = cursor;
			}
		}
	}
}


LyXParagraph* LyXText::GetParFromID(int id)
{
  LyXParagraph* result = FirstParagraph();
  while (result && result->GetID() != id)
    result = result->next;
  return result;
}


// undo functions
bool  LyXText::TextUndo()
{ // returns False if no undo possible
  Undo *undo = parameters->undostack.Pop();
  if (undo){
    FinishUndo();
    if (!undo_frozen)
	    parameters->redostack.Push(CreateUndo(undo->kind, 
						  GetParFromID(undo->number_of_before_par),
						  GetParFromID(undo->number_of_behind_par)));
  }
  return TextHandleUndo(undo);
}


bool LyXText::TextRedo()
{ // returns False if no redo possible
  Undo *undo = parameters->redostack.Pop();
  if (undo){
    FinishUndo();
    if (!undo_frozen)
      parameters->undostack.Push(CreateUndo(undo->kind, 
					    GetParFromID(undo->number_of_before_par),
					    GetParFromID(undo->number_of_behind_par)));
  }
  return TextHandleUndo(undo);
}


bool  LyXText::TextHandleUndo(Undo* undo){ // returns False if no undo possible
  bool result = false;
  if (undo){
    LyXParagraph* before = GetParFromID(undo->number_of_before_par); 
    LyXParagraph* behind = GetParFromID(undo->number_of_behind_par); 
    LyXParagraph* tmppar;
    LyXParagraph* tmppar2;
    LyXParagraph* tmppar3;
    LyXParagraph* tmppar4;
    LyXParagraph* endpar;
    LyXParagraph* tmppar5;
    
    /*
    if (before){
      before->text[before->last] = 0;
      printf("before: %s\n", before->text);
    }
    if (behind){
      behind->text[behind->last] = 0;
      printf("behind: %s\n", behind->text);
    }
    */ 

    // if there's no before take the beginning of the document for redoing
    if (!before)
      SetCursorIntern(FirstParagraph(), 0);

    // replace the paragraphs with the undo informations

    tmppar3 = undo->par;
    undo->par = NULL; // otherwise the undo destructor would delete the paragraph
    tmppar4 = tmppar3;
    if (tmppar4){
      while (tmppar4->next)
	tmppar4 = tmppar4->next;
    } // get last undo par
    
    // now remove the old text if there is any
    if (before != behind || (!behind && !before)){
      if (before)
	tmppar5 = before->next;
      else
	tmppar5 = parameters->paragraph;
      tmppar2 = tmppar3;
      while (tmppar5 && tmppar5 != behind){
	tmppar = tmppar5;
	tmppar5 = tmppar5->next;
	// a memory optimization for edit: Only layout information
	// is stored in the undo. So restore the text informations.
	if (undo->kind == LYX_UNDO_EDIT){
	  tmppar2->text = tmppar->text;
	  tmppar->text = NULL;
	  tmppar2 = tmppar2->next;
	}
	delete tmppar;
      }
    }
    
    // put the new stuff in the list if there is one
    if (tmppar3){
      if (before)
	before->next = tmppar3;
      else
	parameters->paragraph = tmppar3;
      tmppar3->previous = before;
    }
    else {
      if (!before)
	parameters->paragraph = behind;
    }
    if (tmppar4) {
      tmppar4->next = behind;
      if (behind)
	behind->previous = tmppar4;
    }
    
    
    // Set the cursor for redoing
    if (before){
      SetCursorIntern(before->FirstSelfrowPar(), 0);
      // check wether before points to a closed float and open it if necessary
      if (before && before->footnoteflag == LYX_CLOSED_FOOTNOTE
	  && before->next && before->next->footnoteflag != LYX_NO_FOOTNOTE){
	tmppar4 =before;
	while (tmppar4->previous && 
	       tmppar4->previous->footnoteflag == LYX_CLOSED_FOOTNOTE)
	  tmppar4 = tmppar4->previous;
	while (tmppar4 && tmppar4->footnoteflag == LYX_CLOSED_FOOTNOTE){
	  tmppar4->footnoteflag = LYX_OPEN_FOOTNOTE;
	  tmppar4 = tmppar4->next;
	}
      }
    }
    
    // open a cosed footnote at the end if necessary
    if (behind && behind->previous && 
	behind->previous->footnoteflag != LYX_NO_FOOTNOTE &&
	behind->footnoteflag == LYX_CLOSED_FOOTNOTE){
      while (behind && behind->footnoteflag == LYX_CLOSED_FOOTNOTE){
	behind->footnoteflag = LYX_OPEN_FOOTNOTE;
	behind = behind->next;
      }
    }
    
    // calculate the endpar for redoing the paragraphs.
    if (behind){
      if (behind->footnoteflag != LYX_CLOSED_FOOTNOTE)
	endpar = behind->LastPhysicalPar()->Next();
      else
	endpar = behind->NextAfterFootnote()->LastPhysicalPar()->Next();
    }
    else
      endpar = behind;
    
    tmppar = GetParFromID(undo->number_of_cursor_par);
    RedoParagraphs(cursor, endpar); 
    if (tmppar){
      SetCursorIntern(tmppar, undo->cursor_pos);
      UpdateCounters(cursor.row);
    }
    result = true;
    delete undo;
  }
  FinishUndo();
  return result;
}


void LyXText::FinishUndo()
{ // makes sure the next operation will be stored
  undo_finished = True;
}


void LyXText::FreezeUndo()
{ // this is dangerous and for internal use only
  undo_frozen = True;
}


void LyXText::UnFreezeUndo()
{ // this is dangerous and for internal use only
  undo_frozen = False;
}


void LyXText::SetUndo(char kind, LyXParagraph *before, LyXParagraph *behind)
{
  if (!undo_frozen)
    parameters->undostack.Push(CreateUndo(kind, before, behind));
  parameters->redostack.Clear();
}


void LyXText::SetRedo(char kind, LyXParagraph *before, LyXParagraph *behind)
{
  parameters->redostack.Push(CreateUndo(kind, before, behind));
}


Undo* LyXText::CreateUndo(char kind, LyXParagraph *before,
			  LyXParagraph *behind)
{
  int before_number = -1;
  int behind_number = -1;
  if (before)
    before_number = before->GetID();
  if (behind)
    behind_number = behind->GetID();
  // LYX_UNDO_EDIT  and LYX_UNDO_FINISH are
  // always finished. (no overlapping there)
  // overlapping only with insert and delete inside one paragraph: 
  // Nobody wants all removed  character
  // appear one by one when undoing. 
  // EDIT is special since only layout information, not the
  // contents of a paragaph are stored.
  if (!undo_finished && kind != LYX_UNDO_EDIT && 
      kind != LYX_UNDO_FINISH){
    // check wether storing is needed
    if (parameters->undostack.Top() && 
	parameters->undostack.Top()->kind == kind &&
	parameters->undostack.Top()->number_of_before_par ==  before_number &&
	parameters->undostack.Top()->number_of_behind_par ==  behind_number ){
      // no undo needed
      return NULL;
    }
  }
  // create a new Undo
  LyXParagraph* undopar;
  LyXParagraph* tmppar;
  LyXParagraph *tmppar2;

  LyXParagraph* start = NULL;
  LyXParagraph* end = NULL;
  
  if (before)
    start = before->next;
  else
    start = FirstParagraph();
  if (behind)
    end = behind->previous;
  else {
    end = FirstParagraph();
    while (end->next)
      end = end->next;
  }

  if (start && end && start != end->next && (before != behind || (!before && !behind))) {
    tmppar = start;
    tmppar2 = tmppar->Clone();
    tmppar2->SetID(tmppar->GetID());

    // a memory optimization: Just store the layout information when only edit
    if (kind == LYX_UNDO_EDIT){
      if (tmppar2->text)
	delete[] tmppar2->text;
      tmppar2->text = NULL;
    }

    undopar = tmppar2;
  
    while (tmppar != end && tmppar->next) {
      tmppar = tmppar->next;
      tmppar2->next = tmppar->Clone();
      tmppar2->next->SetID(tmppar->GetID());
      // a memory optimization: Just store the layout information when only edit
      if (kind == LYX_UNDO_EDIT){
	if (tmppar2->next->text)
	delete[] tmppar2->next->text;
	tmppar2->next->text = NULL;
      }
      tmppar2->next->previous = tmppar2;
      tmppar2=tmppar2->next;
    }
    tmppar2->next = NULL;
  }
  else
    undopar = NULL; // nothing to replace (undo of delete maybe)
  
  int cursor_par = cursor.par->ParFromPos(cursor.pos)->GetID();
  int cursor_pos =  cursor.par->PositionInParFromPos(cursor.pos);

  Undo* undo = new Undo(kind, 
			before_number, behind_number,  
			cursor_par, cursor_pos, 
			undopar);
  
  undo_finished = False;
  return undo;
}


void LyXText::SetCursorParUndo()
{
  SetUndo(LYX_UNDO_FINISH, 
	  cursor.par->ParFromPos(cursor.pos)->previous, 
	  cursor.par->ParFromPos(cursor.pos)->next); 
}

