/* Copyright (c) 1995 John E. Davis
 * All rights reserved.
 * 
 * You may distribute under the terms of either the GNU General Public
 * License or the Perl Artistic License.
 */

/*-----------------------------------------------------------------
 * File:	slprepr.c
 *
 * preprocessing routines
 *
 *
 * various preprocessing tokens supported
 *
 * #ifdef  TOKEN1 TOKEN2 ...
 *	- True if any of TOKEN1 TOKEN2 ... are defined
 *
 * #ifndef TOKEN1 TOKEN2 ...
 *	- True if none of TOKEN1 TOKEN2 ... are defined
 *
 * #iftrue
 * #ifnfalse
 *	- always True
 *
 * #iffalse
 * #ifntrue
 *	- always False
 *
 * #if$ENV
 *	- True if the enviroment variable ENV is set
 *
 * #ifn$ENV
 *	- True if the enviroment variable ENV is not set
 *
 * #if$ENV TOKEN1 TOKEN2 ... *TOKENa TOKENb* *TOKENc*
 *	- True if the contents of enviroment variable ENV match
 *	  any of TOKEN1 TOKEN2 ... *TOKENa TOKENb* *TOKENc*, in which
 *	  the '*' character is a wildcard match
 *
 * #ifn$ENV TOKEN1 TOKEN2 ... *TOKENa TOKENb* *TOKENc*
 * #if!$ENV TOKEN1 TOKEN2 ... *TOKENa TOKENb* *TOKENc*
 *	- True if the contents of enviroment variable ENV does not match
 *	  any of TOKEN1 TOKEN2 ... *TOKENa TOKENb* *TOKENc*, in which
 *	  the '*' character is a wildcard match
 *
 * #elif...
 * #else
 * #endif
 *
 *
 * Note: the wildcard matching for $ENV can only be at the ends!
 *
 * mj olesen
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>

#include "config.h"

#ifndef NO_STDLIB_H
#include <stdlib.h>
#endif

#include "slang.h"

int SLprep_open_prep (SLPreprocess_Type *pt)
{
   pt->this_level = 0;
   pt->exec_level = 0;
   pt->prev_exec_level = 0;
   pt->comment_char = '%';
   pt->preprocess_char = '#';
   return 0;
}

void SLprep_close_prep (SLPreprocess_Type *pt)
{
}

#define MAX_DEFINES 10
#define MAX_DEFINE_LEN 16

static char SLdefines[MAX_DEFINES][MAX_DEFINE_LEN];

int SLdefine_for_ifdef (char *s)
{
   int n, i;
   char *place;
   
   for (i = 0; i < MAX_DEFINES; i++)
     {
	place = SLdefines[i];
	if (*place == 0)
	  {
	     n = strlen (s);
	     if (n > MAX_DEFINE_LEN - 2) n = MAX_DEFINE_LEN - 2;
	     *place++ = (char) n;
	     strncpy(place, s, n);
	     *(place + n) = 0;
	     return 1;
	  }
     }
   return 0;
}

static int is_any_defined(char *buf, char comment)
{
   char *sys, *buf_save = buf;
   int i = 0, n;
   register char ch;

   while ((i < MAX_DEFINES) && (sys = SLdefines[i], (n = (int) *sys++) != 0))
     {
	buf = buf_save;
	while (1)
	  {
	     while ((ch = *buf), ch && (ch != '\n') && (ch <= ' ')) buf++;
	     if ((ch <= '\n') || (ch == comment)) break;
	     if (!strncmp(buf, sys, n))
	       {
		  buf += n;
		  if ((*buf <= ' ') || (*buf == comment)) return 1;
	       }
	     else
	       {
		  while ((ch = *buf), (ch > ' ') && (ch != comment)) buf++;
	       }
	  }
	i++;
     }
   return 0;
}

#define TOKENSZ		31
#define GLOB_STAR	'*'
static int is_env_defined (unsigned char *buf, char comment)
{
   char *env, token[TOKENSZ + 1], *b, *bmax;
   int token_length;

   if ((*buf <= ' ') || ((char) *buf == comment)) return 0;	/* no token */

   b = token; bmax = b + TOKENSZ;
   
   while ((b < bmax) && (*buf > ' ')) *b++ = (char) *buf++;
   *b = '\0';
   if (*buf > ' ') return 0;	/* failed, token too long */

   if (NULL == (env = (char *) getenv(token)))
     return 0;			/* not defined */

   while ((*buf == ' ') || (*buf == '\t')) buf++;	/* leading spaces */
   if ((*buf == '\n') || ((char)*buf == comment) || (*buf == 0))
     return 1;			/* no tokens, but getenv() worked */

   do
     {
	b = token; bmax = b + TOKENSZ;
	while ((b < bmax) && (*buf > ' ')) *b++ = (char) *buf++;
	*b = '\0';
	if (*buf > ' ') break;	       /* Oops!  Too long */

	b--;
	token_length = (int) (b - token);    /* actually, one less than length
					      * since length is used only if globbing
					      * is performed.
					      */
	/* rudimentary globbing: */
	if (*b == GLOB_STAR)	       /* ...something* */
	  {
	     if (*token == GLOB_STAR)
	       {
		  *b = '\0';
		  if (strstr (env, &token[1]))
		    return 1;				/* `*str*' */
	       }
	     else if (! strncmp (env, token, token_length))
	       return 1;				/* `str*' */
	  }
	else if (*token == GLOB_STAR)  /* *something */
	  {
	     int n = strlen (env) - token_length;
	     if ((n >= 0) && ! strncmp (env + n, token + 1, token_length))
	       return 1;				/* `*str' */
	  }
	else if (! strcmp (env, token))
	  return 1;					/* `str' */
	
	while ((*buf == ' ') || (*buf == '\t')) buf++;
     }
   while (*buf && (*buf != '\n') && ((char) *buf != comment));

   return 0;
}

int SLprep_line_ok (char *buf, SLPreprocess_Type *pt)
{
   int level, prev_exec_level, exec_level;
   
   if ((buf == NULL) || (pt == NULL)) return (1);
   if (*buf == '\n') return (0);
   if (*buf == pt->comment_char) return (0);     /* since '%' is a comment */
   
   if (*buf != pt->preprocess_char)
     return (pt->this_level == pt->exec_level);
   
   level = pt->this_level;
   exec_level = pt->exec_level;
   prev_exec_level = pt->prev_exec_level;
   
   buf++;
   
   /* Allow '#!' to pass.  This could be a shell script with something
    like '#! /local/bin/slang'  */
   if ((*buf == '!') && (pt->preprocess_char == '#'))
     return 0;
   
   /* Allow whitespace as in '#   ifdef'  */
   while ((*buf == ' ') || (*buf == '\t')) buf++;
   if (*buf < 'a') return (level == exec_level);
   
   
   if (!strncmp(buf, "endif", 5))
     {
	if (level == exec_level) 
	  {
	     exec_level--;
	     prev_exec_level = exec_level;
	  }
	level--;
	if (level < prev_exec_level) prev_exec_level = level;
	goto done;
     }
   
   if ((buf[0] == 'e') && (buf[1] == 'l'))   /* else, elifdef, ... */
     {
	if ((level == exec_level + 1) 
	    && (prev_exec_level != level))
	  {
	     /* We are in position to execute */
	     buf += 2;
	     if ((buf[0] == 's') && (buf[1] == 'e'))
	       {
		  /* "else" */
		  exec_level = level;
		  goto done;
	       }

	     /* drop through to ifdef testing.  First set variable
	      * to values appropriate for ifdef testing.
	      */
	     level--;		       /* now == to exec level */
	  }
	else 
	  {
	     if (level == exec_level)
	       {
		  exec_level--;
	       }
	     goto done;
	  }
     }

   if ((buf[0] == 'i') && (buf[1] == 'f'))
     {
	int truth;
	
	if (level != exec_level) 
	  {
	     /* Not interested */
	     level++;
	     goto done;
	  }
	
	level++;
	
	buf += 2;
	if (buf[0] == 'n')
	  {
	     truth = 0;
	     buf++;
	  }
	else truth = 1;
	
	if (!strncmp (buf, "def", 3))
	  truth = (truth == is_any_defined(buf + 3, pt->comment_char));
	else if (!strncmp (buf, "false", 5))
	  truth = !truth;
	else if (*buf == '$')
	  truth = (truth == is_env_defined ((unsigned char *)buf + 1, pt->comment_char));
	else if (0 != strncmp (buf, "true", 4))
	  return 1;		       /* let it bomb */
	
	if (truth)
	  {
	     exec_level = level;
	     prev_exec_level = exec_level;
	  }
     }
   else return 1;  /* let it bomb. */

   done:
   
   if (exec_level < 0) return 1;
   
   pt->this_level = level;
   pt->exec_level = exec_level;
   pt->prev_exec_level = prev_exec_level;
   return 0;
}

   

#if 0
int main ()
{
   char buf[1024];
   SLPreprocess_Type pt;
   
   SLprep_open_prep (&pt);
   
   SLdefine_for_ifdef ("UNIX");
   
   while (NULL != fgets (buf, sizeof (buf) - 1, stdin))
     {
	if (SLprep_line_ok (buf, &pt))
	  {
	     fputs (buf, stdout);
	  }
     }
   
   SLprep_close_prep (&pt);
   return 0;
}
#endif
