/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: control.c,v 1.4 94/07/19 14:07:50 nau Exp $";

/* control panel
 */

#include <stdio.h>
#include <stdlib.h>

#include "global.h"

#include "data.h"
#include "dialog.h"
#include "error.h"
#include "misc.h"
#include "control.h"

#include <X11/Shell.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Toggle.h>

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Widget	OnOffWidgets[MAX_LAYER+3],	/* widget groups for on(off selection */
				DrawingWidgets[MAX_LAYER];

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	Widget		AddLabel(String, Widget, Widget, Widget);
static	void		AddDrawingLayerSelection(Widget);
static	void		AddOnOffSelection(Widget);
static	void		UpdateDrawingLayerSelection(void);
static	void		UpdateOnOffSelection(void);
static	Cardinal	GetGroupOfLayer(Cardinal);
static	void		ChangeGroupVisibility(Cardinal, Boolean);
static	void		CB_GetOnOff(Widget, XtPointer, XtPointer);
static	void		CB_GetDrawingLayer(Widget, XtPointer, XtPointer);

/* ---------------------------------------------------------------------------
 * adds a label without border, surrounded by additional space
 * at the specified position
 */
static Widget AddLabel(String Label, Widget Parent, Widget Top, Widget Left)
{
	Widget		label;
	Dimension	height;

	label = XtVaCreateManagedWidget("label", labelWidgetClass,
		Parent,
		XtNlabel, Label,
		XtNfromHoriz, Top,
		XtNfromVert, Left,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);
	XtVaGetValues(label, XtNheight, &height, NULL);
	XtVaSetValues(label, XtNheight, 3*height/2, NULL);
	return(label);
}

/* ---------------------------------------------------------------------------
 * adds the layer selection buttons to our dialog (downwards)
 */
static void AddDrawingLayerSelection(Widget Parent)
{
			Widget			last;
			int				i;
			XtTranslations	table;
			char			name[10];
	static	String			translations = 
		"<Btn1Down>,<Btn1Up>: set() notify() \n";

		/* install some changes in translation table to make sure that
		 * always ONE element is selected
		 */
	table = XtParseTranslationTable(translations);

		/* insert a label at the top */
	last = AddLabel("current", Parent, NULL, NULL);

	for (i = 0; i < MAX_LAYER; i++)
	{
		sprintf(name, "layer%-i", i+1);
		DrawingWidgets[i] = last = XtVaCreateManagedWidget(name, toggleWidgetClass,
			Parent,
			XtNforeground, PCB->Layer[i].Color,
			XtNfromHoriz, NULL,
			XtNfromVert, last,
			XtNleft, XtChainLeft,
			XtNright, XtChainLeft,
			XtNtop, XtChainTop,
			XtNbottom, XtChainTop,
			NULL);

			/* the first entry identifies the radiogroup,
			 * no widgets are added to this group, because the identification
			 * may change (function is to be performed by UpdateDrawingLayer())
			 * we also have to make sure that no NULL is passed to XtNradioData
			 */
		XtVaSetValues(last,
			XtNradioGroup, DrawingWidgets[0],
			NULL);
		XtOverrideTranslations(last, table);
		XtAddCallback(last, XtNcallback, CB_GetDrawingLayer, (XtPointer) i);
	}
}

/* ---------------------------------------------------------------------------
 * adds the layer On/Off selection buttons to our dialog (downwards)
 * the label for the layer buttons are set by UpdateOnOffSelection()
 */
static void AddOnOffSelection(Widget Parent)
{
	Widget		last;
	int			i;
	char		*text,
				name[10];
	Pixel		color;

		/* insert a label at the top */
	last = AddLabel("on/off", Parent, NULL, NULL);

	for (i = 0; i < MAX_LAYER+3; i++)
	{
		switch(i)
		{
			case MAX_LAYER:			/* settings for elements */
				color = PCB->ElementColor;
				text = "pkg. outline";
				strcpy(name, "elements");
				break;

			case MAX_LAYER+1:		/* settings for pins */
				color = PCB->PinColor;
				text = "pkg. pins";
				strcpy(name, "pins");
				break;

			case MAX_LAYER+2:		/* settings for vias */
				color = PCB->ViaColor;
				text = "vias";
				strcpy(name, "vias");
				break;

			default:				/* layers */
				color = PCB->Layer[i].Color;
				text = "";
				sprintf(name, "layer%-i", i+1);
				break;
		}
		OnOffWidgets[i] = last = XtVaCreateManagedWidget(name, toggleWidgetClass,
			Parent,
			XtNlabel, text,
			XtNforeground, color,
			XtNfromHoriz, NULL,
			XtNfromVert, last,
			XtNleft, XtChainLeft,
			XtNright, XtChainLeft,
			XtNtop, XtChainTop,
			XtNbottom, XtChainTop,
			NULL);
		XtAddCallback(last, XtNcallback, CB_GetOnOff, (XtPointer) i);
	}
}

/* ---------------------------------------------------------------------------
 * updates the layer selection buttons
 */
static void UpdateDrawingLayerSelection(void)
{
	int		i;

		/* set new radio data, PCB may have changed */
	for (i = 0; i < MAX_LAYER; i++)
		XtVaSetValues(DrawingWidgets[i],
			XtNlabel, UNKNOWN(PCB->Layer[i].Name),
			XtNradioData, (XtPointer) &PCB->Layer[i],
			NULL);

		/* switch one button ON and reset the flags that are set
		 * by the callback functions of the toggle widgets
		  */
	XawToggleSetCurrent(DrawingWidgets[0], (XtPointer) CURRENT);
	RedrawOnEnter = REDRAW_NONE;
}

/* ---------------------------------------------------------------------------
 * updates the layer On/Off selection buttons
 */
static void UpdateOnOffSelection(void)
{
	int			i;

		/* the buttons for elements, vias and pins never
		 * change their label, so we only check the layer buttons
		 */
	for (i = 0; i < MAX_LAYER; i++)
		XtVaSetValues(OnOffWidgets[i],
			XtNlabel, UNKNOWN(PCB->Layer[i].Name),
			XtNstate, PCB->Layer[i].On ? True : False,
			NULL);

		/* now set the state of elements, pins and vias */
	XtVaSetValues(OnOffWidgets[i++],
		XtNstate, PCB->ElementOn ? True : False, NULL);
	XtVaSetValues(OnOffWidgets[i++],
		XtNstate, PCB->PinOn ? True : False, NULL);
	XtVaSetValues(OnOffWidgets[i++],
		XtNstate, PCB->ViaOn ? True : False, NULL);
}

/* ----------------------------------------------------------------------
 * searches the group to which a layer belongs to
 * returns MAX_LAYER if no group is found
 */
static Cardinal GetGroupOfLayer(Cardinal Layer)
{
	int		group,
			i;

	for (group = 0; group < MAX_LAYER; group++)
		for (i = 0; i < PCB->LayerGroups.Number[group]; i++)
			if (PCB->LayerGroups.Entries[group][i] == Layer)
				return(group);
	return(MAX_LAYER);
}

/* ----------------------------------------------------------------------
 * changes the visibility of all layers in a group
 */
static void ChangeGroupVisibility(Cardinal Layer, Boolean On)
{
	int		group,
			layer,
			i;

		/* change at least the passed layer */
	PCB->Layer[Layer].On = On;

	if ((group = GetGroupOfLayer(Layer)) < MAX_LAYER)
		for (i = 0; i < PCB->LayerGroups.Number[group]; i++)
		{
			layer = PCB->LayerGroups.Entries[group][i];
			PCB->Layer[layer].On = On;
		}
	UpdateOnOffSelection();
}

/* ---------------------------------------------------------------------------
 * callback routine, called when current layer is changed
 */
static void CB_GetDrawingLayer(Widget W, XtPointer ClientData, XtPointer CallData)
{
	int	newtop = (int) ClientData,
		i;
	
		/* reorder layer stack, first find position of selected one */
	for (i = 0; i < MAX_LAYER; i++)
		if (LayerStack[i] == newtop)
			break;

		/* bring this element to the top of the stack */
	for(; i; i--)
		LayerStack[i] = LayerStack[i-1];

	LayerStack[0] = newtop;
	RedrawOnEnter = REDRAW_CURRENT;

		/* switch layer on if not already visible */
	if (!CURRENT->On)
	{
		ChangeGroupVisibility(newtop, True);
		RedrawOnEnter = REDRAW_ALL;
	}
}

/* ---------------------------------------------------------------------------
 * callback routine, called when On/Off flag of layer is changed
 */
static void CB_GetOnOff(Widget W, XtPointer ClientData, XtPointer CallData)
{
	Boolean	state,
			current = False;
	int		layer,
			i,
			group;

		/* get new state of widget */
	XtVaGetValues(W, XtNstate, &state, NULL);
	layer = (int) ClientData;

		/* set redraw flag if object exists */
	switch(layer)
	{
		case MAX_LAYER:			/* settings for elements */
			PCB->ElementOn = state;
			if (PCB->ElementN)
				RedrawOnEnter = REDRAW_ALL;
			break;

		case MAX_LAYER+1:		/* settings for pins */
			PCB->PinOn = state;
			if (PCB->ElementN)
				RedrawOnEnter = REDRAW_ALL;
			break;

		case MAX_LAYER+2:		/* settings for vias */
			PCB->ViaOn = state;
			if (PCB->ViaN)
				RedrawOnEnter = REDRAW_ALL;
			break;

		default:				/* layers, current one can't be switched off */
				/* check if current one is in that group */
			if ((group = GetGroupOfLayer(layer)) < MAX_LAYER)
				for (i = 0; i < PCB->LayerGroups.Number[group]; i++)
					if (PCB->LayerGroups.Entries[group][i] == LayerStack[0])
					{
						current = True;
						break;
					}
			if (current)
				ChangeGroupVisibility(layer, True);
			else
			{
				ChangeGroupVisibility(layer, state);
				RedrawOnEnter = REDRAW_ALL;
			}
			break;
	}
}

/* ---------------------------------------------------------------------------
 * initializes dialog box to get settings like
 *   zoom
 *   grid
 *   drawing layer
 *   ...
 * returns popup shell widget
 */
Widget InitControlPanel(Widget Parent, Widget Top, Widget Left)
{
	Widget	masterform,
			form[2];

	masterform = XtVaCreateManagedWidget("controlMasterForm", formWidgetClass,
		Parent,
		XtNfromHoriz, Left,
		XtNfromVert, Top,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);

		/* form[0]: used to switch layers on/off
		 * form[1]: used to select the drawing layer
		 */
	form[0] = XtVaCreateManagedWidget("onOffSelection", formWidgetClass,
		masterform,
		XtNfromVert, NULL,
		XtNfromHoriz, NULL,
		NULL);
	form[1] = XtVaCreateManagedWidget("currentSelection", formWidgetClass,
		masterform,
		XtNfromVert, form[0],
		XtNfromHoriz, NULL,
		NULL);

		/* we now add the other widgets */
	AddOnOffSelection(form[0]);
	AddDrawingLayerSelection(form[1]);
	return(masterform);
}

/* ---------------------------------------------------------------------------
 * updates label and sizes in control panel
 */
void UpdateControlPanel(void)
{
	UpdateOnOffSelection();
	UpdateDrawingLayerSelection();
}

