/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "Compat.h"
#include "Util.h"
#include "TextFieldP.h"

#define INVALIDATE_SELECTION(w) \
    (w)->textfield.sel_set = False; \
    if ((w)->textfield.curr_sel != None) { \
        XtDisownSelection((Widget)(w), \
			  (w)->textfield.curr_sel, \
			  (w)->textfield.sel_time); \
        (w)->textfield.curr_sel = None; \
    }
		

static XtResource resources[] = {
#define offset(field) XtOffsetOf(TextFieldRec, textfield.field)
    {XtNbuffer, XtCBuffer, XtRString, sizeof(String),
     offset(buffer), XtRImmediate, (XtPointer)NULL},
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
     offset(foreground_pixel), XtRString, XtDefaultForeground},
    {XtNhighlightForeground, XtCBackground, XtRPixel, sizeof(Pixel),
     offset(highlight_fg), XtRString, XtDefaultBackground},
    {XtNhighlightBackground, XtCForeground, XtRPixel, sizeof(Pixel),
     offset(highlight_bg), XtRString, XtDefaultForeground},
    {XtNfocusColor, XtCForeground, XtRPixel, sizeof(Pixel),
     offset(focus_pixel), XtRString, XtDefaultForeground},
    {XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     offset(font), XtRString, XtDefaultFont},
    {XtNinternalWidth, XtCInternalWidth, XtRDimension, sizeof(Dimension),
     offset(internal_width), XtRImmediate, (XtPointer)8},
    {XtNinternalHeight, XtCInternalHeight, XtRDimension, sizeof(Dimension),
     offset(internal_height), XtRImmediate, (XtPointer)3},
    {XtNpreferredChars, XtCPreferredChars, XtRCardinal, sizeof(Cardinal),
     offset(preferred_chars), XtRImmediate, (XtPointer)32},
    {XtNcallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
     offset(callback), XtRCallback, (XtPointer)NULL},
    {XtNtabCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
     offset(tab_callback), XtRCallback, (XtPointer)NULL},
    {XtNfocusCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
     offset(focus_callback), XtRCallback, (XtPointer)NULL},
    {XtNborderIn, XtCBorderIn, XtRBoolean, sizeof(Boolean),
     offset(border_in), XtRImmediate, (XtPointer)True},
    {XtNdisplayCaret, XtCDisplayCaret, XtRBoolean, sizeof(Boolean),
     offset(display_caret), XtRImmediate, (XtPointer)False},
    {XtNfocusRoot, XtCFocusRoot, XtRWidget, sizeof(Widget),
     offset(focus_root), XtRImmediate, (XtPointer)NULL},
    {XtNfocusHack, XtCHack, XtRBoolean, sizeof(Boolean),
     offset(focus_hack), XtRImmediate, (XtPointer)True},
    {XtNprintFocus, XtCDebug, XtRBoolean, sizeof(Boolean),
     offset(print_focus), XtRImmediate, (XtPointer)False},
#undef offset
};

static void Initialize(Widget, Widget, ArgList, Cardinal*);
static void Destroy(Widget);
static void Redisplay(Widget, XEvent*, Region);
static void Resize(Widget);
static void Realize(Widget, XtValueMask*, XSetWindowAttributes*);
static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal*);
static XtGeometryResult QueryGeometry(Widget, XtWidgetGeometry*,
				      XtWidgetGeometry*);

static void aquire_focus(Widget, XEvent*, String*, Cardinal*);
static void beginning_of_line(Widget, XEvent*, String*, Cardinal*);
static void backward_character(Widget, XEvent*, String*, Cardinal*);
static void delete_next_character(Widget, XEvent*, String*, Cardinal*);
static void end_of_line(Widget, XEvent*, String*, Cardinal*);
static void forward_character(Widget, XEvent*, String*, Cardinal*);
static void multiply(Widget, XEvent*, String*, Cardinal*);
static void kill_to_end_of_line(Widget, XEvent*, String*, Cardinal*);
static void redraw(Widget, XEvent*, String*, Cardinal*);
static void notify(Widget, XEvent*, String*, Cardinal*);
static void tab_notify(Widget, XEvent*, String*, Cardinal*);
static void transpose_characters(Widget, XEvent*, String*, Cardinal*);
static void kill_selection(Widget, XEvent*, String*, Cardinal*);
static void delete_previous_character(Widget, XEvent*, String*, Cardinal*);
static void set_border_color(Widget, XEvent*, String*, Cardinal*);
static void insert_character(Widget, XEvent*, String*, Cardinal*);
static void insert_string(Widget, XEvent*, String*, Cardinal*);
static void set_mark(Widget, XEvent*, String*, Cardinal*);
static void set_position(Widget, XEvent*, String*, Cardinal*);
static void extend_start(Widget, XEvent*, String*, Cardinal*);
static void extend_adjust(Widget, XEvent*, String*, Cardinal*);
static void extend_end(Widget, XEvent*, String*, Cardinal*);
static void insert_selection(Widget, XEvent*, String*, Cardinal*);
static void disown_selection(Widget, XEvent*, String*, Cardinal*);
static void display_caret(Widget, XEvent*, String*, Cardinal*);

static XtActionsRec actions[] = {
    {"aquire-focus",			aquire_focus},
    {"beginning-of-line",		beginning_of_line},
    {"backward-character",		backward_character},
    {"delete-next-character",		delete_next_character},
    {"end-of-line",			end_of_line},
    {"forward-character",		forward_character},
    {"multiply",			multiply},
    {"kill-to-end-of-line",		kill_to_end_of_line},
    {"redraw",				redraw},
    {"notify",				notify},
    {"tab-notify",			tab_notify},
    {"transpose-characters",		transpose_characters},
    {"kill-selection",			kill_selection},
    {"delete-previous-character",	delete_previous_character},
    {"set-border-color",		set_border_color},
    {"insert-character",		insert_character},
    {"insert-string",			insert_string},
    {"set-mark",			set_mark},
    {"set-position",			set_position},
    {"extend-start",			extend_start},
    {"extend-adjust",			extend_adjust},
    {"extend-end",			extend_end},
    {"disown-selection",		disown_selection},
    {"insert-selection",		insert_selection},
    {"display-caret",			display_caret},
};

static char translations[] =
"Ctrl<Key>A:		beginning-of-line() \n"
"Ctrl<Key>B:		backward-character() \n"
"Ctrl<Key>D:		delete-next-character() \n"
"Ctrl<Key>E:		end-of-line() \n"
"Ctrl<Key>F:		forward-character() \n"
"Ctrl<Key>G:		multiply(0) disown-selection() \n"
"Ctrl<Key>K:		kill-to-end-of-line() \n"
"Ctrl<Key>L:		redraw() \n"
"Ctrl<Key>M:		notify() \n"
"Ctrl<Key>T:		transpose-characters() \n"
"Ctrl<Key>U:		multiply(4) \n"
"Ctrl<Key>W:		kill-selection(PRIMARY) \n"
"Ctrl<Key>space:	set-mark() \n"
"<Key>Left:		backward-character() \n"
"<Key>Right:		forward-character() \n"
"<Key>Return:		notify() \n"
"<Key>BackSpace:	delete-previous-character() \n"
"<Key>Delete:		delete-previous-character() \n"
"<Key>Tab:		tab-notify() \n"
"<FocusIn>:		set-border-color(focusColor) display-caret(on) \n"
"<FocusOut>:		set-border-color(background) display-caret(off) \n"
"<Btn1Down>:		aquire-focus() set-position() \n"
"<Btn1Motion>:		extend-adjust() \n"
"<Btn1Up>:		extend-end(PRIMARY) \n"
"<Btn3Down>:		extend-start() \n"
"<Btn3Motion>:		extend-adjust() \n"
"<Btn3Up>:		extend-end(PRIMARY) \n"
"<Btn2Up>:		insert-selection(PRIMARY) \n"
"Ctrl<Key>0:		multiply(0) \n"
"Ctrl<Key>1:		multiply(1) \n"
"Ctrl<Key>2:		multiply(2) \n"
"Ctrl<Key>3:		multiply(3) \n"
"Ctrl<Key>4:		multiply(4) \n"
"Ctrl<Key>5:		multiply(5) \n"
"Ctrl<Key>6:		multiply(6) \n"
"Ctrl<Key>7:		multiply(7) \n"
"Ctrl<Key>8:		multiply(8) \n"
"Ctrl<Key>9:		multiply(9) \n"
"<Key>:			insert-character() \n";

TextFieldClassRec textFieldClassRec = {
    {                                   /* core fields                  */
        (WidgetClass) &shadowClassRec,  /* superclass                   */
        "TextField",                    /* class_name                   */
        sizeof(TextFieldRec),	        /* widget_size                  */
        NULL,                           /* class_initialize             */
        NULL,                           /* class_part_initialize        */
        FALSE,                          /* class_inited                 */
        Initialize,                     /* initialize                   */
        NULL,                           /* initialize_hook              */
        Realize,			/* realize                      */
        actions,                        /* actions                      */
        XtNumber(actions),              /* num_actions                  */
        resources,                      /* resources                    */
        XtNumber(resources),            /* num_resources                */
        NULLQUARK,                      /* xrm_class                    */
        TRUE,                           /* compress_motion              */
#if (XtSpecificationRelease < 4)
	True,				/* compress_exposure		*/
#elif (XtSpecificationRelease < 6)
        XtExposeCompressMaximal,	/* compress_exposure		*/
#else
        XtExposeCompressMaximal | XtExposeNoRegion, /* compress_exposure*/
#endif
        FALSE,                          /* compress_enterleave          */
        FALSE,                          /* visible_interest             */
        Destroy,                        /* destroy                      */
        Resize,                         /* resize                       */
        Redisplay,                      /* expose                       */
        SetValues,                      /* set_values                   */
        NULL,                           /* set_values_hook              */
        XtInheritSetValuesAlmost,       /* set_values_almost            */
        NULL,                           /* get_values_hook              */
        NULL,				/* accept_focus                 */
        XtVersion,                      /* version                      */
        NULL,                           /* callback_private             */
        translations,                   /* tm_table                     */
        QueryGeometry,                  /* query_geometry               */
        XtInheritDisplayAccelerator,    /* display_accelerator          */
        NULL                            /* extension                    */
    },
    {					/* shadow fields		*/
	XtInheritPixelOffset,		/* pixel_offset			*/
	True,				/* use_arm_for_background	*/
	XtInheritAllocShadowColors,	/* alloc_shadow_colors		*/
	XtInheritAllocShadowPixmaps,	/* alloc_shadow_pixmaps		*/
	XtInheritAllocArmColor,		/* alloc_arm_color		*/
	NULL,				/* alloc_arm_pixmap		*/
	XtInheritAllocGCs,		/* alloc_gcs			*/
	NULL,				/* extension			*/
    },
    {                                   /* textfield fields             */
        0                               /* extension                    */
    }
};

WidgetClass textFieldWidgetClass = (WidgetClass)&textFieldClassRec;

/*************************************************************************/

static void init_gcs(TextFieldWidget w)
{
    XGCValues	values;

    values.font = w->textfield.font->fid;
    values.foreground = w->textfield.foreground_pixel;
    if (w->shadow.alloced_arm_pixel)
	values.background = w->shadow.arm_pixel;
    else
	values.background = w->core.background_pixel;
    w->textfield.default_gc =
	XtGetGC((Widget)w, GCForeground | GCBackground | GCFont, &values);
    values.foreground = w->textfield.highlight_fg;
    values.background = w->textfield.highlight_bg;
    w->textfield.highlight_gc =
	XtGetGC((Widget)w, GCForeground | GCBackground | GCFont, &values);
}

static void free_gcs(TextFieldWidget w)
{
    XtReleaseGC((Widget)w, w->textfield.default_gc);
    XtReleaseGC((Widget)w, w->textfield.highlight_gc);
}

static void undraw_caret(TextFieldWidget w)
{
    int y = w->shadow.shadow_width;
    int height = w->core.height - 2 * y;

    if (height > 0)
	XClearArea(XtDisplay(w), XtWindow(w),
		   w->textfield.caret_pos, y, 1, height, False);
}

static void draw_caret(TextFieldWidget w)
{
    int h1 = w->textfield.font->ascent + w->textfield.font->descent + 4;
    int h2 = w->core.height - 2 * w->shadow.shadow_width;

    if (h1 > h2) {
	if (h2 < 0)
	    return;
	h1 = h2;
    }

    XDrawLine(XtDisplay(w), XtWindow(w), w->textfield.default_gc,
	      w->textfield.caret_pos, ((int)w->core.height - h1) / 2 + 1,
	      w->textfield.caret_pos, h1);
}

static void get_preferred_sizes(TextFieldWidget w,
				Dimension *width, Dimension *height)
{
    *width = 2 * (w->textfield.internal_width +
		  w->shadow.shadow_width) +
	w->textfield.preferred_chars *
	(w->textfield.font->max_bounds.lbearing +
	 w->textfield.font->max_bounds.rbearing);

    *height = 2 * (w->textfield.internal_height +
		   w->shadow.shadow_width) +
	w->textfield.font->ascent + w->textfield.font->descent;
}

static int event_to_position(TextFieldWidget w, XEvent *event)
{
    int		ex, ey;
    int		x, pos;

    if (!get_event_xy(event, &ex, &ey))
	return -1;

    x = w->textfield.internal_width + w->shadow.shadow_width;
    if (ex < x) {
	if (w->textfield.first > 0)
	    return w->textfield.first - 1;
	else
	    return 0;
    }

    pos =
	MyXWidthToChars(w->textfield.font,
			&w->textfield.buffer[w->textfield.first],
			w->textfield.len - w->textfield.first, ex - x);

    return w->textfield.first + pos;
}

static void calc_caret_pos(TextFieldWidget w)
{
    w->textfield.caret_pos = w->shadow.shadow_width +
	w->textfield.internal_width;
    if (w->textfield.first < w->textfield.pos)
	w->textfield.caret_pos +=
	    XTextWidth(w->textfield.font,
		       &w->textfield.buffer[w->textfield.first],
		       w->textfield.pos - w->textfield.first);
}

static void calc_shown(TextFieldWidget w)
{
    int	width = w->core.width - 2 * (w->shadow.shadow_width +
				     w->textfield.internal_width);

    if (w->textfield.buffer && w->textfield.first < w->textfield.len)
	w->textfield.shown =
	    MyXWidthToChars(w->textfield.font,
			    &w->textfield.buffer[w->textfield.first],
			    w->textfield.len - w->textfield.first, width);
    else
	w->textfield.shown = 0;
}

static void fit_first(TextFieldWidget w)
{
    if (w->textfield.pos < w->textfield.first) {
	w->textfield.first = w->textfield.pos;
	calc_shown(w);
    } else {
	while (w->textfield.pos > w->textfield.first + w->textfield.shown &&
	       w->textfield.first < w->textfield.len) {
	    w->textfield.first++;
	    calc_shown(w);
	}
    }
}

static void change_pos(TextFieldWidget w, int pos, Boolean len_changed)
{
    if (XtIsRealized((Widget)w))
	undraw_caret(w);

    if (len_changed)
	calc_shown(w);

    if (pos < 0)
	pos = 0;
    if (pos > w->textfield.len)
	pos = w->textfield.len;

    if (w->textfield.sel_set) {
	int	old_pos = w->textfield.pos;

	if (old_pos == w->textfield.sel_start)
	    w->textfield.sel_start = pos;
	else if (old_pos == w->textfield.sel_stop)
	    w->textfield.sel_stop = pos;

	if (w->textfield.sel_start > w->textfield.sel_stop) {
	    Cardinal	temp = w->textfield.sel_start;

	    w->textfield.sel_start = w->textfield.sel_stop;
	    w->textfield.sel_stop = temp;
	}
    }

    w->textfield.pos = pos;
    fit_first(w);
    calc_caret_pos(w);

    Redisplay((Widget)w, NULL, NULL);
}

/*************************************************************************/



static void delete_sel_callback(Widget gw, XtPointer client_data,
				Atom *selection, Atom *type,
				XtPointer value, unsigned long *len,
				int *format)
{
    /* nothing to do here */
}

static void get_sel_callback(Widget gw, XtPointer client_data, Atom *selection,
			     Atom *type, XtPointer value, unsigned long *len,
			     int *format)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    if (!w->textfield.waiting_for_sel || *type == XT_CONVERT_FAIL || !value) {
	XtFree((char *)value);
	w->textfield.waiting_for_sel = False;
	return;
    }
    w->textfield.waiting_for_sel = False;

    if (w->textfield.alloc < w->textfield.len + *len + 8) {
	w->textfield.alloc = w->textfield.len + *len + 8;
	w->textfield.buffer =
	    XtRealloc(w->textfield.buffer, w->textfield.alloc);
    }

    memmove(&w->textfield.buffer[w->textfield.pos + *len],
	    &w->textfield.buffer[w->textfield.pos],
	    w->textfield.len - w->textfield.pos);
    w->textfield.len += *len;
    w->textfield.buffer[w->textfield.len] = '\0';
    memmove(&w->textfield.buffer[w->textfield.pos], (char *)value, *len);
    XtFree((char *)value);
    calc_shown(w);
    Redisplay((Widget)w, NULL, NULL);
}

static Boolean convert_sel_proc(Widget gw, Atom *sel, Atom *target,
				Atom *type, XtPointer *value,
				unsigned long *len, int *format)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    Display		*disp = XtDisplay(w);

    if (w->textfield.curr_sel == None ||
	*sel != w->textfield.curr_sel)
	return False;

    if (*target == XInternAtom(disp, "TARGETS", False)) {
	Atom		*std_targets, *atom;
	unsigned long	std_length, n;

	if (!cvt_std_sel((Widget)w, w->textfield.sel_time,
			 sel, target, type,
			 (XPointer *)&std_targets, &std_length,
			 format))
	    return False;

	*value = (XtPointer)XtMalloc((std_length + 8) * sizeof(Atom));
	atom = *(Atom **)value;
	n = std_length;

	n++; *atom++ = XA_STRING;
	n++; *atom++ = XInternAtom(disp, "TEXT", False);
	n++; *atom++ = XInternAtom(disp, "LENGTH", False);
	n++; *atom++ = XInternAtom(disp, "LIST_LENGTH", False);
	n++; *atom++ = XInternAtom(disp, "DELETE", False);

	memcpy(atom, std_targets, std_length * sizeof(Atom));
	XtFree((char *)std_targets);

	*len = n;
	*type = XA_ATOM;
	*format = 32;

	return True;
    }

    if (*target == XA_STRING ||
	*target == XInternAtom(disp, "TEXT", False)) {
	char	*buffer;

	*len = w->textfield.sel_stop - w->textfield.sel_start;
	buffer = XtNewString(&w->textfield.buffer[w->textfield.sel_start]);
	buffer[*len] = '\0';

	*value = (XtPointer)buffer;
	*type = XA_STRING;
	*format = 8;

	return True;
    }

    if (*target == XInternAtom(disp, "LENGTH", False)) {
	long	*length = (long *)XtMalloc(sizeof(long));

	*length = w->textfield.sel_stop - w->textfield.sel_start;
	*value = (XPointer)length;
	*type = XA_INTEGER;
	*len = 1;
	*format = 32;

	return True;
    }

    if (*target == XInternAtom(disp, "LIST_LENGTH", False)) {
	long	*length = (long *)XtMalloc(sizeof(long));

	*length = 1;
	*value = (XPointer)length;
	*type = XA_INTEGER;
	*format = 32;

	return True;
    }

    if (*target == XInternAtom(disp, "DELETE", False)) {
	char		*buffer = w->textfield.buffer;
	Cardinal	pos = w->textfield.sel_start;		
	int		n = w->textfield.len - w->textfield.sel_stop;

	if (n > 0)
	    memmove(&buffer[pos], &buffer[w->textfield.sel_stop], n);
	buffer[pos + n] = '\0';
	w->textfield.len = pos + n;
	w->textfield.sel_stop = w->textfield.pos = pos;
	w->textfield.sel_set = False;
	change_pos(w, pos, True);

	*value = NULL;
	*len = 0;
	*format = 32;
	*type = None;

	return True;
    }

    return cvt_std_sel((Widget)w, w->textfield.sel_time,
		       sel, target, type, (XPointer *)value, len, format);
}

static void lose_sel_proc(Widget gw, Atom *sel)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.sel_start = w->textfield.sel_stop;
    w->textfield.curr_sel = None;
    Redisplay((Widget)w, NULL, NULL);
}

/*************************************************************************/

static void aquire_focus(Widget gw, XEvent *event,
			 String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    if (w->textfield.focus_root && w->textfield.active) {
	XtCallbackList	c_list = w->textfield.focus_callback;

	XtSetKeyboardFocus(w->textfield.focus_root, (Widget)w);
	if (c_list)
	    XtCallCallbackList((Widget)w, c_list, NULL);
    }
}

static void beginning_of_line(Widget gw, XEvent *event,
			      String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    change_pos(w, 0, False);
}

static void backward_character(Widget gw, XEvent *event,
			       String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			new_pos = w->textfield.pos - w->textfield.multiply;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    change_pos(w, new_pos, False);
}

static void delete_next_character(Widget gw, XEvent *event,
				  String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			mult = w->textfield.multiply;
    int			pos = w->textfield.pos;

    if (pos + mult >= w->textfield.len) {
	w->textfield.buffer[pos] = '\0';
	w->textfield.len = pos;
    } else {
	strcpy(&w->textfield.buffer[pos], &w->textfield.buffer[pos + mult]);
	w->textfield.len -= mult;
    }
    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);
    calc_shown(w);
    Redisplay((Widget)w, NULL, NULL);
}

static void end_of_line(Widget gw, XEvent *event,
			String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    change_pos(w, w->textfield.len, False);
}

static void forward_character(Widget gw, XEvent *event,
			      String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			new_pos = w->textfield.pos + w->textfield.multiply;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    change_pos(w, new_pos, False);
}

static void multiply(Widget gw, XEvent *event,
		     String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    Boolean		ok = False;

    if (*no_params == 1) {
	long	factor = atol(params[0]);
	long	new_mult = w->textfield.multiply * factor;

	if (0 < new_mult && new_mult <= 16384) {
	    ok = True;
	    w->textfield.multiply = (Cardinal)new_mult;
	}
    }

    w->textfield.waiting_for_sel = False;
    if (!ok) {
	XBell(XtDisplay(w), 0);
	w->textfield.multiply = 1;
    }
}

static void kill_to_end_of_line(Widget gw, XEvent *event,
				String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.buffer[w->textfield.pos] = '\0';
    w->textfield.len = w->textfield.pos;
    INVALIDATE_SELECTION(w);
    w->textfield.waiting_for_sel = False;
    w->textfield.multiply = 1;
    calc_shown(w);
    Redisplay((Widget)w, NULL, NULL);
}

static void redraw(Widget gw, XEvent *event,
		   String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    Redisplay((Widget)w, NULL, NULL);
}

static void notify(Widget gw, XEvent *event,
		   String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    XtCallbackList	c_list = w->textfield.callback;

    if (!w->textfield.active)
	return;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);
    if (c_list)
	XtCallCallbackList((Widget)w, c_list, (XtPointer)w->textfield.buffer);
}

static void tab_notify(Widget gw, XEvent *event,
		       String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    XtCallbackList	c_list = w->textfield.tab_callback;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);
    if (c_list)
	XtCallCallbackList((Widget)w, c_list, (XtPointer)w->textfield.buffer);
}

static void transpose_characters(Widget gw, XEvent *event,
				String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			pos = w->textfield.pos;
    char		c;

    if (w->textfield.pos == 0) {
	XBell(XtDisplay(w), 0);
	return;
    } else if (w->textfield.pos == w->textfield.len)
	return;

    c = w->textfield.buffer[w->textfield.pos];
    w->textfield.buffer[w->textfield.pos] =
	w->textfield.buffer[w->textfield.pos-1];
    w->textfield.buffer[pos-1] = c;
    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);
    change_pos(w, (pos < w->textfield.len) ? pos + 1 : pos, False);
}

static void kill_selection(Widget gw, XEvent *event,
			   String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    Display		*disp = XtDisplay(w);
    Atom		atom = XA_PRIMARY;
    Time		time = get_event_time(event);

    w->textfield.multiply = 1;

    if (*no_params > 0)
	atom = XInternAtom(disp, params[0], False);

    w->textfield.waiting_for_sel = True;
    XtGetSelectionValue((Widget)w, atom, XInternAtom(disp, "DELETE", False),
			delete_sel_callback, NULL, time);
}

static void delete_previous_character(Widget gw, XEvent *event,
				      String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			new_pos, mult = w->textfield.multiply;

    if (w->textfield.pos == 0)
	return;
    if (mult > w->textfield.pos)
	mult = w->textfield.pos;
    new_pos = w->textfield.pos - mult;
    w->textfield.len -= mult;
    memmove(&w->textfield.buffer[new_pos],
	    &w->textfield.buffer[w->textfield.pos],
	    w->textfield.len - new_pos);
    w->textfield.buffer[w->textfield.len] = '\0';
    
    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);
    change_pos(w, new_pos, True);
}

static void set_border_color(Widget gw, XEvent *event,
			     String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    Arg			arg;

    if (*no_params != 1) {
	XBell(XtDisplay(w), 0);
	return;
    }

    if (w->textfield.print_focus &&
	(event->type == FocusIn ||
	 event->type == FocusOut)) {
	if (event->type == FocusIn)
	    fprintf(stderr, "FocusIn:  ");
	else
	    fprintf (stderr, "FocusOut: ");
	fprintf(stderr, "%s\n", XtName((Widget)w));

	fprintf(stderr, "          serial:     %ld\n", event->xfocus.serial);
	fprintf(stderr, "          send_event: %s\n",
		event->xfocus.send_event ? "True" : "False");
	fprintf(stderr, "          window:     0x%lx\n", event->xfocus.window);
	fprintf(stderr, "          mode:       ");
	switch (event->xfocus.mode) {
	case NotifyNormal:
	    fprintf(stderr, "NotifyNormal\n");
	    break;
	case NotifyGrab:
	    fprintf(stderr, "NotifyGrab\n");
	    break;
	case NotifyUngrab:
	    fprintf(stderr, "NotifyUngrab\n");
	    break;
	default:
	    fprintf(stderr, "\n");
	    break;
	}
	fprintf(stderr, "          detail:     ");
	switch (event->xfocus.detail) {
	case NotifyAncestor:
	    fprintf(stderr, "NotifyAncestor\n");
	    break;
	case NotifyVirtual:
	    fprintf(stderr, "NotifyVirtual\n");
	    break;
	case NotifyInferior:
	    fprintf(stderr, "NotifyInferior\n");
	    break;
	case NotifyNonlinear:
	    fprintf(stderr, "NotifyNonlinear\n");
	    break;
	case NotifyNonlinearVirtual:
	    fprintf(stderr, "NotifyNonlinearVirtual\n");
	    break;
	case NotifyPointer:
	    fprintf(stderr, "NotifyPointer\n");
	    break;
	case NotifyPointerRoot:
	    fprintf(stderr, "NotifyPointerRoot\n");
	    break;
	case NotifyDetailNone:
	    fprintf(stderr, "NotifyDetailNone\n");
	    break;
	default:
	    fprintf(stderr, "\n");
	    break;
	}
    }

    if (w->textfield.focus_hack &&
	event->type == FocusIn &&
	!event->xfocus.send_event)
	return;

    if (strlen(params[0]) < 15) {
	char	buffer[16];
	char	*c;
	int	i;

	c = params[0];
	i = 0;
	do {
	    buffer[i++] =
		isupper((unsigned char)*c) ?
		tolower((unsigned char)*c) : *c;
	} while (*c++ != '\0');

	if (strcmp(buffer, XtNbackground) == 0) {
	    XtSetArg(arg, XtNborderColor, w->core.background_pixel);
	    XtSetValues((Widget)w, &arg, 1);
	    return;
	} else if (strcmp(buffer, XtNforeground) == 0) {
	    XtSetArg(arg, XtNborderColor, w->textfield.foreground_pixel);
	    XtSetValues((Widget)w, &arg, 1);
	    return;
	} else if (strcmp(buffer, "focuscolor") == 0) {
	    XtSetArg(arg, XtNborderColor, w->textfield.focus_pixel);
	    XtSetValues((Widget)w, &arg, 1);
	    return;
	}
    }

    XBell(XtDisplay(w), 0);
}

static void insert_character(Widget gw, XEvent *event,
			     String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    char		buffer[16];
    KeySym		keysym;
    int			pos = w->textfield.pos;
    int			mult = w->textfield.multiply;
    int			len, new_pos;
    int			insert_len;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);

    len = XLookupString((XKeyEvent *)event, buffer,
			sizeof(buffer), &keysym, NULL);
    if (len <= 0)
	return;
    if (XTextWidth(w->textfield.font, buffer, len) <= 0) {
	XBell(XtDisplay(w), 0);
	return;
    }

    insert_len = len * mult;
    if (insert_len + w->textfield.len >= w->textfield.alloc) {
	int n = insert_len + 2 * w->textfield.alloc;

	w->textfield.buffer = XtRealloc(w->textfield.buffer, n);
	w->textfield.alloc = n;
    }

    new_pos = pos + insert_len;
    memmove(&w->textfield.buffer[new_pos], &w->textfield.buffer[pos],
	    w->textfield.len - pos);
    w->textfield.len += insert_len;
    while (mult-- > 0) {
	memmove(&w->textfield.buffer[pos], buffer, len);
	pos += len;
    }
    w->textfield.buffer[w->textfield.len] = '\0';
    undraw_caret(w);
    change_pos(w, new_pos, True);
}

static void insert_string(Widget gw, XEvent *event,
			  String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			pos = w->textfield.pos;
    int			mult = w->textfield.multiply;
    int			len, new_pos;
    int			insert_len;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);

    if (*no_params != 1) {
	XBell(XtDisplay(w), 0);
	return;
    }

    len = strlen(params[0]);
    if (len <= 0)
	return;

    insert_len = mult * len;
    if (insert_len + w->textfield.len >= w->textfield.alloc) {
	int	n = insert_len + 2 * w->textfield.alloc;

	w->textfield.buffer = XtRealloc(w->textfield.buffer, n);
	w->textfield.alloc = n;
    }

    new_pos = pos + insert_len;
    memmove(&w->textfield.buffer[new_pos], &w->textfield.buffer[pos],
	    w->textfield.len - pos);
    w->textfield.len += insert_len;
    while (mult-- > 0) {
	memmove(&w->textfield.buffer[pos], params[0], len);
	pos += len;
    }
    w->textfield.buffer[w->textfield.len] = '\0';
    undraw_caret(w);
    change_pos(w, new_pos, True);
}

static void set_mark(Widget gw, XEvent *event,
		     String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    INVALIDATE_SELECTION(w);

    if (w->textfield.len > 0) {
	int	pos = w->textfield.pos;

	w->textfield.sel_start = pos;
	w->textfield.sel_stop = pos + 1;
	Redisplay((Widget)w, NULL, NULL);
    }
}

static void set_position(Widget gw, XEvent *event,
			 String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			pos;

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    if (w->textfield.len == 0)
	return;

    INVALIDATE_SELECTION(w);

    pos = event_to_position(w, event);
    if (pos < 0) {
	XBell(XtDisplay(w), 0);
	return;
    }

    w->textfield.sel_start = pos;
    w->textfield.sel_stop = pos;
    change_pos(w, pos, False);
    w->textfield.sel_set = True;
}

static void extend_start(Widget gw, XEvent *event,
			 String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			pos;

    if (!w->textfield.sel_set) {
	set_position((Widget)w, event, params, no_params);
	return;
    }

    w->textfield.multiply = 1;
    w->textfield.waiting_for_sel = False;
    if (w->textfield.len == 0)
	return;

    pos = event_to_position(w, event);
    if (pos < 0) {
	XBell(XtDisplay(w), 0);
	return;
    }

    if (pos <= w->textfield.sel_start)
	w->textfield.sel_start = pos;
    else if (pos >= w->textfield.sel_stop)
	w->textfield.sel_stop = pos;
    else {
	if (pos - w->textfield.sel_start < w->textfield.sel_stop - pos)
	    w->textfield.sel_start = pos;
	else
	    w->textfield.sel_stop = pos;
    }
    w->textfield.pos = pos; /* must do this first... */
    change_pos(w, pos, False);
}

static void extend_adjust(Widget gw, XEvent *event,
			  String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			pos;

    w->textfield.waiting_for_sel = False;
    w->textfield.multiply = 1;

    pos = event_to_position(w, event);
    if (pos < 0) {
	XBell(XtDisplay(w), 0);
	return;
    }

    change_pos(w, pos, False);
}

static void extend_end(Widget gw, XEvent *event,
		       String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.waiting_for_sel = False;
    w->textfield.multiply = 1;

    if (w->textfield.sel_set &&
	(w->textfield.sel_start < w->textfield.sel_stop)) {
	Atom	atom = XA_PRIMARY;
	Time	time = get_event_time(event);

	if (*no_params > 0)
	    atom = XInternAtom(XtDisplay(w), params[0], False);

	if (XtOwnSelection((Widget)w, atom, time, convert_sel_proc,
			   lose_sel_proc, NULL)) {
	    w->textfield.curr_sel = atom;
	    w->textfield.sel_time = time;
	}
    }
}

static void insert_selection(Widget gw, XEvent *event,
			     String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			pos;
    Atom		atom = XA_PRIMARY;
    Time		time = get_event_time(event);

    w->textfield.multiply = 1;

    if (*no_params > 0)
	atom = XInternAtom(XtDisplay(w), params[0], False);

    pos = event_to_position(w, event);
    if (pos < 0)
	return;
    change_pos(w, pos, False);

    w->textfield.waiting_for_sel = True;
    XtGetSelectionValue((Widget)w, atom, XA_STRING,
			get_sel_callback, NULL, time);
}

static void disown_selection(Widget gw, XEvent *event,
			     String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.multiply = 1;
    INVALIDATE_SELECTION(w);
    Redisplay((Widget)w, NULL, NULL);
}

static void display_caret(Widget gw, XEvent *event,
			  String *params, Cardinal *no_params)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    if (w->textfield.focus_hack &&
	event->type == FocusIn &&
	!event->xfocus.send_event)
	return;

    if (*no_params == 1 && strlen(params[0]) < 15) {
	char	buffer[16];
	char	*c;
	int	i;

	c = params[0];
	i = 0;
	do {
	    buffer[i++] =
		isupper((unsigned char)*c) ?
		tolower((unsigned char)*c) : *c;
	} while (*c++ != '\0');

	if (strcmp(buffer, "on") == 0 || strcmp(buffer, "true") == 0) {
	    w->textfield.display_caret = True;
	    draw_caret(w);
	    return;
	} else if (strcmp(buffer, "off") == 0 ||
		   strcmp(buffer, "false") == 0) {
	    w->textfield.display_caret = False;
	    undraw_caret(w);
	    Redisplay((Widget)w, NULL, NULL);
	    return;
	}
    }

    XBell(XtDisplay(w), 0);
}

/*************************************************************************/

static void Initialize(Widget grequest, Widget gnew,
		       ArgList args, Cardinal *no_args)
{
    TextFieldWidget	new = (TextFieldWidget)gnew;
    Dimension		width, height;

    init_gcs(new);
    get_preferred_sizes(new, &width, &height);

    if (new->textfield.buffer) {
	new->textfield.buffer = XtNewString(new->textfield.buffer);
	new->textfield.len = strlen(new->textfield.buffer);
	new->textfield.alloc = new->textfield.len + 1;
    } else {
	new->textfield.buffer = XtMalloc(256);
	new->textfield.alloc = 256;
	new->textfield.len = 0;
	new->textfield.buffer[0] = '\0';
    }

    new->textfield.first = 0;
    new->textfield.pos = 0;
    new->textfield.multiply = 1;
    new->textfield.sel_start = 0;
    new->textfield.sel_stop = 0;
    new->textfield.curr_sel = None;
    new->textfield.waiting_for_sel = False;
    new->textfield.active = True;

    if (new->core.width == 0)
	new->core.width  = width;
    if (new->core.height == 0)
	new->core.height = height;
    new->core.border_pixel = new->core.background_pixel;

    calc_shown(new);
    calc_caret_pos(new);
}

static void Destroy(Widget gw)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    free_gcs(w);
}

static void Resize(Widget gw)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    change_pos(w, w->textfield.pos, True);
}

static void Realize(Widget gw, XtValueMask *mask,
		    XSetWindowAttributes *attributes)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    change_pos(w, w->textfield.pos, True);
    shadowWidgetClass->core_class.realize((Widget)w, mask, attributes);
}

static void Redisplay(Widget gw, XEvent *event, Region region)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    Display		*disp = XtDisplay(w);
    Window		win = XtWindow(w);
    int			x = w->shadow.shadow_width;
    int			height = w->core.height - 2 * x;
    int			width;

    if (!XtIsRealized((Widget)w) || height <= 0)
	return;

    if (w->textfield.shown > 0) {
	int		y = (int)(w->core.height + w->textfield.font->ascent -
				  w->textfield.font->descent) / 2;
	int		pos = w->textfield.first;
	char		*buffer = &w->textfield.buffer[pos];
	int		len = w->textfield.shown;

	if (w->textfield.internal_width > 0) {
	    XClearArea(disp, win, x, x,
		       w->textfield.internal_width, height, False);
	    x += w->textfield.internal_width;
	}

	if (!w->textfield.sel_set ||
	    w->textfield.sel_start >= w->textfield.sel_stop ||
	    w->textfield.sel_stop <= pos ||
	    w->textfield.sel_start >= pos + len) {
	    XDrawImageString(disp, win, w->textfield.default_gc,
			     x, y, buffer, len);
	    x += XTextWidth(w->textfield.font, buffer, len);
	} else {
	    int	n;

	    if (w->textfield.sel_start > pos) {
		n = w->textfield.sel_start - pos;

		if (n > len)
		    n = len;
		XDrawImageString(disp, win, w->textfield.default_gc,
				 x, y, buffer, n);
		x += XTextWidth(w->textfield.font, buffer, n);
		buffer += n;
		pos += n;
		len -= n;
	    }

	    n = w->textfield.sel_stop - pos;
	    if (n > len) {
		if (len > 0) {
		    XDrawImageString(disp, win, w->textfield.highlight_gc,
				     x, y, buffer, len);
		    x += XTextWidth(w->textfield.font, buffer, len);
		}
	    } else {
		XDrawImageString(disp, win, w->textfield.highlight_gc,
				 x, y, buffer, n);
		x += XTextWidth(w->textfield.font, buffer, n);
		buffer += n;
		len -= n;

		XDrawImageString(disp, win, w->textfield.default_gc,
				 x, y, buffer, len);
		x += XTextWidth(w->textfield.font, buffer, len);
	    }
	}

	width = w->core.width - w->shadow.shadow_width - x;
	if (width > 0)
	    XClearArea(disp, win, x, w->shadow.shadow_width,
		       width, height, False);
    } else {
	int	width = w->core.width - 2 * x;

	XClearArea(disp, win, x, x, width, height, False);
    }
	
    if (w->textfield.display_caret)
	draw_caret(w);

    ShadowDrawShadows((ShadowWidget)w, 0, 0, w->core.width,
		      w->core.height, w->textfield.border_in);
}

static Boolean SetValues(Widget gcurrent,
			 Widget grequest,
			 Widget gnew,
			 ArgList args,
			 Cardinal *num_args)
{
    TextFieldWidget	new = (TextFieldWidget)gnew;
    TextFieldWidget	current = (TextFieldWidget)gcurrent;
    Boolean		redisplay = False;

    if (new->core.background_pixel != current->core.background_pixel ||
	new->shadow.arm_pixel != current->shadow.arm_pixel ||
	new->shadow.alloced_arm_pixel != current->shadow.alloced_arm_pixel ||
	new->textfield.foreground_pixel !=
	current->textfield.foreground_pixel ||
	new->textfield.highlight_bg != current->textfield.highlight_bg ||
	new->textfield.highlight_fg != current->textfield.highlight_fg) {
	free_gcs(current);
	init_gcs(new);
	redisplay = True;
    }

    if (new->textfield.buffer != current->textfield.buffer) {
	XtFree(current->textfield.buffer);
	if (new->textfield.buffer) {
	    int	n = strlen(new->textfield.buffer);
	    
	    new->textfield.buffer = XtNewString(new->textfield.buffer);
	    new->textfield.first = 0;
	    new->textfield.pos = 0;
	    new->textfield.len = n;
	    new->textfield.alloc = n + 1;
	} else {
	    new->textfield.buffer = XtNewString("");
	    new->textfield.pos = 0;
	    new->textfield.len = 0;
	    new->textfield.alloc = 1;
	}

	INVALIDATE_SELECTION(new);
	new->textfield.curr_sel = None;
	new->textfield.waiting_for_sel = False;
	
	calc_shown(new);
	calc_caret_pos(new);
	change_pos(new, new->textfield.len, False);
	redisplay = True;
    }

    if (new->textfield.font != current->textfield.font ||
	new->textfield.internal_width !=
	current->textfield.internal_width ||
	new->textfield.internal_height !=
	current->textfield.internal_height) {
	Dimension	width, height;

	get_preferred_sizes(new, &width, &height);
	(void)XtMakeResizeRequest((Widget)new, width, height, NULL, NULL);
	redisplay = True;
    }

    return redisplay;
}

static XtGeometryResult QueryGeometry(Widget gw,
				      XtWidgetGeometry *intended,
				      XtWidgetGeometry *preferred)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    Dimension		width, height;
    Dimension		intended_width, intended_height;

    get_preferred_sizes(w, &width, &height);
    preferred->request_mode = CWWidth | CWHeight;
    preferred->width = width;
    preferred->height = height;

    if (intended->request_mode & CWWidth)
	intended_width = intended->width;
    else
	intended_width = w->core.width;
    if (intended->request_mode & CWHeight)
	intended_height = intended->height;
    else
	intended_height = w->core.height;

    if (width == w->core.width && height == w->core.height)
	return XtGeometryNo;
    else if (width == intended_width && height == intended_height)
	return XtGeometryYes;
    else
	return XtGeometryAlmost;
}

/*************************************************************************/

void TextFieldSetActive(Widget gw, int active)
{
    TextFieldWidget	w = (TextFieldWidget)gw;

    w->textfield.active = active;
}

void TextFieldSetBuffer(Widget gw, char *buffer)
{
    TextFieldWidget	w = (TextFieldWidget)gw;
    int			n;

    if (!buffer)
	buffer = "";

    n = strlen(buffer);

    w->textfield.first = 0;
    w->textfield.pos = 0;
    w->textfield.len = n;

    if (n + 8 > w->textfield.alloc) {
	w->textfield.alloc = 2 * n + 1;
	w->textfield.buffer = XtRealloc(w->textfield.buffer, 2 * n + 1);
    }

    strcpy(w->textfield.buffer, buffer);

    INVALIDATE_SELECTION(w);
    w->textfield.curr_sel = None;
    w->textfield.waiting_for_sel = False;

    calc_shown(w);
    calc_caret_pos(w);
    change_pos(w, w->textfield.len, False);

    if (XtIsRealized((Widget)w)) {
	XClearWindow(XtDisplay(w), XtWindow(w));
	Redisplay((Widget)w, NULL, NULL);
    }
}
