/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <X11/Shell.h>
#include "kedit.h"
#include "kill.h"
#include "pixmaps.h"
#include "widgets.h"
#include "xutil.h"
#include "../Widgets/Knapp.h"
#include "../Widgets/Layout.h"
#include "../Widgets/Menu.h"
#include "../Widgets/Message.h"
#include "../Widgets/MenuKnapp.h"
#include "../Widgets/PopdownSh.h"
#include "../Widgets/Sash.h"
#include "../Widgets/ScrBar.h"
#include "../Widgets/ScrList.h"
#include "../Widgets/SeparatorG.h"
#include "../Widgets/StringG.h"
#include "../Widgets/TextField.h"
#include "../Widgets/Util.h"

#define LINE_LEN 128

#define SAME_FIELD_TYPE(f1, f2) \
( (((f1) == KillFieldMsgid) && ((f2 == KillFieldMsgid)) ) || \
  (!((f1 == KillFieldMsgid)) && !((f2) == KillFieldMsgid)) )

#define SAME_ACTION_TYPE(a1, a2) (!(a1) == !(a2))

static struct {
    Widget	shell;
    Widget	list;
    Widget	scrbar;
    Widget	field_knapp;
    Widget	field_shell;
    Widget	field_menu;
    Widget	scope_knapp;
    Widget	scope_shell;
    Widget	scope_menu;
    Widget	action_knapp;
    Widget	action_shell;
    Widget	action_menu;
    Widget	update_knapp;
    Widget	add_knapp;
    Widget	delete_knapp;
    Widget	group_field;
    Widget	expr_field;
    Widget	color_field;
    Widget	messageid_gadget;
    Widget	subject_gadget;
    Widget	from_gadget;
    Widget	xref_gadget;
} kill_widgets;

static KillField	temp_field = KillFieldSubject;
static KillScope	temp_scope = KillScopeArticle;
static KillAction	temp_action = KillActionKill;

/****************************************************************************/

static void fix_pixmap(KILL_NODE *node)
{
    Display	*disp = XtDisplay(kill_widgets.shell);

    if (node->hot == 0)
	create_hot_pixmap(node);
    else {
	Window		win = XtWindow(kill_widgets.shell);
	GC		gc;
	XGCValues	values;

	if (node->alloced_pixel)
	    values.foreground = node->pixel;
	else
	    values.foreground = global.default_hot_pixel;
	gc = XCreateGC(disp, win, GCForeground, &values);
	XFillRectangle(disp, global.hot_pixmaps[node->hot], gc,
		       0, 0, HOT_PIXMAP_SIZE, HOT_PIXMAP_SIZE);
	XFreeGC(disp, gc);
    }
}

static void check_if_valid(KILL_NODE *node)
{
    if (node->expr_str && node->group_str && node->group_re &&
	((node->field == KillFieldMsgid && node->msgid_valid) ||
	 node->expr_re) &&
	(node->action == KillActionKill|| node->color))
	node->valid = True;
    else
	node->valid = False;
}

static void print_kill_info(char *buffer, KILL_NODE *node)
{
    memset(buffer, ' ', LINE_LEN);

    if (node->valid)
	if (node->applicable && node->unused)
	    buffer[0] = '!';
	else
	    buffer[0] = ' ';
    else
	buffer[0] = '-';

    buffer[1] = "MSFX"[node->field];
    buffer[2] = "ASTt"[node->scope];
    buffer[3] = "KHh"[node->action];

    if (node->group_str) {
	int	n = strlen(node->group_str);

	if (n > 42) n = 42;
	strncpy(&buffer[5], node->group_str, n);
    }

    if (node->expr_str) {
	int	n = strlen(node->expr_str);

	if (n > LINE_LEN - 44)
	    n = LINE_LEN - 44;
	strncpy(&buffer[44], node->expr_str, n);
    }

    buffer[LINE_LEN - 1] = '\0';
}

static void update_list_entry(long n)
{
    if (n >= 0) {
	char	buffer[LINE_LEN];
	Pixmap	pixmap;

	print_kill_info(buffer, global.kill_list[n]);
	pixmap = global.hot_pixmaps[global.kill_list[n]->hot];
	ScrListSetLine(kill_widgets.list, n, buffer, pixmap);
    }
}

static void update_field_knapp(KillField field)
{
    temp_field = field;
    KnappSetLabelNo(kill_widgets.field_knapp, field, True);
}

static void update_field_menu_sensitives(KillField field,
					 Boolean all_sensitive)
{
    XtSetSensitive(kill_widgets.messageid_gadget,
		   all_sensitive || field == KillFieldMsgid);
    XtSetSensitive(kill_widgets.subject_gadget,
		   all_sensitive || field != KillFieldMsgid);
    XtSetSensitive(kill_widgets.from_gadget,
		   all_sensitive || field != KillFieldMsgid);
    XtSetSensitive(kill_widgets.xref_gadget,
		   all_sensitive || field != KillFieldMsgid);
}

static void update_scope_knapp(KillScope scope)
{
    temp_scope = scope;
    KnappSetLabelNo(kill_widgets.scope_knapp, scope, True);
}

static void update_action_knapp(KillAction action)
{
    temp_action = action;
    KnappSetLabelNo(kill_widgets.action_knapp, action, True);
}

static void update_kill_controls(long n)
{
    if (n >= 0) {
	KILL_NODE	*node = global.kill_list[n];

	update_field_knapp(node->field);
	update_field_menu_sensitives(node->field, False);
	update_scope_knapp(node->scope);
	update_action_knapp(node->action);

	TextFieldSetBuffer(kill_widgets.group_field, node->group_str);
	TextFieldSetBuffer(kill_widgets.expr_field, node->expr_str);
	TextFieldSetBuffer(kill_widgets.color_field, node->color);
    } else {
	update_field_knapp(KillFieldSubject);
	update_field_menu_sensitives(KillFieldSubject, True);
	update_scope_knapp(KillScopeArticle);
	update_action_knapp(KillActionKill);

	TextFieldSetBuffer(kill_widgets.group_field, NULL);
	TextFieldSetBuffer(kill_widgets.expr_field, NULL);
	TextFieldSetBuffer(kill_widgets.color_field, NULL);
    }
}

static void init_kill_editor(void)
{
    long	shown, n;

    ScrListClearLines(kill_widgets.list);

    for (n = 0 ; n < global.no_alloc_kill_list ; n++) {
	KILL_NODE	*node = global.kill_list[n];
	char		buffer[LINE_LEN];
	Pixmap		pixmap;

	if (!node) break;

	print_kill_info(buffer, node);
	pixmap =
	    node->action == KillActionKill ?
	    None : global.hot_pixmaps[node->hot];
	ScrListAddLine(kill_widgets.list, buffer, pixmap);
    }

    ScrListGetFirstShownSize(kill_widgets.list, NULL, &shown, &n);
    ScrBarSetLengthsAndPos(kill_widgets.scrbar, n, shown, 0);

    update_kill_controls(-1);
    XtSetSensitive(kill_widgets.color_field, False);
    XtSetSensitive(kill_widgets.add_knapp, True);
    XtSetSensitive(kill_widgets.delete_knapp, False);

    if (global.curr_group) {
	char	*temp;

	temp = regexp_escape_string(global.curr_group->name, True);
	TextFieldSetBuffer(kill_widgets.group_field, temp);
	XtFree(temp);
    }
}

static void close_knapp_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    XtPopdown(kill_widgets.shell);
}

static void clear_knapp_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    long	n = ScrListGetFirstSelected(kill_widgets.list);

    if (n >= 0) ScrListSetSelected(kill_widgets.list, n, False);
    update_kill_controls(-1);
    XtSetSensitive(kill_widgets.add_knapp, True);
    XtSetSensitive(kill_widgets.delete_knapp, False);
    XtSetKeyboardFocus(kill_widgets.shell,
		       kill_widgets.expr_field);
    XtSetSensitive(kill_widgets.color_field, False);
}

static void delete_knapp_callback(Widget w,
				  XtPointer client_data,
				  XtPointer call_data)
{
    long	n = ScrListGetFirstSelected(kill_widgets.list);

    if (n >= 0) {
	Display		*disp = XtDisplay(kill_widgets.shell);
	KILL_NODE	*node = global.kill_list[n];
	Arg		arg;
	long		i, shown;

	ScrListDeleteLine(kill_widgets.list, n);

	for (i = n ; i < global.no_alloc_kill_list - 1 ; i++)
	    global.kill_list[i] = global.kill_list[i + 1];
	global.kill_list[i] = NULL;

	if (node->alloced_pixel) {
	    Colormap	cmap;

	    XtSetArg(arg, XtNcolormap, &cmap);
	    XtGetValues(main_widgets.shell, &arg, 1);

	    XFreeColors(disp, cmap, &node->pixel, 1, 0);
	}

	if (node->hot > 0) {
	    Pixmap	pixmap = global.hot_pixmaps[node->hot];

	    purge_hot(node->hot);
	    global.hot_pixmaps[node->hot] = None;
	    if (pixmap != None) XFreePixmap(disp, pixmap);
	}

	if (node->expr_re) {
	    regfree(node->expr_re);
	    XtFree((char *)node->expr_re);
	    node->expr_re = NULL;
	}
	if (node->group_re) {
	    regfree(node->group_re);
	    XtFree((char *)node->group_re);
	    node->group_re = NULL;
	}
	XtFree(node->expr_str);
	XtFree(node->group_str);
	XtFree(node->color);
	XtFree((char *)node);

	ScrListGetFirstShownSize(kill_widgets.list, &i, &shown, &n);
	ScrBarSetLengthsAndPos(kill_widgets.scrbar, n, shown, i);

	XtSetSensitive(kill_widgets.add_knapp, True);
    }
}

static void add_knapp_callback(Widget w,
			       XtPointer client_data,
			       XtPointer call_data)
{
    KILL_NODE	*node;
    Arg		args[3];
    char	*c;
    Boolean	valid = True;
    regex_t	re;

    if (ScrListGetFirstSelected(kill_widgets.list) >= 0)
	return;

    node = (KILL_NODE *)XtMalloc(sizeof(KILL_NODE));
    node->applicable = False;
    node->unused = False;
    node->expr_re = NULL;
    node->group_re = NULL;
    node->field = temp_field;
    node->scope = temp_scope;

    XtSetArg(args[0], XtNbuffer, &c);
    XtGetValues(kill_widgets.group_field, args, 1);
    if (c && c[0] != '\0') {
	node->group_str = XtNewString(c);
	if (regcomp(&re, c, REGEXP_COMPILE_FLAGS) != 0)
	    valid = False;
	else {
	    node->group_re = (regex_t *)XtMalloc(sizeof(regex_t));
	    memcpy(node->group_re, &re, sizeof(regex_t));
	}
    } else {
	node->group_str = NULL;
	valid = False;
    }

    XtSetArg(args[0], XtNbuffer, &c);
    XtGetValues(kill_widgets.expr_field, args, 1);
    if (temp_field == KillFieldMsgid) {
	node->msgid_valid = False;

	if (c) {
	    while (IS_SPACE(*c))
		c++;
	    node->expr_str = XtNewString(c);
	    if (*c == '<') {
		char	*p = c + strlen(c) - 1;

		while (IS_SPACE(*p))
		    p--;
		if (*p == '>')
		    node->msgid_valid = True;
	    }
	}

	if (!node->msgid_valid) {
	    node->expr_str = NULL;
	    node->msgid_valid = False;
	    valid = False;
	}
    } else {
	node->msgid_valid = False;
	if (c && c[0] != '\0') {
	    node->expr_str = XtNewString(c);
	    if (regcomp(&re, c, REGEXP_COMPILE_FLAGS) != 0)
		valid = False;
	    else {
		node->expr_re = (regex_t *)XtMalloc(sizeof(regex_t));
		memcpy(node->expr_re, &re, sizeof(regex_t));
	    }
	} else {
	    node->expr_str = NULL;
	    valid = False;
	}
    }

    node->action = temp_action;
    if (temp_action == KillActionKill) {
	node->hot = 0;
	node->alloced_pixel = False;
	node->color = NULL;
    } else {
	Display		*disp = XtDisplay(kill_widgets.shell);
	Colormap	cmap;
	XColor		col;
	int		ok;

	XtSetArg(args[0], XtNbuffer, &c);
	XtSetArg(args[1], XtNcolormap, &cmap);
	XtGetValues(kill_widgets.color_field, args, 2);
	if (!c)
	    c = "";
	node->color = XtNewString(c);

	ok = XParseColor(disp, cmap, node->color, &col);
	if (ok && XAllocColor(disp, cmap, &col)) {
	    node->pixel = col.pixel;
	    node->alloced_pixel = True;
	} else {
	    node->pixel = global.default_hot_pixel;
	    node->alloced_pixel = False;
	    popup_colornotice(!ok);
	}

	create_hot_pixmap(node);
    }

    node->valid = valid;
    if (!valid && global.bell)
	XBell(XtDisplay(w), 0);

    kill_add_node(node, True);
}

static void update_knapp_callback(Widget w,
				  XtPointer client_date,
				  XtPointer call_data)
{
    if (update_kill_file())
	set_message("Kill file updated.", False);
    else
	set_message("Failed to update kill file!", True);
}

static void list_sel_callback(Widget w,
			      XtPointer client_data,
			      XtPointer call_data)
{
    long	n = ScrListGetFirstSelected(kill_widgets.list);

    if (n >= 0) {
	KillAction	action = global.kill_list[n]->action;

	update_kill_controls(n);
	XtSetSensitive(kill_widgets.add_knapp, False);
	XtSetSensitive(kill_widgets.delete_knapp, True);
#if (XtSpecificationRelease >= 6)
	if (action != KillActionKill &&
	    kill_widgets.color_field ==
	    XtGetKeyboardFocusWidget(kill_widgets.shell)) {
	    XtSetKeyboardFocus(kill_widgets.shell,
			       kill_widgets.expr_field);
	}
#endif
	XtSetSensitive(kill_widgets.color_field, action != KillActionKill);
    } else {
	XtSetSensitive(kill_widgets.add_knapp, True);
	XtSetSensitive(kill_widgets.delete_knapp, False);
    }
}

static void list_dnd_callback(Widget w,
			      XtPointer client_data,
			      XtPointer call_data)
{
    long	*index = (long *)call_data;
    long	n;
    KILL_NODE	*temp;

    index[2] = False;

    if (global.busy || !global.kill_list)
	return;

    n = 0;
    while (global.kill_list[n])
	n++;
    if (index[0] >= n || index[1] >= n)
	return;

    index[2] = True;
    temp = global.kill_list[index[0]];
    if (index[0] < index[1])
	for (n = index[0] ; n < index[1] ; n++)
	    global.kill_list[n] = global.kill_list[n + 1];
    else
	for (n = index[0] ; n > index[1] ; n--)
	    global.kill_list[n] = global.kill_list[n - 1];
    global.kill_list[index[1]] = temp;
}

static void list_resize_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    ScrListReport	*report = (ScrListReport *)call_data;

    ScrBarSetLengthsAndPos(kill_widgets.scrbar, report->canvas_length,
			   report->slider_length, report->slider_position);
}

static void scrbar_callback(Widget w,
			    XtPointer client_data,
			    XtPointer call_data)
{
    ScrBarReport	*report = (ScrBarReport *)call_data;
    long		shown, size;

    ScrListSetFirst(kill_widgets.list, report->slider_position);
    ScrListGetFirstShownSize(kill_widgets.list, NULL, &shown, &size);
    if ((report->slider_length != shown || report->canvas_length != size) &&
	report->slider_position + shown < size)
	ScrBarSetLengthsAndPos(kill_widgets.scrbar,
			       size, shown, report->slider_position);
}

static void field_menu_callback(Widget w,
				XtPointer client_data,
				XtPointer call_data)
{
    KillField	field = (KillField)client_data;
    long	n = ScrListGetFirstSelected(kill_widgets.list);

    if (n >= 0) {
	KILL_NODE	*node = global.kill_list[n];

	if (SAME_FIELD_TYPE(node->field, field)) {
	    global.kill_list[n]->field = field;
	    update_list_entry(n);
	    update_field_knapp(field);
	} else {
	    if (global.bell)
		XBell(XtDisplay(w), 0);
	}
    } else {
	char	*buffer;
	Arg	arg;

	XtSetArg(arg, XtNbuffer, &buffer);
	XtGetValues(kill_widgets.expr_field, &arg, 1);

	if (buffer && buffer[0] != '\0') {
	    if (SAME_FIELD_TYPE(temp_field, field))
		update_field_knapp(field);
	    else if (global.bell)
		XBell(XtDisplay(w), 0);
	    update_field_menu_sensitives(temp_field, False);
	} else {
	    buffer = NULL;

	    switch (field) {
	    case KillFieldMsgid:
		if (global.curr_art) {
		    buffer = XtMalloc(global.curr_art->hash_len + 3);
		    sprintf(buffer, "<%s>", global.curr_art->msgid);
		}
		break;
	    case KillFieldSubject:
		if (global.curr_art)
		    buffer =
			regexp_escape_string(global.curr_art->subject->subject,
					     False);
		else if (global.curr_subj)
		    buffer =
			regexp_escape_string(global.curr_subj->subject, False);
		break;
	    case KillFieldFrom:
		if (global.curr_art && global.curr_art->from)
		    buffer =
			regexp_escape_string(global.curr_art->from, False);
		break;
	    case KillFieldXref:
		break;
	    }

	    TextFieldSetBuffer(kill_widgets.expr_field, buffer);
	    XtFree(buffer);

	    update_field_knapp(field);
	    if (buffer)
		update_field_menu_sensitives(field, False);
	}
    }
}

static void scope_menu_callback(Widget w,
				XtPointer client_data,
				XtPointer call_data)
{
    KillScope	scope = (KillScope)client_data;
    long	n = ScrListGetFirstSelected(kill_widgets.list);

    if (n >= 0) {
	global.kill_list[n]->scope = scope;
	update_list_entry(n);
    }

    update_scope_knapp(scope);
}

static void action_menu_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    KillAction	action = (KillAction)client_data;
    long	n = ScrListGetFirstSelected(kill_widgets.list);
    Arg		arg;

    if (action == KillActionKill) {
	if (n >= 0) {
	    KILL_NODE	*node = global.kill_list[n];

	    if (node->alloced_pixel) {
		Display		*disp = XtDisplay(kill_widgets.shell);
		Colormap	cmap;

		XtSetArg(arg, XtNcolormap, &cmap);
		XtGetValues(kill_widgets.shell, &arg, 1);

		XFreeColors(disp, cmap, &node->pixel, 1, 0);
		node->alloced_pixel = False;
	    }

	    if (node->hot > 0) {
		purge_hot(node->hot);
		node->hot = 0;
	    }

	    node->action = KillActionKill;
	    update_list_entry(n);
	}

	TextFieldSetBuffer(kill_widgets.color_field, NULL);
#if (XtSpecificationRelease >= 6)
	if (kill_widgets.color_field ==
	    XtGetKeyboardFocusWidget(kill_widgets.shell)) {
	    XtSetKeyboardFocus(kill_widgets.shell,
			       kill_widgets.expr_field);
	}
#endif
	XtSetSensitive(kill_widgets.color_field, False);
	
	update_action_knapp(KillActionKill);
    } else {
	if (n >= 0) {
	    KILL_NODE	*node = global.kill_list[n];

	    node->action = action;
	    update_list_entry(n);
	}

	XtSetSensitive(kill_widgets.color_field, True);

	update_action_knapp(action);
    }
}

static void group_field_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    char	*buffer = (char *)call_data;
    long	n = ScrListGetFirstSelected(kill_widgets.list);
    regex_t	re;
    int		code;

    if (!buffer)
	buffer = "";

    code = regcomp(&re, buffer, REGEXP_COMPILE_FLAGS);
    if (code != 0)
	popup_regexpnotice(code, &re);

    if (n >= 0) {
	KILL_NODE	*node = global.kill_list[n];

	XtFree(node->group_str);
	node->group_str = XtNewString(buffer);

	if (code == 0) {
	    if (node->group_re)
		regfree(node->group_re);
	    else
		node->group_re = (regex_t *)XtMalloc(sizeof(regex_t));
	    memcpy(node->group_re, &re, sizeof(regex_t));
	} else if (node->group_re) {
	    regfree(node->group_re);
	    XtFree((char *)node->group_re);
	    node->group_re = NULL;
	}

	check_if_valid(node);
	update_list_entry(n);
    } else if (code == 0) {
	regfree(&re);
    }

    if (code == 0)
	XtSetKeyboardFocus(kill_widgets.shell, kill_widgets.expr_field);
}

static void expr_field_callback(Widget w,
				XtPointer client_data,
				XtPointer call_data)
{
    char	*buffer = (char *)call_data;
    long	n = ScrListGetFirstSelected(kill_widgets.list);
    Boolean	valid = False;

    /* assert SAME_FIELD_TYPE(node->field, temp_field) */
    if (!buffer)
	buffer = "";

    if (temp_field == KillFieldMsgid) {
	char	*c = buffer;

	while (isspace((unsigned char)*c))
	    c++;
	c = XtNewString(c);
	if (c[0] == '<') {
	    char	*p;

	    p = c + strlen(c) - 1;
	    while (isspace((unsigned char)*p))
		p--;
	    if (p[0] == '>') {
		p[1] = '\0';
		p = strchr(c, '@');
		if (p) {
		    do {
			p++;
			if ('A' <= p[0] && p[0] <= 'Z')
			    p[0] += 'a' - 'A';
		    } while (*p != '>');
		    valid = True;
		}
	    }
	}
			
	if (n >= 0) {
	    KILL_NODE	*node = global.kill_list[n];

	    XtFree(node->expr_str);
	    node->expr_str = c;
	    if (node->expr_re)
		regfree(node->expr_re);
	    node->expr_re = NULL;
	    node->msgid_valid = valid;
	    check_if_valid(node);
	    update_list_entry(n);
	}

	if (!valid) {
	    if (global.bell)
		XBell(XtDisplay(w), 0);
	    popup_notice("exprfieldnotice", "Invalid Message-ID.",
			 "OK", NULL, NULL, 2000, NULL, NULL, XtGrabNone);
	}

	TextFieldSetBuffer(kill_widgets.expr_field, c);
	
	if (n < 0)
	    XtFree(c);
    } else {
	regex_t	re;
	int	code;

	code = regcomp(&re, buffer, REGEXP_COMPILE_FLAGS);
	valid = (code == 0);
	if (!valid)
	    popup_regexpnotice(code, &re);

	if (n >= 0) {
	    KILL_NODE	*node = global.kill_list[n];

	    XtFree(node->expr_str);
	    node->expr_str = XtNewString(buffer);

	    if (valid) {
		if (node->expr_re)
		    regfree(node->expr_re);
		else
		    node->expr_re = (regex_t *)XtMalloc(sizeof(regex_t));
		memcpy(node->expr_re, &re, sizeof(regex_t));
	    } else {
		if (node->expr_re) {
		    regfree(node->expr_re);
		    XtFree((char *)node->expr_re);
		    node->expr_re = NULL;
		}
	    }

	    check_if_valid(node);
	    update_list_entry(n);
	} else if (valid) {
	    regfree(&re);
	}

	if (!valid) {
	    if (global.bell)
		XBell(XtDisplay(w), 0);
	    popup_notice("exprfieldnotice", "Parse error in regexp.",
			 "OK", NULL, NULL, 2000, NULL, NULL, XtGrabNone);
	}
    }

    if (valid) {
	if (temp_action == KillActionKill)
	    XtSetKeyboardFocus(kill_widgets.shell, kill_widgets.group_field);
	else
	    XtSetKeyboardFocus(kill_widgets.shell, kill_widgets.color_field);
    }

    if (buffer[0] == '\0')
	update_field_menu_sensitives(KillFieldMsgid, True);
    else
	update_field_menu_sensitives(temp_field, False);
}

static void color_field_callback(Widget w,
				 XtPointer client_data,
				 XtPointer call_data)
{
    char	*buffer = (char *)call_data;
    long	n = ScrListGetFirstSelected(kill_widgets.list);
    Display	*disp = XtDisplay(w);
    Colormap	cmap = 0;
    XColor	col;
    Arg		arg;
    int		ok;

    if (temp_action == KillActionKill)
	return;

    if (!buffer)
	buffer = "";

    XtSetArg(arg, XtNcolormap, &cmap);
    XtGetValues(w, &arg, 1);

    if (n >= 0) {
	KILL_NODE	*node = global.kill_list[n];

	if (node->alloced_pixel)
	    XFreeColors(disp, cmap, &node->pixel, 1, 0);
	node->alloced_pixel = False;
	XtFree(node->color);
	node->color = XtNewString(buffer);

	ok = XParseColor(disp, cmap, buffer, &col);
	if (ok && XAllocColor(disp, cmap, &col)) {
	    node->alloced_pixel = True;
	    node->pixel = col.pixel;
	} else {
	    popup_colornotice(!ok);
	}

	fix_pixmap(node);
	update_list_entry(n);
    } else {
	ok = XParseColor(disp, cmap, buffer, &col);
	if (!ok) {
	    if (global.bell)
		XBell(XtDisplay(w), 0);
	    popup_colornotice(True);
	}
    }

    if (ok)
	XtSetKeyboardFocus(kill_widgets.shell, kill_widgets.group_field);
}

/*************************************************************************/

static void create_kill_widgets(void)
{
    Arg		args[8];
    Widget	layout, temp;

    XtSetArg(args[0], XtNallowShellResize, True);
    XtSetArg(args[1], XtNinput, True);
    XtSetArg(args[2], XtNcolormap, global.cmap);
    XtSetArg(args[3], XtNvisual, global.visual);
    XtSetArg(args[4], XtNdepth, global.depth);
    kill_widgets.shell =
	XtCreatePopupShell("killeditor", topLevelShellWidgetClass,
			   main_widgets.shell, args, 5);

    layout =
	XtVaCreateManagedWidget("killayout", layoutWidgetClass,
				kill_widgets.shell,
				XtVaTypedArg, XtNlayout, XtRString,
#include "layouts/kill.l"
				(int)sizeof(String), NULL);

    XtSetArg(args[0], XtNatMostOne, True);
    XtSetArg(args[1], XtNatLeastOne, False);
    XtSetArg(args[2], XtNpixmapWidth, HOT_PIXMAP_SIZE);
    XtSetArg(args[3], XtNpixmapHeight, HOT_PIXMAP_SIZE);
    XtSetArg(args[4], XtNdepthOne, False);
    XtSetArg(args[5], XtNallowDnd, True);
    XtSetArg(args[6], XtNusePixmaps, True);
    kill_widgets.list =
	XtCreateManagedWidget("killist", scrListWidgetClass, layout, args, 7);
    XtAddCallback(kill_widgets.list, XtNselectCallback,
		  list_sel_callback, NULL);
    XtAddCallback(kill_widgets.list, XtNresizeCallback,
		  list_resize_callback, NULL);
    XtAddCallback(kill_widgets.list, XtNdndCallback, list_dnd_callback, NULL);

    XtSetArg(args[0], XtNallowOff, True);
    kill_widgets.scrbar =
	XtCreateManagedWidget("scrbar", scrBarWidgetClass, layout, args, 1);
    XtAddCallback(kill_widgets.scrbar, XtNscrollCallback,
		  scrbar_callback, NULL);

    XtSetArg(args[0], XtNcursor, None);
    XtCreateManagedWidget("sash", sashWidgetClass, layout, args, 1);

    temp = XtCreateManagedWidget("close", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(temp, XtNcallback, close_knapp_callback, NULL);

    kill_widgets.delete_knapp =
	XtCreateManagedWidget("delete", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(kill_widgets.delete_knapp, XtNcallback,
		  delete_knapp_callback, NULL);

    temp = XtCreateManagedWidget("clear", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(temp, XtNcallback, clear_knapp_callback, NULL);

    kill_widgets.add_knapp =
	XtCreateManagedWidget("add", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(kill_widgets.add_knapp, XtNcallback,
		  add_knapp_callback, NULL);

    kill_widgets.update_knapp =
	XtCreateManagedWidget("update", knappWidgetClass, layout, NULL, 0);
    XtAddCallback(kill_widgets.update_knapp, XtNcallback,
		  update_knapp_callback, NULL);

    XtSetArg(args[0], XtNmenuName, "fieldshell");
    XtSetArg(args[1], XtNresizable, False);
    kill_widgets.field_knapp =
	XtCreateManagedWidget("fieldknapp", menuKnappWidgetClass,
			      layout, args, 2);

    XtSetArg(args[0], XtNcolormap, global.cmap);
    XtSetArg(args[1], XtNvisual, global.visual);
    XtSetArg(args[2], XtNdepth, global.depth);
    kill_widgets.field_shell =
	XtCreatePopupShell("fieldshell", popdownShellWidgetClass,
			   kill_widgets.shell, args, 3);

    kill_widgets.field_menu =
	XtCreateManagedWidget("fieldmenu", menuWidgetClass,
			      kill_widgets.field_shell, NULL, 0);

    XtCreateManagedWidget("fieldmessage", messageWidgetClass, layout, NULL, 0);
    
    kill_widgets.messageid_gadget =
	MenuCreateGadget("messageid", stringGadgetClass,
			 kill_widgets.field_menu, NULL, 0);
    XtAddCallback(kill_widgets.messageid_gadget, XtNcallback,
		  field_menu_callback, (XtPointer)KillFieldMsgid);
    MenuCreateGadget("separator", separatorGadgetClass,
		     kill_widgets.field_menu, NULL, 0);
    kill_widgets.subject_gadget =
	MenuCreateGadget("subject", stringGadgetClass,
			 kill_widgets.field_menu, NULL, 0);
    XtAddCallback(kill_widgets.subject_gadget, XtNcallback,
		  field_menu_callback, (XtPointer)KillFieldSubject);
    kill_widgets.from_gadget =
	MenuCreateGadget("from", stringGadgetClass,
			 kill_widgets.field_menu, NULL, 0);
    XtAddCallback(kill_widgets.from_gadget, XtNcallback,
		  field_menu_callback, (XtPointer)KillFieldFrom);
    kill_widgets.xref_gadget =
	MenuCreateGadget("xref", stringGadgetClass,
			 kill_widgets.field_menu, NULL, 0);
    XtAddCallback(kill_widgets.xref_gadget, XtNcallback,
		  field_menu_callback, (XtPointer)KillFieldXref);

    XtSetArg(args[0], XtNmenuName, "scopeshell");
    XtSetArg(args[1], XtNresizable, False);
    kill_widgets.scope_knapp =
	XtCreateManagedWidget("scopeknapp", menuKnappWidgetClass,
			      layout, args, 2);

    XtSetArg(args[0], XtNcolormap, global.cmap);
    XtSetArg(args[1], XtNvisual, global.visual);
    XtSetArg(args[2], XtNdepth, global.depth);
    kill_widgets.scope_shell =
	XtCreatePopupShell("scopeshell", popdownShellWidgetClass,
			   kill_widgets.shell, args, 3);

    kill_widgets.scope_menu =
	XtCreateManagedWidget("scopemenu", menuWidgetClass,
			      kill_widgets.scope_shell, NULL, 0);

    XtCreateManagedWidget("scopemessage", messageWidgetClass, layout, NULL, 0);

    temp = MenuCreateGadget("article", stringGadgetClass,
			    kill_widgets.scope_menu, NULL, 0);
    XtAddCallback(temp, XtNcallback,
		  scope_menu_callback, (XtPointer)KillScopeArticle);
    temp = MenuCreateGadget("subject", stringGadgetClass,
			    kill_widgets.scope_menu, NULL, 0);
    XtAddCallback(temp, XtNcallback,
		  scope_menu_callback, (XtPointer)KillScopeSubject);
    temp = MenuCreateGadget("thread", stringGadgetClass,
			    kill_widgets.scope_menu, NULL, 0);
    XtAddCallback(temp, XtNcallback,
		  scope_menu_callback, (XtPointer)KillScopeThread);
    temp = MenuCreateGadget("subthread", stringGadgetClass,
			    kill_widgets.scope_menu, args, 0);
    XtAddCallback(temp, XtNcallback,
		  scope_menu_callback, (XtPointer)KillScopeSubthread);

    XtSetArg(args[0], XtNresizable, False);
    XtSetArg(args[1], XtNmenuName, "actionshell");
    kill_widgets.action_knapp =
	XtCreateManagedWidget("actionknapp", menuKnappWidgetClass,
			      layout, args, 2);

    XtSetArg(args[0], XtNcolormap, global.cmap);
    XtSetArg(args[1], XtNvisual, global.visual);
    XtSetArg(args[2], XtNdepth, global.depth);
    kill_widgets.action_shell =
	XtCreatePopupShell("actionshell", popdownShellWidgetClass,
			   kill_widgets.shell, args, 3);

    kill_widgets.action_menu =
	XtCreateManagedWidget("actionmenu", menuWidgetClass,
			      kill_widgets.action_shell, NULL, 0);

    temp = MenuCreateGadget("kill", stringGadgetClass,
			    kill_widgets.action_menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, action_menu_callback,
		  (XtPointer)KillActionKill);
    temp = MenuCreateGadget("hot", stringGadgetClass,
			    kill_widgets.action_menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, action_menu_callback,
		  (XtPointer)KillActionHot);
    temp = MenuCreateGadget("hotish", stringGadgetClass,
			    kill_widgets.action_menu, NULL, 0);
    XtAddCallback(temp, XtNcallback, action_menu_callback,
		  (XtPointer)KillActionHotish);

    XtSetArg(args[0], XtNfocusRoot, kill_widgets.shell);
    kill_widgets.group_field =
	XtCreateManagedWidget("groupfield", textFieldWidgetClass,
			      layout, args, 1);
    XtAddCallback(kill_widgets.group_field, XtNcallback,
		  group_field_callback, NULL);
    kill_widgets.expr_field =
	XtCreateManagedWidget("exprfield", textFieldWidgetClass,
			      layout, args, 1);
    XtAddCallback(kill_widgets.expr_field, XtNcallback,
		  expr_field_callback, NULL);
    XtSetArg(args[1], XtNpreferredChars, 16);
    kill_widgets.color_field =
	XtCreateManagedWidget("colorfield", textFieldWidgetClass,
			      layout, args, 2);
    XtAddCallback(kill_widgets.color_field, XtNcallback,
		  color_field_callback, NULL);

    XtCreateManagedWidget("groupmessage", messageWidgetClass, layout, NULL, 0);
    XtCreateManagedWidget("exprmessage", messageWidgetClass, layout, NULL, 0);
    XtCreateManagedWidget("colormessage", messageWidgetClass, layout, NULL, 0);

    XtSetKeyboardFocus(kill_widgets.shell, kill_widgets.expr_field);
    XtRealizeWidget(kill_widgets.shell);
    XtInstallAllAccelerators(kill_widgets.shell, kill_widgets.shell);

    add_WM_DELETE_WINDOW_callback(kill_widgets.shell,
				  close_knapp_callback, NULL);

    if (global.busy)
	set_busy_kill(True);
}

void popup_kill_editor(void)
{
    if (!kill_widgets.shell)
	create_kill_widgets();
    init_kill_editor();
    XtPopup(kill_widgets.shell, XtGrabNone);
}

void popdown_kill_editor(void)
{
    if (kill_widgets.shell)
	XtPopdown(kill_widgets.shell);
}

void set_busy_kill(int busy)
{
#if 0
    if (!kill_widgets.shell)
	return;

    XDefineCursor(XtDisplay(kill_widgets.shell), XtWindow(kill_widgets.shell),
		  busy ? global.busy_cursor : global.cursor);
#endif
}

void kill_add_node(KILL_NODE *node, int append)
{
    long	first, shown, size;
    long	i;

    for (i = 0 ; i < global.no_alloc_kill_list ; i++)
	if (!global.kill_list[i])
	    break;

    if (i + 3 > global.no_alloc_kill_list) {
	int	j = global.no_alloc_kill_list;

	global.no_alloc_kill_list = 2 * (global.no_alloc_kill_list + 1);
	global.kill_list =
	    (KILL_NODE **)XtRealloc((char *)global.kill_list,
				    global.no_alloc_kill_list *
				    sizeof(KILL_NODE*));

	while (j < global.no_alloc_kill_list)
	    global.kill_list[j++] = NULL;
    }

    if (append)
	global.kill_list[i] = node;
    else {
	memmove(global.kill_list + 1, global.kill_list,
		(global.no_alloc_kill_list - 1) * sizeof(KILL_NODE*));
	global.kill_list[0] = node;
	i = 0;
    }

    if (!kill_widgets.shell || !is_popped_up(kill_widgets.shell))
	popup_kill_editor();
    else if (!append)
	init_kill_editor();
    else {
	char	buffer[LINE_LEN];
	Pixmap	pixmap;

	print_kill_info(buffer, node);
	if (node->action == KillActionKill)
	    pixmap = None;
	else
	    pixmap = global.hot_pixmaps[node->hot];
	ScrListAddLine(kill_widgets.list, buffer, pixmap);
    }
    ScrListSetSelected(kill_widgets.list, i, True);
    ScrListGetFirstShownSize(kill_widgets.list, &first, &shown, &size);
    if (i < first) {
	ScrListSetFirst(kill_widgets.list, i);
	ScrListGetFirstShownSize(kill_widgets.list, &first, &shown, &size);
    } else if (i >= first + shown) {
	first = i - shown + 1;
	if (first < 0)
	    first = 0;
	ScrListSetFirst(kill_widgets.list, first);
	ScrListGetFirstShownSize(kill_widgets.list, &first, &shown, &size);
    }
    ScrBarSetLengthsAndPos(kill_widgets.scrbar, size, shown, first);
    update_kill_controls(i);
    XtSetSensitive(kill_widgets.add_knapp, False);
    XtSetSensitive(kill_widgets.delete_knapp, True);
    XtSetSensitive(kill_widgets.color_field, node->action != KillActionKill);
}
