/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include "font.h"
#include "parse.h"
#include "util.h"

const char month_names[] = "JanFebMarAprMayJunJulAugSepOctNovDec";

char *time_t_to_date(time_t t, char *buffer)
{
    struct tm	*tm;

    tm = gmtime(&t);
    if (!tm) {
	*buffer = '\0';
	return buffer;
    }

    sprintf(buffer, "%d %3.3s %4d %02d:%02d:%02d GMT",
	    tm->tm_mday, month_names + 3 * tm->tm_mon, 1900 + tm->tm_year,
	    tm->tm_hour, tm->tm_min, tm->tm_sec);

    return buffer;
}

static long tm_to_secs(struct tm *tm)
{
    static int	acc_month_days[] = {
	0,
	31,
	31 + 28,
	31 + 28 + 31,
	31 + 28 + 31 + 30,
	31 + 28 + 31 + 30 + 31,
	31 + 28 + 31 + 30 + 31 + 30,
	31 + 28 + 31 + 30 + 31 + 30 + 31,
	31 + 28 + 31 + 30 + 31 + 30 + 31 + 31,
	31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30,
	31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31,
	31 + 28 + 31 + 30 + 31 + 30 + 31 + 31 + 30 + 31 + 30,
    };
    long	i, result = 0;

    i = tm->tm_year - 70;
    if (i < 0)
	return 0;

    switch ((unsigned int)i % 4) {
    case 0:
	result = 0;
	break;
    case 1:
	result = 365;
	break;
    case 2:  /* Leap year, except 2000 */
	if (i != 30 && tm->tm_mon > 1)
	    result = 365 + 365 + 1;
	else
	    result = 365 + 365;
	break;
    case 3:
	result = 365 + 365 + 365 + 1;
	break;
    }

    if (i > 30)  /* past 2000 => we got one day too much ??? */
	result--;

    result += (i / 4) * (365 + 365 + 365 + 365 + 1);
    result += acc_month_days[(unsigned int)tm->tm_mon % 12];
    result += tm->tm_mday - 1;
    result = 24 * result + tm->tm_hour;
    result = 60 * result + tm->tm_min;
    result = 60 * result + tm->tm_sec;

    return result;
}

time_t parsedate(char *str)
{
    struct tm	tm;
    char	c1, c2, c3;

    while (str[0] == ' ')
	str++;

    if (str[0] < '0')
	return PARSEDATE_ERROR;
    else if (str[0] > '9') {
	c1 = str[1];
	if (c1 < 'a')
	    c1 += ('a' - 'A');
	c2 = str[2];
	if (c2 < 'a')
	    c2 += ('a' - 'A');

	switch (str[0]) {
	case 'M':
	case 'm':
	    if (c1 != 'o' || c2 != 'n')
		return PARSEDATE_ERROR;
	    break;
	case 'T':
	case 't':
	    if ((c1 != 'u' || c2 != 'e') &&
		(c1 != 'h' && c2 != 'u'))
		return PARSEDATE_ERROR;
	    break;
	case 'W':
	case 'w':
	    if (c1 != 'e' || c2 != 'd')
		return PARSEDATE_ERROR;
	    break;
	case 'F':
	case 'f':
	    if (c1 != 'r' || c2 != 'i')
		return PARSEDATE_ERROR;
	    break;
	case 'S':
	case 's':
	    if ((c1 != 'a' || c2 != 't') &&
		(c1 != 'u' || c2 != 'n'))
		return PARSEDATE_ERROR;
	    break;
	default:
	    return PARSEDATE_ERROR;
	}

	str += 3;
	while (str[0] == ' ')
	    str++;
	if (str[0] == ',') {
	    str++;
	    while (str[0] == ' ')
		str++;
	}

	if (str[0] < '0' || str[0] > '9')
	    return PARSEDATE_ERROR;
    }

    tm.tm_mday = str[0] - '0';
    str++;
    while (str[0] >= '0' && str[0] <= '9') {
	tm.tm_mday *= 10;
	tm.tm_mday += (str[0] - '0');
	str++;
    }

    while (str[0] == ' ')
	str++;

    c1 = str[1];
    if (c1 < 'a')
	c1 += ('a' - 'A');
    c2 = str[2];
    if (c2 < 'a')
	c2 += ('a' - 'A');
    switch (str[0]) {
    case 'J':
    case 'j':
	if (c1 == 'a') {
	    if (c2 == 'n')
		tm.tm_mon = 0;
	    else
		return PARSEDATE_ERROR;
	} else if (c1 == 'u') {
	    if (c2 == 'n')
		tm.tm_mon = 5;
	    else if (c2 == 'l')
		tm.tm_mon = 6;
	    else
		return PARSEDATE_ERROR;
	} else {
	    return PARSEDATE_ERROR;
	}
	break;
    case 'F':
    case 'f':
	if (c1 == 'e' && c2 == 'b')
	    tm.tm_mon = 1;
	else
	    return PARSEDATE_ERROR;
	break;
    case 'M':
    case 'm':
	if (c1 == 'a') {
	    if (c2 == 'r')
		tm.tm_mon = 2;
	    else if (c2 == 'y')
		tm.tm_mon = 4;
	    else
		return PARSEDATE_ERROR;
	} else {
	    return PARSEDATE_ERROR;
	}
	break;
    case 'A':
    case 'a':
	if (c1 == 'p') {
	    if (c2 == 'r')
		tm.tm_mon = 3;
	    else
		return PARSEDATE_ERROR;
	} else if (c1 == 'u') {
	    if (c2 == 'g')
		tm.tm_mon = 7;
	    else
		return PARSEDATE_ERROR;
	} else {
	    return PARSEDATE_ERROR;
	}
	break;
    case 'S':
    case 's':
	if (c1 == 'e' && c2 == 'p')
	    tm.tm_mon = 8;
	else
	    return PARSEDATE_ERROR;
	break;
    case 'O':
    case 'o':
	if (c1 == 'c' && c2 == 't')
	    tm.tm_mon = 9;
	else
	    return PARSEDATE_ERROR;
	break;
    case 'N':
    case 'n':
	if (c1 == 'o' && c2 == 'v')
	    tm.tm_mon = 10;
	else
	    return PARSEDATE_ERROR;
	break;
    case 'D':
    case 'd':
	if (c1 == 'e' && c2 == 'c')
	    tm.tm_mon = 11;
	else
	    return PARSEDATE_ERROR;
	break;
    default:
	return PARSEDATE_ERROR;
    }

    str += 3;
    while (str[0] == ' ')
	str++;

    if (str[0] < '0' || str[2] > '9')
	return PARSEDATE_ERROR;
    c1 = str[1];
    c2 = str[2];
    c3 = str[3];
    if (c1 < '0' || c1 > '9')
	return PARSEDATE_ERROR;

    if (c2 < '0' || c2 > '9') {
	tm.tm_year = 10 * (str[0] - '0') + (c1 - '0');
	str += 2;
    } else {
	if (c2 < '0' || c2 > '9' || c3 < '0' || c3 > '9')
	    return PARSEDATE_ERROR;
	tm.tm_year = 1000 * (str[0] - '0' - 1) + 100 * (c1 - '0' - 9) +
	    10 * (c2 - '0') + (c3 - '0');
	str += 4;
    }

    while (str[0] == ' ')
	str++;

    if (str[0] < '0' || str[0] > '9' || str[1] < '0' ||
	str[1] > '9' || str[2] != ':')
	return PARSEDATE_ERROR;

    tm.tm_hour = 10 * (str[0] - '0') + str[1] - '0';
    str += 3;

    if (str[0] < '0' || str[0] > '9' || str[1] < '0' || str[1] > '9' ||
	(str[2] != ' ' && str[2] != ':' && str[2] != '\0'))
	return PARSEDATE_ERROR;

    tm.tm_min = 10 * (str[0] - '0') + str[1] - '0';

    if (str[2] == '\0') {
	tm.tm_sec = 0;
	return (time_t)tm_to_secs(&tm);
    } else {
	long	offset;

	if (str[2] == ':') {
	    str += 3;

	    if (str[0] < '0' || str[0] > '9' || str[1] < '0' ||
		str[1] > '9' || (str[2] != ' ' && str[2] != '\0'))
		return PARSEDATE_ERROR;

	    tm.tm_sec = 10 * (str[0] - '0') + str[1] - '0';
	    str += 2;
	} else {
	    tm.tm_sec = 0;
	}

	while (str[0] == ' ')
	    str++;

	c1 = str[1];
	c2 = str[2];
	switch (str[0]) {
	case '-':
	case '+':
	    if (c1 < '0' || c1 > '9' || c2 < '0' || c2 > '9')
		return PARSEDATE_ERROR;
	    if (str[0] == '+')
		offset = - 3600 * (10 * (c1 - '0') + (c2 - '0'));
	    else
		offset = 3600 * (10 * (c1 - '0') + (c2 - '0'));
	    c1 = str[3];
	    c2 = str[4];
	    if (c1 >= '0' && c1 <= '9' && c2 >= '0' && c2 <= '9') {
		if (str[0] == '+')
		    offset -= 60 * (c1 - '0') + (c2 - '0');
		else
		    offset += 60 * (c1 - '0') + (c2 - '0');
	    }
	    break;
	case 'C':
	    if (c2 == 'T' || c2 == 't') { /* CST, CDT, CET */
		if (c1 == 'S' || c1 == 's')
		    offset = 6 * 3600;
		else if (c1 == 'D' || c1 == 'd')
		    offset = 5 * 3600;
		else if (c1 == 'E' || c1 == 'e')
		    offset = - 3600;
		else
		    return PARSEDATE_ERROR;
	    } else if (c2 == 'S' || c2 == 's') { /* CES */
		if (c1 == 'E' || c1 == 'e')
		    offset = - 3600;
		else
		    return PARSEDATE_ERROR;
	    } else {
		return PARSEDATE_ERROR;
	    }
	    break;
	case 'E':
	    if (c2 == 'T' || c2 == 't') { /* EST, EDT, EET */
		if (c1 == 'S' || c1 == 's')
		    offset = 5 * 3600;
		else if (c1 == 'D' || c1 == 'd')
		    offset = 4 * 3600;
		else if (c1 == 'E' || c1 == 'e')
		    offset = - 2 * 3600;
		else
		    return PARSEDATE_ERROR;
	    } else {
		return PARSEDATE_ERROR;
	    }
	    break;
	case 'M':
	    if (c2 == 'T' || c2 == 't') { /* MST, MDT, MET */
		if (c1 == 'S' || c1 == 's')
		    offset = 7 * 3600;
		else if (c1 == 'D' || c1 == 'd')
		    offset = 6 * 3600;
		else if (c1 == 'E' || c1 == 'e')
		    offset = -3600;
		else
		    return PARSEDATE_ERROR;
	    } else if (c2 == 'Z' || c2 == 'z') { /* MEZ */
		if (c1 == 'E' || c1 == 'e')
		    offset = -3600;
		else
		    return PARSEDATE_ERROR;
	    } else {
		return PARSEDATE_ERROR;
	    }
	    break;
	case 'P':
	    if (c2 == 'T' || c2 == 't') { /* PST, PDT */
		if (c1 == 'S' || c1 == 's')
		    offset = 8 * 3600;
		else if (c1 == 'D' || c1 == 'd')
		    offset = 7 * 3600;
		else
		    return PARSEDATE_ERROR;
	    } else {
		return PARSEDATE_ERROR;
	    }
	    break;
	default:
	    offset = 0;
	    break;
	}

	return (time_t)(tm_to_secs(&tm) + offset);
    }
}

/*************************************************************************/

char *eat_re(char *str)
{
    while (*str == 'R' || *str == 'r') {
	if (str[1] == 'e' || str[1] == 'E') {
	    if (str[2] == ':')
		str += 3;
	    else if (str[2] == '^' && str[4] == ':' &&
		     (str[3] >= '0' || str[2] <= '9'))
		str += 5;
	    else
		break;
	} else {
	    break;
	}

	while (*str == ' ')
	    str++;
    }

    return str;
}

char *get_msgid(char *beg, char *end)
{
    char *mid;

    while (*beg == ' ')
	beg++;

    if (beg >= end || *beg++ != '<')
	return NULL;

    while (end > beg && *end == ' ')
	*end-- = '\0';

    if (*end != '>')
	return NULL;

    *end = '\0';

    if ((mid = strchr(beg, '@')))
	for (mid++ ; mid < end ; mid++)
	    if ('A' <= *mid && *mid <= 'Z')
		*mid += 'a' - 'A';

    return beg;
}

char *parse_author(char *from)
{
    char	*c, *p1, *p2;

    c = from;
    while (isspace((unsigned char)*c) || *c == '"')
	c++;
    if (*c == '\0')
	return "<none>";

    p1 = strchr(c, '<');
    if (p1 && p1 > c) {
	do {
	    p1--;
	} while (p1 > c && isspace((unsigned char)*p1));

	if (p1 > c) {
	    if (p1[0] == '"')
		p1[0] = '\0';
	    else
		p1[1] = '\0';
	    return c;
	}
    }

    p1 = strchr(c, '(');
    if (p1) {
	do {
	    p1++;
	} while (isspace((unsigned char)*p1) || *p1 == '(');

	if (*p1 != '\0' && *p1 != ')') {
	    p2 = strchr(p1, ')');
	    if (p2) {
		do {
		    p2--;
		} while (isspace((unsigned char)*p2));

		if (p2 > p1) {
		    p2[1] = '\0';
		    return p1;
		}
	    }
	}
    }

    return c;
}

/*********************************************************************/

MimeEnc parse_content_enc(char *header)
{
    char	*c = header + 26;
    int		len;

    for (;;) {
	while (IS_SPACE(*c))
	    c++;

	if (*c == '(')
	    while (*c != '\0' && *c != ')')
		c++;
	else
	    break;
    }

    if (*c == '"') {
	char	*p;

	c++;
	p = strchr(c, '"');
	if (!p)
	    return MimeEncUnknown;
	len = p - c;
    } else {
	char	*p = c;

	while (*p != '\0' && !isspace((unsigned char)*p))
	    p++;
	len = p - c;
    }

#define IS_ENC(enc) \
    (len == sizeof enc - 1 && case_lstrncmp(c, enc + 1, sizeof enc - 1) == 0)

    switch (*c++) {
    case '7':
	if (IS_ENC("7bit"))
	    return MimeEncNone;
	break;
    case '8':
	if (IS_ENC("8bit"))
	    return MimeEncNone;
	break;
    case 'b':
    case 'B':
	if (IS_ENC("binary"))
	    return MimeEncNone;
	else if (IS_ENC("base64"))
	    return MimeEncBase64;
	break;
    case 'q':
    case 'Q':
	if (IS_ENC("quoted-printable"))
	    return MimeEncQP;
	break;
    case 'U':
    case 'u':
	if (IS_ENC("uue") ||
	    IS_ENC("uuencode") ||
	    IS_ENC("uuencoded"))
	    return MimeEncUue;
	break;
    case 'x':
    case 'X':
	if (IS_ENC("x-uue") ||
	    IS_ENC("x-uuencode") ||
	    IS_ENC("x-uuencoded"))
	    return MimeEncUue;
	break;
    }

#undef IS_ENC

    return MimeEncUnknown;
}

static int get_token(char **c, char **p)
{
    int	len;

    if (**c == '"') {
	*p = ++*c;
	while (**p != '\0' && **p != '"')
	    (*p)++;
	if (**p == '\0')
	    len = -1;
	else
	    len = (*p)++ - *c;
    } else {
	*p = *c;
	while (**p != '\0' && **p != ' ' &&
	       **p != '\t' && **p != '/' &&
	       **p != ';' && **p != '(' &&
	       **p != '=' && **p != '"')
	    (*p)++;
	len = *p - *c;
    }

    return len;
}

int parse_content_type(char **headers,
		       char *type_buf, int type_len,
		       char *subtype_buf, int subtype_len,
		       MimeArg *args, int n_args)
{
    enum {
	LookingForType,
	LookingForSlash,
	LookingForSubtype,
	LookingForParam,
	LookingForEqual,
	LookingForValue,
	LookingForSemicolon
    }		state = LookingForType;
    char	*c, *p;
    int		len, n;

    c = headers[0] + 13;
    n = 0;
    do {
	for (;;) {
	    while (IS_SPACE(*c))
		c++;

	    if (*c == '\0')
		break;
	    if (*c == '(') { /* skip comments */
		while (*c != '\0' && *c != ')')
		    c++;
		if (*c == ')')
		    c++;
		continue;
	    }

	    switch (state) {
	    case LookingForType:
		len = get_token(&c, &p);
		if (len < 0 || len + 4 > type_len)
		    return False;
		memcpy_lower(type_buf, c, len);
		type_buf[len] = '\0';
		c = p;
		state = LookingForSlash;
		break;
	    case LookingForSlash:
		if (*c++ != '/')
		    return False;
		state = LookingForSubtype;
		break;
	    case LookingForSubtype:
		len = get_token(&c, &p);
		if (len < 0 || len + 4 > subtype_len)
		    return False;
		memcpy_lower(subtype_buf, c, len);
		subtype_buf[len] = '\0';
		c = p;
		state = LookingForSemicolon;
		break;
	    case LookingForSemicolon:
		if (*c++ != ';')
		    return True; /* lenience */
		state = LookingForParam;
		break;
	    case LookingForParam:
		if (n >= n_args)
		    return True;
		len = get_token(&c, &p);
		if (len < 0)
		    return True; /* lenience */
		args[n].name = XtMalloc(len + 1);
		memcpy_lower(args[n].name, c, len);
		args[n].name[len] = '\0';
		c = p;
		state = LookingForEqual;
		break;
	    case LookingForEqual:
		if (*c++ != '=')
		    return True; /* lenience */
		state = LookingForValue;
		break;
	    case LookingForValue:
		len = get_token(&c, &p);
		if (len < 0)
		    return True; /* lenience */
		args[n].value = XtMalloc(len + 1);
		memcpy(args[n].value, c, len);
		args[n++].value[len] = '\0';
		c = p;
		state = LookingForSemicolon;
		break;
	    }
	}

	c = *++headers;
    } while (c && IS_SPACE(*c));

    return (state != LookingForType &&
	    state != LookingForSlash &&
	    state != LookingForSubtype);
}

char *get_charset(MimeArg *args)
{
    while (args->value)
	if (strcmp(args->name, "charset") == 0)
	    return args->value;
	else
	    args++;

    return NULL;
}

char *next_enc_word(char *header, EncWordData *data)
{
    for (header = strstr(header, "=?") ; header ;
	 header = strstr(header + 1, "=?")) {
	char		*c1, *c2;
	MimeFont	*f;

	if ((c1 = strchr(header + 2, '?')) &&
	    (c1[1] == 'q' || c1[1] == 'Q' || c1[1] == 'B' || c1[1] == 'b') &&
	    c1[2] == '?' && (c2 = strstr(c1 + 3, "?="))) {
	    *c1 = '\0';
	    f = get_font(header + 2);
	    *c1 = '?';
	    if (f && f->header_font && (!f->funcs || f->head_enc_hack)) {
		data->word  = c1 + 3;
		data->end   = c2 + 2;
		data->len   = c2 - c1 - 3;
		data->is_qp = (c1[1] == 'Q' || c1[1] == 'q');
		data->font  = f->header_font;
		return header;
	    }
	}
    }

    return NULL;
}
