/*
 * arg.c - common argument processing support functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: arg.c,v 1.20 98/09/14 13:47:24 abe Exp $";
#endif


#include "lsof.h"


_PROTOTYPE(static int ckfd_range,(char *first, char *dash, char *last, int *lo, int *hi));
_PROTOTYPE(static void enter_fd_lst,(char *nm, int lo, int hi));
_PROTOTYPE(static int enter_nwad,(struct nwad *n, int sp, int ep, char *s, struct hostent *he));


/*
 * ckfd_range() - check fd range
 */

static int
ckfd_range(first, dash, last, lo, hi)
	char *first;			/* starting character */
	char *dash;			/* '-' location */
	char *last;			/* '\0' location */
	int *lo;			/* returned low value */
	int *hi;			/* returned high value */
{
	char *cp;
/*
 * See if the range character pointers make sense.
 */
	if (first >= dash || dash >= last) {
	    (void) fprintf(stderr, "%s: illegal FD range for -d: ", Pn);
	    safestrprt(first, stderr, 1);
	    return(1);
	}
/*
 * Assemble and check the high and low values.
 */
	for (cp = first, *lo = 0; *cp && cp < dash; cp++) {
	    if (!isdigit((unsigned char)*cp)) {

FD_range_nondigit:

		(void) fprintf(stderr, "%s: non-digit in -d FD range: ", Pn);
		safestrprt(first, stderr, 1);
		return(1);
	    }
	    *lo = (*lo * 10) + (int)(*cp - '0');
	}
	for (cp = dash+1, *hi = 0; *cp && cp < last; cp++) {
	    if (!isdigit((unsigned char)*cp))
		goto FD_range_nondigit;
	    *hi = (*hi * 10) + (int)(*cp - '0');
	}
	if (*lo >= *hi) {
	    (void) fprintf(stderr, "%s: -d FD range's low >= its high: ", Pn);
	    safestrprt(first, stderr, 1);
	    return(1);
	}
	return(0);
}


/*
 * ck_file_arg() - check file arguments
 */

int
ck_file_arg(i, ac, av)
	int i;			/* first file argument index */
	int ac;			/* argument count */
	char *av[];		/* argument vector */
{
	unsigned char ad, an;
	short err = 0;
	int errno_save;
	char *fnm, *fsnm, *path;
	int fsm, ftype, j;
	MALLOC_S l;
	struct mounts *mp;
	static struct mounts **mmp = (struct mounts **)NULL;
	int mx, nm;
	static int nma = 0;
	struct stat sb;
	struct sfile *sfp;

#if	defined(CKFA_EXPDEV)
	dev_t dev, rdev;
#endif	/* defined(CKFA_EXPDEV) */

#if	defined(HASPROCFS)
	int pfsnl = -1;
	pid_t pid;
	struct procfsid *pfi;
#endif	/* defined(HASPROCFS) */

/*
 * Loop through arguments.
 */
	for (; i < ac; i++) {
	    if (!(path = Readlink(av[i]))) {
		err = 1;
		continue;
	    }
	/*
	 * Remove a terminating `/' from Readlink()-expanded path, but never
	 * from an argument-specified path.  Don't remove `/' from a single
	 * character path name, either.
	 *
	 * Check for file system argument.
	 */
	    if (path != av[i] && (j = strlen(path)) > 1 && path[j-1] == '/')
		path[j-1] = '\0';
	    for (ftype = 1, mp = readmnt(), nm = 0;
		 (Ffilesys != 1) && mp;
		 mp = mp->next)
	    {
		fsm = 0;
		if (strcmp(mp->dir, path) == 0)
		    fsm++;
		else if (Ffilesys == 2 || (mp->fs_mode & S_IFMT) == S_IFBLK) {
		    if (mp->fsnmres && strcmp(mp->fsnmres, path) == 0)
			fsm++;
		}
		if (!fsm)
		    continue;
		ftype = 0;
	    /*
	     * Skip duplicates.
	     */
		for (mx = 0; mx < nm; mx++) {
		    if (strcmp(mp->dir, mmp[mx]->dir) == 0
		    &&  mp->dev == mmp[mx]->dev
		    &&  mp->inode == mmp[mx]->inode)
			break;
		}
		if (mx < nm)
		    continue;
	    /*
	     * Allocate space for and save another mount point match and
	     * the type of match -- directory name (mounted) or file system
	     * name (mounted-on).
	     */
		if (nm >= nma) {
		    nma += 5;
		    l = (MALLOC_S)(nma * sizeof(struct mounts *));
		    if (mmp)
			mmp = (struct mounts **)realloc((MALLOC_P *)mmp, l);
		    else
			mmp = (struct mounts **)malloc(l);
		    if (!mmp) {
			(void) fprintf(stderr,
			    "%s: no space for mount pointers\n", Pn);
			Exit(1);
		    }
		}
		mmp[nm++] = mp;
	    }
	    if (Ffilesys == 2 && nm == 0) {
		(void) fprintf(stderr, "%s: not a file system: ", Pn);
		safestrprt(av[i], stderr, 1);
		err = 1;
		continue;
	    }
	/*
	 * Loop through the file system matches.  If there were none, make one
	 * pass through the loop, using simply the path name.
	 */
	    mx = 0;
	    do {

	    /*
	     * Allocate an sfile structure and fill in the type and link.
	     */
		if (!(sfp = (struct sfile *)malloc(sizeof(struct sfile)))) {
		    (void) fprintf(stderr, "%s: no space for files\n", Pn);
		    Exit(1);
		}
		sfp->next = Sfile;
		Sfile = sfp;
		sfp->f = 0;
		if ((sfp->type = ftype)) {

		/*
		 * For a non-file system path, use the path as the file name
		 * and set a NULL file system name.
		 */
		    fnm = path;
		    fsnm = (char *)NULL;
		/*
		 * Stat the path to obtain its characteristics.
		 */
		    if (statsafely(fnm, &sb) != 0) {
			errno_save = errno;
			(void) fprintf(stderr, "%s: status error on ", Pn);
			safestrprt(fnm, stderr, 0);
			(void) fprintf(stderr, ": %s\n", strerror(errno_save));
			Sfile = sfp->next;
			(void) free((FREE_P *)sfp);
			err = 1;
			break;
		    }
		    sfp->i = sb.st_ino;
		    sfp->mode = sb.st_mode & S_IFMT;
		
#if	defined(CKFA_EXPDEV)
		/*
		 * Expand device numbers before saving, so that they match the
		 * already-expanded local mount info table device numbers.
		 * (This is an EP/IX 2.1.1 and above artifact.)
		 */
		    dev = expdev(sb.st_dev);
		    rdev = expdev(sb.st_rdev);
#endif	/* defined(CKFA_EXPDEV) */

		    if (sfp->mode == S_IFBLK || sfp->mode == S_IFCHR

#if	defined(CKFA_XDEVTST)
		    ||  CKFA_XDEVTST
#endif	/* defined(CKFA_XDEVTST) */

		    )

#if	defined(CKFA_EXPDEV)
			sfp->dev = rdev;
		    else
			sfp->dev = dev;
#else	/* !defined(CKFA_EXPDEV) */
			sfp->dev = sb.st_rdev;
		    else
			sfp->dev = sb.st_dev;
#endif	/* defined(CKFA_EXPDEV) */

#if	defined(CKFA_MPXCHAN)
		/*
		 * Save a (possible) multiplexed channel number.  (This is an
		 * AIX artifact.)
		 */
		    sfp->ch = getchan(path);
#endif	/* defined(CKFA_MPXCHAN) */

		} else {
		    mp = mmp[mx++];

#if	defined(HASPROCFS)
		/*
		 * If this is a /proc file system, set the search flag and
		 * abandon the sfile entry.
		 */
		    if (mp == Mtprocfs) {
			Sfile = sfp->next;
			(void) free((FREE_P *)sfp);
			Procsrch = 1;
			continue;
		    }
#endif	/* defined(HASPROCFS) */

		/*
		 * Derive file name and file system name for a mount point.
		 *
		 * Save the device number, inode number, and modes.
		 */
		    fnm = mp->dir;
		    fsnm = mp->fsname;
		    sfp->dev = mp->dev;
		    sfp->i = mp->inode;
		    sfp->mode = mp->mode & S_IFMT;
		}
	    /*
	     * Store the file name and file system name pointers in the sfile
	     * structure, allocating space as necessary.
	     */
		if (!fnm || fnm == path) {
		    sfp->name = fnm;
		    an = 0;
		} else {
		    if (!(sfp->name = mkstrcpy(fnm, (MALLOC_S *)NULL))) {
			(void) fprintf(stderr,
			    "%s: no space for file name: ", Pn);
			safestrprt(fnm, stderr, 1);
			Exit(1);
		    }
		    an = 1;
		}
		if (!fsnm || fsnm == path) {
		    sfp->devnm = fsnm;
		    ad = 0;
		} else {
		    if (!(sfp->devnm = mkstrcpy(fsnm, (MALLOC_S *)NULL))) {
			(void) fprintf(stderr,
			    "%s: no space for file system name: ", Pn);
			safestrprt(fsnm, stderr, 1);
			Exit(1);
		    }
		    ad = 1;
		}
		if (!(sfp->aname = mkstrcpy(av[i], (MALLOC_S *)NULL))) {
		    (void) fprintf(stderr,
			"%s: no space for argument file name: ", Pn);
			safestrprt(av[i], stderr, 1);
		    Exit(1);
		}

#if	defined(HASPROCFS)
	    /*
	     * See if this is an individual member of a proc file system.
	     */
		if (!Mtprocfs || Procsrch)
		    continue;

# if	defined(HASFSTYPE)
		if (strcmp(sb.st_fstype, HASPROCFS) != 0)
		    continue;
# endif	/* defined(HASFSTYPE) */

		if (pfsnl == -1)
		    pfsnl = strlen(Mtprocfs->dir);
		if (!pfsnl)
		    continue;
		if (strncmp(Mtprocfs->dir, path, pfsnl) != 0)
		    continue;
		if (path[pfsnl] != '/')

# if	defined(HASPINODEN)
		    pid = 0;
# else	/* !defined(HASPINODEN) */
		    continue;
# endif	/* defined(HASPINODEN) */

		else {
		    for (j = pfsnl+1; path[j]; j++) {
			if (!isdigit((unsigned char)path[j]))
			    break;
		    }
		    if (path[j] || (j - pfsnl - 1) < 1
		    ||  (sfp->mode & S_IFMT) != S_IFREG)

# if	defined(HASPINODEN)
			pid = 0;
# else	/* !defined(HASPINODEN) */
			continue;
# endif	/* defined(HASPINODEN) */

		    else
			pid = atoi(&path[pfsnl+1]);
		}
		if (!(pfi = (struct procfsid *)malloc((MALLOC_S)
			     sizeof(struct procfsid))))
		{
		    (void) fprintf(stderr, "%s: no space for %s ID: ",
			Pn, Mtprocfs->dir);
		    safestrprt(path, stderr, 1);
		    Exit(1);
		}
		pfi->pid = pid;
		pfi->f = 0;
		pfi->nm = sfp->aname;
		pfi->next = Procfsid;
		Procfsid = pfi;

# if	defined(HASPINODEN)
		pfi->inode = (unsigned long)sfp->i;
# endif	/* defined(HASPINODEN) */

	    /*
	     * Abandon the Sfile entry, lest it be used in is_file_named().
	     */
		Sfile = sfp->next;
		if (ad)
		    (void) free((FREE_P *)sfp->devnm);
		if (an)
		    (void) free((FREE_P *)sfp->name);
		(void) free((FREE_P *)sfp);
#endif	/* defined(HASPROCFS) */

	    } while (mx < nm);
	}
	return((int)err);
}


#if	defined(HASDCACHE)
/*
 * ctrl_dcache() - enter device cache control
 */

int
ctrl_dcache(c)
	char *c;			/* control string */
{
	int rc = 0;
	
	if (!c) {
	    (void) fprintf(stderr,
		"%s: no device cache option control string\n", Pn);
	    return(1);
	}
/*
 * Decode argument function character.
 */
	switch (*c) {
	case '?':
	    if (*(c+1) != '\0') {
		(void) fprintf(stderr, "%s: nothing should follow -D?\n", Pn);
		return(1);
	    }
	    DChelp = 1;
	    return(0);
	case 'b':
	case 'B':
	    if (Setuidroot

#if	!defined(WILLDROPGID)
	    ||  Myuid
#endif	/* !defined(WILLDROPGID) */

	    )
		rc = 1;
	    else
		DCstate = 1;
	    break;
	case 'r':
	case 'R':
	    if (Setuidroot && *(c+1))
		rc = 1;
	    else
		DCstate = 2;
	    break;
	case 'u':
	case 'U':
	    if (Setuidroot

#if	!defined(WILLDROPGID)
	    ||  Myuid
#endif	/* !defined(WILLDROPGID) */

	    )
		rc = 1;
	    else
		DCstate = 3;
	    break;
	case 'i':
	case 'I':
	    if (*(c+1) == '\0') {
		DCstate = 0;
		return(0);
	    }
	    /* fall through */
	default:
	    (void) fprintf(stderr, "%s: unknown -D option: ", Pn);
	    safestrprt(c, stderr, 1);
	    return(1);
	}
	if (rc) {
	    (void) fprintf(stderr, "%s: -D option restricted to root: ", Pn);
	    safestrprt(c, stderr, 1);
	    return(1);
	}
/*
 * Skip to optional path name and save it.
 */
	for (c++; *c && (*c == ' ' || *c == '\t'); c++)
	    ;
	if (strlen(c)) {
	    if (!(DCpathArg = mkstrcpy(c, (MALLOC_S)NULL))) {
		(void) fprintf(stderr, "%s: no space for -D path: ", Pn);
		safestrprt(c, stderr, 1);
		Exit(1);
	    }
	}
	return(0);
}
#endif	/* defined(HASDCACHE) */


/*
 * enter_fd() - enter file descriptor list for searching
 */

int
enter_fd(f)
	char *f;			/* file descriptor list pointer */
{
	char buf[32], c, *cp1, *cp2, *dash;
	int err, hi, lo;
	char *fc;
/*
 *  Check for non-empty list and make a copy.
 */
	if (!f || (strlen(f) + 1) < 2) {
	    (void) fprintf(stderr, "%s: no file descriptor specified\n", Pn);
	    return(1);
	}
	if (!(fc = mkstrcpy(f, (MALLOC_S *)NULL))) {
	    (void) fprintf(stderr, "%s: no space for fd string: ", Pn);
	    safestrprt(f, stderr, 1);
	    Exit(1);
	}
/*
 * Isolate each file descriptor in the comma-separated list, then enter it
 * in the file descriptor string list.  If a descriptor has the form:
 *
 *	[0-9]+-[0-9]+
 *
 * treat it as an ascending range of file descriptor numbers.
 */
	for (cp1 = fc, err = 0; *cp1;) {
	    for (cp2 = cp1, dash = (char *)NULL; *cp2 && *cp2 != ','; cp2++) {
		if (*cp2 == '-')
		    dash = cp2;
	    }
	    if ((c = *cp2) != '\0')
		*cp2 = '\0';
	    if (cp2 > cp1) {
		if (dash) {
		    if (ckfd_range(cp1, dash, cp2, &lo, &hi))
			err = 1;
		    else
			(void) enter_fd_lst((char *)NULL, lo, hi);
		} else
		    (void) enter_fd_lst(cp1, 0, 0);
	    }
	    if (c == '\0')
		break;
	    cp1 = cp2 + 1;
	}
	(void) free((FREE_P *)fc);
	return(err);
}


/*
 * enter_fd_lst() - make an entry in the FD list, Fdl
 */

static void
enter_fd_lst(nm, lo, hi)
	char *nm;			/* FD name (none if NULL) */
	int lo;				/* FD low boundary (if nm NULL) */
	int hi;				/* FD high boundary (if nm NULL) */
{
	char *cp;
	struct fd_lst *f;
	int n;
/*
 * Allocate an fd_lst entry.
 */
	if (!(f = (struct fd_lst *)malloc((MALLOC_S)sizeof(struct fd_lst)))) {
	   (void) fprintf(stderr, "%s: no space for FD list entry\n", Pn);
	   Exit(1);
	}
	if (nm) {

	/*
	 * Process an FD name.  First see if it contains only digits; if it
	 * does, convert them to an integer and set the low and high
	 * boundaries to the result.
	 *
	 * If the name has a non-digit, store it as a string, and set the
	 * boundaries to impossible values (i.e., low > high).
	 */
	    for (cp = nm, n = 0; *cp; cp++) {
		if (!isdigit((unsigned char)*cp))
		    break;
		n = (n * 10) + (int)(*cp - '0');
	    }
	    if (*cp) {
		if (!(f->nm = mkstrcpy(nm, (MALLOC_S *)NULL))) {
		    (void) fprintf(stderr,
			"%s: no space for copy of: %s\n", Pn, nm);
		    Exit(1);
		}
		lo = 1;
		hi = 0;
	    } else {
		f->nm = (char *)NULL;
		lo = hi = n;
	    }
	} else
	    f->nm = (char *)NULL;
	f->hi = hi;
	f->lo = lo;
	f->next = Fdl;
	Fdl = f;
}


/*
 * enter_id() - enter PGRP or PID for searching
 */

int
enter_id(ty, p)
	enum IDType ty;			/* type: PGRP or PID */
	char *p;			/* process group ID string pointer */
{
	char *cp;
	int i, id, mx, n;
	struct int_lst *s;

	if (!p) {
	    (void) fprintf(stderr, "%s: no process%s ID specified\n",
		Pn, (ty == PGRP) ? " group" : "");
	    return(1);
	}
/*
 * Set up variables for the type of ID.
 */
	switch (ty) {
	case PGRP:
	    mx = Mxpgrp;
	    n = Npgrp;
	    s = Spgrp;
	    break;
	case PID:
	    mx = Mxpid;
	    n = Npid;
	    s = Spid;
	    break;
	default:
	    (void) fprintf(stderr, "%s: enter_id \"", Pn);
	    safestrprt(p, stderr, 0);
	    (void) fprintf(stderr, "\", invalid type: %d\n", ty);
	    Exit(1);
	}
/*
 * Convert and store the ID.
 */
	for (cp = p; *cp;) {

	/*
	 * Assemble ID.
	 */
	    for (id = 0; *cp && *cp != ','; cp++) {

#if	defined(__STDC__)
		if (!isdigit((unsigned char)*cp))
#else
		if (!isascii(*cp) || ! isdigit((unsigned char)*cp))
#endif	/* __STDC__ */

		{
		    (void) fprintf(stderr, "%s: illegal process%s ID: ",
			Pn, (ty == PGRP) ? " group" : "");
		    safestrprt(p, stderr, 1);
		    return(1);
		}
		id = (id * 10) + *cp - '0';
	    }
	    if (*cp)
		cp++;
	/*
	 * Avoid entering duplicates.
	 */
	    for (i = 0; i < n; i++) {
		if (id == s[i].i)
		    break;
	    }
	    if (i < n)
		continue;
	/*
	 * Allocate table table space.
	 */
	    if (n >= mx) {
		mx += IDINCR;
		if (!s)
		    s = (struct int_lst *)malloc(
			(MALLOC_S)(sizeof(struct int_lst) * mx));
		else
		    s = (struct int_lst *)realloc((MALLOC_P *)s,
			(MALLOC_S)(sizeof(struct int_lst) * mx));
		if (!s) {
		    (void) fprintf(stderr, "%s: no space for %d process%s IDs",
			Pn, mx, (ty == PGRP) ? " group" : "");
		    Exit(1);
		}
	    }
	    s[n].f = 0;
	    s[n++].i = id;
	}
/*
 * Save variables for the type of ID.
 */
	if (ty == PGRP) {
	    Mxpgrp = mx;
	    Npgrp = n;
	    Spgrp = s;
	} else {
	    Mxpid = mx;
	    Npid = Npuns = n;
	    Spid = s;
	}
	return(0);
}


/*
 * enter_network_address() - enter Internet address for searching
 */

int
enter_network_address(na)
	char *na;			/* Internet address string pointer */
{
	int ae, ep, i, pr, sp;
	unsigned char *ap;
	char *cp, *p, *wa;
	struct hostent *he = (struct hostent *)NULL;
	char *hn = (char *)NULL;
	MALLOC_S l;
	struct nwad n;
	int pt = 0;
	int pu = 0;
	struct servent *se, *se1;
	char *sn = (char *)NULL;
	MALLOC_S snl = 0;

	if (!na) {
	    (void) fprintf(stderr, "%s: no network address specified\n", Pn);
	    return(1);
	}
/*
 * Initialize network address structure.
 */
	n.proto = (char *)NULL;
	n.af = 0;
	for (i = 0; i < MAX_AF_ADDR; i++) {
	    n.a[i] = 0;
	}
/*
 * Process protocol name, optionally followed by a '@' and a host name or
 * Internet address, or a ':' and a service name or port number.
 */
	wa = na;
	if (*wa && *wa != '@' && *wa != ':') {
	    for (p = wa; *wa && *wa != '@' && *wa != ':'; wa++)
		;
	    if ((l = wa - p)) {
		if (!(n.proto = mkstrcat(p, l, (char *)NULL, -1, (char *)NULL,
			        -1, (MALLOC_S *)NULL)))
		{
		    (void) fprintf(stderr,
			"%s: no space for protocol name from ", Pn);
		    safestrprt(na, stderr, 1);
nwad_exit:
		    if (n.proto)
			(void) free((FREE_P *)n.proto);
		    if (hn)
			(void) free((FREE_P *)hn);
		    if (sn)
			(void) free((FREE_P *)sn);
		    return(1);
		}
	    /*
	     * The protocol name should be "tcp" or "udp".
	     */
		if (strcasecmp(n.proto, "tcp") != 0
		&&  strcasecmp(n.proto, "udp") != 0)
		{
		    (void) fprintf(stderr,
			"%s: unknown protocol name (%s) in ", Pn, n.proto);
		    safestrprt(na, stderr, 1);
		    goto nwad_exit;
		}
	    /*
	     * Convert protocol name to lower case.
	     */
		for (p = n.proto; *p; p++) {
		    if (*p >= 'A' && *p <= 'Z')
			*p = *p - 'A' + 'a';
		}
	    }
	}
/*
 * Process an IPv4 address (1.2.3.4), IPv6 address ([1:2:3:4:5:6:7:8]),
 * or host name, preceded by a '@' and optionally followed by a colon
 * and a service name or port number.
 */
	if (*wa == '@') {
	    wa++;
	    if (!*wa || *wa == ':') {

unacc_address:
		(void) fprintf(stderr,
		    "%s: unacceptable Internet address in ", Pn);
		safestrprt(na, stderr, 1);
		goto nwad_exit;
	    }
	    if (*wa < '0' || *wa > '9' && *wa != '[') {

	    /*
	     * Assemble host name.
	     */
		for (p = wa; *p && *p != ':'; p++)
		    ;
		if ((l = p - wa)) {
		    if (!(hn = mkstrcat(wa, l, (char *)NULL, -1, (char *)NULL,
			       -1, (MALLOC_S *)NULL)))
		    {
			(void) fprintf(stderr,
			    "%s: no space for host name: ", Pn);
			safestrprt(na, stderr, 1);
			goto nwad_exit;
		    }

#if	defined(HASIPv6)
		    if (!(he = gethostbyname2(hn, AF_INET))
		    ||  he->h_addrtype != AF_INET) {
			if (!(he = gethostbyname2(hn, AF_INET6))
			||  he->h_addrtype != AF_INET6)
			    he = (struct hostent *)NULL;
		    }
		    if (!he)
#else	/* !defined(HASIPv6) */
		    if (!(he = gethostbyname(hn)))
#endif	/* defined(HASIPv6) */

		    {
			fprintf(stderr, "%s: unknown host name (%s) in ",
			    Pn, hn);
			safestrprt(na, stderr, 1);
			goto nwad_exit;
		    }
		    ap = (unsigned char *)he->h_addr;
		    n.af = he->h_addrtype;

#if	defined(HASIPv6)
		    for (i = 0;
			 i < (he->h_length - 1) && i < (MAX_AF_ADDR - 1);
			 i++)
		    {
			 n.a[i] = *ap++;
		    }
		    n.a[i] = *ap;
#else	/* !defined(HASIPv6) */
		    n.a[0] = *ap++;
		    n.a[1] = *ap++;
		    n.a[2] = *ap++;
		    n.a[3] = *ap;
#endif	/* defined(HASIPv6) */

		}
		wa = p;
	    } else if (*wa == '[') {

#if	defined(HASIPv6)
	    /*
	     * Assemble IPv6 address.
	     */
		if (!(cp = strrchr(++wa, ']')))
		    goto unacc_address;
		*cp = '\0';
		i = inet_pton(AF_INET6, wa, (void *)&n.a);
		*cp = ']';
		if (i != 1)
		    goto unacc_address;
		for (ae = i = 0; i < MAX_AF_ADDR; i++) {
		    if ((ae |= n.a[i]))
			break;
		}
		if (!ae)
		    goto unacc_address;
		n.af = AF_INET6;
		wa = cp + 1;
#else	/* !defined(HASIPv6) */
		(void) fprintf(stderr,
		    "%s: unsupported IPv6 address in ", Pn);
		safestrprt(na, stderr, 1);
		goto nwad_exit;
#endif


	    } else {

	    /*
	     * Assemble IPv4 address.
	     */
		for (i = 0; *wa; wa++) {
		    if (*wa == ':')
			break;
		    if (*wa == '.') {
			i++;
			if (i >= MIN_AF_ADDR)
			    break;
			continue;
		    }
		    if (*wa < '0' || *wa > '9')
			goto unacc_address;
		    ae = (10 * n.a[i]) + *wa - '0';
		    if (ae > 255)
			goto unacc_address;
		    n.a[i] = ae;
		}
		if (i != (MIN_AF_ADDR - 1)
		||  (!n.a[0] && !n.a[1] && !n.a[2] && !n.a[3]))
		    goto unacc_address;
		n.af = AF_INET;
	    }
	}
/*
 * If there is no port number, enter the address.
 */
	if (!*wa) {
	    if (enter_nwad(&n, -1, -1, na, he))
		goto nwad_exit;
	    if (sn)
		(void) free((FREE_P *)sn);
	    return(0);
	}
/*
 * Process a service name or port number list, preceded by a colon.
 *
 * Entries of the list are separated with commas; elements of a numeric range
 * are specified with a separating minus sign (`-'); all service names must
 * belong to the same protocol; embedded spaces are not allowed.  An embedded
 * minus sign in a name is taken to be part of the name, the starting entry
 * of a range can't be a service name.
 */
	if (*wa != ':' || *(wa + 1) == '\0') {

unacc_port:
	    (void) fprintf(stderr,
		"%s: unacceptable port specification in ", Pn);
	    safestrprt(na, stderr, 1);
	    goto nwad_exit;
	}
	for (++wa; wa && *wa; wa++) {
	    for (ep = pr = sp = 0; *wa; wa++) {
		if (*wa < '0' || *wa > '9') {

		/*
		 * Convert service name to port number, using already-specified
		 * protocol name.  A '-' is taken to be part of the name; hence
		 * the staring entry of a range can't be a service name.
		 */
		    for (p = wa; *wa && *wa != ','; wa++)
			;
		    if (!(l = wa - p)) {
			(void) fprintf(stderr,
			    "%s: invalid service name: ", Pn);
			safestrprt(na, stderr, 1);
			goto nwad_exit;
		    }
		    if (sn) {
			if (l > snl) {
			    sn = (char *)realloc((MALLOC_P *)sn, l + 1);
			    snl = l;
			}
		    } else {
			sn = (char *)malloc(l + 1);
			snl = l;
		    }
		    if (!sn) {
			(void) fprintf(stderr,
			    "%s: no space for service name: ", Pn);
			safestrprt(na, stderr, 1);
			goto nwad_exit;
		    }
		    (void) strncpy(sn, p, l);
		    *(sn + l) = '\0';
		    if (n.proto) {

		    /*
		     * If the protocol has been specified, look up the port
		     * number for the service name for the specified protocol.
		     */
			if (!(se = getservbyname(sn, n.proto))) {
			    (void) fprintf(stderr,
				"%s: unknown service %s for %s in ",
				Pn, sn, n.proto);
			    safestrprt(na, stderr, 1);
			    goto nwad_exit;
			}
			pt = (int)ntohs(se->s_port);
		    } else {

		    /*
		     * If no protocol has been specified, look up the port
		     * numbers for the service name for both TCP and UDP.
		     */
			if((se = getservbyname(sn, "tcp")))
			    pt = (int)ntohs(se->s_port);
			if ((se1 = getservbyname(sn, "udp")))
			    pu = (int)ntohs(se1->s_port);
			if (!se && !se1) {
			    (void) fprintf(stderr,
				"%s: unknown service %s in ", Pn, sn);
			    safestrprt(na, stderr, 1);
			    goto nwad_exit;
			}
			if (se && se1 && pt != pu) {
			    (void) fprintf(stderr,
				"%s: TCP=%d and UDP=%d %s ports conflict;\n",
				Pn, pt, pu, sn);
			    (void) fprintf(stderr,
				"      specify \"tcp:%s\" or \"udp:%s\".\n",
				sn, sn);
			    goto nwad_exit;
			}
			if (!se && se1)
			    pt = pu;
		    }
		    if (pr)
			ep = pt;
		    else {
			sp = pt;
			if (*wa == '-')
			    pr++;
		    }
		} else {

		/*
		 * Assemble port number.
		 */
		    for (; *wa && *wa != ','; wa++) {
			if (*wa == '-') {
			    if (pr)
				goto unacc_port;
			    pr++;
			    break;
			}
			if (*wa < '0' || *wa > '9')
			    goto unacc_port;
			if (pr)
			    ep = (ep * 10) + *wa - '0';
			else
			    sp = (sp * 10) + *wa - '0';
		    }
		}
		if (!*wa || *wa == ',')
		    break;
		if (pr)
		    continue;
		goto unacc_port;
	    }
	    if (!pr)
		ep = sp;
	    if (ep < sp)
		goto unacc_port;
	/*
	 * Enter completed port or port range specification.
	 */
	    if (enter_nwad(&n, sp, ep, na, he))
		goto nwad_exit;
	    if (!*wa)
		break;
	}
	if (sn)
	    (void) free((FREE_P *)sn);
	return(0);
}

/*
 * enter_nwad() - enter nwad structure
 */

static int
enter_nwad(n, sp, ep, s, he)
	struct nwad *n;			/* pointer to partially completed
					 * nwad (less port) */
	int sp;				/* starting port number */
	int ep;				/* ending port number */
	char *s;			/* string that states the address */
	struct hostent *he;		/* pointer to hostent struct from which
					 * network address came */
{
	int ac;
	unsigned char *ap;
	static int na = 0;
	struct nwad nc;
	struct nwad *np;
/*
 * Allocate space for the argument specification.
 */
	if (strlen(s)) {
	    if (!(n->arg = mkstrcpy(s, (MALLOC_S *)NULL))) {
		(void) fprintf(stderr,
		    "%s: no space for Internet argument: ", Pn);
		safestrprt(s, stderr, 1);
		Exit(1);
	    }
	} else
	    n->arg = (char *)NULL;
/*
 * Loop through all hostent addresses.
 */
	for (ac = 1, nc = *n;;) {

	/*
	 * Test address specification -- it must contain at least one of:
	 * protocol, Internet address or port.  If correct, link into search
	 * list.
	 */
	    if (!nc.proto
	    &&  !nc.a[0] && !nc.a[1] && !nc.a[2] && !nc.a[3]

#if	defined(HASIPv6)
	    &&  (nc.af != AF_INET6
	    ||   (!nc.a[4]  && !nc.a[5]  && !nc.a[6]  && !nc.a[7]
	    &&    !nc.a[8]  && !nc.a[9]  && !nc.a[10] && !nc.a[11]
	    &&    !nc.a[12] && !nc.a[13] && !nc.a[14] && !nc.a[15]))
#endif	/* defined(HASIPv6) */

	    &&  sp == -1) {
		(void) fprintf(stderr,
		    "%s: incomplete Internet address specification: ", Pn);
		safestrprt(s, stderr, 1);
		return(1);
	    }
	/*
	 * Limit the network address chain length to MAXNWAD for reasons of
	 * search efficiency.
	 */
	    if (na >= MAXNWAD) {
		(void) fprintf(stderr,
		    "%s: network address limit (%d) exceeded: ",
		    Pn, MAXNWAD);
		safestrprt(s, stderr, 1);
		return(1);
	    }
	/*
	 * Allocate space for the address specification.
	 */
	    if ((np = (struct nwad *)malloc(sizeof(struct nwad))) == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for network address from: ", Pn);
		safestrprt(s, stderr, 1);
		return(1);
	    }
	/*
	 * Construct and link the address specification.
	 */
	    *np = nc;
	    np->sport = sp;
	    np->eport = ep;
	    np->f = 0;
	    np->next = Nwad;
	    Nwad = np;
	    na++;
	/*
	 * If the network address came from gethostbyname(), advance to
	 * the next address; otherwise quit.
	 */
	    if (!he)
		break;
	    if (!(ap = (unsigned char *)he->h_addr_list[ac++]))
		break;

#if	defined(HASIPv6)
	    {
		int i;

		for (i = 0;
		     (i < (he->h_length - 1)) && (i < (MAX_AF_ADDR - 1));
		     i++)
		{
		    nc.a[i] = *ap++;
		}
		nc.a[i] = *ap;
	    }
#else	/* !defined(HASIPv6) */
	    nc.a[0] = *ap++;
	    nc.a[1] = *ap++;
	    nc.a[2] = *ap++;
	    nc.a[3] = *ap;
#endif	/* defined(HASIPv6) */

	}
	return(0);
}


/*
 * enter_str_lst() - enter a string on a list
 */

int
enter_str_lst(opt, s, lp)
	char *opt;			/* option name */
	char *s;			/* string to enter */
	struct str_lst **lp;		/* string's list */
{
	char *cp;
	MALLOC_S len;
	struct str_lst *lpt;

	if (!s || *s == '-') {
	    (void) fprintf(stderr, "%s: missing %s option value\n",
		Pn, opt);
	    return(1);
	}
	if (!(cp = mkstrcpy(s, &len))) {
	    (void) fprintf(stderr, "%s: no string copy space: ", Pn);
	    safestrprt(s, stderr, 1);
	    return(1);
	}
	if ((lpt = (struct str_lst *)malloc(sizeof(struct str_lst))) == NULL) {
	    (void) fprintf(stderr, "%s: no list space: ", Pn);
	    safestrprt(s, stderr, 1);
	    (void) free((FREE_P *)cp);
	    return(1);
	}
	lpt->f = 0;
	lpt->str = cp;
	lpt->len = (int)len;
	lpt->next = *lp;
	*lp = lpt;
	return(0);
}


/*
 * enter_uid() - enter User Identifier for searching
 */

int
enter_uid(u)
	char *u;			/* User IDentifier string pointer */
{
	int err, excl, i, j, lnml, nn;
	MALLOC_S len;
	char lnm[LOGINML], *lp;
	struct passwd *pw;
	char *s;
	uid_t uid;

	if (!u) {
	    (void) fprintf(stderr, "%s: no UIDs specified\n", Pn);
	    return(1);
	}
	for (err = 0, s = u; *s;) {

	/*
	 * Assemble next User IDentifier.
	 */
	    for (excl = i = j = lnml = nn = uid = 0; *s && *s != ','; i++, s++)
	    {
		if (j >= LOGINML-1) {
		    (void) fprintf(stderr, "%s: illegal UID in ", Pn);
		    safestrprt(u, stderr, 1);
		    return(1);
		}
		if (i == 0 && *s == '^') {
		    excl = 1;
		    continue;
		}
		lnm[lnml++] = *s;
		if (nn)
		    continue;

#if	defined(__STDC__)
		if (isdigit((unsigned char)*s))
#else	/* !defined(__STDC__) */
		if (isascii(*s) && isdigit((unsigned char)*s))
#endif	/* defined(__STDC__) */

		    uid = (uid * 10) + *s - '0';
		else
		    nn++;
	    }
	    if (*s)
		s++;
	    if (nn) {
	       lnm[lnml++] = '\0';
		if ((pw = getpwnam(lnm)) == NULL) {
		    (void) fprintf(stderr, "%s: can't get UID for ", Pn);
		    safestrprt(lnm, stderr, 1);
		    err = 1;
		    continue;
		} else
		    uid = pw->pw_uid;
	    }

#if	defined(HASSECURITY)
	/*
	 * If the security mode is enabled, only the root user may list files
	 * belonging to user IDs other than the real user ID of this lsof
	 * process.
	 */
	    if (Myuid && uid != Myuid) {
		(void) fprintf(stderr,
		    "%s: ID %d request rejected because of security mode.\n",
		    Pn, uid);
		err = 1;
		continue;
	    }
#endif	/* HASSECURITY */

	/*
	 * Avoid entering duplicates.
	 */
	    for (i = j = 0; i < Nuid; i++) {
		if (uid != Suid[i].uid)
		    continue;
		if (Suid[i].excl == excl) {
		    j = 1;
		    continue;
		}
		(void) fprintf(stderr,
		    "%s: UID %d has been included and excluded\n",
			Pn, (int)uid);
		err = j = 1;
		break;
	    }
	    if (j)
		continue;
	/*
	 * Allocate space for User IDentifier.
	 */
	    if (Nuid >= Mxuid) {
		Mxuid += UIDINCR;
		len = (MALLOC_S)(Mxuid * sizeof(struct seluid));
		if (!Suid)
		    Suid = (struct seluid *)malloc(len);
		else
		    Suid = (struct seluid *)realloc((MALLOC_P *)Suid, len);
		if (!Suid) {
		    (void) fprintf(stderr, "%s: no space for UIDs", Pn);
		    Exit(1);
		}
	    }
	    if (nn) {
		if (!(lp = mkstrcpy(lnm, (MALLOC_S)NULL))) {
		    (void) fprintf(stderr, "%s: no space for login: ", Pn);
		    safestrprt(lnm, stderr, 1);
		    Exit(1);
		}
		Suid[Nuid].lnm = lp;
	    } else
		Suid[Nuid].lnm = (char *)NULL;
	    Suid[Nuid].uid = uid;
	    Suid[Nuid++].excl = excl;
	    if (excl)
		Nuidexcl++;
	    else
		Nuidincl++;
	}
	return(err);
}
