/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

/* ASCII resource control file I/O routines and local data */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <varargs.h>
#include <string.h>
#include <math.h>
#include "common.h"
#include "version.h"

static char *rcfn; /* pointer to the RC file name string for local routines */

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                        Local Utility routines                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Convert binary password to ASCII printable characters */

static char *passwd2ascii(unsigned char *pwd)
{
	static unsigned char apwd[2*MAXLEN_PASSWD+1];
	unsigned char msh, lsh, *p;
	int i;

	if (*pwd) {
		for (i=0, p=apwd; i<MAXLEN_PASSWD; i++, pwd++) {
			msh = (*pwd & 0xF0) >> 4;
			lsh = *pwd & 0x0F;
			*p++ = lsh + 'A';
			*p++ = msh + 'A';
		}
		*p = 0;
	}
	else *apwd = 0;
	return apwd;
}

/* Convert IP address bytes to ASCII string */

static char *IP2str(unsigned char *ip)
{
	static char IPstr[16];

	sprintf(IPstr, "%u.%u.%u.%u",
			ip[0],ip[1],ip[2],ip[3]);
	return IPstr;
}

/* Convert speed bit-mask to ASCII string */

static char *speed2str(unsigned char speed)
{
	static char sstr[8][8] = {"1200","2400","4800","9600",
							  "19200","38400","57600","115200"};
	static int snum[8] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};
	int i;

	for (i=0; i<8; i++)
		if (speed & snum[i])
			return sstr[i];
	return NULL;
}

/* Convert ASCII password to binary */

static char *ascii2passwd(char *buf)
{
	static unsigned char pwd[MAXLEN_PASSWD+1];
	unsigned char msh, lsh, *p;
	int i = 0;

	if (*buf) {
		for (p=pwd; i<MAXLEN_PASSWD; i++) {
			lsh = (*buf++ - 'A') & 0x0F;
			msh = (*buf++ - 'A') & 0x0F;
			pwd[i] = (msh << 4) | lsh;
		}
	}
	pwd[i] = 0;
	return pwd;
}

/* Convert ASCII speed string to bit-mask */

static unsigned int str2speed(char *buf)
{
	static char sstr[8][8] = {"1200","2400","4800","9600",
							  "19200","38400","57600","115200"};
	static int snum[8] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};
	int i;

	for (i=0; i<8; i++)
		if (strcmp(buf, sstr[i]) == 0)
			return snum[i];
	return 0;
}

/* Convert ASCII string to IP address */

static unsigned char *str2IP(char *buf)
{
	static unsigned char ip[4];
	unsigned int iip[4];
	int i;

	i = sscanf(buf, "%3u.%3u.%3u.%3u", &iip[0],
			   &iip[1], &iip[2], &iip[3]);
	if (i < 4)
		return NULL;
	for (i=0; i<4; i++)
		ip[i] = iip[i];
	return ip;
}

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

static void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

	if (errno < sys_nerr)
		sprintf(emsg, "xISP: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xISP: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	exit(1);
}

/* Function for writing strings in the .xisprc resource control file */

static int rcPrint(va_alist) va_dcl
{
	int bw;
	va_list ap;
	char *fmt;
	FILE *fp;

	va_start(ap);
	fp = va_arg(ap, FILE*);
	fmt = va_arg(ap, char*);
	bw = vfprintf(fp, fmt, ap);
	va_end(ap);
	if (bw <= 0)
		doErr("outputAllXisprc");
	return bw;
}

/* Function for writing out the entire .xisprc file */

static void outputAllXisprc(xisprc_t *p, FILE *rcfp)
{
	int i, n;

	rcPrint(rcfp, "xISP-%s\n", Version);
	for (i=0; i<MAXNUM_ISP; i++, p++) {
		rcPrint(rcfp, "-%d-\n", i);
		rcPrint(rcfp, "DESCR: %s\n", p->descr);
		rcPrint(rcfp, "ACCOUNT: %s\n", p->account);
		if (*(p->account))
			rcPrint(rcfp, "PASSWD: %s\n", passwd2ascii(p->passwd));
		else
			rcPrint(rcfp, "PASSWD: \n");
		rcPrint(rcfp, "NAME: %s\n", p->name);
		rcPrint(rcfp, "REMOTENAME: %s\n", p->rname);
		rcPrint(rcfp, "MAXATTEMPTS: %d\n", p->maxAttempts);
		rcPrint(rcfp, "SLEEPDELAY: %d\n", p->sleepDelay);
		rcPrint(rcfp, "CONNECTWAIT: %d\n", p->connectWait);
		rcPrint(rcfp, "NUMPHONES: %d\n", p->numPhones);
		for (n=0; n<p->numPhones; n++)
			rcPrint(rcfp, " PHONE%d: %s\n", n, p->phone[n]);
		rcPrint(rcfp, "NUMSLINES: %d\n", p->numSlines);
		for (n=0; n<p->numSlines; n++) {
			rcPrint(rcfp, " SLINE%dE: %s\n", n, p->sline[n]);
			rcPrint(rcfp, " SLINE%dS: %s\n", n, p->sline[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "CBDELAY: %d\n", p->CBDelay);
		rcPrint(rcfp, "NUMCBSLINES: %d\n", p->numCBSlns);
		for (n=0; n<p->numCBSlns; n++) {
			rcPrint(rcfp, " CBSLINE%dE: %s\n", n, p->CBsln[n]);
			rcPrint(rcfp, " CBSLINE%dS: %s\n", n, p->CBsln[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "MODEMDEVICE: %s\n", p->modemDevice);
		rcPrint(rcfp, "MODEMSPEED: %s\n", speed2str(p->modemSpeed));
		rcPrint(rcfp, "MODEMRESET: %s\n", p->modemReset);
		rcPrint(rcfp, "MODEMINIT: %s\n", p->modemInit);
		rcPrint(rcfp, "OPEROPTS: %X\n", p->operOpts);
		rcPrint(rcfp, "COMPLEVEL: %d\n", p->compLevel);
		rcPrint(rcfp, "ASYNCMAP: %s\n", p->asyncmap);
		rcPrint(rcfp, "ESCAPE: %s\n", p->escape);
		rcPrint(rcfp, "LOCALIP: %s\n", IP2str(p->localIP));
		rcPrint(rcfp, "REMOTEIP: %s\n", IP2str(p->remoteIP));
		rcPrint(rcfp, "NETMASK: %s\n", IP2str(p->netmask));
		rcPrint(rcfp, "DNS1: %s\n", IP2str(p->dns1));
		rcPrint(rcfp, "DNS2: %s\n", IP2str(p->dns2));
		rcPrint(rcfp, "MTU: %u\n", p->mtu);
		rcPrint(rcfp, "MRU: %u\n", p->mru);
	}
}

/* Function for retrieving the .xisprc version. Note that it
   must be called after fopen() and before all other input */

static char *rcVersion(FILE *fp)
{
	static char version[8] = {0};

	if (fscanf(fp, "xISP-%7[.0-9] ", version) < 1) {
		fputs("xISP: rcVersion: error reading file version!\n", stderr);
		exit(1);
	}
	return version;
}

/* Function for retrieving the ISP sequence number. Note that
   it must be called at the beginning of each ISP record. */

static void rcISP(FILE *fp, int nISP)
{
	int n;

	if (fscanf(fp, "-%3d- ", &n) < 1) {
		fprintf(stderr, "xISP: rcISP: error reading ISP number in %s\n",
				rcfn);
		exit(1);
	}
	if (n != nISP) {
		fprintf(stderr, "xISP: rcISP: ISP sequence number wrong in %s\n",
				rcfn);
		fprintf(stderr, "xISP: rcISP: expected %d, got %d\n", nISP, n);
		exit(1);
	}
}

/* Prints error message if error occurs while
   processing .xisprc. Prints offending line */

static void rcLineError(char *function, char *line)
{
	fprintf(stderr, "xISP: %s: error reading %s\n", function, rcfn);
	fprintf(stderr, "xISP: %s: offending line: [%s]\n", function, line);
	exit(1);
}

/* Reads parameters from .xisprc file, doing error checking. NOTE: it
   assumes a line less than 256 bytes and a parameter less than 128 */

#define MAXLEN_LINE 256
#define MAXLEN_PARAM 128

static char *readParam(FILE *fp, char *name, char type,
					   int llimit, int ulimit, void *data)
{
	static char line[MAXLEN_LINE];
	char sparam[MAXLEN_PARAM] = {0}, *p, *endp;
	int iparam, hex = 0, len;

	if (fgets(line, MAXLEN_LINE, fp) == NULL) {
		fprintf(stderr, "xISP: readParam: %s: premature EOF\n", rcfn);
		exit(1);
	}
	line[strlen(line)-1] = 0;
	if ((p=strchr(line,':')) == NULL)
		rcLineError("readParam", line);
	if (strncmp(name, line, (int)(p-line))) {
		fprintf(stderr, "xISP: readParam: expected [%s], got [%s]\n",
				name, line);
		exit(1);
	}
	for (++p; *p==' '; p++);
	strncpy(sparam, p, sizeof(sparam)-1);
	switch (type) {
		case 'S':
			len = strlen(sparam);
			if (ulimit<len || len<llimit)
				rcLineError("readParam", line);
			strcpy((char *)data, sparam);
			break;

		case 'X':
			hex = 1;
		case 'I':
			iparam = strtol(sparam, &endp, (hex)? 16:10);
			if ((!hex && (ulimit<iparam || iparam<llimit)) ||
				 endp == sparam || *endp )
				rcLineError("readParam", line);
			*((int *)data) = iparam;
			break;

		default: break;
	}
	return(line);
}

/* Function for reading in the entire .xisprc file */

static void inputAllXisprc(xisprc_t *p, FILE *rcfp)
{
	int i, j, n;
	char buf[2*MAXLEN_PASSWD+1], pname[32] = {0}, *line;
	unsigned char *ipp;

	for (i=0; i<MAXNUM_ISP; i++, p++) {
		rcISP(rcfp, i);
		readParam(rcfp, "DESCR", 'S', 0, MAXLEN_DESCR, p->descr);
		readParam(rcfp, "ACCOUNT", 'S', 0, MAXLEN_ACCOUNT, p->account);
		if (! *(p->account))
			n = 0;
		else
			n = 2*MAXLEN_PASSWD;
		readParam(rcfp, "PASSWD", 'S', n, n, buf);
		readParam(rcfp, "NAME", 'S', 0, MAXLEN_UNR, p->name);
		readParam(rcfp, "REMOTENAME", 'S', 0, MAXLEN_UNR, p->rname);
		strcpy(p->passwd, ascii2passwd(buf));
		readParam(rcfp, "MAXATTEMPTS", 'I', 0, 255, &(p->maxAttempts));
		readParam(rcfp, "SLEEPDELAY", 'I', 0, 255, &(p->sleepDelay));
		readParam(rcfp, "CONNECTWAIT", 'I', 0, 255, &(p->connectWait));
		readParam(rcfp, "NUMPHONES", 'I', 0, MAXNUM_TELS, &(p->numPhones));
		for (j=0; j<p->numPhones; j++) {
			sprintf(pname, " PHONE%d", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_PHONE, p->phone[j]);
		}
		readParam(rcfp, "NUMSLINES", 'I', 0, MAXNUM_SLINES, &(p->numSlines));
		for (j=0; j<p->numSlines; j++) {
			sprintf(pname, " SLINE%dE", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE, p->sline[j]);
			sprintf(pname, " SLINE%dS", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE,
					  p->sline[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "CBDELAY", 'I', 0, 255, &(p->CBDelay));
		readParam(rcfp, "NUMCBSLINES", 'I', 0, MAXNUM_SLINES, &(p->numCBSlns));
		for (j=0; j<p->numCBSlns; j++) {
			sprintf(pname, " CBSLINE%dE", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE, p->CBsln[j]);
			sprintf(pname, " CBSLINE%dS", j);
			readParam(rcfp, pname, 'S', 0, MAXLEN_SLINE,
					  p->CBsln[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "MODEMDEVICE", 'S', 0, MAXLEN_DEVICE, p->modemDevice);
		line = readParam(rcfp, "MODEMSPEED", 'S', 4, 6, buf);
		if (! (n=str2speed(buf)))
			rcLineError("inputAllXisprc", line);
		p->modemSpeed = n;
		readParam(rcfp, "MODEMRESET", 'S', 0, MAXLEN_MODEM, p->modemReset);
		readParam(rcfp, "MODEMINIT", 'S', 0, MAXLEN_MODEM, p->modemInit);
		readParam(rcfp, "OPEROPTS", 'X', 0, 0, &(p->operOpts));
		readParam(rcfp, "COMPLEVEL", 'I', 9, 15, &(p->compLevel));
		readParam(rcfp, "ASYNCMAP", 'S', 2, MAXDIG_ASYNCMAP, p->asyncmap);
		readParam(rcfp, "ESCAPE", 'S', 0, MAXLEN_ESCAPE, p->escape);
		line = readParam(rcfp, "LOCALIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->localIP, ipp, 4);
		line = readParam(rcfp, "REMOTEIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->remoteIP, ipp, 4);
		line = readParam(rcfp, "NETMASK", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->netmask, ipp, 4);
		line = readParam(rcfp, "DNS1", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->dns1, ipp, 4);
		line = readParam(rcfp, "DNS2", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->dns2, ipp, 4);
		readParam(rcfp, "MTU", 'I', 48, 1500, &(p->mtu));
		readParam(rcfp, "MRU", 'I', 48, 1500, &(p->mru));
	}
}

/* Initializes the xisprc data structure. */

void initXisprc(xisprc_t *p)
{
	int i;

	p->descr[0] = 0;
	p->account[0] = 0;
	p->passwd[0] = 0;
	p->name[0] = 0;
	p->rname[0] = 0;
	p->maxAttempts = MAXNUM_RETRY;
	p->sleepDelay = MAXSEC_DELAY;
	p->connectWait = MAXSEC_CNWAIT;
	p->numPhones = 0;
	for (i=0; i<MAXNUM_TELS; i++)
		p->phone[i][0] = 0;
	p->numSlines = p->numCBSlns = 0;
	for (i=0; i<(2*MAXNUM_SLINES); i++)
		p->sline[i][0] = p->CBsln[i][0] = 0;
	p->CBDelay = MAXSEC_CBDELAY;
	strcpy(p->modemDevice, MODEM_DEVICE);
	p->modemSpeed = MODEM_SPEED;
	strcpy(p->modemReset, MODEM_RESET);
	strcpy(p->modemInit, MODEM_INIT);
	p->operOpts = OPER_OPTS;
	p->compLevel = COMP_LEVEL;
	strcpy(p->asyncmap, PPPD_HASYNCMAP);
	strcpy(p->escape, PPPD_ESCAPE);
	memcpy(p->localIP, LOCAL_IP, 4);
	memcpy(p->remoteIP, REMOTE_IP, 4);
	memcpy(p->netmask, NETMASK, 4);
	memcpy(p->dns1, DNS, 4);
	memcpy(p->dns2, DNS, 4);
	p->mtu = MTU;
	p->mru = MRU;
}

void RCSq(void)
{
	char *rcsp;
	rcsp = RCSid;
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                           Interface routines                            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Read all xisprc data records from the user file, or create
   a new file if it's the first time this program is run. */

void readAllXisprc(char *rcfname, xisprc_t *rcdata)
{
	FILE *rcfp;
	int i;

	rcfn = rcfname;
	rcfp = fopen(rcfn, "r");
	if (rcfp != NULL) {
		if (strcmp(rcVersion(rcfp), Version)) {
			fprintf(stderr, "xISP: %s is not a v%s file!\n", rcfn, Version);
			fprintf(stderr, "xISP: Please run xisprccv with no arguments.\n");
			exit(1);
		}
		inputAllXisprc(&rcdata[1], rcfp);
		fclose(rcfp);
	}
	else {
		for (i=0; i<(MAXNUM_ISP+1); i++)
			initXisprc(&rcdata[i]);
		rcfp = fopen(rcfn, "w");
		if (rcfp == NULL)
			doErr("readAllXisprc: creating: fopen");
		outputAllXisprc(&rcdata[1], rcfp);
		fclose(rcfp);
	}
}

/* Read the specified xisprc data record from the user file */

void readXisprc(char *rcfname, xisprc_t *rcdata, int rec)
{
	FILE *rcfp;
	xisprc_t tmpOptions[MAXNUM_ISP];

	rcfn = rcfname;
	rcfp = fopen(rcfn, "r");
	if (rcfp != NULL) {
		rcVersion(rcfp);
		inputAllXisprc(tmpOptions, rcfp);
		memcpy(&rcdata[rec], &tmpOptions[rec-1], sizeof(xisprc_t));
		fclose(rcfp);
	}
	else
		doErr("readXisprc: fopen");
	
}

/* Write the specified xisprc data record to the user file. For
   the time being, there is no intelligence for "seeking" into the
   ASCII .xisprc file, so the entire file is re-written each time */

void writeXisprc(char *rcfname, xisprc_t *rcdata, int rec)
{
	FILE *rcfp;

	rcfn = rcfname;
	rcfp = fopen(rcfn, "w");
	if (rcfp != NULL) {
		outputAllXisprc(&rcdata[1], rcfp);
		fclose(rcfp);
	}
	else
		doErr("writeXisprc: fopen");
}

