/*
 * PCMCIA Installation module
 * Luis Francisco Gonzalez
 *
 * The idea is to develop some modules that enable the user to create
 * the PCMCIA configuration.
 * 
 * There will be a special routine that will handle communication with
 * cardmgr so that at any one time the state is known. The rationale
 * is that some people need to access pcmcia drivers to install the
 * system even before the later configuration stages are reached.
 *
 * Files:
 * /etc/pcmcia.conf:        Controller type and options.
 * /etc/pcmcia/config.opts: Allows to exclude irq, I/O ports
 *                          and memory window for probing.
 */

/*
 * LFG FIXME: export check_dir to some utils file.
 */         
#include <stdio.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <newt.h>
#include "dinstall.h"
#include "lang.h"

/*
 * Module variables
 */
static char pcmcia_id[]="$Id: pcmcia.c,v 1.5 1998/06/28 00:03:38 ezanard Exp $";
static char* controller=NULL;
static char* pcic_opts=NULL;
static char* core_opts=NULL;
static char* cardmgr_opts=NULL;
static char* serial_link=NULL;
static char* serial_opts=NULL;
static char* inittab=NULL;
static int use_inittab=0;
static char* cdrom_do_fstab=NULL;
static char* cdrom_do_mount=NULL;
static char* cdrom_fstype=NULL;
static char* cdrom_opts=NULL;
static char* cdrom_mountpt=NULL;
static char** pcmcia_choices=NULL;
static char* pcmcia_values=NULL;
static char pcmcia_configured=0;

/*
 * PCMCIA Check list
 */
#define _WIDTH  (65)   /* Width of the choices menu.     */
#define _HEIGHT (18)   /* Height of the choices menu.    */
#define OPTS (3)       /* Number of choices in the menu. */

/*
 * Function prototypes
 */
extern int check_dir(char*);
void input_parameter(char*, char*, char**);
int activate_pcmcia();
static int _activate_pcmcia();
static int _cardmgr(char);
static void pcmcia_serial_file();
static void pcmcia_cdrom_file();
static void pcmcia_conf_file();
static int write_files();
static int write_lowlevel_files();
static int write_serial_opts();
static int write_cdrom_opts();
/* static int write_network_config(); */
static int select_components();

/*
 * Directories
 */
#ifndef _TESTING_
#define PCMCIA_ETC_DIR       "/target/etc"
#define PCMCIA_CONF_FILE     "/target/etc/pcmcia.conf"
#define PCMCIA_CONF_DIR      "/target/etc/pcmcia"
#define PCMCIA_SERIAL_OPTS   "/target/etc/pcmcia/serial.opts"
#define PCMCIA_CDROM_OPTS    "/target/etc/pcmcia/cdrom.opts"
#define PCMCIA_HOSTS_FILE    "/target/etc/hosts"
#define PCMCIA_NETWORK_OPTS  "/target/etc/pcmcia/network.opts"
#define PCMCIA_VAR_RUN       "/var/run"
#define PCMCIA_UNCONFIGURED  "/target/lib/modules/%s/pcmcia/.unconfigured"
#else
#define PCMCIA_ETC_DIR       "etc"
#define PCMCIA_CONF_FILE     "etc/pcmcia.conf"
#define PCMCIA_CONF_DIR      "etc/pcmcia"
#define PCMCIA_SERIAL_OPTS   "etc/pcmcia/serial.opts"
#define PCMCIA_CDROM_OPTS    "etc/pcmcia/cdrom.opts"
#define PCMCIA_HOSTS_FILE    "etc/hosts"
#define PCMCIA_NETWORK_OPTS  "etc/pcmcia/network.opts"
#define PCMCIA_VAR_RUN       "var/run"
#define PCMCIA_UNCONFIGURED  "lib/modules/%s/pcmcia/.unconfigured"
#endif

#define CARDMGR_START 1
#define CARDMGR_STOP  2
#define PCMCIA_NOT_CFG   1
#define PCMCIA_MOD_ERROR 2

/*
 * Messages
 */

#define HEADER_TXT "#\n# This file was automatically generated during the Debian installation\n#\n"
#define WRT_HDR(FILEP) fprintf((FILEP),"%s",HEADER_TXT)

#define SERIAL_TXT "# Serial device configuration\n"
#define CDROM_TXT  "# CDROM device configuration\n"
#define NETWORK_TXT "# Network adapter configuration\n"


int _cardmgr(char action)
{
  struct stat tmp;

  switch(action)
    {
    case CARDMGR_START:
      snprintf(prtbuf,PRTBUFSIZE,
	       "/sbin/cardmgr %s ; sleep 3",
	       cardmgr_opts);
#ifdef _TESTING_
      fprintf(stderr,"start cardmgr: %s\n",prtbuf);
#endif
      system(prtbuf); 
      break;
    case CARDMGR_STOP:
      if(!(stat("/var/run/cardmgr.pid",&tmp)))
	system("kill -TERM `cat /var/run/cardmgr.pid` 2>/dev/null ; "
	       "sleep 3");
      break;
    }

  /*
   * Every time that cardmgr comes up, a new resolv.conf file is created.
   * This will copy it to the right place.
   */
  if(!stat("/etc/resolv.conf",&tmp)&&!S_ISLNK(tmp.st_mode))
    system("cp /etc/resolv.conf /target/etc/resolv.conf");
    
  return 0;
}

int _activate_pcmcia()
{
  struct stat tmp;
  
  if(pcmcia_configured)
    {
      /*
       * Stop the card services if running.
       */
      if(_cardmgr(CARDMGR_STOP))
	return PCMCIA_MOD_ERROR;
#ifndef _TESTING_
      /*
       * Creates the /var/run directory if not there
       */
      if(stat(PCMCIA_VAR_RUN,&tmp)!=0)
	if(mkdir(PCMCIA_VAR_RUN,0755))
	  return PCMCIA_MOD_ERROR;
      
      /*
       * Link /target/etc/pcmcia to /etc/pcmcia
       */
      getcwd(prtbuf,PRTBUFSIZE);
      chdir("/etc");
      symlink("/target/etc/pcmcia","pcmcia");
      chdir(prtbuf);

      getcwd(prtbuf,PRTBUFSIZE);
      chdir("/lib");
      if(!stat("/lib/modules",&tmp))
	unlink("modules");
      symlink("/target/lib/modules/","modules");
      chdir(prtbuf);
    
#endif
      snprintf(prtbuf,PRTBUFSIZE,
	       "/sbin/insmod -s "
#ifndef _TESTING_
	       "/target"
#endif
	       "/lib/modules/%s/pcmcia/pcmcia_core.o %s",
	       KVER,
	       core_opts);
      system(prtbuf);
      snprintf(prtbuf,PRTBUFSIZE,
	       "/sbin/insmod -s "
#ifndef _TESTING_
	       "/target"
#endif
	       "/lib/modules/%s/pcmcia/%s.o %s",
	       KVER,
	       controller,
	       pcic_opts);
      system(prtbuf);
      snprintf(prtbuf,PRTBUFSIZE,
	       "/sbin/insmod -s "
#ifndef _TESTING_
	       "/target"
#endif
	       "/lib/modules/%s/pcmcia/ds.o ",
	       KVER);
      system(prtbuf);
      
      /*
       * Start the cardservices.
       */
       if(_cardmgr(CARDMGR_START))
	return PCMCIA_MOD_ERROR;
    }
  else
    return PCMCIA_NOT_CFG;
  
  return 0;
}

int activate_pcmcia()
{
  int rtn=0;

  newtSuspend();
  rtn=_activate_pcmcia();
  newtResume();

  switch(rtn)
    {
    case PCMCIA_NOT_CFG:
      problemBox(MSG_PCMCIA_UNCONFIGURED, MSG_PCMCIA_ERROR);
      break;
    case PCMCIA_MOD_ERROR:
      problemBox(MSG_PCMCIA_ERROR_L,MSG_PCMCIA_ERROR);
      break;
    default:
    }

  return rtn;
}

void input_parameter(char* text, char* title, char** variable)
{
  char* def=NULL;
  if(*variable)
    {
      def=strdup(*variable);
      free(*variable);
    }
  else
    def=strdup("");
  (*variable)=inputBox(text,title,def);
  free(def);

  if((*variable)==NULL)
    (*variable)=strdup("");
}

void pcmcia_serial_file()
{
  int loop=1;
  while(loop)
    {
      input_parameter(MSG_SERIAL_LINK_L,MSG_SERIAL_LINK,&serial_link);
      input_parameter(MSG_SETSERIAL_OPTS_L,
		      MSG_SETSERIAL_OPTS,
		      &serial_opts);

      if(yesNoBox(MSG_INITTAB_ENTRY_L,MSG_INITTAB_ENTRY)!=0)
	{
	  use_inittab=1;
	  input_parameter(MSG_INITTAB_COMMAND,MSG_INITTAB_ENTRY,&inittab);
	}
      else
	  use_inittab=0;
	
      /*
       * Confirm values
       */
      snprintf(prtbuf,PRTBUFSIZE, MSG_PCMCIA_CONFIRM_SERIAL1,
	       serial_link,
	       serial_opts);
      if(use_inittab)
	snprintf(prtbuf+strlen(prtbuf),PRTBUFSIZE,
		 MSG_PCMCIA_CONFIRM_SERIAL2,
		 inittab);
      snprintf(prtbuf+strlen(prtbuf), PRTBUFSIZE,
	       MSG_PCMCIA_CONFIRM_SERIAL3);

      if(yesNoBox(prtbuf,MSG_CONFIRMATION)!=0)
	loop=0;
    }
}

void pcmcia_cdrom_file()
{
    int loop=1;
  while(loop)
    {
      if(yesNoBox(MSG_CDROM_FSTAB_L,MSG_CDROM_FSTAB)!=0)
	{
	  free(cdrom_do_fstab);
	  cdrom_do_fstab=strdup("y");
	}
      else
	{
	  free(cdrom_do_fstab);
	  cdrom_do_fstab=strdup("n");
	}

      if(yesNoBox(MSG_CDROM_MOUNT_L,MSG_CDROM_MOUNT)!=0)
	{
	  free(cdrom_do_mount);
	  cdrom_do_mount=strdup("y");
	}
      else
	{
	  free(cdrom_do_mount);
	  cdrom_do_mount=strdup("n");
	}

      input_parameter(MSG_CDROM_FSTYPE_L,MSG_CDROM_FSTYPE,&cdrom_fstype);
      input_parameter(MSG_CDROM_OPTS_L,MSG_CDROM_OPTS,&cdrom_opts);
      input_parameter(MSG_CDROM_MOUNTPT_L,MSG_CDROM_MOUNTPT,&cdrom_mountpt);

      /*
       * Confirm values
       */
      snprintf(prtbuf,PRTBUFSIZE,
	       MSG_PCMCIA_CONFIRM_CDROM,
	       cdrom_do_fstab,
	       cdrom_do_mount,
	       cdrom_opts,
	       cdrom_mountpt);

      if(yesNoBox(prtbuf,MSG_CONFIRMATION)!=0)
	loop=0;
    }
}

void pcmcia_conf_file()
{
  int loop=1;
  char *text;
  char *title;
  char *def=NULL;


  while(loop)
    {
      /*
       * Type of controller
       */  
      text=strdup(MSG_CONTROLLER_L);
      title=strdup(MSG_CONTROLLER);
      
      for(;;)
	{
	  if(controller)
	    {
	      def=strdup(controller);
	      free(controller);
	    }
	  else
	    def=strdup("i82365");
	  controller=inputBox(text,title,def);
	  free(def);
	  if((strcmp(controller,"i82365"))&&
	     (strcmp(controller,"tcic")))
	    problemBox(MSG_INVALID_CONTROLLER,MSG_PROBLEM);
	  else
	    break;
	}
      free(title);
      free(text);
      
      /*
       * PCIC_OPTS, CORE_OPTS, CARDMGR_OPTS
       */
      input_parameter(MSG_PCIC_OPTS_L,MSG_PCIC_OPTS,&pcic_opts);
      input_parameter(MSG_CORE_OPTS_L,MSG_CORE_OPTS,&core_opts);
      input_parameter(MSG_CARDMGR_OPTS_L,MSG_CARDMGR_OPTS,&cardmgr_opts);
      
      /*
       * Confirm values
       */
      snprintf(prtbuf,PRTBUFSIZE, MSG_PCMCIA_CONFIRM_CONTROLLER,
	       controller,
	       pcic_opts,
	       core_opts,
	       cardmgr_opts);
      if(yesNoBox(prtbuf,MSG_CONFIRMATION)!=0)
	loop=0;
    } 
}

static int write_files()
{
  if(write_lowlevel_files()!=0)
    {
      problemBox(MSG_PCMCIA_ERROR_WRITE,MSG_PCMCIA_ERROR);
      return 1;
    }
  else if(write_serial_opts()!=0)
    {
      problemBox(MSG_PCMCIA_ERROR_WRITE,MSG_PCMCIA_ERROR);
      return 1;
    }
  else if(write_cdrom_opts()!=0)
    {
      problemBox(MSG_PCMCIA_ERROR_WRITE,MSG_PCMCIA_ERROR);
      return 1;
    }
  /*  else if(write_network_config()!=0)
      return 1; */

  pcmcia_configured=1;

  return 0;
}

static int write_lowlevel_files()
{
  FILE* p_file;

  /*
   * Creates the /etc directory
   */
  if(check_dir(PCMCIA_ETC_DIR)==-1)
    if(mkdir(PCMCIA_ETC_DIR,0755))
      return 1;

  /*
   * /etc/pcmcia.conf
   */  
  if((p_file=fopen(PCMCIA_CONF_FILE,"w"))==NULL)
    return 1;
  WRT_HDR(p_file);
  fprintf(p_file,
	  "PCMCIA=yes\n"
	  "# Should be either i82365 or tcic\nPCIC=%s\n"
	  "# Put socket driver timing parameters here\nPCIC_OPTS=%s\n"
	  "# Put pcmcia_core options here\nCORE_OPTS=%s\n"
	  "# Put cardmgr options here\nCARDMGR_OPTS=%s\n",
	  controller,
	  pcic_opts,
	  core_opts,
	  cardmgr_opts);
  fclose(p_file);

  /*
   * Creates /etc/pcmcia
   */
  if(check_dir(PCMCIA_CONF_DIR)==-1)
    if(mkdir(PCMCIA_CONF_DIR,0755))
      return 1;

  return 0;
}

static int write_serial_opts()
{
  FILE* p_file;
  /*
   * serial.opts
   */
  if((p_file=fopen(PCMCIA_SERIAL_OPTS,"w"))==NULL)
    return 1;

  WRT_HDR(p_file);
  fprintf(p_file,SERIAL_TXT);
  fprintf(p_file,
	  "#\n# The address format is \"scheme,socket,instance\".\n#\n");
  fprintf(p_file,"case \"$ADDRESS\" in\n*,*,*)\n");
  fprintf(p_file,"\t#Symbolic link to dialout device\n"
	  "\tLINK=\"%s\"\n\t# Options for 'setserial'\n"
	  "\tSERIAL_OPTS=\"%s\"\n"
	  "\t# Should we create an inittab entry for this port?\n",
	  serial_link,
	  serial_opts);
  if(use_inittab)
    fprintf(p_file,"\tINITTAB=\"%s\"\n",inittab);
  else
    fprintf(p_file,"\t#INITTAB=\"%s\"\n",inittab);
  
  fprintf(p_file,"\t;;\nesac\n");
  fclose(p_file);
  
  return 0;
}

static int write_cdrom_opts()
{
  FILE* p_file;
  /*
   * cdrom.opts
   */
  if((p_file=fopen(PCMCIA_CDROM_OPTS,"w"))==NULL)
    return 1;
  WRT_HDR(p_file);
  fprintf(p_file,CDROM_TXT);
  fprintf(p_file,
	  "#\n# The address format is \"scheme,socket\".\n#\n");
  fprintf(p_file,
	  "case \"$ADDRESS\" in\n*,*)\n");
  fprintf(p_file,"\tDO_FSTAB=\"%s\" ; DO_MOUNT=\"%s\"\n"
	  "\tFSTYPE=\"%s\"\n"
	  "\tOPTS=\"%s\"\n"
	  "\tMOUNTPT=\"%s\"\n"
	  "\t;;\nesac\n",
	  cdrom_do_fstab,
	  cdrom_do_mount,
	  cdrom_fstype,
	  cdrom_opts,
	  cdrom_mountpt);
  fclose(p_file);

  return 0;
}

/*
 * Mark what we need to configure
 */
static int select_components()
{
  char msg[]=MSG_PCMCIA_DRIVERS;
  char title[]=MSG_PCMCIA_TITLE;

  return check_box(msg, title, _HEIGHT, _WIDTH,
		   pcmcia_choices, &pcmcia_values, OPTS);  
  
}


int configure_pcmcia()
{
  int ix,rtn=0;
  #ifdef _TESTING_
  problemBox(pcmcia_id,MSG_VERSION);
  #endif
  controller=strdup("i82365");
  pcic_opts=strdup("");
  core_opts=strdup("");
  cardmgr_opts=strdup("");
  serial_link=strdup("/dev/modem");
  serial_opts=strdup("");
  inittab=strdup("/sbin/getty");
  cdrom_do_fstab=strdup("n");
  cdrom_do_mount=strdup("n");
  cdrom_fstype=strdup("iso9660");
  cdrom_opts=strdup("ro");
  cdrom_mountpt=strdup("/cdrom");

  if(!pcmcia_values)
    {
      pcmcia_values=calloc(OPTS+1,sizeof(char));
      for(ix=0;ix<OPTS;ix++)
	pcmcia_values[ix]=' ';
      pcmcia_values[ix]='\0';
    }

  pcmcia_choices=calloc(OPTS,sizeof(char*));
  pcmcia_choices[0]=strdup(MSG_CONTROLLER_PCMCIA);
  pcmcia_choices[1]=strdup(MSG_SERIAL_PCMCIA);
  pcmcia_choices[2]=strdup(MSG_CDROM_PCMCIA);
  /*  pcmcia_choices[3]=strdup(MSG_NETWORK_PCMCIA); */

  if(select_components()!=DLG_OKAY)
    return 255;
  else
    for(ix=0;ix<OPTS;ix++)
      if(pcmcia_values[ix]!=' ')
	switch(ix)
	  {
	  case 0:
	    pcmcia_conf_file(); break;
	  case 1:
	    pcmcia_serial_file(); break;
	  case 2:
	    pcmcia_cdrom_file(); break;
	    /*
	  case 3:
	    pcmcia_network_file(); break;
	    */
	  }
  
  for(ix=0;ix<OPTS;ix++)
    free(pcmcia_choices[ix]);
  free(pcmcia_choices);

  if((rtn=write_files())!=0)
    return rtn;
  
  rtn=activate_pcmcia();
  
  /*
   * Remove the .unconfigured file
   */
  snprintf(prtbuf,PRTBUFSIZE,PCMCIA_UNCONFIGURED,KVER);
  unlink(prtbuf);
  
  return rtn;
}

/*
 * To test, compile using:
 * cc -o ./the_test -D_TESTING_ -DKVER=\"2.0.33\" -g pcmcia.c boxes.c -lnewt -lslang
 */
#ifdef _TESTING_
/*
 * check_dir
 * Stats some dirname and checks whether it exists and whether it's a 
 * directory.
 */
int check_dir(char* dirname)
{
  struct stat check;
  if(stat(dirname,&check)==-1)
    return -1;
  else
    return S_ISDIR(check.st_mode);
}

void main(void)
{
  newtInit();
  newtCls();

  newtDrawRootText(0, 0, "PCMCIA Config test program");
  newtOpenWindow(7, 2, 65, 18, "Root Window");
  newtPushHelpLine(NULL);
  configure_pcmcia();

  free(controller);
  free(pcic_opts);
  free(core_opts);
  free(cardmgr_opts);
  free(serial_link);
  free(serial_opts);
  free(inittab);
  free(cdrom_do_fstab);
  free(cdrom_do_mount);
  free(cdrom_fstype);
  free(cdrom_opts);
  free(cdrom_mountpt);
  

  newtPopWindow();
  newtFinished();
}
#endif
