/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>	/* strstr, strtok, strlen, strcmp */
#include <ctype.h>	/* tolower */
#include <regex.h>

#include <gtk/gtk.h>

#include "xqf.h"
#include "utils.h"
#include "psearch.h"


enum psearch_mode { PSEARCH_MODE_SUBSTRING = 0, PSEARCH_MODE_REGEXP = 1 };
enum psearch_case { PSEARCH_CASE_SENSITIVE = 0, PSEARCH_CASE_INSENSITIVE = 1 };


struct psearch_data {
  enum psearch_mode mode;
  enum psearch_case csens;
  void *pattern;
};


static struct psearch_data psearch = { 
  PSEARCH_MODE_SUBSTRING, 
  PSEARCH_CASE_INSENSITIVE, 
  NULL
};


static char *psearch_history = NULL;


static char *lowcasestrstr (const char *str, const char *substr) {
  int slen = strlen (str);
  int sublen = strlen (substr);
  const char *end;
  int i;

  if (slen < sublen)
    return NULL;

  end = &str[slen - sublen + 1];

  while (str < end) {
    for (i = 0; i < sublen; i++) {
      if (substr[i] != tolower (str[i]))
	goto loop;
    }
    return (char *) str;

  loop:
    str++;
  }

  return NULL;
}


int test_player (struct player *p) {
  GSList *tmp;

  if (!psearch.pattern || !p->name)
    return FALSE;

  switch (psearch.mode) {

  case PSEARCH_MODE_SUBSTRING:

    switch (psearch.csens) {

    case PSEARCH_CASE_SENSITIVE:
      for (tmp = (GSList *) psearch.pattern; tmp; tmp = tmp->next) {
	if (strstr (p->name, (char *) tmp->data))
	  return TRUE;
      }
      return FALSE;

    case PSEARCH_CASE_INSENSITIVE:
      for (tmp = (GSList *) psearch.pattern; tmp; tmp = tmp->next) {
	if (lowcasestrstr (p->name, (char *) tmp->data))
	  return TRUE;
      }
      return FALSE;
    }
    return FALSE;

  case PSEARCH_MODE_REGEXP:
    if (regexec ((regex_t *) psearch.pattern, p->name, 0, NULL, 0) == 0)
      return TRUE;
    return FALSE;
  }
  return FALSE;
}


static enum psearch_res psearch_set_pattern (char *str, 
                             enum psearch_mode mode, enum psearch_case csens) {
  const char delim[] = " \t\n\r";
  char *tmp, *tok;
  int regex_flags;
  regex_t *preg;
  int res;

  if (psearch.pattern) {
    switch (psearch.mode) {

    case PSEARCH_MODE_SUBSTRING:
      g_slist_foreach ((GSList *) psearch.pattern, (GFunc) free, NULL);
      g_slist_free ((GSList *) psearch.pattern);
      break;

    case PSEARCH_MODE_REGEXP:
      regfree ((regex_t *) psearch.pattern);
      g_free (psearch.pattern);
      break;
    }

    psearch.pattern = NULL;
  }

  tmp = strdup_strip (str);

  if (!tmp)
    return PSEARCH_CANCEL;

  switch (mode) { 

  case PSEARCH_MODE_SUBSTRING:
    if (csens == PSEARCH_CASE_INSENSITIVE) {
      for (tok = tmp; *tok; tok++)
	*tok = tolower (*tok);
    }

    tok = strtok (tmp, delim);
    (GSList *) psearch.pattern = 
                  g_slist_prepend ((GSList *) psearch.pattern, g_strdup (tok));
    
    while ((tok = strtok (NULL, delim)) != NULL) {
      (GSList *) psearch.pattern = 
	          g_slist_prepend ((GSList *) psearch.pattern, g_strdup (tok));
    }

    g_free (tmp);
    break;

  case PSEARCH_MODE_REGEXP:
    preg = g_malloc (sizeof (regex_t));

    regex_flags = (csens == PSEARCH_CASE_INSENSITIVE)? REG_ICASE : 0;
    regex_flags |= REG_NOSUB;

    res = regcomp (preg, tmp, regex_flags);
    g_free (tmp);

    if (res) {
      g_free (preg);
      dialog_ok (NULL, "Bad regular expression");
      return PSEARCH_CANCEL;
    }

    psearch.pattern = preg;
    break;
  }

  psearch.mode = mode;
  psearch.csens = csens;
  return PSEARCH_OK;
}


static const char *psearch_mode_desc[3] = {
  "case sensitive substring match",
  "case insensitive substring match",
  "regular expression match"
};


static GtkWidget *case_buttons[2];
static GtkWidget *mode_buttons[2];
static GtkWidget *psearch_entry;


static enum psearch_res psearch_ok;


static void psearch_entry_activate_callback (GtkWidget *widget, gpointer data) {
  enum psearch_mode mode;
  enum psearch_case csens;
  char *str;

  mode  = GTK_TOGGLE_BUTTON (mode_buttons[1])->active;
  csens = GTK_TOGGLE_BUTTON (case_buttons[1])->active;

  str = gtk_entry_get_text (GTK_ENTRY (psearch_entry));

  if (psearch.mode == mode && psearch.csens == csens && psearch_history && 
                                              !strcmp (str, psearch_history)) {
    psearch_ok = PSEARCH_USELAST;
  }
  else {
    if (psearch_history)
      g_free (psearch_history);
    psearch_history = g_strdup (str);
    psearch_ok = psearch_set_pattern (str, mode, csens);
  }
}


enum psearch_res psearch_dialog (void) {
  GtkWidget *window;
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *button;
  GtkWidget *label;
  GtkWidget *frame;
  GSList *group;

  psearch_ok = PSEARCH_CANCEL;

  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_title (GTK_WINDOW (window), "Find Player");
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (window_delete_event_callback), NULL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_grab_add (GTK_WIDGET (window));

  register_window (window);

  main_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (main_vbox), 16);
  gtk_container_add (GTK_CONTAINER (window), main_vbox);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, TRUE, TRUE, 0);

  /* Message */

  label = gtk_label_new ("Find Player:");
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  /* Entry */

  psearch_entry = gtk_entry_new_with_max_length (128);
  gtk_widget_set_usize (GTK_WIDGET (psearch_entry), 128, -1);
  if (psearch_history) {
    gtk_entry_set_text (GTK_ENTRY (psearch_entry), psearch_history);
    gtk_entry_select_region (GTK_ENTRY (psearch_entry), 0, 
                                                     strlen (psearch_history));
  }	
  gtk_signal_connect (GTK_OBJECT (psearch_entry), "activate",
		      GTK_SIGNAL_FUNC (psearch_entry_activate_callback), NULL);
  gtk_signal_connect_object (GTK_OBJECT (psearch_entry), "activate",
                    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (hbox), psearch_entry, TRUE, TRUE, 0);
  gtk_widget_grab_focus (GTK_WIDGET (psearch_entry));
  gtk_widget_show (psearch_entry);

  /* Ok Button */

  button = gtk_button_new_with_label (" Ok ");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		             GTK_SIGNAL_FUNC (psearch_entry_activate_callback),
			     GTK_OBJECT (psearch_entry));
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
	            GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  /* Cancel Button */

  button = gtk_button_new_with_label (" Cancel ");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_widget_show (button);

  gtk_widget_show (hbox);

  /* Mode Buttons */

  hbox = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, FALSE, 0);

  /* Substring/RegExp Mode Buttons */

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 2);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  mode_buttons[0] = gtk_radio_button_new_with_label (NULL, "substring match");
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (mode_buttons[0]));
  mode_buttons[1] = gtk_radio_button_new_with_label (group,
                                                   "regular expression match");
  gtk_toggle_button_set_state (
                         GTK_TOGGLE_BUTTON (mode_buttons[psearch.mode]), TRUE);

  gtk_box_pack_start (GTK_BOX (vbox), mode_buttons[0], TRUE, TRUE, 0);
  gtk_widget_show (mode_buttons[0]);

  gtk_box_pack_start (GTK_BOX (vbox), mode_buttons[1], TRUE, TRUE, 0);
  gtk_widget_show (mode_buttons[1]);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  /* Case Sensitivity Buttons */

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 2);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  case_buttons[0] = gtk_radio_button_new_with_label (NULL, "case sensitive");
  group = gtk_radio_button_group (GTK_RADIO_BUTTON (case_buttons[0]));
  case_buttons[1] = gtk_radio_button_new_with_label (group, "case insensitive");

  gtk_toggle_button_set_state (
                         GTK_TOGGLE_BUTTON (case_buttons[psearch.csens]), TRUE);

  gtk_box_pack_start (GTK_BOX (vbox), case_buttons[0], TRUE, TRUE, 0);
  gtk_widget_show (case_buttons[0]);

  gtk_box_pack_start (GTK_BOX (vbox), case_buttons[1], TRUE, TRUE, 0);
  gtk_widget_show (case_buttons[1]);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  gtk_widget_show (hbox);

  gtk_widget_show (main_vbox);
  gtk_widget_show (window);

  gtk_main ();

  unregister_window (window);

  return psearch_ok;
}

