/*
 * ratMessage.c --
 *
 *	This file contains code which implements the message entities.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"
#include "ratPGP.h"
#include <signal.h>

/*
 * An array of commands. It contains one entry for each internal message
 * type (as defined by RatMessageType).
 */
static MessageProcInfo *messageProcInfo;

/*
 * The number of replies created. This is used to create new unique
 * message handlers.
 */
static int numReplies = 0;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numBodies = 0;

static void RatBodyDelete(Tcl_Interp *interp, BodyInfo *bodyInfoPtr);
static BodyInfo *RatFindFirstText(BodyInfo *bodyInfoPtr);


/*
 *----------------------------------------------------------------------
 *
 * RatInitMessages --
 *
 *      Initialize the message data structures.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The messageProcInfo array is allocated and initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatInitMessages()
{
    messageProcInfo = (MessageProcInfo*)ckalloc(3*sizeof(MessageProcInfo));
    RatStdMessagesInit(&messageProcInfo[RAT_CCLIENT_MESSAGE]);
    RatDbMessagesInit(&messageProcInfo[RAT_DBASE_MESSAGE]);
    RatFrMessagesInit(&messageProcInfo[RAT_FREE_MESSAGE]);
}


/*
 *----------------------------------------------------------------------
 *
 * RatMessageCmd --
 *
 *      Main std mail entity procedure. This routine implements the mail
 *	commands mentioned in ../INTERFACE.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	many.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatMessageCmd(ClientData clientData,Tcl_Interp *interp,int argc,char *argv[])
{
    MessageInfo *msgPtr = (MessageInfo*) clientData;
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'h') && (strncmp(argv[1], "headers", length) == 0)
	    && (length > 1)) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " headers decoding_level\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return RatMessageGetHeader(interp,
		(*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr),
		argv[2]);

    } else if ((c == 'b') && (strncmp(argv[1], "body", length) == 0)
	    && (length > 1)) {
	if (!msgPtr->bodyInfoPtr) {
	    msgPtr->bodyInfoPtr =
		    (*messageProcInfo[msgPtr->type].createBodyProc)(interp,
		    msgPtr);
	    RatPGPBodyCheck(interp, messageProcInfo, &msgPtr->bodyInfoPtr);
	    Tcl_CreateCommand(interp, msgPtr->bodyInfoPtr->cmdName,
		    RatBodyCmd, (ClientData) msgPtr->bodyInfoPtr, NULL);
	}
	Tcl_SetResult(interp, msgPtr->bodyInfoPtr->cmdName, TCL_STATIC);
	return TCL_OK;
    
    } else if ((c == 'r') && (strncmp(argv[1], "rawBody", length) == 0)
	    && (length > 1)) {

	Tcl_SetResult(interp,
		(*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr),
		TCL_VOLATILE);
	return TCL_OK;

    } else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0)
	    && (length > 1)) {
	ENVELOPE *env = (*messageProcInfo[msgPtr->type].envelopeProc)(msgPtr);
	int i;
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " get fields\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (i=2; i<argc; i++) {
	    if (!strcasecmp(argv[i], "return_path")) {
		RatInitAddresses(interp, env->return_path);
	    } else if (!strcasecmp(argv[i], "from")) {
		RatInitAddresses(interp, env->from);
	    } else if (!strcasecmp(argv[i], "sender")) {
		RatInitAddresses(interp, env->sender);
	    } else if (!strcasecmp(argv[i], "reply_to")) {
		RatInitAddresses(interp, env->reply_to);
	    } else if (!strcasecmp(argv[i], "to")) {
		RatInitAddresses(interp, env->to);
	    } else if (!strcasecmp(argv[i], "cc")) {
		RatInitAddresses(interp, env->cc);
	    } else if (!strcasecmp(argv[i], "bcc")) {
		RatInitAddresses(interp, env->bcc);
	    } else {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "bad field \"", argv[1],
			"\": must be one of return_path, from, sender, ",
			"reply_to, to, cc or bcc", (char*)NULL);
		return TCL_ERROR;
	    }
	}
	return TCL_OK;

    } else if ((c == 'r') && (strncmp(argv[1], "reply", length) == 0)
	    && (length > 1)) {
	/*
	 * Construct a reply to a message. We should really handle
	 * different character sets here. /MaF
	 */
	ENVELOPE *env = (*messageProcInfo[msgPtr->type].envelopeProc)(msgPtr);
	char handler[32], buf[1024], *subject;
	BodyInfo *bodyInfoPtr;
	ADDRESS *adrPtr;
	Tcl_DString ds;
	Tcl_DStringInit(&ds);

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " reply to\"", (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(handler, "reply%d", numReplies++);
	if (!strcasecmp(argv[2], "sender")) {
	    /*
	     * We should construct a reply which should go only to one person.
	     * We look for the address in the following fields:
	     *  Reply-To:, From:. Sender:, From
	     * As sson as an address is found the search stops.
	     */
	    if (env->reply_to) {
		adrPtr = env->reply_to;
	    } else if (env->from) {
		adrPtr = env->from;
	    } else if (env->sender) {
		adrPtr = env->sender;
	    } else {
		adrPtr = env->return_path;
	    }
	    for (;adrPtr; adrPtr = adrPtr->next) {
		RatAddressTranslate(interp, adrPtr);
		if (adrPtr->mailbox) {
		    if (Tcl_DStringLength(&ds)) {
			Tcl_DStringAppend(&ds, ", ", -1);
		    }
		    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
		}
	    }
	    Tcl_SetVar2(interp, handler, "to", Tcl_DStringValue(&ds),
		    TCL_GLOBAL_ONLY);
	    Tcl_DStringFree(&ds);
	} else {
	    /*
	     * We should construct a reply which goes to everybody who has
	     * recieved this message. This is done by first collecting all
	     * addresses found in: Reply-To:, From:, Sender:, To: and Cc:.
	     * Then go though this list and eliminate myself and any
	     * duplicates. Now we use the first element of the list as To:
	     * and the rest as Cc:.
	     */
	    ADDRESS **recipientPtrPtr = (ADDRESS**)ckalloc(16*sizeof(ADDRESS*));
	    int numAllocated = 16;
	    int numRecipients = 0;
	    int inList = 0;
	    ADDRESS *to = NULL;
	    int i, j;

#define SCANLIST(x) for (adrPtr = (x); adrPtr; adrPtr = adrPtr->next) { \
			if (numRecipients == numAllocated) { \
			    numAllocated += 16; \
			    recipientPtrPtr = (ADDRESS**)REALLOC( \
				    recipientPtrPtr, \
				    numAllocated*sizeof(ADDRESS*)); \
			} \
			recipientPtrPtr[numRecipients++] = adrPtr; \
	    	    }

	    SCANLIST(env->reply_to);
	    SCANLIST(env->from);
	    SCANLIST(env->to);
	    SCANLIST(env->cc);

	    for (i=0; i<numRecipients; i++) {
		adrPtr = recipientPtrPtr[i];
		if (!adrPtr->host) {
		    inList = (inList)? 0 : 1;
		    continue;
		}
		if (RatAddressIsMe(interp, adrPtr, 1)) {
		    continue;
		}
		RatAddressTranslate(interp, adrPtr);
		for (j=0; j<i; j++) {
		    if (!RatAddressCompare(adrPtr, recipientPtrPtr[j])) {
			break;
		    }
		}
		if (j < i) {
		    continue;
		}
		if (!to) {
		    to = adrPtr;
		} else {
		    if (Tcl_DStringLength(&ds)) {
			Tcl_DStringAppend(&ds, ", ", 2);
		    }
		    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
		}
	    }
	    if (Tcl_DStringLength(&ds)) {
		Tcl_SetVar2(interp, handler, "cc", Tcl_DStringValue(&ds),
			TCL_GLOBAL_ONLY);
	    }
	    if (!to && numRecipients) {
		to = recipientPtrPtr[0];
	    }
	    if (to) {
		if (strcmp(to->host, currentHost)) {
		    sprintf(buf, "%s@%s", to->mailbox, to->host);
		} else {
		    sprintf(buf, "%s", to->mailbox);
		}
		Tcl_SetVar2(interp, handler, "to", buf, TCL_GLOBAL_ONLY);
	    }
	    ckfree(recipientPtrPtr);
	}
	if (env->subject) {
	    subject = RatDecodeHeader(interp, env->subject);
	    subject = RatDecodeHeaderFull(interp, subject);
	    if (strncasecmp("re:", subject, 3)) {
		strcpy(buf, "Re: ");
		strncpy(buf+strlen(buf), subject, sizeof(buf)-strlen(buf));
		Tcl_SetVar2(interp, handler, "subject", buf, TCL_GLOBAL_ONLY);
	    } else {
		Tcl_SetVar2(interp, handler, "subject",subject,TCL_GLOBAL_ONLY);
	    }
	} else {
	    Tcl_SetVar2(interp, handler, "subject",
		    Tcl_GetVar2(interp, "option", "no_subject",TCL_GLOBAL_ONLY),
		    TCL_GLOBAL_ONLY);
	}
	if (env->message_id) {
	    Tcl_SetVar2(interp, handler, "in_reply_to", env->message_id,
		    TCL_GLOBAL_ONLY);
	}
	bodyInfoPtr = RatFindFirstText(msgPtr->bodyInfoPtr);
	if (bodyInfoPtr) {
	    char *leader, *newPtr, *srcPtr, *dstPtr, *attrFormat, *attr;
	    unsigned char *decodedBody;
	    ListExpression *exprPtr;
	    RatEncoding encoding;
	    int outLength, lines, skipSig;
	    BODY *bodyPtr;
	    char *dataPtr;
	    unsigned long len;

	    bodyPtr = bodyInfoPtr->bodyPtr;
	    dataPtr =(*messageProcInfo[bodyInfoPtr->msgPtr->type].fetchBodyProc)
		    (bodyInfoPtr, &len);

	    switch(bodyPtr->encoding) {
	    case ENC7BIT:		encoding = RAT_7BIT; break;
	    case ENC8BIT:		encoding = RAT_8BIT; break;
	    case ENCBINARY:		encoding = RAT_BINARY; break;
	    case ENCBASE64:		encoding = RAT_BASE64; break;
	    case ENCQUOTEDPRINTABLE:	encoding = RAT_QP; break;
	    default:			encoding = RAT_UNKOWN; break;
	    }
	    decodedBody = RatDecode(interp, (unsigned char*)dataPtr, len,
		    encoding, &outLength, NULL, NULL);
	    for (srcPtr = (char*)decodedBody, lines = 0; *srcPtr; srcPtr++){
		if ('\n' == *srcPtr) {
		    lines++;
		}
	    }
	    attrFormat = Tcl_GetVar2(interp, "option", "attribution",
		    TCL_GLOBAL_ONLY);
	    if (attrFormat && strlen(attrFormat)
		    && (exprPtr = RatParseList(attrFormat))) {
		attr = RatDoList(interp, exprPtr,
			messageProcInfo[msgPtr->type].getInfoProc,
			(ClientData)msgPtr, 0);
		RatFreeListExpression(exprPtr);
	    } else {
		attr = NULL;
	    }
	    leader = Tcl_GetVar2(interp, "option","reply_lead",TCL_GLOBAL_ONLY);
	    Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "skip_sig",
		    TCL_GLOBAL_ONLY), &skipSig);
	    newPtr = (char*)ckalloc(strlen((char*)decodedBody) +
		    (strlen(leader)+1)*lines+1+(attr?strlen(attr)+2:0));
	    if (attr) {
		strcpy(newPtr, attr);
		strcat(newPtr, "\n");
		dstPtr = newPtr+strlen(newPtr);
	    } else {
		dstPtr = newPtr;
	    }
	    for (srcPtr = (char*)decodedBody; *srcPtr;) {
		if (skipSig && !strncmp("-- ", srcPtr, 3) &&
			(srcPtr[3] == '\r' || srcPtr[3] == '\n')) {
		    break;
		}
		strcpy(dstPtr, leader);
		dstPtr += strlen(leader);
		if (strncmp(leader, srcPtr, strlen(leader))) {
		    *dstPtr++ = ' ';
		}
		do {
		    if ('\r' != *srcPtr) {
			*dstPtr++ = *srcPtr;
		    }
		} while (*srcPtr && '\n' != *srcPtr++);
	    }
	    *dstPtr = '\0';
	    Tcl_SetVar2(interp, handler, "data", newPtr, TCL_GLOBAL_ONLY);
	    ckfree(decodedBody);
	    ckfree(newPtr);
	}
	Tcl_SetResult(interp, handler, TCL_VOLATILE);
	return TCL_OK;

    } else if ((c == 's') && (strncmp(argv[1], "save", length) == 0)
	    && (length > 1)) {
	Tcl_Channel channel;

	if (NULL == (channel = Tcl_GetChannel(interp, argv[2], NULL))) {
	    return TCL_ERROR;
	}
	RatMessageSave(interp, channel, msgPtr);
	return TCL_OK;

    } else if ((c == 'c') && (strncmp(argv[1], "copy", length) == 0)
	    && (length > 1)) {
	char **listArgv, *folder, *cPtr, flags[128], date[128];
	MAILSTREAM *stream;
	struct stat sbuf;
	Tcl_DString ds;
	STRING string;
	int listArgc, result, i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " copy vfolder_def\"", (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_SplitList(interp, argv[2], &listArgc, &listArgv);

	/*
	 * If the destination is dbase then we call RatInsert
	 */
	if (!strcmp("dbase", listArgv[1])) {
	    result = RatInsertMsg(interp, msgPtr, listArgv[3], listArgv[5],
		    listArgv[4]);
	    free(listArgv);
	    return result;
	}

	/*
	 * If the destination is a dynamic folder then we have to do some
	 * magic.
	 */
	if (!strcmp("dynamic", listArgv[1])) {
	    char *name, **newArgv;

	    name = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		    (ClientData)msgPtr, RAT_FOLDER_MAIL_REAL, 0);
	    if (!name || !strlen(name)) {
		name = currentMailboxName;
	    }
	    newArgv = (char**)malloc(3*sizeof(char*)+strlen(listArgv[0])+
		    strlen(listArgv[3])+strlen(name)+3);
	    newArgv[0] = (char*)newArgv+3*sizeof(char*);
	    strcpy(newArgv[0], listArgv[0]);
	    newArgv[1] = "file";
	    newArgv[2] = "";
	    newArgv[3] = newArgv[0]+strlen(newArgv[0])+1;
	    strcpy(newArgv[3], listArgv[3]);
	    cPtr = newArgv[3]+strlen(newArgv[3]);
	    *cPtr++ = '/';
	    for (i=0; name[i] && name[i] != '@'; i++) {
		*cPtr++ = name[i];
	    }
	    *cPtr = '\0';
	    free(listArgv);
	    listArgv = newArgv;
	}

	/*
	 * Try to create nonexisting files
	 */
	if (!strcmp("file", listArgv[1]) && stat(listArgv[3], &sbuf)) {
	    int mode, fd;

	    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
		    TCL_GLOBAL_ONLY), &mode);
	    fd = open(listArgv[3], O_CREAT, mode);
	    close(fd);
	}

	/*
	 * Try the simple (but common) case where both source
	 * and destination are c-client messages. 
	 */
	folder = RatLindex(interp, listArgv[3], 0);
	if (RAT_CCLIENT_MESSAGE == msgPtr->type
		&& RatStdEasyCopyingOK(msgPtr, listArgv[1], folder,
			(listArgc > 4 ? listArgv[4] : NULL))) {
	    free(listArgv);
	    return RatStdMessageCopy(interp, msgPtr, folder);
	}

	/*
	 * We will need the message text so please retrieve it
	 */
	Tcl_DStringInit(&ds);
	RatMessageGet(interp, msgPtr, &ds, flags, date);
	INIT(&string,mail_string,Tcl_DStringValue(&ds),Tcl_DStringLength(&ds));
	stream = OpenStdFolder(interp, folder, listArgv[1], listArgv[4], NULL);
	free(listArgv);
	if (stream) {
	    if (NULL != (cPtr = strstr(flags, RAT_FLAGGED_STR))) {
		i = strlen(RAT_FLAGGED_STR);
		if (flags == cPtr) {
		    if (' ' == cPtr[i]) {
			i++;
		    }
		} else {
		    cPtr--;
		    i++;
		}
		strcpy(cPtr, cPtr+i);
	    }
	    if (!mail_append_full(stream, folder, flags, date, &string)){
		CloseStdFolder(interp, stream);
		Tcl_SetResult(interp, "mail_append failed", TCL_STATIC);
		return TCL_ERROR;
	    }
	    CloseStdFolder(interp, stream);
	}
	return TCL_OK;

    } else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0)
	    && (length > 1)) {
	ListExpression *exprPtr;
	char *result;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " list format\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (exprPtr = RatParseList(argv[2]))) {
	    Tcl_SetResult(interp, "Illegal list format", TCL_STATIC);
	    return TCL_ERROR;
	}
	result = RatDoList(interp, exprPtr,
		messageProcInfo[msgPtr->type].getInfoProc,
		(ClientData)msgPtr, 0);
	Tcl_ResetResult(interp);
	Tcl_AppendElement(interp, result);
	RatFreeListExpression(exprPtr);
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be one of header, body, rawBody reply, or get",
		(char*)NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageGetHeader --
 *
 *      Gets the header of a message
 *
 * Results:
 *	The header is reutned som a list in interp->result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
int
RatMessageGetHeader(Tcl_Interp *interp, char *srcHeader, char *level)
{
    char *header, *element, *listArgv[2];
    char *dstPtr, *srcPtr = srcHeader;
    Tcl_DString ds;

    if (!srcHeader) {
	RatLog(interp, RAT_FATAL, interp->result, 0);
	exit(1);
    }
    header = (char*) ckalloc (strlen(srcHeader)+1);
    if (!strncmp("From ", srcPtr, 5)) {
	while ('\n' != *srcPtr) {
	    srcPtr++;
	}
	if ('\r' == *(++srcPtr)) {
	    srcPtr++;
	}
    }
    Tcl_DStringInit(&ds);
    while (*srcPtr) {
	listArgv[0] = dstPtr = header;
	while (*srcPtr && ':' != *srcPtr && ' ' != *srcPtr) {
	    *dstPtr++ = *srcPtr++;
	}
	*dstPtr = '\0';
	listArgv[1] = ++dstPtr;
	do {
	    srcPtr++;
	} while (' ' == *srcPtr || '\t' == *srcPtr);
	do {
	    for (; *srcPtr && '\n' != *srcPtr; srcPtr++) {
		if ('\r' != *srcPtr) {
		    *dstPtr++ = *srcPtr;
		}
	    }
	    while ('\n' == *srcPtr || '\r' == *srcPtr) {
		srcPtr++;
	    }
	} while (*srcPtr && (' ' == *srcPtr || '\t' == *srcPtr));
	*dstPtr = '\0';
	if (!strcmp(level, "full")) {
	    listArgv[1] = RatDecodeHeader(interp, listArgv[1]);
	    listArgv[1] = RatDecodeHeaderFull(interp, listArgv[1]);
	} else if (!strcmp(level, "split")) {
	    listArgv[1] = RatDecodeHeader(interp, listArgv[1]);
	}
	element = Tcl_Merge(2, listArgv);
	Tcl_DStringAppendElement(&ds, element);
	ckfree(element);
    }
    ckfree(header);
    Tcl_DStringResult(interp, &ds);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageSetIndex --
 *
 *      Notifies the given message that its index in the folder
 *	has changed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The messages MessageInfo is changed.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatMessageSetIndex(Tcl_Interp *interp, char *cmdName, int index)
{
    MessageInfo *msgPtr;
    Tcl_CmdInfo info;

    Tcl_GetCommandInfo(interp, cmdName, &info);
    msgPtr = (MessageInfo*)info.clientData;
    (*messageProcInfo[msgPtr->type].setIndexProc)(msgPtr, index);
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageDelete --
 *
 *      Deletes the given message.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The message and all its bodyparts are deleted from the interpreter
 *	and all the structures are freed.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatMessageDelete(Tcl_Interp *interp, char *msgName)
{
    Tcl_CmdInfo cmdInfo;
    MessageInfo *msgPtr;
    char buf[256];

    if (0 == Tcl_GetCommandInfo(interp, msgName, &cmdInfo)) {
	Tcl_AppendResult(interp, "No such message: ", msgName, NULL);
	return TCL_ERROR;
    }
    msgPtr = (MessageInfo*)cmdInfo.clientData;

    (*messageProcInfo[msgPtr->type].msgDeleteProc)(msgPtr);
    if (msgPtr->bodyInfoPtr) {
	if (msgPtr->bodyInfoPtr->altPtr) {
	    RatBodyDelete(interp, msgPtr->bodyInfoPtr->altPtr);
	}
	if (msgPtr->bodyInfoPtr->decodedTextPtr) {
	    Tcl_DStringFree(msgPtr->bodyInfoPtr->decodedTextPtr);
	    free(msgPtr->bodyInfoPtr->decodedTextPtr);
	}
	if (msgPtr->bodyInfoPtr->secPtr) {
	    RatBodyDelete(interp, msgPtr->bodyInfoPtr->secPtr);
	} else {
	    RatBodyDelete(interp, msgPtr->bodyInfoPtr);
	}
    }
    sprintf(buf, "msgInfo_%s", msgPtr->name);
    Tcl_UnsetVar(interp, buf, TCL_GLOBAL_ONLY);
    Tcl_DeleteCommand(interp, msgName);
    ckfree(msgPtr->name);
    ckfree(msgPtr);

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * CreateBodyInfo --
 *
 *      Create and somewhat initialize a BodyInfo structure.
 *
 * Results:
 *	A pointer to a BodyInfo structure.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
BodyInfo*
CreateBodyInfo(MessageInfo *msgPtr)
{
    BodyInfo *bodyInfoPtr;
    int pad = sizeof(char*) - sizeof(BodyInfo)%sizeof(char*);

    if (sizeof(char*) == pad) {
	pad = 0;
    }

    bodyInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+pad+16);

    bodyInfoPtr->cmdName = (char*)bodyInfoPtr + pad + sizeof(BodyInfo);

    sprintf(bodyInfoPtr->cmdName, "RatBody%d", numBodies++);
    bodyInfoPtr->firstbornPtr = NULL;
    bodyInfoPtr->nextPtr = NULL;
    bodyInfoPtr->containedEntity = NULL;
    bodyInfoPtr->type = msgPtr->type;
    bodyInfoPtr->msgPtr = msgPtr;
    bodyInfoPtr->secPtr = NULL;
    bodyInfoPtr->altPtr = NULL;
    bodyInfoPtr->decodedTextPtr = NULL;
    bodyInfoPtr->encoded = 0;
    bodyInfoPtr->sigStatus = RAT_UNSIGNED;
    bodyInfoPtr->pgpOutput = NULL;
    return bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * RatBodyCmd --
 *
 *      Main bodypart entity procedure. This routine implements the
 *	bodypart commands mentioned in ../INTERFACE.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	many.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatBodyCmd(ClientData clientData,Tcl_Interp *interp,int argc,char *argv[])
{
    BodyInfo *bodyInfoPtr = (BodyInfo*) clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    unsigned long length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "children", length) == 0)
	    && (length > 2)) {
	BodyInfo *partInfoPtr;
	Tcl_DString resultDS;

	if (TYPEMULTIPART != bodyPtr->type) {
	    return TCL_OK;
	}
	Tcl_DStringInit(&resultDS);
	(*messageProcInfo[bodyInfoPtr->type].makeChildrenProc)
		(interp, bodyInfoPtr);
	for (partInfoPtr = bodyInfoPtr->firstbornPtr; partInfoPtr;
		partInfoPtr = partInfoPtr->nextPtr) {
	    RatPGPBodyCheck(interp, messageProcInfo, &partInfoPtr);
	    Tcl_CreateCommand(interp, partInfoPtr->cmdName, RatBodyCmd,
		    (ClientData) partInfoPtr, NULL);
	    Tcl_DStringAppendElement(&resultDS, partInfoPtr->cmdName);
	}
	Tcl_DStringResult(interp, &resultDS);
	return TCL_OK;

    } else if ((c == 'm') && (strncmp(argv[1], "message", length) == 0)
	    && (length > 1)) {
	char *body;

	if (!bodyInfoPtr->containedEntity) {
	    if (TYPEMESSAGE != bodyPtr->type &&
		    !strcasecmp(bodyPtr->subtype, "rfc822")) {
		Tcl_SetResult(interp, "Not an message/rfc822 bodypart",
			TCL_STATIC);
		return TCL_ERROR;
	    }
	    body = (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
		    (bodyInfoPtr, &length);
	    if (body && *body) {
		bodyInfoPtr->containedEntity =
			RatFrMessageCreate(interp, body, length, NULL);
		Tcl_SetResult(interp, bodyInfoPtr->containedEntity, TCL_STATIC);
	    }
	} else {
	    Tcl_SetResult(interp, bodyInfoPtr->containedEntity, TCL_STATIC);
	}
	return TCL_OK;

    } else if ((c == 't') && (strncmp(argv[1], "type", length) == 0)
	    && (length > 1)) {
	Tcl_AppendElement(interp, body_types[bodyPtr->type]);
	if (bodyPtr->subtype) {
	    Tcl_AppendElement(interp, bodyPtr->subtype);
	} else {
	    Tcl_AppendElement(interp, "");
	}
	return TCL_OK;

    } else if ((c == 'p') && (strncmp(argv[1], "params", length) == 0)) {
	PARAMETER *parameter;
	Tcl_DString result;

	Tcl_DStringInit(&result);
	for (parameter = bodyPtr->parameter; parameter;
		parameter = parameter->next) {
	    Tcl_DStringStartSublist(&result);
	    Tcl_DStringAppendElement(&result, parameter->attribute);
	    Tcl_DStringAppendElement(&result, parameter->value);
	    Tcl_DStringEndSublist(&result);
	}
	Tcl_DStringResult(interp, &result);
	return TCL_OK;

    } else if ((c == 'p') && (strncmp(argv[1], "parameter", length) == 0)) {
	PARAMETER *parameter;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " parameter attribute\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (parameter = bodyPtr->parameter; parameter;
		parameter = parameter->next) {
	    if ( 0 == strcasecmp(argv[2], parameter->attribute)) {
		Tcl_SetResult(interp, parameter->value, TCL_VOLATILE);
		break;
	    }
	}
	return TCL_OK;

    } else if ((c == 'i') && (strncmp(argv[1], "id", length) == 0)
	    && (length > 1)) {
	if (bodyPtr->id) {
	    Tcl_SetResult(interp, bodyPtr->id, TCL_VOLATILE);
	}
	return TCL_OK;

    } else if ((c == 'd') && (strncmp(argv[1], "description", length) == 0)
	    && (length > 1)) {
	if (bodyPtr->description) {
	    Tcl_SetResult(interp, bodyPtr->description, TCL_VOLATILE);
	}
	return TCL_OK;

    } else if ((c == 's') && (strncmp(argv[1], "size", length) == 0)
	    && (length > 1)) {
	sprintf(interp->result, "%ld", bodyPtr->size.bytes);
	return TCL_OK;

    } else if ((c == 'l') && (strncmp(argv[1], "lines", length) == 0)
	    && (length > 1)) {
	sprintf(interp->result, "%ld", bodyPtr->size.lines);
	return TCL_OK;

    } else if ((c == 'e') && (strncmp(argv[1], "encoding", length) == 0)
	    && (length > 1)) {
	char *enc;

	switch(bodyPtr->encoding) {
	case ENC7BIT:		enc = "7bit"; break;
	case ENC8BIT:		enc = "8bit"; break;
	case ENCBINARY:		enc = "binary"; break;
	case ENCBASE64:		enc = "base64"; break;
	case ENCQUOTEDPRINTABLE:enc = "quoted-printable"; break;
	default:		enc = "unkown"; break;
	}
	Tcl_SetResult(interp, enc, TCL_STATIC);
	return TCL_OK;

    } else if ((c == 'd') && (strncmp(argv[1], "data", length) == 0)
	    && (length > 1)) {
	unsigned char *body, *selection;
	char *isCharset = "us-ascii", *toCharset = "us-ascii", *alias;
	int encoded, outLength;
	PARAMETER *parameter;
	RatEncoding encoding;

	if (4 != argc && 3 != argc) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " data encoded ?charset?\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[2], &encoded)) {
	    return TCL_ERROR;
	}
	if (4 == argc) {
	    for (parameter = bodyPtr->parameter; parameter;
		    parameter = parameter->next) {
		if ( 0 == strcasecmp("charset", parameter->attribute)) {
		    isCharset = parameter->value;
		    toCharset = argv[3];
		}
	    }
	    if ((alias = Tcl_GetVar2(interp, "charsetAlias", isCharset,
		    TCL_GLOBAL_ONLY))) {
		isCharset = alias;
	    }
	    if ((alias = Tcl_GetVar2(interp, "charsetAlias", toCharset,
		    TCL_GLOBAL_ONLY))) {
		toCharset = alias;
	    }
	}

	switch(bodyPtr->encoding) {
	case ENC7BIT:		encoding = RAT_7BIT; break;
	case ENC8BIT:		encoding = RAT_8BIT; break;
	case ENCBINARY:		encoding = RAT_BINARY; break;
	case ENCBASE64:		encoding = RAT_BASE64; break;
	case ENCQUOTEDPRINTABLE:encoding = RAT_QP; break;
	default:		encoding = RAT_UNKOWN; break;
	}
	body = (unsigned char *)
		(*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
		(bodyInfoPtr, &length);
	if (body) {
	    if (encoded) {
		selection = (unsigned char*)malloc(length+1);
		memcpy(selection, body, length);
		selection[length] = '\0';
		Tcl_SetResult(interp, (char*)selection, TCL_DYNAMIC);
	    } else {
		Tcl_SetResult(interp, (char*)RatDecode(interp, body, length,
			encoding, &outLength, isCharset, toCharset),
			TCL_DYNAMIC);
	    }
	    return TCL_OK;
	} else {
	    Tcl_SetResult(interp, "[Body not available]\n", TCL_STATIC);
	    return TCL_OK;
	}

    } else if ((c == 's') && (strncmp(argv[1], "saveData", length) == 0)
	    && (length > 1)) {
	int encoded, convertNL;
	Tcl_Channel channel;

	if (5 != argc) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " saveData fileId encoded convertNL\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (channel = Tcl_GetChannel(interp, argv[2], NULL))) {
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[3], &encoded)) {
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[4], &convertNL)) {
	    return TCL_ERROR;
	}
	return RatBodySave(interp, channel, bodyInfoPtr, encoded, convertNL);

    } else if ((c == 'd') && (strncmp(argv[1], "dsn", length) == 0)
	    && (length > 1)) {
	char *body;

	if (TYPEMESSAGE != bodyPtr->type &&
		!strcasecmp(bodyPtr->subtype, "delivery-status")) {
	    Tcl_SetResult(interp, "Not an message/delivery-status bodypart",
		    TCL_STATIC);
	    return TCL_ERROR;
	}
	body = (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
		(bodyInfoPtr, &length);
	if (*body) {
	    return RatDSNExtract(interp, body, length);
	} else {
	    Tcl_SetResult(interp, "No body", TCL_STATIC);
	    return TCL_ERROR;
	}

    } else if ((c == 'g') && (strncmp(argv[1], "getShowCharset", length) == 0)
	    && (length > 7)) {
	char *charset = "us-ascii", *alias;
	PARAMETER *parmPtr;

	if (TYPETEXT != bodyPtr->type) {
	    Tcl_AppendElement(interp, "good");
	    Tcl_AppendElement(interp, "us-ascii");
	    return TCL_OK;
	}
	for (parmPtr = bodyPtr->parameter; parmPtr; parmPtr = parmPtr->next) {
	    if (!strcasecmp(parmPtr->attribute, "charset")) {
		charset = parmPtr->value;
		break;
	    }
	}

	charset = cpystr(charset);
	lcase(charset);

	/*
	 * - See if this charset is an alias and resolve that if so.
	 * - Check if this is a charset we do have a font for
	 *   return good and the charset in that case.
	 * - Check if this is a character set we know about and convert.
	 *   return lose if that is the case.
	 * - return none.
	 */
	if ((alias=Tcl_GetVar2(interp,"charsetAlias",charset,TCL_GLOBAL_ONLY))){
	    free(charset);
	    charset = cpystr(alias);
	}
	if (Tcl_GetVar2(interp, "fontEncoding", charset, TCL_GLOBAL_ONLY)) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendElement(interp, "good");
	    Tcl_AppendElement(interp, charset);
	    free(charset);
	    return TCL_OK;
	}
	/*
	 * This converting part is not implemented yet
	 */
	Tcl_ResetResult(interp);
	Tcl_AppendElement(interp, "none");
	Tcl_AppendElement(interp, "");
	free(charset);
	return TCL_OK;

    } else if ((c == 'f') && (strncmp(argv[1], "findShowCommand", length) == 0)
	    && (length > 1)) {
	return RatMcapFindCmd(interp, bodyInfoPtr);

    } else if ((c == 'f') && (strncmp(argv[1], "filename", length) == 0)
	    && (length > 1)) {
	PARAMETER *parmPtr;

	for (parmPtr = bodyPtr->parameter; parmPtr; parmPtr = parmPtr->next) {
	    if (!strcasecmp(parmPtr->attribute, "filename")
		    || !strcasecmp(parmPtr->attribute, "name")) {
		Tcl_SetResult(interp, parmPtr->value, TCL_VOLATILE);
		return TCL_OK;
	    }
	}
	if (bodyPtr->description && !strchr(bodyPtr->description, ' ')) {
	    Tcl_SetResult(interp, bodyPtr->description, TCL_VOLATILE);
	    return TCL_OK;
	}
	return TCL_OK;

    } else if ((c == 'e') && (strncmp(argv[1], "encoded", length) == 0)
	    && (length > 1)) {
	sprintf(interp->result, "%d", bodyInfoPtr->encoded);
	return TCL_OK;

    } else if ((c == 's') && (strncmp(argv[1], "sigstatus", length) == 0)
	    && (length > 1)) {
	char *status = NULL;

	switch (bodyInfoPtr->sigStatus) {
	case RAT_UNSIGNED:	status = "pgp_none"; break;
	case RAT_UNCHECKED:	status = "pgp_unchecked"; break;
	case RAT_SIG_GOOD:	status = "pgp_good"; break;
	case RAT_SIG_BAD:	status = "pgp_bad"; break;
	}
	Tcl_SetResult(interp, status, TCL_STATIC);
	return TCL_OK;

    } else if ((c == 'c') && (strncmp(argv[1], "checksig", length) == 0)
	    && (length > 2)) {
	RatPGPChecksig(interp, messageProcInfo, bodyInfoPtr);
	return TCL_OK;

    } else if ((c == 'g') && (strncmp(argv[1], "getPGPOutput", length) == 0)
	    && (length > 3)) {
	if (bodyInfoPtr->pgpOutput) {
	    Tcl_SetResult(interp, Tcl_DStringValue(bodyInfoPtr->pgpOutput),
		    TCL_VOLATILE);
	} else {
	    Tcl_ResetResult(interp);
	}
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be one of children, message, type, parameter, id",
		", description, data, saveData, dsn, getShowCharset",
		", getShowCommand, filename, encoded, sigstatus, checksig",
		" or getPGPOutput", NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatBodyDelete --
 *
 *      Deletes the given body.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The bodypart and all its siblings are deleted from the interpreter
 *	and all the structures are freed.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatBodyDelete(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    BodyInfo *siblingInfoPtr, *nextSiblingInfoPtr;
    Tcl_DeleteCommand(interp, bodyInfoPtr->cmdName);
    siblingInfoPtr = bodyInfoPtr->firstbornPtr;
    (*messageProcInfo[bodyInfoPtr->type].bodyDeleteProc)(bodyInfoPtr);
    while (siblingInfoPtr) {
	nextSiblingInfoPtr = siblingInfoPtr->nextPtr;
	RatBodyDelete(interp, siblingInfoPtr);
	siblingInfoPtr = nextSiblingInfoPtr;
    }
    if (bodyInfoPtr->containedEntity) {
	RatMessageDelete(interp, bodyInfoPtr->containedEntity);
    }
    if (bodyInfoPtr->pgpOutput) {
	free(bodyInfoPtr->pgpOutput);
    }
    ckfree(bodyInfoPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageSave --
 *
 *      Saves the given message to the given file.
 *
 * Results:
 *	The message is appended to the given file.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatMessageSave(Tcl_Interp *interp, Tcl_Channel channel, MessageInfo *msgPtr)
{
    Tcl_DString ds;
    char *data;

    Tcl_DStringInit(&ds);
    data = (*messageProcInfo[msgPtr->type].getEnvelopeProc)(interp, msgPtr);
    RatStringPuts((void*)&ds, data);
    Tcl_Write(channel, Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));
    Tcl_DStringSetLength(&ds, 0);
    data = (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr);
    RatStringPuts((void*)&ds, data);
    Tcl_Write(channel, Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));
    Tcl_DStringSetLength(&ds, 0);
    Tcl_Write(channel, "\n", 1);
    data = (*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr);
    RatStringPuts((void*)&ds, data);
    Tcl_Write(channel, Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));
    Tcl_DStringFree(&ds);
    Tcl_Write(channel, "\n", 1);
}

/*
 *----------------------------------------------------------------------
 *
 * RatMessageGet --
 *
 *      Retrieves a message in textual form. The text is placed in the
 *	supplied Tcl_DString.
 *
 * Results:
 *	No result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatMessageGet(Tcl_Interp *interp, MessageInfo *msgPtr, Tcl_DString *ds,
	      char *flags, char *date)
{
    char *data;
    int seen;

    data = (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr);
    Tcl_DStringAppend(ds, data, strlen(data));
    seen  = (*msgPtr->folderInfoPtr->getFlagProc)(msgPtr->folderInfoPtr,
	    interp, msgPtr->msgNo, RAT_SEEN);
    data = (*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr);
    Tcl_DStringAppend(ds, data, strlen(data));
    if (!seen) {
        (*msgPtr->folderInfoPtr->setFlagProc)(msgPtr->folderInfoPtr,
	    interp, msgPtr->msgNo, RAT_SEEN, 0);
    }
    if (flags) {
	data = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_FLAGS, 0);
	strcpy(flags, data);
	data = (*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_DATE_IMAP4, 0);
	strcpy(date, data);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatInsertCmd --
 *
 *      Inserts the given message into the database
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatInsertCmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    Tcl_CmdInfo cmdInfo;
    MessageInfo *msgPtr;

    if (argc != 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" msgId keywords exDate exType\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (0 == Tcl_GetCommandInfo(interp, argv[1], &cmdInfo)) {
	Tcl_AppendResult(interp, "No such message: ", argv[1], NULL);
	return TCL_ERROR;
    }
    msgPtr = (MessageInfo*)cmdInfo.clientData;
    return RatInsertMsg(interp, msgPtr, argv[2], argv[3], argv[4]);
}


/*
 *----------------------------------------------------------------------
 *
 * RatInsertMsg --
 *
 *      Inserts the given message into the database
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatInsertMsg (Tcl_Interp *interp, MessageInfo *msgPtr, char *keywords,
	char *exDate, char *exType)
{
    char *to, *from, *cc, *subject, *flags, **listArgv, **elemArgv;
    MESSAGECACHE elt;
    int date = 0, listArgc, elemArgc, exTime;
    char *eFrom, *header, *body;
    Tcl_DString dString;
    int result, i;
    struct tm tm;

    to = from = cc = subject = flags = NULL;
    if (TCL_OK != RatMessageGetHeader(interp,
	    (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr),
	    "full")) {
	return TCL_ERROR;
    }
    Tcl_SplitList(interp, interp->result, &listArgc, &listArgv);
    for (i=0; i<listArgc; i++) {
	Tcl_SplitList(interp, listArgv[i], &elemArgc, &elemArgv);
	if (!strcasecmp(elemArgv[0], "to")) {
	    to = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "from")) {
	    from = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "cc")) {
	    cc = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "subject")) {
	    subject = cpystr(elemArgv[1]);
	} else if (!strcasecmp(elemArgv[0], "status") ||
		   !strcasecmp(elemArgv[0], "x-status")) {
	    if (flags) {
		flags=(char*)realloc(flags,strlen(flags)+strlen(elemArgv[1])+1);
		strcpy(&flags[strlen(flags)], elemArgv[1]);
	    } else {
		flags = cpystr(elemArgv[1]);
	    }
	} else if (!strcasecmp(elemArgv[0], "date")) {
	    if (T == mail_parse_date(&elt, elemArgv[1])) {
		tm.tm_sec = elt.seconds;
		tm.tm_min = elt.minutes;
		tm.tm_hour = elt.hours;
		tm.tm_mday = elt.day;
		tm.tm_mon = elt.month - 1;
		tm.tm_year = elt.year+70;
		tm.tm_wday = 0;
		tm.tm_yday = 0;
		tm.tm_isdst = -1;
		date = (int)mktime(&tm);
	    } else {
		date = 0;
	    }
	}
	ckfree(elemArgv);
    }
    ckfree(listArgv);
    if (0 == date) {
	date = atoi((*messageProcInfo[msgPtr->type].getInfoProc)(interp,
		(ClientData)msgPtr, RAT_FOLDER_DATE_N, 0));
    }
    Tcl_DStringInit(&dString);
    eFrom = (*messageProcInfo[msgPtr->type].getEnvelopeProc)(interp, msgPtr);
    header = (*messageProcInfo[msgPtr->type].getHeadersProc)(interp, msgPtr);
    Tcl_DStringAppend(&dString, header, strlen(header));
    body = (*messageProcInfo[msgPtr->type].fetchTextProc)(interp, msgPtr);
    Tcl_DStringAppend(&dString, body, strlen(body));
    Tcl_ResetResult(interp);
    exTime = atoi(exDate);
    if (!strcmp("none", exType)) {
	exTime = 0;
    }
    result = RatDbInsert(interp, to, from, cc, subject, date, flags, keywords,
	    exTime, exType, eFrom, Tcl_DStringValue(&dString),
	    Tcl_DStringLength(&dString));
    Tcl_DStringFree(&dString);
    ckfree(to);
    ckfree(from);
    ckfree(cc);
    ckfree(subject);
    ckfree(flags);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * RatParseList --
 *
 *      Parse a list expression (almost like a printf format string)
 *
 * Results:
 *	A structure representing the parsed expression, or null if
 *	there is a syntax error in the format string.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

ListExpression*
RatParseList(char *format)
{
    ListExpression *expPtr;
    int i, w, expIndex, bufLen, num;
    char buf[1024];

    for(i=num=0; '\0' != format[i]; i++) {
	if ('%' == format[i] && format[i+1] && '%' != format[i+1]) {
	    while (format[++i] && ('-' == format[i] || isdigit(format[i])));
	    if (!strchr("snmrRbBdDSi", format[i])) {
		return NULL;
	    }
	    num++;
	}
    }
    expPtr = (ListExpression*)malloc(sizeof(ListExpression));
    expPtr->size = num;
    expPtr->preString = (char**)malloc(num*sizeof(char*));
    expPtr->typeList =(RatFolderInfoType*)malloc(num*sizeof(RatFolderInfoType));
    expPtr->fieldWidth = (int*)malloc(num*sizeof(int));
    expPtr->leftJust = (int*)malloc(num*sizeof(int));
    for (i = expIndex = bufLen = 0; format[i]; i++) {
	if ('%' == format[i]) {
	    if ('%' == format[++i]) {
		buf[bufLen++] = format[i];
		continue;
	    }
	    buf[bufLen] = '\0';
	    expPtr->preString[expIndex] = cpystr(buf);
	    if ('-' == format[i]) {
		expPtr->leftJust[expIndex] = 1;
		i++;
	    } else {
		expPtr->leftJust[expIndex] = 0;
	    }
	    w=0;
	    while (isdigit(format[i])) {
		w = w*10+format[i++]-'0';
	    }
	    expPtr->fieldWidth[expIndex] = w;
	    switch(format[i]) {
	    case 's': expPtr->typeList[expIndex++] = RAT_FOLDER_SUBJECT; break;
	    case 'n': expPtr->typeList[expIndex++] = RAT_FOLDER_NAME; break;
	    case 'm': expPtr->typeList[expIndex++] = RAT_FOLDER_MAIL; break;
	    case 'r': expPtr->typeList[expIndex++] = RAT_FOLDER_NAME_RECIPIENT;
		    break;
	    case 'R': expPtr->typeList[expIndex++] = RAT_FOLDER_MAIL_RECIPIENT;
		    break;
	    case 'b': expPtr->typeList[expIndex++] = RAT_FOLDER_SIZE; break;
	    case 'B': expPtr->typeList[expIndex++] = RAT_FOLDER_SIZE_F; break;
	    case 'd': expPtr->typeList[expIndex++] = RAT_FOLDER_DATE_F; break;
	    case 'D': expPtr->typeList[expIndex++] = RAT_FOLDER_DATE_N; break;
	    case 'S': expPtr->typeList[expIndex++] = RAT_FOLDER_STATUS; break;
	    case 'i': expPtr->typeList[expIndex++] = RAT_FOLDER_INDEX; break;
	    }
	    bufLen = 0;
	} else {
	    buf[bufLen++] = format[i];
	}
    }
    if (bufLen) {
	buf[bufLen] = '\0';
	expPtr->postString = cpystr(buf);
    } else {
	expPtr->postString = NULL;
    }
    return expPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFreeListExpression --
 *
 *      Frees all memory associated with a list expression.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Some memory is freed.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatFreeListExpression(ListExpression *exPtr)
{
    int i;

    for (i=0; i<exPtr->size; i++) {
	free(exPtr->preString[i]);
    }
    free(exPtr->preString);
    free(exPtr->typeList);
    free(exPtr->fieldWidth);
    free(exPtr->leftJust);
    if (exPtr->postString) {
	free(exPtr->postString);
    }
    free(exPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * RatDoList --
 *
 *      Print the list information about a message.
 *
 * Results:
 *	A pointer to a static area which contains the resulting string.
 *	This area is safe until the next call to this function.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatDoList(Tcl_Interp *interp, ListExpression *exprPtr, RatInfoProc *infoProc,
	ClientData clientData, int index)
{
    static Tcl_DString ds;
    static int initialized = 0;
    char *s, *format;
    int i, j, length;

    if (!initialized) {
	initialized = 1;
	Tcl_DStringInit(&ds);
    } else {
	Tcl_DStringSetLength(&ds, 0);
    }

    for (i=0; i<exprPtr->size; i++) {
	if (exprPtr->preString[i]) {
	    Tcl_DStringAppend(&ds, exprPtr->preString[i], -1);
	}
	s = (*infoProc)(interp, clientData, exprPtr->typeList[i], index);
	if (!s) {
	    for (j=0; j<exprPtr->fieldWidth[i]; j++) {
		Tcl_DStringAppend(&ds, " ", 1);
	    }
	    continue;
	}
	s = RatDecodeHeader(interp, s);
	s = RatDecodeHeaderFull(interp, s);
	if (exprPtr->fieldWidth[i]) {
	    if (strlen(s) > exprPtr->fieldWidth[i]) {
		Tcl_DStringAppend(&ds, s, exprPtr->fieldWidth[i]);
	    } else {
		length = Tcl_DStringLength(&ds);
		Tcl_DStringSetLength(&ds, length+exprPtr->fieldWidth[i]);
		if (exprPtr->leftJust[i]) {
		    format = "%-*s";
		} else {
		    format = "%*s";
		}
		sprintf(Tcl_DStringValue(&ds)+length, format,
			exprPtr->fieldWidth[i], s);
	    }
	} else {
	    Tcl_DStringAppend(&ds, s, -1);
	}
    }
    if (exprPtr->postString) {
	Tcl_DStringAppend(&ds, exprPtr->postString, -1);
    }
    return Tcl_DStringValue(&ds);
}


/*
 *----------------------------------------------------------------------
 *
 * RatBodySave --
 *
 *      Save a bodypart to an open channel.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatBodySave(Tcl_Interp *interp,Tcl_Channel channel, BodyInfo *bodyInfoPtr,
	    int encoded, int convertNL)
{
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    RatEncoding encoding;
    unsigned char *body;
    int result = 0, i;
    unsigned long length;

    if (NULL == (body = (unsigned char *)
	    (*messageProcInfo[bodyInfoPtr->type].fetchBodyProc)
	    (bodyInfoPtr, &length))) {
	Tcl_SetResult(interp, "[Body not available]\n", TCL_STATIC);
	return TCL_OK;
    }
    if (!encoded) {
	int outLength;

	switch(bodyPtr->encoding) {
	case ENC7BIT:		encoding = RAT_7BIT; break;
	case ENC8BIT:		encoding = RAT_8BIT; break;
	case ENCBINARY:		encoding = RAT_BINARY; break;
	case ENCBASE64:		encoding = RAT_BASE64; break;
	case ENCQUOTEDPRINTABLE:encoding = RAT_QP; break;
	default:		encoding = RAT_UNKOWN; break;
	}
	body = RatDecode(interp, body, length, encoding, &outLength,
		NULL, NULL);
	length = outLength;
    }
    if (convertNL) {
	/* 
	 * This isn't really elegant but since the channel is buffered
	 * we shouldn't suffer too badly.
	 */
	for (i=0; i<length && -1 != result; i++) {
	    if ('\r' == body[i] && '\n' == body[i+1]) {
		i++;
	    }
	    result = Tcl_Write(channel, (char*)(&body[i]), 1);
	}
    } else {
	result = Tcl_Write(channel, (char*)body, length);
    }
    if (!encoded) {
	ckfree(body);
    }
    if (-1 == result) {
	Tcl_AppendResult(interp, "error writing : ",
		Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * RatFindFirstText --
 *
 *      Finds the first text part of a message
 *
 * Results:
 *	A pointer to the BodyInfo of the first text part, or NULL if none
 *	are found.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static BodyInfo*
RatFindFirstText(BodyInfo *bodyInfoPtr)
{
    BodyInfo *b2Ptr;

    for (; NULL != bodyInfoPtr; bodyInfoPtr = bodyInfoPtr->nextPtr) {
	if (TYPETEXT == bodyInfoPtr->bodyPtr->type) {
	    return bodyInfoPtr;
	}
	if (bodyInfoPtr->firstbornPtr && (NULL !=
		(b2Ptr = RatFindFirstText(bodyInfoPtr->firstbornPtr)))) {
	    return b2Ptr;
	}
    }
    return NULL;
}
