/*
 * netplan access lists: read and write the access list file and verify
 * whether a file is accessible or not based on the access list. The
 * functions in this file are called from netplan.c. The functions are:
 *
 * acl_read	read an access list file into memory
 * acl_write	write memory to an access list file
 * acl_verify	verify the rights a client has on a file based on the list
 * acl_exit	shut down, release the list
 *
 * The syntax for the access list file is a sequence of rules like this:
 *
 *	name | * : [permit | deny] [read] [write] [delete] [netmask n.n.n.n]
 *	           [[user | group | host] data [data ...]]
 *
 * <name> is the file the rule applies to; an asterisk (*) applies to all
 * files. Permit is the default. If none of read,write,delete are specified,
 * all three are the default. The netmask applies to the client's IP address,
 * it is derived from that address according to the usual rules if missing.
 * <data> is one or more numerical UIDs, numerical GIDs, or numerical n.n.n.n
 * IP addresses for user, group, and host rules, respectively. User is the
 * default. Trailing n=0 IP address components are not assumed to denote nets,
 * use the netmask specifier for subnet masking. All whitespace is ignored.
 * Pound signs (#) introduce comments that extend to the end of the line.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#ifdef IBM
#include <sys/select.h>
#endif
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netdb.h>
#ifdef NEWSOS4
#include <machine/endian.h>
#endif
#include <netinet/in.h>
#include <netinet/tcp.h>
#include "netplan.h"

typedef enum {aUSER=0, aGROUP, aHOST} Mode;

typedef struct Acl {
	struct Acl	*next;		/* next access list entry in chain */
	char		*name;		/* applies to this file, 0=all */
	BOOL		deny;		/* TRUE=deny, FALSE=permit */
	BOOL		read;		/* permit/deny reading */
	BOOL		write;		/* permit/deny writing */
	BOOL		delete;		/* permit/deny deleting */
	Mode		mode;		/* check user, group, or host? */
	unsigned int	netmask;	/* if aHOST, netmask, 0=default */
	int		ndata;		/* number of uint's in data list */
	int		maxdata;	/* allocated # of uin's in data list */
	unsigned int	*data;		/* uids, gids, or IP addr list */
} Acl;

static Acl		*acl_root;	/* root of current access list */
static char		*acl_fname;	/* file name pf acl file opened */

static BOOL getword(FILE *, char *, int);
static unsigned int ipaddr_atoi(char *);
static char *ip_itoa(unsigned int);

#if 0					/* debugging only */
main(int argc, char **argv)
{
	int r, w, d;

	acl_read("acl");
	acl_write("acl.echo");
	acl_verify(&r, &w, &d, argv[1],
			atoi(argv[2]), atoi(argv[3]), ipaddr_atoi(argv[4]));
	printf("read %s, write %s, delete %s\n",
		r ? "yes" : "no", w ? "yes" : "no", d ? "yes" : "no");
	acl_exit();
}
#endif


/*
 * forget the previous access list and read a new one ffrom the given file.
 * This must be done before the first verify because an empty access list
 * means deny everything. Returns FALSE if reading fails; in this case the
 * access list is left empty even if the file was ok partially, on the
 * theory that it's better to deny everything than permitting too much.
 */

int acl_read(
	char		*fname)		/* access list file name to read */
{
	FILE		*fp;		/* open file */
	Acl		*tail = 0;	/* previous access list entry */
	Acl		*acl;		/* current new access list entry */
	char		word[256];	/* next word from input stream */
	int		phase = 0;	/* 0=name, 1=mode, 2=data */
	int		ret = TRUE;	/* return code */

	acl_exit();
	if (!(fp = fopen(fname, "r"))) {
		perror(fname);
		return(FALSE);
	}
	while (ret && getword(fp, word, sizeof(word))) {
		if (phase == 1) {				/* read, ... */
			if (*word >= '0' && *word <= '9') {
				if (!acl->read && !acl->write && !acl->delete)
					acl->read = acl->write = acl->delete =
									TRUE;
				phase = 2;
			} else {
				if (!strcmp(word, "permit"))
					acl->deny = FALSE;
				else if (!strcmp(word, "deny"))
					acl->deny = TRUE;
				else if (!strcmp(word, "read"))
					acl->read = TRUE;
				else if (!strcmp(word, "write"))
					acl->write = TRUE;
				else if (!strcmp(word, "delete"))
					acl->delete = TRUE;
				else if (!strcmp(word, "user"))
					acl->mode = aUSER;
				else if (!strcmp(word, "group"))
					acl->mode = aGROUP;
				else if (!strcmp(word, "host"))
					acl->mode = aHOST;
				else if (!strcmp(word, "netmask")) {
					getword(fp, word, sizeof(word));
					acl->netmask = ipaddr_atoi(word);
				} else
					phase = 0;
			}
		}
		if (phase == 2) {				/* data */
			if (*word < '0' || *word > '9') {
				phase = 0;

			} else if (acl->ndata <= acl->maxdata &&
				   !(acl->data = acl->maxdata
					? realloc(acl->data, acl->maxdata*3/2)
					: malloc(acl->maxdata = 16))) {
						perror(fname);
						ret = FALSE;
			} else
				acl->data[acl->ndata++] = acl->mode == aHOST
					? ipaddr_atoi(word)
					: atoi(word);
		}
		if (phase == 0) {				/* name : */
			if (!(acl = (Acl *)malloc(sizeof(Acl)))) {
				perror(fname);
				ret = FALSE;
				break;
			}
			memset(acl, 0, sizeof(Acl));
			if (tail)
				tail->next = acl;
			else
				tail = acl_root = acl;
			if (*word != '*' && !(acl->name = strdup(word))) {
				perror(fname);
				ret = FALSE;
			} else
			if (getword(fp, word, sizeof(word)) && *word != ':') {
				ret = FALSE;
				fprintf(stderr, "%s: no ':' after \"%s\"\n",
					fname, acl->name ? acl->name : "*");
			} else
				phase = 1;
		}
	}
	fclose(fp);
	if (!(acl_fname = strdup(fname))) {
		perror(fname);
		ret = FALSE;
	}
	if (!ret)
		acl_exit();
	return(ret);
}


/*
 * a subfunction of the previous, which reads the access list file word by
 * word. A word is either a sequence of one or more characters of the char
 * set [a-zA-Z0-9_.], or a single character that is not in the set and is
 * not a comma or a minus. Whitespace, comma, and minus are ignored. This
 * makes the access list file free-format, newlines are treated like spaces.
 * Pound signs (#) introduce comments that extend to the end of the line.
 */

#define ISBLANK(c) strchr(",- \t\n\r", c)
#define ISWORD(c) (c >= 'a' && c <= 'z' ||\
		   c >= 'A' && c <= 'Z' ||\
		   c >= '0' && c <= '9' || c == '_' || c == '.')

static BOOL getword(
	FILE		*fp,		/* file to read from */
	char		*word,		/* word buffer to read into */
	int		max)		/* size of word buffer, > 1 */
{
	char		*p = word;	/* next free char in word buffer */
	int		c;		/* current char read from file */

	for (;;) {
		while (ISBLANK(c = fgetc(fp)));
		if (c != '#')
			break;
		while ((c = fgetc(fp)) != EOF && c != '\n');
	}
	*p++ = c;
	while (--max > 1 && (c = fgetc(fp)) != EOF && ISWORD(c))
		*p++ = c;
	*p = 0;
	ungetc(c, fp);
	return(c != EOF);
}


/*
 * write an access list back to a given file, in the same format expected
 * by acl_read. If writing fails (probably because the file is not writable),
 * return FALSE. This function is currently not used because there is no way
 * to modify the list as it was read from the file originally, so why bother?
 */

int acl_write(
	char		*fname)		/* access list file name to write */
{
	FILE		*fp;		/* open file */
	Acl		*acl;		/* current new access list entry */
	int		i;		/* for scanning data list */

	if (!(fp = fopen(fname, "w"))) {
		perror(fname);
		return(FALSE);
	}
	fprintf(fp, "# netplan database access permissions file\n");
	fprintf(fp, "#\n");
	fprintf(fp, "# file|'*' ':' permit|deny\n");
	fprintf(fp, "#	[read] [write] [delete] [netmask n.n.n.n]\n");
	fprintf(fp, "#	[[user|group|host] n*|n.n.n.n*]\n");
	fprintf(fp, "#\n");

	for (acl=acl_root; acl; acl=acl->next) {
		fprintf(fp, "%s: %s", acl->name ? acl->name : "*",
				      acl->deny ? "deny" : "permit");
		if (acl->read)
			fprintf(fp, " read");
		if (acl->write)
			fprintf(fp, " write");
		if (acl->delete)
			fprintf(fp, " delete");
		if (acl->netmask)
			fprintf(fp, " netmask %s", ip_itoa(acl->netmask));
		if (acl->ndata)
			switch(acl->mode) {
			  case aUSER:	fprintf(fp, "\n\tuser");   break;
			  case aGROUP:	fprintf(fp, "\n\tgroup");  break;
			  case aHOST:	fprintf(fp, "\n\thost");   break;
			}
		for (i=0; i < acl->ndata; i++)
			if (acl->mode == aHOST)
				fprintf(fp, " %s", ip_itoa(acl->data[i]));
			else
				fprintf(fp, " %d", acl->data[i]);
		fprintf(fp, "\n");
	}
	fclose(fp);
	return(TRUE);
}


/*
 * simple conversion functions for IP addresses: 32-bit unsigned ints to and
 * from four-part dotted-decimal notation. Symbolic IP addresses are not
 * supported.
 */

static unsigned int ipaddr_atoi(
	char		*addr)		/* n.n.n.n string to convert */
{
	unsigned int	a=0, b=0, c=0, d=0;

	sscanf(addr, "%d.%d.%d.%d", &a, &b, &c, &d);
	return((a&255)<<24 | (b&255)<<16 | (c&255)<<8 | (d&255));
}


static char *ip_itoa(
	unsigned int	ip)		/* IP address to convert */
{
	static char	addr[16];	/* n.n.n.n address string buffer */

	sprintf(addr, "%d.%d.%d.%d", ip>>24, ip>>16&255, ip>>8&255, ip&255);
	return(addr);
}


/*
 * the reason why all of the above is done: return the rights that the user
 * with the user ID uid and the group ID gid, who connected from IP address
 * ip, has for <file>. The right to read, write, and delete are distinguished.
 * The access list is scanned for rules matching the file name and for default
 * rules (marked by * in the access list file). This is called whenever a
 * client opens a file.
 */

void acl_verify(
	int		*r,		/* set to TRUE if reading is ok */
	int		*w,		/* set to TRUE if writing is ok */
	int		*d,		/* set to TRUE if deleting is ok */
	char		*name,		/* file name to verify */
	unsigned int	uid,		/* user ID to verify */
	unsigned int	gid,		/* group ID to verify */
	unsigned int	ip)		/* IP address to verify */
{
	Acl		*acl;		/* current acl entry */
	unsigned int	netmask, nm;	/* default,current netmask for <ip> */
	int		i;		/* for searching data[] */

	*r = *w = *d = FALSE;
	netmask = !(ip & 0x80000000) ? 0xff000000
		: !(ip & 0x40000000) ? 0xffff0000
				     : 0xffffff00;
	for (acl=acl_root; acl; acl=acl->next) {
		if (acl->name && strcmp(acl->name, name))
			continue;
		switch(acl->mode) {
		  case aUSER:
			for (i=0; i < acl->ndata; i++)
				if (acl->data[i] == uid)
					break;
			break;

		  case aGROUP:
			for (i=0; i < acl->ndata; i++)
				if (acl->data[i] == gid)
					break;
			break;

		  case aHOST:
			nm = acl->netmask ? acl->netmask : netmask;
			for (i=0; i < acl->ndata; i++)
				if ((acl->data[i] & nm) == (ip & nm))
					break;
		}
		if (!acl->ndata || i < acl->ndata) {
			if (acl->read)	  *r = !acl->deny;
			if (acl->write)	  *w = !acl->deny;
			if (acl->delete)  *d = !acl->deny;
		}
	}
}


/*
 * shut down access lists. Release all memory. This is also done when the
 * access list file is re-read to keep rules from accumulating.
 */

void acl_exit(void)
{
	Acl		*acl, *next;	/* current and next acl entry */

	for (acl=acl_root; acl; acl=next) {
		next = acl->next;
		if (acl->name)
			free(acl->name);
		if (acl->data)
			free(acl->data);
		free(acl);
	}
	acl_root = 0;
}
