/* worklog */
/* program for tracking time spent on different projects */
/* Truxton Fulton */
/* March 27, 1995 */
/* gcc worklog.c -lncurses -o worklog */

/* Thanks to :
     Tim Newsome <drz@cybercomm.net> (25 Dec 1995)
     Egil Kvaleberg <egilk@sn.no> (5 Sep 1996)
     Mark Sebastian Fischer <Mark.Sebastian.Fischer@horz.technopark.gmd.de> (1 Nov 1996)
   for suggestions, bug reports, bug fixes, and improvements
*/

/*
  This set of defines is commented out because the directory (.worklog) should be part of the DEFAULT_DIR_ENVVAR
  
  #define         DEFAULT_LOG_FILE        ".worklog/worklog"
  #define         DEFAULT_CONFIG_FILE     ".worklog/projects"
*/

#define		DEFAULT_LOG_FILE	"time.log"
#define		DEFAULT_CONFIG_FILE	"projects"

#define VERSIONS "1.4"
#define STRLEN 2048

#define IDLE_X 5
#define IDLE_Y 22
#define CR 10

/* #include <ncurses/ncurses.h> */
#include <curses.h>

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <signal.h>
#include <unistd.h>

struct project_list
{
  char *name ;
  char key ;
  struct project_list *next ;
  long time ;
} *project_head=NULL ;

int num_projects ;
int x_update,y_update ;
struct project_list *project_update ;
char temps[STRLEN] ;
char starttimes[STRLEN] ;
char *log_filename ;
int update_skip_refresh ;

void read_config_file(filename)
     char *filename ;
{
  FILE *f ;
  char firstchar,colonchar ;
  struct project_list *project ;
  char names[STRLEN] ;
  int linenum ;

  num_projects=0 ;
  linenum=0 ;
  project=project_head ;
  f=fopen(filename,"r") ;
  if(f==NULL)
    {
      fprintf(stderr,"ERROR: cannot open project configuration file : %s\n",filename) ;
      perror("fopen") ;
      exit(-1) ;
    }
  while(!feof(f))
    {
      linenum++ ;
      for(firstchar=0;!feof(f) && firstchar<=13;)
	fscanf(f,"%c",&firstchar) ;
      if(!feof(f))
	{
	  if(firstchar=='#')
	    {
	      fscanf(f,"%*[^\n]") ;
	    }
	  else
	    {
	      /* must read colon seperator */
	      fscanf(f,"%c",&colonchar) ;
	      if(colonchar!=':')
		{
		  fprintf(stderr,"ERROR: second character in project line must be a colon.\n") ;
		  fprintf(stderr,"       on line %d is '%c'\n",linenum,colonchar) ;
		  exit(-1) ;
		}
	      project=(struct project_list *)malloc(sizeof(struct project_list)) ;
	      project->key=firstchar ;
	      project->next=project_head ;
	      project->time=0 ;
	      project_head=project ;
	      fscanf(f,"%[^\n]",names) ;
	      project->name=(char *)malloc(1+strlen(names)) ;
	      sprintf(project->name,"%s",names) ;
	      num_projects++ ;
	    }
	}
    }
  fclose(f) ;
}


void draw_main_screen()
{
  struct project_list *project ;
  int i ;

  sprintf(temps,"Worklog version %s",VERSIONS) ;
  attron(A_BOLD) ;
  mvaddstr(0,5,temps) ;
  attroff(A_BOLD) ;
  mvaddstr(1,1,"---------------------------") ;  /* this fixes a weirdness when using 'screen' !! */
  for(i=2,project=project_head;project!=NULL;project=project->next,i++)
    {
      move(i,5) ;
      sprintf(temps,"%c",project->key) ;
      attron(A_BOLD) ;
      mvaddstr(i,5,temps) ;
      attroff(A_BOLD) ;
      mvaddstr(i,7,project->name) ;
    }
  i++ ;
  move(i,0) ;
  clrtoeol() ;
  attron(A_BOLD) ;
  mvaddstr(i,3,"ESC") ;
  attroff(A_BOLD) ;
  mvaddstr(i,7,"Quit") ;
  i++ ;
  move(i,0) ;
  clrtoeol() ;
  i++ ;
  move(i,0) ;
  clrtoeol() ;
  i++ ;
  move(i,0) ;
  clrtoeol() ;
  i++ ;
  move(i,0) ;
  clrtoeol() ;
}


void alarm_handler(flag)
     int flag ;
{
  double seconds,minutes,hours ;

  signal(SIGALRM,alarm_handler) ;
  project_update->time+=1 ;
  if(!update_skip_refresh)
    {
      seconds=(double) project_update->time ;
      if(seconds<60.0)
	sprintf(temps,"%0.2f seconds",seconds) ;
      else
	{
	  minutes=seconds / 60.0 ;
	  if(minutes<60.0)
	    sprintf(temps,"%0.2f minutes",minutes) ;
	  else
	    {
	      hours = seconds / 3600.0 ;
	      sprintf(temps,"%0.2f hours",hours) ;
	    }
	}
      attron(A_UNDERLINE) ;
      mvaddstr(y_update,x_update,temps) ;
      attroff(A_UNDERLINE) ;
      mvaddstr(y_update,x_update+strlen(temps),"      ") ;
    }
}


void exit_handler(flag)
     int flag ;
{
  struct project_list *project ;
  FILE *f ;
  double seconds,minutes,hours ;
  time_t t ;

  f=fopen(log_filename,"a") ;
  if(f==NULL)
    {
      fprintf(stderr,"ERROR: cannot open project log file : %s\n",log_filename) ;
      perror("fopen") ;
      exit(-1) ;
    }

  fprintf(f,"-- Worklog summary begins : %s --\n",starttimes) ;
  for(project=project_head;project!=NULL;project=project->next)
    if(project->time>0)
      {
	seconds=(double) project->time ;
	if(seconds<60.0)
	  sprintf(temps,"%0.2f seconds",seconds) ;
	else
	  {
	    minutes=seconds / 60.0 ;
	    if(minutes<60.0)
	      sprintf(temps,"%0.2f minutes",minutes) ;
	    else
	      {
		hours = seconds / 3600.0 ;
		sprintf(temps,"%0.2f hours",hours) ;
	      }
	  }
	fprintf(f,"%s : total %s\n",project->name,temps) ;
      }
  time(&t) ;
  sprintf(temps,"%s",asctime(localtime(&t))) ;
  if(temps[strlen(temps)-1]<14)
    temps[strlen(temps)-1]=0 ;
  fprintf(f,"-- Worklog summary ends : %s --\n\n",temps) ;
  fclose(f) ;
  endwin() ;
  fprintf(stderr,"\n") ;
  exit(0) ;
}


char clock_on(key)
     char key ;
{
  struct project_list *project ;
  int i,j,found,quit ;
  char keypress ;
  struct itimerval timer_value ;
  int modification,success ;
  FILE *f,*fa ;
  time_t t ;
  long initial_time,delta_time ;
  double seconds,minutes,hours ;
  char comments[STRLEN] ;
  char specific_timefile[STRLEN] ;

  comments[0]=0 ;
  keypress=key ;
  for(found=0,i=2,project=project_head;!found && project!=NULL;project=project->next,i++)
    if(project->key==key)		     /* try to find exact match */
      {
	found=1;
	break;
      }
  
  if(!found)  
    for(found=0,i=2,project=project_head;!found && project!=NULL;project=project->next,i++)
      if(tolower(project->key)==tolower(key)) /* try to find
						 case-insensitive match */
	{
	  found=1;
	  key=project->key;		     /* remember exact key */
	  break;
	}
  
  if(found)
    {
      initial_time=project->time ;
      j=strlen(project->name)+10 ;
      y_update=i ;
      x_update=j ;
      attron(A_BOLD) ;
      mvaddstr(y_update,1,"->") ;
      attroff(A_BOLD) ;
      
      i=num_projects+3 ;
      attron(A_BOLD) ;
      mvaddstr(i,1,"SPACE") ;
      attroff(A_BOLD) ;
      mvaddstr(i,7,"Pause clock") ;
      i++ ;
      attron(A_BOLD) ;
      mvaddstr(i,4,"CR") ;
      attroff(A_BOLD) ;
      mvaddstr(i,7,"Stop clock and optionally enter description") ;
      i++ ;
      attron(A_BOLD) ;
      mvaddstr(i,5,"+") ;
      attroff(A_BOLD) ;
      mvaddstr(i,7,"Adjust by increasing time") ;
      i++ ;
      attron(A_BOLD) ;
      mvaddstr(i,5,"-") ;
      attroff(A_BOLD) ;
      mvaddstr(i,7,"Adjust by decreasing time") ;
      i++ ;
      attron(A_BOLD) ;
      mvaddstr(i,3,"ESC") ;
      attroff(A_BOLD) ;
      mvaddstr(i,7,"Quit") ;
      
      project_update=project ;
      found=1 ;
      j=strlen(project->name)+10 ;
      timer_value.it_value.tv_sec=1 ;
      timer_value.it_value.tv_usec=0 ;
      timer_value.it_interval.tv_sec=1 ;
      timer_value.it_interval.tv_usec=0 ;
      setitimer(ITIMER_REAL,&timer_value,0) ;
      
      for(quit=0;!quit;)
	{
	  update_skip_refresh=0 ;
	  nodelay(stdscr,TRUE) ;
	  for(keypress=ERR;keypress==ERR;usleep(1000))
	    keypress=mvgetch(IDLE_Y,IDLE_X) ;
	  update_skip_refresh=1 ;
	  nodelay(stdscr,FALSE) ;
	  switch(keypress)
	    {
	    case ' ' :
	      timer_value.it_value.tv_sec=0 ;
	      timer_value.it_value.tv_usec=0 ;
	      timer_value.it_interval.tv_sec=0 ;
	      timer_value.it_interval.tv_usec=0 ;
	      setitimer(ITIMER_REAL,&timer_value,0) ;
	      attron(A_BLINK) ;
	      mvaddstr(y_update,1,"->") ;
	      attroff(A_BLINK) ;
	      
	      i=num_projects+9 ;
	      mvaddstr(i,5,"-- Clock paused.  Press any key to resume --") ;
	      move(i,49) ;
	      keypress=getch() ;
	      move(i,0) ;
	      clrtoeol() ;
	      
	      attron(A_BOLD) ;
	      mvaddstr(y_update,1,"->") ;
	      attroff(A_BOLD) ;
	      timer_value.it_value.tv_sec=1 ;
	      timer_value.it_value.tv_usec=0 ;
	      timer_value.it_interval.tv_sec=1 ;
	      timer_value.it_interval.tv_usec=0 ;
	      setitimer(ITIMER_REAL,&timer_value,0) ;
	      break ;
	    case '+' :
	      i=num_projects+9 ;
	      attrset(A_BOLD) ;
	      mvaddstr(i,5,"Enter number of minutes to add : ") ;
	      attrset(A_BOLD) ;
	      echo();
	      success=(mvscanw(i,38,"%d",&modification)==1) ;
	      noecho();
	      if(success)
		if(modification<0)
		  success=0 ;
	      attrset(A_NORMAL) ;
	      move(i,0) ;
	      clrtoeol() ;
	      if(success)
		{
		  project->time+=60*modification ;
		}
	      else
		{
		  beep() ;
		  mvaddstr(i,5,"Error : input value must be a positive integer") ;
		  mvaddstr(i+1,5,"-- Press any key --") ;
		  move(i+1,24) ;
		  keypress=getch() ;
		  move(i,0) ;
		  clrtoeol() ;
		  move(i+1,0) ;
		  clrtoeol() ;
		}
	      break ;
	    case '-' :
	      i=num_projects+9 ;
	      attrset(A_BOLD) ;
	      mvaddstr(i,5,"Enter number of minutes to subtract : ") ;
	      attrset(A_BOLD) ;
	      echo();
	      success=(mvscanw(i,43,"%d",&modification)==1) ;
	      noecho();
	      if(success)
		if(modification<0)
		  success=0 ;
	      attrset(A_NORMAL) ;
	      move(i,0) ;
	      clrtoeol() ;
	      if(success)
		{
		  project->time-=60*modification ;
		}
	      else
		{
		  beep() ;
		  mvaddstr(i,5,"Error : input value must be a positive integer") ;
		  mvaddstr(i+1,5,"-- Press any key --") ;
		  move(i+1,24) ;
		  keypress=getch() ;
		  move(i,0) ;
		  clrtoeol() ;
		  move(i+1,0) ;
		  clrtoeol() ;
		}
	      break ;
	    case CR :
	      i=num_projects+9 ;
	      attrset(A_BOLD) ;
	      mvaddstr(i,5,"Enter comment : ") ;
	      attrset(A_BOLD) ;
	      echo();
	      mvgetstr(i,21,comments) ;
	      noecho();
	      attrset(A_NORMAL) ;
	      move(i,0) ;
	      clrtobot() ;
	      quit=1 ;
	      break ;
	    default :
	      beep() ;
	      quit=1 ;
	      break ;
	    }
	}
      sprintf(specific_timefile,"%s.%c",log_filename,key) ;
      f=fopen(log_filename,"a") ;
      if(f==NULL)
	{
	  fprintf(stderr,"ERROR: cannot open project log file : %s\n",log_filename) ;
	  perror("fopen") ;
	  exit(-1) ;
	}
      fa=fopen(specific_timefile,"a") ;
      if(fa==NULL)
	{
	  fprintf(stderr,"ERROR: cannot open specific project log file : %s\n",specific_timefile) ;
	  perror("fopen") ;
	  exit(-1) ;
	}
      fprintf(f,"%s ",project->name) ;
      delta_time=project->time-initial_time ;
      seconds=(double) delta_time ;
      hours = seconds / 3600.0 ;
      sprintf(temps,"%8.2f hours : ",hours) ;
      fprintf(fa,"%s",temps) ;
      if(strlen(comments)>0)
	{
	  fprintf(f,"(%s) ",comments) ;
	  fprintf(fa,"%s",comments) ;
	}
      else
	fprintf(fa,"<no description>") ;
      if(delta_time>0)
	{
	  seconds=(double) delta_time ;
	  if(seconds<60.0)
	    sprintf(temps,"%0.2f seconds",seconds) ;
	  else
	    {
	      minutes=seconds / 60.0 ;
	      if(minutes<60.0)
		sprintf(temps,"%0.2f minutes",minutes) ;
	      else
		{
		  hours = seconds / 3600.0 ;
		  sprintf(temps,"%0.2f hours",hours) ;
		}
	    }
	}
      fprintf(f,": %s : finished ",temps) ;
      fprintf(fa," : [%s] : finished ",temps) ;
      time(&t) ;
      sprintf(temps,"%s",asctime(localtime(&t))) ;
      if(temps[strlen(temps)-1]<14)
	temps[strlen(temps)-1]=0 ;
      fprintf(f,"%s\n",temps) ;
      fprintf(fa,"%s\n",temps) ;
      fclose(f) ;
      fclose(fa) ;
      
      timer_value.it_value.tv_sec=0 ;
      timer_value.it_value.tv_usec=0 ;
      timer_value.it_interval.tv_sec=0 ;
      timer_value.it_interval.tv_usec=0 ;
      setitimer(ITIMER_REAL,&timer_value,0) ;
      update_skip_refresh=0 ;
      alarm_handler(0) ;
      mvaddstr(y_update,1,"  ") ;
    }
  else
    if(key>' ')
      {
	beep() ;
	keypress=CR ;
      }
  return(keypress) ;
}


void main(argc,argv)
     int argc ;
     char **argv ;
{
  char *config_filename ;
  char keypress ;
  int quit,i ;
  FILE *lf ;
  time_t t ;
  time(&t) ;

  if(argc>1)
    {
      if(argc>3 ||
	 (strstr(argv[1],"-h")==argv[1]) ||
	 (strstr(argv[1],"-?")==argv[1]))
	{
	  fprintf(stderr,"USAGE: %s [<project config file> [<log file>]]\n",argv[0]) ;
	  exit(0) ;
	}
    }
#ifdef DEFAULT_DIR_ENVVAR
  else
    {					     /* no args given */
      const char* default_dir;
      if((default_dir=getenv(DEFAULT_DIR_ENVVAR)))
	if(chdir(default_dir))
	  {
	    fprintf(stderr,"WARNING: Could not change to default directory ($%s = %s)\n",DEFAULT_DIR_ENVVAR,default_dir) ;
	    perror("chdir");
	  }
	else
	  {
	    fprintf(stderr,"INFO: Using default directory %s (environment variable %s)\n",default_dir,DEFAULT_DIR_ENVVAR) ;
	  }
    }
#endif /* def DEFAULT_DIR_ENVVAR */
  
  if (argc>2)
    log_filename=argv[2] ;
  else
    log_filename=DEFAULT_LOG_FILE;
  if (argc>1)
    config_filename=argv[1] ;
  else
    config_filename=DEFAULT_CONFIG_FILE;
  read_config_file(config_filename) ;
  lf=fopen(log_filename,"a") ;
  if(lf==NULL)
    {
      fprintf(stderr,"ERROR: cannot open project log file : %s\n",log_filename) ;
      perror("fopen") ;
      exit(-1) ;
    }
  fclose(lf) ;

  sprintf(starttimes,"%s",asctime(localtime(&t))) ;
  if(starttimes[strlen(starttimes)-1]<14)
    starttimes[strlen(starttimes)-1]=0 ;

  initscr() ;
  cbreak() ;
  noecho() ;

  signal(SIGALRM,alarm_handler) ;

  signal(SIGHUP,exit_handler) ;
  signal(SIGINT,exit_handler) ;
  signal(SIGQUIT,exit_handler) ;
  signal(SIGTERM,exit_handler) ;

  quit=0 ;
  keypress=CR ;
  draw_main_screen() ;
  while(!quit)
    {
      if(keypress==CR)
	keypress=mvgetch(IDLE_Y,IDLE_X) ;
      keypress=clock_on(keypress) ;
      draw_main_screen() ;
      if(keypress==ERR)
	keypress=mvgetch(IDLE_Y,IDLE_X) ;
      if(keypress==27)
	{
	  i=num_projects+5 ;
	  attron(A_BLINK) ;
	  mvaddstr(i,5,"Press ESC once more to quit") ;
	  beep() ;
	  attroff(A_BLINK) ;
	  keypress=mvgetch(IDLE_Y,IDLE_X) ;
	  mvaddstr(i,5,"                           ") ;
	  if(keypress==27)
	    quit=1 ;
	  keypress=CR ;
	}
    }
  exit_handler(0) ;
}
