/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Some I/O routines for GF1/CS4231/InterWave chips
 */

#include "driver.h"

void gus_delay( gus_card_t *card )
{
  int i;
  
  if ( card -> gf1.enh_mode ) return;
  for ( i = 0; i < 6; i++ )
    {
      MB();
      INB( GUSP( card, DRAM ) );
    }
}

void gus_delay1( int loops )
{
  gus_delay2( 16 );
}

void gus_delay2( int loops )
{
  int i;
  
  while( loops-- > 0 )
    for ( i = 0; i < 16; i++ ) INB( 0x80 );
}

/*
 *  =======================================================================
 */

/*
 *  ok.. stop of control registers (wave & ramp) need some special things..
 *       big UltraClick (tm) elimination...
 */

static inline void __gus_ctrl_stop( gus_card_t *card, unsigned char reg )
{
  register unsigned char value;

  OUTB( reg | 0x80, card -> gf1.reg_regsel ); MB();
  value = INB( card -> gf1.reg_data8 ); MB();
  OUTB( reg, card -> gf1.reg_regsel ); MB();
  OUTB( (value | 0x03) & ~(0x80 | 0x20), card -> gf1.reg_data8 ); MB();
} 

static inline void __gus_write8( gus_card_t *card, unsigned char reg, unsigned char data )
{
  OUTB( reg, card -> gf1.reg_regsel ); MB();
  OUTB( data, card -> gf1.reg_data8 ); MB();
}

static inline unsigned char __gus_look8( gus_card_t *card, unsigned char reg )
{
  OUTB( reg, card -> gf1.reg_regsel ); MB();
  return INB( card -> gf1.reg_data8 );
}

static inline void __gus_write16( gus_card_t *card, unsigned char reg, unsigned int data )
{
  OUTB( reg, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)data, card -> gf1.reg_data16 ); MB();
}
       
static inline unsigned short __gus_look16( gus_card_t *card, unsigned char reg )
{
  OUTB( reg, card -> gf1.reg_regsel ); MB();
  return INW( card -> gf1.reg_data16 );
}

static inline void __gus_adlib_write( gus_card_t *card, unsigned char reg, unsigned char data )
{
  OUTB( reg, card -> gf1.reg_timerctrl );
  INB( card -> gf1.reg_timerctrl );
  INB( card -> gf1.reg_timerctrl );
  OUTB( data, card -> gf1.reg_timerdata );
  INB( card -> gf1.reg_timerctrl );
  INB( card -> gf1.reg_timerctrl );
}

static inline void __gus_write_addr( gus_card_t *card, unsigned char reg, unsigned int addr, int w_16bit )
{
  if ( card -> gf1.enh_mode )
    {
      if ( w_16bit )
        addr = ( ( addr >> 1 ) & ~0x0000000f ) | ( addr & 0x0000000f );
      __gus_write8( card, GF1_VB_UPPER_ADDRESS, (unsigned char)( ( addr >> 26 ) & 0x03 ) );
    }
   else
  if ( w_16bit )
    addr = ( addr & 0x00c0000f ) | ( ( addr & 0x003ffff0 ) >> 1 );
  __gus_write16( card, reg, (unsigned short)( addr >> 11 ) );
  __gus_write16( card, reg + 1, (unsigned short)( addr << 5 ) );
}

static inline unsigned int __gus_read_addr( gus_card_t *card, unsigned char reg, short w_16bit )
{
  unsigned int res;
  
  res = ( (unsigned int)__gus_look16( card, reg | 0x80 ) << 11 ) & 0xfff800;
  res |= ( (unsigned int)__gus_look16( card, ( reg + 1 ) | 0x80 ) >> 5 ) & 0x0007ff;
  if ( card -> gf1.enh_mode )
    {
      res |= (unsigned int)__gus_look8( card, GF1_VB_UPPER_ADDRESS | 0x80 ) << 26;
      if ( w_16bit )
        res = ( ( res << 1 ) & 0xffffffe0 ) | ( res & 0x0000000f );
    }
   else
  if ( w_16bit )
    res = ( ( res & 0x001ffff0 ) << 1 ) | ( res & 0x00c0000f );
  return res;
}


/*
 *  =======================================================================
 */
    
void gus_ctrl_stop( gus_card_t *card, unsigned char reg )
{
  __gus_ctrl_stop( card, reg );
} 

void gus_write8( gus_card_t *card, unsigned char reg, unsigned char data )
{
  __gus_write8( card, reg, data );
}

unsigned char gus_look8( gus_card_t *card, unsigned char reg )
{
  return __gus_look8( card, reg );
}

void gus_write16( gus_card_t *card, unsigned char reg, unsigned int data )
{
  __gus_write16( card, reg, data );
}
       
unsigned short gus_look16( gus_card_t *card, unsigned char reg )
{
  return __gus_look16( card, reg );
}

void gus_adlib_write( gus_card_t *card, unsigned char reg, unsigned char data )
{
  __gus_adlib_write( card, reg, data );
}

void gus_write_addr( gus_card_t *card, unsigned char reg, unsigned int addr, short w_16bit )
{
  __gus_write_addr( card, reg, addr, w_16bit );
}

unsigned int gus_read_addr( gus_card_t *card, unsigned char reg, short w_16bit )
{
  return __gus_read_addr( card, reg, w_16bit );
}

/*
 *
 */

void gus_i_ctrl_stop( gus_card_t *card, unsigned char reg )
{
  unsigned long flags;
  
  CLI( &flags );
  __gus_ctrl_stop( card, reg );
  STI( &flags );
}

void gus_i_write8( gus_card_t *card, unsigned char reg, unsigned char data )
{
  unsigned long flags;
  
  CLI( &flags );
  __gus_write8( card, reg, data );
  STI( &flags );
}

unsigned char gus_i_look8( gus_card_t *card, unsigned char reg )
{
  unsigned long flags;
  unsigned char res;

  CLI( &flags );
  res = __gus_look8( card, reg );
  STI( &flags );
  return res;
}

void gus_i_write16( gus_card_t *card, unsigned char reg, unsigned int data )
{
  unsigned long flags;
  
  CLI( &flags );
  __gus_write16( card, reg, data );
  STI( &flags );
}

unsigned short gus_i_look16( gus_card_t *card, unsigned char reg )
{
  unsigned long flags;
  unsigned short res;

  CLI( &flags );
  res = __gus_look16( card, reg );
  STI( &flags );
  return res;
}

void gus_i_adlib_write( gus_card_t *card, unsigned char reg, unsigned char data )
{
  unsigned long flags;
  
  CLI( &flags );
  __gus_adlib_write( card, reg, data );
  STI( &flags );
}

void gus_i_write_addr( gus_card_t *card, unsigned char reg, unsigned int addr, short w_16bit )
{
  unsigned long flags;
  
  CLI( &flags );
  __gus_write_addr( card, reg, addr, w_16bit );
  STI( &flags );
}

unsigned int gus_i_read_addr( gus_card_t *card, unsigned char reg, short w_16bit )
{
  unsigned int res;
  unsigned long flags;
  
  CLI( &flags );  
  res = __gus_read_addr( card, reg, w_16bit );
  STI( &flags );
  return res;
}

/*
 *
 */

void gus_dram_addr( gus_card_t *card, unsigned int addr )
{
  unsigned long flags;

  CLI( &flags );
  OUTB( 0x43, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)addr, card -> gf1.reg_data16 ); MB();
  OUTB( 0x44, card -> gf1.reg_regsel ); MB();
  OUTB( (unsigned char)(addr >> 16), card -> gf1.reg_data8 ); MB();
  STI( &flags );
}

void gus_poke( gus_card_t *card, unsigned int addr, unsigned char data )
{
  unsigned long flags;

  CLI( &flags );
  OUTB( 0x43, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)addr, card -> gf1.reg_data16 ); MB();
  OUTB( 0x44, card -> gf1.reg_regsel ); MB();
  OUTB( (unsigned char)(addr >> 16), card -> gf1.reg_data8 ); MB();
  OUTB( data, card -> gf1.reg_dram );
  STI( &flags );
}

unsigned char gus_peek( gus_card_t *card, unsigned int addr )
{
  unsigned long flags;
  unsigned char res;

  CLI( &flags );
  OUTB( 0x43, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)addr, card -> gf1.reg_data16 ); MB();
  OUTB( 0x44, card -> gf1.reg_regsel ); MB();
  OUTB( (unsigned char)(addr >> 16), card -> gf1.reg_data8 ); MB();
  res = INB( card -> gf1.reg_dram );
  STI( &flags );
  return res;
}

#ifdef GUSCFG_PNP

void gus_pokew( gus_card_t *card, unsigned int addr, unsigned short data )
{
  unsigned long flags;

  CLI( &flags );
  OUTB( 0x43, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)addr, card -> gf1.reg_data16 ); MB();
  OUTB( 0x44, card -> gf1.reg_regsel ); MB();
  OUTB( (unsigned char)(addr >> 16), card -> gf1.reg_data8 ); MB();
  OUTB( 0x51, card -> gf1.reg_regsel ); MB();
  OUTW( data, card -> gf1.reg_data16 );
  STI( &flags );
}

unsigned short gus_peekw( gus_card_t *card, unsigned int addr )
{
  unsigned long flags;
  unsigned short res;

  CLI( &flags );
  OUTB( 0x43, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)addr, card -> gf1.reg_data16 ); MB();
  OUTB( 0x44, card -> gf1.reg_regsel ); MB();
  OUTB( (unsigned char)(addr >> 16), card -> gf1.reg_data8 ); MB();
  OUTB( 0x51, card -> gf1.reg_regsel ); MB();
  res = INW( card -> gf1.reg_data16 );
  STI( &flags );
  return res;
}

void gus_dram_setmem( gus_card_t *card, unsigned int addr, unsigned short value, unsigned int count )
{
  unsigned short port;
  unsigned long flags;
  
  addr &= ~1;
  count >>= 1;
  port = GUSP( card, GF1DATALOW );
  CLI( &flags );
  OUTB( 0x43, card -> gf1.reg_regsel ); MB();
  OUTW( (unsigned short)addr, card -> gf1.reg_data16 ); MB();
  OUTB( 0x44, card -> gf1.reg_regsel ); MB();
  OUTB( (unsigned char)(addr >> 16), card -> gf1.reg_data8 ); MB();
  OUTB( 0x51, card -> gf1.reg_regsel );
  while ( count-- )
    OUTW( value, port );
  STI( &flags );
}

#endif

/*
 *
 */

void gus_reselect_active_voices( gus_card_t *card )
{
  unsigned short i, j, k, voices;

  static unsigned short voices_tbl[ 32 - 14 + 1 ] = {
    44100, 41160, 38587, 36317, 34300, 32494, 30870, 29400, 28063, 26843,
    25725, 24696, 23746, 22866, 22050, 21289, 20580, 19916, 19293
  };
  
#if 0
  PRINTK( "gus_reselect_active_voices - begin\n" );
#endif
  /* at first.. shut down all currently used voices */
  /* this depends on code.. so I'm now using callbacks */
  for ( i = 0; i < GF1_VOICE_RANGES; i++ )
    if ( card -> gf1.voice_ranges[ i ].voices_change_start )
      card -> gf1.voice_ranges[ i ].voices_change_start( card );
  voices = 0;
  for ( i = 0; i < GF1_VOICE_RANGES; i++ )
    voices += ( card -> gf1.voice_ranges[ i ].voices =
                card -> gf1.voice_ranges[ i ].rvoices );
  if ( voices > 32 )
    {
      /* ok.. order - 1) EFFECTS, 2) PCM, 3) SYNTH */
      voices = 32;
      i = GF1_VOICE_RANGES;
      while ( i-- > 0 )
        {
          if ( card -> gf1.voice_ranges[ i ].voices > voices )
            card -> gf1.voice_ranges[ i ].voices = voices;
          voices -= card -> gf1.voice_ranges[ i ].voices;
        }
      voices = 32;
    }
  for ( i = j = 0; i < GF1_VOICE_RANGES - 1; i++ )
    {
      k = card -> gf1.voice_ranges[ i ].voices;
      card -> gf1.voice_ranges[ i ].min = !k ? 31 : j;
      card -> gf1.voice_ranges[ i ].max = !k ? 0 : j + (k - 1);
      if ( k ) j = card -> gf1.voice_ranges[ i ].max + 1;
    }
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].max = 31;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].min = 32 - card -> gf1.voice_ranges[ GF1_VOICE_RANGE_EFFECT ].voices;
#if 0
  printk( "syn_active_voice = %i, pcm_active_voices = %i\n", card -> gf1.syn_active_voices, card -> gf1.pcm_active_voices );
#endif
  if ( voices < 14 ) voices = 14;
  card -> gf1.active_voices = voices;
  card -> gf1.playback_freq =
  	card -> gf1.enh_mode ? 44100 : voices_tbl[ voices - 14 ];
  if ( !card -> gf1.enh_mode )
    {
      unsigned char reset, old_setup, new_setup;
    
      /*
       * ok.. seems to be trouble with old GF1 chip and active voices select register..
       * I'm now writing correction value (32) and real value (voices).
       * This may cause some unwanted interrupts, but I handle all of them.
       * Don't change this code anymore.. (Perex -> Perex)
       */
      new_setup = 0xc0 | ( voices - 1 );
      old_setup = gus_i_look8( card, GF1_GB_ACTIVE_VOICES );
      if ( old_setup != new_setup )	/* ignore equal hits */
        {
          reset = gus_i_look8( card, GF1_GB_RESET );
          gus_i_write8( card, GF1_GB_RESET, reset & 5 );	/* turn off DAC */
          gus_i_write8( card, GF1_GB_ACTIVE_VOICES, 0xc0 | ( 32 - 1 ) );
          gus_delay1( 10 );
          gus_i_write8( card, GF1_GB_ACTIVE_VOICES, new_setup );
          gus_delay1( 10 );
          gus_i_write8( card, GF1_GB_RESET, reset );		/* restore */
        }
    }
  /* send ack that voice reselection was done */
  for ( i = 0; i < GF1_VOICE_RANGES; i++ )
    if ( card -> gf1.voice_ranges[ i ].voices_change_stop )
      card -> gf1.voice_ranges[ i ].voices_change_stop( card );
#if 0
  PRINTK( "gus_reselect_active_voices - end\n" );
#endif
}

#ifdef GUSCFG_CODEC

void codec_outm( gus_card_t *card, unsigned char reg, unsigned char mask, unsigned char value )
{
  unsigned char r, tmp;
  unsigned long flags;

  CLI( &flags );
  r = ( INB( CODECP( card, REGSEL ) ) & 0xe0 ) | reg;
  OUTB( r, CODECP( card, REGSEL ) ); MB();
  tmp = ( INB( CODECP( card, REG ) ) & mask ) | value;
  OUTB( tmp, CODECP( card, REG ) ); MB();
  STI( &flags );
}

void codec_out( gus_card_t *card, unsigned char reg, unsigned char value )
{
  unsigned long flags;
  int timeout = 900000;

  while ( timeout-- > 0 && ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT ) );
  if ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT )
    PRINTK( "codec_out: auto calibration time out\n" );
  CLI( &flags );
  OUTB( card -> codec.mce_bit | reg, CODECP( card, REGSEL ) );
  OUTB( value, CODECP( card, REG ) ); MB();
  STI( &flags );
}
    
unsigned char codec_in( gus_card_t *card, unsigned char reg )
{
  unsigned long flags;
  unsigned char res;
  int timeout = 900000;

  while ( timeout-- > 0 && ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT ) );
  if ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT )
    PRINTK( "codec_in: auto calibration time out\n" );
  CLI( &flags );
  OUTB( card -> codec.mce_bit | reg, CODECP( card, REGSEL ) ); MB();
  res = INB( CODECP( card, REG ) );
  STI( &flags );
  return res;
}

#endif /* GUSCFG_CODEC */

#if defined( GUSCFG_DEBUG ) && defined( GUSCFG_CODEC )

void codec_debug( gus_card_t *card )
{
  printk( "CODEC REGS:       INDEX = 0x%02x  ", INB( CODECP( card, REGSEL ) ) );
  printk( "                 STATUS = 0x%02x\n", INB( CODECP( card, STATUS ) ) );
  printk( "  0x00: left input      = 0x%02x  ", codec_in( card, 0x00 ) );
  printk( "  0x10: alt 1 (CFIG 2)  = 0x%02x\n", codec_in( card, 0x10 ) );
  printk( "  0x01: right input     = 0x%02x  ", codec_in( card, 0x01 ) );
  printk( "  0x11: alt 2 (CFIG 3)  = 0x%02x\n", codec_in( card, 0x11 ) );
  printk( "  0x02: GF1 left input  = 0x%02x  ", codec_in( card, 0x02 ) );
  printk( "  0x12: left line in    = 0x%02x\n", codec_in( card, 0x12 ) );
  printk( "  0x03: GF1 right input = 0x%02x  ", codec_in( card, 0x03 ) );
  printk( "  0x13: right line in   = 0x%02x\n", codec_in( card, 0x13 ) );
  printk( "  0x04: CD left input   = 0x%02x  ", codec_in( card, 0x04 ) );
  printk( "  0x14: timer low       = 0x%02x\n", codec_in( card, 0x14 ) );
  printk( "  0x05: CD right input  = 0x%02x  ", codec_in( card, 0x05 ) );
  printk( "  0x15: timer high      = 0x%02x\n", codec_in( card, 0x15 ) );
  printk( "  0x06: left output     = 0x%02x  ", codec_in( card, 0x06 ) );
  printk( "  0x16: left MIC (PnP)  = 0x%02x\n", codec_in( card, 0x16 ) );
  printk( "  0x07: right output    = 0x%02x  ", codec_in( card, 0x07 ) );
  printk( "  0x17: right MIC (PnP) = 0x%02x\n", codec_in( card, 0x17 ) );
  printk( "  0x08: playback format = 0x%02x  ", codec_in( card, 0x08 ) );
  printk( "  0x18: IRQ status      = 0x%02x\n", codec_in( card, 0x18 ) );
  printk( "  0x09: iface (CFIG 1)  = 0x%02x  ", codec_in( card, 0x09 ) );
  printk( "  0x19: left line out   = 0x%02x\n", codec_in( card, 0x19 ) );
  printk( "  0x0a: pin control     = 0x%02x  ", codec_in( card, 0x0a ) );
  printk( "  0x1a: mono control    = 0x%02x\n", codec_in( card, 0x1a ) );
  printk( "  0x0b: init & status   = 0x%02x  ", codec_in( card, 0x0b ) );
  printk( "  0x1b: right line out  = 0x%02x\n", codec_in( card, 0x1b ) );
  printk( "  0x0c: revision & mode = 0x%02x  ", codec_in( card, 0x0c ) );
  printk( "  0x1c: record format   = 0x%02x\n", codec_in( card, 0x1c ) );
  printk( "  0x0d: loopback        = 0x%02x  ", codec_in( card, 0x0d ) );
  printk( "  0x1d: var freq (PnP)  = 0x%02x\n", codec_in( card, 0x1d ) );
  printk( "  0x0e: ply upr count   = 0x%02x  ", codec_in( card, 0x0e ) );
  printk( "  0x1e: ply lwr count   = 0x%02x\n", codec_in( card, 0x1e ) );
  printk( "  0x0f: rec upr count   = 0x%02x  ", codec_in( card, 0x0f ) );
  printk( "  0x1f: rec lwr count   = 0x%02x\n", codec_in( card, 0x1f ) );
}

#endif

#ifdef GUSCFG_DEBUG

void gus_print_voice_registers( gus_card_t *card )
{
  unsigned char mode;
  int voice, ctrl;
  
  voice = card -> gf1.active_voice;
  printk( " -%i- GF1  voice ctrl, ramp ctrl  = 0x%x, 0x%x\n", voice, ctrl = gus_i_read8( card, 0 ), gus_i_read8( card, 0x0d ) );
  printk( " -%i- GF1  frequency              = 0x%x\n", voice, gus_i_read16( card, 1 ) );
  printk( " -%i- GF1  loop start, end        = 0x%x (0x%x), 0x%x (0x%x)\n", voice, gus_i_read_addr( card, 2, ctrl & 4 ), gus_i_read_addr( card, 2, ( ctrl & 4 ) ^ 4 ), gus_i_read_addr( card, 4, ctrl & 4 ), gus_i_read_addr( card, 4, ( ctrl & 4 ) ^ 4 ) );
  printk( " -%i- GF1  ramp start, end, rate  = 0x%x, 0x%x, 0x%x\n", voice, gus_i_read8( card, 7 ), gus_i_read8( card, 8 ), gus_i_read8( card, 6 ) );
  printk( " -%i- GF1  volume                 = 0x%x\n", voice, gus_i_read16( card, 9 ) );
  printk( " -%i- GF1  position               = 0x%x (0x%x)\n", voice, gus_i_read_addr( card, 0x0a, ctrl & 4 ), gus_i_read_addr( card, 0x0a, ( ctrl & 4 ) ^ 4 ) );
  if ( card -> pnp_flag && gus_i_read8( card, 0x19 ) & 0x01 )	/* enhanced mode */
    {
      mode = gus_i_read8( card, 0x15 );
      printk( " -%i- GFA1 mode                   = 0x%x\n", voice, mode );
      if ( mode & 0x01 )   	/* Effect processor */
        {
          printk( " -%i- GFA1 effect address         = 0x%x\n", voice, gus_i_read_addr( card, 0x11, ctrl & 4 ) );
          printk( " -%i- GFA1 effect volume          = 0x%x\n", voice, gus_i_read16( card, 0x16 ) );
          printk( " -%i- GFA1 effect volume final    = 0x%x\n", voice, gus_i_read16( card, 0x1d ) );
          printk( " -%i- GFA1 effect acumulator      = 0x%x\n", voice, gus_i_read8( card, 0x14 ) );
        }
      if ( mode & 0x20 )
        {
          printk( " -%i- GFA1 left offset            = 0x%x (%i)\n", voice, gus_i_read16( card, 0x13 ), gus_i_read16( card, 0x13 ) >> 4 );
          printk( " -%i- GFA1 left offset final      = 0x%x (%i)\n", voice, gus_i_read16( card, 0x1c ), gus_i_read16( card, 0x1c ) >> 4 );
          printk( " -%i- GFA1 right offset           = 0x%x (%i)\n", voice, gus_i_read16( card, 0x0c ), gus_i_read16( card, 0x0c ) >> 4 );
          printk( " -%i- GFA1 right offset final     = 0x%x (%i)\n", voice, gus_i_read16( card, 0x1b ), gus_i_read16( card, 0x1b ) >> 4 );
        }
       else
        printk( " -%i- GF1  pan                    = 0x%x\n", voice, gus_i_read8( card, 0x0c ) );
    }
   else
    printk( " -%i- GF1  pan                    = 0x%x\n", voice, gus_i_read8( card, 0x0c ) );
}

void gus_print_global_registers( gus_card_t *card )
{
  unsigned char global_mode = 0x00;
  
  if ( card -> pnp_flag )
    global_mode = gus_i_read8( card, GF1_GB_GLOBAL_MODE );
  printk( " -G- GF1 active voices            = 0x%x\n", gus_i_look8( card, GF1_GB_ACTIVE_VOICES ) );
  printk( " -G- GF1 global mode              = 0x%x\n", gus_i_look8( card, GF1_GB_GLOBAL_MODE ) );
  if ( global_mode & 0x02 )		/* LFO enabled? */
    printk( " -G- GF1 LFO base                 = 0x%x\n", gus_i_look16( card, GF1_GW_LFO_BASE ) );
  printk( " -G- GF1 voices IRQ read          = 0x%x\n", gus_i_look8( card, GF1_GB_VOICES_IRQ_READ ) );
  printk( " -G- GF1 DRAM DMA control         = 0x%x\n", gus_i_look8( card, GF1_GB_DRAM_DMA_CONTROL ) );
  printk( " -G- GF1 DRAM DMA high/low        = 0x%x/0x%x\n", gus_i_look8( card, GF1_GB_DRAM_DMA_HIGH ), gus_i_read16( card, GF1_GW_DRAM_DMA_LOW ) );
  printk( " -G- GF1 DRAM IO high/low         = 0x%x/0x%x\n", gus_i_look8( card, GF1_GB_DRAM_IO_HIGH ), gus_i_read16( card, GF1_GW_DRAM_IO_LOW ) );
  if ( !card -> pnp_flag )
    printk( " -G- GF1 record DMA control       = 0x%x\n", gus_i_look8( card, GF1_GB_REC_DMA_CONTROL ) );
  printk( " -G- GF1 DRAM IO 16               = 0x%x\n", gus_i_look16( card, GF1_GW_DRAM_IO16 ) );
  if ( card -> gf1.enh_mode )
    {
      printk( " -G- GFA1 memory config           = 0x%x\n", gus_i_look16( card, GF1_GW_MEMORY_CONFIG ) );
      printk( " -G- GFA1 memory control          = 0x%x\n", gus_i_look8( card, GF1_GB_MEMORY_CONTROL ) );
      printk( " -G- GFA1 FIFO record base        = 0x%x\n", gus_i_look16( card, GF1_GW_FIFO_RECORD_BASE_ADDR ) );
      printk( " -G- GFA1 FIFO playback base      = 0x%x\n", gus_i_look16( card, GF1_GW_FIFO_PLAY_BASE_ADDR ) );
      printk( " -G- GFA1 interleave control      = 0x%x\n", gus_i_look16( card, GF1_GW_INTERLEAVE ) );
    }
}

void gus_print_setup_registers( gus_card_t *card )
{
  printk( " -S- mix control                  = 0x%x\n", INB( GUSP( card, MIXCNTRLREG ) ) );
  printk( " -S- IRQ status                   = 0x%x\n", INB( GUSP( card, IRQSTAT ) ) );
  printk( " -S- timer control                = 0x%x\n", INB( GUSP( card, TIMERCNTRL ) ) );
  printk( " -S- timer data                   = 0x%x\n", INB( GUSP( card, TIMERDATA ) ) );
  printk( " -S- status read                  = 0x%x\n", INB( GUSP( card, REGCNTRLS ) ) );
  printk( " -S- Sound Blaster control        = 0x%x\n", gus_i_look8( card, GF1_GB_SOUND_BLASTER_CONTROL ) );
  printk( " -S- AdLib timer 1/2              = 0x%x/0x%x\n", gus_i_look8( card, GF1_GB_ADLIB_TIMER_1 ), gus_i_look8( card, GF1_GB_ADLIB_TIMER_2 ) );
  printk( " -S- reset                        = 0x%x\n", gus_i_look8( card, GF1_GB_RESET ) );
  if ( card -> pnp_flag )
    {
      printk( " -S- compatibility                = 0x%x\n", gus_i_look8( card, GF1_GB_COMPATIBILITY ) );
      printk( " -S- decode control               = 0x%x\n", gus_i_look8( card, GF1_GB_DECODE_CONTROL ) );
      printk( " -S- version number               = 0x%x\n", gus_i_look8( card, GF1_GB_VERSION_NUMBER ) );
      printk( " -S- MPU-401 emul. control A/B    = 0x%x/0x%x\n", gus_i_look8( card, GF1_GB_MPU401_CONTROL_A ), gus_i_look8( card, GF1_GB_MPU401_CONTROL_B ) );
      printk( " -S- emulation IRQ                = 0x%x\n", gus_i_look8( card, GF1_GB_EMULATION_IRQ ) );
    }
} 

void gus_peek_print_block( gus_card_t *card, unsigned int addr, int count, int w_16bit )
{
  if ( !w_16bit )
    {
      while ( count-- > 0 )
        printk( count > 0 ? "%02x:" : "%02x", gus_peek( card, addr++ ) );
    }
   else
    {
      while ( count-- > 0 )
        {
          printk( count > 0 ? "%04x:" : "%04x", gus_peek( card, addr ) | ( gus_peek( card, addr + 1 ) << 8 ) );
          addr += 2;
        }
    }
}

#endif
