// wmmixer - a mixer designed for WindowMaker
// 07/05/98  Release 0.8
// Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>
// This software comes with ABSOLUTELY NO WARRANTY
// This software is free software, and you are welcome to redistribute it
// under certain conditions
// See the README file for a more complete notice.

#define WINDOWMAKER FALSE
#define USESHAPE FALSE
#define NAME "wmmixer"
#define CLASS "WMMixer"

#define DEFAULTDEVICE "/dev/mixer"
#define BACKCOLOR "#282828"
#define LEDCOLOR "green"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>

#include "mixctl.h"

Pixmap back;
Pixmap tile;
Pixmap disp;
Pixmap mask;
Pixmap symb;
Pixmap nrec;

unsigned long fcolor,bcolor;

#include "XPM/wmmixer.xpm"
#include "XPM/tile.xpm"
#include "XPM/symbols.xpm"
#include "XPM/norec.xpm"

bool wmaker = WINDOWMAKER;
bool ushape = USESHAPE;
char txtdpy[256] = "";
char txtdev[256] = DEFAULTDEVICE;
char ledcol[256] = LEDCOLOR;
char bckcol[256] = BACKCOLOR;

Atom _XA_GNUSTEP_WM_FUNC;
Atom deleteWin;

Display *dpy;
Window Win[2];
Window Root;
XWMHints *hints;
GC WinGC;
int activeWin;

int curDev=0;
int curShowRec;
int curRec;
int curLeft;
int curRight;

// For new buttons
int btnstate=0;
#define btnNext  1
#define btnPrev  2
#define btnRec   4

// With repeat next/prev if you hold down mouse button
#define btnrepeat 5
int btntimer=0;

int Channels=0;
int Channel[20];
int sympic[20]={0,7,8,2,1,6,4,5,3,9,9,9,9,9,9,9,9,9,9,9};

bool dragging=FALSE;

MixCtl *mixctl;

// Standard dock-app stuff
void initXWin(int argc, char *argv[]);
void freeXWin();
void createWin(Window *win);
unsigned long getColor(char *colorName, float dim);

// Custom dock-app stuff
void scanArgs(int argc, char *argv[]);
void readFile();
void checkVol(bool forced=TRUE);
void pressEvent(XButtonEvent *xev);
void releaseEvent(XButtonEvent *xev);
void motionEvent(XMotionEvent *xev);
void repaint();
void update();
void drawLeft();
void drawRight();
void drawBtns(int btns);
void drawBtn(int x, int y, int w, int h, bool down);

int main(int argc,char *argv[])
{
   scanArgs(argc,argv);
   initXWin(argc,argv);

   XGCValues gcv;
   unsigned long gcm;
   gcm = GCForeground|GCBackground|GCGraphicsExposures;
   gcv.graphics_exposures = False;
   WinGC = XCreateGC(dpy, Root, gcm, &gcv);

   XpmAttributes pixatt;
   XpmColorSymbol ledcols[4]={{"led_color_high", NULL, 0},
                              {"led_color_med",  NULL, 0},
                              {"led_color_low",  NULL, 0},
                              {"back_color",     NULL, 0}};
   ledcols[0].pixel=getColor(ledcol, 1.00);
   ledcols[1].pixel=getColor(ledcol, 1.65);
   ledcols[2].pixel=getColor(ledcol, 2.60);
   ledcols[3].pixel=getColor(bckcol, 1.00);
   pixatt.numsymbols=4;
   pixatt.colorsymbols=ledcols;
   pixatt.exactColors=FALSE;
   pixatt.closeness=40000;
   pixatt.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness;
   XpmCreatePixmapFromData(dpy, Root, wmmixer_xpm, &back, &mask, &pixatt);
   XpmCreatePixmapFromData(dpy, Root, tile_xpm, &tile, NULL, &pixatt);
   XpmCreatePixmapFromData(dpy, Root, symbols_xpm, &symb, NULL, &pixatt);
   XpmCreatePixmapFromData(dpy, Root, norec_xpm, &nrec, NULL, &pixatt);
   disp = XCreatePixmap(dpy, Root, 64, 64, DefaultDepth(dpy,DefaultScreen(dpy)));

   fcolor=ledcols[0].pixel;
   bcolor=ledcols[2].pixel;

   // Install mask or copy background tile
   if(wmaker || ushape)
      XShapeCombineMask(dpy, Win[activeWin], ShapeBounding, 0, 0, mask, ShapeSet);
   else
      XCopyArea(dpy, tile, disp, WinGC, 0,0,64,64,0,0);

   // Copy background
   XSetClipMask(dpy, WinGC, mask);
   XCopyArea(dpy, back, disp, WinGC, 0,0,64,64,0,0);
   XSetClipMask(dpy, WinGC, None);

   mixctl = new MixCtl(txtdev);

   for(int i=0;i<mixctl->getNrDevices();i++){
      if(mixctl->getSupport(i)){
         Channel[Channels]=i;
         Channels++;
      }
   }

   readFile();

   if(Channels==0)
      fprintf(stderr,"Sorry, no supported channels found\n");
   else{
      checkVol(TRUE);

      XEvent event;
      XSelectInput(dpy, Win[activeWin], ExposureMask | ButtonPressMask | ButtonReleaseMask | ButtonMotionMask);
      XMapWindow(dpy, Win[0]);

      bool finished=FALSE;
      while(!finished){
         while(XPending(dpy)){
            XNextEvent(dpy,&event);
            switch(event.type){
             case Expose:
                repaint();
             break;
             case ButtonPress:
                pressEvent(&event.xbutton);
             break;
             case ButtonRelease:
                releaseEvent(&event.xbutton);
             break;
             case MotionNotify:
                motionEvent(&event.xmotion);
             break;
             case ClientMessage:
                if(event.xclient.data.l[0]==deleteWin)
                   finished=TRUE;
             break;
            }
         }

         // Repeat next/prev
         if(btnstate & (btnPrev | btnNext)){
            btntimer++;
            if(btntimer>=btnrepeat){
               if(btnstate & btnNext)
                  curDev++;
               else
                  curDev--;
               if(curDev<0)
                  curDev=Channels-1;
               if(curDev>=Channels)
                  curDev=0;
               checkVol(TRUE);      // forced because of changed mount point
               btntimer=0;
            }
         }
         else{
            // Check volume, and update as needed
            checkVol(FALSE);
         }
         XFlush(dpy);
         usleep(50000L);
      }
   }
   XFreeGC(dpy, WinGC);
   XFreePixmap(dpy, disp);
   XFreePixmap(dpy, mask);
   XFreePixmap(dpy, back);
   XFreePixmap(dpy, tile);
   XFreePixmap(dpy, symb);
   XFreePixmap(dpy, nrec);
   freeXWin();
   delete mixctl;
   return 0;
}

void initXWin(int argc, char *argv[]){
   if((dpy=XOpenDisplay(txtdpy))==NULL){
      fprintf(stderr,"You're probably trying to run an X app from the console, you idiot! RTFM\n");
      exit(1);
   }
   _XA_GNUSTEP_WM_FUNC = XInternAtom(dpy, "_GNUSTEP_WM_FUNCTION", False);
   deleteWin = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
   Root=DefaultRootWindow(dpy);
   createWin(&Win[0]);
   createWin(&Win[1]);
   XWMHints hints;
   XSizeHints shints;
   hints.window_group = Win[0];
   shints.min_width=64;
   shints.min_height=64;
   shints.max_width=64;
   shints.max_height=64;
   shints.x=0;
   shints.y=0;
   if(wmaker){
      hints.initial_state = WithdrawnState;
      hints.icon_window = Win[1];
      hints.flags = WindowGroupHint | StateHint | IconWindowHint;
      shints.flags = PMinSize | PMaxSize | PPosition;
      activeWin=1;
   }
   else{
      hints.initial_state = NormalState;
      hints.flags = WindowGroupHint | StateHint;
      shints.flags = PMinSize | PMaxSize;
      activeWin=0;
   }
   XSetWMHints(dpy, Win[0], &hints);
   XSetWMNormalHints(dpy, Win[0], &shints);
   XSetCommand(dpy, Win[0], argv, argc);
   XStoreName(dpy, Win[0], NAME);
   XSetIconName(dpy, Win[0], NAME);
   XSetWMProtocols(dpy, Win[activeWin], &deleteWin, 1);
}

void freeXWin(){
   XDestroyWindow(dpy, Win[0]);
   XDestroyWindow(dpy, Win[1]);
   XCloseDisplay(dpy);
}

void createWin(Window *win){
   XClassHint classHint;
   *win = XCreateSimpleWindow(dpy, Root, 10, 10, 64, 64,0,0,0);
   classHint.res_name = NAME;
   classHint.res_class = CLASS;
   XSetClassHint(dpy, *win, &classHint);
}

unsigned long getColor(char *colorName, float dim)
{
   XColor Color;
   XWindowAttributes Attributes;

   XGetWindowAttributes(dpy, Root, &Attributes);
   Color.pixel = 0;

   XParseColor (dpy, Attributes.colormap, colorName, &Color);
   Color.red=(unsigned short)(Color.red/dim);
   Color.blue=(unsigned short)(Color.blue/dim);
   Color.green=(unsigned short)(Color.green/dim);
   Color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor (dpy, Attributes.colormap, &Color);

   return Color.pixel;
}

void scanArgs(int argc, char *argv[]){
   for(int i=1;i<argc;i++){
      if(strcmp(argv[i],"-h")==0 || strcmp(argv[i],"-help")==0 || strcmp(argv[i],"--help")==0){
         fprintf(stderr,"wmmixer - a mixer designed for WindowMaker\n07/05/98  Release 0.8\n");
         fprintf(stderr,"Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>\n");
         fprintf(stderr,"This software comes with ABSOLUTELY NO WARRANTY\n");
         fprintf(stderr,"This software is free software, and you are welcome to redistribute it\n");
         fprintf(stderr,"under certain conditions\n");
         fprintf(stderr,"See the README file for a more complete notice.\n\n");
         fprintf(stderr,"usage:\n\n   %s [options]\n\noptions:\n\n",argv[0]);
         fprintf(stderr,"   -h | -help | --help    display this help screen\n");
         fprintf(stderr,"   -w                     use WithdrawnState    (for WindowMaker)\n");
         fprintf(stderr,"   -s                     shaped window\n");
         fprintf(stderr,"   -l led_color           use the specified color for led display\n");
         fprintf(stderr,"   -b back_color          use the specified color for backgrounds\n");
         fprintf(stderr,"   -d mix_device          use specified device  (rather than /dev/mixer)\n");
         fprintf(stderr,"   -display display       select target display (see X manual pages)\n\n");
         exit(0);
      }
      if(strcmp(argv[i],"-w")==0)
         wmaker=TRUE;
      if(strcmp(argv[i],"-s")==0)
         ushape=TRUE;
      if(strcmp(argv[i],"-d")==0){
	 if(i<argc-1){
            i++;
            sprintf(txtdev,"%s",argv[i]);
         }
         continue;
      }
      if(strcmp(argv[i],"-l")==0){
	 if(i<argc-1){
            i++;
            sprintf(ledcol,"%s",argv[i]);
	 }
         continue;
      }
      if(strcmp(argv[i],"-b")==0){
	 if(i<argc-1){
            i++;
            sprintf(bckcol,"%s",argv[i]);
	 }
         continue;
      }
      if(strcmp(argv[i],"-display")==0){
	 if(i<argc-1){
            i++;
            sprintf(txtdpy,"%s",argv[i]);
         }
         continue;
      }
   }
}

void readFile(){
   FILE *rcfile;
   char rcfilen[256];
   char buf[256];
   int done;
   sprintf(rcfilen,"%s/.wmmixer",getenv("HOME"));
   if((rcfile=fopen(rcfilen,"r"))!=NULL){
      Channels=0;
      do{
         fgets(buf,250,rcfile);
         if((done=feof(rcfile))==0){
            buf[strlen(buf)-1]=0;
            if(strncmp(buf,"addchannel ",strlen("addchannel "))==0){
               sscanf(buf,"addchannel %i",&Channel[Channels]);
               if(Channel[Channels]<mixctl->getNrDevices()){
                  if(mixctl->getSupport(Channel[Channels]))
                     Channels++;
	       }
            }
            if(strncmp(buf,"setmono ",strlen("setmono "))==0){
               if(Channels>0){
                  int value;
                  sscanf(buf,"setmono %i",&value);
                  mixctl->setLeft(Channel[Channels-1],value);
                  mixctl->setRight(Channel[Channels-1],value);
                  mixctl->writeVol(Channel[Channels-1]);
               }
            }
            if(strncmp(buf,"setleft ",strlen("setleft "))==0){
	       if(Channels>0){
                  int value;
                  sscanf(buf,"setleft %i",&value);
                  mixctl->setLeft(Channel[Channels-1],value);
                  mixctl->writeVol(Channel[Channels-1]);
	       }
            }
            if(strncmp(buf,"setright ",strlen("setright "))==0){
	       if(Channels>0){
                  int value;
                  sscanf(buf,"setleft %i",&value);
                  mixctl->setRight(Channel[Channels-1],value);
                  mixctl->writeVol(Channel[Channels-1]);
	       }
            }
            if(strncmp(buf,"setrecsrc ",strlen("setrecsrc "))==0){
	       if(Channels>0)
                  mixctl->setRec(Channel[Channels-1],(strncmp(buf+strlen("setrecsrc "),"TRUE",strlen("TRUE"))==0));
            }
         }
      }  while(done==0);
      fclose(rcfile);
      mixctl->writeRec();       // delayed write, to avoid interference from low-level drivers.
   }
}

void checkVol(bool forced){
   mixctl->readVol(Channel[curDev],TRUE);
   int nl=mixctl->readLeft(Channel[curDev]);
   int nr=mixctl->readRight(Channel[curDev]);
   bool nrec=mixctl->readRec(Channel[curDev],TRUE);
   if(forced){
      curLeft=nl;
      curRight=nr;
      curRec=nrec;
      if(nrec)
         btnstate|=btnRec;
      else
         btnstate&=~btnRec;      
      curShowRec=mixctl->getRecords(Channel[curDev]);
      update();   // Update everything
      repaint();
   }
   else{
      if(nl!=curLeft || nr!=curRight || nrec!=curRec){
         if(nl!=curLeft){
            curLeft=nl;
            drawLeft();
         }
         if(nr!=curRight){
            curRight=nr;
            drawRight();
         }
         if(nrec!=curRec){
            curRec=nrec;
            if(nrec)
               btnstate|=btnRec;
            else
               btnstate&=~btnRec;
            drawBtns(btnRec);
         }
         repaint();
      }
   }
}

void pressEvent(XButtonEvent *xev){
   int x=xev->x;
   int y=xev->y;
   if(x>=5 && y>=33 && x<=16 && y<=43){
      curDev--;
      if(curDev<0)
         curDev=Channels-1;
      btnstate|=btnPrev;
      btntimer=0;
      drawBtns(btnPrev);
      checkVol(TRUE);
      return;
   }
   if(x>=17 && y>=33 && x<=28 && y<=43){
      curDev++;
      if(curDev>=Channels)
         curDev=0;
      btnstate|=btnNext;
      btntimer=0;
      drawBtns(btnNext);
      checkVol(TRUE);
      return;
   }
   if(x>=37 && x<=56 && y>=8 && y<=56){
      int v=((60-y)*100)/(3*17);
      dragging=TRUE;
      if(x<=50)
         mixctl->setLeft(Channel[curDev], v);
      if(x>=45)
         mixctl->setRight(Channel[curDev], v);
      mixctl->writeVol(Channel[curDev]);
      checkVol(FALSE);
      return;
   }
   if(x>=5 && y>=47 && x<=28 && y<=57){
      mixctl->setRec(Channel[curDev], !mixctl->readRec(Channel[curDev],FALSE));
      mixctl->writeRec();
      checkVol(FALSE);
   }
}

void releaseEvent(XButtonEvent *xev){
   dragging=FALSE;
   btnstate&=~(btnPrev | btnNext);
   drawBtns(btnPrev | btnNext);
   repaint();
}

void motionEvent(XMotionEvent *xev){
   int x=xev->x;
   int y=xev->y;
   if(x>=37 && x<=56 && y>=8 && dragging){
      int v=((60-y)*100)/(3*17);
      if(v<0)
         v=0;
      if(x<=50)
         mixctl->setLeft(Channel[curDev], v);
      if(x>=45)
         mixctl->setRight(Channel[curDev], v);
      mixctl->writeVol(Channel[curDev]);
      checkVol(FALSE);
   }
}

void repaint(){
   XCopyArea(dpy, disp, Win[activeWin], WinGC, 0, 0, 64, 64, 0, 0);
   XEvent xev;
   while(XCheckTypedEvent(dpy, Expose, &xev));
}

void update(){
   // Only needs to be called when the current device is changed

   XCopyArea(dpy, symb, disp, WinGC, sympic[Channel[curDev]]*22,0,22,22, 6,5);
   drawLeft();
   drawRight();
   drawBtns(btnRec);
}

void drawLeft(){
   XSetForeground(dpy, WinGC, fcolor);
   for(int i=0;i<17;i++){
      if(i==(curLeft*17)/100)
         XSetForeground(dpy, WinGC, bcolor);
      XFillRectangle(dpy, disp, WinGC, 37, 54-3*i, 9, 2);
   }
}

void drawRight(){
   XSetForeground(dpy, WinGC, fcolor);
   for(int i=0;i<17;i++){
      if(i==(curRight*17)/100)
         XSetForeground(dpy, WinGC, bcolor);
      XFillRectangle(dpy, disp, WinGC, 48, 54-3*i, 9, 2);
   }
}

void drawBtns(int btns){
   if(btns & btnPrev)
      drawBtn(5,33,12,11,(btnstate & btnPrev));
   if(btns & btnNext)
      drawBtn(17,33,12,11,(btnstate & btnNext));
   if(btns & btnRec){
      drawBtn(5,47,24,11,(btnstate & btnRec));
      if(!curShowRec)
         XCopyArea(dpy, nrec, disp, WinGC, 0,0,6,7,14,49);
      else
         XCopyArea(dpy, back, disp, WinGC, 14,49,6,7,14,49);
   }
}

void drawBtn(int x, int y, int w, int h, bool down){
   if(!down)
      XCopyArea(dpy, back, disp, WinGC, x,y,w,h,x,y);
   else{
      XCopyArea(dpy, back, disp, WinGC, x,y,1,h-1,x+w-1,y+1);
      XCopyArea(dpy, back, disp, WinGC, x+w-1,y+1,1,h-1,x,y);
      XCopyArea(dpy, back, disp, WinGC, x,y,w-1,1,x+1,y+h-1);
      XCopyArea(dpy, back, disp, WinGC, x+1,y+h-1,w-1,1,x,y);
   }
}
