#include <stdio.h>
#include <stdarg.h>

#include "arena.h"
#include "bridge.h"
#ifdef ARENA_DEBUG	/* QingLong.14-12-96 */
#  include "debug.h"
#endif
#include "display.h"
#include "forms.h"
#include "html.h"
#include "main.h"
#include "scrollbar.h"
#include "status.h"
#include "util.h"
#include "w3c_libhack.h"


/* The multithreaded library complicates the history mechannism.
   Several documents may be requested simultaneously,
   and they may arrive in a different order than they were clicked in.
   Therefore, putting a document into the history is a two-stem porcess:
   when the user clicks, HistoryRecord put it into the history,
   but does not enable it -- then,
   when a document arrives, HistoryVerify enables it.
 */

/* Simply set a history mode type, or if arg == 0, just return current.
 * Didn't want any global variable to communicate with main.c (arg list
 * parsing).
 */
int HistoryMode(int new_mode)
{
 /* Start with current mode set to whatever was in arena.h */
 static int curr_hist_mode = HISTORY_MODE;

 if (new_mode != 0)
   {
    int temp = curr_hist_mode;
    curr_hist_mode = new_mode;
    return temp;
   }
 return curr_hist_mode;
}

/* Set a history delete ok flag...
 * Used by executeAction() in menu.c to indicate that "next" HistoryRecord()
 * should do "mark for death" iff(HISTORY_TYPE_SEMI_DELETE).  Do this so we
 * don't have a global variable!
 */
static Bool History_OK_TO_MARK_FOR_DEATH = False;

void HistoryAllowMarkForDeath(Bool flag)
{
 History_OK_TO_MARK_FOR_DEATH = flag;
}

/* Just return the value of the mark for death flag...
 * Used by HistoryRecord and GotoDoc to control the deletion history mechanisms
 */
Bool HistoryCanMarkForDeath()
{
 return  History_OK_TO_MARK_FOR_DEATH;
}



History* NewHistory()
{
 History* h;

 if ((h = (History*)Arena_CAlloc(sizeof(History), 1, False)))
   {
    h->state = HISTORY_NOTREGISTERED;
    h->title = NULL;
    h->tag = NULL;
    h->y = 0;
   }

 return h;
}

void FreeHistory(History* h)
{
 if (h)
   {
    h->anchor = NULL;
    h->state = HISTORY_NOTREGISTERED;
    Free(h->title);
    Free(h->tag);
    h->y = 0;
   }
 Free(h);
}


/* go to home document */

Bool HomeDoc(GC theGC, ViewWindow theViewWindow)
{
 Bool TotalSuccess = False;
 Doc* thedocument;
#ifdef ARENA_DEBUG
 char Iam[] = "HomeDoc";
#endif


#if 1
 thedocument = (Doc*)HTAnchor_document(context->home_anchor);
# else
 thedocument = (Doc*)HTAnchor_document(HTAnchor_parent((HTAnchor*)context->home_anchor));
#endif
 if (thedocument)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE && VERBOSE_TRACE) Arena_TracePrint(Iam, ")\n");
#endif

    TotalSuccess = GotoDoc(theGC, theViewWindow,
			   HTList_count(context->history) - 1);
   }
  else
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE) Arena_TracePrint(Iam, " home_doc freed??????\n");
#endif
    TotalSuccess = ((libLoadAnchor((HTAnchor*)context->home_anchor, NULL,
				   True, True, True, False, False, True))
		    != NULL);

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE) HistoryList(context);
#endif
   }

 return TotalSuccess;
}


Bool ForwardDoc(GC theGC, ViewWindow theViewWindow)
{
 History* h = NULL;
 int pos; 

#ifdef ARENA_DEBUG
 char Iam[] = "ForwardDoc";
 if (HISTORY_TRACE && VERBOSE_TRACE) Arena_TracePrint(Iam, ")\n");
#endif

 pos = context->history_pos;
 while (pos > 0)
   {
    pos--;
    h = (History*)HTList_objectAt(context->history, pos);
    if (h == NULL)
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       if (HISTORY_TRACE) Arena_TracePrint(Iam, " no history to be found\n");
#endif
       continue;
      }
    if (h->state == HISTORY_VERIFIED) break;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE)
      Arena_TracePrint(Iam,
		       " selected item either not loaded or deleted %d.\n",
		       (int)(h->state));
#endif
    h = NULL;
   } /* End ``while (pos > 0)'' */

 if (h == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE) Arena_TracePrint(Iam, " no history to be found.\n");
#endif
    return False;
   }

 /* from here we assume success */
 HistoryAllowMarkForDeath(False);
 return GotoDoc(theGC, theViewWindow, pos);
}


Bool BackDoc(GC theGC, ViewWindow theViewWindow)
{
 History* h = NULL;
 int pos; 
#ifdef ARENA_DEBUG
 char Iam[] = "BackDoc";
#endif


 pos = context->history_pos;

 {
  int history_length = HTList_count(context->history);

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
  if (HISTORY_TRACE)
    Arena_TracePrint(Iam,
		     " context->history_pos = %d, history_length = %d.\n",
		     pos, history_length);
#endif

  while (pos < (history_length - 1))
    {
     pos++;
     h = (History*)HTList_objectAt(context->history, pos);

     if (h == NULL)
       {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
	if (HISTORY_TRACE)
	  Arena_TracePrint(Iam,
			   " position %d not found in the history.\n", pos);
#endif
	continue;
       }

     if (h->state == HISTORY_VERIFIED) break;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
     if (HISTORY_TRACE)
       Arena_TracePrint(Iam,
			" selected item not yet confirmed loaded %d\n",
			(int)(h->state));	
#endif
     h = NULL;
    }

  if (h == NULL)
    {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
     if (HISTORY_TRACE) Arena_TracePrint(Iam, " no history to be found.\n");
#endif
     return False;
    }
 }

 /* from here we assume success */
 HistoryAllowMarkForDeath(False);
 return GotoDoc(theGC, theViewWindow, pos);
}



Bool GotoDoc(GC theGC, ViewWindow theViewWindow, int pos)
{
 Bool TotalSuccess = False;
 History* h = NULL;
 HTAnchor* a;
 Doc* d;

#ifdef ARENA_DEBUG
 char Iam[] = "GotoDoc";
 if (HISTORY_TRACE && VERBOSE_TRACE) Arena_TracePrint(Iam, ")\n");
#endif

 h = (History*)HTList_objectAt(context->history, pos);
 if (h == NULL)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE) Arena_TracePrint(Iam, " no history to be found.\n");
#endif
    return False;
   }

 if (h->state != HISTORY_VERIFIED)
   {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE)
      Arena_TracePrint(Iam,
		       " selected item not yet confirmed loaded %d.\n",
		       (int)(h->state));	
#endif
    return False;
   }

 /* from here we assume success */

 context->history_pos = pos;
 context->current_history = (History*)HTList_objectAt(context->history,
						      context->history_pos);
 a = h->anchor;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (HISTORY_TRACE)
   Arena_TracePrint(Iam,
		    " pos = %d, a = \"%s\".\n",
		    context->history_pos,
		    HTAnchor_physical(HTAnchor_parent(a)));
#endif

 if ((d = (Doc*)HTAnchor_document(HTAnchor_parent(a))) != NULL)
   {
    if ((d->state == DOC_LOADED) || (d->state == DOC_PROCESSED))
      {
       Bool Reparse = (CurrentDoc != d);

       if (CurrentDoc->edited_field)
	 FinishCurrentFieldEditing(theGC, theViewWindow);

       CurrentDoc = (Doc*)d;
       CurrentDoc->show_raw = False;

       /* Once again...
	* CurrentDoc* may be a doc used by more than one anchor!
	* Set the #tag to OUR anchor's named tag pointer!
	*/
       Free(CurrentDoc->tag);
       CurrentDoc->tag = HTChildAnchor_tag((HTChildAnchor*)a);

       /* By freeing and not resetting ->Fragment,
	* history->y will control the positioning of the document.
	* If you want each ChildAnchor to ALWAYS go to the fragment name tag,
	* then just allow the reset of the ->Fragment.
	*/
       Free(CurrentDoc->Fragment);
       /* CurrentDoc->Fragment = HTChildAnchor_tag((HTChildAnchor*)a);       */

       /* Do reparse if necessary...
	* and/or redisplay
	*/
       if (Reparse)
	 NewBuffer(CurrentDoc);
       else
	 {
	  if (CurrentDoc->Fragment)
	    context->current_history->y = Y_of_Fragment(CurrentDoc,
							CurrentDoc->Fragment);
	  MoveYDisplay(CurrentDoc, context->current_history->y);
	 }

       /* Regardless of what is decided above about the ->Fragment
	* Free it here... Then history->y will prevail... until the NEXT time!
	*/
       Free(CurrentDoc->Fragment);
       TotalSuccess = True;
      }
     else
      {
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       Arena_TracePrint(Iam,
			" WEIRD, doc is reg'd but state is lame (%d).\n",
			d->state);
#endif
      }
   }
  else
   {
#if 1	/* 16-04-97 Juergen Ruehle <jr@falbala.informatik.uni-kiel.de> */
    TotalSuccess = ((libLoadAnchor(a, NULL,
				   True, False, False, False, False,  True))
		    != NULL);
# else
    TotalSuccess = ((libLoadAnchor(a, NULL,
				   True, False, False, False, False, False))
		    != NULL);
#endif
   }

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (HISTORY_TRACE) HistoryList(context);
#endif
 return TotalSuccess;
}

/* Do one thing... Add an entry to the history list! */
History* HistoryAddOne(HTAnchor* a)
{
 History* h = NULL;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 char Iam[] = "HistoryAddOne";
 if (HISTORY_TRACE) Arena_TracePrint(Iam, " anchor "POINTER_FORMAT"\n", a);
#endif

 /* Then, create a new history record to be put onto the history
  * We are remembering the actual anchor of the request... Therefore,
  * we have lots of ChildAnchor* problems later on!
  */
 if ((h = NewHistory()) == NULL) return NULL;
 h->state = HISTORY_REGISTERED;
 h->anchor = a;

 if ((HTAnchor*)HTAnchor_parent(a) != a)
   {
    HTChildAnchor* c = (HTChildAnchor*)a;
    h->tag = HTChildAnchor_tag(c);
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
    if (HISTORY_TRACE)
      Arena_TracePrint(Iam, " recording CHILD tag \"%s\"\n", h->tag);
#endif
   }

 /* Create the "list", if necessary, and add our new history entry to it! */
 if (!context->history) context->history = HTList_new();
 HTList_addObject(context->history, h);

 /* 16-04-97 Juergen Ruehle <jr@falbala.informatik.uni-kiel.de> */
 if (context->history_pos >= 0) context->history_pos++;
 return h;
}

/* Perform setting of "mark for death" for all history entries "after" this
 * one...  and clean up the history list
 */
Bool HistoryMarkForDeath()
{
 int mode, i;
 Bool DidMark = False;

 History* h;

#ifdef ARENA_DEBUG
 char Iam[] = "HistoryMarkForDeath";
 if (HISTORY_TRACE) Arena_TracePrint(Iam, " Begin... \n");
#endif

 /* glk.20.12.97
  * First, let's take care of the special case of DELETING type history modes.
  * HISTORY_TYPE_DELETE will always "mark for death" all history entries
  * after current postion.
  * HISTORY _TYPE_SEMI_HYPERCHAIN will do the same, BUT ONLY if popup menu
  * has been used to select "branch position".  Please note:  The use of
  * Back/Reload/Forward does NOT create the ...CanMarkForDeath == True!
  */
 mode = HistoryMode(HISTORY_QUERY);
 if (mode == HISTORY_TYPE_DELETE ||
     (mode == HISTORY_TYPE_SEMI_HYPERCHAIN && HistoryCanMarkForDeath()))
   {
    /* We're putting a new doc into the history list.  First mark the rest
     * of the history for death since this is a deleting history mechanism
     */
     if ((i = context->history_pos) > 0)
       {
	for (i--; i >= 0; --i)
	  {
#ifdef ARENA_DEBUG
	    if (HISTORY_TRACE) Arena_TracePrint(Iam, " in loop %d.\n", i);
#endif
	    h = (History *)HTList_objectAt(context->history, i);
	    if (h && h->state == HISTORY_VERIFIED)
	      {
	       h->state = HISTORY_DELETED;
	       DidMark = True;
#ifdef ARENA_DEBUG
	       if (HISTORY_TRACE) Arena_TracePrint(Iam, " deleting %d.\n", i);
#endif
	      }
	  }   /* end for                      */
       }  /* end if context->history > 0      */
     if (DidMark)
       HistoryCleanUp();
   }   /* end if "delete type" HistoryMode()  */
 return DidMark;
}


/* Do a full recording of an anchor into the history list...
 * This includes clean up of list if necessary!
 */
History* HistoryRecord(HTAnchor* a)
{
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 char Iam[] = "HistoryRecord";
 if (HISTORY_TRACE) Arena_TracePrint(Iam, " anchor "POINTER_FORMAT"\n", a);
#endif

 /* If we are using one of the "deleting history mechanisms"...
  * Set the delete marker for any entries "above" our current position
  */
 HistoryMarkForDeath();

 /* Do NOT allow mark for death the next time in here...
  * Unless reset to True by executeAction() from menu.c
  * Add the new one to our list
  */
 HistoryAllowMarkForDeath(False);

 /* NOW!
  * Add the new anchor to the list!!
  */
 return  HistoryAddOne(a);
}

/* Actually perform the removal of "marked for delete" history entries.
 */
void HistoryCleanUp(void)
{
 History* h;
 HTList* l;
 HTList* tmp_l;

#ifdef ARENA_DEBUG
 int i = 0;
 char Iam[] = "HistoryCleanUp";
 if (HISTORY_TRACE) Arena_TracePrint(Iam, " Begin... \n");
#endif

 /* We may not be running in HISTORY_TYPE_DELETE or ..._SEMI_HYPERCHAIN, but...
  * a history entry may still be marked for death... This occurs when an
  * attempt to open a document fails!
  */
 l = context->history;
 while ((h = (History*)HTList_nextObject(l)))
   {
    if (h->state == HISTORY_DELETED)
      {
#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
       if (HISTORY_TRACE)
	  Arena_TracePrint(Iam, " deleting %d \"%s\"\n",
		i, HTAnchor_physical(HTAnchor_parent((HTAnchor*)h->anchor)));
#endif
       FreeHistory(h);

       /* Remove the linked list object... we've just release its contents */
       tmp_l = l->next;
       HTList_removeObject(context->history, l->object);
       context->history_pos--;

       /* dumps if error on start_up since we'll remove the object "l" is
	* pointing to...
	*/
       l = tmp_l;
      }
#ifdef ARENA_DEBUG
    else /* if we just deleted element, it moved ahead automatically */
      i++;
#endif
   }

#ifdef ARENA_DEBUG
 if (HISTORY_TRACE) HistoryList(context);
#endif
}


Bool HistoryDelete(HTAnchor* a)
{
 History* h;
 HTList* l = context->history;
 Bool theAnchorFound = False;

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 char Iam[] = "HistoryDelete";
 if (HISTORY_TRACE) Arena_TracePrint(Iam, " anchor "POINTER_FORMAT"\n", a);
#endif

 while ((h = (History*)HTList_nextObject(l)))
   {
    if (a == h->anchor)
      {
       theAnchorFound = True;
       h->state = HISTORY_DELETED;
#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
       {
	char* theaddress;
	theaddress = HTAnchor_address(a);
	if (HISTORY_TRACE)
	  Arena_TracePrint(Iam, " a->parent->address = \"%s\"\n", theaddress);
	Free(theaddress);
       }
#endif
      }
   }

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (HISTORY_TRACE) HistoryList(context);
#endif
 return theAnchorFound;
}


Bool HistoryVerify(HTAnchor* a)
{
 History* h;
 HTList* l;

 Bool theAnchorFound = False;
 int count = 0;
 HTParentAnchor* Panchor = HTAnchor_parent(a);

#ifdef ARENA_DEBUG
 char Iam[] = "HistoryVerify";
#endif

#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
 if (HISTORY_TRACE)
   Arena_TracePrint(Iam, " anchor "POINTER_FORMAT"\n", a);
#endif

 /* The incoming anchor has been loaded
  * and should be verified on the history list.
  * As it is verified, we should delete the rest of the branch.
  * If we were just moving around in history (i.e. HistoryRecord),
  * the mark-for-death had to be on at a stage not called.
  */
 HistoryCleanUp();

 /* NOW, try and find our anchor in the history list.
  * If we find it, set it to Verified, and return it as our current
  * history position... BUT, make sure it is the "lastest" entry that we
  * return as "the" one.  Any other history entries that have the "same"
  * parent, will also be marked verified.
  */
 l = context->history;
 while ((h = (History*)HTList_nextObject(l)))
   {
    if (!theAnchorFound)
      {
       if (h->anchor == a && h->state == HISTORY_REGISTERED)
	 {
	   h->state = HISTORY_VERIFIED;
	   theAnchorFound = True;
	   context->history_pos = count;
	   context->current_history = h;
	   if (CurrentDoc->title)
	     StrAllocCopy(h->title, CurrentDoc->title);

#ifdef ARENA_DEBUG	/* QingLong.11-12-96 */
	   if (HISTORY_TRACE)
	     Arena_TracePrint(Iam, " context->history_pos = %d\n",
			      context->history_pos);
#endif
	 }
      }
    else
      {
       /* Same parent?  Then mark it verified */
       if ((HTAnchor*)HTAnchor_parent(h->anchor) == (HTAnchor*)Panchor)
	 if (h->state == HISTORY_REGISTERED)
	   h->state = HISTORY_VERIFIED;
      }
    count++;
   }

#ifdef ARENA_DEBUG	/* QingLong.12-12-96 */
 if (HISTORY_TRACE) HistoryList(context);
#endif
 return theAnchorFound;
}
