/* NCSA CCI for X Windows
** Software Development Group
** National Center for Supercomputing Applications
** University of Illinois at Urbana-Champaign
** 605 E. Springfield, Champaign, IL 61820
** mosaic@ncsa.uiuc.edu

** Copyright  (C)  1995 Board of Trustees of the University of Illinois
*/

#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include <InterViews/layout.h>
#include <InterViews/style.h>
#include <IV-look/kit.h>
#include <hyperg/widgets/msgbox.h>
#include <hyperg/widgets/fchooser.h>
#include <hyperg/utils/str.h>
#include <InterViews/enter-scope.h>
#include "cciCaller.h"
#include "cciViewer.h"

extern "C" {
  extern MCCIPort MCCIConnect(char *serverAddress,
			      int port,
			      void (*callBack)(),
			      void *callBackData);
  extern int MCCISendOutput(MCCIPort serverPort,
			    char *mimeType, 
                            void (*callBack)(char *, char *, int , void *),
                            void *callBackData);
  extern int MCCIGet(MCCIPort serverPort,
		     char *uri, 
		     int output,
		     int absRelative,
		     char *additionalHeader);
  extern int MCCINBGet(MCCIPort serverPort,
		       char *uri, 
		       int output,
		       int absRelative,
		       char *additionalHeader);
  extern void MCCIInitialize();
  extern int MCCIIsConnected(MCCIPort serverPort);
  extern int MCCIDisconnect(MCCIPort serverPort);
  extern int MCCIPoll(MCCIPort serverPort);
  extern int MCCISendAnchor(MCCIPort serverPort,
                            int status,
                            void (*callBack)(char *, void *),
                            void *callBackData);
  extern void MCCIRetrieveURLFromFile(MCCIPort serverPort,
				      char *fileName,char **url);
  extern int MCCIGetAnnotation(MCCIPort serverPort,
			int type,
			char *url,
			int *numberPublic,
			char ***publicAnnotations,
			int *numberGroup,
			char ***groupAnnotations,
			int *numberPrivate,
			char ***privateAnnotations);

  extern int MCCIPutAnnotation(MCCIPort serverPort,
			       int type,
			       char *url,
			       char *annotation,
			       int annotationLength);

}

CCICaller* CCICaller::me_;

// Return the CCI information which Mosaic writes to a file when CCI is invoked
void CCICaller::defaultPortInfo(int& portNum, RString& hostname) const {
  RString port_fn(getenv("HOME"));
  port_fn += "/.mosaiccciport";
  ifstream portstream(port_fn.string());
  if (portstream) {
    RString portstring;
    portstring.gLine(portstream);
    int colon = portstring.index(':');
    if (colon > 0) {
      RString num = portstring.gRight(colon+1);
      portNum = atoi(num.string());
      hostname = portstring.gSubstrIndex(0, colon-1);
    }
  }
}

// Connect either to the given port and host, or to the default  
boolean CCICaller::connect(int portNum, char* hostname) {
  if (!connected_) {
    RString hostString = hostname;
    if (portNum == 0) 
      defaultPortInfo(portNum, hostString);
    makeConnection(portNum, hostString);
  }
  if (connected_)
    return true;
  return false;  // mpi
}

// Actually make the CCI connection to the given host/port
void CCICaller::makeConnection(int portNum, const RString& hostname) {
  disconnect();
  if (portNum > 0) {
    MCCIInitialize();
    mosaicPort_ = MCCIConnect((char*)hostname.string(), portNum, 0, 0);
    if (connected_ = MCCIIsConnected(mosaicPort_)) {
      MCCISendOutput(mosaicPort_, (char*)mimeType_.string(),
		     handleLoad, NULL);
      MCCISendAnchor(mosaicPort_, MCCI_SEND_AFTER, 
		     handleFetch, NULL);
      updateMenu();
    }
  }
}

void CCICaller::updateMenu() const {
}
  
// Ask Mosaic to fetch the given url
void CCICaller::follow(const RString& url) {
  if (MCCINBGet(mosaicPort_, (char*)url.string(), 
		MCCI_DEFAULT, MCCI_DEFAULT, 0) == MCCI_FAIL)
    notConnected();
}

// Connect to the given host/port.  Disconnect to whatever we are already 
// connected to.
boolean CCICaller::reconnect(int portNum, const RString& hostname) {
  makeConnection(portNum, hostname);
  return true;
}

CCICaller::~CCICaller() {
  disconnect();
}

void CCICaller::disconnect() {
  if (connected_) {
    MCCIDisconnect(mosaicPort_);
    connected_ = false;
  }
}

// Check to see if anything has arrived on the CCI port.
boolean CCICaller::poll() {
  if (connected_) {
    if (MCCIPoll(mosaicPort_) == -1)
      notConnected();
  }
  return connected_;
}

// This is called if we discover we are no longer connected.
void CCICaller::notConnected() {
  disconnect();
  updateMenu();
  hideAnchors();
}

void CCICaller::hideAnchors() const {
}

// Constructor sets static variable to this object
CCICaller::CCICaller(const char* mimeType) {
  me_ = this;
  connected_ = false;
  mimeType_ = mimeType;
  mosaicPort_ = NULL;
}


RString CCICaller::matchName(const char* name) const {
  char* urlname = 0;
  MCCIRetrieveURLFromFile(mosaicPort_, (char*)name, &urlname);
  if (urlname && (strlen(urlname) > 0)) {
    RString rfile = urlname;
    free(urlname);
    return rfile;
  }
  else
    return name;
}

boolean CCICaller::putAnnotation(int type,
				 const char *annotation) {
    return MCCIPutAnnotation(mosaicPort_, type, 
			     (char*)currentURL().string(), 
			     (char*)annotation,
			     strlen(annotation));
}

RString CCICaller::currentURL() const {
  return RString::lambda();
}

RString CCICaller::getAnchors(const char* name) {
    int numAnnotations[num_annotation_types];
    char** annotations[num_annotation_types];
    RString allTogether;

    MCCIGetAnnotation(mosaicPort_, MCCI_PRIVATE_ANNOTATION,
		      (char*)name,
		      &numAnnotations[PUBLIC_ANNOTATION],
		      &annotations[PUBLIC_ANNOTATION],
		      &numAnnotations[GROUP_ANNOTATION],
		      &annotations[GROUP_ANNOTATION],
		      &numAnnotations[PRIVATE_ANNOTATION],
		      &annotations[PRIVATE_ANNOTATION]);
    
    for (int i = 0; i < num_annotation_types; ++i) {
      for (int j = 0; j < numAnnotations[i]; ++j) {
	allTogether += annotations[i][j];
      }
    }
    
    return allTogether;
} 

HgLanguage::Language CCICaller::language() const {
  return HgLanguage::Default;
}

Window* CCICaller::window() const {
  return NULL;
}

void CCICaller::handleFetch(char *anchor, void *callBackData) {
  me_->handleFetch(anchor);
}


void CCICaller::handleLoad(char *, char *data, int length,
			   void *) {

  me_->handleLoad(data, length);
}

void CCICaller::handleFetch(char *anchor) {
}


void CCICaller::handleLoad(char *data, int length) {
}

SceneCCICaller::SceneCCICaller(CCISceneViewer* scene) :
CCICaller("x-world/x-vrml") {
  wait_ = true;
  lastAsked_ = "";
  viewer_ = scene;
}

void SceneCCICaller::handleLoad(char *data, int length) {
  if (wait_) {
    lastAsked_ = "";
    do {
      poll(); // wait until we get the anchor
    } while (lastAsked_.length() == 0);
  }
  wait_ = true;
  viewer_->handleLoad(data, length, lastAsked_);
}

void SceneCCICaller::handleFetch(char *anchor) {
  RString anc = anchor;
  if (anc.indexa(".wrl") >= 0) 
    lastAsked_ = anc;
}

void SceneCCICaller::follow(const RString& url) {
  poll(); // make sure nothing is pending
  wait_ = false;
  lastAsked_ = url;
  if (MCCIGet(mosaicPort_, (char*)url.string(), 
		MCCI_DEFAULT, MCCI_DEFAULT, 0) == MCCI_FAIL)
    notConnected();
  wait_ = true;
}  

