;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime/Llib/dsssl.scm               */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Sat Jul  3 11:30:29 1997                          */
;*    Last change :  Fri Feb  6 16:55:35 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    Bigloo support for Dsssl (Iso/Iec 10179:1996)                    */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __dsssl

   (import  (__error                   "Llib/error.scm"))
   
   (use     (__type                    "Llib/type.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__r4_control_features_6_9 "Ieee/control.scm")
	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")
	    (__r4_strings_6_7          "Ieee/string.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")

	    (__evenv                   "Eval/evenv.scm"))

   (extern  (macro c-keyword?::bool           (::obj)     "KEYWORDP")
	    (c-string->keyword::keyword       (::string)  "string_to_keyword")
	    (macro c-keyword->string::bstring (::keyword) "KEYWORD_TO_STRING")
	    (macro cnst->integer::long        (::obj)     "CCNST"))
   
   (export  (inline keyword?::bool           ::obj)
	    (inline keyword->string::bstring ::keyword)
	    (inline string->keyword::keyword ::bstring)
	    (make-dsssl-function-prelude     ::obj ::obj ::obj ::procedure)
	    (dsssl-get-key-arg               ::obj ::keyword ::obj)
	    (dsssl-check-key-args!           ::obj ::obj)
	    (dsssl-formals->scheme-formals   ::obj ::procedure))
	    
   (pragma  (c-keyword? (predicate-of keyword) no-cfa-top)
	    (keyword? side-effect-free)
	    (c-string->keyword no-cfa-top)
	    (string->keyword no-cfa-top)))
 
;*---------------------------------------------------------------------*/
;*    keyword? ...                                                     */
;*---------------------------------------------------------------------*/
(define-inline (keyword? obj)
   (c-keyword? obj))

;*---------------------------------------------------------------------*/
;*    keyword->string ...                                              */
;*---------------------------------------------------------------------*/
(define-inline (keyword->string keyword)
   (c-keyword->string keyword))

;*---------------------------------------------------------------------*/
;*    string->keyword ...                                              */
;*---------------------------------------------------------------------*/
(define-inline (string->keyword string)
   (c-string->keyword string))

;*---------------------------------------------------------------------*/
;*    make-dsssl-function-prelude ...                                  */
;*    -------------------------------------------------------------    */
;*    This function decodes a DSSSL formal parameter list and          */
;*    produce a header decoding the actuals.                           */
;*    -------------------------------------------------------------    */
;*    We implement a finite automata by representing each state        */
;*    by a function. It is much easier to implement this than          */
;*    with variables and loops.                                        */
;*---------------------------------------------------------------------*/
(define (make-dsssl-function-prelude where formals body err)
   
   (define (scheme-state args)
      (cond
	 ((not (pair? args))
	  body)
	 ((and (not (symbol? (car args))) (not (pair? (car args))))
	  ;; either it is a Dsssl named constant or an error.
	  (case (car args)
	     ((#!optional)
	      (enter-dsssl-state (cdr args) optional-state))
	     ((#!rest)
	      (enter-dsssl-state (cdr args) rest-state))
	     ((#!key)
	      (enter-dsssl-state (cdr args) no-rest-key-state))
	     (else
	      (err where "Illegal formal list" formals))))
	 (else
	  ;; regular Scheme formal, we simply skip
	  (scheme-state (cdr args)))))

   (define (enter-dsssl-state args next-state)
      (let loop ((as args))
	 (cond
	    ((null? as)
	     (next-state args #unspecified))
	    ((not (pair? as))
	     (err where "Illegal formal list" formals))
	    ((and (not (symbol? (car as))) (not (pair? (car as))))
	     (loop (cdr as)))
	    (else
	     (match-case (car as)
		((? symbol?)
		 (let ((dsssl-arg (gensym 'dsssl)))
		    `(let ((,dsssl-arg ,(car as)))
			,(next-state args dsssl-arg))))
		(((? symbol?) ?-)
		 (let ((dsssl-arg (gensym 'dsssl)))
		    `(let ((,dsssl-arg ,(car (car as))))
			,(next-state args dsssl-arg)))))))))

   (define (optional-state args dsssl-arg)
      (define (one-optional-arg arg initializer)
	 (let ((tmp (gensym 'tmp)))
	    `(let ((,arg (if (null? ,dsssl-arg)
			     ,initializer
			     (let ((,tmp (car ,dsssl-arg)))
				(set! ,dsssl-arg (cdr ,dsssl-arg))
				,tmp))))
		,(optional-state (cdr args) dsssl-arg))))
      (cond
	 ((null? args)
	  body)
	 ((not (pair? args))
	  (err where "Illegal Dsssl formal list (#!optional)" formals))
	 ((and (not (symbol? (car args))) (not (pair? (car args))))
	  ;; either it is a Dsssl named constant or an error.
	  (case (car args)
	     ((#!rest)
	      (rest-state (cdr args) dsssl-arg))
	     ((#!key)
	      (rest-state (cdr args) dsssl-arg))
	     (else
	      (err where "Illegal Dsssl formal list (#!optional)" formals))))
	 (else
	  ;; an optional Dsssl formal
	  (match-case (car args)
	     (((and (? symbol?) ?arg) ?initializer)
	      (one-optional-arg arg initializer))
	     ((and (? symbol?) ?arg)
	      (one-optional-arg arg #f))
	     (else
	      (err where "Illegal Dsssl formal list (#!optional)" formals))))))

   (define (rest-state args dsssl-arg)
      (cond
	 ((not (pair? args))
	  (err where "Illegal Dsssl formal list (#!rest)" formals))
	 (else
	  (match-case (car args)
	     ((and (? symbol?) ?id)
	      `(let ((,id ,dsssl-arg))
		  ,(exit-rest-state (cdr args) dsssl-arg)))
	     (else
	      (error where "Illegal Dsssl formal list (#!rest)" formals))))))
      
   (define (exit-rest-state args dsssl-arg)
      (cond
	 ((null? args)
	  body)
	 ((not (pair? args))
	  (err where "Illegal Dsssl formal list (#!rest)" formals))
	 ((eq? (car args) #!key)
	  (rest-key-state (cdr args) dsssl-arg))
	 (else
	  (err where "Illegal Dsssl formal list (#!rest)" formals))))

   (define (rest-key-state args dsssl-arg)
      (cond
	 ((null? args)
	  body)
	 (else
	  `(begin
	      (dsssl-check-key-args! ,dsssl-arg '())
	      ,(key-state args dsssl-arg)))))
   
   (define (no-rest-key-state args dsssl-arg)
      (define (formal-keyword-list args)
	 (map (lambda (arg)
		 (match-case arg
		    ((and (? symbol?) ?arg)
		     (symbol->keyword arg))
		    (((and (? symbol?) ?arg) ?-)
		     (symbol->keyword arg))
		    (else
		     (err where "Illegal Dsssl formal list (#!key)" formals))))
	      args))
      (cond
	 ((null? args)
	  body)
	 (else
	  `(begin
	      (dsssl-check-key-args! ,dsssl-arg ',(formal-keyword-list args))
	      ,(key-state args dsssl-arg)))))
   
   (define (key-state args dsssl-arg)
      (define (one-key-arg arg initializer)
	 `(let ((,arg (dsssl-get-key-arg ,dsssl-arg
					 ,(symbol->keyword arg)
					 ,initializer)))
	     ,(key-state (cdr args) dsssl-arg)))
      (cond
	 ((null? args)
	  body)
	 ((not (pair? args))
	  (err where "Illegal Dsssl formal list (#!key)" formals))
	 ((and (not (symbol? (car args))) (not (pair? (car args))))
	  (err where "Illegal Dsssl formal list (#!key)" formals))
	 (else
	  ;; an optional Dsssl formal
	  (match-case (car args)
	     (((and (? symbol?) ?arg) ?initializer)
	      (one-key-arg arg initializer))
	     ((and (? symbol?) ?arg)
	      (one-key-arg arg #f))
	     (else
	      (err where "Illegal Dsssl formal list (#!key)" formals))))))
   
   (scheme-state formals))
   
;*---------------------------------------------------------------------*/
;*    symbol->keyword ...                                              */
;*---------------------------------------------------------------------*/
(define (symbol->keyword symbol)
   (string->keyword (symbol->string symbol)))

;*---------------------------------------------------------------------*/
;*    dsssl-check-key-args! ...                                        */
;*    -------------------------------------------------------------    */
;*    This function checks that dsssl args are, at runtime,            */
;*    correctly formed. That is, the dsssl-args variable must hold     */
;*    a serie of pairs where the first element is a keyword.           */
;*    Furthermore, if key-list is non-nil, we check that for each      */
;*    pair, the key is present in key-list.                            */
;*---------------------------------------------------------------------*/
(define (dsssl-check-key-args! dsssl-args key-list)
   (if (null? key-list)
       (let loop ((args dsssl-args))
	  (cond
	     ((null? args)
	      #t)
	     ((or (not (pair? args))
		  (null? (cdr args))
		  (not (keyword? (car args))))
	      (error "dsssl formal parsing"
		     "Illegal #!keys parameters"
		     dsssl-args))
	     (else
	      (loop (cddr args)))))
       (let loop ((args dsssl-args))
	  (cond
	     ((null? args)
	      #t)
	     ((or (not (pair? args))
		  (null? (cdr args))
		  (not (keyword? (car args)))
		  (not (memq (car args) key-list)))
	      (error "dsssl formal parsing"
		     "Illegal #!keys parameters"
		     dsssl-args))
	     (else
	      (loop (cddr args)))))))
   
;*---------------------------------------------------------------------*/
;*    dsssl-get-key-arg ...                                            */
;*    -------------------------------------------------------------    */
;*    dsssl args have already been tested. We know for sure that       */
;*    it is a serie of pairs where first elements are keywords.        */
;*---------------------------------------------------------------------*/
(define (dsssl-get-key-arg dsssl-args keyword initializer)
   (let loop ((args dsssl-args))
      (cond
	 ((null? args)
	  initializer)
	 ((eq? (car args) keyword)
	  (cadr args))
	 (else
	  (loop (cddr args))))))
   
;*---------------------------------------------------------------------*/
;*    id-sans-type ...                                                 */
;*    -------------------------------------------------------------    */
;*    This function remove the type from an identifier. Thas is,       */
;*    provided the symbol `id::type', it returns `id'.                 */
;*---------------------------------------------------------------------*/
(define (id-sans-type::symbol id::symbol)
   (let* ((string (symbol->string id))
	  (len    (string-length string)))
      (let loop ((walker  0))
	 (cond
	    ((=fx walker len)
	     id)
	    ((and (char=? (string-ref string walker) #\:)
		  (<fx walker (-fx len 1))
		  (char=? (string-ref string (+fx walker 1)) #\:))
	     (string->symbol (substring string 0 walker)))
	    (else
	     (loop (+fx walker 1)))))))

;*---------------------------------------------------------------------*/
;*    dsssl-formals->scheme-formals ...                                */
;*    -------------------------------------------------------------    */
;*    This function parses a formal argument list and removes          */
;*    the Dsssl named constant in order to construct a regular Scheme  */
;*    formal parameter list.                                           */
;*      eg:   x y #!optional z #!rest r #!key k -> x y . z             */
;*    -------------------------------------------------------------    */
;*    This function does not check the whole correctness of the        */
;*    formal parameter list. It only checks until the first            */
;*    Dsssl formal parameter is found.                                 */
;*---------------------------------------------------------------------*/
(define (dsssl-formals->scheme-formals formals err)
   
   (define (dsssl-named-constant? obj)
      (memq obj '(#!optional #!rest #!key)))

   (define (dsssl-defaulted-formal? obj)
      (match-case obj
	 ((?- ?-)
	  #t)
	 (else
	  #f)))

   (define (dsssl-default-formal obj)
      (car obj))

   (let loop ((args  formals)
	      (dsssl #f))
      (cond
	 ((null? args)
	  '())
	 ((not (pair? args))
	  (cond
	     (dsssl
	      (err "Can't use both DSSSL named constant"
		   "and `.' notation"
		   formals))
	     ((not (symbol? args))
	      (err "Illegal formal parameter" "symbol expected" formals))
	     (else
	      (id-sans-type args))))
	 ((not (symbol? (car args)))
	  (cond
	     ((dsssl-named-constant? (car args))
	      (loop (cdr args) #t))
	     ((not dsssl)
	      (err "Illegal formal parameter" "symbol expected" formals))
	     ((dsssl-defaulted-formal? (car args))
	      (id-sans-type (dsssl-default-formal (car args))))
	     (else
	      (err "Illegal formal parameter"
		   "symbol or named constant expected"
		   formals))))
	 (dsssl
	  (id-sans-type (car args)))
	 (else
	  (cons (id-sans-type (car args))
		(loop (cdr args) #f))))))

   
