package biss.awt;

import java.awt.FontMetrics;
import java.awt.Graphics;

/**
 * auxiliary class implementing line based text drawing support
 * (used by TextField and TextArea)
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class LineBuffer
  extends Object
{
	FontMetrics Fm;
	public int Len;
	int MaxLen;
	int TabChars;
	int TabDelta;
	public char[] Text;
	int CharHeight;
	int MaxDescent;

public LineBuffer( FontMetrics fm) {
	Len      = 0;
	MaxLen   = 100;
	TabChars = 4;
	Text     = new char[ MaxLen];

	setMetrics( fm);
}

public LineBuffer( FontMetrics fm, LineBuffer lb){
	this( fm);
	insert( lb.Text, 0, lb.Len, -1);
}

public LineBuffer( FontMetrics fm, String s){
	this( fm);
	insert( s, -1);
}

public LineBuffer( FontMetrics fm, char[] ca, int offs, int end){
	this( fm);
	insert( ca, offs, end - offs, -1);
}

public LineBuffer( FontMetrics fm, char[] ca, int offs, int end, int tabWidth){
	this( fm);
	setTabWidth( tabWidth);
	insert( ca, offs, end - offs, -1);
}

public void append( String s) {
	insert( s, -1);
}

public void append( char c){
	insert( c, -1);
}

public void append( char[] ca, int offs, int end) {
	insert( ca, offs, end - offs, -1);
}

public String asString(){
	return String.valueOf( Text, 0, Len);
}

public int charsWidth(){
	return charsWidth( 0, Len, 0);
}

public int charsWidth( int sIdx, int len, int curPos){
	int ns, xLen = curPos;
	for ( int idx = sIdx; idx < sIdx + len; idx++) {
		if ( Text[idx] == '\t'){
			for ( ns = 0;;ns += TabDelta){
				if ( ns > xLen){
					xLen = ns;
					break;
				}
			}
		}
		else
			xLen += Fm.charWidth( Text[idx]);
	}
	return xLen - curPos;
}

public void draw( int sIdx, int len, Graphics g, int xPos,
           int yPos, int xoffs) {

	int idx, nextStop, si = -1;
	int eIdx = sIdx + len;
	boolean setStart = true;

	for ( idx=sIdx; idx<eIdx; idx++) {
		if ( Text[idx] == '\t'){
			if ( si > -1) {
				g.drawChars( Text, si, idx - si, xPos, yPos);
				xPos += Fm.charsWidth( Text, si, idx - si);
				si = -1;
			}	
			setStart = true;
			for ( nextStop = xoffs;;nextStop += TabDelta){
				if ( nextStop > xPos){
					xPos = nextStop;
					break;
				}
			}
		}
		else if ( setStart){
			si = idx;
			setStart = false;
		}
	}

	if ( si > -1 )
		g.drawChars( Text, si, idx - si, xPos, yPos);

}

public void free() {
	Len = 0;
}

public int indexOf( char c){
	for ( int i = 0; i < Len; i++){
		if ( Text[i] == c)
			return i;
	}
	return -1;
}

public boolean insert( String s, int iPos){
	if ( s != null) {
		char[] ca = new char[ s.length()];
		s.getChars( 0, s.length(), ca, 0);
		return insert( ca, 0, s.length(), iPos);
	}
	return false;
}

public boolean insert( char c, int iPos) {
	if ( (iPos == -1) || (iPos > Len))
		iPos = Len;

	verifyLen( 1);

	for( int idx = Len - 1; idx >= iPos; idx--)
		Text[idx + 1] = Text[idx];
	Text[iPos] = c;
	Len ++;	

	if ( iPos == Len - 1)
		return false;

	return true;	
}

public boolean insert( char[] ca, int offs, int len, int iPos) {
	int idx;

	if ( (iPos == -1) || (iPos > Len))
		iPos = Len;

	verifyLen( len);

	for( idx = Len - 1; idx >= iPos; idx--)
		Text[idx + len] = Text[idx];
	for( idx = 0; idx < len; idx++)
		Text[iPos + idx] = ca[offs + idx];
	Len += len;	

	if ( iPos == Len - len)
		return false;

	return true;	
}

public boolean isBlankLine () {
	int  i;
	char a;

	if ( Len == 0 )
		return true;

	// we iterate backwards because of indent levels
	for ( i=Len-1; i>=0; i-- ) {
		a = Text[i];
		if ( (a != ' ') || (a != '\t') )
			return false;
	}

	return true;
}

public int leadingTabs() {
	int lt = 0;
	for ( int idx=0; idx<Len; idx++){
		if ( Text[idx] != '\t' )
			break;
		lt++;
	}
	return lt;
}

public int posFor( int cIdx){
	return charsWidth( 0, cIdx, 0);
}

public void remove( int offs, int end){
	for ( int i=offs; i<Len; i++){
		if ( i + end - offs >= Len)
			break;
		Text[i] = Text[i + end - offs];
	}	
	Len -= end - offs;
}

public int removeAll( char c){
	int i, j, ol = Len;
	for ( i = 0; i < Len; i++){
		if ( Text[i] == c){
			for ( j = i + 1; j < Len; j++)
				Text[j-1] = Text[j];
			i--;
			Len--;
		}
	}

	return ol - Len;
}

public void replace( int offs, int end, String s){
	remove( offs, end);
	insert( s, offs);
}

public void setLen( int i){
	Len = i;
}

public void setMetrics( FontMetrics fm) {
	Fm = fm;
	if ( Fm != null){
		CharHeight = Fm.getHeight();
		MaxDescent = Fm.getMaxDescent();
		TabDelta = Fm.charWidth( 'x') * TabChars;
	}
	else
		TabDelta = 0;
}

public void setTabWidth( int numChars) {
	TabChars = numChars;
	if ( Fm != null)
		TabDelta = Fm.charWidth( 'x') * TabChars;
	else
		TabDelta = 0;
}

public String toString() {
	return new String( Text, 0, Len);
}

void verifyLen( int addLen){
	if ( Len + addLen >= MaxLen){
		MaxLen += Math.max( addLen, MaxLen / 2);
		char[] cn = new char[ MaxLen];
		for ( int i = 0; i < Len; i++)
			cn[i] = Text[i];
		Text = cn;
	}
}
}
