/*
** Copyright (c) Massachusetts Institute of Technology 1994-1997.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. Works derived from this code are not distributed for
**        commercial gain without explicit permission from MIT 
**        (for permission contact lclint-request@larch.lcs.mit.edu).
*/
/*
** mapping.c
**
** Module for lsymbol maps.
**
**  AUTHOR:
**	Yang Meng Tan, Massachusetts Institute of Technology
*/

# include "lclintMacros.nf"
# include "llbasic.h"

/*@constant int MAPPING_SIZE; @*/
# define MAPPING_SIZE 127

/* use lower-order bits by masking out higher order bits */

/*@-macrofcndecl@*/
# define mmash(key)  ((key) & MAPPING_SIZE)
/*@=macrofcndecl@*/

static void mappair_free (/*@null@*/ /*@only@*/ mappair *p)
{
  if (p == NULL) 
    {
      return;
    }
  else
    {
      mappair_free (p->next);
      sfree (p);
    }
}

void mapping_free (/*@only@*/ mapping *m)
{
  int i;

  for (i = 0; i <= MAPPING_SIZE; i++)
    {
      mappair_free (m->buckets[i]);
    }
  
  sfree (m->buckets);
  sfree (m);
}

/*@only@*/ mapping *
mapping_create (void)
{
  int i;
  mapping *t = (mapping *) dmalloc (sizeof (*t));

  t->buckets = (mappair **) dmalloc ((MAPPING_SIZE + 1) * sizeof (*t->buckets));
  t->count = 0;

  for (i = 0; i <= MAPPING_SIZE; i++)
    {
      t->buckets[i] = (mappair *) 0;
    }

  return t;
}

lsymbol
mapping_find (mapping * t, lsymbol domain)
{
  mappair *entry;
  unsigned int key;

  key = mmash (domain);
  entry = t->buckets[key];
  for (; entry != NULL; entry = entry->next)
    {
      if (entry->domain == domain)
	return entry->range;
    }

  return lsymbol_undefined;
}

void
mapping_bind (mapping *t, lsymbol domain, lsymbol range)
{
 /* add the binding (domain -> range) to t */
 /* can assume that the binding is a new one in m, so no need
    to check. */
  mappair *entry;
  mappair *newentry = (mappair *) dmalloc (sizeof (*newentry));
  unsigned int key;

  key = mmash (domain);
  /*@-deparrays@*/ entry = t->buckets[key]; /*@=deparrays@*/
  newentry->domain = domain;
  newentry->range = range;
  newentry->next = entry;

  t->buckets[key] = newentry; 
  t->count++;
}
