(* Some notion of synthetic events *)
open Tk
open Widget
open Protocol

(* To each event is associated a table of (widget, callback) *)
let events = Hashtblc.create 37

(* Notes:
 *   "cascading" events (on the same event) are not supported 
 *   Only one binding active at a time for each event on each widget.
 *)

(* Get the callback table associated with <name>. Initializes if required *)
let get_event name =
  try Hashtblc.find events name 
  with
    Not_found ->
      let h = Hashtblc.create 37 in
       Hashtblc.add events name h;
       (* Initialize the callback invocation mechanism, based on 
          variable trace
        *)
       let var = "camltk_events(" ^ name ^")" in
       let rec set () =
	 Textvariable.handle (Textvariable.coerce var)
	 (fun () ->
	    begin match tkEval [| TkToken "set"; TkToken var |] with
	      "all" -> (* Invoke all callbacks *)
		Hashtblc.do_table_rev 
      	       	  (fun p f -> 
      	       	     try 
      	       	      f (cTKtoCAMLwidget p) 
      	       	     with _ -> ())
                  h
	    | p -> (* Invoke callback for p *)
		try
		  let w = cTKtoCAMLwidget p
      	       	  and f = Hashtblc.find h p in
		    f w
      	        with
      	       	  _ -> ()
            end; 
      	    set ()(* reactivate the callback *)
            ) in
       set();
       h 

(* Remove binding for event <name> on widget <w> *)
let remove w name =   
  Hashtblc.remove (get_event name) (Widget.name w)

(* Adds <f> as callback for widget <w> on event <name> *)
let bind w name f =
  remove w name;
  Hashtblc.add (get_event name) (Widget.name w) f

(* Sends event <name> to all widgets *)
let broadcast name =
  tkEval [| TkToken "set"; 
      	    TkToken ("camltk_events(" ^ name ^")");
	    TkToken "all" |];
  ()	    

(* Sends event <name> to widget <w> *)
let send name w =
  tkEval [| TkToken "set"; 
      	    TkToken ("camltk_events(" ^ name ^")");
	    TkToken (Widget.name w) |];
  ()	    

(* Remove all callbacks associated to widget <w> *)
let remove_callbacks w =
  Hashtblc.iter (fun _ h -> Hashtblc.remove h (Widget.name w)) events

let _ =
  add_destroy_hook remove_callbacks
