#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "chess.h"
#include "data.h"
#include "epdglue.h"

/* modified 01/07/98 */
/*
********************************************************************************
*                                                                              *
*   SearchRoot() is the recursive routine used to implement the alpha/beta     *
*   negamax search (similar to minimax but simpler to code.)  SearchRoot() is  *
*   only called when ply=1.  it is somewhat different from Search() in that    *
*   some things (null move search, hash lookup, etc.) are not useful at the    *
*   root of the tree.  SearchRoot() calls Search() to search any positions     *
*   that are below ply=1.                                                      *
*                                                                              *
********************************************************************************
*/
int SearchRoot(int alpha, int beta, int wtm, int depth)
{
  register int first_move=1;
  register int initial_alpha, value;
  register int extensions, begin_root_nodes;
/*
 ----------------------------------------------------------
|                                                          |
|   initialize.  set NextMove() status to 0 so it will     |
|   know what has to be done.                              |
|                                                          |
 ----------------------------------------------------------
*/
  in_check[2]=0;
  extended_reason[2]=no_extension;
  initial_alpha=alpha;
  RepetitionCheck(1,wtm);
  in_check[1]=Check(wtm);
  next_status[1].phase=ROOT_MOVES;
  killer_count1[2]=0;
  killer_count2[2]=0;
/*
 ----------------------------------------------------------
|                                                          |
|   now iterate through the move list and search the       |
|   resulting positions.  note that Search() culls any     |
|   move that is not legal by using Check().  the special  |
|   case is that we must find one legal move to search to  |
|   confirm that it's not a mate or draw.                  |
|                                                          |
 ----------------------------------------------------------
*/
  while ((current_phase[1]=NextRootMove(wtm))) {
    extended_reason[1]=0;
#if !defined(FAST)
    if (1 <= trace_level)
      SearchTrace(1,depth,wtm,alpha,beta,"SearchRoot",current_phase[1]);
#endif
/*
 ----------------------------------------------------------
|                                                          |
|   if we push a pawn to the 7th rank, we need to look     |
|   deeper to see if it can promote.                       |
|                                                          |
 ----------------------------------------------------------
*/
    extensions=-60;
    if (Piece(current_move[1])==pawn &&
        ((wtm && To(current_move[1]) > H5 && TotalBlackPieces<16 &&
         !And(mask_pawn_passed_w[To(current_move[1])],BlackPawns)) ||
        (!wtm && To(current_move[1]) < A4 && TotalWhitePieces<16 &&
         !And(mask_pawn_passed_b[To(current_move[1])],WhitePawns)) ||
        push_extensions[To(current_move[1])]) &&
        Swap(From(current_move[1]),To(current_move[1]),wtm) >= 0) {
      extended_reason[1]|=passed_pawn_extension;
      passed_pawn_extensions_done++;
      extensions+=PUSH_PASSED_PAWN;
    }
/*
 ----------------------------------------------------------
|                                                          |
|   now make the move and search the resulting position.   |
|                                                          |
 ----------------------------------------------------------
*/
    MakeMove(1,current_move[1],wtm);
/*
 ----------------------------------------------------------
|                                                          |
|   if the move to be made checks the opponent, then we    |
|   need to remember that he's in check and also extend    |
|   the depth by one ply for him to get out.               |
|                                                          |
 ----------------------------------------------------------
*/
    if (Check(ChangeSide(wtm))) {
      in_check[2]=1;
      extended_reason[2]=check_extension;
      check_extensions_done++;
      extensions+=IN_CHECK;
    }
    else {
      in_check[2]=0;
      extended_reason[2]=no_extension;
    }
/*
 ----------------------------------------------------------
|                                                          |
|   now call Search to produce a value for this move.      |
|                                                          |
 ----------------------------------------------------------
*/
    begin_root_nodes=nodes_searched;
    extensions=Min(extensions,0);
    if (first_move) {
      value=-ABSearch(-beta,-alpha,ChangeSide(wtm),
                      depth+extensions,2,DO_NULL);
      if (abort_search) {
        UnMakeMove(1,current_move[1],wtm);
        return(alpha);
      }
      first_move=0;
    }
    else {
      value=-ABSearch(-alpha-1,-alpha,ChangeSide(wtm),
                      depth+extensions,2,DO_NULL);
      if (abort_search) {
        UnMakeMove(1,current_move[1],wtm);
        return(alpha);
      }
      if ((value > alpha) && (value < beta)) {
        value=-ABSearch(-beta,-alpha,ChangeSide(wtm),
                        depth+extensions,2,DO_NULL);
        if (abort_search) {
          UnMakeMove(1,current_move[1],wtm);
          return(alpha);
        }
      }
    }
    root_nodes[root_move]=nodes_searched-begin_root_nodes;
    if (value > alpha) {
      SearchOutput(value,beta);
      if(value >= beta) {
        HistoryRefutation(1,depth,wtm);
        UnMakeMove(1,current_move[1],wtm);
        StoreRefutation(1,depth,wtm,value,0);
        return(value);
      }
      alpha=value;
      root_value=alpha;
      beta=alpha+30;
      root_beta=beta;
    }
    UnMakeMove(1,current_move[1],wtm);
  }
/*
 ----------------------------------------------------------
|                                                          |
|   all moves have been searched.  if none were legal,     |
|   return either MATE or DRAW depending on whether the    |
|   side to move is in check or not.                       |
|                                                          |
 ----------------------------------------------------------
*/
  if (first_move == 1) {
    value=(Check(wtm)) ? -(MATE-1) :
                             ((wtm==root_wtm) ? DrawScore() : -DrawScore());
    if (value >=alpha && value <beta) {
      SavePVS(1,value,0);
#if !defined(FAST)
      if (1 <= trace_level) printf("Search() no moves!  ply=1\n");
#endif
    }
    return(value);
  }
  else {
    if (alpha != initial_alpha) {
      memcpy(&pv[0].path[1],&pv[1].path[1],(pv[1].path_length)*4);
      memcpy(&pv[0].path_hashed,&pv[1].path_hashed,3);
      pv[0].path[0]=current_move[0];
      HistoryBest(1,depth,wtm);
    }
    StoreBest(1,depth,wtm,alpha,initial_alpha,0);
    return(alpha);
  }
}

/* modified 11/26/96 */
/*
********************************************************************************
*                                                                              *
*   SearchOutput() is used to print the principal variation whenever it        *
*   changes.  one additional feature is that SearchOutput() will try to do     *
*   something about variations truncated by the transposition table.  if the   *
*   variation was cut short by a transposition table hit, then we can make the *
*   last move, add it to the end of the variation and extend the depth of the  *
*   variation to cover it.                                                     *
*                                                                              *
********************************************************************************
*/
void SearchOutput(int value, int bound)
{
#define PrintOK() (nodes_searched>noise_level || value>(MATE-100))
  register int *mv, *mvp;
  register int wtm;
  int temp_root_nodes;

/*
 ----------------------------------------------------------
|                                                          |
|   first, move the best move to the top of the ply-1 move |
|   list if it's not already there, so that it will be the |
|   first move tried in the next iteration.                |
|                                                          |
 ----------------------------------------------------------
*/
  wtm=root_wtm;
  if (!abort_search) {
    whisper_value=(analyze_mode && !root_wtm) ? -value : value;
    whisper_depth=iteration_depth;
    for (mvp=last[0];mvp<last[1];mvp++) if(current_move[1]== *mvp) break;
    if (mvp != last[0]) {
      temp_root_nodes=root_nodes[mvp-last[0]];
      for (mv=mvp;mv>last[0];mv--) {
        *mv=*(mv-1);
        root_nodes[mv-last[0]]=root_nodes[mv-1-last[0]];
      }
      root_nodes[0]=temp_root_nodes;
      *last[0]=current_move[1];
      easy_move=0;
    }
    end_time=ReadClock(time_type);
/*
 ----------------------------------------------------------
|                                                          |
|   if this is not a fail-high move, then output the PV    |
|   by walking down the path being backed up.              |
|                                                          |
 ----------------------------------------------------------
*/
    if(value < bound) {
      UnMakeMove(1,pv[1].path[1],root_wtm);
      DisplayPV(6, wtm, end_time-start_time, value, pv[1]);
      MakeMove(1,pv[1].path[1],root_wtm);
    }
    else {
      if (PrintOK()) {
        Print(2,"               %2i   %s     ++   ",iteration_depth,
        DisplayTime(end_time-start_time));
        UnMakeMove(1,current_move[1],wtm);
        if (display_options&64) Print(2,"%d. ",move_number);
        if ((display_options&64) && !wtm) Print(2,"... ");
        Print(2,"%s!!\n",OutputMove(&current_move[1],1,wtm));

        whisper_text[0]=0;
        if (display_options&64)
          sprintf(whisper_text," %d.",move_number);
        if ((display_options&64) && !wtm)
          sprintf(whisper_text+strlen(whisper_text)," ...");
        sprintf(whisper_text+strlen(whisper_text)," %s!!",
                OutputMove(&current_move[1],1,wtm));
        MakeMove(1,current_move[1],wtm);
        Whisper(6,iteration_depth,end_time-start_time,whisper_value,
                nodes_searched,-1,whisper_text);
      }
      if (current_move[1] != pv[1].path[1]) {
        pv[1].path[1]=current_move[1];
        pv[1].path_length=1;
        pv[1].path_hashed=0;
        pv[1].path_iteration_depth=iteration_depth;
      }
    }
  }
}

/* modified 05/20/96 */
/*
********************************************************************************
*                                                                              *
*   SearchTrace() is used to print the search trace output each time a node is *
*   traversed in the tree.                                                     *
*                                                                              *
********************************************************************************
*/
void SearchTrace(int ply, int depth, int wtm, int alpha, int beta, char* name,
                 int phase)
{
  int i;
  for (i=1;i<ply;i++) printf("  ");
  printf("%d  %s d:%5.2f [%s,",ply,OutputMove(&current_move[ply],ply,wtm),
         (float) depth/ (float) INCREMENT_PLY,DisplayEvaluation(alpha));
  printf("%s] n:%d %s(%d)\n", DisplayEvaluation(beta),
         (nodes_searched),name,phase);
}
